"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.create = create;
const language_core_1 = require("@vue/language-core");
const html = require("vscode-html-languageservice");
const nameCasing_1 = require("../nameCasing");
const utils_1 = require("../utils");
function create({ getComponentNames, getElementNames, getComponentProps }) {
    return {
        name: 'vue-missing-props-hints',
        capabilities: {
            inlayHintProvider: {},
        },
        create(context) {
            let intrinsicElementNames;
            return {
                async provideInlayHints(document, range, cancellationToken) {
                    const info = (0, utils_1.getEmbeddedInfo)(context, document, 'template');
                    if (!info) {
                        return;
                    }
                    const { sourceScript, root } = info;
                    const enabled = await context.env.getConfiguration?.('vue.inlayHints.missingProps') ?? false;
                    if (!enabled) {
                        return;
                    }
                    const scanner = getScanner(context, document);
                    if (!scanner) {
                        return;
                    }
                    const result = [];
                    const casing = await (0, nameCasing_1.checkCasing)(context, sourceScript.id);
                    const components = await getComponentNames(root.fileName) ?? [];
                    const componentProps = {};
                    intrinsicElementNames ??= new Set(await getElementNames(root.fileName) ?? []);
                    let token;
                    let current;
                    while ((token = scanner.scan()) !== html.TokenType.EOS) {
                        if (token === html.TokenType.StartTag) {
                            const tagName = scanner.getTokenText();
                            const tagOffset = scanner.getTokenOffset();
                            const checkTag = tagName.includes('.')
                                ? tagName
                                : components.find(component => component === tagName || (0, language_core_1.hyphenateTag)(component) === tagName);
                            if (intrinsicElementNames.has(tagName) || !checkTag) {
                                continue;
                            }
                            if (tagOffset < document.offsetAt(range.start)) {
                                continue;
                            }
                            if (tagOffset > document.offsetAt(range.end)) {
                                break;
                            }
                            if (!componentProps[checkTag]) {
                                if (cancellationToken.isCancellationRequested) {
                                    break;
                                }
                                componentProps[checkTag] = (await getComponentProps(root.fileName, checkTag) ?? [])
                                    .filter(prop => prop.required)
                                    .map(prop => prop.name);
                            }
                            current = {
                                unburnedRequiredProps: [...componentProps[checkTag]],
                                labelOffset: scanner.getTokenOffset() + scanner.getTokenLength(),
                            };
                        }
                        else if (token === html.TokenType.AttributeName) {
                            if (current) {
                                let attrText = scanner.getTokenText();
                                if (attrText === 'v-bind') {
                                    current.unburnedRequiredProps = [];
                                }
                                else {
                                    // remove modifiers
                                    if (attrText.includes('.')) {
                                        attrText = attrText.split('.')[0];
                                    }
                                    // normalize
                                    if (attrText.startsWith('v-bind:')) {
                                        attrText = attrText.slice('v-bind:'.length);
                                    }
                                    else if (attrText.startsWith(':')) {
                                        attrText = attrText.slice(':'.length);
                                    }
                                    else if (attrText.startsWith('v-model:')) {
                                        attrText = attrText.slice('v-model:'.length);
                                    }
                                    else if (attrText === 'v-model') {
                                        attrText = root.vueCompilerOptions.target >= 3 ? 'modelValue' : 'value'; // TODO: support for experimentalModelPropName?
                                    }
                                    else if (attrText.startsWith('v-on:')) {
                                        attrText = 'on-' + (0, language_core_1.hyphenateAttr)(attrText.slice('v-on:'.length));
                                    }
                                    else if (attrText.startsWith('@')) {
                                        attrText = 'on-' + (0, language_core_1.hyphenateAttr)(attrText.slice('@'.length));
                                    }
                                    current.unburnedRequiredProps = current.unburnedRequiredProps.filter(propName => {
                                        return attrText !== propName
                                            && attrText !== (0, language_core_1.hyphenateAttr)(propName);
                                    });
                                }
                            }
                        }
                        else if (token === html.TokenType.StartTagSelfClose || token === html.TokenType.StartTagClose) {
                            if (current) {
                                for (const requiredProp of current.unburnedRequiredProps) {
                                    result.push({
                                        label: `${requiredProp}!`,
                                        paddingLeft: true,
                                        position: document.positionAt(current.labelOffset),
                                        kind: 2,
                                        textEdits: [{
                                                range: {
                                                    start: document.positionAt(current.labelOffset),
                                                    end: document.positionAt(current.labelOffset),
                                                },
                                                newText: ` :${casing.attr === nameCasing_1.AttrNameCasing.Kebab ? (0, language_core_1.hyphenateAttr)(requiredProp) : requiredProp}=`,
                                            }],
                                    });
                                }
                                current = undefined;
                            }
                        }
                    }
                    return result;
                },
            };
        },
    };
    function getScanner(context, document) {
        if (document.languageId === 'html') {
            return context.inject('html/languageService').createScanner(document.getText());
        }
        else {
            const pugDocument = context.inject('pug/pugDocument', document);
            if (pugDocument) {
                return context.inject('pug/languageService').createScanner(pugDocument);
            }
        }
    }
}
//# sourceMappingURL=vue-missing-props-hints.js.map