static char rcsid[] = "$Id: regiondb.c 223349 2020-10-28 02:49:25Z twu $";
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifndef HAVE_MEMCPY
# define memcpy(d,s,n) bcopy((s),(d),(n))
#endif
#ifndef HAVE_MEMMOVE
# define memmove(d,s,n) bcopy((s),(d),(n))
#endif

#include "regiondb.h"
#include "regiondbdef.h"

#ifdef WORDS_BIGENDIAN
#include "bigendian.h"
#else
#include "littleendian.h"
#endif

#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>		/* For memcpy */

#include <sys/mman.h>		/* For munmap */

#ifdef HAVE_UNISTD_H
#include <unistd.h>		/* For lseek and close */
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>		/* For off_t */
#endif
#if HAVE_DIRENT_H
# include <dirent.h>
# define NAMLEN(dirent) strlen((dirent)->d_name)
#else
# define dirent direct
# define NAMLEN(dirent) (dirent)->d_namlen
# if HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# if HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# if HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif

#include "assert.h"
#include "mem.h"
#include "fopen.h"
#include "types.h"		/* For Oligospace_T */
#include "filesuffix.h"
#include "intersect.h"
#include "genome128_hr.h"

#if !defined(LARGE_GENOMES)
#include "merge-diagonals-simd-uint4.h"
#elif !defined(HAVE_AVX512) && !defined(HAVE_AVX2)
#include "merge-diagonals-heap.h" /* For Merge_diagonals_uint8 */
#else
#include "merge-diagonals-simd-uint8.h" /* For Merge_diagonals_uint8 */
#endif

#ifdef GSNAP
#include "univdiag.h"
#include "univdiagdef.h"
#endif


#ifdef HAVE_PTHREAD
#include <pthread.h>
#endif

#if defined(HAVE_SSE2)
#include <emmintrin.h>
#endif
#if defined(HAVE_SSE4_1)
#include <smmintrin.h>
#endif
#ifdef HAVE_AVX2
#include <immintrin.h>
#endif
#ifdef HAVE_AVX512
#include <immintrin.h>
#endif



#ifdef DEBUG0
#define debug0(x) x
#else
#define debug0(x)
#endif

/* fill procedures */
#ifdef DEBUG1
#define debug1(x) x
#else
#define debug1(x)
#endif

/* intersection */
#ifdef DEBUG8
#define debug8(x) x
#else
#define debug8(x)
#endif

/* Summary of results */
#ifdef DEBUG9
#define debug9(x) x
#else
#define debug9(x)
#endif

/* binary_search */
#ifdef DEBUG10
#define debug10(x) x
#else
#define debug10(x)
#endif


static int region1part;

#define T Regiondb_T

/* See regiondbdef.h for a a description of the data structure */

/************************************************************************
 *   Debugging procedures
 ************************************************************************/

#if 0
static void
print_vector_hex (__m128i x) {
  UINT4 *s = (UINT4 *) &x;

  /* printf("%08X %08X %08X %08X\n",s[0],s[1],s[2],s[3]); */
  printf("%08X %08X %08X %08X\n",s[3],s[2],s[1],s[0]);
  return;
}

static void
print_vector_uint (__m128i x) {
  UINT4 *s = (UINT4 *) &x;

  /* printf("%d %d %d %d\n",s[0],s[1],s[2],s[3]); */
  printf("%u %u %u %u\n",s[3],s[2],s[1],s[0]);
  return;
}
#endif



#ifndef PMAP

/*                      87654321 */
#define LOW_TWO_BITS  0x00000003

#if (defined(DEBUG0) || defined(DEBUG1) || defined(DEBUG2))
static char *
shortoligo_nt (Oligospace_T oligo, Width_T oligosize) {
  char *nt;
  Width_T i, j;
  Oligospace_T lowbits;

  nt = (char *) CALLOC(oligosize+1,sizeof(char));
  j = oligosize-1;
  for (i = 0; i < oligosize; i++) {
    lowbits = oligo & LOW_TWO_BITS;
    switch (lowbits) {
    case RIGHT_A: nt[j] = 'A'; break;
    case RIGHT_C: nt[j] = 'C'; break;
    case RIGHT_G: nt[j] = 'G'; break;
    case RIGHT_T: nt[j] = 'T'; break;
    }
    oligo >>= 2;
    j--;
  }

  return nt;
}
#endif

#endif


#if !defined(UTILITYP)
/* Have to use UINT4, and not UINT2, because highi could be 65536 */
static UINT4
binary_search_with_term (UINT4 lowi, UINT4 highi, UINT2 *positions, UINT4 term, UINT4 goal) {
  UINT4 middlei;

  debug10(printf("entered binary search_with_term with lowi=%d, highi=%d, term=%u, goal=%u\n",
		 lowi,highi,term,goal));

  while (lowi < highi) {
    middlei = lowi + ((highi - lowi) / 2);
    debug10(printf("  binary: %d:%u %d:%u %d:%u   vs. %u\n",
		   lowi,positions[lowi] + term,middlei,positions[middlei] + term,
		   highi-1,positions[highi-1] + term,goal));
    if (goal < positions[middlei] + term) {
      highi = middlei;
    } else if (goal > positions[middlei] + term) {
      lowi = middlei + 1;
    } else {
      debug10(printf("binary search returns %d\n",middlei));
      return middlei;
    }
  }

  debug10(printf("binary search returns %d\n",highi));
  return highi;
}
#endif


#if 0
/* Not sure if we need SIMD unsigned addition */
#ifdef LARGE_GENOMES
static Univcoord_T *
add_region_term (Univcoord_T *diagonals, int n, Univcoord_T region_term) {
  Univcoord_T *ptr;

#if defined(HAVE_AVX512)
  __m512i _region_term_x8, _diagonals_x8;
#endif
#if defined(HAVE_AVX2)
  __m256i _region_term_x4, _diagonals_x4;
#endif


  ptr = &(diagonals[0]);

#if defined(HAVE_AVX512)
  _region_term_x8 = _mm512_set1_epi64(region_term);
	
  while (ptr + 8 < &(diagonals[n])) {
    _diagonals_x8 = _mm512_loadu_si512((__m512i *) ptr); /* Read 8 diagonals */
    _mm512_storeu_si512((__m512i *) ptr, _mm512_add_epi64(_diagonals_x8, _region_term_x8));
    ptr += 8;
  }
#endif

#if defined(HAVE_AVX2)
  _region_term_x4 = _mm256_set1_epi64x(region_term);

  while (ptr + 4 < &(diagonals[n])) {
    _diagonals_x4 = _mm256_loadu_si256((__m256i *) ptr); /* Read 4 diagonals */
    _mm256_storeu_si256((__m256i *) ptr, _mm256_add_epi64(_diagonals_x8, _region_term_x8));
    ptr += 4;
  }
#endif

  while (ptr < &(diagonals[n])) {
    *ptr++ += region_term;
  }

  return diagonals;
}

#else
static Univcoord_T *
add_region_term (Univcoord_T *diagonals, int n, Univcoord_T region_term) {
  Univcoord_T *ptr;

#if defined(HAVE_AVX512)
  __m512i _region_term_x16, _diagonals_x16;
#endif
#if defined(HAVE_AVX2)
  __m256i _region_term_x8, _diagonals_x8;
#endif
#if defined(HAVE_SSE2)
  __m128i _region_term_x4, _diagonals_x4;
#endif


  ptr = &(diagonals[0]);

#if defined(HAVE_AVX512)
  _region_term_x16 = _mm512_set1_epi32(region_term);
	
  while (ptr + 16 < &(diagonals[n])) {
    _diagonals_x16 = _mm512_loadu_si512((__m512i *) ptr); /* Read 16 diagonals */
    _mm512_storeu_si512((__m512i *) ptr, _mm512_add_epi32(_diagonals_x16, _region_term_x16));
    ptr += 16;
  }
#endif

#if defined(HAVE_AVX2)
  _region_term_x8 = _mm256_set1_epi32(region_term);

  while (ptr + 8 < &(diagonals[n])) {
    _diagonals_x8 = _mm256_loadu_si256((__m256i *) ptr); /* Read 8 diagonals */
    _mm256_storeu_si256((__m256i *) ptr, _mm256_add_epi32(_diagonals_x8, _region_term_x8));
    ptr += 8;
  }
#endif

#if defined(HAVE_SSE2)
  _region_term_x4 = _mm_set1_epi32(region_term);

  while (ptr + 4 < &(diagonals[n])) {
    _diagonals_x4 = mm_loadu_si128((__m128i *) ptr); /* Read 4 diagonals */
    _mm_storeu_si128((__m128i *) ptr, _mm_add_epi32(_diagonals_x4, _region_term_x4));
    ptr += 4;
  }
#endif

  while (ptr < &(diagonals[n])) {
    *ptr++ += region_term;
  }

  return diagonals;
}
#endif
#endif


#if !defined(UTILITYP)
static int
binary_search_uint2 (int lowi, int highi, UINT2 *positions, UINT2 goal) {
  int middlei;

  debug10(printf("entered binary search with lowi=%d, highi=%d, goal=%u\n",lowi,highi,goal));

  while (lowi < highi) {
    middlei = lowi + ((highi - lowi) / 2);
    debug10(printf("  binary: %d:%u %d:%u %d:%u   vs. %u\n",
		   lowi,positions[lowi],middlei,positions[middlei],
		   highi,positions[highi],goal));
    if (goal < positions[middlei]) {
      highi = middlei;
    } else if (goal > positions[middlei]) {
      lowi = middlei + 1;
    } else {
      debug10(printf("binary search returns %d\n",middlei));
      return middlei;
    }
  }

  debug10(printf("binary search returns %d\n",highi));
  return highi;
}
#endif


#if !defined(UTILITYP)
/* Modeled after Intersect_exact in intersect.c */
/* Assumes that npositionsa and npositionsb are both greater than 0 */
static int
intersect_within_region (Univcoord_T *diagonals_ptr,
			 UINT2 *positionsa, int npositionsa, int diagterma,
			 UINT2 *positionsb, int npositionsb, int diagtermb,
			 Univcoord_T region_term) {
  int ndiagonals = 0;
  Univcoord_T adj_region_term;	/* region_term + diagterm */
  UINT2 local_goal, delta;
  UINT2 *positions0, *positions1;
  int npositions0, npositions1, j;

#ifdef DEBUG8
  printf("intersect_within_region:\n");
  for (j = 0; j < npositionsa; j++) {
    printf("%hu ",positionsa[j]);
  }
  printf("\n");
  for (j = 0; j < npositionsb; j++) {
    printf("%hu ",positionsb[j]);
  }
  printf("\n\n");
#endif

  assert(npositionsa > 0 && npositionsb > 0);
  if (npositionsa < npositionsb) {
    positions0 = positionsa;
    positions1 = positionsb;
    npositions0 = npositionsa;
    npositions1 = npositionsb;

    adj_region_term = region_term + (Univcoord_T) diagtermb;  /* local_goal based on larger list */
    delta = (UINT2) (diagterma - diagtermb); /* list0 + (diagterm0 - diagterm1) = list1 */
    debug8(printf("delta is %hu\n",delta));
    /* delta is essentially positive, so no need to handle overflow */

  } else {
    positions0 = positionsb;
    positions1 = positionsa;
    npositions0 = npositionsb;
    npositions1 = npositionsa;

    adj_region_term = region_term + (Univcoord_T) diagterma;  /* local_goal based on larger list */
    delta = (UINT2) (diagtermb - diagterma); /* list0 + (diagterm0 - diagterm1) = list1 */
    debug8(printf("delta is %hu\n",delta));
    /* delta is essentially negative: a large number close to 65536 */

    /* Therefore, need to advance second list so local_goal does not overflow */
    while (npositions0 > 0 && (*positions0) < (UINT2) -delta) {
      debug8(printf("Advancing positions0 past %hu\n",*positions0));
      ++positions0;
      --npositions0;
    }

  }
  /* npositions0 is the smaller list */

  debug8(printf("intersect_within_region with %d positions <= %d positions\n",
		npositions0,npositions1));

  while (npositions0 > 0) {
    local_goal = (*positions0) + delta;
    debug8(printf("intersection list 0: %d:%hu => local_goal %hu\n",npositions0,*positions0,local_goal));
    if (npositions1 > 0 && *positions1 < local_goal) {
      j = 1;
      while (j < npositions1 && positions1[j] < local_goal) {
	j <<= 1;		/* gallop by 2 */
      }
      if (j >= npositions1) {
	j = binary_search_uint2(j >> 1,npositions1,positions1,local_goal);
      } else {
	j = binary_search_uint2(j >> 1,j,positions1,local_goal);
      }
      positions1 += j;
      npositions1 -= j;
    }

    if (npositions1 <= 0) {
      return ndiagonals;

    } else if ((*positions1) == local_goal) {
      /* Found local goal.  Save and advance */
      debug8(printf("    intersection list 1: %d:%hu  found\n",npositions1,*positions1));
      diagonals_ptr[ndiagonals++] = (Univcoord_T) local_goal + adj_region_term;
      ++positions1;
      --npositions1;
    }

    ++positions0;
    --npositions0;
  }
  debug8(printf("\n"));

  return ndiagonals;
}
#endif



#if !defined(UTILITYP)
/* Modeled after Intersect_exact in intersect.c */
/* Assumes that npositions0 and npositions1 are both greater than 0 */
static int
intersect_between_regions (Univcoord_T *diagonals_ptr,
			   UINT2 *positions0, int npositions0, int diagterm0,
			   UINT2 *positions1, int npositions1, int diagterm1,
			   Univcoord_T region_term_1) {
  int ndiagonals = 0;
  UINT2 local_goal, diagterm, delta;
#ifdef DEBUG8
  int j;
#endif


#ifdef DEBUG8
  printf("intersect_between_regions with diagterm0 %d and diagterm1 %d:\n",diagterm0,diagterm1);
  for (j = 0; j < npositions0; j++) {
    printf("%hu ",positions0[j]);
  }
  printf("\n");
  for (j = 0; j < npositions1; j++) {
    printf("%hu ",positions1[j]);
  }
  printf("\n\n");
#endif

  diagterm = (UINT2) diagterm1;	/* local_goal based on second list */
  delta = (UINT2) (diagterm0 - diagterm1); /* list0 + (diagterm0 - diagterm1) = list1 */
  debug8(printf("delta is %hu\n",delta));

  while (npositions0 > 0) {
    local_goal = (*positions0) + delta;
    debug8(printf("intersection list 0: %d:%hu => local_goal %hu\n",npositions0,*positions0,local_goal));
    if (npositions1 > 0 && *positions1 < local_goal) {
      positions1 += 1;
      npositions1 -= 1;
    }

    if (npositions1 <= 0) {
      return ndiagonals;

    } else if ((*positions1) == local_goal) {
      /* Found local goal.  Save and advance */
      debug8(printf("    intersection list 1: %d:%hu  found\n",npositions1,*positions1));
      /* Previously used region_term_0 when diagterm1 was negative, so diagterm was near REGION_LENGTH */
      /* Now using region_term_1 */
      debug8(printf("    storing local_goal %hu + diagterm %d + region_term_1 %u\n",
		    local_goal,diagterm,region_term_1));
      diagonals_ptr[ndiagonals++] = (Univcoord_T) (local_goal + diagterm) + region_term_1;
      ++positions1;
      --npositions1;
    }

    ++positions0;
    --npositions0;
  }
  debug8(printf("\n"));

  return ndiagonals;
}
#endif



/* Output for GSNAP: Returns (UINT4 *) aligned memory because of call to
   Merge_diagonals_uint4.  Need to free with FREE_ALIGN */
/* Output for GSNAPL: Returns (UINT8 *) non-aligned memory.  Free with FREE */
#if !defined(UTILITYP)
static int
read_with_bounds_doubles (Univcoord_T *stream, T this, Localspace_T oligo0, Localspace_T oligo1,
			  Univcoord_T low, Univcoord_T high, int diagterm0, int diagterm1, int oligosep) {
  int nentries = 0;
  int low_regioni, high_regioni, regioni;
  UINT2 *region_offsets, *region_positions, *prev_region_positions;
  Univcoord_T *stream_orig, region_term;
  int ptr0, end0, ptr1, end1, end;	/* Have to use int and not UINT2, because values could be 65536 */
  int between_ptr0, between_end1;	/* Have to use int because we evaluate between_ptr0-1 */
#ifdef DEBUG0
  int i;
#endif


  debug0(printf("read_with_bounds_doubles: oligo0 = %06X, oligo1 = %06X, diagterms %d and %d\n",
		oligo0,oligo1,diagterm0,diagterm1));

  low_regioni = low/REGION_LENGTH;
  high_regioni = high/REGION_LENGTH;

  debug0(printf("low %llu => region %d.  high %llu => region %d\n",
                 low,low_regioni,high,high_regioni));

  if (high_regioni == low_regioni) {
    debug0(printf("Case 1: One region\n"));
    regioni = low_regioni;

    region_offsets = &(this->regiondb[regioni * this->region_size]);
    ptr0 = (oligo0 == 0) ? 0 : region_offsets[oligo0 - 1];
    if ((end0 = region_offsets[oligo0]) < ptr0) {
      end0 = REGION_LENGTH;
    }
    debug0(printf("offset pointers for oligo0 are %u and %u\n",ptr0,end0));

    ptr1 = (oligo1 == 0) ? 0 : region_offsets[oligo1 - 1];
    if ((end1 = region_offsets[oligo1]) < ptr1) {
      end1 = REGION_LENGTH;
    }
    debug0(printf("offset pointers for oligo1 are %u and %u\n",ptr1,end1));

    if (end0 == ptr0 || end1 == ptr1) {
      return 0;

    } else {
      region_positions = &(region_offsets[this->offsets_size]);
      region_term = regioni * REGION_LENGTH;
      /* printf("region_term %llu = regioni %d * REGION_LENGTH\n",region_term,regioni); */
      
      ptr0 = binary_search_with_term(ptr0,end0,region_positions,region_term,low);
      end = ptr0;
      while (end < end0 && region_positions[end] + region_term <= high) {
	end++;
      }
      end0 = end;

      ptr1 = binary_search_with_term(ptr1,end1,region_positions,region_term,low);
      end = ptr1;
      while (end < end1 && region_positions[end] + region_term <= high) {
	end++;
      }
      end1 = end;

      if (end0 == ptr0 || end1 == ptr1) {
	return 0;
      } else {
	return intersect_within_region(stream,&(region_positions[ptr0]),/*npositionsa*/end0-ptr0,diagterm0,
				       &(region_positions[ptr1]),/*npositionsa*/end1-ptr1,diagterm1,
				       region_term);
      }
    }

  } else {
    debug0(printf("Case 2: Multiple regions that must be merged\n"));
    stream_orig = stream;

    /* low region */
    regioni = low_regioni;
    region_offsets = &(this->regiondb[regioni * this->region_size]);
    region_positions = &(region_offsets[this->offsets_size]);
    region_term = regioni * REGION_LENGTH;
    /* printf("region_term %llu = regioni %d * REGION_LENGTH\n",region_term,regioni); */

    ptr0 = (oligo0 == 0) ? 0 : region_offsets[oligo0 - 1];
    if ((end0 = region_offsets[oligo0]) < ptr0) {
      end0 = REGION_LENGTH;
    }
    debug0(printf("offset pointers for oligo0 in low region are %u and %u\n",ptr0,end0));

    ptr1 = (oligo1 == 0) ? 0 : region_offsets[oligo1 - 1];
    if ((end1 = region_offsets[oligo1]) < ptr1) {
      end1 = REGION_LENGTH;
    }
    debug0(printf("offset pointers for oligo1 in low region are %u and %u\n",ptr1,end1));

    if (end0 == ptr0 || end1 == ptr1) {
      /* Skip stream */
    } else {
      ptr0 = binary_search_with_term(ptr0,end0,region_positions,region_term,low);
      /* For low region, do not alter end0 */

      ptr1 = binary_search_with_term(ptr1,end1,region_positions,region_term,low);
      /* For low region, do not alter end1 */

      if (end0 == ptr0 || end1 == ptr1) {
	/* Skip stream */
      } else {
	stream += intersect_within_region(stream,&(region_positions[ptr0]),/*npositionsa*/end0-ptr0,diagterm0,
					  &(region_positions[ptr1]),/*npositionsa*/end1-ptr1,diagterm1,
					  region_term);
      }
    }

    /* middle regions */
    for (regioni = low_regioni + 1; regioni < high_regioni; regioni++) {
      /* Between previous region and this one */
      prev_region_positions = region_positions;
      /* prev_region_term = region_term; */

      between_ptr0 = end0;
      while (between_ptr0 > ptr0 && prev_region_positions[between_ptr0-1] >= REGION_LENGTH - oligosep) {
	between_ptr0--;
      }

      region_offsets = &(this->regiondb[regioni * this->region_size]);
      region_positions = &(region_offsets[this->offsets_size]);
      region_term = regioni * REGION_LENGTH;
      /* printf("region_term %llu = regioni %d * REGION_LENGTH\n",region_term,regioni); */

      between_end1 = ptr1 = (oligo1 == 0) ? 0 : region_offsets[oligo1 - 1];
      if ((end1 = region_offsets[oligo1]) < ptr1) {
	end1 = REGION_LENGTH;
      }
      debug0(printf("offset pointers for oligo1 before middle region are %u and %u\n",ptr1,end1));

      while (between_end1 < end1 && region_positions[between_end1] < oligosep) {
	between_end1++;
      }
      
      /* Intersection of prev_region_offsets[between_ptr0..end0) and
	 region_offsets[ptr1..between_end1) */
#ifdef DEBUG8
      for (i = between_ptr0; i < end0; i++) {
	printf("Prev %d: %hu\n",i,prev_region_positions[i]);
      }
      for (i = ptr1; i < between_end1; i++) {
	printf("Next %d: %hu\n",i,region_positions[i]);
      }
#endif

      if (end0 == between_ptr0 || between_end1 == ptr1) {
	/* Skip intersection */
      } else {
	stream += intersect_between_regions(stream,&(prev_region_positions[between_ptr0]),	
					    end0-between_ptr0,diagterm0,
					    &(region_positions[ptr1]),between_end1-ptr1,diagterm1,
					    /*region_term_1*/region_term);
      }

      /* Handle this region */
      ptr0 = (oligo0 == 0) ? 0 : region_offsets[oligo0 - 1];
      if ((end0 = region_offsets[oligo0]) < ptr0) {
	end0 = REGION_LENGTH;
      }
      debug0(printf("offset pointers for oligo0 in middle region are %u and %u\n",ptr0,end0));

      ptr1 = (oligo1 == 0) ? 0 : region_offsets[oligo1 - 1];
      if ((end1 = region_offsets[oligo1]) < ptr1) {
	end1 = REGION_LENGTH;
      }
      debug0(printf("offset pointers for oligo1 in middle region are %u and %u\n",ptr1,end1));

      if (end0 == ptr0 || end1 == ptr1) {
	/* Skip stream */
      } else {
	stream += intersect_within_region(stream,&(region_positions[ptr0]),/*npositionsa*/end0-ptr0,diagterm0,
					  &(region_positions[ptr1]),/*npositionsa*/end1-ptr1,diagterm1,
					  region_term);
      }
    }


    /* Between previous region and high region */
    prev_region_positions = region_positions;
    /* prev_region_term = region_term; */

    between_ptr0 = end0;
    while (between_ptr0 > ptr0 && prev_region_positions[between_ptr0-1] >= REGION_LENGTH - oligosep) {
      between_ptr0--;
    }

    regioni = high_regioni;
    region_offsets = &(this->regiondb[regioni * this->region_size]);
    region_positions = &(region_offsets[this->offsets_size]);
    region_term = regioni * REGION_LENGTH;
    /* printf("region_term %llu = regioni %d * REGION_LENGTH\n",region_term,regioni); */

    between_end1 = ptr1 = (oligo1 == 0) ? 0 : region_offsets[oligo1 - 1];
    if ((end1 = region_offsets[oligo1]) < ptr1) {
      end1 = REGION_LENGTH;
    }
    debug0(printf("offset pointers for oligo1 before high region are %u and %u\n",ptr1,end1));

    while (between_end1 < end1 && region_positions[between_end1] < oligosep) {
      between_end1++;
    }
      
    /* Intersection of prev_region_offsets[between_ptr0..end0) and
       region_offsets[ptr1..between_end1) */
#ifdef DEBUG8
    for (i = between_ptr0; i < end0; i++) {
      printf("Prev %d: %hu\n",i,prev_region_positions[i]);
    }
    for (i = ptr1; i < between_end1; i++) {
      printf("Next %d: %hu\n",i,region_positions[i]);
    }
#endif

    if (end0 == between_ptr0 || between_end1 == ptr1) {
      /* Skip intersection */
    } else {
      stream += intersect_between_regions(stream,&(prev_region_positions[between_ptr0]),
					  end0-between_ptr0,diagterm0,
					  &(region_positions[ptr1]),between_end1-ptr1,diagterm1,
					  /*region_term_1*/region_term);
    }

    /* Handle high region */
    ptr0 = (oligo0 == 0) ? 0 : region_offsets[oligo0 - 1];
    if ((end0 = region_offsets[oligo0]) < ptr0) {
      end0 = REGION_LENGTH;
    }
    debug0(printf("offset pointers for oligo0 in high region are %u and %u\n",ptr0,end0));

    ptr1 = (oligo1 == 0) ? 0 : region_offsets[oligo1 - 1];
    if ((end1 = region_offsets[oligo1]) < ptr1) {
      end1 = REGION_LENGTH;
    }
    debug0(printf("offset pointers for oligo1 in high region are %u and %u\n",ptr1,end1));

    if (end0 == ptr0 || end1 == ptr1) {
      /* Skip stream */
    } else {
      /* For high region, do not alter ptr0 */
      end = ptr0;
      while (end < end0 && region_positions[end] + region_term <= high) {
	end++;
      }
      end0 = end;

      /* For high region, do not alter ptr1 */
      end = ptr1;
      while (end < end1 && region_positions[end] + region_term <= high) {
	end++;
      }
      end1 = end;

      if (end0 == ptr0 || end1 == ptr1) {
	/* Skip stream */
      } else {
	stream += intersect_within_region(stream,&(region_positions[ptr0]),/*npositionsa*/end0-ptr0,diagterm0,
					  &(region_positions[ptr1]),/*npositionsa*/end1-ptr1,diagterm1,
					  region_term);
      }
    }

    nentries = stream - stream_orig;

#ifdef DEBUG0
    printf("%d entries:",nentries);
    for (i = 0; i < nentries; i++) {
      printf(" %u",stream_orig[i]);
    }
    printf("\n");
#endif

    return nentries;
  }
}
#endif


#define VALUE_A 0
#define VALUE_C 1
#define VALUE_G 2
#define VALUE_T 3


/* Number of possible genestrands: 3.  Number of possible nucleotides: 4 */
static Localspace_T forward_conv[3][4];
static Localspace_T revcomp_conv[3][4];

#ifdef GSNAP
static int char_to_int[128] =
  { -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* 1..10 */
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* 11..20 */
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* 21..30 */
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* 31..40 */
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* 41..50 */
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, /* 51..60 */
    -1, -1, -1, -1,

  /* A,  B,  C,  D,  E,  F,  G,  H,  I,  J, */
     0, -1,  1, -1, -1, -1,  2, -1, -1, -1, /* 65..74 */

  /* K,  L,  M,  N,  O,  P,  Q,  R,  S,  T  */
    -1, -1, -1, -1, -1, -1, -1, -1, -1,  3, /* 75..84 */

  /* U,  V,  W,  X,  Y,  Z  */
    -1, -1, -1, -1, -1, -1,

    -1, -1, -1, -1, -1, -1,

  /* a,  b,  c,  d,  e,  f,  g,  h,  i,  j, */
     0, -1,  1, -1, -1, -1,  2, -1, -1, -1, /* 97..106 */

  /* k,  l,  m,  n,  o,  p,  q,  r,  s,  t  */
    -1, -1, -1, -1, -1, -1, -1, -1, -1,  3, /* 107..116 */

  /* u,  v,  w,  x,  y,  z  */
    -1, -1, -1, -1, -1, -1,

    -1, -1, -1, -1, -1};
#endif


#ifdef GSNAP
static Localspace_T
nt_oligo_plus (bool *validp, char *queryptr, int genestrand) {
  Localspace_T oligo = 0U;
  int i, d;
  char c;

  *validp = true;
  for (i = 0; i < region1part; i++) {
    oligo <<= 2;

    c = queryptr[i];
    if ((d = char_to_int[(int) c]) < 0) {
      *validp = false;
      return 0;
    } else {
      oligo += forward_conv[genestrand][d];
    }
  }

  return oligo;
}
#endif


#ifdef GSNAP
static Localspace_T
nt_oligo_minus (bool *validp, char *queryptr, int genestrand) {
  Localspace_T oligo = 0U;
  int i, d;
  char c;

  *validp = true;
  for (i = 0; i < region1part; i++) {
    oligo <<= 2;

    c = queryptr[i];
    if ((d = char_to_int[(int) c]) < 0) {
      *validp = false;
      return 0;
    } else {
      oligo += revcomp_conv[genestrand][d];
    }
  }

  return oligo;
}
#endif



#if 0

#define MAX_NEIGHBORS 0		/* was 3 */

/* Modified from kmer-search.c */
static Univdiag_T *
find_local_sets (int *nloci, List_T **left_diagonals, List_T **right_diagonals,
		 int subopt_count, Univcoord_T *values, int nvalues,
		 Compress_T query_compress, int pos5, int pos3, Genome_T genomebits,
		 int region1part, bool plusp) {
  Univdiag_T *middle_diagonals, *out;
  Univcoord_T *ptr, *end, *first, nearby_value, left, diagonal;
#if 0
  int trim5, trim3;
#else
  int querystart, queryend;
#endif
  int nmismatches5, nmismatches3;
  int *indices, *outi;
  int nneighbors;
  int i, k;

#ifdef TRIM_CHECK
  int trim5_old, trim3_old;
  int nmismatches5_old, nmismatches3_old;
#endif


  out = middle_diagonals = (Univdiag_T *) MALLOC(nvalues*sizeof(Univdiag_T));
  outi = indices = (int *) MALLOC(nvalues*sizeof(int));

  /* Get all values that have subopt_count or more */
  ptr = values;
  end = &(values[nvalues]);
  while (ptr < end) {
    first = ptr;
    debug0(printf("DIAGONAL %u",*first));
    if (ptr + subopt_count - 1 >= end) {
      /* End of list fails */
      debug0(printf(" => Goes beyond end of list\n"));
      ptr = end;

    } else if (ptr[subopt_count - 1] != *first) {
      /* Fails */
      debug0(printf(" => Fails\n"));
      if (ptr[subopt_count - 1] != ptr[subopt_count - 2]) {
	/* Can jump immediately */
	ptr += subopt_count - 1;
      } else {
	/* Advance forward until we see a new value */
	while (ptr < end && *ptr == *first) {
	  ptr++;
	}
      }

    } else {
      /* Contender */
      /* left = *first - querylength; */
      left = *first;
#if 0
      trim_ends(&trim5,&trim3,&nmismatches5,&nmismatches3,poly_p,query_compress,left,querylength,genomebits,plusp,15);
#else
      querystart = Genome_first_kmer_left(&nmismatches5,genomebits,query_compress,left,pos5,pos3,
					  plusp,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/region1part);
      queryend = Genome_first_kmer_right(&nmismatches3,genomebits,query_compress,left,pos5,pos3,
					 plusp,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/region1part);
#endif

      debug0(printf(" => MIDDLE DIAGONAL at left %u has query %d..%d\n",left,querystart,queryend));
      *out++ = Univdiag_new(querystart,queryend,/*univdiagonal*/left);
      *outi++ = ptr - values;	/* Save index so we can find local diagonals later */

      ptr += subopt_count;
      while (ptr < end && *ptr == *first) {
	ptr++;
      }
    }
  }

  *nloci = out - &(middle_diagonals[0]);
  *left_diagonals = (List_T *) CALLOC(*nloci,sizeof(List_T));
  *right_diagonals = (List_T *) CALLOC(*nloci,sizeof(List_T));

  /* Add nearby diagonals */
  /* TODO: Check for too many nearby diagonals, indicating a repetitive oligo */
  for (i = 0; i < *nloci; i++) {
    /* diagonal = middle_diagonals[i]->univdiagonal + querylength; */
    diagonal = middle_diagonals[i]->univdiagonal;

    k = indices[i];
    nneighbors = 0;
    while (nneighbors <= MAX_NEIGHBORS && k - 1 >= 0 && diagonal - values[k - 1] < 200000) {
      nearby_value = values[--k];
      debug0(printf("Potential before %u is %u\n",diagonal,nearby_value));

      /* left = nearby_value - querylength; */
      left = nearby_value;
#if 0
      trim_ends(&trim5,&trim3,&nmismatches5,&nmismatches3,poly_p,query_compress,left,querylength,genomebits,plusp,15);
#else
      querystart = Genome_first_kmer_left(&nmismatches5,genomebits,query_compress,left,pos5,pos3,
					  plusp,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/region1part);
      queryend = Genome_first_kmer_right(&nmismatches3,genomebits,query_compress,left,pos5,pos3,
					 plusp,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/region1part);
#endif

      debug0(printf("LEFT DIAGONAL at left %u has query %d..%d",left,querystart,queryend));
      if (querystart >= middle_diagonals[i]->querystart) {
	debug0(printf(" => skipping because querystart %d >= middle querystart %d",
		      querystart,middle_diagonals[i]->querystart));
      } else {
	(*left_diagonals)[i] =
	  List_push((*left_diagonals)[i],(void *) Univdiag_new(querystart,queryend,/*univdiagonal*/left));
	nneighbors += 1;
      }
      debug0(printf("\n"));
      while (k - 1 >= 0 && values[k - 1] == nearby_value) {
	k--;
      }
    }

    k = indices[i] + subopt_count - 1;
    while (k + 1 < nvalues && values[k + 1] == diagonal) {
      k++;
    }

    nneighbors = 0;
    while (nneighbors <= MAX_NEIGHBORS && k + 1 < nvalues && values[k + 1] - diagonal < 200000) {
      nearby_value = values[++k];
      debug0(printf("Potential after %u is %u\n",diagonal,nearby_value));
      
      /* left = nearby_value - querylength; */
      left = nearby_value;
#if 0
      trim_ends(&trim5,&trim3,&nmismatches5,&nmismatches3,poly_p,query_compress,left,querylength,genomebits,plusp,15);
#else
      querystart = Genome_first_kmer_left(&nmismatches5,genomebits,query_compress,left,pos5,pos3,
					  plusp,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/region1part);
      queryend = Genome_first_kmer_right(&nmismatches3,genomebits,query_compress,left,pos5,pos3,
					 plusp,genestrand,/*query_unk_mismatch_local_p*/false,/*kmer*/region1part);
#endif

      debug0(printf("RIGHT DIAGONAL at left %u has query %d..%d",left,querystart,queryend));
      if (queryend <= middle_diagonals[i]->queryend) {
	debug0(printf(" => skipping because queryend %d <= middle queryend %d",
		      queryend,middle_diagonals[i]->queryend));
      } else {
	(*right_diagonals)[i] =
	  List_push((*right_diagonals)[i],(void *) Univdiag_new(querystart,queryend,/*univdiagonal*/left));
	nneighbors += 1;
      }
      debug0(printf("\n"));
      while (k + 1 < nvalues && values[k + 1] == nearby_value) {
	k++;
      }
    }
  }

  FREE(indices);
  return middle_diagonals;
}

#endif


#if 0

/* Copied from kmer-search.c */
static int
most_prevalent_uint (int *nloci, Univcoord_T *values, int nvalues) {
  int max_count, count;
  Univcoord_T *out, *ptr, *end, *first;

  assert(nvalues > 0);

  ptr = out = &(values[0]);	/* Reset */
  end = &(values[nvalues]);

  max_count = 1;
  while (ptr < end) {
    first = ptr;
    debug0(printf("DIAGONAL %u",*first));
    if (ptr + max_count - 1 >= end) {
      /* End of list fails */
      debug0(printf(" => Goes beyond end of list\n"));
      ptr = end;

    } else if (ptr[max_count - 1] != *first) {
      /* Fails */
      debug0(printf(" => Fails\n"));
      if (ptr[max_count - 1] != ptr[max_count - 2]) {
	/* Can jump immediately */
	ptr += max_count - 1;
      } else {
	/* Advance forward until we see a new value */
	while (ptr < end && *ptr == *first) {
	  ptr++;
	}
      }

    } else {
      /* Contender */
      ptr += max_count;
      while (ptr < end && *ptr == *first) {
	ptr++;
      }
      debug0(printf(" => Count %ld, index %d => printing\n",ptr - first,first - values));
      if ((count = ptr - first) > max_count) {
	out = &(values[0]);	/* Reset */
	max_count = count;
      }
      *out++ = *first;
    }
  }

  *nloci = out - &(values[0]);
  return max_count;
}

#endif


#ifdef GSNAP
/* Copied from kmer-search.c */
static int
most_prevalent_count (Univcoord_T *values, int nvalues) {
  int max_count, count;
  Univcoord_T *ptr, *end, *first;

  assert(nvalues > 0);

  ptr = values;
  end = &(values[nvalues]);

  max_count = 1;
  while (ptr < end) {
    first = ptr;
    debug0(printf("DIAGONAL %llu\n",*first));
    if (ptr + max_count - 1 >= end) {
      /* End of list fails */
      debug0(printf(" => Goes beyond end of list\n"));
      ptr = end;

    } else if (ptr[max_count - 1] != *first) {
      /* Fails */
      debug0(printf(" => Fails\n"));
      if (ptr[max_count - 1] != ptr[max_count - 2]) {
	/* Can jump immediately */
	ptr += max_count - 1;
      } else {
	/* Advance forward until we see a new value */
	while (ptr < end && *ptr == *first) {
	  ptr++;
	}
      }

    } else {
      /* Contender */
      ptr += max_count;
      while (ptr < end && *ptr == *first) {
	ptr++;
      }
      debug0(printf(" => Count %ld\n",ptr - first));
      if ((count = ptr - first) > max_count) {
	max_count = count;
      }
    }
  }

  return max_count;
}


/* Modified from kmer-search.c */
/* Overwrites values */
static void
find_prevalent (int *nloci, Univcoord_T *values, int nvalues, int subopt_count) {
  Univcoord_T *out, *ptr, *end, *first;
  /* int querystart, queryend; */
  /* int nmismatches5, nmismatches3; */

  out = &(values[0]);		/* overwrite */

  /* Get all values that have subopt_count or more */
  ptr = values;
  end = &(values[nvalues]);
  while (ptr < end) {
    first = ptr;
    debug0(printf("DIAGONAL %llu\n",*first));
    if (ptr + subopt_count - 1 >= end) {
      /* End of list fails */
      debug0(printf(" => Goes beyond end of list\n"));
      ptr = end;

    } else if (ptr[subopt_count - 1] != *first) {
      /* Fails */
      debug0(printf(" => Fails\n"));
      if (ptr[subopt_count - 1] != ptr[subopt_count - 2]) {
	/* Can jump immediately */
	ptr += subopt_count - 1;
      } else {
	/* Advance forward until we see a new value */
	while (ptr < end && *ptr == *first) {
	  ptr++;
	}
      }

    } else {
      /* Contender */
      *out++ = *first;
      
      ptr += subopt_count;
      while (ptr < end && *ptr == *first) {
	ptr++;
      }
    }
  }

  *nloci = out - &(values[0]);
  return;
}


#define SUBOPT 2


/* Output for GSNAP: Returns (UINT4 *) aligned memory because of call
   to Merge_diagonals_uint4.  Internal streams are also aligned from
   call to read_with_bounds */
/* Output for GSNAPL: Returns (UINT8 *) non-aligned memory.  Internal
   streams are also non-aligned from call to read_with_bounds */

#if !defined(UTILITYP)
/* queryptr is either queryuc_ptr (plus) or queryrc (minus) */
Univcoord_T *
Regiondb_get_univdiagonals (int *nentries, T this, int oligolen,
			    char *queryptr, int pos5, int pos3, int querylength,
			    Univcoord_T low_position, Univcoord_T high_position,
			    bool plusp, int genestrand,
			    Univcoord_T **stream_alloc, int *streamsize_alloc,
			    bool remove_repetitive_p) {
  Univcoord_T *diagonals, *stream, *p, *q, last_diagonal;
  int low_regioni, high_regioni, nregions;
  Localspace_T oligo0, oligo1;
  bool validp_0, validp_1;
  int querypos, oligosep;
  int ndiagonals, max_diagonals, total_n = 0, n;
  int max_count, subopt_count;
  int streami = 0;
#if defined(DEBUG0) || defined(DEBUG9)
  int i;
#endif
  

  debug0(printf("Entered Regiondb_get_diagonals with queryptr %s\n",queryptr));
  debug0(printf("  querypos %d..%d, low %llu, and high %llu\n",
		pos5,pos3,low_position,high_position));

  if (low_position >= high_position) {
    /* Have to check for this because a substring that is at the
       beginning or end of a chromosome may attempt to find a local
       piece that is beyond the chromosome, indicated by low == high */
    debug0(printf("  exiting because low %llu >= high %llu\n",
		  low_position,high_position));
    *nentries = 0;
    return (Univcoord_T *) NULL;

  } else {
    low_regioni = low_position/REGION_LENGTH;
    high_regioni = high_position/REGION_LENGTH;
    nregions = (high_regioni - low_regioni + 1);

    /* At most, each region can contain 65536 positions */
    max_diagonals = nregions * REGION_LENGTH + /* seams between regions*/(nregions - 1)*2;
    oligosep = oligolen - region1part;
  }

  if (plusp) {
    for (querypos = pos5; querypos <= pos3 - oligolen; querypos++) {
      oligo0 = nt_oligo_plus(&validp_0,&(queryptr[querypos]),genestrand);
      oligo1 = nt_oligo_plus(&validp_1,&(queryptr[querypos+oligosep]),genestrand);
      debug0(printf("Oligos: %s (valid %d) and %s (valid %d)\n",
		    shortoligo_nt(oligo0,this->region1part),validp_0,
		    shortoligo_nt(oligo1,this->region1part),validp_1));

      if (validp_0 == true && validp_1 == true) {
#ifndef LARGE_GENOMES
	stream = (Univcoord_T *) MALLOC_ALIGN(max_diagonals*sizeof(Univcoord_T));
	debug_align(printf("Allocating %p to %p -- Malloc of 1 bytes requested from %s:%d\n",
			   stream,stream,__FILE__,__LINE__));
#elif defined(HAVE_AVX512) || defined(HAVE_AVX2)
	stream = (Univcoord_T *) MALLOC_ALIGN(max_diagonals*sizeof(Univcoord_T));
#else
	stream = (Univcoord_T *) MALLOC(max_diagonals*sizeof(Univcoord_T));
#endif
	
	if ((n = read_with_bounds_doubles(stream,this,oligo0,oligo1,low_position,high_position,
					  /*diagterm0*/querylength - querypos,
					  /*diagterm1*/querylength - (querypos+oligosep),
					  oligosep)) == 0) {
	  /* Skip stream */
	  debug9(printf("At querypos %d, 0 entries:\n",querypos));

#ifndef LARGE_GENOMES
	  FREE_ALIGN(stream);
#elif defined(HAVE_AVX512) || defined(HAVE_AVX2)
	  FREE_ALIGN(stream);
#else
	  FREE(stream);
#endif
	  
	} else {
#ifdef DEBUG9
	  printf("At querypos %d, %d entries:",querypos,n);
	  for (i = 0; i < n; i++) {
	    printf(" %llu",stream[i]);
   	  }
	  printf("\n");
#endif

	  stream_alloc[streami] = stream;
	  streamsize_alloc[streami] = n;
	  total_n += n;
	  streami++;
	}
      }
    }

  } else {
    for (querypos = pos5; querypos <= pos3 - oligolen; querypos++) {
      oligo0 = nt_oligo_minus(&validp_0,&(queryptr[querypos]),genestrand);
      oligo1 = nt_oligo_minus(&validp_1,&(queryptr[querypos+oligosep]),genestrand);
      debug0(printf("Oligos: %s (valid %d) and %s (valid %d)\n",
		    shortoligo_nt(oligo0,this->region1part),validp_0,
		    shortoligo_nt(oligo1,this->region1part),validp_1));

      if (validp_0 == true && validp_1 == true) {
#ifndef LARGE_GENOMES
	stream = (Univcoord_T *) MALLOC_ALIGN(max_diagonals*sizeof(Univcoord_T));
	debug_align(printf("Allocating %p to %p -- Malloc of 1 bytes requested from %s:%d\n",
			   stream,stream,__FILE__,__LINE__));
#elif defined(HAVE_AVX512) || defined(HAVE_AVX2)
	stream = (Univcoord_T *) MALLOC_ALIGN(max_diagonals*sizeof(Univcoord_T));
#else
	stream = (Univcoord_T *) MALLOC(max_diagonals*sizeof(Univcoord_T));
#endif

	if ((n = read_with_bounds_doubles(stream,this,oligo0,oligo1,low_position,high_position,
					  /*diagterm0*/querylength - querypos,
					  /*diagterm1*/querylength - (querypos+oligosep),
					  oligosep)) == 0) {
	  /* Skip stream */
	  debug9(printf("At querypos %d, 0 entries:\n",querypos));

#ifndef LARGE_GENOMES
	  FREE_ALIGN(stream);
#elif defined(HAVE_AVX512) || defined(HAVE_AVX2)
	  FREE_ALIGN(stream);
#else
	  FREE(stream);
#endif

	} else {
#ifdef DEBUG9
	  printf("At querypos %d, %d entries:",querypos,n);
	  for (i = 0; i < n; i++) {
	    printf(" %llu",stream[i]);
	  }
	  printf("\n");
#endif
	  stream_alloc[streami] = stream;
	  streamsize_alloc[streami] = n;
	  total_n += n;
	  streami++;
	}
      }
    }
  }

  /* Merge streams */
  if (streami == 0) {
    *nentries = 0;
    return (Univcoord_T *) NULL;

  } else {
#ifdef LARGE_GENOMES
    diagonals = Merge_diagonals_uint8(&ndiagonals,stream_alloc,streamsize_alloc,/*nstreams*/streami);
#else
    diagonals = Merge_diagonals_uint4(&ndiagonals,stream_alloc,streamsize_alloc,/*nstreams*/streami);
#endif
    while (--streami >= 0) {
      stream = stream_alloc[streami];
#ifndef LARGE_GENOMES
      FREE_ALIGN(stream);	/* Aligned */
#elif defined(HAVE_AVX512) || defined(HAVE_AVX2)
      FREE_ALIGN(stream);	/* Aligned */
#else
      FREE(stream);
#endif
    }

#if 0
    /* max_count = */ most_prevalent_uint(&(*nentries),diagonals,ndiagonals);
#else
    max_count = most_prevalent_count(diagonals,ndiagonals);
    if ((subopt_count = max_count - SUBOPT) < 1) {
      subopt_count = 1;
    }
    find_prevalent(&(*nentries),diagonals,ndiagonals,subopt_count);
#endif

    if (remove_repetitive_p == true) {
#ifdef DEBUG0
      printf("\n");
      printf("%d entries before removing repetitive:",*nentries);
      for (i = 0; i < *nentries; i++) {
	printf(" %llu",diagonals[i]);
      }
      printf("\n");
#endif

      /* Skip repetitive diagonals */
      q = diagonals;
      last_diagonal = *q;
      for (p = &(diagonals[1]); p < &(diagonals[*nentries]); p++) {
	assert(*p > last_diagonal);
	if (*p == last_diagonal + 1) {
	  /* Skip: repetitive */
	} else {
	  *++q = *p;
	}
	last_diagonal = *p;
      }
      *nentries = (q - diagonals) + 1;
    }
    
#ifdef DEBUG0
    printf("\n");
    printf("%d entries:",*nentries);
    for (i = 0; i < *nentries; i++) {
      printf(" %llu",diagonals[i]);
    }
    printf("\n");
#endif

    return diagonals;
  }
}
#endif
#endif


#ifdef PMAP
#define BASE_KMER_SAMPLING 3   /* e.g., 677 */
#define KMER_SAMPLING 2   /* e.g., 77 */
#else
#define BASE_KMER_SAMPLING 5   /* e.g., 12153 */
#define KMER_SAMPLING 3   /* e.g., 153 */
#endif


static Regiondb_filenames_T
Regiondb_filenames_new (char *regiondb_filename, char *regiondb_basename_ptr,
			char *regiondb_region1info_ptr) {
  Regiondb_filenames_T new = (Regiondb_filenames_T) MALLOC(sizeof(*new));

  new->regiondb_filename = regiondb_filename;
  new->regiondb_basename_ptr = regiondb_basename_ptr;
  new->regiondb_region1info_ptr = regiondb_region1info_ptr;

  return new;
}

void
Regiondb_filenames_free (Regiondb_filenames_T *old) {

  FREE((*old)->regiondb_filename);

  FREE(*old);

  return;
}


Regiondb_filenames_T
Regiondb_get_filenames (
#ifdef PMAP
		       Alphabet_T *alphabet, Alphabet_T required_alphabet,
#endif
		       Width_T *region1part, Width_T *region1interval,
		       char *genomesubdir, char *fileroot, char *idx_filesuffix, char *snps_root,
		       Width_T required_region1part, Width_T required_interval) {
  char *regiondb_filename, *regiondb_basename_ptr, *regiondb_region1info_ptr;

  char *base_filename, *filename;
#ifdef PMAP
  char *pattern1, *pattern2, *a;
  int patternlength1, patternlength2, alphabet_strlen;
  Alphabet_T found_alphabet;
#else
  char *pattern;
  char tens;
#endif
  char interval_char, digit_string[2], *p, *q;
  Width_T found_region1part = 0, found_interval = 0;
  int patternlength;

  char ones;
  char *regiondb_suffix;
  struct dirent *entry;
  DIR *dp;


  if (snps_root == NULL) {
    regiondb_suffix = "regiondb";
  } else {
    regiondb_suffix = (char *) CALLOC(strlen("regiondb.")+strlen(snps_root)+1,sizeof(char));
    sprintf(regiondb_suffix,"regiondb.%s",snps_root);
  }


#ifdef PMAP
  *alphabet = NALPHABETS + 1;
#endif
  *region1part = 0;
  *region1interval = 1000;
  base_filename = (char *) NULL;

  if ((dp = opendir(genomesubdir)) == NULL) {
    fprintf(stderr,"Unable to open directory %s\n",genomesubdir);
    exit(9);
  }

#ifdef PMAP
  pattern1 = (char *) CALLOC(strlen(fileroot)+strlen(".")+1,sizeof(char)); /* e.g., "hg19." */
  sprintf(pattern1,"%s.",fileroot);
  patternlength1 = strlen(pattern1);

  pattern2 = (char *) CALLOC(strlen(".")+strlen(idx_filesuffix)+1,sizeof(char)); /* e.g., ".pr" */
  sprintf(pattern2,".%s",idx_filesuffix);
  patternlength2 = strlen(pattern2);

  digit_string[1] = '\0';	/* Needed for atoi */
  while ((entry = readdir(dp)) != NULL) {
    filename = entry->d_name;
    if (!strncmp(filename,pattern1,patternlength1)) {
      a = &(filename[strlen(pattern1)]); /* Points after fileroot, e.g., "hg19." */
      if ((p = strstr(a,pattern2)) != NULL && (q = strstr(p,locoffsetsstrm_suffix)) != NULL && !strcmp(q,locoffsetsstrm_suffix)) {
	if ((found_alphabet = Alphabet_find(a)) != AA0) {
	  alphabet_strlen = p - a;
	  p += patternlength2;

	  if (q - p == KMER_SAMPLING) {
	    /* Latest style, e.g., pf77 */
	    if (sscanf(p,"%c%c",&ones,&interval_char) == 2) {
	      digit_string[0] = ones;
	      found_region1part = atoi(digit_string);

	      digit_string[0] = interval_char;
	      found_interval = atoi(digit_string);
	    } else {
	      abort();
	    }

#if 0
	  } else if (q - p == BASE_KMER_SAMPLING) {
	    /* Previous style, e.g., pf677.  No longer supported. */
	    if (sscanf(p,"%c%c%c",&ones0,&ones,&interval_char) == 3) {
	      digit_string[0] = ones0;
	      found_basesize = atoi(digit_string);

	      digit_string[0] = ones;
	      found_region1part = atoi(digit_string);

	      digit_string[0] = interval_char;
	      found_interval = atoi(digit_string);
	    } else {
	      abort();
	    }
#endif

	  } else {
	    /* fprintf(stderr,"Cannot parse part between %s and offsets in filename %s\n",idx_filesuffix,filename); */
	    if (snps_root != NULL) {
	      FREE(regiondb_suffix);
	    }
	    return (Filenames_T) NULL;
	  }

	  if ((required_alphabet == AA0 || found_alphabet == required_alphabet) &&
	      (required_region1part == 0 || found_region1part == required_region1part) &&
	      (required_interval == 0 || found_interval == required_interval)) {
	    if (required_alphabet == AA0 && found_alphabet > *alphabet) {
	      /* Skip, since we have already found an earlier alphabet */
	    } else if (required_region1part == 0 && found_region1part < *region1part) {
	      /* Skip, since we have already found a larger region1part */
	    } else if (required_interval == 0 && found_interval > *region1interval) {
	      /* Skip, since we have already found a smaller interval */
	    } else {
	      patternlength = patternlength1 + alphabet_strlen + patternlength2;
	      /* *basesize = found_basesize; */
	      *region1part = found_region1part;
	      *region1interval = found_interval;
	      *alphabet = found_alphabet;
	      FREE(base_filename);
	      base_filename = (char *) CALLOC(strlen(filename)+1,sizeof(char));
	      strcpy(base_filename,filename);
	    }
	  }
	}
      }
    }
  }

  FREE(pattern2);
  FREE(pattern1);

#else

  pattern = (char *) CALLOC(strlen(fileroot)+strlen(".")+strlen(idx_filesuffix)+1,sizeof(char));
  sprintf(pattern,"%s.%s",fileroot,idx_filesuffix);
  patternlength = strlen(pattern);

  digit_string[1] = '\0';	/* Needed for atoi */
  while ((entry = readdir(dp)) != NULL) {
    filename = entry->d_name;
    if (!strncmp(filename,pattern,patternlength)) {
      p = &(filename[strlen(pattern)]); /* Points after idx_filesuffix, e.g., "ref" */
      if ((q = strstr(p,regiondb_suffix)) != NULL && !strcmp(q,regiondb_suffix)) {

	if (q - p == KMER_SAMPLING) {
	  /* New style, e.g., ref153 */
	  if (sscanf(p,"%c%c%c",&tens,&ones,&interval_char) == 3) {
	    digit_string[0] = tens;
	    found_region1part = 10*atoi(digit_string);
	    digit_string[0] = ones;
	    found_region1part += atoi(digit_string);

	    digit_string[0] = interval_char;
	    found_interval = atoi(digit_string);
	  } else {
	    abort();
	  }

	} else {
	  fprintf(stderr,"Cannot parse part between %s and offsets in filename %s: found %ld characters, expecting %d\n",
		  idx_filesuffix,filename,q-p,BASE_KMER_SAMPLING);
	  abort();
	  if (snps_root != NULL) {
	    FREE(regiondb_suffix);
	  }
	  return (Regiondb_filenames_T) NULL;
	}

	if ((required_region1part == 0 || found_region1part == required_region1part) &&
	    (required_interval == 0 || found_interval == required_interval)) {
	  if (required_region1part == 0 && found_region1part < *region1part) {
	    /* Skip, since we have already found a larger region1part */
	  } else if (required_interval == 0 && found_interval > *region1interval) {
	    /* Skip, since we have already found a smaller interval */
	  } else {
	    *region1part = found_region1part;
	    *region1interval = found_interval;
	    FREE(base_filename);
	    base_filename = (char *) CALLOC(strlen(filename)+1,sizeof(char));
	    strcpy(base_filename,filename);
	  }
	}
      }
    }
  }

  FREE(pattern);
#endif


  if (closedir(dp) < 0) {
    fprintf(stderr,"Unable to close directory %s\n",genomesubdir);
  }

  /* Construct full filenames */
  if (base_filename == NULL) {
    /* offsetspages_filename = (char *) NULL; */
    /* offsetsmeta_filename = (char *) NULL; */
    /* offsetsstrm_filename = (char *) NULL; */
    /* positions_high_filename = (char *) NULL; */
    /* positions_low_filename = (char *) NULL; */
    if (snps_root != NULL) {
      FREE(regiondb_suffix);
    }
    return (Regiondb_filenames_T) NULL;

  } else {
    regiondb_filename = (char *) CALLOC(strlen(genomesubdir)+strlen("/")+strlen(base_filename)+1,sizeof(char));
    regiondb_basename_ptr = &(regiondb_filename[strlen(genomesubdir)+strlen("/")]);
    regiondb_region1info_ptr = &(regiondb_basename_ptr[patternlength]);

    sprintf(regiondb_filename,"%s/%s",genomesubdir,base_filename);
    if (Access_file_exists_p(regiondb_filename) == false) {
      fprintf(stderr,"Regiondb filename %s does not exist\n",regiondb_filename);
      FREE(regiondb_filename);
      FREE(base_filename);
      if (snps_root != NULL) {
	FREE(regiondb_suffix);
      }
      return (Regiondb_filenames_T) NULL;
    }

    FREE(base_filename);

    fprintf(stderr,"Looking for region files in directory %s\n",genomesubdir);
    fprintf(stderr,"  Regiondb file is %s\n",regiondb_basename_ptr);
    return Regiondb_filenames_new(regiondb_filename,regiondb_basename_ptr,regiondb_region1info_ptr);
  }
}



static Oligospace_T
power (int base, Width_T exponent) {
#ifdef OLIGOSPACE_NOT_LONG
  Oligospace_T result = 1U;
#else
  Oligospace_T result = 1UL;
#endif
  int i;

  for (i = 0; i < exponent; i++) {
    result *= base;
  }
  return result;
}


T
Regiondb_new_genome (Width_T *region1part, Width_T *region1interval,
		     char *genomesubdir, char *fileroot, char *idx_filesuffix, char *snps_root,
#ifdef PMAP
		     Alphabet_T *alphabet, int *alphabet_size, Alphabet_T required_alphabet,
#endif
		     Width_T required_region1part, Width_T required_interval,
		     Access_mode_T regiondb_access, bool sharedp,
		     bool multiple_sequences_p, bool preload_shared_memory_p, bool unload_shared_memory_p) {
  T new = (T) MALLOC(sizeof(*new));

  Regiondb_filenames_T filenames;
  char *comma;
  double seconds;

  if ((filenames = Regiondb_get_filenames(
#ifdef PMAP
					 &(*alphabet),required_alphabet,
#endif
					 &new->region1part,&new->region1interval,
					 genomesubdir,fileroot,idx_filesuffix,snps_root,
					 required_region1part,required_interval)) == NULL) {
    /* For backward compatibility and large genomes, allow alignment without regiondb */
    /* fprintf(stderr,"Cannot find genomic index files in either current or old format.  Looking for files containing %s\n",idx_filesuffix); */
    FREE(new);
    return (T) NULL;

  }

  *region1part = new->region1part;
  *region1interval = new->region1interval;

#ifdef PMAP
  *alphabet_size = Alphabet_get_size(*alphabet);
#endif
  new->offsets_size = power(4,*region1part);
  new->region_size = new->offsets_size + REGION_LENGTH;

  if (regiondb_access == USE_ALLOCATE) {
    if (snps_root) {
      fprintf(stderr,"Allocating memory for %s (%s) regiondb, kmer %d, interval %d...",
	      idx_filesuffix,snps_root,new->region1part,new->region1interval);
    } else {
      fprintf(stderr,"Allocating memory for %s regiondb, kmer %d, interval %d...",
	      idx_filesuffix,new->region1part,new->region1interval);
    }

    if (
#ifndef HAVE_MMAP
	0 &&
#endif
	multiple_sequences_p == false && preload_shared_memory_p == false && unload_shared_memory_p == false) {
#ifdef HAVE_MMAP
      new->regiondb = (UINT2 *) Access_mmap(&new->regiondb_fd,&new->regiondb_len,&seconds,
					    filenames->regiondb_filename,/*randomp*/false);
      new->regiondb_access = MMAPPED;
#endif
    } else if (sharedp == true) {
      new->regiondb = (UINT2 *) Access_allocate_shared(&new->regiondb_access,&new->regiondb_shmid,&new->regiondb_key,
						       &new->regiondb_fd,&new->regiondb_len,&seconds,
						       filenames->regiondb_filename,sizeof(UINT4));
    } else {
      new->regiondb = (UINT2 *) Access_allocate_private(&new->regiondb_access,&new->regiondb_len,&seconds,
							filenames->regiondb_filename,sizeof(UINT4));
    }

    if (new->regiondb == NULL) {
      fprintf(stderr,"insufficient memory (need to use a lower batch mode (-B)\n");
      exit(9);
    } else {
      comma = Genomicpos_commafmt(new->regiondb_len);
      fprintf(stderr,"done (%s bytes",comma);
      FREE(comma);
      if (multiple_sequences_p == true) {
	fprintf(stderr,", %.2f sec",seconds);
      }
      fprintf(stderr,")\n");
    }

#ifdef HAVE_MMAP
  } else if (regiondb_access == USE_MMAP_PRELOAD || regiondb_access == USE_MMAP_ONLY) {
    new->regiondb = (UINT2 *) Access_mmap(&new->regiondb_fd,&new->regiondb_len,&seconds,
					  filenames->regiondb_filename,/*randomp*/true);

    if (new->regiondb == NULL) {
      fprintf(stderr,"Insufficient memory for mmap of %s (will use disk file instead, but program will be slow)\n",
	      filenames->regiondb_filename);
      new->regiondb_access = FILEIO;
    } else {
      new->regiondb_access = MMAPPED;
    }
#endif

  } else {
    fprintf(stderr,"Don't recognize regiondb_access %d\n",regiondb_access);
    abort();
  }

  Regiondb_filenames_free(&filenames);

  return new;
}


void
Regiondb_free (T *old) {
  if (*old) {

    if ((*old)->regiondb_access == ALLOCATED_PRIVATE) {
      FREE((*old)->regiondb);

    } else if ((*old)->regiondb_access == ALLOCATED_SHARED) {
      Access_deallocate((*old)->regiondb,(*old)->regiondb_shmid,(*old)->regiondb_key);

#ifdef HAVE_MMAP
    } else if ((*old)->regiondb_access == MMAPPED) {
      munmap((void *) (*old)->regiondb,(*old)->regiondb_len);
      close((*old)->regiondb_fd);
#endif
    }
      
    FREE(*old);
  }
  return;
}


/* Modified from Oligo_setup */
void
Regiondb_setup (int region1part_in, int mode) {
  region1part = region1part_in;

  if (mode == STANDARD) {
    memcpy(forward_conv[+0],((Localspace_T[]){/*   */VALUE_A, /*   */VALUE_C, /*   */VALUE_G, /*   */VALUE_T}),4*sizeof(Localspace_T));
    memcpy(revcomp_conv[+0],((Localspace_T[]){/*   */VALUE_A, /*   */VALUE_C, /*   */VALUE_G, /*   */VALUE_T}),4*sizeof(Localspace_T));

  } else if (mode == CMET_STRANDED) {
    memcpy(forward_conv[+0],((Localspace_T[]){/*   */VALUE_A, /*C>T*/VALUE_T, /*   */VALUE_G, /*   */VALUE_T}),4*sizeof(Localspace_T));
    memcpy(revcomp_conv[+0],((Localspace_T[]){/*   */VALUE_A, /*   */VALUE_C, /*G>A*/VALUE_A, /*   */VALUE_T}),4*sizeof(Localspace_T));

  } else if (mode == CMET_NONSTRANDED) {
    /* genestrand: 0 or +1 */
    /* Queryuc_ptr from gplus strand: (C==T).  Matches genomicfwd kmer */
    /* Queryrc from gminus strand: (G==A),  Then revcomp to match genomic genomicfwd kmer: (C==T) */

    /* genestrand: +2 (pcr step) */
    /* Queryuc_ptr from gminus strand: (C==T). Then pcr: (G==A).  Matches genomicfwd kmer */
    /* Queryrc from gplus strand: (G==A).  Then pcr: (C==T).  Then revcomp to match genomicfwd kmer: (G==A) */

    memcpy(forward_conv[+1],((Localspace_T[]){/*   */VALUE_A, /*C>T*/VALUE_T, /*   */VALUE_G, /*   */VALUE_T}),4*sizeof(Localspace_T));
    memcpy(forward_conv[+2],((Localspace_T[]){/*   */VALUE_A, /*   */VALUE_C, /*G>A*/VALUE_A, /*   */VALUE_T}),4*sizeof(Localspace_T));

    memcpy(revcomp_conv[+1],((Localspace_T[]){/*   */VALUE_A, /*   */VALUE_C, /*G>A*/VALUE_A, /*   */VALUE_T}),4*sizeof(Localspace_T));
    memcpy(revcomp_conv[+2],((Localspace_T[]){/*   */VALUE_A, /*C>T*/VALUE_T, /*   */VALUE_G, /*   */VALUE_T}),4*sizeof(Localspace_T));

  } else if (mode == ATOI_STRANDED) {
    memcpy(forward_conv[+0],((Localspace_T[]){/*A>G*/VALUE_G, /*   */VALUE_C, /*   */VALUE_G, /*   */VALUE_T}),4*sizeof(Localspace_T));
    memcpy(revcomp_conv[+0],((Localspace_T[]){/*   */VALUE_A, /*   */VALUE_C, /*   */VALUE_G, /*T>C*/VALUE_C}),4*sizeof(Localspace_T));

  } else if (mode == ATOI_NONSTRANDED) {
    memcpy(forward_conv[+1],((Localspace_T[]){/*A>G*/VALUE_G, /*   */VALUE_C, /*   */VALUE_G, /*   */VALUE_T}),4*sizeof(Localspace_T));
    memcpy(forward_conv[+2],((Localspace_T[]){/*   */VALUE_A, /*   */VALUE_C, /*   */VALUE_G, /*T>C*/VALUE_C}),4*sizeof(Localspace_T));

    memcpy(revcomp_conv[+1],((Localspace_T[]){/*   */VALUE_A, /*   */VALUE_C, /*   */VALUE_G, /*T>C*/VALUE_C}),4*sizeof(Localspace_T));
    memcpy(revcomp_conv[+2],((Localspace_T[]){/*A>G*/VALUE_G, /*   */VALUE_C, /*   */VALUE_G, /*   */VALUE_T}),4*sizeof(Localspace_T));

  } else if (mode == TTOC_STRANDED) {
    memcpy(forward_conv[+0],((Localspace_T[]){/*   */VALUE_A, /*   */VALUE_C, /*   */VALUE_G, /*T>C*/VALUE_C}),4*sizeof(Localspace_T));
    memcpy(revcomp_conv[+0],((Localspace_T[]){/*A>G*/VALUE_G, /*   */VALUE_C, /*   */VALUE_G, /*   */VALUE_T}),4*sizeof(Localspace_T));

  } else if (mode == TTOC_NONSTRANDED) {
    memcpy(forward_conv[+1],((Localspace_T[]){/*   */VALUE_A, /*   */VALUE_C, /*   */VALUE_G, /*T>C*/VALUE_C}),4*sizeof(Localspace_T));
    memcpy(forward_conv[+2],((Localspace_T[]){/*A>G*/VALUE_G, /*   */VALUE_C, /*   */VALUE_G, /*   */VALUE_T}),4*sizeof(Localspace_T));

    memcpy(revcomp_conv[+1],((Localspace_T[]){/*A>G*/VALUE_G, /*   */VALUE_C, /*   */VALUE_G, /*   */VALUE_T}),4*sizeof(Localspace_T));
    memcpy(revcomp_conv[+2],((Localspace_T[]){/*   */VALUE_A, /*   */VALUE_C, /*   */VALUE_G, /*T>C*/VALUE_C}),4*sizeof(Localspace_T));

  } else {
    fprintf(stderr,"Unexpected mode %d\n",mode);
    exit(9);
  }
}

