/***********************************************************************************

    Copyright (C) 2007-2024 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include "diarydata.hpp"
#include "diary.hpp"
#include "tableelem.hpp"
#include "../lifeograph.hpp"
#include "../strings.hpp"


namespace LIFEO
{

// CONSTANTS =======================================================================================
REG_PROP_GETTERS( VT::SO::PROP_0 )
REG_PROP_GETTERS( VT::SO::PROP_1 )

REG_PROP_GETTERS( VT::SAS::PROP_0 )
REG_PROP_GETTERS( VT::SAS::PROP_1 )
REG_PROP_GETTERS( VT::SAS::PROP_2 )

REG_PROP_GETTERS( VT::SUMF::PROP_0 )
REG_PROP_GETTERS( VT::SUMF::PROP_1 )
REG_PROP_GETTERS( VT::SUMF::PROP_2 )

REG_PROP_GETTERS( VT::ETS::PROP_0 )
REG_PROP_GETTERS( VT::ETS::PROP_1 )
REG_PROP_GETTERS( VT::ETS::PROP_2 )
REG_PROP_GETTERS( VT::ETS::PROP_3 )
REG_PROP_GETTERS( VT::ETS::PROP_4 )
REG_PROP_GETTERS( VT::ETS::PROP_5 )

REG_PROP_GETTERS( VT::CS::PROP_0 )
REG_PROP_GETTERS( VT::CS::PROP_1 )
REG_PROP_GETTERS( VT::CS::PROP_2 )

REG_PROP_GETTERS( VT::TCAo::PROP_0 )
REG_PROP_GETTERS( VT::TCAo::PROP_1 )
REG_PROP_GETTERS( VT::TCAo::PROP_2 )
REG_PROP_GETTERS( VT::TCAo::PROP_3 )
REG_PROP_GETTERS( VT::TCAo::PROP_4 )
REG_PROP_GETTERS( VT::TCAo::PROP_5 )
REG_PROP_GETTERS( VT::TCAo::PROP_6 )
REG_PROP_GETTERS( VT::TCAf::PROP_0 )
REG_PROP_GETTERS( VT::TCAf::PROP_1 )
REG_PROP_GETTERS( VT::TCAf::PROP_2 )
REG_PROP_GETTERS( VT::TCAf::PROP_3 )
REG_PROP_GETTERS( VT::TCAu::PROP_0 )
REG_PROP_GETTERS( VT::TCAu::PROP_1 )
REG_PROP_GETTERS( VT::TCAu::PROP_2 )
REG_PROP_GETTERS( VT::TCAu::PROP_3 )

// DIARYELEMENT ====================================================================================
// STATIC MEMBERS
const R2Pixbuf                      DiaryElement::s_pixbuf_null;
const Ustring                       DiaryElement::s_type_names[] =
{
    "", "Chapter Ctg (not used anymore)", _( "Theme" ), _( "Filter" ), _( "Chart" ), _( "Table" ),
    _( "Paragraph" ), _( "Diary" ), "Multiple Entries", _( "Entry" ), "Chapter (not used anymore)",
    "Header"
};

DiaryElement::DiaryElement() : NamedElement( "" ), m_status( ES::EXPANDED ) {}

DiaryElement::DiaryElement( Diary* const ptr2diary,
                            const Ustring& name,
                            ElemStatus status )
:   NamedElement( name ), m_p2diary( ptr2diary ),
    m_status( status ),
    m_id( ptr2diary ? ptr2diary->create_new_id( this ) : DEID_UNSET )
{
}

DiaryElement::~DiaryElement()
{
    if( m_p2diary != nullptr )
        m_p2diary->erase_id( m_id );
}

// DIARYELEMENT DATA SOURCE (Entries and Paragraphs) ===============================================
void
DiaryElemDataSrc::set_todo_status( ElemStatus s )
{
    m_status -= ( m_status & ES::FILTER_TODO );
    m_status |= s;
}

SI
DiaryElemDataSrc::get_todo_status_si() const
{
    switch( get_todo_status() )
    {
        case ES::TODO:
            return SI::TODO;
        case ES::PROGRESSED:
            return SI::PROGRESSED;
        case ES::DONE:
            return SI::DONE;
        case ES::CANCELED:
            return SI::CANCELED;
        case ( ES::NOT_TODO | ES::TODO ):
        case ( ES::NOT_TODO | ES::PROGRESSED ):
        case ( ES::NOT_TODO | ES::DONE ):
        case ( ES::NOT_TODO | ES::CANCELED ):
            return SI::AUTO;
        case ES::NOT_TODO:
        default:    // should never be the case
            return SI::_VOID_;
    }
}

// COORDS ==========================================================================================
double
Coords::get_distance( const Coords& p1, const Coords& p2 )
{
#if __GNUC__ > 9
    const static double D{ 6371 * 2 } ;      // mean diameter of Earth in kilometers
    const static double to_rad{ HELPERS::PI/180 }; // degree to radian conversion multiplier
    const double φ1{ p1.latitude * to_rad };  // in radians
    const double φ2{ p2.latitude * to_rad };  // in radians
    const double Δφ{ φ2 - φ1 };
    const double Δλ{ ( p2.longitude - p1.longitude ) * to_rad };

    const double a = pow( sin( Δφ / 2 ), 2 ) + cos( φ1 ) * cos( φ2 ) * pow( sin( Δλ / 2 ), 2 );

    return( D * atan2( sqrt( a ), sqrt( 1 - a ) ) );
    // per Wikipedia article about haversine formula, asin( sqrt( a ) ) should also work
#else // Unicode identifier support was added in GCC 10!
    const static double D{ 6371 * 2 } ;
    const static double to_rad{ HELPERS::PI/180 };
    const double phi1{ p1.latitude * to_rad };
    const double phi2{ p2.latitude * to_rad };
    const double dp{ phi2 - phi1 };
    const double dl{ ( p2.longitude - p1.longitude ) * to_rad };

    const double a = pow( sin( dp / 2 ), 2 ) + cos( phi1 ) * cos( phi2 ) * pow( sin( dl / 2 ), 2 );

    return( D * atan2( sqrt( a ), sqrt( 1 - a ) ) );
#endif
}

// NAME AND VALUE ==================================================================================
NameAndValue
NameAndValue::parse( const Ustring& text )
{
    NameAndValue nav;
    char lf{ '=' }; // =, \, #, $(unit)
    int divider{ 0 };
    int trim_length{ 0 };
    int trim_length_unit{ 0 };
    bool negative{ false };
    Wchar c;

    for( Ustring::size_type i = 0; i < text.size(); i++ )
    {
        c = text.at( i );
        switch( c )
        {
            case '\\':
                if( lf == '#' || lf == '$' )
                {
                    nav.unit += c;
                    trim_length_unit = 0;
                    lf = '$';
                }
                else if( lf == '\\' )
                {
                    nav.name += c;
                    trim_length = 0;
                    lf = '=';
                }
                else // i.e. ( lf == '=' )
                    lf = '\\';
                break;
            case '=':
                if( nav.name.empty() || lf == '\\' )
                {
                    nav.name += c;
                    trim_length = 0;
                    lf = '=';
                }
                else if( lf == '#' || lf == '$' )
                {
                    nav.unit += c;
                    trim_length_unit = 0;
                    lf = '$';
                }
                else // i.e. ( lf == '=' )
                {
                    nav.status |= NameAndValue::HAS_EQUAL;
                    lf = '#';
                }
                break;
            case ' ':
            case '\t':
                // if( lf == '#' ) just ignore
                if( lf == '=' || lf == '\\' )
                {
                    if( !nav.name.empty() ) // else ignore
                    {
                        nav.name += c;
                        trim_length++;
                    }
                }
                else if( lf == '$' )
                {
                    nav.unit += c;
                    trim_length_unit++;
                }
                break;
            case ',':
            case '.':
                if( divider || lf == '$' ) // note that if divider, lf must be #
                {
                    nav.unit += c;
                    trim_length_unit = 0;
                    lf = '$';
                }
                else if( lf == '#' )
                    divider = 1;
                else
                {
                    nav.name += c;
                    trim_length = 0;
                    lf = '=';
                }
                break;
            case '-':
                if( negative || lf == '$' ) // note that if negative, lf must be #
                {
                    nav.unit += c;
                    trim_length_unit = 0;
                    lf = '$';
                }
                else if( lf == '#' )
                    negative = true;
                else
                {
                    nav.name += c;
                    trim_length = 0;
                    lf = '=';
                }
                break;
            case '0': case '1': case '2': case '3': case '4':
            case '5': case '6': case '7': case '8': case '9':
                if( lf == '#' )
                {
                    nav.status |= NameAndValue::HAS_VALUE;
                    nav.value *= 10;
                    nav.value += ( c - '0' );
                    if( divider )
                        divider *= 10;
                }
                else if( lf == '$' )
                {
                    nav.unit += c;
                    trim_length_unit = 0;
                }
                else
                {
                    nav.name += c;
                    trim_length = 0;
                    lf = '='; // reset ( lf == \ ) case
                }
                break;
            default:
                if( lf == '#' || lf == '$' )
                {
                    nav.unit += c;
                    trim_length_unit = 0;
                    lf = '$';
                }
                else
                {
                    nav.name += c;
                    trim_length = 0;
                    lf = '=';
                }
                break;
        }
    }

    if( lf == '$' )
        nav.status |= ( NameAndValue::HAS_NAME | NameAndValue::HAS_UNIT );
    else if( ! nav.name.empty() )
        nav.status |= NameAndValue::HAS_NAME;

    if( trim_length )
        nav.name.erase( nav.name.size() - trim_length, trim_length );
    if( trim_length_unit )
        nav.unit.erase( nav.unit.size() - trim_length_unit, trim_length_unit );

    if( lf == '=' && ! nav.name.empty() ) // implicit boolean tag
        nav.value = 1;
    else
    {
        if( divider > 1 )
            nav.value /= divider;
        if( negative )
            nav.value *= -1;
    }

    PRINT_DEBUG( "tag parsed | name: ", nav.name, "; value: ", nav.value, "; unit: ", nav.unit );

    return nav;
}

// THEMES ==========================================================================================
// STATIC MEMBERS
const Color Theme::s_color_match1( "#33FF33" );
const Color Theme::s_color_match2( "#009900" );
const Color Theme::s_color_link1( "#6666FF" );
const Color Theme::s_color_link2( "#000099" );
const Color Theme::s_color_broken1( "#FF3333" );
const Color Theme::s_color_broken2( "#990000" );

const Color Theme::s_color_todo( "#FF0000" );
const Color Theme::s_color_done( "#66BB00" );
const Color Theme::s_color_canceled( "#AA8855" );

Theme::Theme( Diary* const d, const Ustring& name, const Ustring& )
// last Ustring is a dummy to mimic StringDefElem constructors
:   DiaryElement( d, name )
{
}

Theme::Theme( Diary* const d,
              const Ustring& name,
              const Ustring& str_font,
              const std::string& str_base,
              const std::string& str_text,
              const std::string& str_heading,
              const std::string& str_subheading,
              const std::string& str_highlight )
:   DiaryElement( d, name ),
    font( str_font ), color_base( str_base ), color_text( str_text ),
    color_heading( str_heading ), color_subheading( str_subheading ),
    color_highlight( str_highlight )
{
    calculate_derived_colors();
}

Theme::Theme( Diary* const d, const Ustring& name, const Theme* theme )
:   DiaryElement( d, name ),
    font( theme->font ),
    color_base( theme->color_base ),
    color_text( theme->color_text ),
    color_heading( theme->color_heading ),
    color_subheading( theme->color_subheading ),
    color_highlight( theme->color_highlight )
{
    calculate_derived_colors();
}

void
Theme::copy_to( Theme* target ) const
{
    target->font = font;
    target->image_bg = image_bg;
    target->color_base = color_base;
    target->color_text = color_text;
    target->color_heading = color_heading;
    target->color_subheading = color_subheading;
    target->color_highlight = color_highlight;

    target->calculate_derived_colors();
}

void
Theme::calculate_derived_colors()
{
    color_subsubheading = midtone( color_text, color_subheading, 0.5 );
    color_inline_tag =    midtone( color_base, color_highlight, 0.2 );
    color_mid =           midtone( color_base, color_text, 0.65 );
    color_pale =          midtone( color_base, color_text, 0.3 );
    color_region_bg =     midtone( color_base, color_text, 0.1 );
    color_match_bg =      contrast2( color_base, s_color_match1, s_color_match2 );
    color_link =          contrast2( color_base, s_color_link1, s_color_link2 );
    color_link_broken =   contrast2( color_base, s_color_broken1, s_color_broken2 );

    // TODO: 3.2: we may change the coefficients below depending on the difference between the...
    // ... contrasting colors using get_color_diff( Theme::s_color_done, theme->color_base )...
    // ... generally, when get_color_diff is < 1.0 contrast is not satisfactory
    color_open =          midtone( s_color_todo, color_text );
    color_open_bg =       midtone( s_color_todo, color_base, 0.7 );

    color_done =          midtone( s_color_done, color_text );
    color_done_text =     midtone( s_color_done, color_text, 0.7 );
    color_done_bg =       midtone( s_color_done, color_base, 0.7 );

    color_canceled =      midtone( s_color_canceled, color_text );
    color_canceled_bg =   midtone( s_color_canceled, color_base, 0.7 );
}

String
Theme::get_css_class_def() const
{
    const auto& name    { get_css_class_name() };
    const int   size    { font.get_size_is_absolute() ? font.get_size()
                                                      : ( font.get_size() / PANGO_SCALE ) };
    const auto& c_text  { color_text.to_string() };
    const auto& c_base  { color_base.to_string() };
    const auto& font_f  { font.get_family().empty() ? "sans" : font.get_family() };
    const auto& bg_img  { image_bg.empty() ? "none"
                                           : STR::compose( "url(\"", image_bg, "\")" ) };

    return( STR::compose(
            "textview#", name, " { "
                "color: ",              c_text, "; "
                "font-family: ",        font_f, "; "
                "font-size: ",          size,   "pt; "
                "caret-color: ",        c_text, "; }\n"
            "textview#", name, " text selection { "
                "color: ",              c_base, "; "
                "background: ",   color_heading.to_string(), "; }\n"
            "textview#", name, " text:selected { "
                "color: ",              c_base, "; "
                "background: ",   color_heading.to_string(), "; }\n"
            "textview#", name, " text { "
                "background-color: ",   c_base, "; "
                "background-image: ",   bg_img, "; }\n"
            "button.", name, " { "
                "background-color: ",   c_base, "; "
                "background-image: ",   bg_img, "; }\n"
            "entry#noncustom text { background-color: initial; background-image: initial; }\n"
            "combobox#noncustom entry text { background-color: initial; "
                                            "background-image: initial; }" ) );
    // noncustom specialization is to revert children of the main textview to defaults
}

// THEMESYSTEM =====================================================================================
ThemeSystem::ThemeSystem( const Ustring& f,
                          const std::string& cb,
                          const std::string& ct,
                          const std::string& ch,
                          const std::string& csh,
                          const std::string& chl )
:   Theme( nullptr, "Lifeograph", f, cb, ct, ch, csh, chl )
{
}

ThemeSystem*
ThemeSystem::get()
{
    static ThemeSystem *s_theme{ nullptr };

    if( s_theme == nullptr )
        s_theme = new ThemeSystem( "Sans 10", "white", "black", "#B72525", "#963F3F", "#FFBBBB" );

    return s_theme;
}

// CHARTS ==========================================================================================
ChartData::ChartData( Diary* d, int t )
: m_properties( t ), m_td( new TableData( d ) ), m_p2diary( d )
{ }

ChartData::~ChartData()
{
    delete m_td;
}

void
ChartData::clear()
{
    m_properties = 0;
    m_unit.clear();
    clear_points();
}

void
ChartData::set_diary( Diary* diary )
{
    m_p2diary = diary;
    m_td->set_diary( diary );
}

String
ChartData::get_as_string() const
{
    String chart_def;

    // table:
    chart_def += STR::compose( "Gt", m_table_id, '\n' );

    // columns:
    chart_def += STR::compose( "Gx", m_tcidx, '\n' );
    chart_def += STR::compose( "Gy", m_tcidy, '\n' );
    chart_def += STR::compose( "Gu", m_tcidu, '\n' );
    chart_def += STR::compose( "Gf", m_tcidf, '\n' );
    chart_def += STR::compose( "Gv", m_filter_v, '\n' );

    // options:
    chart_def += "Go_"; // _ was for TAGGED_ONLY but it is no longer in use

    switch( m_properties & STYLE_MASK )
    {
        default:                chart_def += 'L'; break;
        case STYLE_BARS:        chart_def += 'B'; break;
        case STYLE_PIE:         chart_def += 'P'; break;
    }
    switch( m_properties & PERIOD_MASK )
    {
        default:                chart_def += 'D'; break; // daily
        case PERIOD_WEEKLY:     chart_def += 'W'; break;
        case PERIOD_MONTHLY:    chart_def += 'M'; break;
        case PERIOD_YEARLY:     chart_def += 'Y'; break;
    }
    switch( m_properties & COMBINE_MASK )
    {
        default:                            chart_def += 'P'; break;    // cumulative periodic
        case COMBINE_CUMULATIVE_CONTINUOUS: chart_def += 'C'; break;
        case COMBINE_AVERAGE:               chart_def += 'A'; break;
    }

    return chart_def;
}

void
ChartData::set_from_string( const Ustring& chart_def )
{
    String      line;
    StringSize  line_offset{ 0 };

    clear();

    while( STR::get_line( chart_def, line_offset, line ) )
    {
        if( line.size() < 2 )   // should never occur
            continue;

        switch( line[ 1 ] )
        {
            case 't':   // table
                set_table( std::stoul( line.substr( 2 ) ) );
                break;
            case 'x':   // x-axis
                m_tcidx = std::stoul( line.substr( 2 ) );
                if( Diary::d->get_version() < 2018 && m_tcidx < TableData::COL_ID_MAX )
                    m_tcidx += TableData::COL_ID_MIN; // convert from index to id
                break;
            case 'y':   // y-axis
                m_tcidy = std::stoul( line.substr( 2 ) );
                if( Diary::d->get_version() < 2018 && m_tcidy < TableData::COL_ID_MAX )
                    m_tcidy += TableData::COL_ID_MIN; // convert from index to id
                break;
            case 'u':   // underlay
                m_tcidu = std::stoul( line.substr( 2 ) );
                if( Diary::d->get_version() < 2018 && m_tcidu < TableData::COL_ID_MAX )
                    m_tcidu += TableData::COL_ID_MIN; // convert from index to id
                break;
            case 'f':   // filter
                m_tcidf = std::stoul( line.substr( 2 ) );
                if( Diary::d->get_version() < 2018 && m_tcidf < TableData::COL_ID_MAX )
                    m_tcidf += TableData::COL_ID_MIN; // convert from index to id
                break;
            case 'v':   // filter value
                m_filter_v = line.substr( 2 );
                break;
            case 'o':
                switch( line[ 3 ] )
                {
                    case 'L': m_properties |= STYLE_LINE; break;
                    case 'B': m_properties |= STYLE_BARS; break;
                    case 'P': m_properties |= STYLE_PIE; break;
                }
                switch( line[ 4 ] )
                {
                    case 'D': m_properties |= PERIOD_DAILY; break;
                    case 'W': m_properties |= PERIOD_WEEKLY; break;
                    case 'M': m_properties |= PERIOD_MONTHLY; break;
                    case 'Y': m_properties |= PERIOD_YEARLY; break;
                }
                switch( line[ 5 ] )
                {
                    case 'P': m_properties |= COMBINE_CUMULATIVE_PERIODIC; break;
                    case 'C': m_properties |= COMBINE_CUMULATIVE_CONTINUOUS; break;
                    case 'A': m_properties |= COMBINE_AVERAGE; break;
                }
                break;
            default:
                PRINT_DEBUG( "Unrecognized chart string: ", line );
                break;
        }
    }

    refresh_type();
    refresh_unit();
}

void
ChartData::set_from_string_old( const Ustring& chart_def )
{
    // TODO: 3.1: to be improved but it may not worth to effort to implement upgrading all sorts of...
    //       ...older charts
    String      line;
    StringSize  line_offset{ 0 };

    m_properties = 0;
    m_table_id = TABLE_ID_ALL_ENTRIES;
    m_tcidx = TableData::COL_ID_MIN; // date column
    m_tcidy = COLUMN_COUNT;
    m_tcidu = COLUMN_NONE;

    while( STR::get_line( chart_def, line_offset, line ) )
    {
        if( line.size() < 2 )   // should never occur
            continue;

        switch( line[ 1 ] )
        {
            case 'y':   // y axis
            {
                switch( line[ 2 ] )
                {
                    // case 'c':   // count: alrready set
                    //     break;
                    case 'l':   // text length
                        m_tcidx = ( TableData::COL_ID_MIN + 2 ); // size column
                        break;
                    case 'm':   // map path length
                        break;
                    case 't':   // tag value <- hard to support
                        break;
                    case 'p':   // tag value for para <-hard to support
                        m_table_id = TABLE_ID_ALL_PARAGRAPHS;
                        break;
                }
                break;
            }
            //case 'p': // para filter tag DROPPED
            /*case 'f':   // filter
                if( Diary::d->is_old() )
                    filter_entry = m_p2diary->get_filter( line.substr( 2 ) );
                else
                {
                    if( line[ 2 ] == 'e' )
                        filter_entry = m_p2diary->get_filter( std::stoul( line.substr( 3 ) ) );
                    else
                    if( line[ 2 ] == 'p' )
                        filter_para = m_p2diary->get_filter( std::stoul( line.substr( 3 ) ) );
                }
                break;*/
            case 'o':
                switch( line[ 3 ] )
                {
                    case 'Y': m_tcidu = COLUMN_PREV_YEAR; break;
                    case 'P':  break; // planned - TODO: 3.1
                }
                switch( line[ 4 ] )
                {
                    case 'W': m_properties |= PERIOD_WEEKLY; break;
                    case 'M': m_properties |= PERIOD_MONTHLY; break;
                    case 'Y': m_properties |= PERIOD_YEARLY; break;
                }
                switch( line[ 5 ] )
                {
                    case 'P': m_properties |= COMBINE_CUMULATIVE_PERIODIC|STYLE_BARS; break;
                    case 'C': m_properties |= COMBINE_CUMULATIVE_CONTINUOUS|STYLE_BARS; break;
                    case 'A': m_properties |= COMBINE_AVERAGE|STYLE_LINE; break;
                }
                break;
        }
    }
}

unsigned int
ChartData::calculate_distance( const DateV d1, const DateV d2 ) const
{
    switch( m_properties & PERIOD_MASK )
    {
        case PERIOD_DAILY:
            return Date::calculate_days_between_abs( d1, d2 );
        case PERIOD_WEEKLY:
            return Date::calculate_weeks_between_abs( d1, d2 );
        case PERIOD_MONTHLY:
            return Date::calculate_months_between_abs( d1, d2 );
        case PERIOD_YEARLY:
        default:
            return labs( int( Date::get_year( d1 ) ) - int( Date::get_year( d2 ) ) );
    }
}

void
ChartData::forward_date( DateV& date ) const
{
    switch( m_properties & PERIOD_MASK )
    {
        case PERIOD_DAILY:      Date::forward_days( date, 1 ); break;
        case PERIOD_WEEKLY:     Date::forward_days( date, 7 ); break;
        case PERIOD_MONTHLY:
            if( Date::get_month( date ) == 12 )
                date = Date::make( Date::get_year( date ) + 1, 1, 1 );
            else
                date = Date::make( Date::get_year( date ), Date::get_month( date ) + 1, 1 );
            break;
        case PERIOD_YEARLY:
            date = Date::make( Date::get_year( date ) + 1, 1, 1 );
            break;
    }
}

void
ChartData::clear_points()
{
    values_date.clear();
    values_str.clear();
    values_str2index.clear();
    values_index2str.clear();
    values_num.clear();

    v_min = Constants::INFINITY_PLS;
    v_max = Constants::INFINITY_MNS;

    m_span = 0;
}

void
ChartData::add_value_date( DateV date, const Value vy, const Value vu, DiaryElement* elem )
{
    switch( get_period() )
    {
        case PERIOD_DAILY:      date = Date::isolate_YMD( date ); break;
        case PERIOD_WEEKLY:     Date::backward_to_week_start( date ); break;
        case PERIOD_MONTHLY:    Date::backward_to_month_start( date ); break;
        case PERIOD_YEARLY:     Date::backward_to_year_start( date ); break;
    }

    if( values_date.empty() )
    {
        values_date[ date ] = { vy, vu, 1, { elem } };
        return;
    }
    else if( values_date.find( date ) == values_date.end() ) // insert the intermediate values
    {
        DateV           d;
        unsigned int    steps_between;

        if( date < values_date.begin()->first )
        {
            d = date;
            steps_between = calculate_distance( d, values_date.begin()->first );
        }
        else
        {
            d = values_date.rbegin()->first;
            steps_between = calculate_distance( d, date );
        }

        for( unsigned int i = 1; i < steps_between; i++ )
        {
            forward_date( d );
            values_date[ d ] = { 0.0, 0.0, 0, {} };
        }
    }

    switch( get_combining() )
    {
        case COMBINE_CUMULATIVE_PERIODIC:
        case COMBINE_CUMULATIVE_CONTINUOUS:
            values_date[ date ].add_cumulative( vy, vu, elem );
            break;
        case ChartData::COMBINE_AVERAGE:
            values_date[ date ].add_average( vy, vu, elem );

            break;
    }
}

void
ChartData::fill_in_intermediate_date_values()
{
    // auto get_next_real = [ this ]( DateV d ) -> DateV
    // {

    // };

    switch( get_combining() )
    {
        case ChartData::COMBINE_CUMULATIVE_PERIODIC: // do nothing
            break;
        case ChartData::COMBINE_CUMULATIVE_CONTINUOUS:
        {
            Value vp{ 0.0 }, up{ 0.0 };

            for( auto& vd : values_date )
            {
                vd.second.v += vp;
                vd.second.u += up;

                vp = vd.second.v;
                up = vd.second.u;
            }
            break;
        }
        case ChartData::COMBINE_AVERAGE://---is this really meaningful?
            // if( vd.second.c == 0 )
            // {
            //     auto d = vd.first;
            //     auto vi = vy;
            //     auto ui = vu;
            //     const auto steps_between = calculate_distance( d, get_next_real() );
            //     const auto v_offset = ( vy - values_date.begin()->second.v ) / steps_between;
            //     const auto u_offset = ( vu - values_date.begin()->second.u ) / steps_between;
            //     vd.second.v = vi + v_offset * step_i;
            // }
            break;
    }
}

void
ChartData::fill_in_date_underlays()
{
    auto        it_vs{ values_date.begin() }; // source
    unsigned    offset;

    switch( get_period() )
    {
        case ChartData::PERIOD_DAILY:
            //offset = Date::get_days_in_year( Date::get_year( d ) - 1 );
            offset = 365; // TODO: 3.1: incorporate leap years
            break;
        case ChartData::PERIOD_WEEKLY: offset = 52; break;
        // case ChartData::PERIOD_MONTHLY:
        default:                       offset = 12; break;
    }

    if( values_date.size() > offset )
    {
        auto it_vt{ values_date.begin() }; // target
        std::advance( it_vt, offset );

        while( it_vt != values_date.end() )
        {
            it_vt->second.u = it_vs->second.v;
            ++it_vs;
            ++it_vt;
        }
    }
}

void
ChartData::add_value_str( const Ustring& str, const Value vy, const Value vu, DiaryElement* elem )
{
    auto&& it{ values_str2index.find( str ) };

    if( it == values_str2index.end() )
    {
        const auto iv{ values_str.size() };
        values_str2index[ str ] = iv;
        values_index2str[ iv ] = str;
        values_str[ iv ] = { vy, vu, 1, { elem } };
    }
    else
    {
        switch( get_combining() )
        {
            case ChartData::COMBINE_CUMULATIVE_PERIODIC:
            case ChartData::COMBINE_CUMULATIVE_CONTINUOUS:
                values_str[ ( *it ).second ].add_cumulative( vy, vu, elem );
                break;
            case ChartData::COMBINE_AVERAGE:
                values_str[ ( *it ).second ].add_average( vy, vu, elem );
                break;
        }
    }
}

void
ChartData::add_value_num( const Value vx, const Value vy, const Value vu, DiaryElement* elem )
{
    if( values_num.find( vx ) == values_num.end() )
        values_num[ vx ] = { vy, vu, 1, { elem } };
    else
    {
        switch( get_combining() )
        {
            case ChartData::COMBINE_CUMULATIVE_PERIODIC:
            case ChartData::COMBINE_CUMULATIVE_CONTINUOUS:
                values_num[ vx ].add_cumulative( vy, vu, elem );
                break;
            case ChartData::COMBINE_AVERAGE:
                values_num[ vx ].add_average( vy, vu, elem );
                break;
        }
    }
}

void
ChartData::update_span()
{
    switch( get_type() )
    {
        case TYPE_DATE:     m_span = values_date.size();    break;
        case TYPE_NUMBER:   m_span = values_num.size();     break;
        default:            m_span = values_str.size();     break;
    }
}

void
ChartData::update_min_max()
{
    switch( get_type() )
    {
        case TYPE_DATE:   update_min_max( values_date ); break;
        case TYPE_NUMBER: update_min_max( values_num ); break;
        case TYPE_STRING: update_min_max( values_str ); break;
    }

    // try to improve the min of the graph to optimize between screen real estate usage and...
    // ...giving a better idea of the actual ratios:
    if( v_min > 0 /* MAYBE: && get_style() == STYLE_BARS */ )
    {
        if( v_min <= ( v_max - v_min ) )
            v_min = 0.0;
        else
            v_min -= ( v_max - v_min );
    }
}

void
ChartData::calculate_points()
{
    if( !m_p2diary ) return;

    clear_points();

    auto col{ m_td->m_columns_map[ m_tcidx ] };

    if( !col ) return;

    col->allocate_filter_stacks();

    for( auto& line : m_td->m_lines )
    {
        if( m_tcidf != ChartData::COLUMN_NONE && !m_filter_v.empty() &&
            line->get_col_v_txt_by_cid( m_tcidf ) != m_filter_v )
            continue;

        const Value vy{ m_tcidy == ChartData::COLUMN_COUNT
                        ? 1.0
                        : line->get_col_v_num_by_cid( m_tcidy ) };
        const Value vu{ m_tcidu < ChartData::COLUMN_NONE
                        ? line->get_col_v_num_by_cid( m_tcidu )
                        : 0.0 };
        const auto  ic{ col->get_index() };

        switch( get_type() )
        {
            case ChartData::TYPE_DATE:
                if( col->get_type() == TableColumn::TCT_GANTT )
                {
                    // TODO: 3.1: subdivide the value when necessary
                    // if( m_tcidy != ChartData::COLUMN_COUNT )
                    // {
                    //     const Value vy_sub{ vy / period };
                    //     const Value vu_sub{ vu / period };
                    // }

                    for( auto segment : line->m_periods )
                        for( auto vx = segment.dbgn; vx < segment.dend; forward_date( vx ) )
                            add_value_date( vx, vy, vu, line->m_p2elem );
                }
                else
                {
                    const auto& vx{ line->get_col_v_date( ic ) };
                    if( vx != Date::NOT_SET )
                        add_value_date( vx, vy, vu, line->m_p2elem );
                }
                break;
            case ChartData::TYPE_NUMBER:
            {
                const auto& vx{ line->get_col_v_num( ic ) };
                add_value_num( vx, vy, vu, line->m_p2elem );
                break;
            }
            default:
            {
                const auto& vx{ line->get_col_v_txt( ic ) };
                add_value_str( vx, vy, vu, line->m_p2elem );
                break;
            }
        }
    }

    col->deallocate_filter_stacks();

    if( get_combining() == ChartData::COMBINE_CUMULATIVE_CONTINUOUS )
        fill_in_intermediate_date_values();

    if( get_type() == ChartData::TYPE_DATE &&
        get_period() != ChartData::PERIOD_YEARLY &&
        m_tcidu == ChartData::COLUMN_PREV_YEAR )
        fill_in_date_underlays();

    update_min_max();
    update_span();
}

void
ChartData::set_properties( int t )
{
    if( 0 == ( t & STYLE_MASK ) )       t |= ( m_properties & STYLE_MASK );
    if( 0 == ( t & TYPE_MASK ) )        t |= ( m_properties & TYPE_MASK );
    if( 0 == ( t & PERIOD_MASK ) )      t |= ( m_properties & PERIOD_MASK );
    if( 0 == ( t & COMBINE_MASK ) )     t |= ( m_properties & COMBINE_MASK );

    m_properties = t;
}

void
ChartData::set_table( DEID id )
{
    m_table_id = id;
    m_tcidx = -1;
    m_tcidy = ChartData::COLUMN_COUNT;
    m_tcidu = ChartData::COLUMN_NONE;
    m_tcidf = ChartData::COLUMN_NONE;
    m_filter_v.clear();

    refresh_table();
    refresh_type();
    refresh_unit();
}

void
ChartData::refresh_table()
{
    m_td->clear();

    switch( m_table_id )
    {
        case TABLE_ID_ALL_ENTRIES:
            m_td->set_from_string( TABLE_DEF_ALL_ENTRIES );
            break;
        case TABLE_ID_ALL_PARAGRAPHS:
            m_td->set_from_string( TABLE_DEF_ALL_PARAGRAPHS );
            break;
        default:
        {
            auto table { m_p2diary->get_table( m_table_id ) };
            if( !table ) return;
            m_td->set_from_string( table->get_definition() );
            break;
        }
    }

    m_td->m_grouping_depth = 0; // disable grouping again
    // TODO: 3.1: (MAYBE): a special table populator may be created for charts as they disregard grouping
    m_td->populate_lines( false );
}

void
ChartData::refresh_type()
{
    auto col{ m_td->m_columns_map[ m_tcidx ] };

    if( !col ) return;

    if( col->get_type() == TableColumn::TCT_DATE || col->get_type() == TableColumn::TCT_GANTT )
        set_properties( TYPE_DATE );
    else
    if( col->is_numeric() )
        set_properties( TYPE_NUMBER );
    else
        set_properties( TYPE_STRING );
}

void
ChartData::refresh_unit()
{
    if( m_tcidy != COLUMN_COUNT )
    {
        auto col{ m_td->m_columns_map[ m_tcidy ] };
        if( col )
        {
            m_unit = col->get_unit();
            return;
        }
    }

    m_unit = "";
}

const Ustring ChartElem::DEFINITION_DEFAULT{ STR::compose( "Gt",
                                                           TABLE_ID_ALL_ENTRIES,
                                                           "\nGo_LMP" ) };
const Ustring ChartElem::DEFINITION_DEFAULT_Y{ STR::compose( "Gt",
                                                             TABLE_ID_ALL_ENTRIES,
                                                             "\nGo_LYP" ) };

const R2Pixbuf&
ChartElem::get_icon() const
{
    return Lifeograph::icons->chart;
}

} // end of namespace LIFEO
