// Copyright (c) 2010, Jens Peter Secher <jpsecher@gmail.com>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

// Shell process container and utilities.

class Process
{
	public static function runButExitOnError
	(
		command : String,
		args : Array<String>,
		logger : ILog,
		/// By default, the output is returned in an iterator.
		? level : Int = 0xFFFF,
		/// By default, only the exit code 0 is accepted.
		? acceptableExitCodes : Array<Int>,
		/// By default, CWD is not changed.
		? workingDirectory : String
	)
	: Iter<String>
	{
		// Change working directory if necessary.
		var cwd : String = null;
		if( workingDirectory != null )
		{			
			cwd = Sys.getCwd();
			Sys.setCwd( workingDirectory );
		}
		var p = new Process( command, args );
		var context = new List<String>();
		for( line in p.stdouterr() )
		{
			// Put all output in the log.
			logger.log( line );
			// If the logger's verbosity is at least as high as the info level,
			// then also display the line.
			if( logger.verbosity >= level )
			{
				logger.info( level, line );
			}
			// Maintain the context.
			else
			{
				context.add( line );
			}
		}
		// Change working directory back if necessary.
		if( cwd != null )
		{			
			Sys.setCwd( cwd );
		}
		// Check process exit code.
		if( acceptableExitCodes == null ) acceptableExitCodes = [0];
		if( ! Lambda.has( acceptableExitCodes, p.process.exitCode() ) )
		{
			// Process failed, abort with context.
			Lambda.iter( context, function(line) logger.info( 0, line ) );
			logger.error( p.command + " failed." );
			Sys.exit( 1 );
		}
		// Process succeeded.
		return new MakeIter( context.iterator() );
	}

	public var process (default,null) : sys.io.Process;
	
	public function new( command: String, args: Array<String> )
	{
		this.command = command + " " + args.join( " " );
		process = new sys.io.Process( command, args );
	}

	//
	// Combine stdout and stderr into one iterator.
	//
	public function stdouterr() : Iter<String>
	{
		return new ProcessTwoStreamIterator( process.stdout, process.stderr, process );
	}

	var command (default,null) : String;	
}
