/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      Hidayat Khan <huk2209@gmail.com>
 */

#define SIMDE_TEST_X86_AVX512_INSN avg

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/set.h>
#include <simde/x86/avx512/avg.h>

static int
test_simde_mm_mask_avg_epu8 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const uint8_t src[16];
    const simde__mmask16 k;
    const uint8_t a[16];
    const uint8_t b[16];
    const uint8_t r[16];
  } test_vec[] = {
    { { UINT8_C(223), UINT8_C( 94), UINT8_C(238), UINT8_C(159), UINT8_C(163), UINT8_C( 41), UINT8_C(132), UINT8_C(235),
        UINT8_C(179), UINT8_C(199), UINT8_C(121), UINT8_C(192), UINT8_C(144), UINT8_C(137), UINT8_C(199), UINT8_C(242) },
      UINT16_C(  880),
      { UINT8_C( 38), UINT8_C(204), UINT8_C( 95), UINT8_C( 86), UINT8_C( 95), UINT8_C(207), UINT8_C(238), UINT8_C(187),
        UINT8_C(158), UINT8_C(238), UINT8_C(129), UINT8_C( 60), UINT8_C( 59), UINT8_C(113), UINT8_C(182), UINT8_C( 35) },
      { UINT8_C(126), UINT8_C(124), UINT8_C(159), UINT8_C(184), UINT8_C( 44), UINT8_C(132), UINT8_C( 32), UINT8_C(176),
        UINT8_C(198), UINT8_C(151), UINT8_C( 42), UINT8_C( 37), UINT8_C(208), UINT8_C(244), UINT8_C( 40), UINT8_C(127) },
      { UINT8_C(223), UINT8_C( 94), UINT8_C(238), UINT8_C(159), UINT8_C( 70), UINT8_C(170), UINT8_C(135), UINT8_C(235),
        UINT8_C(178), UINT8_C(195), UINT8_C(121), UINT8_C(192), UINT8_C(144), UINT8_C(137), UINT8_C(199), UINT8_C(242) } },
    { { UINT8_C(  1), UINT8_C(  3), UINT8_C(187), UINT8_C( 89), UINT8_C(251), UINT8_C(189), UINT8_C( 42), UINT8_C( 80),
        UINT8_C( 90), UINT8_C(205), UINT8_C(190), UINT8_C(157), UINT8_C(162), UINT8_C( 75), UINT8_C(201), UINT8_C(144) },
      UINT16_C( 3057),
      { UINT8_C( 39), UINT8_C( 73), UINT8_C(228), UINT8_C(182), UINT8_C(168), UINT8_C(194), UINT8_C( 73), UINT8_C( 32),
        UINT8_C(224), UINT8_C(227), UINT8_C( 70), UINT8_C(226), UINT8_C(117),    UINT8_MAX, UINT8_C( 44), UINT8_C(192) },
      { UINT8_C( 64), UINT8_C(180), UINT8_C(205), UINT8_C( 57), UINT8_C(184), UINT8_C( 62), UINT8_C(173), UINT8_C(184),
        UINT8_C(102), UINT8_C(195), UINT8_C( 31), UINT8_C(254), UINT8_C(204), UINT8_C(228), UINT8_C( 49), UINT8_C( 40) },
      { UINT8_C( 52), UINT8_C(  3), UINT8_C(187), UINT8_C( 89), UINT8_C(176), UINT8_C(128), UINT8_C(123), UINT8_C(108),
        UINT8_C(163), UINT8_C(211), UINT8_C(190), UINT8_C(240), UINT8_C(162), UINT8_C( 75), UINT8_C(201), UINT8_C(144) } },
    { { UINT8_C( 15), UINT8_C( 24), UINT8_C(132), UINT8_C(130), UINT8_C(210), UINT8_C( 62), UINT8_C(197), UINT8_C(109),
        UINT8_C( 97), UINT8_C(  2), UINT8_C( 46), UINT8_C(162), UINT8_C( 42), UINT8_C( 88), UINT8_C(245), UINT8_C(123) },
      UINT16_C( 7289),
      { UINT8_C(  0), UINT8_C(136), UINT8_C( 83), UINT8_C(112), UINT8_C(183), UINT8_C( 64), UINT8_C( 47), UINT8_C( 79),
        UINT8_C(103), UINT8_C(247), UINT8_C( 96), UINT8_C(189), UINT8_C( 23), UINT8_C(119), UINT8_C(180), UINT8_C(161) },
      { UINT8_C(197), UINT8_C( 69), UINT8_C(160), UINT8_C(138), UINT8_C(105), UINT8_C( 52), UINT8_C(  1), UINT8_C(  2),
        UINT8_C(248), UINT8_C( 44), UINT8_C(225), UINT8_C(162), UINT8_C(121), UINT8_C( 29), UINT8_C( 82), UINT8_C(180) },
      { UINT8_C( 99), UINT8_C( 24), UINT8_C(132), UINT8_C(125), UINT8_C(144), UINT8_C( 58), UINT8_C( 24), UINT8_C(109),
        UINT8_C( 97), UINT8_C(  2), UINT8_C(161), UINT8_C(176), UINT8_C( 72), UINT8_C( 88), UINT8_C(245), UINT8_C(123) } },
    { { UINT8_C(228), UINT8_C( 83), UINT8_C(134), UINT8_C( 72), UINT8_C( 93), UINT8_C(252), UINT8_C(135), UINT8_C( 44),
        UINT8_C(207), UINT8_C(254), UINT8_C( 26), UINT8_C( 36), UINT8_C(251), UINT8_C(191), UINT8_C(198), UINT8_C(162) },
      UINT16_C(10439),
      { UINT8_C(177), UINT8_C( 33), UINT8_C(146), UINT8_C(155), UINT8_C(248), UINT8_C(254), UINT8_C(125), UINT8_C(  3),
        UINT8_C( 59), UINT8_C(138), UINT8_C(  1), UINT8_C(174), UINT8_C(242), UINT8_C(100), UINT8_C(200), UINT8_C( 21) },
      { UINT8_C(174), UINT8_C(241), UINT8_C(168), UINT8_C(118), UINT8_C( 64), UINT8_C(174), UINT8_C(230), UINT8_C(175),
        UINT8_C(180), UINT8_C( 53), UINT8_C(234), UINT8_C(150), UINT8_C( 37), UINT8_C(168), UINT8_C( 83), UINT8_C(  3) },
      { UINT8_C(176), UINT8_C(137), UINT8_C(157), UINT8_C( 72), UINT8_C( 93), UINT8_C(252), UINT8_C(178), UINT8_C( 89),
        UINT8_C(207), UINT8_C(254), UINT8_C( 26), UINT8_C(162), UINT8_C(251), UINT8_C(134), UINT8_C(198), UINT8_C(162) } },
    { { UINT8_C(122), UINT8_C(250), UINT8_C(189), UINT8_C(160), UINT8_C(115), UINT8_C(149), UINT8_C(245), UINT8_C( 33),
        UINT8_C(124), UINT8_C( 29), UINT8_C(124), UINT8_C(108), UINT8_C(224), UINT8_C( 32), UINT8_C(195), UINT8_C(231) },
      UINT16_C(12761),
      { UINT8_C(247), UINT8_C(133), UINT8_C(200), UINT8_C(215), UINT8_C(115), UINT8_C(235), UINT8_C( 55), UINT8_C(169),
        UINT8_C( 16), UINT8_C(  0), UINT8_C(228), UINT8_C( 54), UINT8_C(237), UINT8_C(236), UINT8_C(223), UINT8_C(171) },
      { UINT8_C( 16), UINT8_C(145), UINT8_C(101), UINT8_C(177), UINT8_C( 72), UINT8_C(123), UINT8_C( 46), UINT8_C(141),
        UINT8_C(192), UINT8_C( 90), UINT8_C(253), UINT8_C( 21), UINT8_C( 45), UINT8_C(185), UINT8_C( 42), UINT8_C(  3) },
      { UINT8_C(132), UINT8_C(250), UINT8_C(189), UINT8_C(196), UINT8_C( 94), UINT8_C(149), UINT8_C( 51), UINT8_C(155),
        UINT8_C(104), UINT8_C( 29), UINT8_C(124), UINT8_C(108), UINT8_C(141), UINT8_C(211), UINT8_C(195), UINT8_C(231) } },
    { { UINT8_C(175), UINT8_C(113), UINT8_C( 53), UINT8_C(251), UINT8_C( 62), UINT8_C(180), UINT8_C( 14), UINT8_C(218),
        UINT8_C(  6), UINT8_C(177), UINT8_C(124), UINT8_C(219), UINT8_C(130), UINT8_C(183), UINT8_C(159), UINT8_C( 60) },
      UINT16_C(31348),
      { UINT8_C(147), UINT8_C( 33), UINT8_C(100), UINT8_C( 13), UINT8_C(120), UINT8_C(107), UINT8_C(111), UINT8_C(243),
        UINT8_C(207), UINT8_C(130), UINT8_C(  1), UINT8_C(224), UINT8_C( 43), UINT8_C(186), UINT8_C( 31), UINT8_C( 65) },
      { UINT8_C( 99), UINT8_C(  7), UINT8_C(250), UINT8_C( 91), UINT8_C( 51), UINT8_C( 29), UINT8_C(220), UINT8_C(147),
        UINT8_C(201), UINT8_C( 14), UINT8_C( 39), UINT8_C(168), UINT8_C(126), UINT8_C( 67), UINT8_C(188), UINT8_C(  5) },
      { UINT8_C(175), UINT8_C(113), UINT8_C(175), UINT8_C(251), UINT8_C( 86), UINT8_C( 68), UINT8_C(166), UINT8_C(218),
        UINT8_C(  6), UINT8_C( 72), UINT8_C(124), UINT8_C(196), UINT8_C( 85), UINT8_C(127), UINT8_C(110), UINT8_C( 60) } },
    { { UINT8_C(242), UINT8_C( 69), UINT8_C(162), UINT8_C(119), UINT8_C(254), UINT8_C(223), UINT8_C(  4), UINT8_C(103),
        UINT8_C( 71), UINT8_C(203), UINT8_C(102), UINT8_C(161), UINT8_C( 74), UINT8_C(192), UINT8_C( 71), UINT8_C( 18) },
      UINT16_C(37728),
      { UINT8_C(164), UINT8_C(184), UINT8_C(107), UINT8_C(170), UINT8_C(176), UINT8_C( 87), UINT8_C( 21), UINT8_C(189),
        UINT8_C(121), UINT8_C( 65), UINT8_C(220), UINT8_C(247), UINT8_C(123), UINT8_C(116), UINT8_C(175), UINT8_C( 66) },
      { UINT8_C( 59), UINT8_C( 56), UINT8_C(140), UINT8_C(215), UINT8_C( 57), UINT8_C( 79), UINT8_C(  6), UINT8_C(198),
        UINT8_C(158), UINT8_C( 36), UINT8_C( 45), UINT8_C(148), UINT8_C(219), UINT8_C(148), UINT8_C(243), UINT8_C(124) },
      { UINT8_C(242), UINT8_C( 69), UINT8_C(162), UINT8_C(119), UINT8_C(254), UINT8_C( 83), UINT8_C( 14), UINT8_C(103),
        UINT8_C(140), UINT8_C( 51), UINT8_C(102), UINT8_C(161), UINT8_C(171), UINT8_C(192), UINT8_C( 71), UINT8_C( 95) } },
    { { UINT8_C(235), UINT8_C(169), UINT8_C(130), UINT8_C( 40), UINT8_C(218), UINT8_C(  0), UINT8_C(170), UINT8_C(139),
        UINT8_C(222), UINT8_C(105), UINT8_C( 13), UINT8_C(153), UINT8_C( 80), UINT8_C(242), UINT8_C(224), UINT8_C(166) },
      UINT16_C(  700),
      { UINT8_C( 54), UINT8_C(208), UINT8_C( 42), UINT8_C( 57), UINT8_C( 30), UINT8_C(  7), UINT8_C( 16), UINT8_C( 87),
        UINT8_C( 69), UINT8_C(135), UINT8_C( 64), UINT8_C(151), UINT8_C(101), UINT8_C(  6), UINT8_C( 98), UINT8_C( 55) },
      { UINT8_C(207), UINT8_C(241), UINT8_C(248), UINT8_C(218), UINT8_C( 34), UINT8_C(178), UINT8_C(118), UINT8_C(254),
        UINT8_C(126), UINT8_C(179), UINT8_C( 49), UINT8_C(198), UINT8_C(179), UINT8_C( 64), UINT8_C( 60), UINT8_C(  3) },
      { UINT8_C(235), UINT8_C(169), UINT8_C(145), UINT8_C(138), UINT8_C( 32), UINT8_C( 93), UINT8_C(170), UINT8_C(171),
        UINT8_C(222), UINT8_C(157), UINT8_C( 13), UINT8_C(153), UINT8_C( 80), UINT8_C(242), UINT8_C(224), UINT8_C(166) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i src = simde_mm_loadu_epi8(test_vec[i].src);
    simde__m128i a = simde_mm_loadu_epi8(test_vec[i].a);
    simde__m128i b = simde_mm_loadu_epi8(test_vec[i].b);
    simde__m128i r = simde_mm_mask_avg_epu8(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_u8x16(r, simde_mm_loadu_epi8(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm_maskz_avg_epu8 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask16 k;
    const uint8_t a[16];
    const uint8_t b[16];
    const uint8_t r[16];
  } test_vec[] = {
    { UINT16_C( 4520),
      { UINT8_C(177), UINT8_C( 79), UINT8_C(221), UINT8_C(139), UINT8_C(188), UINT8_C(149), UINT8_C( 47), UINT8_C(212),
        UINT8_C( 46), UINT8_C( 99), UINT8_C(  6), UINT8_C( 44), UINT8_C(211), UINT8_C( 14), UINT8_C(103), UINT8_C(124) },
      { UINT8_C( 38), UINT8_C(167), UINT8_C(225), UINT8_C( 40), UINT8_C( 61), UINT8_C( 80), UINT8_C( 59), UINT8_C( 97),
        UINT8_C(246), UINT8_C(222), UINT8_C(189), UINT8_C( 77), UINT8_C(225), UINT8_C( 45), UINT8_C( 52), UINT8_C(179) },
      { UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C( 90), UINT8_C(  0), UINT8_C(115), UINT8_C(  0), UINT8_C(155),
        UINT8_C(146), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(218), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0) } },
    { UINT16_C(56465),
      { UINT8_C(144), UINT8_C( 79), UINT8_C( 55), UINT8_C(116), UINT8_C(  8), UINT8_C( 74), UINT8_C(160), UINT8_C( 23),
        UINT8_C( 28), UINT8_C( 86), UINT8_C( 52), UINT8_C(116), UINT8_C( 57), UINT8_C(188), UINT8_C(114), UINT8_C(224) },
      { UINT8_C( 95), UINT8_C( 57), UINT8_C( 28), UINT8_C( 75), UINT8_C(205), UINT8_C(213), UINT8_C(194), UINT8_C( 20),
        UINT8_C(242), UINT8_C(212), UINT8_C( 50), UINT8_C(130), UINT8_C(194), UINT8_C( 27), UINT8_C(233), UINT8_C(192) },
      { UINT8_C(120), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(107), UINT8_C(  0), UINT8_C(  0), UINT8_C( 22),
        UINT8_C(  0), UINT8_C(  0), UINT8_C( 51), UINT8_C(123), UINT8_C(126), UINT8_C(  0), UINT8_C(174), UINT8_C(208) } },
    { UINT16_C(58442),
      { UINT8_C(248), UINT8_C(114), UINT8_C( 53), UINT8_C( 75), UINT8_C( 60), UINT8_C(224), UINT8_C( 66), UINT8_C(126),
        UINT8_C(183), UINT8_C( 52), UINT8_C( 73), UINT8_C( 36), UINT8_C( 12), UINT8_C( 14), UINT8_C(211), UINT8_C(237) },
      { UINT8_C(230), UINT8_C(156), UINT8_C( 31), UINT8_C(235), UINT8_C( 13), UINT8_C( 84), UINT8_C(108), UINT8_C(118),
        UINT8_C(154), UINT8_C(237), UINT8_C( 31), UINT8_C( 53), UINT8_C( 28), UINT8_C(216), UINT8_C(122), UINT8_C(  0) },
      { UINT8_C(  0), UINT8_C(135), UINT8_C(  0), UINT8_C(155), UINT8_C(  0), UINT8_C(  0), UINT8_C( 87), UINT8_C(  0),
        UINT8_C(  0), UINT8_C(  0), UINT8_C( 52), UINT8_C(  0), UINT8_C(  0), UINT8_C(115), UINT8_C(167), UINT8_C(119) } },
    { UINT16_C(31261),
      { UINT8_C(128), UINT8_C(105), UINT8_C(105), UINT8_C(133), UINT8_C( 37), UINT8_C( 61), UINT8_C(137), UINT8_C(126),
        UINT8_C( 96), UINT8_C(176), UINT8_C(161), UINT8_C( 59), UINT8_C(188), UINT8_C(173), UINT8_C(182), UINT8_C( 71) },
      { UINT8_C( 30), UINT8_C(228), UINT8_C(  8), UINT8_C(148), UINT8_C( 57), UINT8_C(106), UINT8_C( 88), UINT8_C( 65),
        UINT8_C( 78), UINT8_C(104), UINT8_C( 89), UINT8_C(221), UINT8_C(  7), UINT8_C(168), UINT8_C(224), UINT8_C( 69) },
      { UINT8_C( 79), UINT8_C(  0), UINT8_C( 57), UINT8_C(141), UINT8_C( 47), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
        UINT8_C(  0), UINT8_C(140), UINT8_C(  0), UINT8_C(140), UINT8_C( 98), UINT8_C(171), UINT8_C(203), UINT8_C(  0) } },
    { UINT16_C(19393),
      { UINT8_C(  8), UINT8_C( 63), UINT8_C( 71), UINT8_C(118), UINT8_C( 15), UINT8_C(126), UINT8_C(  0), UINT8_C(177),
        UINT8_C(101), UINT8_C( 31), UINT8_C(100), UINT8_C(116), UINT8_C(121), UINT8_C( 15), UINT8_C( 83), UINT8_C(183) },
      { UINT8_C(240), UINT8_C(132), UINT8_C(225), UINT8_C(129), UINT8_C(219), UINT8_C( 81), UINT8_C(204), UINT8_C( 84),
        UINT8_C( 93), UINT8_C(119), UINT8_C( 30), UINT8_C( 24), UINT8_C( 44), UINT8_C(166), UINT8_C(183), UINT8_C( 86) },
      { UINT8_C(124), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(102), UINT8_C(131),
        UINT8_C( 97), UINT8_C( 75), UINT8_C(  0), UINT8_C( 70), UINT8_C(  0), UINT8_C(  0), UINT8_C(133), UINT8_C(  0) } },
    { UINT16_C(41252),
      { UINT8_C(208), UINT8_C(182), UINT8_C( 60), UINT8_C(  7), UINT8_C( 93), UINT8_C(241), UINT8_C( 49), UINT8_C(121),
        UINT8_C(175), UINT8_C(110), UINT8_C(183), UINT8_C(116), UINT8_C(110), UINT8_C( 30), UINT8_C( 98), UINT8_C(228) },
      { UINT8_C(213), UINT8_C( 34), UINT8_C( 23), UINT8_C(211), UINT8_C( 37), UINT8_C(180), UINT8_C(142), UINT8_C( 24),
        UINT8_C(100), UINT8_C(164), UINT8_C( 50), UINT8_C(110), UINT8_C(254), UINT8_C(135), UINT8_C(148), UINT8_C(147) },
      { UINT8_C(  0), UINT8_C(  0), UINT8_C( 42), UINT8_C(  0), UINT8_C(  0), UINT8_C(211), UINT8_C(  0), UINT8_C(  0),
        UINT8_C(138), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C( 83), UINT8_C(  0), UINT8_C(188) } },
    { UINT16_C(59646),
      { UINT8_C(  1), UINT8_C(149), UINT8_C( 22), UINT8_C( 22), UINT8_C(243), UINT8_C( 85), UINT8_C(239), UINT8_C(217),
        UINT8_C(154), UINT8_C(103), UINT8_C(109), UINT8_C( 37), UINT8_C( 53), UINT8_C(251), UINT8_C( 11), UINT8_C( 72) },
      { UINT8_C( 24), UINT8_C(222), UINT8_C(191), UINT8_C(119), UINT8_C(253), UINT8_C(184), UINT8_C( 59), UINT8_C( 25),
        UINT8_C( 93), UINT8_C( 80), UINT8_C(194), UINT8_C( 70), UINT8_C(163), UINT8_C(111), UINT8_C( 27), UINT8_C(249) },
      { UINT8_C(  0), UINT8_C(186), UINT8_C(107), UINT8_C( 71), UINT8_C(248), UINT8_C(135), UINT8_C(149), UINT8_C(121),
        UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C( 54), UINT8_C(  0), UINT8_C(181), UINT8_C( 19), UINT8_C(161) } },
    { UINT16_C(50361),
      { UINT8_C( 29), UINT8_C(184), UINT8_C(187), UINT8_C(163), UINT8_C(100), UINT8_C(202), UINT8_C(129), UINT8_C(220),
        UINT8_C(112), UINT8_C( 12), UINT8_C( 67), UINT8_C(252), UINT8_C(226), UINT8_C(146), UINT8_C( 40), UINT8_C(117) },
      { UINT8_C( 65), UINT8_C(228), UINT8_C( 24), UINT8_C(137), UINT8_C(136), UINT8_C(209), UINT8_C(197), UINT8_C(194),
        UINT8_C( 37), UINT8_C(103), UINT8_C(218), UINT8_C(232), UINT8_C(118), UINT8_C( 96), UINT8_C( 24), UINT8_C(106) },
      { UINT8_C( 47), UINT8_C(  0), UINT8_C(  0), UINT8_C(150), UINT8_C(118), UINT8_C(206), UINT8_C(  0), UINT8_C(207),
        UINT8_C(  0), UINT8_C(  0), UINT8_C(143), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C( 32), UINT8_C(112) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i a = simde_mm_loadu_epi8(test_vec[i].a);
    simde__m128i b = simde_mm_loadu_epi8(test_vec[i].b);
    simde__m128i r = simde_mm_maskz_avg_epu8(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_u8x16(r, simde_mm_loadu_epi8(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm_mask_avg_epu16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const uint16_t src[8];
    const simde__mmask8 k;
    const uint16_t a[8];
    const uint16_t b[8];
    const uint16_t r[8];
  } test_vec[] = {
    { { UINT16_C(13001), UINT16_C(49556), UINT16_C(11089), UINT16_C(65127), UINT16_C(35056), UINT16_C( 1312), UINT16_C( 9004), UINT16_C(13332) },
      UINT8_C(175),
      { UINT16_C(62617), UINT16_C(24352), UINT16_C(22336), UINT16_C(47694), UINT16_C(54709), UINT16_C(59240), UINT16_C( 1665), UINT16_C(29730) },
      { UINT16_C(15758), UINT16_C(28426), UINT16_C(26767), UINT16_C(26014), UINT16_C(41205), UINT16_C(51151), UINT16_C(60304), UINT16_C(21251) },
      { UINT16_C(39188), UINT16_C(26389), UINT16_C(24552), UINT16_C(36854), UINT16_C(35056), UINT16_C(55196), UINT16_C( 9004), UINT16_C(25491) } },
    { { UINT16_C(61031), UINT16_C( 6143), UINT16_C(45008), UINT16_C(57419), UINT16_C( 9108), UINT16_C(11394), UINT16_C( 2778), UINT16_C(28206) },
      UINT8_C( 24),
      { UINT16_C(23679), UINT16_C(48897), UINT16_C(37527), UINT16_C(18836), UINT16_C(12479), UINT16_C(63802), UINT16_C( 7655), UINT16_C(37835) },
      { UINT16_C(11313), UINT16_C( 3876), UINT16_C( 7048), UINT16_C(40312), UINT16_C(40891), UINT16_C(43622), UINT16_C(15457), UINT16_C(51128) },
      { UINT16_C(61031), UINT16_C( 6143), UINT16_C(45008), UINT16_C(29574), UINT16_C(26685), UINT16_C(11394), UINT16_C( 2778), UINT16_C(28206) } },
    { { UINT16_C(36702), UINT16_C( 8916), UINT16_C(13843), UINT16_C(  647), UINT16_C(51785), UINT16_C( 6217), UINT16_C(27008), UINT16_C(20100) },
      UINT8_C(193),
      { UINT16_C( 3770), UINT16_C(45967), UINT16_C( 3106), UINT16_C( 6053), UINT16_C(60586), UINT16_C( 4509), UINT16_C(64909), UINT16_C( 6528) },
      { UINT16_C(61670), UINT16_C(48673), UINT16_C(32774), UINT16_C(51222), UINT16_C(41333), UINT16_C(28038), UINT16_C(19729), UINT16_C(45013) },
      { UINT16_C(32720), UINT16_C( 8916), UINT16_C(13843), UINT16_C(  647), UINT16_C(51785), UINT16_C( 6217), UINT16_C(42319), UINT16_C(25771) } },
    { { UINT16_C(22996), UINT16_C(64305), UINT16_C(47039), UINT16_C( 9690), UINT16_C(48265), UINT16_C( 8077), UINT16_C(32576), UINT16_C( 4691) },
      UINT8_C( 10),
      { UINT16_C(40983), UINT16_C(13302), UINT16_C(51251), UINT16_C(17264), UINT16_C(27536), UINT16_C(63937), UINT16_C(31547), UINT16_C(53750) },
      { UINT16_C(58466), UINT16_C(31079), UINT16_C( 3675), UINT16_C(25139), UINT16_C(33471), UINT16_C(50261), UINT16_C( 9774), UINT16_C(52132) },
      { UINT16_C(22996), UINT16_C(22191), UINT16_C(47039), UINT16_C(21202), UINT16_C(48265), UINT16_C( 8077), UINT16_C(32576), UINT16_C( 4691) } },
    { { UINT16_C(47876), UINT16_C(58639), UINT16_C( 7248), UINT16_C(41832), UINT16_C(59043), UINT16_C(26562), UINT16_C(56901), UINT16_C(36863) },
      UINT8_C(221),
      { UINT16_C(50607), UINT16_C( 5102), UINT16_C( 8349), UINT16_C(60550), UINT16_C(11356), UINT16_C( 8417), UINT16_C(45837), UINT16_C(24033) },
      { UINT16_C(35428), UINT16_C(56038), UINT16_C(56232), UINT16_C(61113), UINT16_C(49578), UINT16_C(26195), UINT16_C( 5136), UINT16_C(23463) },
      { UINT16_C(43018), UINT16_C(58639), UINT16_C(32291), UINT16_C(60832), UINT16_C(30467), UINT16_C(26562), UINT16_C(25487), UINT16_C(23748) } },
    { { UINT16_C(21351), UINT16_C(38854), UINT16_C(32515), UINT16_C(49947), UINT16_C(59713), UINT16_C(23314), UINT16_C(43905), UINT16_C(60752) },
      UINT8_C(183),
      { UINT16_C(28446), UINT16_C(18064), UINT16_C(45813), UINT16_C(36071), UINT16_C(31067), UINT16_C( 5165), UINT16_C(56876), UINT16_C(22693) },
      { UINT16_C(46420), UINT16_C(17039), UINT16_C(42271), UINT16_C(13345), UINT16_C( 2232), UINT16_C(37090), UINT16_C(43366), UINT16_C( 9416) },
      { UINT16_C(37433), UINT16_C(17552), UINT16_C(44042), UINT16_C(49947), UINT16_C(16650), UINT16_C(21128), UINT16_C(43905), UINT16_C(16055) } },
    { { UINT16_C(16717), UINT16_C(20503), UINT16_C(52611), UINT16_C(27592), UINT16_C(49225), UINT16_C( 3278), UINT16_C(52617), UINT16_C(58528) },
      UINT8_C(176),
      { UINT16_C(44561), UINT16_C(34177), UINT16_C( 2545), UINT16_C(54671), UINT16_C(36436), UINT16_C(40339), UINT16_C(55759), UINT16_C(60619) },
      { UINT16_C(40663), UINT16_C(41299), UINT16_C(63917), UINT16_C(18867), UINT16_C( 7798), UINT16_C(53641), UINT16_C(26059), UINT16_C(20753) },
      { UINT16_C(16717), UINT16_C(20503), UINT16_C(52611), UINT16_C(27592), UINT16_C(22117), UINT16_C(46990), UINT16_C(52617), UINT16_C(40686) } },
    { { UINT16_C(27456), UINT16_C(41153), UINT16_C(60032), UINT16_C(47330), UINT16_C( 2999), UINT16_C( 6670), UINT16_C(47672), UINT16_C(39638) },
      UINT8_C( 18),
      { UINT16_C(55338), UINT16_C(41088), UINT16_C(17079), UINT16_C(42423), UINT16_C(17315), UINT16_C( 3317), UINT16_C(53014), UINT16_C(  248) },
      { UINT16_C(45727), UINT16_C(38187), UINT16_C(29156), UINT16_C(50804), UINT16_C(38165), UINT16_C(60221), UINT16_C(33811), UINT16_C(59539) },
      { UINT16_C(27456), UINT16_C(39638), UINT16_C(60032), UINT16_C(47330), UINT16_C(27740), UINT16_C( 6670), UINT16_C(47672), UINT16_C(39638) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i src = simde_mm_loadu_epi16(test_vec[i].src);
    simde__m128i a = simde_mm_loadu_epi16(test_vec[i].a);
    simde__m128i b = simde_mm_loadu_epi16(test_vec[i].b);
    simde__m128i r = simde_mm_mask_avg_epu16(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_u16x8(r, simde_mm_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm_maskz_avg_epu16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask8 k;
    const uint16_t a[8];
    const uint16_t b[8];
    const uint16_t r[8];
  } test_vec[] = {
    { UINT8_C(163),
      { UINT16_C(48032), UINT16_C(55188), UINT16_C(12951), UINT16_C(45863), UINT16_C( 7585), UINT16_C( 6666), UINT16_C(10347), UINT16_C(21784) },
      { UINT16_C(55826), UINT16_C(23098), UINT16_C(54925), UINT16_C(54432), UINT16_C(28629), UINT16_C( 3981), UINT16_C(45286), UINT16_C(45278) },
      { UINT16_C(51929), UINT16_C(39143), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C( 5324), UINT16_C(    0), UINT16_C(33531) } },
    { UINT8_C( 32),
      { UINT16_C( 7784), UINT16_C(51464), UINT16_C(65205), UINT16_C(30372), UINT16_C(30102), UINT16_C(45454), UINT16_C(17167), UINT16_C(12381) },
      { UINT16_C(35010), UINT16_C(19204), UINT16_C(18245), UINT16_C(19767), UINT16_C(24345), UINT16_C(35860), UINT16_C( 4547), UINT16_C(38580) },
      { UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(40657), UINT16_C(    0), UINT16_C(    0) } },
    { UINT8_C(150),
      { UINT16_C(65141), UINT16_C(20853), UINT16_C(29926), UINT16_C(21344), UINT16_C(51024), UINT16_C(44582), UINT16_C(52777), UINT16_C(58439) },
      { UINT16_C(15050), UINT16_C(49285), UINT16_C(22104), UINT16_C(30833), UINT16_C( 7402), UINT16_C(45888), UINT16_C( 6809), UINT16_C(18470) },
      { UINT16_C(    0), UINT16_C(35069), UINT16_C(26015), UINT16_C(    0), UINT16_C(29213), UINT16_C(    0), UINT16_C(    0), UINT16_C(38455) } },
    { UINT8_C(207),
      { UINT16_C(49291), UINT16_C(31616), UINT16_C(45010), UINT16_C(26247), UINT16_C(28118), UINT16_C(33579), UINT16_C(29123), UINT16_C(49910) },
      { UINT16_C(19996), UINT16_C(51909), UINT16_C( 8994), UINT16_C(39237), UINT16_C( 5756), UINT16_C(50924), UINT16_C(22322), UINT16_C(21862) },
      { UINT16_C(34644), UINT16_C(41763), UINT16_C(27002), UINT16_C(32742), UINT16_C(    0), UINT16_C(    0), UINT16_C(25723), UINT16_C(35886) } },
    { UINT8_C(217),
      { UINT16_C(34614), UINT16_C(47250), UINT16_C( 7698), UINT16_C(16686), UINT16_C(53181), UINT16_C(59815), UINT16_C(39047), UINT16_C(26958) },
      { UINT16_C(60660), UINT16_C(17223), UINT16_C(53149), UINT16_C(21460), UINT16_C(41601), UINT16_C(27295), UINT16_C(22351), UINT16_C( 2906) },
      { UINT16_C(47637), UINT16_C(    0), UINT16_C(    0), UINT16_C(19073), UINT16_C(47391), UINT16_C(    0), UINT16_C(30699), UINT16_C(14932) } },
    { UINT8_C(  4),
      { UINT16_C(20478), UINT16_C(10512), UINT16_C(20701), UINT16_C(45563), UINT16_C(21448), UINT16_C( 1776), UINT16_C(12739), UINT16_C(18480) },
      { UINT16_C( 8358), UINT16_C(19044), UINT16_C(27967), UINT16_C(59654), UINT16_C( 6723), UINT16_C(19513), UINT16_C( 3306), UINT16_C(57872) },
      { UINT16_C(    0), UINT16_C(    0), UINT16_C(24334), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0) } },
    { UINT8_C( 17),
      { UINT16_C(22178), UINT16_C(37794), UINT16_C(40419), UINT16_C(55432), UINT16_C(49929), UINT16_C(16963), UINT16_C(37167), UINT16_C(13772) },
      { UINT16_C(25210), UINT16_C(39450), UINT16_C(28240), UINT16_C(19007), UINT16_C(62966), UINT16_C(38882), UINT16_C(59033), UINT16_C(16682) },
      { UINT16_C(23694), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(56448), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0) } },
    { UINT8_C( 86),
      { UINT16_C( 8663), UINT16_C(52000), UINT16_C(12269), UINT16_C(20492), UINT16_C(56656), UINT16_C(28723), UINT16_C(38241), UINT16_C(26297) },
      { UINT16_C(20566), UINT16_C(52418), UINT16_C( 1010), UINT16_C(36269), UINT16_C(50768), UINT16_C(38701), UINT16_C( 4452), UINT16_C(46909) },
      { UINT16_C(    0), UINT16_C(52209), UINT16_C( 6640), UINT16_C(    0), UINT16_C(53712), UINT16_C(    0), UINT16_C(21347), UINT16_C(    0) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i a = simde_mm_loadu_epi16(test_vec[i].a);
    simde__m128i b = simde_mm_loadu_epi16(test_vec[i].b);
    simde__m128i r = simde_mm_maskz_avg_epu16(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_u16x8(r, simde_mm_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_mask_avg_epu8 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const uint8_t src[32];
    const simde__mmask32 k;
    const uint8_t a[32];
    const uint8_t b[32];
    const uint8_t r[32];
  } test_vec[] = {
    { { UINT8_C( 43), UINT8_C(217), UINT8_C(123), UINT8_C(234), UINT8_C( 47), UINT8_C(229), UINT8_C(173), UINT8_C(208),
        UINT8_C(101), UINT8_C( 58), UINT8_C( 95), UINT8_C( 33), UINT8_C(132), UINT8_C( 71), UINT8_C(220), UINT8_C(101),
        UINT8_C( 39), UINT8_C(213), UINT8_C( 74), UINT8_C(240), UINT8_C(189), UINT8_C(241), UINT8_C(247), UINT8_C(106),
        UINT8_C(168), UINT8_C(218), UINT8_C(210), UINT8_C(243), UINT8_C(145), UINT8_C(135), UINT8_C( 24), UINT8_C(188) },
      UINT32_C(2426835809),
      { UINT8_C(120), UINT8_C( 84), UINT8_C( 96), UINT8_C(222), UINT8_C(142), UINT8_C(191),    UINT8_MAX, UINT8_C( 18),
        UINT8_C(  6), UINT8_C(220), UINT8_C(119), UINT8_C( 46), UINT8_C(177), UINT8_C(194), UINT8_C( 30), UINT8_C(111),
        UINT8_C(179), UINT8_C( 21), UINT8_C(217), UINT8_C( 91), UINT8_C(239), UINT8_C(171), UINT8_C( 79), UINT8_C(129),
        UINT8_C( 51), UINT8_C(103), UINT8_C( 61), UINT8_C(148), UINT8_C(251), UINT8_C(228), UINT8_C( 36), UINT8_C(115) },
      { UINT8_C( 56), UINT8_C(132), UINT8_C( 81), UINT8_C(198), UINT8_C( 67), UINT8_C( 81), UINT8_C(216), UINT8_C( 73),
        UINT8_C( 45), UINT8_C( 79), UINT8_C(119), UINT8_C(222), UINT8_C( 17), UINT8_C(150), UINT8_C( 77), UINT8_C(197),
        UINT8_C(171), UINT8_C( 38), UINT8_C( 32), UINT8_C(155), UINT8_C(210), UINT8_C(111), UINT8_C( 28), UINT8_C(  5),
        UINT8_C(215), UINT8_C( 89), UINT8_C(153), UINT8_C(210), UINT8_C( 61), UINT8_C(189), UINT8_C( 69), UINT8_C(117) },
      { UINT8_C( 88), UINT8_C(217), UINT8_C(123), UINT8_C(234), UINT8_C( 47), UINT8_C(136), UINT8_C(236), UINT8_C(208),
        UINT8_C( 26), UINT8_C(150), UINT8_C( 95), UINT8_C( 33), UINT8_C( 97), UINT8_C( 71), UINT8_C(220), UINT8_C(154),
        UINT8_C( 39), UINT8_C( 30), UINT8_C(125), UINT8_C(240), UINT8_C(189), UINT8_C(141), UINT8_C(247), UINT8_C( 67),
        UINT8_C(168), UINT8_C(218), UINT8_C(210), UINT8_C(243), UINT8_C(156), UINT8_C(135), UINT8_C( 24), UINT8_C(116) } },
    { { UINT8_C( 65), UINT8_C(151), UINT8_C( 59), UINT8_C(132), UINT8_C(232), UINT8_C( 19), UINT8_C(205), UINT8_C( 21),
        UINT8_C( 99), UINT8_C( 69), UINT8_C(243), UINT8_C(116), UINT8_C(219), UINT8_C( 65), UINT8_C( 57), UINT8_C(134),
        UINT8_C(103), UINT8_C( 90), UINT8_C( 33), UINT8_C( 57), UINT8_C(201), UINT8_C( 61), UINT8_C( 62), UINT8_C(160),
        UINT8_C(151), UINT8_C(215), UINT8_C(114), UINT8_C(212), UINT8_C(148), UINT8_C(184), UINT8_C( 74), UINT8_C(213) },
      UINT32_C( 928613711),
      { UINT8_C(153), UINT8_C( 39), UINT8_C( 76), UINT8_C(252), UINT8_C(108), UINT8_C( 63), UINT8_C(112), UINT8_C( 71),
        UINT8_C(128), UINT8_C(170), UINT8_C(205), UINT8_C(232), UINT8_C(  4), UINT8_C(239), UINT8_C( 33), UINT8_C(205),
        UINT8_C( 44), UINT8_C( 96), UINT8_C(110), UINT8_C(195), UINT8_C( 55), UINT8_C(224), UINT8_C(152), UINT8_C(204),
        UINT8_C(152), UINT8_C(226), UINT8_C(161), UINT8_C(231), UINT8_C(103), UINT8_C(251), UINT8_C( 30), UINT8_C(  0) },
      { UINT8_C( 34), UINT8_C(106), UINT8_C(252), UINT8_C(142), UINT8_C(170), UINT8_C(109), UINT8_C(213), UINT8_C( 42),
        UINT8_C( 23), UINT8_C(162), UINT8_C( 18), UINT8_C( 27), UINT8_C(145), UINT8_C( 52), UINT8_C(232), UINT8_C(190),
        UINT8_C(148), UINT8_C( 86), UINT8_C(129), UINT8_C(203), UINT8_C( 55), UINT8_C( 25), UINT8_C(151), UINT8_C(207),
        UINT8_C(251), UINT8_C( 57), UINT8_C(183), UINT8_C( 99), UINT8_C( 52), UINT8_C(213), UINT8_C( 99), UINT8_C( 86) },
      { UINT8_C( 94), UINT8_C( 73), UINT8_C(164), UINT8_C(197), UINT8_C(232), UINT8_C( 19), UINT8_C(163), UINT8_C( 21),
        UINT8_C( 76), UINT8_C( 69), UINT8_C(112), UINT8_C(116), UINT8_C(219), UINT8_C( 65), UINT8_C( 57), UINT8_C(198),
        UINT8_C( 96), UINT8_C( 90), UINT8_C( 33), UINT8_C(199), UINT8_C( 55), UINT8_C( 61), UINT8_C(152), UINT8_C(160),
        UINT8_C(202), UINT8_C(142), UINT8_C(172), UINT8_C(212), UINT8_C( 78), UINT8_C(232), UINT8_C( 74), UINT8_C(213) } },
    { { UINT8_C( 64), UINT8_C( 96), UINT8_C(228), UINT8_C(234), UINT8_C(205), UINT8_C(185), UINT8_C( 20), UINT8_C(228),
        UINT8_C( 91), UINT8_C( 39),    UINT8_MAX, UINT8_C(237), UINT8_C( 91), UINT8_C(231), UINT8_C(171), UINT8_C(239),
        UINT8_C( 62), UINT8_C( 44), UINT8_C(186), UINT8_C(117), UINT8_C( 70), UINT8_C( 82), UINT8_C( 68), UINT8_C( 65),
        UINT8_C(139), UINT8_C(251), UINT8_C(164), UINT8_C(191), UINT8_C(209), UINT8_C(  8), UINT8_C( 21), UINT8_C( 17) },
      UINT32_C( 905705832),
      { UINT8_C(178), UINT8_C( 15), UINT8_C( 25), UINT8_C( 13), UINT8_C( 54), UINT8_C( 24), UINT8_C(250), UINT8_C(145),
           UINT8_MAX, UINT8_C(165), UINT8_C(128), UINT8_C( 61), UINT8_C(210), UINT8_C( 59), UINT8_C(178), UINT8_C( 24),
        UINT8_C(141), UINT8_C(247), UINT8_C( 89), UINT8_C( 24), UINT8_C(242), UINT8_C(254), UINT8_C(215), UINT8_C(195),
        UINT8_C(  6), UINT8_C(236), UINT8_C(212), UINT8_C(110), UINT8_C(229), UINT8_C(207), UINT8_C(163), UINT8_C(151) },
      { UINT8_C(223), UINT8_C(188), UINT8_C(164), UINT8_C( 21), UINT8_C(212), UINT8_C(159), UINT8_C(167), UINT8_C(211),
        UINT8_C( 68), UINT8_C( 39), UINT8_C( 17), UINT8_C( 22), UINT8_C( 98), UINT8_C(195), UINT8_C( 46), UINT8_C(239),
        UINT8_C(186), UINT8_C(136), UINT8_C(  7), UINT8_C(173), UINT8_C(134), UINT8_C(222), UINT8_C(112), UINT8_C(140),
        UINT8_C(202), UINT8_C( 69), UINT8_C(250), UINT8_C(175), UINT8_C( 20), UINT8_C(157), UINT8_C( 70), UINT8_C(243) },
      { UINT8_C( 64), UINT8_C( 96), UINT8_C(228), UINT8_C( 17), UINT8_C(205), UINT8_C( 92), UINT8_C(209), UINT8_C(228),
        UINT8_C(162), UINT8_C( 39),    UINT8_MAX, UINT8_C( 42), UINT8_C(154), UINT8_C(127), UINT8_C(112), UINT8_C(132),
        UINT8_C(164), UINT8_C(192), UINT8_C(186), UINT8_C( 99), UINT8_C(188), UINT8_C(238), UINT8_C(164), UINT8_C(168),
        UINT8_C(104), UINT8_C(251), UINT8_C(231), UINT8_C(191), UINT8_C(125), UINT8_C(182), UINT8_C( 21), UINT8_C( 17) } },
    { { UINT8_C( 89), UINT8_C(235), UINT8_C(  9), UINT8_C( 45), UINT8_C(138), UINT8_C(176), UINT8_C(  0), UINT8_C(206),
        UINT8_C(215), UINT8_C( 17), UINT8_C(229), UINT8_C( 58), UINT8_C(213), UINT8_C( 19), UINT8_C( 41), UINT8_C(143),
        UINT8_C(155), UINT8_C( 49), UINT8_C( 60), UINT8_C( 33), UINT8_C( 15), UINT8_C(173), UINT8_C(173), UINT8_C(218),
        UINT8_C(242), UINT8_C(167), UINT8_C(137), UINT8_C(  6), UINT8_C( 68), UINT8_C(208), UINT8_C(250), UINT8_C(157) },
      UINT32_C(1170867131),
      { UINT8_C(179), UINT8_C(203), UINT8_C( 19), UINT8_C(138), UINT8_C(220), UINT8_C(248), UINT8_C(196), UINT8_C(177),
        UINT8_C( 12), UINT8_C(238), UINT8_C( 65), UINT8_C(167), UINT8_C( 31), UINT8_C(125), UINT8_C(201), UINT8_C( 46),
        UINT8_C( 42), UINT8_C(118), UINT8_C(  8), UINT8_C( 28), UINT8_C( 30), UINT8_C(146), UINT8_C( 35), UINT8_C( 98),
        UINT8_C( 98), UINT8_C( 29), UINT8_C(  0), UINT8_C( 29), UINT8_C( 32), UINT8_C(202), UINT8_C( 98), UINT8_C(211) },
      { UINT8_C(149), UINT8_C(117), UINT8_C( 93), UINT8_C(114), UINT8_C(110), UINT8_C( 34), UINT8_C( 35), UINT8_C(122),
        UINT8_C( 16), UINT8_C(100), UINT8_C( 33), UINT8_C( 47), UINT8_C(226), UINT8_C(234), UINT8_C( 93), UINT8_C( 12),
        UINT8_C( 97), UINT8_C(102), UINT8_C( 41), UINT8_C(127), UINT8_C(248), UINT8_C( 76), UINT8_C(225), UINT8_C( 90),
        UINT8_C(105), UINT8_C(225), UINT8_C(119), UINT8_C(137), UINT8_C(172), UINT8_C(217), UINT8_C( 92), UINT8_C( 65) },
      { UINT8_C(164), UINT8_C(160), UINT8_C(  9), UINT8_C(126), UINT8_C(165), UINT8_C(141), UINT8_C(  0), UINT8_C(150),
        UINT8_C( 14), UINT8_C(169), UINT8_C(229), UINT8_C( 58), UINT8_C(213), UINT8_C( 19), UINT8_C( 41), UINT8_C(143),
        UINT8_C(155), UINT8_C(110), UINT8_C( 60), UINT8_C( 78), UINT8_C( 15), UINT8_C(173), UINT8_C(130), UINT8_C( 94),
        UINT8_C(102), UINT8_C(167), UINT8_C( 60), UINT8_C(  6), UINT8_C( 68), UINT8_C(208), UINT8_C( 95), UINT8_C(157) } },
    { { UINT8_C( 78), UINT8_C(185), UINT8_C(179), UINT8_C(188), UINT8_C(219), UINT8_C(215), UINT8_C( 54), UINT8_C(235),
        UINT8_C( 59), UINT8_C( 88), UINT8_C( 26), UINT8_C( 29), UINT8_C( 66), UINT8_C(120), UINT8_C( 42), UINT8_C(163),
        UINT8_C(222), UINT8_C( 83), UINT8_C( 34), UINT8_C(214), UINT8_C(159), UINT8_C(  4), UINT8_C( 48), UINT8_C(  8),
        UINT8_C(229), UINT8_C(167), UINT8_C(145), UINT8_C(145), UINT8_C(128), UINT8_C(237), UINT8_C(211), UINT8_C(206) },
      UINT32_C(2190182054),
      { UINT8_C( 93), UINT8_C(193), UINT8_C(109), UINT8_C(153), UINT8_C( 25), UINT8_C(136), UINT8_C(182), UINT8_C( 92),
        UINT8_C(  0), UINT8_C(224),    UINT8_MAX, UINT8_C(222), UINT8_C( 51), UINT8_C( 34), UINT8_C(180), UINT8_C(210),
        UINT8_C( 38), UINT8_C(228), UINT8_C(218), UINT8_C( 11), UINT8_C(139), UINT8_C(107), UINT8_C(157), UINT8_C( 11),
        UINT8_C( 88), UINT8_C(112), UINT8_C(217),    UINT8_MAX, UINT8_C(246), UINT8_C(100), UINT8_C(129), UINT8_C( 84) },
      { UINT8_C( 38), UINT8_C(238), UINT8_C(237), UINT8_C( 63), UINT8_C(118), UINT8_C(163), UINT8_C(155), UINT8_C(118),
        UINT8_C(132), UINT8_C(155), UINT8_C( 84), UINT8_C(183), UINT8_C(189), UINT8_C(  8), UINT8_C(138), UINT8_C(227),
        UINT8_C(236), UINT8_C(100), UINT8_C(238), UINT8_C(119), UINT8_C(208), UINT8_C(139), UINT8_C(130), UINT8_C( 40),
        UINT8_C(251), UINT8_C( 92), UINT8_C( 39), UINT8_C(242), UINT8_C(192), UINT8_C(168), UINT8_C( 70), UINT8_C(230) },
      { UINT8_C( 78), UINT8_C(216), UINT8_C(173), UINT8_C(188), UINT8_C(219), UINT8_C(150), UINT8_C( 54), UINT8_C(105),
        UINT8_C( 59), UINT8_C(190), UINT8_C(170), UINT8_C( 29), UINT8_C( 66), UINT8_C(120), UINT8_C( 42), UINT8_C(219),
        UINT8_C(137), UINT8_C(164), UINT8_C( 34), UINT8_C( 65), UINT8_C(159), UINT8_C(  4), UINT8_C( 48), UINT8_C( 26),
        UINT8_C(229), UINT8_C(102), UINT8_C(145), UINT8_C(145), UINT8_C(128), UINT8_C(237), UINT8_C(211), UINT8_C(157) } },
    { { UINT8_C(151), UINT8_C( 51), UINT8_C( 38), UINT8_C( 13), UINT8_C(214), UINT8_C(193), UINT8_C(132), UINT8_C( 90),
        UINT8_C( 92), UINT8_C(216), UINT8_C( 18), UINT8_C( 25), UINT8_C(225), UINT8_C(156), UINT8_C(252), UINT8_C(205),
        UINT8_C(  0), UINT8_C(235), UINT8_C( 69), UINT8_C(208), UINT8_C(118), UINT8_C(199), UINT8_C(249), UINT8_C(114),
        UINT8_C( 35), UINT8_C( 32), UINT8_C(100), UINT8_C(228), UINT8_C(201), UINT8_C(170), UINT8_C(202), UINT8_C( 96) },
      UINT32_C(3010326749),
      { UINT8_C(178), UINT8_C(241), UINT8_C( 14), UINT8_C( 14), UINT8_C(202), UINT8_C( 32), UINT8_C( 40), UINT8_C(171),
        UINT8_C(188), UINT8_C( 36), UINT8_C(120), UINT8_C(188), UINT8_C( 15), UINT8_C(189), UINT8_C(141), UINT8_C(134),
        UINT8_C(133), UINT8_C(134), UINT8_C(248), UINT8_C(168), UINT8_C(166), UINT8_C( 92), UINT8_C(140), UINT8_C(111),
        UINT8_C(  6), UINT8_C( 87), UINT8_C(207), UINT8_C(227), UINT8_C( 71), UINT8_C( 61), UINT8_C(150), UINT8_C(249) },
      { UINT8_C( 46), UINT8_C(164), UINT8_C(  8), UINT8_C(248), UINT8_C(196), UINT8_C( 48), UINT8_C(163), UINT8_C(128),
        UINT8_C( 84), UINT8_C( 28), UINT8_C( 61), UINT8_C(100), UINT8_C(217), UINT8_C(202), UINT8_C(234), UINT8_C( 94),
        UINT8_C( 80), UINT8_C(226), UINT8_C(  7), UINT8_C(246), UINT8_C( 62), UINT8_C(147), UINT8_C(102), UINT8_C( 68),
        UINT8_C(234), UINT8_C( 53), UINT8_C( 39), UINT8_C( 50), UINT8_C(114), UINT8_C(189), UINT8_C( 43), UINT8_C(161) },
      { UINT8_C(112), UINT8_C( 51), UINT8_C( 11), UINT8_C(131), UINT8_C(199), UINT8_C(193), UINT8_C(102), UINT8_C(150),
        UINT8_C( 92), UINT8_C(216), UINT8_C( 18), UINT8_C( 25), UINT8_C(116), UINT8_C(196), UINT8_C(188), UINT8_C(114),
        UINT8_C(107), UINT8_C(235), UINT8_C(128), UINT8_C(207), UINT8_C(118), UINT8_C(120), UINT8_C(121), UINT8_C(114),
        UINT8_C(120), UINT8_C( 70), UINT8_C(100), UINT8_C(228), UINT8_C( 93), UINT8_C(125), UINT8_C(202), UINT8_C(205) } },
    { { UINT8_C( 98), UINT8_C( 51), UINT8_C(153), UINT8_C( 38), UINT8_C( 99), UINT8_C( 61), UINT8_C(167), UINT8_C(184),
        UINT8_C( 89), UINT8_C(228), UINT8_C( 28), UINT8_C( 50), UINT8_C(174), UINT8_C(  6), UINT8_C(145), UINT8_C(254),
        UINT8_C(232), UINT8_C(152), UINT8_C(244), UINT8_C( 38), UINT8_C( 43), UINT8_C( 90), UINT8_C(106), UINT8_C( 22),
        UINT8_C(144), UINT8_C(145), UINT8_C( 72), UINT8_C(  2), UINT8_C( 78), UINT8_C(115), UINT8_C(163), UINT8_C(176) },
      UINT32_C( 181878183),
      { UINT8_C(122), UINT8_C(126), UINT8_C(194), UINT8_C(211), UINT8_C( 98), UINT8_C(222), UINT8_C(  5), UINT8_C( 16),
        UINT8_C(228), UINT8_C(150), UINT8_C( 14), UINT8_C(204), UINT8_C( 46), UINT8_C(  2), UINT8_C(242), UINT8_C( 90),
        UINT8_C( 93), UINT8_C( 92), UINT8_C(112), UINT8_C(237), UINT8_C(237), UINT8_C(184), UINT8_C(239), UINT8_C( 60),
        UINT8_C( 43), UINT8_C(147), UINT8_C(236), UINT8_C(210), UINT8_C(208), UINT8_C(195), UINT8_C(221), UINT8_C( 74) },
      { UINT8_C( 65), UINT8_C(159), UINT8_C( 29), UINT8_C(163), UINT8_C(126), UINT8_C( 34), UINT8_C(179), UINT8_C( 98),
        UINT8_C(185), UINT8_C(193), UINT8_C( 47), UINT8_C(231), UINT8_C(196), UINT8_C( 33), UINT8_C( 65), UINT8_C( 33),
        UINT8_C(126), UINT8_C(177), UINT8_C( 14), UINT8_C(107), UINT8_C(105), UINT8_C(253), UINT8_C(167), UINT8_C(149),
        UINT8_C(144), UINT8_C(148), UINT8_C(103), UINT8_C( 96), UINT8_C( 87), UINT8_C( 68), UINT8_C(170), UINT8_C(153) },
      { UINT8_C( 94), UINT8_C(143), UINT8_C(112), UINT8_C( 38), UINT8_C( 99), UINT8_C(128), UINT8_C(167), UINT8_C( 57),
        UINT8_C(207), UINT8_C(228), UINT8_C( 31), UINT8_C(218), UINT8_C(121), UINT8_C( 18), UINT8_C(145), UINT8_C(254),
        UINT8_C(110), UINT8_C(135), UINT8_C( 63), UINT8_C( 38), UINT8_C(171), UINT8_C( 90), UINT8_C(203), UINT8_C(105),
        UINT8_C(144), UINT8_C(148), UINT8_C( 72), UINT8_C(153), UINT8_C( 78), UINT8_C(115), UINT8_C(163), UINT8_C(176) } },
    { { UINT8_C(228), UINT8_C(199), UINT8_C( 60), UINT8_C( 98), UINT8_C(234), UINT8_C(240), UINT8_C(196), UINT8_C(163),
        UINT8_C(177), UINT8_C(243), UINT8_C(138), UINT8_C(117), UINT8_C( 21), UINT8_C(204), UINT8_C(150), UINT8_C(147),
        UINT8_C(125), UINT8_C(164), UINT8_C(254), UINT8_C(231), UINT8_C(162), UINT8_C(166), UINT8_C(124), UINT8_C( 50),
        UINT8_C( 58), UINT8_C(227), UINT8_C(147), UINT8_C(145), UINT8_C( 40), UINT8_C( 61), UINT8_C( 42), UINT8_C( 12) },
      UINT32_C(4016989957),
      { UINT8_C( 87), UINT8_C( 50), UINT8_C(146), UINT8_C(  8), UINT8_C( 38), UINT8_C( 28), UINT8_C(126), UINT8_C( 59),
        UINT8_C(232), UINT8_C( 20), UINT8_C(206), UINT8_C(102), UINT8_C(185), UINT8_C(204), UINT8_C( 77), UINT8_C( 91),
        UINT8_C(114), UINT8_C(201), UINT8_C(141), UINT8_C(172), UINT8_C(172), UINT8_C( 32), UINT8_C( 62), UINT8_C(212),
        UINT8_C( 94), UINT8_C(104), UINT8_C(224), UINT8_C( 99), UINT8_C(207), UINT8_C( 78), UINT8_C( 82), UINT8_C( 38) },
      { UINT8_C(129), UINT8_C(228), UINT8_C( 47), UINT8_C(167), UINT8_C(  0), UINT8_C(173), UINT8_C(226), UINT8_C(233),
        UINT8_C(193), UINT8_C(176), UINT8_C( 79), UINT8_C(122), UINT8_C(124), UINT8_C(156), UINT8_C(213), UINT8_C(239),
        UINT8_C(101), UINT8_C( 99), UINT8_C(155), UINT8_C( 17), UINT8_C(131), UINT8_C(217), UINT8_C(230), UINT8_C(225),
        UINT8_C( 66), UINT8_C(198), UINT8_C( 68), UINT8_C( 17), UINT8_C( 21), UINT8_C(150), UINT8_C( 56), UINT8_C(150) },
      { UINT8_C(108), UINT8_C(199), UINT8_C( 97), UINT8_C( 98), UINT8_C(234), UINT8_C(240), UINT8_C(196), UINT8_C(163),
        UINT8_C(213), UINT8_C( 98), UINT8_C(143), UINT8_C(117), UINT8_C( 21), UINT8_C(180), UINT8_C(145), UINT8_C(147),
        UINT8_C(125), UINT8_C(150), UINT8_C(148), UINT8_C( 95), UINT8_C(162), UINT8_C(125), UINT8_C(146), UINT8_C( 50),
        UINT8_C( 80), UINT8_C(151), UINT8_C(146), UINT8_C( 58), UINT8_C( 40), UINT8_C(114), UINT8_C( 69), UINT8_C( 94) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i src = simde_x_mm256_loadu_epi8(test_vec[i].src);
    simde__m256i a = simde_x_mm256_loadu_epi8(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi8(test_vec[i].b);
    simde__m256i r = simde_mm256_mask_avg_epu8(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i8x32(r, simde_x_mm256_loadu_epi8(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_maskz_avg_epu8 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask32 k;
    const uint8_t a[32];
    const uint8_t b[32];
    const uint8_t r[32];
  } test_vec[] = {
    { UINT32_C(3998241187),
      { UINT8_C(128), UINT8_C(226), UINT8_C(235), UINT8_C(237), UINT8_C(106), UINT8_C(224), UINT8_C(254), UINT8_C(106),
        UINT8_C(178), UINT8_C(196), UINT8_C(101), UINT8_C(242), UINT8_C( 83), UINT8_C(193), UINT8_C(154), UINT8_C(225),
        UINT8_C(118), UINT8_C( 15), UINT8_C( 70), UINT8_C(198), UINT8_C(232), UINT8_C( 76), UINT8_C(  2), UINT8_C(132),
        UINT8_C( 51), UINT8_C(110), UINT8_C( 22), UINT8_C(214), UINT8_C(191), UINT8_C(102), UINT8_C(196), UINT8_C( 64) },
      { UINT8_C( 72), UINT8_C(176), UINT8_C( 45), UINT8_C(178), UINT8_C(144), UINT8_C( 43), UINT8_C( 28), UINT8_C( 66),
        UINT8_C(239), UINT8_C(130), UINT8_C( 52), UINT8_C( 66), UINT8_C( 67), UINT8_C(207), UINT8_C( 35), UINT8_C(186),
        UINT8_C(222), UINT8_C(105), UINT8_C(128), UINT8_C(198), UINT8_C(181), UINT8_C(131), UINT8_C( 74), UINT8_C(232),
        UINT8_C(241), UINT8_C( 96), UINT8_C(190), UINT8_C(177), UINT8_C(198), UINT8_C(131), UINT8_C(241), UINT8_C( 14) },
      { UINT8_C(100), UINT8_C(201), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(134), UINT8_C(  0), UINT8_C( 86),
        UINT8_C(209), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C( 75), UINT8_C(  0), UINT8_C( 95), UINT8_C(  0),
        UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(207), UINT8_C(  0), UINT8_C( 38), UINT8_C(  0),
        UINT8_C(  0), UINT8_C(103), UINT8_C(106), UINT8_C(196), UINT8_C(  0), UINT8_C(117), UINT8_C(219), UINT8_C( 39) } },
    { UINT32_C(3284147763),
      { UINT8_C( 73), UINT8_C(221), UINT8_C(  6), UINT8_C( 56), UINT8_C( 95), UINT8_C( 58), UINT8_C(123), UINT8_C(162),
        UINT8_C(  9), UINT8_C(158), UINT8_C( 92), UINT8_C(231), UINT8_C(  8), UINT8_C(221), UINT8_C(174), UINT8_C(189),
        UINT8_C( 96), UINT8_C(248), UINT8_C(166), UINT8_C( 81), UINT8_C( 89), UINT8_C(100), UINT8_C(  2), UINT8_C( 31),
        UINT8_C(231), UINT8_C(243), UINT8_C( 46), UINT8_C( 26), UINT8_C( 17), UINT8_C(238), UINT8_C(222), UINT8_C( 90) },
      { UINT8_C(203), UINT8_C(228), UINT8_C(147), UINT8_C( 42), UINT8_C( 30), UINT8_C( 14), UINT8_C(205), UINT8_C( 40),
        UINT8_C(172), UINT8_C( 41), UINT8_C( 15), UINT8_C(180), UINT8_C(  6), UINT8_C(189), UINT8_C(114), UINT8_C(102),
        UINT8_C(182), UINT8_C( 24), UINT8_C(184), UINT8_C( 15), UINT8_C(124), UINT8_C(186), UINT8_C( 46), UINT8_C(100),
        UINT8_C(174), UINT8_C( 92), UINT8_C(126), UINT8_C(191), UINT8_C( 75), UINT8_C( 92), UINT8_C( 26), UINT8_C( 22) },
      { UINT8_C(138), UINT8_C(225), UINT8_C(  0), UINT8_C(  0), UINT8_C( 63), UINT8_C( 36), UINT8_C(  0), UINT8_C(  0),
        UINT8_C(  0), UINT8_C(100), UINT8_C( 54), UINT8_C(206), UINT8_C(  7), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
        UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C( 24), UINT8_C( 66),
        UINT8_C(203), UINT8_C(168), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(124), UINT8_C( 56) } },
    { UINT32_C(1598139712),
      { UINT8_C(187), UINT8_C( 14), UINT8_C(135), UINT8_C(103), UINT8_C( 55), UINT8_C(150), UINT8_C( 28), UINT8_C( 62),
        UINT8_C( 84), UINT8_C(142), UINT8_C(164), UINT8_C( 10), UINT8_C(166), UINT8_C( 92), UINT8_C( 25), UINT8_C( 34),
        UINT8_C( 23), UINT8_C( 71), UINT8_C(134), UINT8_C(197), UINT8_C(164), UINT8_C(  5), UINT8_C(132), UINT8_C(239),
        UINT8_C( 97), UINT8_C(158), UINT8_C(  5), UINT8_C(162), UINT8_C( 75), UINT8_C( 70), UINT8_C(  1), UINT8_C(  6) },
      { UINT8_C( 84), UINT8_C(136), UINT8_C(110), UINT8_C(140), UINT8_C( 30), UINT8_C(138), UINT8_C(202), UINT8_C(114),
        UINT8_C( 24), UINT8_C(110), UINT8_C(124), UINT8_C(190), UINT8_C(203), UINT8_C(149), UINT8_C(224), UINT8_C(226),
        UINT8_C(221), UINT8_C(103), UINT8_C(167), UINT8_C(129), UINT8_C(108), UINT8_C( 43), UINT8_C(112), UINT8_C(205),
        UINT8_C(202), UINT8_C(117), UINT8_C(111), UINT8_C( 21), UINT8_C(188), UINT8_C(112), UINT8_C( 28), UINT8_C( 16) },
      { UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(115), UINT8_C(  0),
        UINT8_C( 54), UINT8_C(  0), UINT8_C(144), UINT8_C(100), UINT8_C(  0), UINT8_C(121), UINT8_C(  0), UINT8_C(130),
        UINT8_C(122), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(122), UINT8_C(  0),
        UINT8_C(150), UINT8_C(138), UINT8_C( 58), UINT8_C( 92), UINT8_C(132), UINT8_C(  0), UINT8_C( 15), UINT8_C(  0) } },
    { UINT32_C( 396135160),
      { UINT8_C( 20), UINT8_C(102), UINT8_C(137), UINT8_C( 44), UINT8_C(213), UINT8_C(  6), UINT8_C(234), UINT8_C(160),
        UINT8_C(155), UINT8_C(202), UINT8_C(130), UINT8_C(120), UINT8_C( 49), UINT8_C( 41), UINT8_C(249), UINT8_C(157),
        UINT8_C( 84), UINT8_C(105), UINT8_C(107), UINT8_C( 30), UINT8_C(223), UINT8_C(218), UINT8_C( 52), UINT8_C(155),
        UINT8_C( 75), UINT8_C( 80), UINT8_C(171), UINT8_C( 67), UINT8_C(218), UINT8_C( 72), UINT8_C( 90), UINT8_C(238) },
      { UINT8_C(174), UINT8_C(228), UINT8_C( 26), UINT8_C(131), UINT8_C(234), UINT8_C(  4), UINT8_C( 35), UINT8_C(133),
        UINT8_C(206), UINT8_C(165), UINT8_C(254), UINT8_C(  0), UINT8_C(206), UINT8_C(247), UINT8_C(157), UINT8_C( 35),
        UINT8_C( 97), UINT8_C(  8), UINT8_C( 65), UINT8_C( 64), UINT8_C(227), UINT8_C(117), UINT8_C(219), UINT8_C( 46),
        UINT8_C(197), UINT8_C(134), UINT8_C(113), UINT8_C(159), UINT8_C(206), UINT8_C(204), UINT8_C(141), UINT8_C(125) },
      { UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C( 88), UINT8_C(224), UINT8_C(  5), UINT8_C(135), UINT8_C(147),
        UINT8_C(  0), UINT8_C(184), UINT8_C(  0), UINT8_C( 60), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C( 96),
        UINT8_C(  0), UINT8_C(  0), UINT8_C( 86), UINT8_C( 47), UINT8_C(225), UINT8_C(  0), UINT8_C(  0), UINT8_C(101),
        UINT8_C(136), UINT8_C(107), UINT8_C(142), UINT8_C(  0), UINT8_C(212), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0) } },
    { UINT32_C(2583734192),
      { UINT8_C(171), UINT8_C( 36), UINT8_C( 31), UINT8_C(122), UINT8_C(201), UINT8_C( 29), UINT8_C(122), UINT8_C(152),
        UINT8_C( 21), UINT8_C( 23), UINT8_C(187), UINT8_C(118), UINT8_C( 32), UINT8_C(252), UINT8_C(182), UINT8_C(  3),
        UINT8_C(114), UINT8_C(145), UINT8_C( 49), UINT8_C( 55), UINT8_C( 23), UINT8_C(162), UINT8_C(215), UINT8_C(230),
        UINT8_C(110), UINT8_C(100), UINT8_C( 99), UINT8_C( 30), UINT8_C( 12), UINT8_C( 99), UINT8_C(184), UINT8_C(183) },
      { UINT8_C(135), UINT8_C(216), UINT8_C( 49), UINT8_C( 81), UINT8_C(245), UINT8_C(171), UINT8_C(233), UINT8_C( 10),
        UINT8_C(195), UINT8_C(164), UINT8_C(128), UINT8_C(227), UINT8_C(160), UINT8_C( 54), UINT8_C(230), UINT8_C( 18),
        UINT8_C(199), UINT8_C( 23), UINT8_C( 74), UINT8_C(223), UINT8_C(185), UINT8_C( 33), UINT8_C(197), UINT8_C( 40),
        UINT8_C(133), UINT8_C( 40), UINT8_C( 70), UINT8_C(145), UINT8_C(139),    UINT8_MAX, UINT8_C( 73), UINT8_C( 19) },
      { UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(223), UINT8_C(100), UINT8_C(  0), UINT8_C( 81),
        UINT8_C(108), UINT8_C( 94), UINT8_C(158), UINT8_C(  0), UINT8_C(  0), UINT8_C(153), UINT8_C(  0), UINT8_C( 11),
        UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
        UINT8_C(  0), UINT8_C( 70), UINT8_C(  0), UINT8_C( 88), UINT8_C( 76), UINT8_C(  0), UINT8_C(  0), UINT8_C(101) } },
    { UINT32_C(3429137111),
      { UINT8_C( 38), UINT8_C( 77), UINT8_C(215), UINT8_C(233), UINT8_C(241), UINT8_C( 87), UINT8_C(204), UINT8_C(145),
        UINT8_C(142), UINT8_C(178), UINT8_C(164), UINT8_C( 85), UINT8_C(201), UINT8_C(238), UINT8_C( 52), UINT8_C(130),
        UINT8_C( 15), UINT8_C(249), UINT8_C(170), UINT8_C(148), UINT8_C( 33), UINT8_C(241), UINT8_C( 38), UINT8_C(173),
        UINT8_C(240), UINT8_C(111), UINT8_C(192), UINT8_C(199), UINT8_C(233), UINT8_C( 36), UINT8_C(147), UINT8_C( 15) },
      { UINT8_C(113), UINT8_C(106), UINT8_C(248), UINT8_C( 98), UINT8_C(194), UINT8_C(196), UINT8_C(243), UINT8_C( 80),
        UINT8_C(118), UINT8_C(151), UINT8_C(165), UINT8_C( 63), UINT8_C(133), UINT8_C(218), UINT8_C(194), UINT8_C(148),
        UINT8_C(211), UINT8_C(108), UINT8_C( 41), UINT8_C(245), UINT8_C( 93), UINT8_C( 79), UINT8_C(162), UINT8_C( 77),
        UINT8_C(190), UINT8_C( 98), UINT8_C( 20), UINT8_C(167), UINT8_C(134), UINT8_C(168), UINT8_C(183), UINT8_C(247) },
      { UINT8_C( 76), UINT8_C( 92), UINT8_C(232), UINT8_C(  0), UINT8_C(218), UINT8_C(  0), UINT8_C(224), UINT8_C(113),
        UINT8_C(  0), UINT8_C(165), UINT8_C(  0), UINT8_C( 74), UINT8_C(167), UINT8_C(228), UINT8_C(123), UINT8_C(  0),
        UINT8_C(  0), UINT8_C(  0), UINT8_C(106), UINT8_C(  0), UINT8_C(  0), UINT8_C(160), UINT8_C(100), UINT8_C(  0),
        UINT8_C(  0), UINT8_C(  0), UINT8_C(106), UINT8_C(183), UINT8_C(  0), UINT8_C(  0), UINT8_C(165), UINT8_C(131) } },
    { UINT32_C(3562647314),
      { UINT8_C(116), UINT8_C( 76), UINT8_C( 36), UINT8_C(234), UINT8_C(228), UINT8_C(202), UINT8_C( 42), UINT8_C(105),
        UINT8_C(164), UINT8_C(236), UINT8_C(254), UINT8_C(119), UINT8_C( 88), UINT8_C( 39), UINT8_C(108), UINT8_C(182),
        UINT8_C(118), UINT8_C( 14), UINT8_C(  3), UINT8_C( 52), UINT8_C(112), UINT8_C( 24), UINT8_C(219), UINT8_C(246),
        UINT8_C(192), UINT8_C(146), UINT8_C(237), UINT8_C(210), UINT8_C( 66), UINT8_C( 70), UINT8_C(167), UINT8_C(182) },
      { UINT8_C(147), UINT8_C(203), UINT8_C(160), UINT8_C(119), UINT8_C(149), UINT8_C(202), UINT8_C(224), UINT8_C( 57),
        UINT8_C(182), UINT8_C(222), UINT8_C(177), UINT8_C( 15), UINT8_C(  5), UINT8_C( 29), UINT8_C(197), UINT8_C(123),
        UINT8_C( 44), UINT8_C(200), UINT8_C(175), UINT8_C(156), UINT8_C(224), UINT8_C(139), UINT8_C(147), UINT8_C(160),
        UINT8_C( 29), UINT8_C(128), UINT8_C(115), UINT8_C( 95), UINT8_C(199), UINT8_C( 26), UINT8_C( 21), UINT8_C( 90) },
      { UINT8_C(  0), UINT8_C(140), UINT8_C(  0), UINT8_C(  0), UINT8_C(189), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
        UINT8_C(173), UINT8_C(229), UINT8_C(216), UINT8_C( 67), UINT8_C(  0), UINT8_C( 34), UINT8_C(  0), UINT8_C(153),
        UINT8_C( 81), UINT8_C(  0), UINT8_C(  0), UINT8_C(104), UINT8_C(168), UINT8_C(  0), UINT8_C(183), UINT8_C(  0),
        UINT8_C(  0), UINT8_C(  0), UINT8_C(176), UINT8_C(  0), UINT8_C(133), UINT8_C(  0), UINT8_C( 94), UINT8_C(136) } },
    { UINT32_C(2077341413),
      { UINT8_C(128), UINT8_C(177), UINT8_C(180), UINT8_C( 55), UINT8_C(144), UINT8_C(101), UINT8_C( 70), UINT8_C(149),
        UINT8_C(131), UINT8_C( 11), UINT8_C( 17), UINT8_C(175), UINT8_C(211), UINT8_C(192), UINT8_C( 75), UINT8_C(180),
        UINT8_C( 75), UINT8_C(222), UINT8_C( 84), UINT8_C(105), UINT8_C( 95), UINT8_C(199), UINT8_C(200), UINT8_C( 38),
        UINT8_C(225), UINT8_C(222), UINT8_C(128), UINT8_C(199), UINT8_C(148), UINT8_C( 81), UINT8_C( 66), UINT8_C( 20) },
      { UINT8_C(  2), UINT8_C(246), UINT8_C( 75), UINT8_C(146), UINT8_C( 92), UINT8_C(145), UINT8_C( 40), UINT8_C(223),
        UINT8_C(156), UINT8_C( 57), UINT8_C(142), UINT8_C(112), UINT8_C(249), UINT8_C(217), UINT8_C( 36), UINT8_C( 69),
        UINT8_C(184), UINT8_C(120), UINT8_C(174), UINT8_C( 23), UINT8_C( 64), UINT8_C(118), UINT8_C( 61), UINT8_C( 33),
        UINT8_C( 84), UINT8_C(189), UINT8_C(232), UINT8_C(232), UINT8_C( 14), UINT8_C( 42), UINT8_C(253), UINT8_C( 16) },
      { UINT8_C( 65), UINT8_C(  0), UINT8_C(128), UINT8_C(  0), UINT8_C(  0), UINT8_C(123), UINT8_C( 55), UINT8_C(186),
        UINT8_C(  0), UINT8_C( 34), UINT8_C( 80), UINT8_C(  0), UINT8_C(230), UINT8_C(205), UINT8_C(  0), UINT8_C(125),
        UINT8_C(130), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C( 80), UINT8_C(  0), UINT8_C(131), UINT8_C( 36),
        UINT8_C(155), UINT8_C(206), UINT8_C(  0), UINT8_C(216), UINT8_C( 81), UINT8_C( 62), UINT8_C(160), UINT8_C(  0) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi8(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi8(test_vec[i].b);
    simde__m256i r = simde_mm256_maskz_avg_epu8(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i8x32(r, simde_x_mm256_loadu_epi8(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_mask_avg_epu16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const uint16_t src[16];
    const simde__mmask16 k;
    const uint16_t a[16];
    const uint16_t b[16];
    const uint16_t r[16];
  } test_vec[] = {
    { { UINT16_C(61285), UINT16_C( 8905), UINT16_C(29130), UINT16_C(28554), UINT16_C(37319), UINT16_C(54261), UINT16_C(53731), UINT16_C(48551),
        UINT16_C(52128), UINT16_C(51552), UINT16_C(39817), UINT16_C(26740), UINT16_C(55116), UINT16_C(46532), UINT16_C(63557), UINT16_C(43569) },
      UINT16_C(64231),
      { UINT16_C(45517), UINT16_C(22379), UINT16_C(12833), UINT16_C( 5864), UINT16_C(52229), UINT16_C(44263), UINT16_C(34953), UINT16_C(59768),
        UINT16_C(  337), UINT16_C(50564), UINT16_C(53354), UINT16_C(11933), UINT16_C(57990), UINT16_C(46886), UINT16_C( 3469), UINT16_C(23217) },
      { UINT16_C( 7358), UINT16_C(57265), UINT16_C(39502), UINT16_C(21494), UINT16_C(56678), UINT16_C(61439), UINT16_C(30565), UINT16_C(46808),
        UINT16_C(23673), UINT16_C(58236), UINT16_C( 6445), UINT16_C(45841), UINT16_C(14331), UINT16_C(34922), UINT16_C( 6980), UINT16_C(  738) },
      { UINT16_C(26438), UINT16_C(39822), UINT16_C(26168), UINT16_C(28554), UINT16_C(37319), UINT16_C(52851), UINT16_C(32759), UINT16_C(53288),
        UINT16_C(52128), UINT16_C(54400), UINT16_C(39817), UINT16_C(28887), UINT16_C(36161), UINT16_C(40904), UINT16_C( 5225), UINT16_C(11978) } },
    { { UINT16_C(37943), UINT16_C(34274), UINT16_C(55342), UINT16_C(38104), UINT16_C(55221), UINT16_C( 7043), UINT16_C(23375), UINT16_C(51409),
        UINT16_C(19895), UINT16_C(58539), UINT16_C(48230), UINT16_C(25239), UINT16_C(  499), UINT16_C(14314), UINT16_C(52508), UINT16_C(21305) },
      UINT16_C( 7009),
      { UINT16_C(36824), UINT16_C(45299), UINT16_C(43299), UINT16_C(42632), UINT16_C(55236), UINT16_C(38145), UINT16_C(47263), UINT16_C(19171),
        UINT16_C(18845), UINT16_C(13318), UINT16_C(63915), UINT16_C(38454), UINT16_C(21040), UINT16_C(26979), UINT16_C(50342), UINT16_C(32389) },
      { UINT16_C(30803), UINT16_C(30255), UINT16_C(46881), UINT16_C(58652), UINT16_C( 7566), UINT16_C(11643), UINT16_C(24277), UINT16_C(29303),
        UINT16_C(32167), UINT16_C(21415), UINT16_C(56694), UINT16_C(42729), UINT16_C(19503), UINT16_C(54543), UINT16_C(37904), UINT16_C(25428) },
      { UINT16_C(33814), UINT16_C(34274), UINT16_C(55342), UINT16_C(38104), UINT16_C(55221), UINT16_C(24894), UINT16_C(35770), UINT16_C(51409),
        UINT16_C(25506), UINT16_C(17367), UINT16_C(48230), UINT16_C(40592), UINT16_C(20272), UINT16_C(14314), UINT16_C(52508), UINT16_C(21305) } },
    { { UINT16_C(33549), UINT16_C(11993), UINT16_C(62778), UINT16_C(51220), UINT16_C(36626), UINT16_C(59381), UINT16_C(27885), UINT16_C(37978),
        UINT16_C(  489), UINT16_C(24551), UINT16_C(53470), UINT16_C( 3333), UINT16_C( 5148), UINT16_C(11491), UINT16_C(14249), UINT16_C(46735) },
      UINT16_C(26810),
      { UINT16_C(62692), UINT16_C(63581), UINT16_C(28604), UINT16_C(45447), UINT16_C(29783), UINT16_C(45341), UINT16_C( 1545), UINT16_C(61618),
        UINT16_C(36965), UINT16_C(27329), UINT16_C(56733), UINT16_C(32894), UINT16_C( 9994), UINT16_C(39351), UINT16_C(29149), UINT16_C(49666) },
      { UINT16_C(24421), UINT16_C( 8634), UINT16_C(17103), UINT16_C( 9938), UINT16_C(61366), UINT16_C(49111), UINT16_C(35317), UINT16_C(23216),
        UINT16_C(28953), UINT16_C(46788), UINT16_C(17230), UINT16_C(22583), UINT16_C(61034), UINT16_C(18674), UINT16_C(62560), UINT16_C(50442) },
      { UINT16_C(33549), UINT16_C(36108), UINT16_C(62778), UINT16_C(27693), UINT16_C(45575), UINT16_C(47226), UINT16_C(27885), UINT16_C(42417),
        UINT16_C(  489), UINT16_C(24551), UINT16_C(53470), UINT16_C(27739), UINT16_C( 5148), UINT16_C(29013), UINT16_C(45855), UINT16_C(46735) } },
    { { UINT16_C(50259), UINT16_C( 8935), UINT16_C(47366), UINT16_C(48456), UINT16_C( 8105), UINT16_C(40572), UINT16_C(11432), UINT16_C(49657),
        UINT16_C(48541), UINT16_C(60536), UINT16_C(44800), UINT16_C(27460), UINT16_C(13981), UINT16_C(64947), UINT16_C(48426), UINT16_C(32451) },
      UINT16_C(43649),
      { UINT16_C(34976), UINT16_C(59747), UINT16_C( 3141), UINT16_C(49416), UINT16_C(45483), UINT16_C(42222), UINT16_C(35698), UINT16_C(60001),
        UINT16_C(25207), UINT16_C(48281), UINT16_C(14285), UINT16_C(33010), UINT16_C( 7476), UINT16_C(63293), UINT16_C(48795), UINT16_C(15265) },
      { UINT16_C( 1350), UINT16_C(35620), UINT16_C(11537), UINT16_C(48205), UINT16_C(15326), UINT16_C(20576), UINT16_C(49862), UINT16_C(15931),
        UINT16_C(54308), UINT16_C(61946), UINT16_C(60427), UINT16_C(16497), UINT16_C(44553), UINT16_C(42039), UINT16_C(55660), UINT16_C(46048) },
      { UINT16_C(18163), UINT16_C( 8935), UINT16_C(47366), UINT16_C(48456), UINT16_C( 8105), UINT16_C(40572), UINT16_C(11432), UINT16_C(37966),
        UINT16_C(48541), UINT16_C(55114), UINT16_C(44800), UINT16_C(24754), UINT16_C(13981), UINT16_C(52666), UINT16_C(48426), UINT16_C(30657) } },
    { { UINT16_C( 1246), UINT16_C(61246), UINT16_C(35633), UINT16_C( 4012), UINT16_C( 3270), UINT16_C(36192), UINT16_C(39886), UINT16_C(62155),
        UINT16_C(50543), UINT16_C(31715), UINT16_C(21681), UINT16_C(48059), UINT16_C(61954), UINT16_C(28511), UINT16_C(16331), UINT16_C(43298) },
      UINT16_C(24644),
      { UINT16_C(30105), UINT16_C(17900), UINT16_C(45701), UINT16_C(58705), UINT16_C( 8255), UINT16_C( 2688), UINT16_C(61202), UINT16_C(63183),
        UINT16_C(33130), UINT16_C( 9546), UINT16_C(19772), UINT16_C(39704), UINT16_C(58300), UINT16_C(57051), UINT16_C( 8077), UINT16_C( 9790) },
      { UINT16_C(10900), UINT16_C( 6507), UINT16_C(48349), UINT16_C( 7422), UINT16_C(32476), UINT16_C(61223), UINT16_C(63086), UINT16_C(55525),
        UINT16_C(12151), UINT16_C(46078), UINT16_C( 5756), UINT16_C(14415), UINT16_C(11001), UINT16_C(34326), UINT16_C(21833), UINT16_C(56748) },
      { UINT16_C( 1246), UINT16_C(61246), UINT16_C(47025), UINT16_C( 4012), UINT16_C( 3270), UINT16_C(36192), UINT16_C(62144), UINT16_C(62155),
        UINT16_C(50543), UINT16_C(31715), UINT16_C(21681), UINT16_C(48059), UINT16_C(61954), UINT16_C(45689), UINT16_C(14955), UINT16_C(43298) } },
    { { UINT16_C( 6015), UINT16_C(23799), UINT16_C(62932), UINT16_C(45177), UINT16_C(41076), UINT16_C(58015), UINT16_C(33942), UINT16_C( 3770),
        UINT16_C(47284), UINT16_C(12481), UINT16_C( 4302), UINT16_C(51305), UINT16_C(32570), UINT16_C(33614), UINT16_C(64468), UINT16_C(21601) },
      UINT16_C(22546),
      { UINT16_C(59056), UINT16_C(10573), UINT16_C(49559), UINT16_C(14025), UINT16_C(24739), UINT16_C(24251), UINT16_C(28526), UINT16_C(12054),
        UINT16_C(58783), UINT16_C( 2112), UINT16_C(31405), UINT16_C(64392), UINT16_C(23806), UINT16_C(24566), UINT16_C( 2480), UINT16_C(25015) },
      { UINT16_C( 1263), UINT16_C(34442), UINT16_C(21702), UINT16_C(27069), UINT16_C(30900), UINT16_C( 8903), UINT16_C(57063), UINT16_C(34385),
        UINT16_C(37315), UINT16_C(28815), UINT16_C( 5900), UINT16_C( 2667), UINT16_C(25203), UINT16_C( 9321), UINT16_C( 8299), UINT16_C(23173) },
      { UINT16_C( 6015), UINT16_C(22508), UINT16_C(62932), UINT16_C(45177), UINT16_C(27820), UINT16_C(58015), UINT16_C(33942), UINT16_C( 3770),
        UINT16_C(47284), UINT16_C(12481), UINT16_C( 4302), UINT16_C(33530), UINT16_C(24505), UINT16_C(33614), UINT16_C( 5390), UINT16_C(21601) } },
    { { UINT16_C( 3876), UINT16_C(60129), UINT16_C(40547), UINT16_C( 5972), UINT16_C( 6934), UINT16_C(64825), UINT16_C(35833), UINT16_C(48259),
        UINT16_C( 4636), UINT16_C(10284), UINT16_C(38953), UINT16_C(40242), UINT16_C(39930), UINT16_C(26049), UINT16_C(18107), UINT16_C(57535) },
      UINT16_C(41045),
      { UINT16_C(47562), UINT16_C( 7742), UINT16_C(21712), UINT16_C( 2618), UINT16_C(13137), UINT16_C(54677), UINT16_C(45552), UINT16_C( 7399),
        UINT16_C( 4570), UINT16_C( 3252), UINT16_C(44718), UINT16_C(28584), UINT16_C(25363), UINT16_C(54197), UINT16_C( 2627), UINT16_C( 3699) },
      { UINT16_C(45763), UINT16_C(37932), UINT16_C(26118), UINT16_C(22686), UINT16_C(13210), UINT16_C(35373), UINT16_C( 5348), UINT16_C(48806),
        UINT16_C(23333), UINT16_C(54219), UINT16_C(29449), UINT16_C( 7490), UINT16_C(63446), UINT16_C( 6896), UINT16_C(25346), UINT16_C(50472) },
      { UINT16_C(46663), UINT16_C(60129), UINT16_C(23915), UINT16_C( 5972), UINT16_C(13174), UINT16_C(64825), UINT16_C(25450), UINT16_C(48259),
        UINT16_C( 4636), UINT16_C(10284), UINT16_C(38953), UINT16_C(40242), UINT16_C(39930), UINT16_C(30547), UINT16_C(18107), UINT16_C(27086) } },
    { { UINT16_C(21525), UINT16_C( 7257), UINT16_C(63419), UINT16_C(21876), UINT16_C(41258), UINT16_C( 4063), UINT16_C(34229), UINT16_C(56269),
        UINT16_C(39136), UINT16_C(60078), UINT16_C(61707), UINT16_C(57863), UINT16_C(63464), UINT16_C(60156), UINT16_C( 9306), UINT16_C(28848) },
      UINT16_C( 2424),
      { UINT16_C(13196), UINT16_C(    1), UINT16_C(11144), UINT16_C(26529), UINT16_C(22074), UINT16_C( 2285), UINT16_C(52529), UINT16_C(57504),
        UINT16_C(44215), UINT16_C(48849), UINT16_C(47502), UINT16_C(35509), UINT16_C( 4260), UINT16_C(21678), UINT16_C( 9856), UINT16_C( 3165) },
      { UINT16_C(24154), UINT16_C(57868), UINT16_C(44426), UINT16_C(50250), UINT16_C(14083), UINT16_C(13772), UINT16_C(27908), UINT16_C(48149),
        UINT16_C(58905), UINT16_C(42874), UINT16_C(12447), UINT16_C(17201), UINT16_C(57152), UINT16_C(49303), UINT16_C(62725), UINT16_C(24524) },
      { UINT16_C(21525), UINT16_C( 7257), UINT16_C(63419), UINT16_C(38390), UINT16_C(18079), UINT16_C( 8029), UINT16_C(40219), UINT16_C(56269),
        UINT16_C(51560), UINT16_C(60078), UINT16_C(61707), UINT16_C(26355), UINT16_C(63464), UINT16_C(60156), UINT16_C( 9306), UINT16_C(28848) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i src = simde_x_mm256_loadu_epi16(test_vec[i].src);
    simde__m256i a = simde_x_mm256_loadu_epi16(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi16(test_vec[i].b);
    simde__m256i r = simde_mm256_mask_avg_epu16(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_u16x16(r, simde_x_mm256_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_maskz_avg_epu16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask16 k;
    const uint16_t a[16];
    const uint16_t b[16];
    const uint16_t r[16];
  } test_vec[] = {
    { UINT16_C(57756),
      { UINT16_C(48284), UINT16_C(  210), UINT16_C(57470), UINT16_C(23667), UINT16_C(34039), UINT16_C(45577), UINT16_C(60747), UINT16_C(36233),
        UINT16_C(61458), UINT16_C(64798), UINT16_C( 3514), UINT16_C(56831), UINT16_C(32324), UINT16_C( 4150), UINT16_C(54047), UINT16_C(48113) },
      { UINT16_C(50319), UINT16_C( 3515), UINT16_C(12196), UINT16_C(39785), UINT16_C(29363), UINT16_C(65102), UINT16_C(55136), UINT16_C(29324),
        UINT16_C(43719), UINT16_C(33391), UINT16_C(28343), UINT16_C(64351), UINT16_C(38381), UINT16_C( 3083), UINT16_C(64872), UINT16_C(63432) },
      { UINT16_C(    0), UINT16_C(    0), UINT16_C(34833), UINT16_C(31726), UINT16_C(31701), UINT16_C(    0), UINT16_C(    0), UINT16_C(32779),
        UINT16_C(52589), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C( 3617), UINT16_C(59460), UINT16_C(55773) } },
    { UINT16_C(33729),
      { UINT16_C(25861), UINT16_C(28338), UINT16_C(25857), UINT16_C(20449), UINT16_C(16740), UINT16_C(61478), UINT16_C(61107), UINT16_C( 8858),
        UINT16_C(20848), UINT16_C(53137), UINT16_C(32332), UINT16_C(22372), UINT16_C(52618), UINT16_C(21076), UINT16_C( 5572), UINT16_C(51670) },
      { UINT16_C(34939), UINT16_C(31800), UINT16_C( 6638), UINT16_C(21195), UINT16_C(61786), UINT16_C( 3394), UINT16_C(56543), UINT16_C(20271),
        UINT16_C(49197), UINT16_C(31006), UINT16_C(33598), UINT16_C(51664), UINT16_C( 9552), UINT16_C( 5147), UINT16_C(61754), UINT16_C(46558) },
      { UINT16_C(30400), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(58825), UINT16_C(14565),
        UINT16_C(35023), UINT16_C(42072), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(49114) } },
    { UINT16_C( 5754),
      { UINT16_C(26673), UINT16_C(64559), UINT16_C(35258), UINT16_C(64750), UINT16_C(52630), UINT16_C(50648), UINT16_C( 1309), UINT16_C(15238),
        UINT16_C(50302), UINT16_C(20158), UINT16_C( 3725), UINT16_C(43379), UINT16_C(44579), UINT16_C(  410), UINT16_C( 5219), UINT16_C(38167) },
      { UINT16_C(18044), UINT16_C(13969), UINT16_C(32719), UINT16_C(25906), UINT16_C( 2637), UINT16_C(27178), UINT16_C(45071), UINT16_C(36261),
        UINT16_C(25717), UINT16_C(  732), UINT16_C(20338), UINT16_C(38315), UINT16_C(18173), UINT16_C(24982), UINT16_C(44378), UINT16_C(55286) },
      { UINT16_C(    0), UINT16_C(39264), UINT16_C(    0), UINT16_C(45328), UINT16_C(27634), UINT16_C(38913), UINT16_C(23190), UINT16_C(    0),
        UINT16_C(    0), UINT16_C(10445), UINT16_C(12032), UINT16_C(    0), UINT16_C(31376), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0) } },
    { UINT16_C(34803),
      { UINT16_C(49677), UINT16_C(16391), UINT16_C(21543), UINT16_C(21066), UINT16_C(23230), UINT16_C(25346), UINT16_C(30695), UINT16_C(50119),
        UINT16_C(14970), UINT16_C( 9491), UINT16_C( 4303), UINT16_C(26219), UINT16_C(50801), UINT16_C(26387), UINT16_C( 1949), UINT16_C(43759) },
      { UINT16_C(63177), UINT16_C(61930), UINT16_C(13642), UINT16_C( 2115), UINT16_C(17807), UINT16_C(30315), UINT16_C(13245), UINT16_C(14138),
        UINT16_C(19821), UINT16_C(15452), UINT16_C(51293), UINT16_C(53154), UINT16_C(46734), UINT16_C(11062), UINT16_C( 9661), UINT16_C(34517) },
      { UINT16_C(56427), UINT16_C(39161), UINT16_C(    0), UINT16_C(    0), UINT16_C(20519), UINT16_C(27831), UINT16_C(21970), UINT16_C(32129),
        UINT16_C(17396), UINT16_C(12472), UINT16_C(27798), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(39138) } },
    { UINT16_C(49179),
      { UINT16_C(25975), UINT16_C(47861), UINT16_C(33901), UINT16_C(55552), UINT16_C(48634), UINT16_C(13324), UINT16_C(31220), UINT16_C(20609),
        UINT16_C(57269), UINT16_C(22552), UINT16_C(42670), UINT16_C(58382), UINT16_C(52177), UINT16_C(42762), UINT16_C( 9553), UINT16_C(51559) },
      { UINT16_C(23691), UINT16_C(63619), UINT16_C(33760), UINT16_C(56017), UINT16_C(56640), UINT16_C(13327), UINT16_C(36950), UINT16_C( 3205),
        UINT16_C(40303), UINT16_C( 7524), UINT16_C(29252), UINT16_C( 5378), UINT16_C( 3133), UINT16_C(36540), UINT16_C( 9009), UINT16_C(48215) },
      { UINT16_C(24833), UINT16_C(55740), UINT16_C(    0), UINT16_C(55785), UINT16_C(52637), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0),
        UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C( 9281), UINT16_C(49887) } },
    { UINT16_C(56191),
      { UINT16_C(24501), UINT16_C(34398), UINT16_C(40762), UINT16_C(18788), UINT16_C(47827), UINT16_C(22745), UINT16_C(18886), UINT16_C(10998),
        UINT16_C(14950), UINT16_C(26780), UINT16_C(55631), UINT16_C( 3188), UINT16_C(42600), UINT16_C(48943), UINT16_C(44898), UINT16_C( 6042) },
      { UINT16_C(63758), UINT16_C(18590), UINT16_C(  664), UINT16_C(27537), UINT16_C(27580), UINT16_C(33732), UINT16_C(47796), UINT16_C( 6829),
        UINT16_C(19188), UINT16_C(17283), UINT16_C(63267), UINT16_C(35663), UINT16_C(32669), UINT16_C(   75), UINT16_C(58670), UINT16_C(15383) },
      { UINT16_C(44130), UINT16_C(26494), UINT16_C(20713), UINT16_C(23163), UINT16_C(37704), UINT16_C(28239), UINT16_C(33341), UINT16_C(    0),
        UINT16_C(17069), UINT16_C(22032), UINT16_C(    0), UINT16_C(19426), UINT16_C(37635), UINT16_C(    0), UINT16_C(51784), UINT16_C(10713) } },
    { UINT16_C(46558),
      { UINT16_C(30341), UINT16_C( 5815), UINT16_C(29922), UINT16_C(42625), UINT16_C(13815), UINT16_C(42080), UINT16_C(21584), UINT16_C(54254),
        UINT16_C( 4759), UINT16_C(59338), UINT16_C(26781), UINT16_C(59494), UINT16_C(37992), UINT16_C(32718), UINT16_C(44240), UINT16_C(21813) },
      { UINT16_C(60451), UINT16_C( 1388), UINT16_C(60768), UINT16_C(22443), UINT16_C( 2851), UINT16_C(29692), UINT16_C(59999), UINT16_C(63046),
        UINT16_C( 4348), UINT16_C(39645), UINT16_C(17272), UINT16_C(57474), UINT16_C(20695), UINT16_C(43104), UINT16_C(38397), UINT16_C( 8445) },
      { UINT16_C(    0), UINT16_C( 3602), UINT16_C(45345), UINT16_C(32534), UINT16_C( 8333), UINT16_C(    0), UINT16_C(40792), UINT16_C(58650),
        UINT16_C( 4554), UINT16_C(    0), UINT16_C(22027), UINT16_C(    0), UINT16_C(29344), UINT16_C(37911), UINT16_C(    0), UINT16_C(15129) } },
    { UINT16_C(27009),
      { UINT16_C(57893), UINT16_C(53335), UINT16_C(31289), UINT16_C(13787), UINT16_C(15085), UINT16_C(13088), UINT16_C( 7216), UINT16_C( 3651),
        UINT16_C(48310), UINT16_C(14673), UINT16_C(10652), UINT16_C(64649), UINT16_C(34513), UINT16_C(52881), UINT16_C( 5030), UINT16_C(52024) },
      { UINT16_C(36853), UINT16_C(11931), UINT16_C(30217), UINT16_C(63076), UINT16_C(33968), UINT16_C(57641), UINT16_C(27808), UINT16_C(22511),
        UINT16_C(16424), UINT16_C(50576), UINT16_C( 6505), UINT16_C(15041), UINT16_C(21408), UINT16_C(17929), UINT16_C(16742), UINT16_C(23314) },
      { UINT16_C(47373), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(13081),
        UINT16_C(32367), UINT16_C(    0), UINT16_C(    0), UINT16_C(39845), UINT16_C(    0), UINT16_C(35405), UINT16_C(10886), UINT16_C(    0) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi16(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi16(test_vec[i].b);
    simde__m256i r = simde_mm256_maskz_avg_epu16(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_u16x16(r, simde_x_mm256_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_avg_epu8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_x_mm512_set_epu8(UINT8_C( 54), UINT8_C( 98), UINT8_C(144), UINT8_C( 33),
                             UINT8_C(227), UINT8_C( 68), UINT8_C( 44), UINT8_C(252),
                             UINT8_C(188), UINT8_C(131), UINT8_C( 22), UINT8_C(137),
                             UINT8_C(110), UINT8_C(230), UINT8_C(116), UINT8_C( 30),
                             UINT8_C(237), UINT8_C(136), UINT8_C(151), UINT8_C(122),
                             UINT8_C( 83), UINT8_C(138), UINT8_C(146), UINT8_C(187),
                             UINT8_C( 83), UINT8_C(179), UINT8_C(167), UINT8_C( 88),
                             UINT8_C(248), UINT8_C( 88), UINT8_C(101), UINT8_C(132),
                             UINT8_C(203), UINT8_C( 36), UINT8_C( 53), UINT8_C(174),
                             UINT8_C( 78), UINT8_C( 97), UINT8_C(124), UINT8_C(201),
                             UINT8_C( 12), UINT8_C(233), UINT8_C(196), UINT8_C(121),
                             UINT8_C( 77), UINT8_C(  2), UINT8_C(189), UINT8_C(251),
                             UINT8_C(252), UINT8_C(129), UINT8_C( 38), UINT8_C(107),
                             UINT8_C(130), UINT8_C(227), UINT8_C(158), UINT8_C(158),
                             UINT8_C(195), UINT8_C( 10), UINT8_C(101), UINT8_C( 55),
                             UINT8_C( 64), UINT8_C(106), UINT8_C(118), UINT8_C( 58)),
      simde_x_mm512_set_epu8(UINT8_C(139), UINT8_C( 91), UINT8_C( 12), UINT8_C(185),
                             UINT8_C( 48), UINT8_C(130), UINT8_C(230), UINT8_C(176),
                             UINT8_C(212), UINT8_C( 20), UINT8_C(110), UINT8_C(217),
                             UINT8_C(186), UINT8_C(186), UINT8_C(132), UINT8_C(114),
                             UINT8_C(188), UINT8_C(251), UINT8_C(225), UINT8_C(230),
                             UINT8_C(165), UINT8_C(108), UINT8_C( 17), UINT8_C(  9),
                             UINT8_C(245), UINT8_C(227), UINT8_C(125), UINT8_C(240),
                             UINT8_C( 47), UINT8_C( 91), UINT8_C(  3), UINT8_C(135),
                             UINT8_C(243), UINT8_C(103), UINT8_C(152), UINT8_C(185),
                             UINT8_C( 49), UINT8_C(  8), UINT8_C( 66), UINT8_C(147),
                             UINT8_C( 50), UINT8_C(167), UINT8_C( 57), UINT8_C(102),
                             UINT8_C(142), UINT8_C( 47), UINT8_C(173), UINT8_C( 47),
                             UINT8_C(172), UINT8_C(237), UINT8_C( 57), UINT8_C(255),
                             UINT8_C(101), UINT8_C(227), UINT8_C(198), UINT8_C(199),
                             UINT8_C( 25), UINT8_C(175), UINT8_C(229), UINT8_C(172),
                             UINT8_C(210), UINT8_C(198), UINT8_C(215), UINT8_C(150)),
      simde_x_mm512_set_epu8(UINT8_C( 97), UINT8_C( 95), UINT8_C( 78), UINT8_C(109),
                             UINT8_C(138), UINT8_C( 99), UINT8_C(137), UINT8_C(214),
                             UINT8_C(200), UINT8_C( 76), UINT8_C( 66), UINT8_C(177),
                             UINT8_C(148), UINT8_C(208), UINT8_C(124), UINT8_C( 72),
                             UINT8_C(213), UINT8_C(194), UINT8_C(188), UINT8_C(176),
                             UINT8_C(124), UINT8_C(123), UINT8_C( 82), UINT8_C( 98),
                             UINT8_C(164), UINT8_C(203), UINT8_C(146), UINT8_C(164),
                             UINT8_C(148), UINT8_C( 90), UINT8_C( 52), UINT8_C(134),
                             UINT8_C(223), UINT8_C( 70), UINT8_C(103), UINT8_C(180),
                             UINT8_C( 64), UINT8_C( 53), UINT8_C( 95), UINT8_C(174),
                             UINT8_C( 31), UINT8_C(200), UINT8_C(127), UINT8_C(112),
                             UINT8_C(110), UINT8_C( 25), UINT8_C(181), UINT8_C(149),
                             UINT8_C(212), UINT8_C(183), UINT8_C( 48), UINT8_C(181),
                             UINT8_C(116), UINT8_C(227), UINT8_C(178), UINT8_C(179),
                             UINT8_C(110), UINT8_C( 93), UINT8_C(165), UINT8_C(114),
                             UINT8_C(137), UINT8_C(152), UINT8_C(167), UINT8_C(104)) },
    { simde_x_mm512_set_epu8(UINT8_C(208), UINT8_C(219), UINT8_C( 57), UINT8_C(220),
                             UINT8_C(  6), UINT8_C( 65), UINT8_C( 21), UINT8_C( 70),
                             UINT8_C( 42), UINT8_C( 69), UINT8_C(205), UINT8_C( 75),
                             UINT8_C( 45), UINT8_C( 93), UINT8_C(126), UINT8_C(  6),
                             UINT8_C(174), UINT8_C( 50), UINT8_C( 17), UINT8_C(221),
                             UINT8_C(217), UINT8_C(127), UINT8_C(111), UINT8_C(120),
                             UINT8_C(131), UINT8_C(148), UINT8_C(141), UINT8_C(152),
                             UINT8_C(144), UINT8_C(136), UINT8_C( 66), UINT8_C( 61),
                             UINT8_C(132), UINT8_C(112), UINT8_C( 28), UINT8_C(152),
                             UINT8_C(127), UINT8_C(231), UINT8_C(109), UINT8_C(161),
                             UINT8_C(158), UINT8_C(  6), UINT8_C(213), UINT8_C(173),
                             UINT8_C(118), UINT8_C(179), UINT8_C( 79), UINT8_C(178),
                             UINT8_C( 46), UINT8_C(236), UINT8_C( 49), UINT8_C( 56),
                             UINT8_C(157), UINT8_C(129), UINT8_C( 46), UINT8_C(200),
                             UINT8_C( 89), UINT8_C(130), UINT8_C(187), UINT8_C(247),
                             UINT8_C(213), UINT8_C( 39), UINT8_C(146), UINT8_C(142)),
      simde_x_mm512_set_epu8(UINT8_C( 76), UINT8_C(227), UINT8_C(192), UINT8_C(129),
                             UINT8_C(228), UINT8_C( 26), UINT8_C(165), UINT8_C(150),
                             UINT8_C(111), UINT8_C( 32), UINT8_C( 11), UINT8_C( 66),
                             UINT8_C( 65), UINT8_C(106), UINT8_C(209), UINT8_C(140),
                             UINT8_C( 66), UINT8_C(  2), UINT8_C(157), UINT8_C(126),
                             UINT8_C( 65), UINT8_C(  8), UINT8_C( 57), UINT8_C( 55),
                             UINT8_C(150), UINT8_C( 40), UINT8_C(164), UINT8_C( 54),
                             UINT8_C( 94), UINT8_C(153), UINT8_C(234), UINT8_C(140),
                             UINT8_C( 45), UINT8_C( 64), UINT8_C(136), UINT8_C(209),
                             UINT8_C(101), UINT8_C(213), UINT8_C(147), UINT8_C(253),
                             UINT8_C( 57), UINT8_C(116), UINT8_C( 96), UINT8_C( 87),
                             UINT8_C(172), UINT8_C( 78), UINT8_C(156), UINT8_C(126),
                             UINT8_C(  3), UINT8_C(150), UINT8_C(133), UINT8_C( 13),
                             UINT8_C(226), UINT8_C( 18), UINT8_C(106), UINT8_C(202),
                             UINT8_C( 95), UINT8_C( 92), UINT8_C( 59), UINT8_C(210),
                             UINT8_C(140), UINT8_C( 87), UINT8_C(213), UINT8_C( 52)),
      simde_x_mm512_set_epu8(UINT8_C(142), UINT8_C(223), UINT8_C(125), UINT8_C(175),
                             UINT8_C(117), UINT8_C( 46), UINT8_C( 93), UINT8_C(110),
                             UINT8_C( 77), UINT8_C( 51), UINT8_C(108), UINT8_C( 71),
                             UINT8_C( 55), UINT8_C(100), UINT8_C(168), UINT8_C( 73),
                             UINT8_C(120), UINT8_C( 26), UINT8_C( 87), UINT8_C(174),
                             UINT8_C(141), UINT8_C( 68), UINT8_C( 84), UINT8_C( 88),
                             UINT8_C(141), UINT8_C( 94), UINT8_C(153), UINT8_C(103),
                             UINT8_C(119), UINT8_C(145), UINT8_C(150), UINT8_C(101),
                             UINT8_C( 89), UINT8_C( 88), UINT8_C( 82), UINT8_C(181),
                             UINT8_C(114), UINT8_C(222), UINT8_C(128), UINT8_C(207),
                             UINT8_C(108), UINT8_C( 61), UINT8_C(155), UINT8_C(130),
                             UINT8_C(145), UINT8_C(129), UINT8_C(118), UINT8_C(152),
                             UINT8_C( 25), UINT8_C(193), UINT8_C( 91), UINT8_C( 35),
                             UINT8_C(192), UINT8_C( 74), UINT8_C( 76), UINT8_C(201),
                             UINT8_C( 92), UINT8_C(111), UINT8_C(123), UINT8_C(229),
                             UINT8_C(177), UINT8_C( 63), UINT8_C(180), UINT8_C( 97)) },
    { simde_x_mm512_set_epu8(UINT8_C( 11), UINT8_C(110), UINT8_C(145), UINT8_C( 98),
                             UINT8_C(192), UINT8_C(110), UINT8_C(161), UINT8_C( 50),
                             UINT8_C(252), UINT8_C(104), UINT8_C(187), UINT8_C( 15),
                             UINT8_C(183), UINT8_C(198), UINT8_C( 79), UINT8_C(  9),
                             UINT8_C( 68), UINT8_C(123), UINT8_C(137), UINT8_C(144),
                             UINT8_C(105), UINT8_C( 90), UINT8_C( 86), UINT8_C( 94),
                             UINT8_C( 93), UINT8_C( 77), UINT8_C(192), UINT8_C(  8),
                             UINT8_C( 44), UINT8_C( 25), UINT8_C( 87), UINT8_C(143),
                             UINT8_C(226), UINT8_C( 88), UINT8_C(154), UINT8_C(252),
                             UINT8_C( 55), UINT8_C(166), UINT8_C(142), UINT8_C(195),
                             UINT8_C(169), UINT8_C(178), UINT8_C(205), UINT8_C( 91),
                             UINT8_C( 65), UINT8_C(198), UINT8_C(127), UINT8_C( 53),
                             UINT8_C(245), UINT8_C( 24), UINT8_C(177), UINT8_C(240),
                             UINT8_C(199), UINT8_C( 24), UINT8_C(143), UINT8_C(108),
                             UINT8_C( 39), UINT8_C(170), UINT8_C(163), UINT8_C(222),
                             UINT8_C( 56), UINT8_C(160), UINT8_C(200), UINT8_C(122)),
      simde_x_mm512_set_epu8(UINT8_C( 57), UINT8_C( 48), UINT8_C(248), UINT8_C( 92),
                             UINT8_C( 57), UINT8_C(154), UINT8_C( 83), UINT8_C(125),
                             UINT8_C( 10), UINT8_C(154), UINT8_C( 72), UINT8_C(111),
                             UINT8_C( 67), UINT8_C( 46), UINT8_C(125), UINT8_C(202),
                             UINT8_C( 45), UINT8_C( 61), UINT8_C( 52), UINT8_C(168),
                             UINT8_C(212), UINT8_C( 16), UINT8_C( 37), UINT8_C(131),
                             UINT8_C( 48), UINT8_C( 90), UINT8_C(210), UINT8_C(233),
                             UINT8_C(148), UINT8_C(  2), UINT8_C( 10), UINT8_C( 62),
                             UINT8_C( 89), UINT8_C(180), UINT8_C( 94), UINT8_C( 22),
                             UINT8_C(254), UINT8_C( 77), UINT8_C(  0), UINT8_C(151),
                             UINT8_C(212), UINT8_C( 38), UINT8_C(106), UINT8_C(205),
                             UINT8_C( 72), UINT8_C(159), UINT8_C(226), UINT8_C( 51),
                             UINT8_C( 53), UINT8_C(162), UINT8_C(189), UINT8_C( 88),
                             UINT8_C( 42), UINT8_C(136), UINT8_C(104), UINT8_C(208),
                             UINT8_C(154), UINT8_C(241), UINT8_C(187), UINT8_C(143),
                             UINT8_C(157), UINT8_C( 95), UINT8_C( 29), UINT8_C( 97)),
      simde_x_mm512_set_epu8(UINT8_C( 34), UINT8_C( 79), UINT8_C(197), UINT8_C( 95),
                             UINT8_C(125), UINT8_C(132), UINT8_C(122), UINT8_C( 88),
                             UINT8_C(131), UINT8_C(129), UINT8_C(130), UINT8_C( 63),
                             UINT8_C(125), UINT8_C(122), UINT8_C(102), UINT8_C(106),
                             UINT8_C( 57), UINT8_C( 92), UINT8_C( 95), UINT8_C(156),
                             UINT8_C(159), UINT8_C( 53), UINT8_C( 62), UINT8_C(113),
                             UINT8_C( 71), UINT8_C( 84), UINT8_C(201), UINT8_C(121),
                             UINT8_C( 96), UINT8_C( 14), UINT8_C( 49), UINT8_C(103),
                             UINT8_C(158), UINT8_C(134), UINT8_C(124), UINT8_C(137),
                             UINT8_C(155), UINT8_C(122), UINT8_C( 71), UINT8_C(173),
                             UINT8_C(191), UINT8_C(108), UINT8_C(156), UINT8_C(148),
                             UINT8_C( 69), UINT8_C(179), UINT8_C(177), UINT8_C( 52),
                             UINT8_C(149), UINT8_C( 93), UINT8_C(183), UINT8_C(164),
                             UINT8_C(121), UINT8_C( 80), UINT8_C(124), UINT8_C(158),
                             UINT8_C( 97), UINT8_C(206), UINT8_C(175), UINT8_C(183),
                             UINT8_C(107), UINT8_C(128), UINT8_C(115), UINT8_C(110)) },
    { simde_x_mm512_set_epu8(UINT8_C(130), UINT8_C( 79), UINT8_C( 82), UINT8_C(  5),
                             UINT8_C(236), UINT8_C(119), UINT8_C( 25), UINT8_C( 77),
                             UINT8_C(139), UINT8_C(103), UINT8_C(204), UINT8_C( 53),
                             UINT8_C( 70), UINT8_C( 81), UINT8_C( 35), UINT8_C(154),
                             UINT8_C(229), UINT8_C( 64), UINT8_C( 94), UINT8_C(131),
                             UINT8_C(217), UINT8_C(  4), UINT8_C(103), UINT8_C(204),
                             UINT8_C( 53), UINT8_C( 46), UINT8_C(123), UINT8_C(123),
                             UINT8_C(251), UINT8_C(227), UINT8_C(153), UINT8_C( 16),
                             UINT8_C( 79), UINT8_C(199), UINT8_C( 15), UINT8_C(177),
                             UINT8_C( 83), UINT8_C( 15), UINT8_C( 37), UINT8_C(215),
                             UINT8_C(178), UINT8_C(145), UINT8_C(240), UINT8_C( 53),
                             UINT8_C(116), UINT8_C(195), UINT8_C(101), UINT8_C(243),
                             UINT8_C(134), UINT8_C( 93), UINT8_C( 64), UINT8_C(103),
                             UINT8_C( 73), UINT8_C( 82), UINT8_C(189), UINT8_C( 37),
                             UINT8_C(202), UINT8_C( 62), UINT8_C(157), UINT8_C( 97),
                             UINT8_C(142), UINT8_C(  6), UINT8_C( 38), UINT8_C( 24)),
      simde_x_mm512_set_epu8(UINT8_C( 57), UINT8_C( 93), UINT8_C(144), UINT8_C( 30),
                             UINT8_C( 11), UINT8_C(223), UINT8_C( 55), UINT8_C(131),
                             UINT8_C( 29), UINT8_C(176), UINT8_C( 52), UINT8_C( 84),
                             UINT8_C( 79), UINT8_C( 47), UINT8_C(128), UINT8_C( 28),
                             UINT8_C( 73), UINT8_C(131), UINT8_C(138), UINT8_C(  1),
                             UINT8_C( 98), UINT8_C(110), UINT8_C(210), UINT8_C(168),
                             UINT8_C(208), UINT8_C(103), UINT8_C(151), UINT8_C( 58),
                             UINT8_C(149), UINT8_C( 16), UINT8_C(113), UINT8_C( 41),
                             UINT8_C(201), UINT8_C( 13), UINT8_C(211), UINT8_C(230),
                             UINT8_C(208), UINT8_C(154), UINT8_C( 49), UINT8_C(127),
                             UINT8_C( 93), UINT8_C(  7), UINT8_C(136), UINT8_C( 53),
                             UINT8_C( 19), UINT8_C( 95), UINT8_C(187), UINT8_C(149),
                             UINT8_C(  0), UINT8_C(161), UINT8_C(228), UINT8_C( 84),
                             UINT8_C( 84), UINT8_C(105), UINT8_C( 90), UINT8_C( 17),
                             UINT8_C(  3), UINT8_C(183), UINT8_C(131), UINT8_C( 24),
                             UINT8_C(201), UINT8_C(  5), UINT8_C(128), UINT8_C( 80)),
      simde_x_mm512_set_epu8(UINT8_C( 94), UINT8_C( 86), UINT8_C(113), UINT8_C( 18),
                             UINT8_C(124), UINT8_C(171), UINT8_C( 40), UINT8_C(104),
                             UINT8_C( 84), UINT8_C(140), UINT8_C(128), UINT8_C( 69),
                             UINT8_C( 75), UINT8_C( 64), UINT8_C( 82), UINT8_C( 91),
                             UINT8_C(151), UINT8_C( 98), UINT8_C(116), UINT8_C( 66),
                             UINT8_C(158), UINT8_C( 57), UINT8_C(157), UINT8_C(186),
                             UINT8_C(131), UINT8_C( 75), UINT8_C(137), UINT8_C( 91),
                             UINT8_C(200), UINT8_C(122), UINT8_C(133), UINT8_C( 29),
                             UINT8_C(140), UINT8_C(106), UINT8_C(113), UINT8_C(204),
                             UINT8_C(146), UINT8_C( 85), UINT8_C( 43), UINT8_C(171),
                             UINT8_C(136), UINT8_C( 76), UINT8_C(188), UINT8_C( 53),
                             UINT8_C( 68), UINT8_C(145), UINT8_C(144), UINT8_C(196),
                             UINT8_C( 67), UINT8_C(127), UINT8_C(146), UINT8_C( 94),
                             UINT8_C( 79), UINT8_C( 94), UINT8_C(140), UINT8_C( 27),
                             UINT8_C(103), UINT8_C(123), UINT8_C(144), UINT8_C( 61),
                             UINT8_C(172), UINT8_C(  6), UINT8_C( 83), UINT8_C( 52)) },
    { simde_x_mm512_set_epu8(UINT8_C( 64), UINT8_C( 98), UINT8_C(116), UINT8_C(169),
                             UINT8_C(168), UINT8_C(153), UINT8_C(192), UINT8_C(140),
                             UINT8_C( 79), UINT8_C( 29), UINT8_C(180), UINT8_C(232),
                             UINT8_C( 89), UINT8_C( 84), UINT8_C(140), UINT8_C(130),
                             UINT8_C(108), UINT8_C(184), UINT8_C(116), UINT8_C( 83),
                             UINT8_C(161), UINT8_C(105), UINT8_C( 17), UINT8_C(119),
                             UINT8_C(  0), UINT8_C(101), UINT8_C(104), UINT8_C(103),
                             UINT8_C(226), UINT8_C( 60), UINT8_C( 23), UINT8_C(129),
                             UINT8_C(141), UINT8_C(179), UINT8_C( 87), UINT8_C(113),
                             UINT8_C(251), UINT8_C(219), UINT8_C(107), UINT8_C(119),
                             UINT8_C( 70), UINT8_C(169), UINT8_C( 11), UINT8_C( 91),
                             UINT8_C( 23), UINT8_C(156), UINT8_C( 99), UINT8_C( 36),
                             UINT8_C(154), UINT8_C( 26), UINT8_C( 23), UINT8_C(168),
                             UINT8_C( 12), UINT8_C(101), UINT8_C(189), UINT8_C(124),
                             UINT8_C(154), UINT8_C( 52), UINT8_C(134), UINT8_C(128),
                             UINT8_C(213), UINT8_C( 85), UINT8_C(189), UINT8_C(209)),
      simde_x_mm512_set_epu8(UINT8_C( 21), UINT8_C( 27), UINT8_C( 57), UINT8_C(175),
                             UINT8_C(191), UINT8_C( 35), UINT8_C( 75), UINT8_C(  5),
                             UINT8_C(168), UINT8_C( 42), UINT8_C(198), UINT8_C(223),
                             UINT8_C(239), UINT8_C(225), UINT8_C(227), UINT8_C(  5),
                             UINT8_C( 85), UINT8_C(196), UINT8_C(184), UINT8_C(155),
                             UINT8_C( 73), UINT8_C(109), UINT8_C(131), UINT8_C(103),
                             UINT8_C(131), UINT8_C(155), UINT8_C(105), UINT8_C( 70),
                             UINT8_C(164), UINT8_C(114), UINT8_C(114), UINT8_C(212),
                             UINT8_C(157), UINT8_C( 34), UINT8_C(220), UINT8_C(250),
                             UINT8_C(254), UINT8_C(227), UINT8_C(240), UINT8_C( 41),
                             UINT8_C(208), UINT8_C( 26), UINT8_C(221), UINT8_C(132),
                             UINT8_C(121), UINT8_C(178), UINT8_C(219), UINT8_C( 55),
                             UINT8_C( 45), UINT8_C( 78), UINT8_C(233), UINT8_C( 53),
                             UINT8_C( 87), UINT8_C(172), UINT8_C(122), UINT8_C(147),
                             UINT8_C(187), UINT8_C(115), UINT8_C( 22), UINT8_C(254),
                             UINT8_C(  9), UINT8_C(218), UINT8_C(224), UINT8_C(214)),
      simde_x_mm512_set_epu8(UINT8_C( 43), UINT8_C( 63), UINT8_C( 87), UINT8_C(172),
                             UINT8_C(180), UINT8_C( 94), UINT8_C(134), UINT8_C( 73),
                             UINT8_C(124), UINT8_C( 36), UINT8_C(189), UINT8_C(228),
                             UINT8_C(164), UINT8_C(155), UINT8_C(184), UINT8_C( 68),
                             UINT8_C( 97), UINT8_C(190), UINT8_C(150), UINT8_C(119),
                             UINT8_C(117), UINT8_C(107), UINT8_C( 74), UINT8_C(111),
                             UINT8_C( 66), UINT8_C(128), UINT8_C(105), UINT8_C( 87),
                             UINT8_C(195), UINT8_C( 87), UINT8_C( 69), UINT8_C(171),
                             UINT8_C(149), UINT8_C(107), UINT8_C(154), UINT8_C(182),
                             UINT8_C(253), UINT8_C(223), UINT8_C(174), UINT8_C( 80),
                             UINT8_C(139), UINT8_C( 98), UINT8_C(116), UINT8_C(112),
                             UINT8_C( 72), UINT8_C(167), UINT8_C(159), UINT8_C( 46),
                             UINT8_C(100), UINT8_C( 52), UINT8_C(128), UINT8_C(111),
                             UINT8_C( 50), UINT8_C(137), UINT8_C(156), UINT8_C(136),
                             UINT8_C(171), UINT8_C( 84), UINT8_C( 78), UINT8_C(191),
                             UINT8_C(111), UINT8_C(152), UINT8_C(207), UINT8_C(212)) },
    { simde_x_mm512_set_epu8(UINT8_C(136), UINT8_C( 77), UINT8_C( 92), UINT8_C(127),
                             UINT8_C(250), UINT8_C( 34), UINT8_C(180), UINT8_C(171),
                             UINT8_C( 31), UINT8_C(112), UINT8_C(162), UINT8_C(182),
                             UINT8_C(240), UINT8_C(136), UINT8_C( 96), UINT8_C( 85),
                             UINT8_C( 41), UINT8_C( 52), UINT8_C( 59), UINT8_C(203),
                             UINT8_C(213), UINT8_C(103), UINT8_C(221), UINT8_C(176),
                             UINT8_C( 99), UINT8_C(  2), UINT8_C( 21), UINT8_C(182),
                             UINT8_C(  7), UINT8_C(176), UINT8_C(125), UINT8_C(109),
                             UINT8_C(233), UINT8_C(242), UINT8_C(  5), UINT8_C(109),
                             UINT8_C( 73), UINT8_C(240), UINT8_C(175), UINT8_C( 24),
                             UINT8_C( 44), UINT8_C(103), UINT8_C(204), UINT8_C( 40),
                             UINT8_C( 78), UINT8_C(117), UINT8_C(221), UINT8_C(168),
                             UINT8_C(194), UINT8_C( 10), UINT8_C( 15), UINT8_C(227),
                             UINT8_C( 13), UINT8_C(241), UINT8_C( 93), UINT8_C( 23),
                             UINT8_C( 31), UINT8_C( 84), UINT8_C(219), UINT8_C( 76),
                             UINT8_C( 52), UINT8_C( 31), UINT8_C( 22), UINT8_C(183)),
      simde_x_mm512_set_epu8(UINT8_C(  0), UINT8_C( 70), UINT8_C(193), UINT8_C(116),
                             UINT8_C(  9), UINT8_C( 85), UINT8_C(190), UINT8_C(154),
                             UINT8_C( 37), UINT8_C(184), UINT8_C(193), UINT8_C(107),
                             UINT8_C( 79), UINT8_C(235), UINT8_C( 33), UINT8_C(125),
                             UINT8_C(111), UINT8_C(108), UINT8_C( 99), UINT8_C(162),
                             UINT8_C(175), UINT8_C(211), UINT8_C( 79), UINT8_C(167),
                             UINT8_C(166), UINT8_C( 89), UINT8_C( 21), UINT8_C(138),
                             UINT8_C( 15), UINT8_C(105), UINT8_C(191), UINT8_C(205),
                             UINT8_C( 34), UINT8_C(122), UINT8_C(209), UINT8_C(143),
                             UINT8_C(216), UINT8_C(162), UINT8_C(239), UINT8_C(190),
                             UINT8_C(212), UINT8_C(246), UINT8_C(251), UINT8_C(206),
                             UINT8_C( 61), UINT8_C( 80), UINT8_C(151), UINT8_C( 93),
                             UINT8_C( 56), UINT8_C(135), UINT8_C(169), UINT8_C(  8),
                             UINT8_C(191), UINT8_C( 66), UINT8_C( 60), UINT8_C( 24),
                             UINT8_C( 93), UINT8_C(156), UINT8_C( 32), UINT8_C( 27),
                             UINT8_C( 48), UINT8_C(210), UINT8_C(231), UINT8_C( 10)),
      simde_x_mm512_set_epu8(UINT8_C( 68), UINT8_C( 74), UINT8_C(143), UINT8_C(122),
                             UINT8_C(130), UINT8_C( 60), UINT8_C(185), UINT8_C(163),
                             UINT8_C( 34), UINT8_C(148), UINT8_C(178), UINT8_C(145),
                             UINT8_C(160), UINT8_C(186), UINT8_C( 65), UINT8_C(105),
                             UINT8_C( 76), UINT8_C( 80), UINT8_C( 79), UINT8_C(183),
                             UINT8_C(194), UINT8_C(157), UINT8_C(150), UINT8_C(172),
                             UINT8_C(133), UINT8_C( 46), UINT8_C( 21), UINT8_C(160),
                             UINT8_C( 11), UINT8_C(141), UINT8_C(158), UINT8_C(157),
                             UINT8_C(134), UINT8_C(182), UINT8_C(107), UINT8_C(126),
                             UINT8_C(145), UINT8_C(201), UINT8_C(207), UINT8_C(107),
                             UINT8_C(128), UINT8_C(175), UINT8_C(228), UINT8_C(123),
                             UINT8_C( 70), UINT8_C( 99), UINT8_C(186), UINT8_C(131),
                             UINT8_C(125), UINT8_C( 73), UINT8_C( 92), UINT8_C(118),
                             UINT8_C(102), UINT8_C(154), UINT8_C( 77), UINT8_C( 24),
                             UINT8_C( 62), UINT8_C(120), UINT8_C(126), UINT8_C( 52),
                             UINT8_C( 50), UINT8_C(121), UINT8_C(127), UINT8_C( 97)) },
    { simde_x_mm512_set_epu8(UINT8_C(234), UINT8_C(192), UINT8_C(234), UINT8_C(237),
                             UINT8_C(  5), UINT8_C(121), UINT8_C(180), UINT8_C( 23),
                             UINT8_C( 48), UINT8_C(184), UINT8_C(208), UINT8_C(106),
                             UINT8_C( 94), UINT8_C(238), UINT8_C(124), UINT8_C( 15),
                             UINT8_C(123), UINT8_C(203), UINT8_C( 16), UINT8_C( 14),
                             UINT8_C(131), UINT8_C(218), UINT8_C( 99), UINT8_C(178),
                             UINT8_C(217), UINT8_C(189), UINT8_C(211), UINT8_C(222),
                             UINT8_C( 27), UINT8_C( 25), UINT8_C( 55), UINT8_C(187),
                             UINT8_C( 24), UINT8_C(250), UINT8_C( 44), UINT8_C(207),
                             UINT8_C(143), UINT8_C(198), UINT8_C(115), UINT8_C(  4),
                             UINT8_C( 18), UINT8_C( 78), UINT8_C( 97), UINT8_C( 62),
                             UINT8_C( 38), UINT8_C(184), UINT8_C(216), UINT8_C(124),
                             UINT8_C(155), UINT8_C( 45), UINT8_C( 66), UINT8_C( 36),
                             UINT8_C(  7), UINT8_C( 22), UINT8_C(106), UINT8_C( 14),
                             UINT8_C( 55), UINT8_C(  9), UINT8_C(104), UINT8_C(136),
                             UINT8_C(167), UINT8_C(129), UINT8_C(  7), UINT8_C( 98)),
      simde_x_mm512_set_epu8(UINT8_C(  4), UINT8_C( 28), UINT8_C(178), UINT8_C(230),
                             UINT8_C(  8), UINT8_C( 80), UINT8_C(138), UINT8_C(121),
                             UINT8_C(141), UINT8_C(178), UINT8_C(150), UINT8_C( 72),
                             UINT8_C(121), UINT8_C( 20), UINT8_C( 28), UINT8_C(243),
                             UINT8_C(255), UINT8_C(153), UINT8_C( 78), UINT8_C(119),
                             UINT8_C( 23), UINT8_C(182), UINT8_C(133), UINT8_C(168),
                             UINT8_C(148), UINT8_C(129), UINT8_C(203), UINT8_C(206),
                             UINT8_C( 25), UINT8_C(232), UINT8_C( 71), UINT8_C( 92),
                             UINT8_C( 90), UINT8_C(159), UINT8_C(196), UINT8_C( 86),
                             UINT8_C(163), UINT8_C(105), UINT8_C(169), UINT8_C(248),
                             UINT8_C(169), UINT8_C( 85), UINT8_C(177), UINT8_C(146),
                             UINT8_C( 77), UINT8_C( 44), UINT8_C(150), UINT8_C( 64),
                             UINT8_C(145), UINT8_C( 33), UINT8_C(163), UINT8_C(255),
                             UINT8_C( 33), UINT8_C( 86), UINT8_C(  3), UINT8_C(162),
                             UINT8_C(241), UINT8_C(203), UINT8_C( 13), UINT8_C( 95),
                             UINT8_C(247), UINT8_C( 54), UINT8_C( 73), UINT8_C(154)),
      simde_x_mm512_set_epu8(UINT8_C(119), UINT8_C(110), UINT8_C(206), UINT8_C(234),
                             UINT8_C(  7), UINT8_C(101), UINT8_C(159), UINT8_C( 72),
                             UINT8_C( 95), UINT8_C(181), UINT8_C(179), UINT8_C( 89),
                             UINT8_C(108), UINT8_C(129), UINT8_C( 76), UINT8_C(129),
                             UINT8_C(189), UINT8_C(178), UINT8_C( 47), UINT8_C( 67),
                             UINT8_C( 77), UINT8_C(200), UINT8_C(116), UINT8_C(173),
                             UINT8_C(183), UINT8_C(159), UINT8_C(207), UINT8_C(214),
                             UINT8_C( 26), UINT8_C(129), UINT8_C( 63), UINT8_C(140),
                             UINT8_C( 57), UINT8_C(205), UINT8_C(120), UINT8_C(147),
                             UINT8_C(153), UINT8_C(152), UINT8_C(142), UINT8_C(126),
                             UINT8_C( 94), UINT8_C( 82), UINT8_C(137), UINT8_C(104),
                             UINT8_C( 58), UINT8_C(114), UINT8_C(183), UINT8_C( 94),
                             UINT8_C(150), UINT8_C( 39), UINT8_C(115), UINT8_C(146),
                             UINT8_C( 20), UINT8_C( 54), UINT8_C( 55), UINT8_C( 88),
                             UINT8_C(148), UINT8_C(106), UINT8_C( 59), UINT8_C(116),
                             UINT8_C(207), UINT8_C( 92), UINT8_C( 40), UINT8_C(126)) },
    { simde_x_mm512_set_epu8(UINT8_C( 83), UINT8_C(142), UINT8_C( 42), UINT8_C(220),
                             UINT8_C( 37), UINT8_C( 62), UINT8_C( 98), UINT8_C(238),
                             UINT8_C(153), UINT8_C( 28), UINT8_C(206), UINT8_C(122),
                             UINT8_C( 85), UINT8_C(118), UINT8_C(175), UINT8_C(149),
                             UINT8_C(220), UINT8_C( 77), UINT8_C( 29), UINT8_C( 65),
                             UINT8_C(199), UINT8_C(150), UINT8_C( 85), UINT8_C(205),
                             UINT8_C(179), UINT8_C(171), UINT8_C( 47), UINT8_C( 79),
                             UINT8_C(186), UINT8_C(189), UINT8_C(  5), UINT8_C( 88),
                             UINT8_C(186), UINT8_C(241), UINT8_C(203), UINT8_C(126),
                             UINT8_C(116), UINT8_C( 53), UINT8_C( 43), UINT8_C(208),
                             UINT8_C(141), UINT8_C(174), UINT8_C(168), UINT8_C( 33),
                             UINT8_C(144), UINT8_C(236), UINT8_C( 92), UINT8_C(185),
                             UINT8_C(194), UINT8_C( 20), UINT8_C(  6), UINT8_C( 82),
                             UINT8_C( 17), UINT8_C(156), UINT8_C(193), UINT8_C(  6),
                             UINT8_C( 65), UINT8_C( 12), UINT8_C( 82), UINT8_C(186),
                             UINT8_C(  9), UINT8_C( 38), UINT8_C(161), UINT8_C(250)),
      simde_x_mm512_set_epu8(UINT8_C(  7), UINT8_C(237), UINT8_C(219), UINT8_C(171),
                             UINT8_C(236), UINT8_C(104), UINT8_C( 84), UINT8_C(  1),
                             UINT8_C(226), UINT8_C( 80), UINT8_C( 28), UINT8_C(225),
                             UINT8_C(190), UINT8_C(201), UINT8_C(130), UINT8_C(143),
                             UINT8_C( 85), UINT8_C(214), UINT8_C(162), UINT8_C(248),
                             UINT8_C(236), UINT8_C(124), UINT8_C(162), UINT8_C(253),
                             UINT8_C(116), UINT8_C(105), UINT8_C(252), UINT8_C(254),
                             UINT8_C( 40), UINT8_C( 36), UINT8_C( 40), UINT8_C(179),
                             UINT8_C(113), UINT8_C(185), UINT8_C( 37), UINT8_C(245),
                             UINT8_C( 41), UINT8_C(  6), UINT8_C(210), UINT8_C( 50),
                             UINT8_C( 82), UINT8_C(223), UINT8_C(202), UINT8_C( 34),
                             UINT8_C(124), UINT8_C( 12), UINT8_C(156), UINT8_C(107),
                             UINT8_C( 75), UINT8_C(221), UINT8_C(108), UINT8_C(193),
                             UINT8_C(  4), UINT8_C(112), UINT8_C(113), UINT8_C(115),
                             UINT8_C(250), UINT8_C( 35), UINT8_C(161), UINT8_C( 10),
                             UINT8_C(151), UINT8_C(213), UINT8_C(128), UINT8_C(  1)),
      simde_x_mm512_set_epu8(UINT8_C( 45), UINT8_C(190), UINT8_C(131), UINT8_C(196),
                             UINT8_C(137), UINT8_C( 83), UINT8_C( 91), UINT8_C(120),
                             UINT8_C(190), UINT8_C( 54), UINT8_C(117), UINT8_C(174),
                             UINT8_C(138), UINT8_C(160), UINT8_C(153), UINT8_C(146),
                             UINT8_C(153), UINT8_C(146), UINT8_C( 96), UINT8_C(157),
                             UINT8_C(218), UINT8_C(137), UINT8_C(124), UINT8_C(229),
                             UINT8_C(148), UINT8_C(138), UINT8_C(150), UINT8_C(167),
                             UINT8_C(113), UINT8_C(113), UINT8_C( 23), UINT8_C(134),
                             UINT8_C(150), UINT8_C(213), UINT8_C(120), UINT8_C(186),
                             UINT8_C( 79), UINT8_C( 30), UINT8_C(127), UINT8_C(129),
                             UINT8_C(112), UINT8_C(199), UINT8_C(185), UINT8_C( 34),
                             UINT8_C(134), UINT8_C(124), UINT8_C(124), UINT8_C(146),
                             UINT8_C(135), UINT8_C(121), UINT8_C( 57), UINT8_C(138),
                             UINT8_C( 11), UINT8_C(134), UINT8_C(153), UINT8_C( 61),
                             UINT8_C(158), UINT8_C( 24), UINT8_C(122), UINT8_C( 98),
                             UINT8_C( 80), UINT8_C(126), UINT8_C(145), UINT8_C(126)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_avg_epu8(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_u8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_avg_epu8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask64 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_x_mm512_set_epu8(UINT8_C(117), UINT8_C(152), UINT8_C(221), UINT8_C(216),
                             UINT8_C(255), UINT8_C( 43), UINT8_C( 10), UINT8_C(211),
                             UINT8_C(214), UINT8_C( 80), UINT8_C(187), UINT8_C(241),
                             UINT8_C(242), UINT8_C(134), UINT8_C( 60), UINT8_C( 93),
                             UINT8_C( 23), UINT8_C(122), UINT8_C( 10), UINT8_C(108),
                             UINT8_C(232), UINT8_C(191), UINT8_C(217), UINT8_C(158),
                             UINT8_C(199), UINT8_C(250), UINT8_C( 81), UINT8_C(211),
                             UINT8_C(229), UINT8_C( 28), UINT8_C(171), UINT8_C( 15),
                             UINT8_C(138), UINT8_C( 52), UINT8_C( 10), UINT8_C(140),
                             UINT8_C( 26), UINT8_C(213), UINT8_C(218), UINT8_C(229),
                             UINT8_C( 66), UINT8_C(204), UINT8_C(  5), UINT8_C(255),
                             UINT8_C(228), UINT8_C(  3), UINT8_C(123), UINT8_C(140),
                             UINT8_C(222), UINT8_C(224), UINT8_C( 98), UINT8_C(103),
                             UINT8_C(237), UINT8_C(138), UINT8_C(179), UINT8_C(224),
                             UINT8_C( 60), UINT8_C(176), UINT8_C( 22), UINT8_C(230),
                             UINT8_C( 60), UINT8_C(244), UINT8_C(191), UINT8_C( 88)),
      UINT64_C(          2117573942),
      simde_x_mm512_set_epu8(UINT8_C( 32), UINT8_C( 22), UINT8_C( 88), UINT8_C(222),
                             UINT8_C( 12), UINT8_C( 90), UINT8_C(155), UINT8_C(252),
                             UINT8_C(242), UINT8_C( 42), UINT8_C(169), UINT8_C(105),
                             UINT8_C( 22), UINT8_C( 34), UINT8_C(113), UINT8_C(184),
                             UINT8_C(216), UINT8_C(186), UINT8_C(232), UINT8_C(159),
                             UINT8_C(188), UINT8_C(250), UINT8_C( 98), UINT8_C(132),
                             UINT8_C(221), UINT8_C( 11), UINT8_C(138), UINT8_C(207),
                             UINT8_C(214), UINT8_C( 24), UINT8_C(222), UINT8_C( 73),
                             UINT8_C(253), UINT8_C(184), UINT8_C(153), UINT8_C( 26),
                             UINT8_C(220), UINT8_C(147), UINT8_C( 37), UINT8_C( 50),
                             UINT8_C( 26), UINT8_C( 78), UINT8_C( 33), UINT8_C( 67),
                             UINT8_C(248), UINT8_C(190), UINT8_C( 29), UINT8_C( 31),
                             UINT8_C( 34), UINT8_C( 40), UINT8_C(189), UINT8_C( 86),
                             UINT8_C( 38), UINT8_C(128), UINT8_C(150), UINT8_C(241),
                             UINT8_C(100), UINT8_C( 53), UINT8_C( 42), UINT8_C( 55),
                             UINT8_C( 87), UINT8_C(241), UINT8_C(251), UINT8_C(171)),
      simde_x_mm512_set_epu8(UINT8_C(229), UINT8_C(148), UINT8_C(139), UINT8_C(168),
                             UINT8_C(149), UINT8_C( 53), UINT8_C(240), UINT8_C(255),
                             UINT8_C(164), UINT8_C(137), UINT8_C( 17), UINT8_C(134),
                             UINT8_C( 22), UINT8_C(243), UINT8_C(  7), UINT8_C(130),
                             UINT8_C(232), UINT8_C(205), UINT8_C(227), UINT8_C(142),
                             UINT8_C(100), UINT8_C(203), UINT8_C(  0), UINT8_C(144),
                             UINT8_C(176), UINT8_C( 89), UINT8_C( 91), UINT8_C(  1),
                             UINT8_C(102), UINT8_C(254), UINT8_C(189), UINT8_C(168),
                             UINT8_C(251), UINT8_C(171), UINT8_C( 24), UINT8_C( 13),
                             UINT8_C( 67), UINT8_C( 49), UINT8_C( 20), UINT8_C(185),
                             UINT8_C(232), UINT8_C( 19), UINT8_C(238), UINT8_C( 58),
                             UINT8_C(109), UINT8_C(140), UINT8_C( 95), UINT8_C( 71),
                             UINT8_C( 47), UINT8_C(118), UINT8_C(241), UINT8_C(225),
                             UINT8_C(186), UINT8_C(175), UINT8_C( 45), UINT8_C( 88),
                             UINT8_C(164), UINT8_C( 95), UINT8_C(253), UINT8_C(227),
                             UINT8_C( 20), UINT8_C(170), UINT8_C(251), UINT8_C( 57)),
      simde_x_mm512_set_epu8(UINT8_C(117), UINT8_C(152), UINT8_C(221), UINT8_C(216),
                             UINT8_C(255), UINT8_C( 43), UINT8_C( 10), UINT8_C(211),
                             UINT8_C(214), UINT8_C( 80), UINT8_C(187), UINT8_C(241),
                             UINT8_C(242), UINT8_C(134), UINT8_C( 60), UINT8_C( 93),
                             UINT8_C( 23), UINT8_C(122), UINT8_C( 10), UINT8_C(108),
                             UINT8_C(232), UINT8_C(191), UINT8_C(217), UINT8_C(158),
                             UINT8_C(199), UINT8_C(250), UINT8_C( 81), UINT8_C(211),
                             UINT8_C(229), UINT8_C( 28), UINT8_C(171), UINT8_C( 15),
                             UINT8_C(138), UINT8_C(178), UINT8_C( 89), UINT8_C( 20),
                             UINT8_C(144), UINT8_C( 98), UINT8_C( 29), UINT8_C(229),
                             UINT8_C( 66), UINT8_C(204), UINT8_C(136), UINT8_C( 63),
                             UINT8_C(228), UINT8_C(165), UINT8_C( 62), UINT8_C( 51),
                             UINT8_C( 41), UINT8_C(224), UINT8_C( 98), UINT8_C(156),
                             UINT8_C(112), UINT8_C(152), UINT8_C(179), UINT8_C(165),
                             UINT8_C( 60), UINT8_C(176), UINT8_C(148), UINT8_C(141),
                             UINT8_C( 60), UINT8_C(206), UINT8_C(251), UINT8_C( 88)) },
    { simde_x_mm512_set_epu8(UINT8_C(248), UINT8_C(153), UINT8_C(124), UINT8_C(192),
                             UINT8_C(251), UINT8_C( 73), UINT8_C( 83), UINT8_C(149),
                             UINT8_C(192), UINT8_C(225), UINT8_C( 11), UINT8_C( 45),
                             UINT8_C(242), UINT8_C(146), UINT8_C(100), UINT8_C(250),
                             UINT8_C(206), UINT8_C(133), UINT8_C(162), UINT8_C( 12),
                             UINT8_C(227), UINT8_C(156), UINT8_C( 97), UINT8_C(141),
                             UINT8_C(103), UINT8_C(177), UINT8_C(102), UINT8_C(250),
                             UINT8_C(236), UINT8_C(151), UINT8_C(250), UINT8_C( 69),
                             UINT8_C( 19), UINT8_C(102), UINT8_C(130), UINT8_C(239),
                             UINT8_C( 26), UINT8_C(151), UINT8_C( 91), UINT8_C(218),
                             UINT8_C(106), UINT8_C(  8), UINT8_C( 85), UINT8_C(190),
                             UINT8_C( 40), UINT8_C(207), UINT8_C( 10), UINT8_C( 15),
                             UINT8_C( 30), UINT8_C( 97), UINT8_C(208), UINT8_C( 26),
                             UINT8_C( 77), UINT8_C(104), UINT8_C(138), UINT8_C( 49),
                             UINT8_C(222), UINT8_C(133), UINT8_C( 22), UINT8_C(114),
                             UINT8_C(226), UINT8_C( 75), UINT8_C(  2), UINT8_C(236)),
      UINT64_C(           474091127),
      simde_x_mm512_set_epu8(UINT8_C( 64), UINT8_C( 45), UINT8_C(186), UINT8_C( 94),
                             UINT8_C(127), UINT8_C(186), UINT8_C(127), UINT8_C(178),
                             UINT8_C(198), UINT8_C( 92), UINT8_C(231), UINT8_C(248),
                             UINT8_C( 21), UINT8_C( 89), UINT8_C(  8), UINT8_C(  1),
                             UINT8_C( 85), UINT8_C(  5), UINT8_C(111), UINT8_C(109),
                             UINT8_C(  6), UINT8_C(229), UINT8_C( 18), UINT8_C( 62),
                             UINT8_C(249), UINT8_C(126), UINT8_C(234), UINT8_C(220),
                             UINT8_C(246), UINT8_C(255), UINT8_C(  1), UINT8_C(115),
                             UINT8_C( 87), UINT8_C( 93), UINT8_C(185), UINT8_C(156),
                             UINT8_C(164), UINT8_C(103), UINT8_C(237), UINT8_C(252),
                             UINT8_C(126), UINT8_C(112), UINT8_C(184), UINT8_C( 45),
                             UINT8_C( 61), UINT8_C(246), UINT8_C( 68), UINT8_C(163),
                             UINT8_C(  5), UINT8_C(127), UINT8_C(109), UINT8_C(194),
                             UINT8_C(167), UINT8_C(139), UINT8_C(130), UINT8_C( 52),
                             UINT8_C(248), UINT8_C(164), UINT8_C(233), UINT8_C(208),
                             UINT8_C(104), UINT8_C(136), UINT8_C(254), UINT8_C(148)),
      simde_x_mm512_set_epu8(UINT8_C( 10), UINT8_C(152), UINT8_C(249), UINT8_C(150),
                             UINT8_C(250), UINT8_C(  9), UINT8_C(168), UINT8_C( 52),
                             UINT8_C( 69), UINT8_C( 91), UINT8_C(134), UINT8_C( 83),
                             UINT8_C( 54), UINT8_C(214), UINT8_C(  9), UINT8_C(100),
                             UINT8_C( 84), UINT8_C( 66), UINT8_C( 99), UINT8_C(199),
                             UINT8_C( 20), UINT8_C(200), UINT8_C(215), UINT8_C( 34),
                             UINT8_C( 96), UINT8_C(125), UINT8_C( 40), UINT8_C(246),
                             UINT8_C( 37), UINT8_C(202), UINT8_C(215), UINT8_C(111),
                             UINT8_C(239), UINT8_C( 73), UINT8_C( 10), UINT8_C( 78),
                             UINT8_C(192), UINT8_C( 57), UINT8_C( 95), UINT8_C( 52),
                             UINT8_C(133), UINT8_C(102), UINT8_C(165), UINT8_C(231),
                             UINT8_C(182), UINT8_C( 23), UINT8_C(129), UINT8_C(213),
                             UINT8_C(123), UINT8_C(235), UINT8_C(187), UINT8_C( 72),
                             UINT8_C(170), UINT8_C( 39), UINT8_C(204), UINT8_C( 88),
                             UINT8_C( 48), UINT8_C( 73), UINT8_C(134), UINT8_C( 45),
                             UINT8_C(127), UINT8_C( 12), UINT8_C( 19), UINT8_C(132)),
      simde_x_mm512_set_epu8(UINT8_C(248), UINT8_C(153), UINT8_C(124), UINT8_C(192),
                             UINT8_C(251), UINT8_C( 73), UINT8_C( 83), UINT8_C(149),
                             UINT8_C(192), UINT8_C(225), UINT8_C( 11), UINT8_C( 45),
                             UINT8_C(242), UINT8_C(146), UINT8_C(100), UINT8_C(250),
                             UINT8_C(206), UINT8_C(133), UINT8_C(162), UINT8_C( 12),
                             UINT8_C(227), UINT8_C(156), UINT8_C( 97), UINT8_C(141),
                             UINT8_C(103), UINT8_C(177), UINT8_C(102), UINT8_C(250),
                             UINT8_C(236), UINT8_C(151), UINT8_C(250), UINT8_C( 69),
                             UINT8_C( 19), UINT8_C(102), UINT8_C(130), UINT8_C(117),
                             UINT8_C(178), UINT8_C( 80), UINT8_C( 91), UINT8_C(218),
                             UINT8_C(106), UINT8_C(107), UINT8_C( 85), UINT8_C(190),
                             UINT8_C( 40), UINT8_C(207), UINT8_C( 99), UINT8_C( 15),
                             UINT8_C( 30), UINT8_C( 97), UINT8_C(208), UINT8_C( 26),
                             UINT8_C(169), UINT8_C( 89), UINT8_C(167), UINT8_C( 49),
                             UINT8_C(222), UINT8_C(119), UINT8_C(184), UINT8_C(127),
                             UINT8_C(226), UINT8_C( 74), UINT8_C(137), UINT8_C(140)) },
    { simde_x_mm512_set_epu8(UINT8_C(221), UINT8_C(100), UINT8_C(182), UINT8_C(149),
                             UINT8_C( 44), UINT8_C(198), UINT8_C( 20), UINT8_C( 23),
                             UINT8_C(105), UINT8_C(188), UINT8_C(118), UINT8_C(243),
                             UINT8_C(175), UINT8_C( 41), UINT8_C(183), UINT8_C(141),
                             UINT8_C(145), UINT8_C( 21), UINT8_C( 99), UINT8_C(117),
                             UINT8_C(242), UINT8_C(144), UINT8_C( 71), UINT8_C( 21),
                             UINT8_C(142), UINT8_C(181), UINT8_C( 66), UINT8_C(137),
                             UINT8_C(194), UINT8_C(226), UINT8_C( 86), UINT8_C(128),
                             UINT8_C(109), UINT8_C( 15), UINT8_C(187), UINT8_C( 22),
                             UINT8_C(243), UINT8_C( 38), UINT8_C(163), UINT8_C(215),
                             UINT8_C( 96), UINT8_C( 79), UINT8_C(232), UINT8_C(216),
                             UINT8_C( 90), UINT8_C( 31), UINT8_C(221), UINT8_C( 22),
                             UINT8_C(144), UINT8_C(219), UINT8_C( 29), UINT8_C( 29),
                             UINT8_C(  7), UINT8_C(  8), UINT8_C(106), UINT8_C(210),
                             UINT8_C( 85), UINT8_C(118), UINT8_C(120), UINT8_C(208),
                             UINT8_C(112), UINT8_C( 80), UINT8_C(173), UINT8_C( 55)),
      UINT64_C(            66155005),
      simde_x_mm512_set_epu8(UINT8_C( 90), UINT8_C(190), UINT8_C(204), UINT8_C( 44),
                             UINT8_C(145), UINT8_C( 10), UINT8_C(145), UINT8_C( 20),
                             UINT8_C(245), UINT8_C(128), UINT8_C(239), UINT8_C(216),
                             UINT8_C(215), UINT8_C(  0), UINT8_C(241), UINT8_C(105),
                             UINT8_C( 81), UINT8_C(  3), UINT8_C( 23), UINT8_C(107),
                             UINT8_C(238), UINT8_C( 80), UINT8_C(150), UINT8_C( 52),
                             UINT8_C( 80), UINT8_C(120), UINT8_C( 83), UINT8_C(139),
                             UINT8_C( 84), UINT8_C(178), UINT8_C( 47), UINT8_C(223),
                             UINT8_C(103), UINT8_C( 66), UINT8_C( 79), UINT8_C( 53),
                             UINT8_C(211), UINT8_C( 20), UINT8_C(111), UINT8_C(197),
                             UINT8_C(238), UINT8_C( 30), UINT8_C( 70), UINT8_C(231),
                             UINT8_C(199), UINT8_C( 18), UINT8_C(252), UINT8_C(101),
                             UINT8_C( 75), UINT8_C( 12), UINT8_C( 85), UINT8_C( 93),
                             UINT8_C(177), UINT8_C(243), UINT8_C( 43), UINT8_C( 45),
                             UINT8_C( 40), UINT8_C(175), UINT8_C( 99), UINT8_C(172),
                             UINT8_C(  7), UINT8_C(156), UINT8_C(188), UINT8_C(130)),
      simde_x_mm512_set_epu8(UINT8_C( 42), UINT8_C( 32), UINT8_C(165), UINT8_C(130),
                             UINT8_C(119), UINT8_C( 88), UINT8_C(110), UINT8_C( 93),
                             UINT8_C( 75), UINT8_C(207), UINT8_C(193), UINT8_C(214),
                             UINT8_C( 54), UINT8_C(185), UINT8_C( 87), UINT8_C(255),
                             UINT8_C(231), UINT8_C(196), UINT8_C(102), UINT8_C(158),
                             UINT8_C(161), UINT8_C(222), UINT8_C(210), UINT8_C( 94),
                             UINT8_C(118), UINT8_C(127), UINT8_C(194), UINT8_C(186),
                             UINT8_C( 80), UINT8_C(125), UINT8_C(244), UINT8_C( 33),
                             UINT8_C(110), UINT8_C(247), UINT8_C(227), UINT8_C(141),
                             UINT8_C(139), UINT8_C( 52), UINT8_C(130), UINT8_C(241),
                             UINT8_C(138), UINT8_C(133), UINT8_C(240), UINT8_C( 72),
                             UINT8_C( 84), UINT8_C( 54), UINT8_C( 76), UINT8_C(208),
                             UINT8_C(177), UINT8_C(100), UINT8_C(198), UINT8_C( 30),
                             UINT8_C( 35), UINT8_C( 68), UINT8_C(216), UINT8_C(  8),
                             UINT8_C( 68), UINT8_C(120), UINT8_C(187), UINT8_C(206),
                             UINT8_C(102), UINT8_C(133), UINT8_C( 95), UINT8_C(110)),
      simde_x_mm512_set_epu8(UINT8_C(221), UINT8_C(100), UINT8_C(182), UINT8_C(149),
                             UINT8_C( 44), UINT8_C(198), UINT8_C( 20), UINT8_C( 23),
                             UINT8_C(105), UINT8_C(188), UINT8_C(118), UINT8_C(243),
                             UINT8_C(175), UINT8_C( 41), UINT8_C(183), UINT8_C(141),
                             UINT8_C(145), UINT8_C( 21), UINT8_C( 99), UINT8_C(117),
                             UINT8_C(242), UINT8_C(144), UINT8_C( 71), UINT8_C( 21),
                             UINT8_C(142), UINT8_C(181), UINT8_C( 66), UINT8_C(137),
                             UINT8_C(194), UINT8_C(226), UINT8_C( 86), UINT8_C(128),
                             UINT8_C(109), UINT8_C( 15), UINT8_C(187), UINT8_C( 22),
                             UINT8_C(243), UINT8_C( 38), UINT8_C(121), UINT8_C(219),
                             UINT8_C(188), UINT8_C( 82), UINT8_C(155), UINT8_C(152),
                             UINT8_C( 90), UINT8_C( 31), UINT8_C(221), UINT8_C(155),
                             UINT8_C(144), UINT8_C( 56), UINT8_C(142), UINT8_C( 62),
                             UINT8_C(  7), UINT8_C(  8), UINT8_C(106), UINT8_C( 27),
                             UINT8_C( 54), UINT8_C(148), UINT8_C(143), UINT8_C(189),
                             UINT8_C( 55), UINT8_C(145), UINT8_C(173), UINT8_C(120)) },
    { simde_x_mm512_set_epu8(UINT8_C(197), UINT8_C(175), UINT8_C(198), UINT8_C( 71),
                             UINT8_C(137), UINT8_C(191), UINT8_C(178), UINT8_C(155),
                             UINT8_C(242), UINT8_C(  4), UINT8_C(232), UINT8_C(161),
                             UINT8_C(106), UINT8_C( 31), UINT8_C(104), UINT8_C( 20),
                             UINT8_C( 65), UINT8_C(248), UINT8_C(181), UINT8_C(128),
                             UINT8_C(175), UINT8_C( 68), UINT8_C(170), UINT8_C( 98),
                             UINT8_C(201), UINT8_C( 10), UINT8_C( 75), UINT8_C( 51),
                             UINT8_C(199), UINT8_C(145), UINT8_C( 87), UINT8_C( 47),
                             UINT8_C(235), UINT8_C(105), UINT8_C( 17), UINT8_C(107),
                             UINT8_C(137), UINT8_C(238), UINT8_C(133), UINT8_C( 81),
                             UINT8_C( 54), UINT8_C(134), UINT8_C(173), UINT8_C( 81),
                             UINT8_C( 21), UINT8_C( 13), UINT8_C(  6), UINT8_C(200),
                             UINT8_C(140), UINT8_C(177), UINT8_C(141), UINT8_C( 81),
                             UINT8_C( 31), UINT8_C( 29), UINT8_C(140), UINT8_C(138),
                             UINT8_C(175), UINT8_C( 98), UINT8_C( 23), UINT8_C(148),
                             UINT8_C(130), UINT8_C( 95), UINT8_C(212), UINT8_C(200)),
      UINT64_C(           518008675),
      simde_x_mm512_set_epu8(UINT8_C(155), UINT8_C(133), UINT8_C( 64), UINT8_C(186),
                             UINT8_C(210), UINT8_C(255), UINT8_C( 70), UINT8_C(210),
                             UINT8_C( 96), UINT8_C( 45), UINT8_C( 57), UINT8_C(248),
                             UINT8_C( 23), UINT8_C( 34), UINT8_C(240), UINT8_C(208),
                             UINT8_C( 74), UINT8_C( 85), UINT8_C(150), UINT8_C( 98),
                             UINT8_C( 81), UINT8_C(149), UINT8_C(213), UINT8_C( 64),
                             UINT8_C(146), UINT8_C(124), UINT8_C(134), UINT8_C(133),
                             UINT8_C( 20), UINT8_C(122), UINT8_C( 57), UINT8_C(241),
                             UINT8_C( 58), UINT8_C( 90), UINT8_C(153), UINT8_C( 57),
                             UINT8_C( 51), UINT8_C(138), UINT8_C( 37), UINT8_C(177),
                             UINT8_C( 13), UINT8_C(116), UINT8_C(177), UINT8_C(238),
                             UINT8_C(169), UINT8_C(177), UINT8_C(173), UINT8_C(231),
                             UINT8_C(226), UINT8_C(216), UINT8_C(126), UINT8_C( 80),
                             UINT8_C(182), UINT8_C( 71), UINT8_C(188), UINT8_C( 53),
                             UINT8_C(255), UINT8_C(200), UINT8_C(176), UINT8_C( 17),
                             UINT8_C(127), UINT8_C( 83), UINT8_C(247), UINT8_C(  0)),
      simde_x_mm512_set_epu8(UINT8_C(226), UINT8_C(  1), UINT8_C(238), UINT8_C(252),
                             UINT8_C( 39), UINT8_C( 85), UINT8_C( 69), UINT8_C( 68),
                             UINT8_C(143), UINT8_C(218), UINT8_C( 28), UINT8_C( 83),
                             UINT8_C(225), UINT8_C( 61), UINT8_C( 37), UINT8_C( 67),
                             UINT8_C( 46), UINT8_C(213), UINT8_C( 32), UINT8_C(183),
                             UINT8_C(230), UINT8_C(  2), UINT8_C(250), UINT8_C(122),
                             UINT8_C(205), UINT8_C(118), UINT8_C(  3), UINT8_C( 17),
                             UINT8_C( 32), UINT8_C( 82), UINT8_C( 40), UINT8_C(  0),
                             UINT8_C( 28), UINT8_C( 37), UINT8_C(253), UINT8_C(171),
                             UINT8_C(164), UINT8_C( 45), UINT8_C(233), UINT8_C(198),
                             UINT8_C(148), UINT8_C( 44), UINT8_C( 28), UINT8_C( 77),
                             UINT8_C( 12), UINT8_C( 81), UINT8_C(153), UINT8_C(  7),
                             UINT8_C(234), UINT8_C( 32), UINT8_C(228), UINT8_C(142),
                             UINT8_C(168), UINT8_C( 42), UINT8_C( 92), UINT8_C( 70),
                             UINT8_C(144), UINT8_C(203), UINT8_C(149), UINT8_C( 41),
                             UINT8_C(206), UINT8_C(198), UINT8_C( 56), UINT8_C( 54)),
      simde_x_mm512_set_epu8(UINT8_C(197), UINT8_C(175), UINT8_C(198), UINT8_C( 71),
                             UINT8_C(137), UINT8_C(191), UINT8_C(178), UINT8_C(155),
                             UINT8_C(242), UINT8_C(  4), UINT8_C(232), UINT8_C(161),
                             UINT8_C(106), UINT8_C( 31), UINT8_C(104), UINT8_C( 20),
                             UINT8_C( 65), UINT8_C(248), UINT8_C(181), UINT8_C(128),
                             UINT8_C(175), UINT8_C( 68), UINT8_C(170), UINT8_C( 98),
                             UINT8_C(201), UINT8_C( 10), UINT8_C( 75), UINT8_C( 51),
                             UINT8_C(199), UINT8_C(145), UINT8_C( 87), UINT8_C( 47),
                             UINT8_C(235), UINT8_C(105), UINT8_C( 17), UINT8_C(114),
                             UINT8_C(108), UINT8_C( 92), UINT8_C(135), UINT8_C( 81),
                             UINT8_C( 81), UINT8_C( 80), UINT8_C(103), UINT8_C( 81),
                             UINT8_C( 21), UINT8_C( 13), UINT8_C(  6), UINT8_C(200),
                             UINT8_C(140), UINT8_C(177), UINT8_C(177), UINT8_C( 81),
                             UINT8_C(175), UINT8_C( 57), UINT8_C(140), UINT8_C( 62),
                             UINT8_C(175), UINT8_C(202), UINT8_C(163), UINT8_C(148),
                             UINT8_C(130), UINT8_C( 95), UINT8_C(152), UINT8_C( 27)) },
    { simde_x_mm512_set_epu8(UINT8_C(121), UINT8_C(  9), UINT8_C(193), UINT8_C(106),
                             UINT8_C( 93), UINT8_C( 44), UINT8_C(  0), UINT8_C(223),
                             UINT8_C(203), UINT8_C(101), UINT8_C( 76), UINT8_C( 37),
                             UINT8_C( 94), UINT8_C(224), UINT8_C(152), UINT8_C(236),
                             UINT8_C(208), UINT8_C( 45), UINT8_C( 88), UINT8_C(163),
                             UINT8_C(104), UINT8_C( 42), UINT8_C(157), UINT8_C( 59),
                             UINT8_C( 90), UINT8_C(187), UINT8_C(107), UINT8_C( 16),
                             UINT8_C(138), UINT8_C(137), UINT8_C(196), UINT8_C( 51),
                             UINT8_C(126), UINT8_C(178), UINT8_C(114), UINT8_C(181),
                             UINT8_C(181), UINT8_C( 19), UINT8_C(113), UINT8_C( 84),
                             UINT8_C( 47), UINT8_C(173), UINT8_C(230), UINT8_C(218),
                             UINT8_C( 64), UINT8_C(150), UINT8_C(107), UINT8_C( 56),
                             UINT8_C(130), UINT8_C(248), UINT8_C( 35), UINT8_C(112),
                             UINT8_C(178), UINT8_C( 75), UINT8_C(231), UINT8_C(  1),
                             UINT8_C(229), UINT8_C(189), UINT8_C( 49), UINT8_C( 75),
                             UINT8_C(217), UINT8_C(188), UINT8_C(205), UINT8_C( 42)),
      UINT64_C(          4020943947),
      simde_x_mm512_set_epu8(UINT8_C(244), UINT8_C(209), UINT8_C(117), UINT8_C( 40),
                             UINT8_C( 42), UINT8_C( 16), UINT8_C(230), UINT8_C(122),
                             UINT8_C(122), UINT8_C(219), UINT8_C(158), UINT8_C(236),
                             UINT8_C( 86), UINT8_C(169), UINT8_C(166), UINT8_C(144),
                             UINT8_C(141), UINT8_C( 79), UINT8_C(123), UINT8_C( 33),
                             UINT8_C(201), UINT8_C(131), UINT8_C(102), UINT8_C( 59),
                             UINT8_C(199), UINT8_C( 19), UINT8_C(252), UINT8_C(201),
                             UINT8_C(170), UINT8_C( 88), UINT8_C(209), UINT8_C( 29),
                             UINT8_C(140), UINT8_C(198), UINT8_C(115), UINT8_C(193),
                             UINT8_C(241), UINT8_C(202), UINT8_C( 84), UINT8_C(255),
                             UINT8_C(  5), UINT8_C(223), UINT8_C(160), UINT8_C( 93),
                             UINT8_C( 97), UINT8_C(124), UINT8_C( 26), UINT8_C(222),
                             UINT8_C(175), UINT8_C(168), UINT8_C( 26), UINT8_C(155),
                             UINT8_C(117), UINT8_C(221), UINT8_C(174), UINT8_C( 92),
                             UINT8_C(115), UINT8_C(243), UINT8_C(104), UINT8_C( 83),
                             UINT8_C( 80), UINT8_C(138), UINT8_C( 34), UINT8_C( 48)),
      simde_x_mm512_set_epu8(UINT8_C(168), UINT8_C(187), UINT8_C(118), UINT8_C( 36),
                             UINT8_C(110), UINT8_C( 81), UINT8_C(219), UINT8_C( 36),
                             UINT8_C(182), UINT8_C(147), UINT8_C( 47), UINT8_C( 12),
                             UINT8_C(227), UINT8_C(175), UINT8_C( 76), UINT8_C(234),
                             UINT8_C( 91), UINT8_C(125), UINT8_C( 98), UINT8_C( 17),
                             UINT8_C(115), UINT8_C( 58), UINT8_C(149), UINT8_C( 90),
                             UINT8_C(115), UINT8_C(232), UINT8_C( 83), UINT8_C( 17),
                             UINT8_C(245), UINT8_C( 20), UINT8_C( 81), UINT8_C( 54),
                             UINT8_C(197), UINT8_C(112), UINT8_C(154), UINT8_C( 13),
                             UINT8_C(  8), UINT8_C(151), UINT8_C(229), UINT8_C(129),
                             UINT8_C(144), UINT8_C(125), UINT8_C( 21), UINT8_C( 55),
                             UINT8_C( 24), UINT8_C( 58), UINT8_C(  7), UINT8_C(127),
                             UINT8_C(150), UINT8_C(222), UINT8_C(105), UINT8_C(207),
                             UINT8_C(223), UINT8_C(121), UINT8_C(  0), UINT8_C(127),
                             UINT8_C(191), UINT8_C(166), UINT8_C(133), UINT8_C(112),
                             UINT8_C(199), UINT8_C( 77), UINT8_C( 42), UINT8_C( 34)),
      simde_x_mm512_set_epu8(UINT8_C(121), UINT8_C(  9), UINT8_C(193), UINT8_C(106),
                             UINT8_C( 93), UINT8_C( 44), UINT8_C(  0), UINT8_C(223),
                             UINT8_C(203), UINT8_C(101), UINT8_C( 76), UINT8_C( 37),
                             UINT8_C( 94), UINT8_C(224), UINT8_C(152), UINT8_C(236),
                             UINT8_C(208), UINT8_C( 45), UINT8_C( 88), UINT8_C(163),
                             UINT8_C(104), UINT8_C( 42), UINT8_C(157), UINT8_C( 59),
                             UINT8_C( 90), UINT8_C(187), UINT8_C(107), UINT8_C( 16),
                             UINT8_C(138), UINT8_C(137), UINT8_C(196), UINT8_C( 51),
                             UINT8_C(169), UINT8_C(155), UINT8_C(135), UINT8_C(181),
                             UINT8_C(125), UINT8_C(177), UINT8_C(157), UINT8_C(192),
                             UINT8_C( 75), UINT8_C(173), UINT8_C( 91), UINT8_C(218),
                             UINT8_C( 61), UINT8_C(150), UINT8_C( 17), UINT8_C( 56),
                             UINT8_C(163), UINT8_C(248), UINT8_C( 66), UINT8_C(181),
                             UINT8_C(170), UINT8_C(171), UINT8_C(231), UINT8_C(  1),
                             UINT8_C(229), UINT8_C(205), UINT8_C( 49), UINT8_C( 75),
                             UINT8_C(140), UINT8_C(188), UINT8_C( 38), UINT8_C( 41)) },
    { simde_x_mm512_set_epu8(UINT8_C( 84), UINT8_C(222), UINT8_C( 36), UINT8_C(132),
                             UINT8_C( 32), UINT8_C(182), UINT8_C( 73), UINT8_C(182),
                             UINT8_C( 77), UINT8_C(116), UINT8_C( 50), UINT8_C( 82),
                             UINT8_C( 68), UINT8_C( 72), UINT8_C( 23), UINT8_C( 32),
                             UINT8_C(202), UINT8_C( 82), UINT8_C( 53), UINT8_C( 71),
                             UINT8_C( 22), UINT8_C( 92), UINT8_C( 42), UINT8_C(133),
                             UINT8_C(215), UINT8_C( 34), UINT8_C( 75), UINT8_C( 63),
                             UINT8_C(139), UINT8_C( 23), UINT8_C(141), UINT8_C( 66),
                             UINT8_C(166), UINT8_C( 99), UINT8_C(183), UINT8_C(237),
                             UINT8_C(213), UINT8_C(192), UINT8_C(235), UINT8_C( 20),
                             UINT8_C(108), UINT8_C(253), UINT8_C(147), UINT8_C(214),
                             UINT8_C( 44), UINT8_C(210), UINT8_C( 20), UINT8_C( 90),
                             UINT8_C(  2), UINT8_C(252), UINT8_C(108), UINT8_C( 27),
                             UINT8_C(207), UINT8_C( 69), UINT8_C(172), UINT8_C( 82),
                             UINT8_C(  9), UINT8_C(  0), UINT8_C( 42), UINT8_C(118),
                             UINT8_C(253), UINT8_C(189), UINT8_C(  6), UINT8_C( 30)),
      UINT64_C(           589511266),
      simde_x_mm512_set_epu8(UINT8_C(159), UINT8_C(199), UINT8_C( 92), UINT8_C(122),
                             UINT8_C( 12), UINT8_C(  6), UINT8_C( 73), UINT8_C(222),
                             UINT8_C( 93), UINT8_C(236), UINT8_C(196), UINT8_C(151),
                             UINT8_C( 37), UINT8_C(206), UINT8_C(135), UINT8_C(234),
                             UINT8_C(134), UINT8_C( 99), UINT8_C(198), UINT8_C( 81),
                             UINT8_C( 12), UINT8_C(102), UINT8_C(195), UINT8_C(222),
                             UINT8_C( 31), UINT8_C( 64), UINT8_C( 80), UINT8_C( 65),
                             UINT8_C(110), UINT8_C(185), UINT8_C(  8), UINT8_C( 98),
                             UINT8_C(113), UINT8_C( 81), UINT8_C(193), UINT8_C(220),
                             UINT8_C(195), UINT8_C(204), UINT8_C(  7), UINT8_C(184),
                             UINT8_C(203), UINT8_C(  3), UINT8_C(246), UINT8_C(115),
                             UINT8_C(106), UINT8_C(122), UINT8_C( 85), UINT8_C(105),
                             UINT8_C( 90), UINT8_C( 75), UINT8_C(186), UINT8_C( 89),
                             UINT8_C( 25), UINT8_C(170), UINT8_C(216), UINT8_C(247),
                             UINT8_C(137), UINT8_C(237), UINT8_C(110), UINT8_C(230),
                             UINT8_C(130), UINT8_C(124), UINT8_C(  6), UINT8_C(245)),
      simde_x_mm512_set_epu8(UINT8_C( 79), UINT8_C( 23), UINT8_C(207), UINT8_C( 80),
                             UINT8_C(207), UINT8_C( 81), UINT8_C(  7), UINT8_C(130),
                             UINT8_C(181), UINT8_C(246), UINT8_C(189), UINT8_C(163),
                             UINT8_C(120), UINT8_C( 35), UINT8_C(200), UINT8_C(  3),
                             UINT8_C( 24), UINT8_C(158), UINT8_C(243), UINT8_C(  3),
                             UINT8_C(150), UINT8_C(228), UINT8_C(184), UINT8_C(101),
                             UINT8_C(219), UINT8_C(121), UINT8_C( 65), UINT8_C( 30),
                             UINT8_C(249), UINT8_C( 33), UINT8_C(115), UINT8_C(122),
                             UINT8_C(244), UINT8_C(143), UINT8_C(149), UINT8_C( 29),
                             UINT8_C(178), UINT8_C(131), UINT8_C(101), UINT8_C(120),
                             UINT8_C( 82), UINT8_C(219), UINT8_C(186), UINT8_C(193),
                             UINT8_C(234), UINT8_C(139), UINT8_C( 28), UINT8_C( 91),
                             UINT8_C(105), UINT8_C(250), UINT8_C(223), UINT8_C(189),
                             UINT8_C(224), UINT8_C(245), UINT8_C(188), UINT8_C( 51),
                             UINT8_C( 24), UINT8_C( 89), UINT8_C( 22), UINT8_C( 52),
                             UINT8_C(174), UINT8_C(224), UINT8_C( 67), UINT8_C(156)),
      simde_x_mm512_set_epu8(UINT8_C( 84), UINT8_C(222), UINT8_C( 36), UINT8_C(132),
                             UINT8_C( 32), UINT8_C(182), UINT8_C( 73), UINT8_C(182),
                             UINT8_C( 77), UINT8_C(116), UINT8_C( 50), UINT8_C( 82),
                             UINT8_C( 68), UINT8_C( 72), UINT8_C( 23), UINT8_C( 32),
                             UINT8_C(202), UINT8_C( 82), UINT8_C( 53), UINT8_C( 71),
                             UINT8_C( 22), UINT8_C( 92), UINT8_C( 42), UINT8_C(133),
                             UINT8_C(215), UINT8_C( 34), UINT8_C( 75), UINT8_C( 63),
                             UINT8_C(139), UINT8_C( 23), UINT8_C(141), UINT8_C( 66),
                             UINT8_C(166), UINT8_C( 99), UINT8_C(171), UINT8_C(237),
                             UINT8_C(213), UINT8_C(192), UINT8_C( 54), UINT8_C(152),
                             UINT8_C(108), UINT8_C(253), UINT8_C(216), UINT8_C(214),
                             UINT8_C( 44), UINT8_C(210), UINT8_C( 57), UINT8_C( 98),
                             UINT8_C(  2), UINT8_C(252), UINT8_C(205), UINT8_C(139),
                             UINT8_C(125), UINT8_C( 69), UINT8_C(202), UINT8_C( 82),
                             UINT8_C(  9), UINT8_C(163), UINT8_C( 66), UINT8_C(118),
                             UINT8_C(253), UINT8_C(189), UINT8_C( 37), UINT8_C( 30)) },
    { simde_x_mm512_set_epu8(UINT8_C(122), UINT8_C( 31), UINT8_C(165), UINT8_C(180),
                             UINT8_C(160), UINT8_C(238), UINT8_C( 21), UINT8_C( 93),
                             UINT8_C(118), UINT8_C( 87), UINT8_C(251), UINT8_C( 43),
                             UINT8_C(165), UINT8_C(232), UINT8_C(151), UINT8_C(210),
                             UINT8_C(191), UINT8_C( 60), UINT8_C(141), UINT8_C(172),
                             UINT8_C(143), UINT8_C(152), UINT8_C( 64), UINT8_C(105),
                             UINT8_C( 15), UINT8_C(129), UINT8_C(227), UINT8_C( 58),
                             UINT8_C( 92), UINT8_C(145), UINT8_C( 98), UINT8_C(228),
                             UINT8_C(166), UINT8_C( 72), UINT8_C(150), UINT8_C( 59),
                             UINT8_C( 16), UINT8_C(  1), UINT8_C(194), UINT8_C( 73),
                             UINT8_C(230), UINT8_C(150), UINT8_C(242), UINT8_C( 38),
                             UINT8_C(140), UINT8_C(234), UINT8_C( 77), UINT8_C( 54),
                             UINT8_C(  3), UINT8_C(126), UINT8_C(200), UINT8_C(132),
                             UINT8_C(171), UINT8_C(245), UINT8_C(169), UINT8_C(148),
                             UINT8_C(166), UINT8_C(  7), UINT8_C(  6), UINT8_C(227),
                             UINT8_C( 91), UINT8_C(103), UINT8_C( 93), UINT8_C(244)),
      UINT64_C(          1742408254),
      simde_x_mm512_set_epu8(UINT8_C(195), UINT8_C(222), UINT8_C( 29), UINT8_C(255),
                             UINT8_C(195), UINT8_C(117), UINT8_C(247), UINT8_C(187),
                             UINT8_C( 69), UINT8_C(188), UINT8_C(174), UINT8_C(241),
                             UINT8_C(175), UINT8_C(167), UINT8_C(226), UINT8_C( 66),
                             UINT8_C(177), UINT8_C(104), UINT8_C(172), UINT8_C(245),
                             UINT8_C( 53), UINT8_C( 17), UINT8_C( 64), UINT8_C( 70),
                             UINT8_C(209), UINT8_C(113), UINT8_C( 86), UINT8_C(118),
                             UINT8_C( 56), UINT8_C( 92), UINT8_C(177), UINT8_C(185),
                             UINT8_C(205), UINT8_C(241), UINT8_C(149), UINT8_C(204),
                             UINT8_C(155), UINT8_C( 35), UINT8_C(114), UINT8_C( 12),
                             UINT8_C(212), UINT8_C( 63), UINT8_C( 66), UINT8_C( 74),
                             UINT8_C( 97), UINT8_C(253), UINT8_C(218), UINT8_C( 22),
                             UINT8_C( 38), UINT8_C( 68), UINT8_C(247), UINT8_C(250),
                             UINT8_C(215), UINT8_C( 66), UINT8_C( 67), UINT8_C( 91),
                             UINT8_C(177), UINT8_C(159), UINT8_C(198), UINT8_C(145),
                             UINT8_C(183), UINT8_C( 26), UINT8_C( 66), UINT8_C(165)),
      simde_x_mm512_set_epu8(UINT8_C(104), UINT8_C(148), UINT8_C(187), UINT8_C(240),
                             UINT8_C(224), UINT8_C(131), UINT8_C(  1), UINT8_C(106),
                             UINT8_C( 77), UINT8_C(151), UINT8_C(127), UINT8_C(197),
                             UINT8_C(118), UINT8_C( 35), UINT8_C( 55), UINT8_C(144),
                             UINT8_C( 54), UINT8_C(201), UINT8_C( 41), UINT8_C(221),
                             UINT8_C(189), UINT8_C( 99), UINT8_C(112), UINT8_C(181),
                             UINT8_C( 52), UINT8_C(200), UINT8_C(153), UINT8_C(231),
                             UINT8_C(146), UINT8_C(105), UINT8_C(102), UINT8_C(227),
                             UINT8_C(214), UINT8_C(182), UINT8_C( 31), UINT8_C( 41),
                             UINT8_C( 93), UINT8_C(234), UINT8_C( 82), UINT8_C( 71),
                             UINT8_C( 52), UINT8_C(241), UINT8_C(224), UINT8_C( 69),
                             UINT8_C( 73), UINT8_C( 93), UINT8_C(195), UINT8_C( 84),
                             UINT8_C(170), UINT8_C(173), UINT8_C(170), UINT8_C( 88),
                             UINT8_C( 60), UINT8_C(109), UINT8_C( 86), UINT8_C(119),
                             UINT8_C(141), UINT8_C(206), UINT8_C( 51), UINT8_C(  6),
                             UINT8_C( 71), UINT8_C(253), UINT8_C( 22), UINT8_C( 92)),
      simde_x_mm512_set_epu8(UINT8_C(122), UINT8_C( 31), UINT8_C(165), UINT8_C(180),
                             UINT8_C(160), UINT8_C(238), UINT8_C( 21), UINT8_C( 93),
                             UINT8_C(118), UINT8_C( 87), UINT8_C(251), UINT8_C( 43),
                             UINT8_C(165), UINT8_C(232), UINT8_C(151), UINT8_C(210),
                             UINT8_C(191), UINT8_C( 60), UINT8_C(141), UINT8_C(172),
                             UINT8_C(143), UINT8_C(152), UINT8_C( 64), UINT8_C(105),
                             UINT8_C( 15), UINT8_C(129), UINT8_C(227), UINT8_C( 58),
                             UINT8_C( 92), UINT8_C(145), UINT8_C( 98), UINT8_C(228),
                             UINT8_C(166), UINT8_C(212), UINT8_C( 90), UINT8_C( 59),
                             UINT8_C( 16), UINT8_C(135), UINT8_C( 98), UINT8_C( 42),
                             UINT8_C(132), UINT8_C(152), UINT8_C(242), UINT8_C( 72),
                             UINT8_C( 85), UINT8_C(234), UINT8_C(207), UINT8_C( 53),
                             UINT8_C(  3), UINT8_C(126), UINT8_C(200), UINT8_C(132),
                             UINT8_C(138), UINT8_C(245), UINT8_C( 77), UINT8_C(148),
                             UINT8_C(166), UINT8_C(  7), UINT8_C(125), UINT8_C( 76),
                             UINT8_C(127), UINT8_C(140), UINT8_C( 44), UINT8_C(244)) },
    { simde_x_mm512_set_epu8(UINT8_C(204), UINT8_C( 72), UINT8_C(160), UINT8_C( 51),
                             UINT8_C( 17), UINT8_C( 91), UINT8_C( 99), UINT8_C(232),
                             UINT8_C(230), UINT8_C( 75), UINT8_C( 98), UINT8_C(249),
                             UINT8_C(240), UINT8_C(193), UINT8_C( 40), UINT8_C(144),
                             UINT8_C( 13), UINT8_C(232), UINT8_C(120), UINT8_C( 62),
                             UINT8_C(136), UINT8_C(  7), UINT8_C(252), UINT8_C(  3),
                             UINT8_C(225), UINT8_C( 43), UINT8_C(176), UINT8_C(138),
                             UINT8_C(205), UINT8_C( 22), UINT8_C(172), UINT8_C( 52),
                             UINT8_C(224), UINT8_C(163), UINT8_C( 92), UINT8_C(198),
                             UINT8_C(193), UINT8_C( 77), UINT8_C(  2), UINT8_C( 58),
                             UINT8_C(117), UINT8_C(166), UINT8_C( 25), UINT8_C( 65),
                             UINT8_C( 70), UINT8_C( 62), UINT8_C(155), UINT8_C(179),
                             UINT8_C(194), UINT8_C(252), UINT8_C(155), UINT8_C(113),
                             UINT8_C( 69), UINT8_C( 48), UINT8_C( 10), UINT8_C( 99),
                             UINT8_C( 95), UINT8_C(207), UINT8_C( 28), UINT8_C(202),
                             UINT8_C(125), UINT8_C(229), UINT8_C(112), UINT8_C( 77)),
      UINT64_C(          3364149510),
      simde_x_mm512_set_epu8(UINT8_C( 11), UINT8_C( 82), UINT8_C( 84), UINT8_C(224),
                             UINT8_C(162), UINT8_C(242), UINT8_C(204), UINT8_C( 68),
                             UINT8_C(182), UINT8_C( 59), UINT8_C( 47), UINT8_C(221),
                             UINT8_C( 34), UINT8_C(151), UINT8_C(204), UINT8_C( 95),
                             UINT8_C(116), UINT8_C( 40), UINT8_C(133), UINT8_C(233),
                             UINT8_C(222), UINT8_C( 50), UINT8_C( 44), UINT8_C( 33),
                             UINT8_C( 29), UINT8_C(204), UINT8_C(121), UINT8_C( 27),
                             UINT8_C(138), UINT8_C(185), UINT8_C(  5), UINT8_C(107),
                             UINT8_C(  3), UINT8_C( 36), UINT8_C( 56), UINT8_C(124),
                             UINT8_C(129), UINT8_C( 58), UINT8_C( 37), UINT8_C(111),
                             UINT8_C(254), UINT8_C(180), UINT8_C( 54), UINT8_C(250),
                             UINT8_C( 42), UINT8_C( 75), UINT8_C(154), UINT8_C(231),
                             UINT8_C(115), UINT8_C(171), UINT8_C(247), UINT8_C(142),
                             UINT8_C(  9), UINT8_C(239), UINT8_C(101), UINT8_C( 47),
                             UINT8_C( 79), UINT8_C(209), UINT8_C(133), UINT8_C(218),
                             UINT8_C(253), UINT8_C( 22), UINT8_C( 28), UINT8_C(216)),
      simde_x_mm512_set_epu8(UINT8_C( 60), UINT8_C( 92), UINT8_C(115), UINT8_C(  5),
                             UINT8_C(100), UINT8_C( 27), UINT8_C(227), UINT8_C(100),
                             UINT8_C( 89), UINT8_C(205), UINT8_C( 67), UINT8_C( 82),
                             UINT8_C(155), UINT8_C(204), UINT8_C(152), UINT8_C(151),
                             UINT8_C(209), UINT8_C(229), UINT8_C(210), UINT8_C( 14),
                             UINT8_C(231), UINT8_C( 89), UINT8_C(236), UINT8_C(229),
                             UINT8_C(214), UINT8_C( 93), UINT8_C( 51), UINT8_C(119),
                             UINT8_C( 13), UINT8_C( 60), UINT8_C(212), UINT8_C(222),
                             UINT8_C( 67), UINT8_C( 38), UINT8_C( 63), UINT8_C(141),
                             UINT8_C( 12), UINT8_C(159), UINT8_C( 36), UINT8_C(134),
                             UINT8_C(  6), UINT8_C(139), UINT8_C( 90), UINT8_C(132),
                             UINT8_C(140), UINT8_C(228), UINT8_C(152), UINT8_C(216),
                             UINT8_C(137), UINT8_C(105), UINT8_C(237), UINT8_C(202),
                             UINT8_C( 29), UINT8_C(230), UINT8_C( 44), UINT8_C(  5),
                             UINT8_C( 39), UINT8_C( 73), UINT8_C(177), UINT8_C(242),
                             UINT8_C(209), UINT8_C( 96), UINT8_C( 58), UINT8_C(209)),
      simde_x_mm512_set_epu8(UINT8_C(204), UINT8_C( 72), UINT8_C(160), UINT8_C( 51),
                             UINT8_C( 17), UINT8_C( 91), UINT8_C( 99), UINT8_C(232),
                             UINT8_C(230), UINT8_C( 75), UINT8_C( 98), UINT8_C(249),
                             UINT8_C(240), UINT8_C(193), UINT8_C( 40), UINT8_C(144),
                             UINT8_C( 13), UINT8_C(232), UINT8_C(120), UINT8_C( 62),
                             UINT8_C(136), UINT8_C(  7), UINT8_C(252), UINT8_C(  3),
                             UINT8_C(225), UINT8_C( 43), UINT8_C(176), UINT8_C(138),
                             UINT8_C(205), UINT8_C( 22), UINT8_C(172), UINT8_C( 52),
                             UINT8_C( 35), UINT8_C( 37), UINT8_C( 92), UINT8_C(198),
                             UINT8_C( 71), UINT8_C( 77), UINT8_C(  2), UINT8_C( 58),
                             UINT8_C(130), UINT8_C(166), UINT8_C( 25), UINT8_C( 65),
                             UINT8_C( 70), UINT8_C(152), UINT8_C(155), UINT8_C(179),
                             UINT8_C(126), UINT8_C(138), UINT8_C(155), UINT8_C(172),
                             UINT8_C( 19), UINT8_C( 48), UINT8_C( 10), UINT8_C( 26),
                             UINT8_C( 95), UINT8_C(207), UINT8_C( 28), UINT8_C(202),
                             UINT8_C(125), UINT8_C( 59), UINT8_C( 43), UINT8_C( 77)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_avg_epu8(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_u32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_avg_epu8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask64 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
{ UINT64_C(          1713497089),
      simde_x_mm512_set_epu8(UINT8_C(151), UINT8_C( 80), UINT8_C(240), UINT8_C(132),
                             UINT8_C(208), UINT8_C( 76), UINT8_C(165), UINT8_C(128),
                             UINT8_C( 54), UINT8_C( 63), UINT8_C(114), UINT8_C(183),
                             UINT8_C(230), UINT8_C(208), UINT8_C(232), UINT8_C(243),
                             UINT8_C(  5), UINT8_C(123), UINT8_C(211), UINT8_C(199),
                             UINT8_C(149), UINT8_C( 47), UINT8_C( 90), UINT8_C(202),
                             UINT8_C(  1), UINT8_C(118), UINT8_C( 37), UINT8_C(249),
                             UINT8_C( 83), UINT8_C( 31), UINT8_C(233), UINT8_C(236),
                             UINT8_C(152), UINT8_C(114), UINT8_C( 63), UINT8_C( 25),
                             UINT8_C(176), UINT8_C( 17), UINT8_C( 37), UINT8_C(212),
                             UINT8_C(144), UINT8_C( 41), UINT8_C(238), UINT8_C( 86),
                             UINT8_C(114), UINT8_C(233), UINT8_C(170), UINT8_C(157),
                             UINT8_C(114), UINT8_C( 25), UINT8_C( 94), UINT8_C( 34),
                             UINT8_C(208), UINT8_C(252), UINT8_C(133), UINT8_C(212),
                             UINT8_C(188), UINT8_C( 19), UINT8_C( 47), UINT8_C(134),
                             UINT8_C(117), UINT8_C( 69), UINT8_C(135), UINT8_C( 66)),
      simde_x_mm512_set_epu8(UINT8_C(180), UINT8_C(237), UINT8_C(100), UINT8_C( 59),
                             UINT8_C(192), UINT8_C(196), UINT8_C(203), UINT8_C( 16),
                             UINT8_C(  0), UINT8_C(167), UINT8_C( 13), UINT8_C( 17),
                             UINT8_C(116), UINT8_C( 41), UINT8_C( 54), UINT8_C(248),
                             UINT8_C(144), UINT8_C(109), UINT8_C( 94), UINT8_C( 19),
                             UINT8_C( 46), UINT8_C(201), UINT8_C(103), UINT8_C(  7),
                             UINT8_C(241), UINT8_C(244), UINT8_C(234), UINT8_C(127),
                             UINT8_C(208), UINT8_C(173), UINT8_C(247), UINT8_C(171),
                             UINT8_C(177), UINT8_C(244), UINT8_C( 76), UINT8_C(191),
                             UINT8_C(166), UINT8_C( 19), UINT8_C( 33), UINT8_C(206),
                             UINT8_C( 89), UINT8_C(216), UINT8_C(139), UINT8_C(111),
                             UINT8_C( 48), UINT8_C(119), UINT8_C(201), UINT8_C( 66),
                             UINT8_C(113), UINT8_C(254), UINT8_C(207), UINT8_C(146),
                             UINT8_C(201), UINT8_C( 44), UINT8_C(125), UINT8_C(195),
                             UINT8_C(221), UINT8_C(218), UINT8_C(229), UINT8_C(236),
                             UINT8_C( 32), UINT8_C(197), UINT8_C(234), UINT8_C(125)),
      simde_x_mm512_set_epu8(UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(179), UINT8_C( 70), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C( 18), UINT8_C( 35), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(189), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(112),
                             UINT8_C(114), UINT8_C(140), UINT8_C(151), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(148), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C( 96)) },
    { UINT64_C(          4250428108),
      simde_x_mm512_set_epu8(UINT8_C(107), UINT8_C(163), UINT8_C(179), UINT8_C( 53),
                             UINT8_C(126), UINT8_C( 10), UINT8_C(123), UINT8_C(143),
                             UINT8_C(215), UINT8_C(148), UINT8_C(197), UINT8_C(220),
                             UINT8_C(232), UINT8_C(205), UINT8_C(188), UINT8_C(218),
                             UINT8_C( 19), UINT8_C(120), UINT8_C(138), UINT8_C( 63),
                             UINT8_C( 24), UINT8_C( 72), UINT8_C( 39), UINT8_C( 31),
                             UINT8_C(164), UINT8_C( 52), UINT8_C( 81), UINT8_C( 39),
                             UINT8_C(186), UINT8_C( 73), UINT8_C( 76), UINT8_C(114),
                             UINT8_C(249), UINT8_C(  4), UINT8_C(201), UINT8_C(188),
                             UINT8_C(120), UINT8_C( 98), UINT8_C(141), UINT8_C(200),
                             UINT8_C( 93), UINT8_C(254), UINT8_C( 78), UINT8_C( 16),
                             UINT8_C( 88), UINT8_C( 71), UINT8_C(144), UINT8_C(138),
                             UINT8_C(  4), UINT8_C(168), UINT8_C( 76), UINT8_C( 88),
                             UINT8_C(159), UINT8_C(107), UINT8_C(228), UINT8_C(197),
                             UINT8_C(209), UINT8_C(111), UINT8_C( 61), UINT8_C(147),
                             UINT8_C(137), UINT8_C( 97), UINT8_C(105), UINT8_C(160)),
      simde_x_mm512_set_epu8(UINT8_C( 95), UINT8_C(238), UINT8_C( 11), UINT8_C( 96),
                             UINT8_C(252), UINT8_C(162), UINT8_C(116), UINT8_C(225),
                             UINT8_C( 52), UINT8_C(254), UINT8_C( 98), UINT8_C( 10),
                             UINT8_C(  5), UINT8_C( 19), UINT8_C(191), UINT8_C( 10),
                             UINT8_C(147), UINT8_C( 52), UINT8_C(171), UINT8_C(224),
                             UINT8_C( 38), UINT8_C( 92), UINT8_C(  6), UINT8_C(185),
                             UINT8_C(177), UINT8_C( 79), UINT8_C(162), UINT8_C(113),
                             UINT8_C(139), UINT8_C( 20), UINT8_C(174), UINT8_C( 82),
                             UINT8_C(136), UINT8_C(114), UINT8_C(204), UINT8_C(188),
                             UINT8_C(236), UINT8_C(209), UINT8_C(166), UINT8_C(169),
                             UINT8_C( 79), UINT8_C(219), UINT8_C( 63), UINT8_C(167),
                             UINT8_C(216), UINT8_C(189), UINT8_C(187), UINT8_C(139),
                             UINT8_C(254), UINT8_C(206), UINT8_C(156), UINT8_C(171),
                             UINT8_C(128), UINT8_C(139), UINT8_C( 97), UINT8_C(237),
                             UINT8_C( 71), UINT8_C(239), UINT8_C(  0), UINT8_C(174),
                             UINT8_C(229), UINT8_C(124), UINT8_C( 45), UINT8_C( 57)),
      simde_x_mm512_set_epu8(UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(193), UINT8_C( 59), UINT8_C(203), UINT8_C(188),
                             UINT8_C(178), UINT8_C(154), UINT8_C(  0), UINT8_C(185),
                             UINT8_C(  0), UINT8_C(237), UINT8_C(  0), UINT8_C( 92),
                             UINT8_C(152), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(187), UINT8_C(116), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(163), UINT8_C(  0),
                             UINT8_C(140), UINT8_C(175), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(183), UINT8_C(111), UINT8_C(  0), UINT8_C(  0)) },
    { UINT64_C(          3228288806),
      simde_x_mm512_set_epu8(UINT8_C( 83), UINT8_C( 39), UINT8_C(120), UINT8_C(115),
                             UINT8_C(205), UINT8_C(228), UINT8_C(102), UINT8_C(158),
                             UINT8_C(179), UINT8_C(121), UINT8_C( 42), UINT8_C(114),
                             UINT8_C(255), UINT8_C(112), UINT8_C( 17), UINT8_C(225),
                             UINT8_C(108), UINT8_C(229), UINT8_C( 66), UINT8_C( 23),
                             UINT8_C( 69), UINT8_C(166), UINT8_C(210), UINT8_C(165),
                             UINT8_C(175), UINT8_C(169), UINT8_C(  1), UINT8_C(245),
                             UINT8_C( 84), UINT8_C(139), UINT8_C( 79), UINT8_C(146),
                             UINT8_C(212), UINT8_C(226), UINT8_C( 33), UINT8_C( 53),
                             UINT8_C( 64), UINT8_C(240), UINT8_C(145), UINT8_C(215),
                             UINT8_C(154), UINT8_C( 13), UINT8_C( 97), UINT8_C(201),
                             UINT8_C( 19), UINT8_C(240), UINT8_C(188), UINT8_C(173),
                             UINT8_C(228), UINT8_C( 88), UINT8_C( 41), UINT8_C( 18),
                             UINT8_C(177), UINT8_C(249), UINT8_C(161), UINT8_C(144),
                             UINT8_C(168), UINT8_C(228), UINT8_C(181), UINT8_C( 34),
                             UINT8_C(226), UINT8_C(255), UINT8_C( 52), UINT8_C(164)),
      simde_x_mm512_set_epu8(UINT8_C( 60), UINT8_C(216), UINT8_C( 36), UINT8_C(217),
                             UINT8_C(176), UINT8_C(146), UINT8_C( 14), UINT8_C(195),
                             UINT8_C(217), UINT8_C(186), UINT8_C(140), UINT8_C(157),
                             UINT8_C(174), UINT8_C(143), UINT8_C(136), UINT8_C(140),
                             UINT8_C(198), UINT8_C( 18), UINT8_C( 72), UINT8_C( 23),
                             UINT8_C(139), UINT8_C(151), UINT8_C( 83), UINT8_C(  3),
                             UINT8_C(152), UINT8_C( 34), UINT8_C( 72), UINT8_C(223),
                             UINT8_C( 84), UINT8_C(166), UINT8_C(140), UINT8_C(210),
                             UINT8_C(238), UINT8_C( 96), UINT8_C(210), UINT8_C(147),
                             UINT8_C(153), UINT8_C(238), UINT8_C(217), UINT8_C( 67),
                             UINT8_C(118), UINT8_C( 12), UINT8_C(237), UINT8_C(171),
                             UINT8_C( 23), UINT8_C( 75), UINT8_C(178), UINT8_C(118),
                             UINT8_C(128), UINT8_C( 11), UINT8_C(225), UINT8_C(116),
                             UINT8_C(179), UINT8_C( 97), UINT8_C( 87), UINT8_C( 53),
                             UINT8_C(223), UINT8_C( 37), UINT8_C( 28), UINT8_C( 24),
                             UINT8_C(153), UINT8_C( 99), UINT8_C(181), UINT8_C( 41)),
      simde_x_mm512_set_epu8(UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(225), UINT8_C(161), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C( 13), UINT8_C(167), UINT8_C(  0),
                             UINT8_C( 21), UINT8_C(  0), UINT8_C(183), UINT8_C(146),
                             UINT8_C(178), UINT8_C( 50), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(173), UINT8_C(124), UINT8_C( 99),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(105), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(177), UINT8_C(117), UINT8_C(  0)) },
    { UINT64_C(            74372392),
      simde_x_mm512_set_epu8(UINT8_C( 19), UINT8_C(169), UINT8_C( 42), UINT8_C(146),
                             UINT8_C( 68), UINT8_C( 97), UINT8_C(131), UINT8_C( 75),
                             UINT8_C( 30), UINT8_C(202), UINT8_C(218), UINT8_C(236),
                             UINT8_C(160), UINT8_C( 84), UINT8_C(108), UINT8_C( 24),
                             UINT8_C(202), UINT8_C(230), UINT8_C(131), UINT8_C(203),
                             UINT8_C( 48), UINT8_C(178), UINT8_C(160), UINT8_C( 82),
                             UINT8_C(240), UINT8_C(188), UINT8_C(191), UINT8_C( 28),
                             UINT8_C(174), UINT8_C(140), UINT8_C(119), UINT8_C(143),
                             UINT8_C(102), UINT8_C( 90), UINT8_C( 86), UINT8_C(242),
                             UINT8_C(207), UINT8_C( 71), UINT8_C(  2), UINT8_C( 28),
                             UINT8_C(194), UINT8_C( 53), UINT8_C(214), UINT8_C(158),
                             UINT8_C( 11), UINT8_C( 81), UINT8_C( 46), UINT8_C( 73),
                             UINT8_C(250), UINT8_C(127), UINT8_C(146), UINT8_C(248),
                             UINT8_C(106), UINT8_C( 95), UINT8_C(130), UINT8_C(129),
                             UINT8_C(153), UINT8_C(235), UINT8_C(236), UINT8_C(185),
                             UINT8_C(106), UINT8_C( 23), UINT8_C(205), UINT8_C(209)),
      simde_x_mm512_set_epu8(UINT8_C(221), UINT8_C(195), UINT8_C( 15), UINT8_C( 71),
                             UINT8_C(113), UINT8_C(109), UINT8_C( 91), UINT8_C(139),
                             UINT8_C(  0), UINT8_C(121), UINT8_C( 48), UINT8_C(109),
                             UINT8_C( 55), UINT8_C(125), UINT8_C(144), UINT8_C( 80),
                             UINT8_C( 48), UINT8_C( 40), UINT8_C( 32), UINT8_C(158),
                             UINT8_C( 64), UINT8_C(225), UINT8_C(246), UINT8_C(250),
                             UINT8_C(216), UINT8_C( 37), UINT8_C( 76), UINT8_C(205),
                             UINT8_C( 27), UINT8_C(254), UINT8_C(155), UINT8_C(246),
                             UINT8_C(218), UINT8_C( 69), UINT8_C(  1), UINT8_C(216),
                             UINT8_C(164), UINT8_C( 62), UINT8_C( 73), UINT8_C(202),
                             UINT8_C( 28), UINT8_C(155), UINT8_C(152), UINT8_C(139),
                             UINT8_C( 24), UINT8_C(201), UINT8_C( 82), UINT8_C(156),
                             UINT8_C(214), UINT8_C( 62), UINT8_C(143), UINT8_C(110),
                             UINT8_C(164), UINT8_C(127), UINT8_C(164), UINT8_C( 20),
                             UINT8_C(221), UINT8_C( 35), UINT8_C( 30), UINT8_C(170),
                             UINT8_C(120), UINT8_C( 91), UINT8_C(187), UINT8_C(207)),
      simde_x_mm512_set_epu8(UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C( 67), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(104), UINT8_C(183), UINT8_C(  0),
                             UINT8_C( 18), UINT8_C(141), UINT8_C( 64), UINT8_C(  0),
                             UINT8_C(232), UINT8_C( 95), UINT8_C(  0), UINT8_C(179),
                             UINT8_C(  0), UINT8_C(111), UINT8_C(  0), UINT8_C( 75),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(133), UINT8_C(  0),
                             UINT8_C(113), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0)) },
    { UINT64_C(          1704849946),
      simde_x_mm512_set_epu8(UINT8_C( 75), UINT8_C(130), UINT8_C(174), UINT8_C(185),
                             UINT8_C(240), UINT8_C(251), UINT8_C(233), UINT8_C(143),
                             UINT8_C(196), UINT8_C( 48), UINT8_C(196), UINT8_C( 72),
                             UINT8_C(201), UINT8_C(123), UINT8_C( 70), UINT8_C(237),
                             UINT8_C( 30), UINT8_C(225), UINT8_C(201), UINT8_C(186),
                             UINT8_C(  9), UINT8_C(202), UINT8_C( 73), UINT8_C(192),
                             UINT8_C( 82), UINT8_C(124), UINT8_C( 29), UINT8_C(  8),
                             UINT8_C( 64), UINT8_C( 66), UINT8_C(254), UINT8_C(144),
                             UINT8_C(122), UINT8_C(124), UINT8_C( 90), UINT8_C( 22),
                             UINT8_C(107), UINT8_C(236), UINT8_C( 37), UINT8_C(140),
                             UINT8_C(185), UINT8_C( 48), UINT8_C(255), UINT8_C(239),
                             UINT8_C(166), UINT8_C(  3), UINT8_C(222), UINT8_C( 36),
                             UINT8_C(239), UINT8_C(218), UINT8_C(100), UINT8_C(226),
                             UINT8_C(118), UINT8_C( 42), UINT8_C(231), UINT8_C(211),
                             UINT8_C(  4), UINT8_C(  8), UINT8_C( 53), UINT8_C( 84),
                             UINT8_C(136), UINT8_C( 61), UINT8_C( 90), UINT8_C(237)),
      simde_x_mm512_set_epu8(UINT8_C( 81), UINT8_C( 81), UINT8_C( 43), UINT8_C( 91),
                             UINT8_C(149), UINT8_C(240), UINT8_C( 71), UINT8_C(161),
                             UINT8_C(204), UINT8_C( 80), UINT8_C( 62), UINT8_C(128),
                             UINT8_C(178), UINT8_C( 60), UINT8_C( 92), UINT8_C( 64),
                             UINT8_C( 44), UINT8_C(188), UINT8_C(121), UINT8_C(108),
                             UINT8_C( 90), UINT8_C( 48), UINT8_C(  8), UINT8_C(192),
                             UINT8_C(189), UINT8_C(101), UINT8_C( 72), UINT8_C(114),
                             UINT8_C(232), UINT8_C(233), UINT8_C(205), UINT8_C(  5),
                             UINT8_C(117), UINT8_C( 79), UINT8_C( 29), UINT8_C(232),
                             UINT8_C( 51), UINT8_C(176), UINT8_C(226), UINT8_C(160),
                             UINT8_C(230), UINT8_C(138), UINT8_C( 64), UINT8_C(111),
                             UINT8_C(190), UINT8_C(200), UINT8_C(126), UINT8_C(168),
                             UINT8_C(222), UINT8_C(  5), UINT8_C(142), UINT8_C(  0),
                             UINT8_C( 60), UINT8_C( 23), UINT8_C(145), UINT8_C(197),
                             UINT8_C( 92), UINT8_C( 21), UINT8_C(182), UINT8_C( 99),
                             UINT8_C(234), UINT8_C(209), UINT8_C(134), UINT8_C( 50)),
      simde_x_mm512_set_epu8(UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(102), UINT8_C( 60), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(206), UINT8_C(  0), UINT8_C(150),
                             UINT8_C(208), UINT8_C(  0), UINT8_C(  0), UINT8_C(175),
                             UINT8_C(178), UINT8_C(102), UINT8_C(  0), UINT8_C(102),
                             UINT8_C(231), UINT8_C(112), UINT8_C(121), UINT8_C(113),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(188), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C( 92),
                             UINT8_C(185), UINT8_C(  0), UINT8_C(112), UINT8_C(  0)) },
    { UINT64_C(           616766838),
      simde_x_mm512_set_epu8(UINT8_C(180), UINT8_C(137), UINT8_C( 51), UINT8_C( 92),
                             UINT8_C(187), UINT8_C( 82), UINT8_C( 53), UINT8_C( 28),
                             UINT8_C(223), UINT8_C(152), UINT8_C(200), UINT8_C(193),
                             UINT8_C(225), UINT8_C(154), UINT8_C(231), UINT8_C( 61),
                             UINT8_C(162), UINT8_C( 56), UINT8_C(127), UINT8_C(187),
                             UINT8_C(196), UINT8_C(180), UINT8_C(126), UINT8_C( 42),
                             UINT8_C(219), UINT8_C(  5), UINT8_C(113), UINT8_C(138),
                             UINT8_C(204), UINT8_C(253), UINT8_C(215), UINT8_C(247),
                             UINT8_C(235), UINT8_C( 43), UINT8_C(234), UINT8_C( 33),
                             UINT8_C(112), UINT8_C(108), UINT8_C(220), UINT8_C( 90),
                             UINT8_C(144), UINT8_C(181), UINT8_C( 55), UINT8_C( 53),
                             UINT8_C( 64), UINT8_C(203), UINT8_C(190), UINT8_C(189),
                             UINT8_C(102), UINT8_C(244), UINT8_C( 66), UINT8_C(205),
                             UINT8_C( 39), UINT8_C(196), UINT8_C(165), UINT8_C(215),
                             UINT8_C(172), UINT8_C( 65), UINT8_C( 68), UINT8_C(204),
                             UINT8_C( 53), UINT8_C( 34), UINT8_C( 78), UINT8_C(127)),
      simde_x_mm512_set_epu8(UINT8_C(141), UINT8_C(142), UINT8_C(201), UINT8_C(183),
                             UINT8_C( 28), UINT8_C(210), UINT8_C( 48), UINT8_C(119),
                             UINT8_C(143), UINT8_C( 53), UINT8_C(224), UINT8_C( 96),
                             UINT8_C(218), UINT8_C(201), UINT8_C(253), UINT8_C(241),
                             UINT8_C( 77), UINT8_C(249), UINT8_C( 74), UINT8_C( 43),
                             UINT8_C(181), UINT8_C(113), UINT8_C(189), UINT8_C(106),
                             UINT8_C( 54), UINT8_C( 32), UINT8_C(202), UINT8_C( 67),
                             UINT8_C(214), UINT8_C( 23), UINT8_C(167), UINT8_C( 40),
                             UINT8_C(139), UINT8_C( 52), UINT8_C( 60), UINT8_C( 38),
                             UINT8_C( 95), UINT8_C( 19), UINT8_C(138), UINT8_C(217),
                             UINT8_C(209), UINT8_C( 61), UINT8_C( 63), UINT8_C(128),
                             UINT8_C(163), UINT8_C( 56), UINT8_C(251), UINT8_C(135),
                             UINT8_C( 69), UINT8_C(144), UINT8_C(129), UINT8_C(103),
                             UINT8_C(244), UINT8_C( 97), UINT8_C(191), UINT8_C( 58),
                             UINT8_C(119), UINT8_C( 19), UINT8_C(105), UINT8_C(174),
                             UINT8_C(180), UINT8_C( 41), UINT8_C(251), UINT8_C(155)),
      simde_x_mm512_set_epu8(UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(147), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C( 64), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(177), UINT8_C(121), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(221), UINT8_C(162),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(154),
                             UINT8_C(142), UINT8_C(147), UINT8_C(  0), UINT8_C(137),
                             UINT8_C(  0), UINT8_C( 42), UINT8_C( 87), UINT8_C(189),
                             UINT8_C(  0), UINT8_C( 38), UINT8_C(165), UINT8_C(  0)) },
    { UINT64_C(           771922964),
      simde_x_mm512_set_epu8(UINT8_C( 84), UINT8_C(193), UINT8_C(196), UINT8_C(188),
                             UINT8_C( 28), UINT8_C( 14), UINT8_C(183), UINT8_C( 63),
                             UINT8_C(195), UINT8_C( 17), UINT8_C( 60), UINT8_C(196),
                             UINT8_C(140), UINT8_C(189), UINT8_C(109), UINT8_C( 63),
                             UINT8_C( 42), UINT8_C(192), UINT8_C(240), UINT8_C(166),
                             UINT8_C( 14), UINT8_C(224), UINT8_C(252), UINT8_C( 17),
                             UINT8_C( 46), UINT8_C(184), UINT8_C(253), UINT8_C( 54),
                             UINT8_C( 94), UINT8_C( 75), UINT8_C(123), UINT8_C(145),
                             UINT8_C(160), UINT8_C(164), UINT8_C(123), UINT8_C(216),
                             UINT8_C(170), UINT8_C(112), UINT8_C(140), UINT8_C( 56),
                             UINT8_C( 30), UINT8_C(182), UINT8_C(108), UINT8_C( 63),
                             UINT8_C( 54), UINT8_C( 68), UINT8_C( 21), UINT8_C( 50),
                             UINT8_C( 82), UINT8_C( 10), UINT8_C(167), UINT8_C(177),
                             UINT8_C(210), UINT8_C(  2), UINT8_C( 48), UINT8_C( 73),
                             UINT8_C(153), UINT8_C( 42), UINT8_C( 43), UINT8_C( 19),
                             UINT8_C( 82), UINT8_C(112), UINT8_C(146), UINT8_C( 77)),
      simde_x_mm512_set_epu8(UINT8_C(117), UINT8_C( 61), UINT8_C(219), UINT8_C(166),
                             UINT8_C(101), UINT8_C(146), UINT8_C(252), UINT8_C(220),
                             UINT8_C(111), UINT8_C(  7), UINT8_C(233), UINT8_C( 32),
                             UINT8_C(239), UINT8_C(243), UINT8_C(204), UINT8_C(249),
                             UINT8_C(179), UINT8_C(241), UINT8_C(226), UINT8_C(120),
                             UINT8_C(244), UINT8_C(118), UINT8_C(248), UINT8_C(141),
                             UINT8_C(179), UINT8_C(252), UINT8_C(219), UINT8_C( 58),
                             UINT8_C( 62), UINT8_C(244), UINT8_C(181), UINT8_C( 77),
                             UINT8_C( 48), UINT8_C(136), UINT8_C(150), UINT8_C( 74),
                             UINT8_C( 87), UINT8_C(176), UINT8_C(194), UINT8_C( 45),
                             UINT8_C(190), UINT8_C(213), UINT8_C( 70), UINT8_C( 70),
                             UINT8_C(177), UINT8_C(254), UINT8_C(160), UINT8_C( 78),
                             UINT8_C(253), UINT8_C(131), UINT8_C(202), UINT8_C(108),
                             UINT8_C(174), UINT8_C(186), UINT8_C(172), UINT8_C(193),
                             UINT8_C(162), UINT8_C(235), UINT8_C(165), UINT8_C(158),
                             UINT8_C(140), UINT8_C(172), UINT8_C(251), UINT8_C(116)),
      simde_x_mm512_set_epu8(UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(137), UINT8_C(  0),
                             UINT8_C(129), UINT8_C(144), UINT8_C(167), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C( 91), UINT8_C(  0),
                             UINT8_C(168), UINT8_C(  0), UINT8_C(  0), UINT8_C(143),
                             UINT8_C(192), UINT8_C( 94), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C( 89),
                             UINT8_C(  0), UINT8_C(142), UINT8_C(  0), UINT8_C(  0)) },
    { UINT64_C(            97696573),
      simde_x_mm512_set_epu8(UINT8_C( 56), UINT8_C(233), UINT8_C(161), UINT8_C( 70),
                             UINT8_C( 29), UINT8_C(119), UINT8_C(231), UINT8_C(252),
                             UINT8_C(132), UINT8_C(198), UINT8_C( 18), UINT8_C(195),
                             UINT8_C( 88), UINT8_C(114), UINT8_C(173), UINT8_C( 54),
                             UINT8_C( 75), UINT8_C(  3), UINT8_C(214), UINT8_C(  2),
                             UINT8_C(139), UINT8_C(186), UINT8_C( 99), UINT8_C(154),
                             UINT8_C(177), UINT8_C(145), UINT8_C(140), UINT8_C(170),
                             UINT8_C(124), UINT8_C( 32), UINT8_C(208), UINT8_C(227),
                             UINT8_C( 14), UINT8_C( 57), UINT8_C( 96), UINT8_C(190),
                             UINT8_C(123), UINT8_C(184), UINT8_C( 51), UINT8_C( 34),
                             UINT8_C(239), UINT8_C( 21), UINT8_C(170), UINT8_C(146),
                             UINT8_C(232), UINT8_C(  0), UINT8_C(180), UINT8_C( 80),
                             UINT8_C( 72), UINT8_C(231), UINT8_C(149), UINT8_C(142),
                             UINT8_C(142), UINT8_C( 86), UINT8_C( 39), UINT8_C(243),
                             UINT8_C( 33), UINT8_C(235), UINT8_C( 93), UINT8_C(129),
                             UINT8_C( 46), UINT8_C(131), UINT8_C(184), UINT8_C(105)),
      simde_x_mm512_set_epu8(UINT8_C(188), UINT8_C(104), UINT8_C(150), UINT8_C( 91),
                             UINT8_C(223), UINT8_C( 10), UINT8_C(154), UINT8_C(110),
                             UINT8_C(234), UINT8_C(193), UINT8_C( 58), UINT8_C(132),
                             UINT8_C( 18), UINT8_C(225), UINT8_C( 95), UINT8_C(  7),
                             UINT8_C(253), UINT8_C( 41), UINT8_C(230), UINT8_C(186),
                             UINT8_C(  9), UINT8_C( 18), UINT8_C(163), UINT8_C( 56),
                             UINT8_C( 96), UINT8_C(249), UINT8_C( 80), UINT8_C( 62),
                             UINT8_C(194), UINT8_C( 89), UINT8_C(163), UINT8_C(133),
                             UINT8_C( 35), UINT8_C(148), UINT8_C(177), UINT8_C( 29),
                             UINT8_C(195), UINT8_C(169), UINT8_C( 29), UINT8_C( 99),
                             UINT8_C(232), UINT8_C( 69), UINT8_C( 58), UINT8_C(168),
                             UINT8_C(144), UINT8_C(242), UINT8_C(220), UINT8_C(  1),
                             UINT8_C( 48), UINT8_C(123), UINT8_C( 36), UINT8_C(240),
                             UINT8_C( 56), UINT8_C(103), UINT8_C(183), UINT8_C(  7),
                             UINT8_C(134), UINT8_C(138), UINT8_C(158), UINT8_C(199),
                             UINT8_C( 65), UINT8_C( 92), UINT8_C(182), UINT8_C(118)),
      simde_x_mm512_set_epu8(UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
                             UINT8_C(  0), UINT8_C(177), UINT8_C(  0), UINT8_C( 67),
                             UINT8_C(236), UINT8_C( 45), UINT8_C(  0), UINT8_C(157),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(200), UINT8_C(  0),
                             UINT8_C( 60), UINT8_C(  0), UINT8_C( 93), UINT8_C(191),
                             UINT8_C( 99), UINT8_C(  0), UINT8_C(111), UINT8_C(125),
                             UINT8_C(  0), UINT8_C(  0), UINT8_C(126), UINT8_C(164),
                             UINT8_C( 56), UINT8_C(112), UINT8_C(  0), UINT8_C(112)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_avg_epu8(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_u8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_avg_epu16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_x_mm512_set_epu16(UINT16_C( 13922), UINT16_C( 36897), UINT16_C( 58180), UINT16_C( 11516),
                              UINT16_C( 48259), UINT16_C(  5769), UINT16_C( 28390), UINT16_C( 29726),
                              UINT16_C( 60808), UINT16_C( 38778), UINT16_C( 21386), UINT16_C( 37563),
                              UINT16_C( 21427), UINT16_C( 42840), UINT16_C( 63576), UINT16_C( 25988),
                              UINT16_C( 52004), UINT16_C( 13742), UINT16_C( 20065), UINT16_C( 31945),
                              UINT16_C(  3305), UINT16_C( 50297), UINT16_C( 19714), UINT16_C( 48635),
                              UINT16_C( 64641), UINT16_C(  9835), UINT16_C( 33507), UINT16_C( 40606),
                              UINT16_C( 49930), UINT16_C( 25911), UINT16_C( 16490), UINT16_C( 30266)),
      simde_x_mm512_set_epu16(UINT16_C( 35675), UINT16_C(  3257), UINT16_C( 12418), UINT16_C( 59056),
                              UINT16_C( 54292), UINT16_C( 28377), UINT16_C( 47802), UINT16_C( 33906),
                              UINT16_C( 48379), UINT16_C( 57830), UINT16_C( 42348), UINT16_C(  4361),
                              UINT16_C( 62947), UINT16_C( 32240), UINT16_C( 12123), UINT16_C(   903),
                              UINT16_C( 62311), UINT16_C( 39097), UINT16_C( 12552), UINT16_C( 17043),
                              UINT16_C( 12967), UINT16_C( 14694), UINT16_C( 36399), UINT16_C( 44335),
                              UINT16_C( 44269), UINT16_C( 14847), UINT16_C( 26083), UINT16_C( 50887),
                              UINT16_C(  6575), UINT16_C( 58796), UINT16_C( 53958), UINT16_C( 55190)),
      simde_x_mm512_set_epu16(UINT16_C( 24799), UINT16_C( 20077), UINT16_C( 35299), UINT16_C( 35286),
                              UINT16_C( 51276), UINT16_C( 17073), UINT16_C( 38096), UINT16_C( 31816),
                              UINT16_C( 54594), UINT16_C( 48304), UINT16_C( 31867), UINT16_C( 20962),
                              UINT16_C( 42187), UINT16_C( 37540), UINT16_C( 37850), UINT16_C( 13446),
                              UINT16_C( 57158), UINT16_C( 26420), UINT16_C( 16309), UINT16_C( 24494),
                              UINT16_C(  8136), UINT16_C( 32496), UINT16_C( 28057), UINT16_C( 46485),
                              UINT16_C( 54455), UINT16_C( 12341), UINT16_C( 29795), UINT16_C( 45747),
                              UINT16_C( 28253), UINT16_C( 42354), UINT16_C( 35224), UINT16_C( 42728)) },
    { simde_x_mm512_set_epu16(UINT16_C( 53467), UINT16_C( 14812), UINT16_C(  1601), UINT16_C(  5446),
                              UINT16_C( 10821), UINT16_C( 52555), UINT16_C( 11613), UINT16_C( 32262),
                              UINT16_C( 44594), UINT16_C(  4573), UINT16_C( 55679), UINT16_C( 28536),
                              UINT16_C( 33684), UINT16_C( 36248), UINT16_C( 37000), UINT16_C( 16957),
                              UINT16_C( 33904), UINT16_C(  7320), UINT16_C( 32743), UINT16_C( 28065),
                              UINT16_C( 40454), UINT16_C( 54701), UINT16_C( 30387), UINT16_C( 20402),
                              UINT16_C( 12012), UINT16_C( 12600), UINT16_C( 40321), UINT16_C( 11976),
                              UINT16_C( 22914), UINT16_C( 48119), UINT16_C( 54567), UINT16_C( 37518)),
      simde_x_mm512_set_epu16(UINT16_C( 19683), UINT16_C( 49281), UINT16_C( 58394), UINT16_C( 42390),
                              UINT16_C( 28448), UINT16_C(  2882), UINT16_C( 16746), UINT16_C( 53644),
                              UINT16_C( 16898), UINT16_C( 40318), UINT16_C( 16648), UINT16_C( 14647),
                              UINT16_C( 38440), UINT16_C( 42038), UINT16_C( 24217), UINT16_C( 60044),
                              UINT16_C( 11584), UINT16_C( 35025), UINT16_C( 26069), UINT16_C( 37885),
                              UINT16_C( 14708), UINT16_C( 24663), UINT16_C( 44110), UINT16_C( 40062),
                              UINT16_C(   918), UINT16_C( 34061), UINT16_C( 57874), UINT16_C( 27338),
                              UINT16_C( 24412), UINT16_C( 15314), UINT16_C( 35927), UINT16_C( 54580)),
      simde_x_mm512_set_epu16(UINT16_C( 36575), UINT16_C( 32047), UINT16_C( 29998), UINT16_C( 23918),
                              UINT16_C( 19635), UINT16_C( 27719), UINT16_C( 14180), UINT16_C( 42953),
                              UINT16_C( 30746), UINT16_C( 22446), UINT16_C( 36164), UINT16_C( 21592),
                              UINT16_C( 36062), UINT16_C( 39143), UINT16_C( 30609), UINT16_C( 38501),
                              UINT16_C( 22744), UINT16_C( 21173), UINT16_C( 29406), UINT16_C( 32975),
                              UINT16_C( 27581), UINT16_C( 39682), UINT16_C( 37249), UINT16_C( 30232),
                              UINT16_C(  6465), UINT16_C( 23331), UINT16_C( 49098), UINT16_C( 19657),
                              UINT16_C( 23663), UINT16_C( 31717), UINT16_C( 45247), UINT16_C( 46049)) },
    { simde_x_mm512_set_epu16(UINT16_C(  2926), UINT16_C( 37218), UINT16_C( 49262), UINT16_C( 41266),
                              UINT16_C( 64616), UINT16_C( 47887), UINT16_C( 47046), UINT16_C( 20233),
                              UINT16_C( 17531), UINT16_C( 35216), UINT16_C( 26970), UINT16_C( 22110),
                              UINT16_C( 23885), UINT16_C( 49160), UINT16_C( 11289), UINT16_C( 22415),
                              UINT16_C( 57944), UINT16_C( 39676), UINT16_C( 14246), UINT16_C( 36547),
                              UINT16_C( 43442), UINT16_C( 52571), UINT16_C( 16838), UINT16_C( 32565),
                              UINT16_C( 62744), UINT16_C( 45552), UINT16_C( 50968), UINT16_C( 36716),
                              UINT16_C( 10154), UINT16_C( 41950), UINT16_C( 14496), UINT16_C( 51322)),
      simde_x_mm512_set_epu16(UINT16_C( 14640), UINT16_C( 63580), UINT16_C( 14746), UINT16_C( 21373),
                              UINT16_C(  2714), UINT16_C( 18543), UINT16_C( 17198), UINT16_C( 32202),
                              UINT16_C( 11581), UINT16_C( 13480), UINT16_C( 54288), UINT16_C(  9603),
                              UINT16_C( 12378), UINT16_C( 53993), UINT16_C( 37890), UINT16_C(  2622),
                              UINT16_C( 22964), UINT16_C( 24086), UINT16_C( 65101), UINT16_C(   151),
                              UINT16_C( 54310), UINT16_C( 27341), UINT16_C( 18591), UINT16_C( 57907),
                              UINT16_C( 13730), UINT16_C( 48472), UINT16_C( 10888), UINT16_C( 26832),
                              UINT16_C( 39665), UINT16_C( 48015), UINT16_C( 40287), UINT16_C(  7521)),
      simde_x_mm512_set_epu16(UINT16_C(  8783), UINT16_C( 50399), UINT16_C( 32004), UINT16_C( 31320),
                              UINT16_C( 33665), UINT16_C( 33215), UINT16_C( 32122), UINT16_C( 26218),
                              UINT16_C( 14556), UINT16_C( 24348), UINT16_C( 40629), UINT16_C( 15857),
                              UINT16_C( 18132), UINT16_C( 51577), UINT16_C( 24590), UINT16_C( 12519),
                              UINT16_C( 40454), UINT16_C( 31881), UINT16_C( 39674), UINT16_C( 18349),
                              UINT16_C( 48876), UINT16_C( 39956), UINT16_C( 17715), UINT16_C( 45236),
                              UINT16_C( 38237), UINT16_C( 47012), UINT16_C( 30928), UINT16_C( 31774),
                              UINT16_C( 24910), UINT16_C( 44983), UINT16_C( 27392), UINT16_C( 29422)) },
    { simde_x_mm512_set_epu16(UINT16_C( 33359), UINT16_C( 20997), UINT16_C( 60535), UINT16_C(  6477),
                              UINT16_C( 35687), UINT16_C( 52277), UINT16_C( 18001), UINT16_C(  9114),
                              UINT16_C( 58688), UINT16_C( 24195), UINT16_C( 55556), UINT16_C( 26572),
                              UINT16_C( 13614), UINT16_C( 31611), UINT16_C( 64483), UINT16_C( 39184),
                              UINT16_C( 20423), UINT16_C(  4017), UINT16_C( 21263), UINT16_C(  9687),
                              UINT16_C( 45713), UINT16_C( 61493), UINT16_C( 29891), UINT16_C( 26099),
                              UINT16_C( 34397), UINT16_C( 16487), UINT16_C( 18770), UINT16_C( 48421),
                              UINT16_C( 51774), UINT16_C( 40289), UINT16_C( 36358), UINT16_C(  9752)),
      simde_x_mm512_set_epu16(UINT16_C( 14685), UINT16_C( 36894), UINT16_C(  3039), UINT16_C( 14211),
                              UINT16_C(  7600), UINT16_C( 13396), UINT16_C( 20271), UINT16_C( 32796),
                              UINT16_C( 18819), UINT16_C( 35329), UINT16_C( 25198), UINT16_C( 53928),
                              UINT16_C( 53351), UINT16_C( 38714), UINT16_C( 38160), UINT16_C( 28969),
                              UINT16_C( 51469), UINT16_C( 54246), UINT16_C( 53402), UINT16_C( 12671),
                              UINT16_C( 23815), UINT16_C( 34869), UINT16_C(  4959), UINT16_C( 48021),
                              UINT16_C(   161), UINT16_C( 58452), UINT16_C( 21609), UINT16_C( 23057),
                              UINT16_C(   951), UINT16_C( 33560), UINT16_C( 51461), UINT16_C( 32848)),
      simde_x_mm512_set_epu16(UINT16_C( 24022), UINT16_C( 28946), UINT16_C( 31787), UINT16_C( 10344),
                              UINT16_C( 21644), UINT16_C( 32837), UINT16_C( 19136), UINT16_C( 20955),
                              UINT16_C( 38754), UINT16_C( 29762), UINT16_C( 40377), UINT16_C( 40250),
                              UINT16_C( 33483), UINT16_C( 35163), UINT16_C( 51322), UINT16_C( 34077),
                              UINT16_C( 35946), UINT16_C( 29132), UINT16_C( 37333), UINT16_C( 11179),
                              UINT16_C( 34764), UINT16_C( 48181), UINT16_C( 17425), UINT16_C( 37060),
                              UINT16_C( 17279), UINT16_C( 37470), UINT16_C( 20190), UINT16_C( 35739),
                              UINT16_C( 26363), UINT16_C( 36925), UINT16_C( 43910), UINT16_C( 21300)) },
    { simde_x_mm512_set_epu16(UINT16_C( 16482), UINT16_C( 29865), UINT16_C( 43161), UINT16_C( 49292),
                              UINT16_C( 20253), UINT16_C( 46312), UINT16_C( 22868), UINT16_C( 35970),
                              UINT16_C( 27832), UINT16_C( 29779), UINT16_C( 41321), UINT16_C(  4471),
                              UINT16_C(   101), UINT16_C( 26727), UINT16_C( 57916), UINT16_C(  6017),
                              UINT16_C( 36275), UINT16_C( 22385), UINT16_C( 64475), UINT16_C( 27511),
                              UINT16_C( 18089), UINT16_C(  2907), UINT16_C(  6044), UINT16_C( 25380),
                              UINT16_C( 39450), UINT16_C(  6056), UINT16_C(  3173), UINT16_C( 48508),
                              UINT16_C( 39476), UINT16_C( 34432), UINT16_C( 54613), UINT16_C( 48593)),
      simde_x_mm512_set_epu16(UINT16_C(  5403), UINT16_C( 14767), UINT16_C( 48931), UINT16_C( 19205),
                              UINT16_C( 43050), UINT16_C( 50911), UINT16_C( 61409), UINT16_C( 58117),
                              UINT16_C( 21956), UINT16_C( 47259), UINT16_C( 18797), UINT16_C( 33639),
                              UINT16_C( 33691), UINT16_C( 26950), UINT16_C( 42098), UINT16_C( 29396),
                              UINT16_C( 40226), UINT16_C( 56570), UINT16_C( 65251), UINT16_C( 61481),
                              UINT16_C( 53274), UINT16_C( 56708), UINT16_C( 31154), UINT16_C( 56119),
                              UINT16_C( 11598), UINT16_C( 59701), UINT16_C( 22444), UINT16_C( 31379),
                              UINT16_C( 47987), UINT16_C(  5886), UINT16_C(  2522), UINT16_C( 57558)),
      simde_x_mm512_set_epu16(UINT16_C( 10943), UINT16_C( 22316), UINT16_C( 46046), UINT16_C( 34249),
                              UINT16_C( 31652), UINT16_C( 48612), UINT16_C( 42139), UINT16_C( 47044),
                              UINT16_C( 24894), UINT16_C( 38519), UINT16_C( 30059), UINT16_C( 19055),
                              UINT16_C( 16896), UINT16_C( 26839), UINT16_C( 50007), UINT16_C( 17707),
                              UINT16_C( 38251), UINT16_C( 39478), UINT16_C( 64863), UINT16_C( 44496),
                              UINT16_C( 35682), UINT16_C( 29808), UINT16_C( 18599), UINT16_C( 40750),
                              UINT16_C( 25524), UINT16_C( 32879), UINT16_C( 12809), UINT16_C( 39944),
                              UINT16_C( 43732), UINT16_C( 20159), UINT16_C( 28568), UINT16_C( 53076)) },
    { simde_x_mm512_set_epu16(UINT16_C( 34893), UINT16_C( 23679), UINT16_C( 64034), UINT16_C( 46251),
                              UINT16_C(  8048), UINT16_C( 41654), UINT16_C( 61576), UINT16_C( 24661),
                              UINT16_C( 10548), UINT16_C( 15307), UINT16_C( 54631), UINT16_C( 56752),
                              UINT16_C( 25346), UINT16_C(  5558), UINT16_C(  1968), UINT16_C( 32109),
                              UINT16_C( 59890), UINT16_C(  1389), UINT16_C( 18928), UINT16_C( 44824),
                              UINT16_C( 11367), UINT16_C( 52264), UINT16_C( 20085), UINT16_C( 56744),
                              UINT16_C( 49674), UINT16_C(  4067), UINT16_C(  3569), UINT16_C( 23831),
                              UINT16_C(  8020), UINT16_C( 56140), UINT16_C( 13343), UINT16_C(  5815)),
      simde_x_mm512_set_epu16(UINT16_C(    70), UINT16_C( 49524), UINT16_C(  2389), UINT16_C( 48794),
                              UINT16_C(  9656), UINT16_C( 49515), UINT16_C( 20459), UINT16_C(  8573),
                              UINT16_C( 28524), UINT16_C( 25506), UINT16_C( 45011), UINT16_C( 20391),
                              UINT16_C( 42585), UINT16_C(  5514), UINT16_C(  3945), UINT16_C( 49101),
                              UINT16_C(  8826), UINT16_C( 53647), UINT16_C( 55458), UINT16_C( 61374),
                              UINT16_C( 54518), UINT16_C( 64462), UINT16_C( 15696), UINT16_C( 38749),
                              UINT16_C( 14471), UINT16_C( 43272), UINT16_C( 48962), UINT16_C( 15384),
                              UINT16_C( 23964), UINT16_C(  8219), UINT16_C( 12498), UINT16_C( 59146)),
      simde_x_mm512_set_epu16(UINT16_C( 17482), UINT16_C( 36602), UINT16_C( 33212), UINT16_C( 47523),
                              UINT16_C(  8852), UINT16_C( 45585), UINT16_C( 41018), UINT16_C( 16617),
                              UINT16_C( 19536), UINT16_C( 20407), UINT16_C( 49821), UINT16_C( 38572),
                              UINT16_C( 33966), UINT16_C(  5536), UINT16_C(  2957), UINT16_C( 40605),
                              UINT16_C( 34358), UINT16_C( 27518), UINT16_C( 37193), UINT16_C( 53099),
                              UINT16_C( 32943), UINT16_C( 58363), UINT16_C( 17891), UINT16_C( 47747),
                              UINT16_C( 32073), UINT16_C( 23670), UINT16_C( 26266), UINT16_C( 19608),
                              UINT16_C( 15992), UINT16_C( 32180), UINT16_C( 12921), UINT16_C( 32481)) },
    { simde_x_mm512_set_epu16(UINT16_C( 60096), UINT16_C( 60141), UINT16_C(  1401), UINT16_C( 46103),
                              UINT16_C( 12472), UINT16_C( 53354), UINT16_C( 24302), UINT16_C( 31759),
                              UINT16_C( 31691), UINT16_C(  4110), UINT16_C( 33754), UINT16_C( 25522),
                              UINT16_C( 55741), UINT16_C( 54238), UINT16_C(  6937), UINT16_C( 14267),
                              UINT16_C(  6394), UINT16_C( 11471), UINT16_C( 36806), UINT16_C( 29444),
                              UINT16_C(  4686), UINT16_C( 24894), UINT16_C(  9912), UINT16_C( 55420),
                              UINT16_C( 39725), UINT16_C( 16932), UINT16_C(  1814), UINT16_C( 27150),
                              UINT16_C( 14089), UINT16_C( 26760), UINT16_C( 42881), UINT16_C(  1890)),
      simde_x_mm512_set_epu16(UINT16_C(  1052), UINT16_C( 45798), UINT16_C(  2128), UINT16_C( 35449),
                              UINT16_C( 36274), UINT16_C( 38472), UINT16_C( 30996), UINT16_C(  7411),
                              UINT16_C( 65433), UINT16_C( 20087), UINT16_C(  6070), UINT16_C( 34216),
                              UINT16_C( 38017), UINT16_C( 52174), UINT16_C(  6632), UINT16_C( 18268),
                              UINT16_C( 23199), UINT16_C( 50262), UINT16_C( 41833), UINT16_C( 43512),
                              UINT16_C( 43349), UINT16_C( 45458), UINT16_C( 19756), UINT16_C( 38464),
                              UINT16_C( 37153), UINT16_C( 41983), UINT16_C(  8534), UINT16_C(   930),
                              UINT16_C( 61899), UINT16_C(  3423), UINT16_C( 63286), UINT16_C( 18842)),
      simde_x_mm512_set_epu16(UINT16_C( 30574), UINT16_C( 52970), UINT16_C(  1765), UINT16_C( 40776),
                              UINT16_C( 24373), UINT16_C( 45913), UINT16_C( 27649), UINT16_C( 19585),
                              UINT16_C( 48562), UINT16_C( 12099), UINT16_C( 19912), UINT16_C( 29869),
                              UINT16_C( 46879), UINT16_C( 53206), UINT16_C(  6785), UINT16_C( 16268),
                              UINT16_C( 14797), UINT16_C( 30867), UINT16_C( 39320), UINT16_C( 36478),
                              UINT16_C( 24018), UINT16_C( 35176), UINT16_C( 14834), UINT16_C( 46942),
                              UINT16_C( 38439), UINT16_C( 29458), UINT16_C(  5174), UINT16_C( 14040),
                              UINT16_C( 37994), UINT16_C( 15092), UINT16_C( 53084), UINT16_C( 10366)) },
    { simde_x_mm512_set_epu16(UINT16_C( 21390), UINT16_C( 10972), UINT16_C(  9534), UINT16_C( 25326),
                              UINT16_C( 39196), UINT16_C( 52858), UINT16_C( 21878), UINT16_C( 44949),
                              UINT16_C( 56397), UINT16_C(  7489), UINT16_C( 51094), UINT16_C( 21965),
                              UINT16_C( 45995), UINT16_C( 12111), UINT16_C( 47805), UINT16_C(  1368),
                              UINT16_C( 47857), UINT16_C( 52094), UINT16_C( 29749), UINT16_C( 11216),
                              UINT16_C( 36270), UINT16_C( 43041), UINT16_C( 37100), UINT16_C( 23737),
                              UINT16_C( 49684), UINT16_C(  1618), UINT16_C(  4508), UINT16_C( 49414),
                              UINT16_C( 16652), UINT16_C( 21178), UINT16_C(  2342), UINT16_C( 41466)),
      simde_x_mm512_set_epu16(UINT16_C(  2029), UINT16_C( 56235), UINT16_C( 60520), UINT16_C( 21505),
                              UINT16_C( 57936), UINT16_C(  7393), UINT16_C( 48841), UINT16_C( 33423),
                              UINT16_C( 21974), UINT16_C( 41720), UINT16_C( 60540), UINT16_C( 41725),
                              UINT16_C( 29801), UINT16_C( 64766), UINT16_C( 10276), UINT16_C( 10419),
                              UINT16_C( 29113), UINT16_C(  9717), UINT16_C( 10502), UINT16_C( 53810),
                              UINT16_C( 21215), UINT16_C( 51746), UINT16_C( 31756), UINT16_C( 40043),
                              UINT16_C( 19421), UINT16_C( 27841), UINT16_C(  1136), UINT16_C( 29043),
                              UINT16_C( 64035), UINT16_C( 41226), UINT16_C( 38869), UINT16_C( 32769)),
      simde_x_mm512_set_epu16(UINT16_C( 11710), UINT16_C( 33604), UINT16_C( 35027), UINT16_C( 23416),
                              UINT16_C( 48566), UINT16_C( 30126), UINT16_C( 35360), UINT16_C( 39186),
                              UINT16_C( 39186), UINT16_C( 24605), UINT16_C( 55817), UINT16_C( 31845),
                              UINT16_C( 37898), UINT16_C( 38439), UINT16_C( 29041), UINT16_C(  5894),
                              UINT16_C( 38485), UINT16_C( 30906), UINT16_C( 20126), UINT16_C( 32513),
                              UINT16_C( 28743), UINT16_C( 47394), UINT16_C( 34428), UINT16_C( 31890),
                              UINT16_C( 34553), UINT16_C( 14730), UINT16_C(  2822), UINT16_C( 39229),
                              UINT16_C( 40344), UINT16_C( 31202), UINT16_C( 20606), UINT16_C( 37118)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_avg_epu16(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_u16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_avg_epu16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const uint16_t src[32];
    const simde__mmask32 k;
    const uint16_t a[32];
    const uint16_t b[32];
    const uint16_t r[32];
  } test_vec[] = {
    { { UINT16_C(19100), UINT16_C(47010), UINT16_C(60087), UINT16_C(28088), UINT16_C(36628), UINT16_C( 9660), UINT16_C(16286), UINT16_C( 8161),
        UINT16_C(63657), UINT16_C(63180), UINT16_C( 6828), UINT16_C(46048), UINT16_C(34775), UINT16_C(18101), UINT16_C(58268), UINT16_C(14486),
        UINT16_C(14382), UINT16_C(58864), UINT16_C(43042), UINT16_C(14162), UINT16_C( 3895), UINT16_C(54876), UINT16_C(15950), UINT16_C(63733),
        UINT16_C(49462), UINT16_C(58350), UINT16_C(53212), UINT16_C(45974), UINT16_C(19286), UINT16_C(62458), UINT16_C(36911), UINT16_C(23851) },
      UINT32_C(3946978248),
      { UINT16_C(38340), UINT16_C(64290), UINT16_C(32420), UINT16_C(62161), UINT16_C(51132), UINT16_C(62442), UINT16_C(55688), UINT16_C(25814),
        UINT16_C(27816), UINT16_C(65048), UINT16_C( 4791), UINT16_C(59121), UINT16_C( 7586), UINT16_C(27203), UINT16_C(34360), UINT16_C(64597),
        UINT16_C(30491), UINT16_C(49144), UINT16_C(51702), UINT16_C(45745), UINT16_C(40080), UINT16_C( 6565), UINT16_C(31605), UINT16_C( 7549),
        UINT16_C(38375), UINT16_C(40731), UINT16_C( 3495), UINT16_C(18821), UINT16_C(51498), UINT16_C(25268), UINT16_C( 2383), UINT16_C(27231) },
      { UINT16_C(22401), UINT16_C(30505), UINT16_C(55840), UINT16_C(45353), UINT16_C(53110), UINT16_C(60362), UINT16_C(18250), UINT16_C(12808),
        UINT16_C( 9437), UINT16_C(34001), UINT16_C(22065), UINT16_C(23502), UINT16_C(33311), UINT16_C(28349), UINT16_C( 7307), UINT16_C( 3288),
        UINT16_C(  371), UINT16_C(38019), UINT16_C(44508), UINT16_C(21061), UINT16_C( 3964), UINT16_C(50750), UINT16_C(18006), UINT16_C(13304),
        UINT16_C(51562), UINT16_C(39864), UINT16_C(34336), UINT16_C(16374), UINT16_C(46088), UINT16_C(37806), UINT16_C(34512), UINT16_C(17568) },
      { UINT16_C(19100), UINT16_C(47010), UINT16_C(60087), UINT16_C(53757), UINT16_C(36628), UINT16_C( 9660), UINT16_C(36969), UINT16_C(19311),
        UINT16_C(18627), UINT16_C(49525), UINT16_C( 6828), UINT16_C(41312), UINT16_C(20449), UINT16_C(18101), UINT16_C(58268), UINT16_C(14486),
        UINT16_C(14382), UINT16_C(43582), UINT16_C(43042), UINT16_C(14162), UINT16_C( 3895), UINT16_C(54876), UINT16_C(24806), UINT16_C(63733),
        UINT16_C(44969), UINT16_C(40298), UINT16_C(53212), UINT16_C(17598), UINT16_C(19286), UINT16_C(31537), UINT16_C(18448), UINT16_C(22400) } },
    { { UINT16_C( 9096), UINT16_C(25816), UINT16_C( 7632), UINT16_C(19638), UINT16_C(62508), UINT16_C(33299), UINT16_C( 2875), UINT16_C(42422),
        UINT16_C(28373), UINT16_C(62785), UINT16_C(14324), UINT16_C(64564), UINT16_C(58091), UINT16_C(48271), UINT16_C(12137), UINT16_C(61696),
        UINT16_C(55379), UINT16_C( 9045), UINT16_C( 3061), UINT16_C( 8560), UINT16_C(33536), UINT16_C(15267), UINT16_C(22926), UINT16_C(25568),
        UINT16_C( 8647), UINT16_C(47960), UINT16_C(36185), UINT16_C(17591), UINT16_C(18287), UINT16_C(55296), UINT16_C(  118), UINT16_C(51657) },
      UINT32_C(3454869208),
      { UINT16_C(23850), UINT16_C(10990), UINT16_C(37600), UINT16_C(28261), UINT16_C(17899), UINT16_C(46034), UINT16_C(10855), UINT16_C(49262),
        UINT16_C( 9911), UINT16_C( 9988), UINT16_C( 1389), UINT16_C(58367), UINT16_C(51461), UINT16_C(57005), UINT16_C(39655), UINT16_C( 4523),
        UINT16_C(39671), UINT16_C(55099), UINT16_C(41004), UINT16_C( 5957), UINT16_C( 6118), UINT16_C(19914), UINT16_C(14658), UINT16_C(63757),
        UINT16_C( 4447), UINT16_C(52256), UINT16_C( 8214), UINT16_C( 7343), UINT16_C(23785), UINT16_C(53498), UINT16_C(42486), UINT16_C(60898) },
      { UINT16_C( 7487), UINT16_C(27588), UINT16_C( 2750), UINT16_C(42115), UINT16_C(19745), UINT16_C(25585), UINT16_C(65158), UINT16_C(58717),
        UINT16_C(32015), UINT16_C( 9905), UINT16_C(24989), UINT16_C(34370), UINT16_C(15549), UINT16_C(46167), UINT16_C(14817), UINT16_C( 8609),
        UINT16_C(26198), UINT16_C( 5260), UINT16_C( 3952), UINT16_C(37304), UINT16_C(43357), UINT16_C(58357), UINT16_C(21159), UINT16_C(47049),
        UINT16_C(31439), UINT16_C(28125), UINT16_C( 8155), UINT16_C(39411), UINT16_C(19035), UINT16_C(15437), UINT16_C(61059), UINT16_C(55901) },
      { UINT16_C( 9096), UINT16_C(25816), UINT16_C( 7632), UINT16_C(35188), UINT16_C(18822), UINT16_C(33299), UINT16_C(38007), UINT16_C(53990),
        UINT16_C(28373), UINT16_C( 9947), UINT16_C(13189), UINT16_C(46369), UINT16_C(33505), UINT16_C(48271), UINT16_C(12137), UINT16_C(61696),
        UINT16_C(32935), UINT16_C( 9045), UINT16_C(22478), UINT16_C(21631), UINT16_C(33536), UINT16_C(39136), UINT16_C(17909), UINT16_C(55403),
        UINT16_C(17943), UINT16_C(47960), UINT16_C( 8185), UINT16_C(23377), UINT16_C(18287), UINT16_C(55296), UINT16_C(51773), UINT16_C(58400) } },
    { { UINT16_C(59988), UINT16_C(50414), UINT16_C(43001), UINT16_C(22102), UINT16_C(19280), UINT16_C(63546), UINT16_C(  925), UINT16_C(27823),
        UINT16_C(35965), UINT16_C(23001), UINT16_C(52651), UINT16_C( 1778), UINT16_C(16151), UINT16_C(39746), UINT16_C(41005), UINT16_C(33397),
        UINT16_C(25482), UINT16_C(33606), UINT16_C(39946), UINT16_C(23514), UINT16_C( 5351), UINT16_C(33875), UINT16_C(  535), UINT16_C(38129),
        UINT16_C(51854), UINT16_C(14829), UINT16_C(57239), UINT16_C(44863), UINT16_C(33054), UINT16_C(19530), UINT16_C(48929), UINT16_C(43982) },
      UINT32_C( 758060066),
      { UINT16_C( 2481), UINT16_C(39048), UINT16_C(56093), UINT16_C(13341), UINT16_C( 3805), UINT16_C(27592), UINT16_C(46808), UINT16_C(28836),
        UINT16_C(58261), UINT16_C(46111), UINT16_C(26980), UINT16_C(34304), UINT16_C(52776), UINT16_C(18993), UINT16_C(24802), UINT16_C(37751),
        UINT16_C(65385), UINT16_C(34348), UINT16_C(18906), UINT16_C(47034), UINT16_C(33623), UINT16_C(12066), UINT16_C(50745), UINT16_C(52895),
        UINT16_C(48809), UINT16_C( 3714), UINT16_C(33319), UINT16_C(20372), UINT16_C(50512), UINT16_C(13210), UINT16_C( 4390), UINT16_C(36806) },
      { UINT16_C(61969), UINT16_C(60182), UINT16_C(53307), UINT16_C(37539), UINT16_C(50515), UINT16_C(36034), UINT16_C(24972), UINT16_C(13659),
        UINT16_C(56608), UINT16_C(18243), UINT16_C(55136), UINT16_C(45207), UINT16_C(12701), UINT16_C(50147), UINT16_C(43586), UINT16_C(21330),
        UINT16_C(26780), UINT16_C(55359), UINT16_C(57913), UINT16_C(35946), UINT16_C(11431), UINT16_C(13081), UINT16_C(29838), UINT16_C(44649),
        UINT16_C(44113), UINT16_C(45557), UINT16_C(35972), UINT16_C( 8546), UINT16_C(17853), UINT16_C(  228), UINT16_C(14063), UINT16_C(35923) },
      { UINT16_C(59988), UINT16_C(49615), UINT16_C(43001), UINT16_C(22102), UINT16_C(19280), UINT16_C(31813), UINT16_C(  925), UINT16_C(27823),
        UINT16_C(35965), UINT16_C(23001), UINT16_C(41058), UINT16_C( 1778), UINT16_C(32739), UINT16_C(39746), UINT16_C(41005), UINT16_C(33397),
        UINT16_C(46083), UINT16_C(44854), UINT16_C(38410), UINT16_C(41490), UINT16_C( 5351), UINT16_C(12574), UINT16_C(  535), UINT16_C(38129),
        UINT16_C(46461), UINT16_C(14829), UINT16_C(34646), UINT16_C(14459), UINT16_C(33054), UINT16_C( 6719), UINT16_C(48929), UINT16_C(43982) } },
    { { UINT16_C(37535), UINT16_C(55396), UINT16_C(52852), UINT16_C( 7268), UINT16_C(32251), UINT16_C(35151), UINT16_C(47345), UINT16_C(17207),
        UINT16_C(11365), UINT16_C(59892), UINT16_C(22201), UINT16_C(30218), UINT16_C(61084), UINT16_C(35702), UINT16_C(51748), UINT16_C(49943),
        UINT16_C(31580), UINT16_C(53659), UINT16_C(   74), UINT16_C(17901), UINT16_C(15485), UINT16_C(28622), UINT16_C( 1525), UINT16_C(23218),
        UINT16_C(42545), UINT16_C(59971), UINT16_C(19965), UINT16_C(39265), UINT16_C(55099), UINT16_C(24356), UINT16_C(15521), UINT16_C(65059) },
      UINT32_C(  30391991),
      { UINT16_C(48318), UINT16_C(15430), UINT16_C( 5368), UINT16_C(60843), UINT16_C(23833), UINT16_C(19271), UINT16_C(35331), UINT16_C(   53),
        UINT16_C(38615), UINT16_C( 4761), UINT16_C(48750), UINT16_C( 3954), UINT16_C(38394), UINT16_C(45325), UINT16_C(56403), UINT16_C( 4787),
        UINT16_C(63896), UINT16_C(37198), UINT16_C(63758), UINT16_C(10110), UINT16_C(50774), UINT16_C(22898), UINT16_C(43088), UINT16_C(10330),
        UINT16_C(62270), UINT16_C(44090), UINT16_C(44209), UINT16_C(43964), UINT16_C(51521), UINT16_C(38237), UINT16_C( 4262), UINT16_C(16039) },
      { UINT16_C(62729), UINT16_C( 6095), UINT16_C(20206), UINT16_C(17471), UINT16_C(45332), UINT16_C(25757), UINT16_C(63321), UINT16_C(39052),
        UINT16_C(51179), UINT16_C(40004), UINT16_C(  115), UINT16_C(46408), UINT16_C(42442), UINT16_C(28746), UINT16_C(61877), UINT16_C(48814),
        UINT16_C(32486), UINT16_C(54486), UINT16_C( 5580), UINT16_C(57368), UINT16_C(46534), UINT16_C( 8260), UINT16_C(53677), UINT16_C(39096),
        UINT16_C(64664), UINT16_C( 2868), UINT16_C(31997), UINT16_C(51136), UINT16_C( 2593), UINT16_C(54839), UINT16_C(58875), UINT16_C(57749) },
      { UINT16_C(55524), UINT16_C(10763), UINT16_C(12787), UINT16_C( 7268), UINT16_C(34583), UINT16_C(22514), UINT16_C(47345), UINT16_C(19553),
        UINT16_C(11365), UINT16_C(22383), UINT16_C(24433), UINT16_C(25181), UINT16_C(40418), UINT16_C(37036), UINT16_C(51748), UINT16_C(26801),
        UINT16_C(48191), UINT16_C(45842), UINT16_C(34669), UINT16_C(33739), UINT16_C(15485), UINT16_C(28622), UINT16_C(48383), UINT16_C(24713),
        UINT16_C(63467), UINT16_C(59971), UINT16_C(19965), UINT16_C(39265), UINT16_C(55099), UINT16_C(24356), UINT16_C(15521), UINT16_C(65059) } },
    { { UINT16_C(27491), UINT16_C(12213), UINT16_C(52608), UINT16_C(17935), UINT16_C(21635), UINT16_C(12390), UINT16_C( 7717), UINT16_C(48584),
        UINT16_C(64539), UINT16_C( 6344), UINT16_C(35193), UINT16_C(39647), UINT16_C( 5779), UINT16_C(36721), UINT16_C( 1787), UINT16_C(24432),
        UINT16_C( 9841), UINT16_C(61838), UINT16_C(40691), UINT16_C(30263), UINT16_C(40690), UINT16_C( 6054), UINT16_C(28348), UINT16_C(55252),
        UINT16_C(40043), UINT16_C(58607), UINT16_C(52773), UINT16_C(47486), UINT16_C(61412), UINT16_C(57416), UINT16_C(47349), UINT16_C(26175) },
      UINT32_C(3528969694),
      { UINT16_C(36715), UINT16_C(23880), UINT16_C(61229), UINT16_C(59764), UINT16_C(18525), UINT16_C(51393), UINT16_C(45285), UINT16_C( 2732),
        UINT16_C(11135), UINT16_C(25539), UINT16_C( 2842), UINT16_C( 4163), UINT16_C(33476), UINT16_C(41590), UINT16_C(52816), UINT16_C(47988),
        UINT16_C(48477), UINT16_C(35353), UINT16_C(36268), UINT16_C( 2419), UINT16_C(13526), UINT16_C(48082), UINT16_C(32485), UINT16_C(25797),
        UINT16_C(35241), UINT16_C(50375), UINT16_C( 2964), UINT16_C(22740), UINT16_C(19085), UINT16_C(56827), UINT16_C(28440), UINT16_C(30105) },
      { UINT16_C(45612), UINT16_C(55551), UINT16_C(29503), UINT16_C( 5602), UINT16_C(46247), UINT16_C(36048), UINT16_C(38450), UINT16_C(56560),
        UINT16_C(47135), UINT16_C(45984), UINT16_C(29891), UINT16_C(20492), UINT16_C( 1982), UINT16_C(55086), UINT16_C(51062), UINT16_C(41804),
        UINT16_C(19577), UINT16_C(47227), UINT16_C(23999), UINT16_C(26318), UINT16_C(40465), UINT16_C(17651), UINT16_C(58164), UINT16_C(21280),
        UINT16_C(49307), UINT16_C(24071), UINT16_C( 4916), UINT16_C(62127), UINT16_C(56602), UINT16_C(37065), UINT16_C( 5796), UINT16_C( 7475) },
      { UINT16_C(27491), UINT16_C(39716), UINT16_C(45366), UINT16_C(32683), UINT16_C(32386), UINT16_C(12390), UINT16_C(41868), UINT16_C(29646),
        UINT16_C(29135), UINT16_C( 6344), UINT16_C(16367), UINT16_C(12328), UINT16_C( 5779), UINT16_C(36721), UINT16_C(51939), UINT16_C(44896),
        UINT16_C(34027), UINT16_C(41290), UINT16_C(30134), UINT16_C(30263), UINT16_C(26996), UINT16_C( 6054), UINT16_C(45325), UINT16_C(55252),
        UINT16_C(40043), UINT16_C(37223), UINT16_C(52773), UINT16_C(47486), UINT16_C(37844), UINT16_C(57416), UINT16_C(17118), UINT16_C(18790) } },
    { { UINT16_C(44898), UINT16_C( 8661), UINT16_C(41740), UINT16_C( 7815), UINT16_C(31298), UINT16_C(30306), UINT16_C(33374), UINT16_C(63946),
        UINT16_C(53570), UINT16_C(30296), UINT16_C( 2020), UINT16_C(65128), UINT16_C(13028), UINT16_C(34958), UINT16_C(49736), UINT16_C(43685),
        UINT16_C(31345), UINT16_C(32203), UINT16_C(21022), UINT16_C(24731), UINT16_C(64973), UINT16_C(11222), UINT16_C(41087), UINT16_C(49444),
        UINT16_C(31857), UINT16_C(21815), UINT16_C(41091), UINT16_C(26451), UINT16_C(58066), UINT16_C( 6895), UINT16_C(38052), UINT16_C( 5572) },
      UINT32_C( 764579599),
      { UINT16_C(12001), UINT16_C(44685), UINT16_C(25387), UINT16_C(43993), UINT16_C(65028), UINT16_C(30060), UINT16_C(42106), UINT16_C(65227),
        UINT16_C( 7748), UINT16_C( 5733), UINT16_C(21760), UINT16_C(42032), UINT16_C(62697), UINT16_C(63673), UINT16_C(19587), UINT16_C(25637),
        UINT16_C(45690), UINT16_C(42259), UINT16_C(60438), UINT16_C( 6736), UINT16_C(48618), UINT16_C(25999), UINT16_C(23137), UINT16_C(42339),
        UINT16_C(51321), UINT16_C(31163), UINT16_C(60189), UINT16_C( 1822), UINT16_C(55263), UINT16_C(25343), UINT16_C( 9507), UINT16_C(40390) },
      { UINT16_C(55767), UINT16_C(60739), UINT16_C(37830), UINT16_C(45063), UINT16_C(38736), UINT16_C(45333), UINT16_C(30961), UINT16_C(27222),
        UINT16_C( 4417), UINT16_C(24292), UINT16_C(  764), UINT16_C(56165), UINT16_C(26073), UINT16_C(64829), UINT16_C( 1162), UINT16_C(24986),
        UINT16_C(56797), UINT16_C(41807), UINT16_C(22129), UINT16_C(49492), UINT16_C(27117), UINT16_C(57203), UINT16_C(51682), UINT16_C( 9033),
        UINT16_C(11739), UINT16_C(55169), UINT16_C(59183), UINT16_C( 2483), UINT16_C(61516), UINT16_C(54790), UINT16_C(41204), UINT16_C(53815) },
      { UINT16_C(33884), UINT16_C(52712), UINT16_C(31609), UINT16_C(44528), UINT16_C(31298), UINT16_C(30306), UINT16_C(33374), UINT16_C(63946),
        UINT16_C( 6083), UINT16_C(15013), UINT16_C(11262), UINT16_C(49099), UINT16_C(13028), UINT16_C(34958), UINT16_C(49736), UINT16_C(25312),
        UINT16_C(31345), UINT16_C(42033), UINT16_C(21022), UINT16_C(24731), UINT16_C(37868), UINT16_C(11222), UINT16_C(41087), UINT16_C(25686),
        UINT16_C(31530), UINT16_C(21815), UINT16_C(59686), UINT16_C( 2153), UINT16_C(58066), UINT16_C(40067), UINT16_C(38052), UINT16_C( 5572) } },
    { { UINT16_C(34430), UINT16_C(61301), UINT16_C(51677), UINT16_C(51888), UINT16_C( 9011), UINT16_C( 5545), UINT16_C(62445), UINT16_C(51256),
        UINT16_C(47392), UINT16_C(20639), UINT16_C(21152), UINT16_C(60505), UINT16_C(24387), UINT16_C(14274), UINT16_C(64255), UINT16_C(32009),
        UINT16_C(32640), UINT16_C(23916), UINT16_C( 7496), UINT16_C(31528), UINT16_C(53568), UINT16_C(11664), UINT16_C(51396), UINT16_C(58869),
        UINT16_C(38274), UINT16_C( 8757), UINT16_C(36583), UINT16_C(10767), UINT16_C(53741), UINT16_C(60514), UINT16_C(27595), UINT16_C(19562) },
      UINT32_C( 866768618),
      { UINT16_C(53747), UINT16_C(13486), UINT16_C(16291), UINT16_C(26465), UINT16_C(22279), UINT16_C(35148), UINT16_C(33260), UINT16_C(54188),
        UINT16_C(47887), UINT16_C(64766), UINT16_C(24716), UINT16_C(22761), UINT16_C(21451), UINT16_C(46756), UINT16_C(19753), UINT16_C( 7657),
        UINT16_C(38687), UINT16_C(49745), UINT16_C(45782), UINT16_C(56873), UINT16_C(30217), UINT16_C(62823), UINT16_C( 5111), UINT16_C( 1993),
        UINT16_C(51150), UINT16_C(23299), UINT16_C(60455), UINT16_C(62131), UINT16_C(22335), UINT16_C(27048), UINT16_C(37284), UINT16_C(50054) },
      { UINT16_C(55081), UINT16_C(65413), UINT16_C(44937), UINT16_C(37853), UINT16_C(17701), UINT16_C( 7304), UINT16_C(20824), UINT16_C(10019),
        UINT16_C(10008), UINT16_C(16258), UINT16_C(13587), UINT16_C(21298), UINT16_C(55948), UINT16_C(12476), UINT16_C(17004), UINT16_C(38388),
        UINT16_C(31001), UINT16_C(41620), UINT16_C(29224), UINT16_C(19765), UINT16_C(48823), UINT16_C( 3946), UINT16_C(36111), UINT16_C(10294),
        UINT16_C(47284), UINT16_C(51303), UINT16_C(39405), UINT16_C(31003), UINT16_C(55156), UINT16_C(57514), UINT16_C(40473), UINT16_C(12917) },
      { UINT16_C(34430), UINT16_C(39450), UINT16_C(51677), UINT16_C(32159), UINT16_C( 9011), UINT16_C(21226), UINT16_C(27042), UINT16_C(32104),
        UINT16_C(47392), UINT16_C(40512), UINT16_C(19152), UINT16_C(60505), UINT16_C(38700), UINT16_C(14274), UINT16_C(18379), UINT16_C(23023),
        UINT16_C(34844), UINT16_C(23916), UINT16_C( 7496), UINT16_C(38319), UINT16_C(53568), UINT16_C(33385), UINT16_C(51396), UINT16_C( 6144),
        UINT16_C(49217), UINT16_C(37301), UINT16_C(36583), UINT16_C(10767), UINT16_C(38746), UINT16_C(42281), UINT16_C(27595), UINT16_C(19562) } },
    { { UINT16_C( 2327), UINT16_C(16596), UINT16_C( 2683), UINT16_C(12941), UINT16_C(63432), UINT16_C(55106), UINT16_C(30853), UINT16_C(14847),
        UINT16_C(26417), UINT16_C( 7681), UINT16_C( 7168), UINT16_C(29848), UINT16_C(17139), UINT16_C( 3156), UINT16_C(51680), UINT16_C(63294),
        UINT16_C( 5075), UINT16_C(20023), UINT16_C(50461), UINT16_C(58753), UINT16_C(50108), UINT16_C(16828), UINT16_C(48187), UINT16_C(27771),
        UINT16_C(31779), UINT16_C( 9099), UINT16_C( 9113), UINT16_C(35992), UINT16_C(60517), UINT16_C(17817), UINT16_C(55222), UINT16_C(35132) },
      UINT32_C( 131560682),
      { UINT16_C(22585), UINT16_C(62956), UINT16_C(43291), UINT16_C(22327), UINT16_C(45669), UINT16_C(35011), UINT16_C(20014), UINT16_C(51115),
        UINT16_C(17265), UINT16_C(54868), UINT16_C(60720), UINT16_C(58907), UINT16_C(22724), UINT16_C(44911), UINT16_C(18124), UINT16_C( 1462),
        UINT16_C(41887), UINT16_C(47866), UINT16_C(12620), UINT16_C(45329), UINT16_C(54755), UINT16_C( 4665), UINT16_C(58403), UINT16_C(38361),
        UINT16_C(11560), UINT16_C(22635), UINT16_C(34586), UINT16_C(57150), UINT16_C(44511), UINT16_C(43918), UINT16_C(17651), UINT16_C(37552) },
      { UINT16_C(43751), UINT16_C(13133), UINT16_C(24284), UINT16_C(49124), UINT16_C( 7475), UINT16_C(22481), UINT16_C(43778), UINT16_C(10988),
        UINT16_C(22488), UINT16_C(62338), UINT16_C(49374), UINT16_C(48594), UINT16_C(24685), UINT16_C(24680), UINT16_C( 6308), UINT16_C(36083),
        UINT16_C(16579), UINT16_C(40895), UINT16_C(42142), UINT16_C(53854), UINT16_C(12481), UINT16_C(49961), UINT16_C( 5595), UINT16_C(46061),
        UINT16_C(28524), UINT16_C(19366), UINT16_C(30767), UINT16_C(39944), UINT16_C(29144), UINT16_C(32253), UINT16_C(61577), UINT16_C(19465) },
      { UINT16_C( 2327), UINT16_C(38045), UINT16_C( 2683), UINT16_C(35726), UINT16_C(63432), UINT16_C(28746), UINT16_C(31896), UINT16_C(31052),
        UINT16_C(26417), UINT16_C( 7681), UINT16_C(55047), UINT16_C(29848), UINT16_C(23705), UINT16_C(34796), UINT16_C(12216), UINT16_C(63294),
        UINT16_C(29233), UINT16_C(44381), UINT16_C(27381), UINT16_C(58753), UINT16_C(33618), UINT16_C(16828), UINT16_C(31999), UINT16_C(42211),
        UINT16_C(20042), UINT16_C(21001), UINT16_C(32677), UINT16_C(35992), UINT16_C(60517), UINT16_C(17817), UINT16_C(55222), UINT16_C(35132) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi16(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi16(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi16(test_vec[i].b);
    simde__m512i r = simde_mm512_mask_avg_epu16(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_u16x32(r, simde_mm512_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_maskz_avg_epu16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask32 k;
    const uint16_t a[32];
    const uint16_t b[32];
    const uint16_t r[32];
  } test_vec[] = {
    { UINT32_C(2786950776),
      { UINT16_C(13284), UINT16_C(64018), UINT16_C(52116), UINT16_C(52299), UINT16_C(41321), UINT16_C(22662), UINT16_C(38227), UINT16_C(57256),
        UINT16_C( 6590), UINT16_C(53296), UINT16_C( 8772), UINT16_C(11782), UINT16_C(51609), UINT16_C( 4485), UINT16_C(41543), UINT16_C(11191),
        UINT16_C(51669), UINT16_C(26917), UINT16_C(28820), UINT16_C(64822), UINT16_C(48145), UINT16_C(25686), UINT16_C(65105), UINT16_C( 3907),
        UINT16_C(29463), UINT16_C(23519), UINT16_C(58773), UINT16_C(12169), UINT16_C( 3758), UINT16_C(62784), UINT16_C(63665), UINT16_C(34336) },
      { UINT16_C(17857), UINT16_C(22256), UINT16_C( 9909), UINT16_C(50771), UINT16_C(43490), UINT16_C(13099), UINT16_C(28327), UINT16_C(48962),
        UINT16_C( 8674), UINT16_C(30490), UINT16_C(41991), UINT16_C(46502), UINT16_C(59314), UINT16_C(25515), UINT16_C(52191), UINT16_C(41194),
        UINT16_C(55825), UINT16_C(50934), UINT16_C(18944), UINT16_C(57997), UINT16_C(47347), UINT16_C(39701), UINT16_C(22310), UINT16_C( 2138),
        UINT16_C(29816), UINT16_C(32640), UINT16_C( 9752), UINT16_C(52021), UINT16_C(57357), UINT16_C(60462), UINT16_C( 6315), UINT16_C(48269) },
      { UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(51535), UINT16_C(42406), UINT16_C(17881), UINT16_C(33277), UINT16_C(    0),
        UINT16_C(    0), UINT16_C(41893), UINT16_C(25382), UINT16_C(29142), UINT16_C(55462), UINT16_C(15000), UINT16_C(46867), UINT16_C(    0),
        UINT16_C(53747), UINT16_C(    0), UINT16_C(23882), UINT16_C(61410), UINT16_C(47746), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0),
        UINT16_C(    0), UINT16_C(28080), UINT16_C(34263), UINT16_C(    0), UINT16_C(    0), UINT16_C(61623), UINT16_C(    0), UINT16_C(41303) } },
    { UINT32_C(4068705266),
      { UINT16_C( 4301), UINT16_C(49620), UINT16_C(59848), UINT16_C(61020), UINT16_C(46656), UINT16_C(47607), UINT16_C(30506), UINT16_C(17208),
        UINT16_C(28061), UINT16_C(43790), UINT16_C(15437), UINT16_C(63895), UINT16_C( 9301), UINT16_C(18357), UINT16_C(14504), UINT16_C(30010),
        UINT16_C( 3656), UINT16_C( 4150), UINT16_C(37624), UINT16_C(14591), UINT16_C(63048), UINT16_C(29681), UINT16_C(10861), UINT16_C( 2742),
        UINT16_C(50327), UINT16_C(58805), UINT16_C(19712), UINT16_C(21982), UINT16_C(37745), UINT16_C( 6557), UINT16_C(55244), UINT16_C( 5263) },
      { UINT16_C(50661), UINT16_C(56613), UINT16_C( 9304), UINT16_C(40982), UINT16_C( 1818), UINT16_C(34579), UINT16_C(51505), UINT16_C(51601),
        UINT16_C(18317), UINT16_C(36526), UINT16_C(35988), UINT16_C( 1507), UINT16_C(32799), UINT16_C(60191), UINT16_C(44631), UINT16_C(15616),
        UINT16_C( 9587), UINT16_C(51994), UINT16_C(12361), UINT16_C(25452), UINT16_C(32568), UINT16_C(27114), UINT16_C(31561), UINT16_C(54834),
        UINT16_C(57538), UINT16_C(22116), UINT16_C(18540), UINT16_C(35932), UINT16_C(31688), UINT16_C( 8311), UINT16_C(30505), UINT16_C(40029) },
      { UINT16_C(    0), UINT16_C(53117), UINT16_C(    0), UINT16_C(    0), UINT16_C(24237), UINT16_C(41093), UINT16_C(41006), UINT16_C(34405),
        UINT16_C(23189), UINT16_C(40158), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(22813),
        UINT16_C( 6622), UINT16_C(28072), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(28788),
        UINT16_C(    0), UINT16_C(40461), UINT16_C(    0), UINT16_C(    0), UINT16_C(34717), UINT16_C( 7434), UINT16_C(42875), UINT16_C(22646) } },
    { UINT32_C(3848828828),
      { UINT16_C(54440), UINT16_C(57416), UINT16_C(12883), UINT16_C(40009), UINT16_C(31918), UINT16_C(28787), UINT16_C(55132), UINT16_C(51655),
        UINT16_C( 8991), UINT16_C(59477), UINT16_C(52382), UINT16_C(50952), UINT16_C(25924), UINT16_C(57443), UINT16_C(52188), UINT16_C(33990),
        UINT16_C( 3743), UINT16_C(62308), UINT16_C(44609), UINT16_C(61327), UINT16_C(  554), UINT16_C(34399), UINT16_C( 9946), UINT16_C(63823),
        UINT16_C(42057), UINT16_C(59361), UINT16_C(59761), UINT16_C(46510), UINT16_C( 4686), UINT16_C(11157), UINT16_C(23517), UINT16_C(32175) },
      { UINT16_C( 5226), UINT16_C(43888), UINT16_C(65474), UINT16_C(60570), UINT16_C(63746), UINT16_C(56434), UINT16_C(49696), UINT16_C(27093),
        UINT16_C(46950), UINT16_C(55121), UINT16_C(65440), UINT16_C(61324), UINT16_C( 8721), UINT16_C(61210), UINT16_C(51581), UINT16_C(59244),
        UINT16_C(56541), UINT16_C(40850), UINT16_C(11483), UINT16_C(56715), UINT16_C(65062), UINT16_C(18105), UINT16_C(36800), UINT16_C( 9903),
        UINT16_C(   70), UINT16_C(59134), UINT16_C(35328), UINT16_C( 4565), UINT16_C(61356), UINT16_C(10752), UINT16_C(27833), UINT16_C(38417) },
      { UINT16_C(    0), UINT16_C(    0), UINT16_C(39179), UINT16_C(50290), UINT16_C(47832), UINT16_C(    0), UINT16_C(    0), UINT16_C(39374),
        UINT16_C(27971), UINT16_C(57299), UINT16_C(58911), UINT16_C(    0), UINT16_C(17323), UINT16_C(59327), UINT16_C(51885), UINT16_C(    0),
        UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(59021), UINT16_C(    0), UINT16_C(26252), UINT16_C(23373), UINT16_C(    0),
        UINT16_C(21064), UINT16_C(    0), UINT16_C(47545), UINT16_C(    0), UINT16_C(    0), UINT16_C(10955), UINT16_C(25675), UINT16_C(35296) } },
    { UINT32_C( 607560776),
      { UINT16_C(49616), UINT16_C(62977), UINT16_C(48063), UINT16_C(32572), UINT16_C(60490), UINT16_C(37030), UINT16_C(42220), UINT16_C(60534),
        UINT16_C(19502), UINT16_C(56318), UINT16_C(65083), UINT16_C(62469), UINT16_C( 5739), UINT16_C(45963), UINT16_C(49594), UINT16_C(35799),
        UINT16_C(55682), UINT16_C(17025), UINT16_C(48788), UINT16_C(57025), UINT16_C(26538), UINT16_C(38510), UINT16_C(58379), UINT16_C(14979),
        UINT16_C(33072), UINT16_C(27669), UINT16_C( 6783), UINT16_C(60000), UINT16_C(60208), UINT16_C(60318), UINT16_C(30124), UINT16_C(12150) },
      { UINT16_C(63310), UINT16_C(57969), UINT16_C(12981), UINT16_C(24512), UINT16_C(11930), UINT16_C(42486), UINT16_C(30995), UINT16_C(17375),
        UINT16_C(62714), UINT16_C(31151), UINT16_C( 4110), UINT16_C(16228), UINT16_C(  763), UINT16_C(43050), UINT16_C(41079), UINT16_C(50903),
        UINT16_C(18583), UINT16_C(19880), UINT16_C(27002), UINT16_C( 5292), UINT16_C(41623), UINT16_C(43706), UINT16_C(39195), UINT16_C( 5614),
        UINT16_C(40334), UINT16_C(40079), UINT16_C(62381), UINT16_C(43483), UINT16_C( 1525), UINT16_C(27729), UINT16_C(10405), UINT16_C(15666) },
      { UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(28542), UINT16_C(    0), UINT16_C(    0), UINT16_C(36608), UINT16_C(    0),
        UINT16_C(    0), UINT16_C(    0), UINT16_C(34597), UINT16_C(    0), UINT16_C(    0), UINT16_C(44507), UINT16_C(    0), UINT16_C(43351),
        UINT16_C(    0), UINT16_C(18453), UINT16_C(37895), UINT16_C(    0), UINT16_C(34081), UINT16_C(41108), UINT16_C(    0), UINT16_C(    0),
        UINT16_C(    0), UINT16_C(    0), UINT16_C(34582), UINT16_C(    0), UINT16_C(    0), UINT16_C(44024), UINT16_C(    0), UINT16_C(    0) } },
    { UINT32_C(3934968688),
      { UINT16_C(13892), UINT16_C(56319), UINT16_C(47577), UINT16_C(62598), UINT16_C(29778), UINT16_C(57354), UINT16_C(39185), UINT16_C(49021),
        UINT16_C(22668), UINT16_C(33128), UINT16_C(47454), UINT16_C( 1005), UINT16_C( 8417), UINT16_C(20800), UINT16_C(51963), UINT16_C(16187),
        UINT16_C(14849), UINT16_C(55834), UINT16_C(41203), UINT16_C(18126), UINT16_C(55316), UINT16_C( 9766), UINT16_C(41841), UINT16_C(64997),
        UINT16_C(19964), UINT16_C(23166), UINT16_C(27654), UINT16_C(59229), UINT16_C(40588), UINT16_C(34616), UINT16_C(29544), UINT16_C(27078) },
      { UINT16_C(57518), UINT16_C(41283), UINT16_C( 4737), UINT16_C(38375), UINT16_C( 3818), UINT16_C(23739), UINT16_C(41137), UINT16_C(44377),
        UINT16_C(55533), UINT16_C(62215), UINT16_C(25924), UINT16_C(53466), UINT16_C( 4611), UINT16_C(27479), UINT16_C( 7558), UINT16_C(13525),
        UINT16_C( 6397), UINT16_C(32469), UINT16_C(48426), UINT16_C( 5396), UINT16_C(53195), UINT16_C(31857), UINT16_C(51824), UINT16_C(23850),
        UINT16_C(12706), UINT16_C(58961), UINT16_C(11158), UINT16_C(39350), UINT16_C( 3390), UINT16_C(50181), UINT16_C(55850), UINT16_C(10488) },
      { UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(16798), UINT16_C(40547), UINT16_C(40161), UINT16_C(    0),
        UINT16_C(39101), UINT16_C(47672), UINT16_C(    0), UINT16_C(27236), UINT16_C( 6514), UINT16_C(    0), UINT16_C(29761), UINT16_C(14856),
        UINT16_C(    0), UINT16_C(44152), UINT16_C(    0), UINT16_C(11761), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(44424),
        UINT16_C(    0), UINT16_C(41064), UINT16_C(    0), UINT16_C(49290), UINT16_C(    0), UINT16_C(42399), UINT16_C(42697), UINT16_C(18783) } },
    { UINT32_C( 497470962),
      { UINT16_C(47754), UINT16_C(21810), UINT16_C(41866), UINT16_C(64210), UINT16_C(64621), UINT16_C( 4183), UINT16_C(43053), UINT16_C(50422),
        UINT16_C(44500), UINT16_C( 4701), UINT16_C(25274), UINT16_C(58838), UINT16_C(52796), UINT16_C(12045), UINT16_C(45979), UINT16_C( 9804),
        UINT16_C(32366), UINT16_C(63611), UINT16_C(19745), UINT16_C(36594), UINT16_C(18761), UINT16_C(30622), UINT16_C(38386), UINT16_C(50747),
        UINT16_C(38978), UINT16_C(64728), UINT16_C(44795), UINT16_C(14305), UINT16_C(61052), UINT16_C( 5990), UINT16_C(45730), UINT16_C( 4157) },
      { UINT16_C(47408), UINT16_C(20744), UINT16_C(64006), UINT16_C(20704), UINT16_C(32323), UINT16_C(13767), UINT16_C(  531), UINT16_C(22011),
        UINT16_C(54170), UINT16_C(38226), UINT16_C(13185), UINT16_C(64973), UINT16_C(13090), UINT16_C(50197), UINT16_C(21222), UINT16_C( 5844),
        UINT16_C(56331), UINT16_C( 4712), UINT16_C(18646), UINT16_C( 6498), UINT16_C(10694), UINT16_C(55887), UINT16_C(18987), UINT16_C(50479),
        UINT16_C(33054), UINT16_C(40795), UINT16_C(10421), UINT16_C(55197), UINT16_C(45659), UINT16_C(16795), UINT16_C(28420), UINT16_C( 4184) },
      { UINT16_C(    0), UINT16_C(21277), UINT16_C(    0), UINT16_C(    0), UINT16_C(48472), UINT16_C( 8975), UINT16_C(21792), UINT16_C(36217),
        UINT16_C(49335), UINT16_C(    0), UINT16_C(19230), UINT16_C(61906), UINT16_C(    0), UINT16_C(    0), UINT16_C(33601), UINT16_C( 7824),
        UINT16_C(    0), UINT16_C(34162), UINT16_C(19196), UINT16_C(    0), UINT16_C(    0), UINT16_C(43255), UINT16_C(    0), UINT16_C(50613),
        UINT16_C(36016), UINT16_C(    0), UINT16_C(27608), UINT16_C(34751), UINT16_C(53356), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0) } },
    { UINT32_C( 555925579),
      { UINT16_C(33800), UINT16_C(52794), UINT16_C(35245), UINT16_C(55464), UINT16_C(55508), UINT16_C(62109), UINT16_C(63577), UINT16_C( 3729),
        UINT16_C(11808), UINT16_C(31973), UINT16_C(32992), UINT16_C(58813), UINT16_C( 5615), UINT16_C(15093), UINT16_C( 6101), UINT16_C(56667),
        UINT16_C(38555), UINT16_C(18604), UINT16_C(21535), UINT16_C(62240), UINT16_C(48428), UINT16_C(34533), UINT16_C(30646), UINT16_C(54932),
        UINT16_C(31397), UINT16_C(34386), UINT16_C( 4346), UINT16_C(60011), UINT16_C(24613), UINT16_C(64292), UINT16_C(32887), UINT16_C( 4824) },
      { UINT16_C(33814), UINT16_C(13658), UINT16_C(31449), UINT16_C( 1321), UINT16_C( 3639), UINT16_C(60811), UINT16_C( 8325), UINT16_C(11204),
        UINT16_C( 5786), UINT16_C(38065), UINT16_C( 7206), UINT16_C(19582), UINT16_C(41852), UINT16_C(62279), UINT16_C( 7971), UINT16_C(14597),
        UINT16_C(24484), UINT16_C(32110), UINT16_C(38873), UINT16_C( 4226), UINT16_C( 3750), UINT16_C(11262), UINT16_C(49710), UINT16_C(51286),
        UINT16_C( 2008), UINT16_C(65372), UINT16_C(56099), UINT16_C(40779), UINT16_C(37502), UINT16_C(41362), UINT16_C(38833), UINT16_C(21978) },
      { UINT16_C(33807), UINT16_C(33226), UINT16_C(    0), UINT16_C(28393), UINT16_C(    0), UINT16_C(    0), UINT16_C(35951), UINT16_C(    0),
        UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C( 7036), UINT16_C(35632),
        UINT16_C(    0), UINT16_C(25357), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(22898), UINT16_C(    0), UINT16_C(    0),
        UINT16_C(16703), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(52827), UINT16_C(    0), UINT16_C(    0) } },
    { UINT32_C(3486664950),
      { UINT16_C(21984), UINT16_C(34528), UINT16_C(56931), UINT16_C(37297), UINT16_C( 2208), UINT16_C(30809), UINT16_C(46351), UINT16_C(13175),
        UINT16_C(49808), UINT16_C( 3794), UINT16_C(25940), UINT16_C( 1711), UINT16_C(35324), UINT16_C(62299), UINT16_C(11986), UINT16_C(45762),
        UINT16_C(41603), UINT16_C(58936), UINT16_C(59776), UINT16_C( 8311), UINT16_C(53489), UINT16_C(  409), UINT16_C( 4229), UINT16_C( 5684),
        UINT16_C( 1747), UINT16_C(10020), UINT16_C(54379), UINT16_C(26669), UINT16_C(35165), UINT16_C(12123), UINT16_C( 7607), UINT16_C(15073) },
      { UINT16_C( 6592), UINT16_C(16416), UINT16_C(38659), UINT16_C(62561), UINT16_C(64103), UINT16_C(60661), UINT16_C(10506), UINT16_C(56578),
        UINT16_C(10032), UINT16_C(39685), UINT16_C(13051), UINT16_C(22531), UINT16_C(24251), UINT16_C(29320), UINT16_C(27004), UINT16_C(15532),
        UINT16_C(52355), UINT16_C(34428), UINT16_C(56675), UINT16_C(51834), UINT16_C(28887), UINT16_C(58039), UINT16_C(47513), UINT16_C(51647),
        UINT16_C(50400), UINT16_C(56165), UINT16_C(26871), UINT16_C(45620), UINT16_C(48327), UINT16_C(17189), UINT16_C(53541), UINT16_C(43135) },
      { UINT16_C(    0), UINT16_C(25472), UINT16_C(47795), UINT16_C(    0), UINT16_C(33156), UINT16_C(45735), UINT16_C(28429), UINT16_C(34877),
        UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(12121), UINT16_C(    0), UINT16_C(    0), UINT16_C(19495), UINT16_C(    0),
        UINT16_C(    0), UINT16_C(46682), UINT16_C(    0), UINT16_C(    0), UINT16_C(41188), UINT16_C(    0), UINT16_C(25871), UINT16_C(28666),
        UINT16_C(26074), UINT16_C(33093), UINT16_C(40625), UINT16_C(36145), UINT16_C(    0), UINT16_C(    0), UINT16_C(30574), UINT16_C(29104) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi16(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi16(test_vec[i].b);
    simde__m512i r = simde_mm512_maskz_avg_epu16(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_u16x32(r, simde_mm512_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_mask_avg_epu8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_maskz_avg_epu8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_mask_avg_epu16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_maskz_avg_epu16)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_mask_avg_epu8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_maskz_avg_epu8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_mask_avg_epu16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_maskz_avg_epu16)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_avg_epu8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_avg_epu8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_avg_epu8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_avg_epu16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_avg_epu16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_avg_epu16)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
