# Copyright (c) 2023 Ansible Project
# GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

import base64
import uuid

import pytest

dpapi_ng = pytest.importorskip("dpapi_ng")

from ansible_collections.microsoft.ad.plugins.plugin_utils._ldap.laps import LAPSDecryptor


def test_decrypt_laps_blob() -> None:
    root_key = (
        b"\xDC\x24\xFF\x6D\xB1\x31\x70\x18"
        b"\x8E\xC9\xFF\xB5\x11\xFA\x41\xA6"
        b"\xEB\x51\xD0\x49\xFE\x8C\xFE\x27"
        b"\xD6\x83\xD5\x1E\xD5\xA1\x0F\xAF"
        b"\x64\x3F\x67\x2E\xE6\xED\x8F\x9F"
        b"\x5E\x11\x18\x72\x7B\x3A\xA9\x38"
        b"\x4F\xF9\x43\xFF\x5B\x04\xF3\x10"
        b"\x53\x0B\x08\x3C\x34\x37\x99\x6E"
    )
    cache = dpapi_ng.KeyCache()
    cache.load_key(root_key, uuid.UUID("bac64fa8-e890-917c-1090-83e7b0f85996"))
    decryptor = LAPSDecryptor()
    decryptor._cache = cache

    laps_blob = (
        b"\xB6\x82\xD9\x01\x64\x7B\xAE\x07"
        b"\xE4\x04\x00\x00\x00\x00\x00\x00"
        b"\x30\x82\x04\x4E\x06\x09\x2A\x86"
        b"\x48\x86\xF7\x0D\x01\x07\x03\xA0"
        b"\x82\x04\x3F\x30\x82\x04\x3B\x02"
        b"\x01\x02\x31\x82\x04\x07\xA2\x82"
        b"\x04\x03\x02\x01\x04\x30\x82\x03"
        b"\xC5\x04\x82\x03\x6C\x01\x00\x00"
        b"\x00\x4B\x44\x53\x4B\x03\x00\x00"
        b"\x00\x69\x01\x00\x00\x11\x00\x00"
        b"\x00\x12\x00\x00\x00\xA8\x4F\xC6"
        b"\xBA\x90\xE8\x7C\x91\x10\x90\x83"
        b"\xE7\xB0\xF8\x59\x96\x08\x03\x00"
        b"\x00\x18\x00\x00\x00\x18\x00\x00"
        b"\x00\x44\x48\x50\x42\x00\x01\x00"
        b"\x00\x87\xA8\xE6\x1D\xB4\xB6\x66"
        b"\x3C\xFF\xBB\xD1\x9C\x65\x19\x59"
        b"\x99\x8C\xEE\xF6\x08\x66\x0D\xD0"
        b"\xF2\x5D\x2C\xEE\xD4\x43\x5E\x3B"
        b"\x00\xE0\x0D\xF8\xF1\xD6\x19\x57"
        b"\xD4\xFA\xF7\xDF\x45\x61\xB2\xAA"
        b"\x30\x16\xC3\xD9\x11\x34\x09\x6F"
        b"\xAA\x3B\xF4\x29\x6D\x83\x0E\x9A"
        b"\x7C\x20\x9E\x0C\x64\x97\x51\x7A"
        b"\xBD\x5A\x8A\x9D\x30\x6B\xCF\x67"
        b"\xED\x91\xF9\xE6\x72\x5B\x47\x58"
        b"\xC0\x22\xE0\xB1\xEF\x42\x75\xBF"
        b"\x7B\x6C\x5B\xFC\x11\xD4\x5F\x90"
        b"\x88\xB9\x41\xF5\x4E\xB1\xE5\x9B"
        b"\xB8\xBC\x39\xA0\xBF\x12\x30\x7F"
        b"\x5C\x4F\xDB\x70\xC5\x81\xB2\x3F"
        b"\x76\xB6\x3A\xCA\xE1\xCA\xA6\xB7"
        b"\x90\x2D\x52\x52\x67\x35\x48\x8A"
        b"\x0E\xF1\x3C\x6D\x9A\x51\xBF\xA4"
        b"\xAB\x3A\xD8\x34\x77\x96\x52\x4D"
        b"\x8E\xF6\xA1\x67\xB5\xA4\x18\x25"
        b"\xD9\x67\xE1\x44\xE5\x14\x05\x64"
        b"\x25\x1C\xCA\xCB\x83\xE6\xB4\x86"
        b"\xF6\xB3\xCA\x3F\x79\x71\x50\x60"
        b"\x26\xC0\xB8\x57\xF6\x89\x96\x28"
        b"\x56\xDE\xD4\x01\x0A\xBD\x0B\xE6"
        b"\x21\xC3\xA3\x96\x0A\x54\xE7\x10"
        b"\xC3\x75\xF2\x63\x75\xD7\x01\x41"
        b"\x03\xA4\xB5\x43\x30\xC1\x98\xAF"
        b"\x12\x61\x16\xD2\x27\x6E\x11\x71"
        b"\x5F\x69\x38\x77\xFA\xD7\xEF\x09"
        b"\xCA\xDB\x09\x4A\xE9\x1E\x1A\x15"
        b"\x97\x3F\xB3\x2C\x9B\x73\x13\x4D"
        b"\x0B\x2E\x77\x50\x66\x60\xED\xBD"
        b"\x48\x4C\xA7\xB1\x8F\x21\xEF\x20"
        b"\x54\x07\xF4\x79\x3A\x1A\x0B\xA1"
        b"\x25\x10\xDB\xC1\x50\x77\xBE\x46"
        b"\x3F\xFF\x4F\xED\x4A\xAC\x0B\xB5"
        b"\x55\xBE\x3A\x6C\x1B\x0C\x6B\x47"
        b"\xB1\xBC\x37\x73\xBF\x7E\x8C\x6F"
        b"\x62\x90\x12\x28\xF8\xC2\x8C\xBB"
        b"\x18\xA5\x5A\xE3\x13\x41\x00\x0A"
        b"\x65\x01\x96\xF9\x31\xC7\x7A\x57"
        b"\xF2\xDD\xF4\x63\xE5\xE9\xEC\x14"
        b"\x4B\x77\x7D\xE6\x2A\xAA\xB8\xA8"
        b"\x62\x8A\xC3\x76\xD2\x82\xD6\xED"
        b"\x38\x64\xE6\x79\x82\x42\x8E\xBC"
        b"\x83\x1D\x14\x34\x8F\x6F\x2F\x91"
        b"\x93\xB5\x04\x5A\xF2\x76\x71\x64"
        b"\xE1\xDF\xC9\x67\xC1\xFB\x3F\x2E"
        b"\x55\xA4\xBD\x1B\xFF\xE8\x3B\x9C"
        b"\x80\xD0\x52\xB9\x85\xD1\x82\xEA"
        b"\x0A\xDB\x2A\x3B\x73\x13\xD3\xFE"
        b"\x14\xC8\x48\x4B\x1E\x05\x25\x88"
        b"\xB9\xB7\xD2\xBB\xD2\xDF\x01\x61"
        b"\x99\xEC\xD0\x6E\x15\x57\xCD\x09"
        b"\x15\xB3\x35\x3B\xBB\x64\xE0\xEC"
        b"\x37\x7F\xD0\x28\x37\x0D\xF9\x2B"
        b"\x52\xC7\x89\x14\x28\xCD\xC6\x7E"
        b"\xB6\x18\x4B\x52\x3D\x1D\xB2\x46"
        b"\xC3\x2F\x63\x07\x84\x90\xF0\x0E"
        b"\xF8\xD6\x47\xD1\x48\xD4\x79\x54"
        b"\x51\x5E\x23\x27\xCF\xEF\x98\xC5"
        b"\x82\x66\x4B\x4C\x0F\x6C\xC4\x16"
        b"\x59\x13\xE2\xBB\xB1\xC4\xA8\x36"
        b"\x61\x13\x90\xEB\xA3\xA4\x67\x40"
        b"\xF2\x9E\x7D\xAC\x49\xA4\xBE\x93"
        b"\xC6\xFE\xBC\xFC\x1F\x13\xAA\xAF"
        b"\xFE\xFE\x2A\x44\x17\x0E\x5B\x2E"
        b"\x03\x17\x9B\x42\x82\x1D\xCD\x06"
        b"\x2C\xAD\xEF\xA6\x6F\x3C\x60\xD5"
        b"\xC5\x3C\x2B\x96\xAC\x08\x02\xA0"
        b"\x7C\xCA\x7F\x51\x60\xD9\xF3\x3A"
        b"\xE2\xFA\x87\xA5\x90\x61\x91\x6B"
        b"\xFD\x89\x3D\x20\x72\x7D\xDE\xCA"
        b"\x47\xD4\x21\x2D\xD9\x0D\x0F\x65"
        b"\xB2\x42\x60\xBD\x9D\xC1\xF1\x19"
        b"\x7B\x5E\x7B\xCE\x08\x05\x00\xC1"
        b"\xEA\x95\xA4\xAB\x60\xBE\x3C\x13"
        b"\x0F\xB6\xB0\x76\xFC\xA0\x6F\x8E"
        b"\xE1\x39\x7E\x58\x84\x53\x6B\xF9"
        b"\x03\x14\xAF\x12\xCF\xB3\x1A\x1A"
        b"\xAC\x10\x51\x72\x83\x17\xF3\xCC"
        b"\x28\x47\xDC\x3F\xE2\x54\x3A\x7E"
        b"\xA1\xFF\x23\xB7\xC9\xD6\x0F\x6B"
        b"\x1E\xD9\x20\xB4\xC6\x0B\x9D\xC4"
        b"\xDF\x08\x7D\xD0\x95\x5D\x01\xDC"
        b"\xFC\x4E\xBC\x9F\xF5\x33\x0C\xF0"
        b"\xF1\x1B\x46\xF9\x4D\x16\x64\x36"
        b"\x73\x52\xCF\xAD\xEC\x72\x17\x0F"
        b"\x4A\xA2\xC8\x5A\xAF\x73\x6C\xAE"
        b"\xF4\x7A\x21\x65\x4B\xBF\xDD\xAB"
        b"\x34\x8D\xDD\x9C\x22\x8E\x2C\xDB"
        b"\xD1\xAD\x08\xE1\x87\x31\xC6\xA4"
        b"\x6E\xF4\xCF\x4C\xF1\xE3\x5E\x0B"
        b"\x10\x65\xFA\x51\x0D\x53\x01\x88"
        b"\xF3\x64\x00\x6F\x00\x6D\x00\x61"
        b"\x00\x69\x00\x6E\x00\x2E\x00\x74"
        b"\x00\x65\x00\x73\x00\x74\x00\x00"
        b"\x00\x64\x00\x6F\x00\x6D\x00\x61"
        b"\x00\x69\x00\x6E\x00\x2E\x00\x74"
        b"\x00\x65\x00\x73\x00\x74\x00\x00"
        b"\x00\x30\x53\x06\x09\x2B\x06\x01"
        b"\x04\x01\x82\x37\x4A\x01\x30\x46"
        b"\x06\x0A\x2B\x06\x01\x04\x01\x82"
        b"\x37\x4A\x01\x01\x30\x38\x30\x36"
        b"\x30\x34\x0C\x03\x53\x49\x44\x0C"
        b"\x2D\x53\x2D\x31\x2D\x35\x2D\x32"
        b"\x31\x2D\x34\x31\x35\x31\x38\x30"
        b"\x38\x37\x39\x37\x2D\x33\x34\x33"
        b"\x30\x35\x36\x31\x30\x39\x32\x2D"
        b"\x32\x38\x34\x33\x34\x36\x34\x35"
        b"\x38\x38\x2D\x35\x31\x32\x30\x0B"
        b"\x06\x09\x60\x86\x48\x01\x65\x03"
        b"\x04\x01\x2D\x04\x28\x8F\xC9\xA4"
        b"\x80\xB4\x86\x54\x29\x23\x70\xE5"
        b"\x13\x2C\xC3\x71\xCE\x0A\xA2\x1B"
        b"\x42\x0D\x6C\xBF\x59\x10\x29\x91"
        b"\xED\xEB\x3D\x1B\x79\x08\xDD\x15"
        b"\x84\x19\x35\xFF\xA0\x30\x2B\x06"
        b"\x09\x2A\x86\x48\x86\xF7\x0D\x01"
        b"\x07\x01\x30\x1E\x06\x09\x60\x86"
        b"\x48\x01\x65\x03\x04\x01\x2E\x30"
        b"\x11\x04\x0C\x03\xDD\xDF\x73\xE1"
        b"\x5C\x1C\xFB\xBD\x6C\x6F\x50\x02"
        b"\x01\x10\x75\x32\x34\x64\x9C\x6E"
        b"\xD7\xB0\xC7\xE0\xE4\x36\x90\xB1"
        b"\x40\x7F\x20\xB3\xB8\x45\xE2\xFD"
        b"\x62\xBE\x7C\x1B\x17\xAE\x0C\xC4"
        b"\x4C\xD7\xD2\x4A\x97\xD9\x4E\x05"
        b"\x4A\x06\x96\xC0\x73\xF2\x94\xF0"
        b"\xAF\x85\xDB\xFF\xD2\x42\x3E\xEE"
        b"\x0B\xCB\x24\xF1\xE6\x75\x7B\xA9"
        b"\xB9\x56\x2F\xBA\x90\x49\x2D\xE8"
        b"\x58\xE9\xCE\x7D\x30\xD3\x46\xA3"
        b"\xDA\x81\x7A\xFA\x18\x92\x50\x72"
        b"\xD2\x6B\x16\xB7\x56\xBD\x9C\xDE"
        b"\xF4\x9D\xD5\x2A\x8B\x2A\x4D\xB0"
        b"\x0D\xFB\xAA\x76\x9F\xAB\xE8\x7A"
        b"\x33\xDC\xE8\xD4\x3B\xF8\x21\x9B"
        b"\x35\x30\x66\x70\x8A\xB5\xDE\xB6"
        b"\xFC\x05\xB2\x20\x22\x2C\x09\xD7"
        b"\x74\x32\x16\x80\x83\x0A\xEE\x42"
        b"\x15\x5D\x56\x22"
    )
    actual = decryptor.decrypt(laps_blob)
    assert actual['update_timestamp'] == 133281382308674404
    assert actual['flags'] == 0
    assert actual['encrypted_value'] == base64.b64encode(laps_blob[16:]).decode()
    assert actual['value'] == '{"n":"Administrator","t":"1d982b607ae7b64","p":"6jr&}yK++{0Q}&"}'
    assert 'debug' not in actual


def test_decrypt_invalid_laps_blob() -> None:
    decryptor = LAPSDecryptor()

    laps_blob = (
        b"\xB6\x82\xD9\x01\x64\x7B\xAE\x07"
        b"\x01\x00\x00\x00\x01\x00\x00\x00"
        b"\x01"
    )

    actual = decryptor.decrypt(laps_blob)
    assert actual['update_timestamp'] == 133281382308674404
    assert actual['flags'] == 1
    assert actual['encrypted_value'] == "AQ=="
    assert 'value' not in actual
    assert 'Failed to decrypt value due to error - NotEnougData' in actual['debug']
