if (typeof Test === 'undefined') require('../tester');

Test('precision', function () {
    var MAX = 1e9;

    Test.areEqual(BigNumber.prototype.sd, BigNumber.prototype.precision);

    BigNumber.config({
        DECIMAL_PLACES: 20,
        ROUNDING_MODE: 7,
        RANGE: 1E9,
        EXPONENTIAL_AT: 1E9
    });

    // Return significant digit count

    var t = function (value, count, includeZeros) {
        Test.areEqual(count, new BigNumber(value).precision(includeZeros));
    }

    t(NaN, null);
    t(Infinity, null, true);
    t(-Infinity, null);

    t(0, 1);
    t(-0, 1);
    t(0, 1, true);
    t(-0, 1, true);
    t(1, 1, true);
    t(-1, 1, true);

    t(100, 1);
    t(100, 1, false);
    t(100, 3, true);

    t('0.001234568900', 8);
    t('0.001234568900', 8, false);
    t('0.001234568900', 8, true);

    t('987654321000000', 9, false);
    t('987654321000000', 15, true);
    t('987654321000000.0012345689000001', 31, false);
    t('987654321000000.0012345689000001', 31, true);

    t('1e+123', 1);
    t('1e+123', 124, true);

    t('1e-123', 1);
    t('1e-123', 1, true);

    t('9.9999e+900000000', 5, false);
    t('9.9999e+900000000', 900000001, true);
    t('-9.9999e+900000000', 5, false);
    t('-9.9999e+900000000', 900000001, true);

    t('1e-900000000', 1, false);
    t('1e-900000000', 1, true);
    t('-1e-900000000', 1, false);
    t('-1e-900000000', 1, true);

    t('0e+0', 1);
    t('0e-4', 1);
    t('   +4352435.435435e-4   ', 13);
    t('   -4352435.435435e+14   ', 21, true);

    // Return a new BigNumber with a maximum number of sd significant digits

    t = function (expected, value, sd, rm) {
        Test.areEqual(expected, new BigNumber(value).precision(sd, rm).valueOf());
    }

    t('0', 0, 1);
    t('0.5', 0.5, 1);
    t('1', 1, 1);
    t('-1', -1, 1);
    t('Infinity', Infinity, 1);
    t('-Infinity', -Infinity, 1);
    t('NaN', NaN, 1);

    t('0', '0', 1);
    t('-0', '-0', 1);
    t('-0', '-0', 1, 0);
    t('-0', '-0', 1, 3);
    t('-0', '-0', 1, 6);
    t('0', '0', 10);
    t('-0', '-0', 20);

    t('123456789.12345678912346789', '123456789.12345678912346789', 26);
    BigNumber.config({ ROUNDING_MODE: 0 });
    t('123456789.12345678912346789', '123456789.12345678912346789', 26);
    BigNumber.config({ ROUNDING_MODE: 1 });
    t('123456789.12345678912346789', '123456789.12345678912346789', 26);

    t('4937809340236234102130.947044664011', '4937809340236234102130.947044664011', 35, 0);
    t('337528093391.5', '337528093391.493107', 13, 6);
    t('7725982105101004219161075340794084739.771523585576612', '7725982105101004219161075340794084739.77152358557661226998117872661100', 52, 3);
    t('52215017456426378512883312970.4861221', '52215017456426378512883312970.4861221', 36, 2);
    t('4985349771216.54991391', '4985349771216.54991391072733', 21, 4);
    t('1101895947.92763', '1101895947.92762954984827601473036438720818296', 15, 4);
    t('20026847', '20026846.9372', 8, 6);
    t('8204561821674316833270', '8204561821674316833266.047893028353', 21, 2);
    t('7419720696218', '7419720696218.21961188299387526', 13, 1);
    t('635000000', '634438520.50126453', 3, 0);
    t('28392130715407969', '28392130715407968.96', 18, 5);
    t('36765276128959576253780217524271972128', '36765276128959576253780217524271972127.99', 38, 2);
    t('88429990502356839.49260500060049004694585659', '88429990502356839.49260500060049004694585658780656', 43, 4);
    t('2768479', '2768479.2081810557738981722324989595', 7, 1);
    t('85601900000000000000000000', '85601959550632180897338814.9', 6, 3);
    t('3911951424308151086107907934806060', '3911951424308151086107907934806062.45495666964', 33, 5);
    t('892162270787217755968010370478903185093', '892162270787217755968010370478903185093.0', 42, 6);
    t('8018330257774248398702693271920270', '8018330257774248398702693271920270.09311146148062547723636185680287073', 33, 1);
    t('88092477731945.1182807043', '88092477731945.118280704259340858', 24, 4);
    t('9252541030951.9012531009707842372724161089075754576', '9252541030951.90125310097078423727241610890757545760', 53, 3);
    t('90993285126.547129568419776', '90993285126.5471295684197756642787492493375', 26, 6);
    t('315137924718839945152625512241078.591350724696151650603', '315137924718839945152625512241078.59135072469615165060341097372', 54, 1);
    t('822866202506325637.7', '822866202506325637.740891965300', 19, 3);
    t('78407276142689114300000000', '78407276142689114391801017.43776', 18, 1);
    t('13199803085352251536787921', '13199803085352251536787920.6016623911545535514918245', 26, 2);
    t('708.66442', '708.66442', 11, 0);
    t('96426394035800000000000000000', '96426394035875521556520436906.967758064780068220269263413347', 12, 1);
    t('297.1386227802795046531729004026', '297.1386227802795046531729004026975', 31, 3);
    t('98291.739', '98291.739', 9, 2);
    t('93', '93.078130', 2, 5);
    t('0.001', '0.00099999999999', 3, 0);
    t('0.000999', '0.00099999999999', 3, 1);
    t('56300000000', '56281970459.86927925182', 3, 6);
    t('565000000000', '565914070913.993334452356', 3, 1);
    t('3979389667144941000000000', '3979389667144941900427982.77071', 16, 3);
    t('52352220229587840556187937111046743.4583563', '52352220229587840556187937111046743.458356312735390953445935426160537204', 42, 4);
    t('36271.3981207141751333261', '36271.39812071417513332613', 24, 1);
    t('4739543704210300000000000000', '4739543704210297904547136327.7628335972128075226361512', 15, 6);
    t('37661669615568080247052723032589226.8', '37661669615568080247052723032589226.77024153', 36, 5);
    t('9645371928851642074344527881477634000', '9645371928851642074344527881477634345.4914341', 34, 4);
    t('698149777646326000000000000', '698149777646325814282220973.088797026232', 15, 2);
    t('18498919087448435591833490000000', '18498919087448435591833489904295.729717528', 26, 5);
    t('325969900850571568339999253781743124500', '325969900850571568339999253781743124504.0', 38, 5);
    t('40693000000', '40693051631.1251783538001747467807789871', 5, 4);
    t('810710395413475993', '810710395413475993.394678', 18, 3);
    t('86568781977428682.3822270220321', '86568781977428682.38222702203211', 30, 5);
    t('427897046936740724.911837668090826906087', '427897046936740724.911837668090826906087', 41, 3);
    t('8248561437515605126456247769501697.764', '8248561437515605126456247769501697.764296656999', 37, 4);
    t('5727898467119496952445688320883475.404402704196931184674675825', '5727898467119496952445688320883475.404402704196931184674675825', 64, 4);
    t('8752958499919994791000000000000000', '8752958499919994791191031106105979.47729', 19, 5);
    t('362154165751.582202112646276', '362154165751.58220211264627597398064', 28, 5);
    t('11000000000000000000000000000000', '11914671819298396166152780469749.700178398572262303378530442966993', 2, 1);
    t('84694.019523885', '84694.019523885', 16, 1);
    t('46198004778598591438177002960.8079419769', '46198004778598591438177002960.807941976894036', 39, 2);
    t('944540023268478392955679.235197603187056475911', '944540023268478392955679.235197603187056475911190458898708904', 45, 5);
    t('6384.0533', '6384.0532734075270642890', 8, 0);
    t('86.52585963444630269524402336034067', '86.5258596344463026952440233603406633365294', 34, 0);
    t('211698906765339555009000000000000', '211698906765339555008927202275920.017666588018743809326', 21, 6);
    t('93695.521776', '93695.521776317122397800801736652801', 11, 3);
    t('21', '20.73571174', 2, 0);
    t('44651390355733081311803973.631', '44651390355733081311803973.6306343', 29, 2);
    t('19628275338135891273639233000', '19628275338135891273639233078.3194337412', 26, 1);
    t('715174732399634482855366624736839961.178974112796', '715174732399634482855366624736839961.1789741127950654699759400963516', 48, 0);
    t('4010185.194165679', '4010185.1941656790873904791', 16, 1);
    t('199763544.3', '199763544.2746', 10, 5);
    t('792604593150000000000000000', '792604593148129965009152274.7', 11, 5);
    t('95649091688620000000000', '95649091688615551968376.399825', 13, 6);
    t('910000000000000000000000', '913366499416274482076806.54925361292970753237933', 2, 4);
    t('5713.74819', '5713.7481900', 13, 6);
    t('167100', '167006.16906099738293329453555919087', 4, 2);
    t('91.8149489203336825896217615649', '91.814948920333682589621761564879978', 30, 6);
    t('385450160421689186188707104582074317519.6', '385450160421689186188707104582074317519.5011', 40, 0);
    t('980700000000000000000000000000', '980624231078939283428476194025.3425674466530668543518761186154209', 4, 0);
    t('338479692666105014616000000000', '338479692666105014616253331245.01389376828879865108105798429640384', 21, 1);
    t('174223066779502216000000000', '174223066779502215997051823.30021471115775', 19, 5);
    t('321274436690589321065314360761529600', '321274436690589321065314360761529598.3926958740287183203559360243861156', 35, 0);
    t('965260000000000000000000000000000000', '965258557685921663672915807775689926.372616852406174796393011112241', 5, 5);
    t('20495856361024494227552150059704.128252538', '20495856361024494227552150059704.12825253784', 41, 0);
    t('347539608767858771651414903881.81676443931828298', '347539608767858771651414903881.816764439318282980413', 47, 6);
    t('319471356791473500000000000000000', '319471356791473556138054433838044.90552027814', 16, 1);
    t('3758410000', '3758411337.659079355464558774017310777636076246', 6, 1);
    t('986791401418080896.725674945574053721614668858858', '986791401418080896.725674945574053721614668858858', 51, 4);
    t('172860000000000', '172864222463390.0005522631729828976762398559', 5, 5);
    t('56.049', '56.049', 6, 6);
    t('8343815925404573683080000', '8343815925404573683077429.4758959133829073879054267509464514', 21, 4);
    t('5454663865.0064130258909884643960591289', '5454663865.006413025890988464396059128886823', 38, 6);
    t('9581036221000167548404063617040830845674.8563402624162914896', '9581036221000167548404063617040830845674.8563402624162914896694713537', 59, 1);
    t('500000000000', '410956568424.9910595380985324', 1, 0);
    t('32006369380000000000000000000000000', '32006369376256480193142996925472582.0', 10, 0);
    t('5222032172895746.90277', '5222032172895746.902766092011', 21, 4);
    t('8439793217442.92', '8439793217442.92137711335321', 15, 4);
    t('9.866065033617488770826998', '9.8660650336174887708269980', 28, 4);
    t('44049352.2266915297896690888554259285496418688', '44049352.22669152978966908885542592854964186877', 45, 0);
    t('26847260395161412327091304326000', '26847260395161412327091304326394.7663271691497120655790427', 29, 3);
    t('1000000000000', '1085137384726.66574913879922622187703130425247', 2, 3);
    t('9.9456', '9.9456', 6, 6);
    t('428952193056.953', '428952193056.953', 15, 6);
    t('1941911780790529313259513.70860258', '1941911780790529313259513.70860257943008396576247829795083', 33, 5);
    t('23968438750336.405474280759814', '23968438750336.405474280759813318106', 29, 0);
    t('315154230499048000000000000000', '315154230499048332963529384927.93303628365695160150210696280635993802', 15, 5);
    t('6500000000', '6480935501.55496974285182162603356273156890', 2, 4);
    t('42302765724017563090210326618736869271', '42302765724017563090210326618736869270.98392129213256644255722', 39, 2);
    t('608747404736337255085100901273291565', '608747404736337255085100901273291565.318094268584914528314761117', 36, 3);
    t('3154176290801000000000000000', '3154176290800715548619630621.0355772810773559541', 13, 4);
    t('38865242750002932570887590800000000000', '38865242750002932570887590890877550733.5505553313425194996432914268372269', 27, 1);
    t('59457914554626139.52', '59457914554626139.52065942543632055255656', 19, 5);
    t('8728384988174071448.69', '8728384988174071448.69274', 21, 4);
    t('8209486790494710645762862448749.80842396605900775935', '8209486790494710645762862448749.80842396605900775935', 54, 1);
    t('963188883728.74069481457014205', '963188883728.740694814570142059', 29, 3);
    t('93249710101545870000000000', '93249710101545874917863173.03705388925', 16, 3);
    t('88179259072684818766023076456.5889523232', '88179259072684818766023076456.58895232320350', 39, 4);
    t('776981797289252251.093401546863758214465', '776981797289252251.093401546863758214465286704664334', 39, 4);
    t('188670879464255056850261739500000000000', '188670879464255056850261739507365918265.9293874736873279', 28, 1);
    t('790372524561301386433116300000000000', '790372524561301386433116293301042112.0', 25, 2);
    t('117626300000000000', '117626347290152284.584779', 7, 3);
    t('353396834933281759072773.805789609148', '353396834933281759072773.805789609148', 37, 4);
    t('4274522263032700000', '4274522263032712850.0041729873', 14, 4);
    t('8350000000000', '8345606845226.45566624', 3, 6);
    t('7139338928.10029575', '7139338928.10029575', 19, 2);
    t('1.28993795996', '1.289937959964596899900', 12, 3);
    t('8777400901795823.67', '8777400901795823.66842604905538596685215930', 18, 6);
    t('829083805.750913', '829083805.75091307615246295390233836347298103', 16, 1);
    t('99745760328169140340', '99745760328169140333.10321', 19, 0);
    t('39561800000', '39561788882.62936468719', 6, 0);
    t('4659674912019020909769477616.1809485971278002', '4659674912019020909769477616.18094859712780016763510', 44, 2);
    t('9513568631.54', '9513568631.5362634854217681854845578621681451031', 12, 2);
    t('71608079246048915608970559007661.255', '71608079246048915608970559007661.255734918', 35, 1);
    t('4710.2762273634840929733075294039', '4710.27622736348409297330752940382966', 32, 0);
    t('491431256000000000000000000000000000000', '491431255824882855240158558614835211186.2941426095181713', 9, 0);
    t('442.6207', '442.6207173360599954126438578214', 7, 3);
    t('37025.890491', '37025.890490940768', 11, 0);
    t('830221444669', '830221444668.904474045532659191532', 12, 4);
    t('9214.33035312663577', '9214.3303531266357743116142146275368', 18, 4);
    t('99114552906189452084831000', '99114552906189452084830651.235559738', 23, 0);
    t('9240000', '9237492.48741663256357', 3, 2);
    t('98638337070765360464290.142476943377927319135', '98638337070765360464290.142476943377927319135', 44, 4);
    t('7730506503679581800000000000000000000', '7730506503679581774208931329691431760.4070160', 17, 6);
    t('6490946891.109036932756', '6490946891.10903693275578875322924308057719606', 22, 0);
    t('1523069.837539194', '1523069.837539193635000383', 16, 6);
    t('67402561079260.60795042', '67402561079260.60795042', 24, 3);
    t('888680146937040000000000000000000000000', '888680146937039418254096574971719707863.634', 15, 0);
    t('1364883198577172425.9044289', '1364883198577172425.9044289360876370', 26, 1);
    t('3664150693823831212.30495532780006476530216846662850648265', '3664150693823831212.304955327800064765302168466628506482646', 57, 2);
    t('3962922390.558883734580946668227696', '3962922390.5588837345809466682276953538', 34, 0);
    t('760000000000000000000000000000000', '758496233861793371677548532966632.786781459964892880', 2, 5);
    t('4640635532500000000000', '4640635532543664084831.61045102695251376591752966891370241786', 11, 3);
    t('6301.05994', '6301.05994456315754349439', 9, 4);
    t('478148873635312804.18861546266', '478148873635312804.18861546266', 29, 3);
    t('1620892431003646248920000000000000000000', '1620892431003646248929805806948171864119.15804899944048563307635492295136', 21, 1);
    t('3776549488473140000000000000000', '3776549488473136161690960746614.23650608177', 15, 4);
    t('521592676144262.737936012120007', '521592676144262.737936012120007', 32, 2);
    t('26772254407885657634772388614.281005', '26772254407885657634772388614.28100503696440301119529564808', 35, 6);
    t('13263721319886876.651', '13263721319886876.6514957936031616', 20, 3);
    t('21706.81', '21706.81298', 7, 1);
    t('62219056946980.9', '62219056946980.98169', 15, 1);
    t('8', '7.996990', 1, 5);
    t('28.4468', '28.4468294985', 6, 5);
    t('738050444.6990561423945', '738050444.699056142394456245079649632004583', 22, 4);
    t('990998710146129000000000000', '990998710146128821111029453.111462787315230353', 15, 5);
    t('975006704000000000000', '975006703986171330311.915262720306957123520839467366', 10, 2);
    t('59966312668618217.386207', '59966312668618217.38620677388406274877243465900683674', 23, 5);
    t('580483689475444082065373780806678615', '580483689475444082065373780806678615.0', 38, 4);
    t('777172152084000', '777172152083690.11416150', 12, 5);
    t('9933.8736549968', '9933.87365499675169000576761894360934', 14, 4);
    t('204363188056330491838545785942779252744.547514555869602587224004', '204363188056330491838545785942779252744.54751455586960258722400467538111498', 63, 3);
    t('53422865000', '53422864158.308155821650920225073', 8, 0);
    t('414.2195850509161124', '414.2195850509161124820290904615692666133996', 19, 1);
    t('293378769200000000', '293378769214089684.3880595714', 10, 1);
    t('425307248707947696963109693000000', '425307248707947696963109692956299.505433947257641', 27, 0);
    t('6464166104.2567204932', '6464166104.2567204932', 21, 5);
    t('640000000000000000000000000000000', '648169318683674318114139037144099.5529786496204790325222242232433387', 2, 3);
    t('1137917683060655236534000000', '1137917683060655236533363155.031079294757778258500907586486084', 22, 2);
    t('6000000000000000000', '5722939367160669775.59152823532798928035370037115863714', 1, 0);
    t('581091710823050000000000000', '581091710823047241756858512.713357759428', 14, 0);
    t('4952915437691054280969583000000', '4952915437691054280969583218972.8616946954635715417980', 25, 1);
    t('9705894787100000000000000000000000', '9705894787058704299917761166078683.9910480951422', 11, 5);
    t('3709152457254657158536681.732705728776998763593525103', '3709152457254657158536681.732705728776998763593525103159733', 52, 3);
    t('186530566473536251263.858877958', '186530566473536251263.858877958474287539', 30, 4);
    t('18185426229191232807040619835808.46825483359797368754', '18185426229191232807040619835808.46825483359797368754', 52, 3);
    t('41.5133038761731262218686', '41.51330387617312622186860688055631484', 24, 4);
    t('892734300000000000000000000000000000000', '892734285316609348922285835347841194677.279150490649933902110687168010', 7, 4);
    t('8867741117000000000000000000', '8867741117136144692773028671.48789537068725372709865944763', 10, 4);
    t('58066.6119296613521', '58066.611929661352058113440', 18, 0);
    t('610095820904.3119523193', '610095820904.311952319394878656597629841829693065', 22, 3);
    t('45147764430505826.0397992803707428783769210234', '45147764430505826.03979928037074287837692102337', 45, 2);
    t('2365934669507425116.685560049269016191', '2365934669507425116.685560049269016191181112056065', 37, 4);
    t('76328567052192004276.32866340568679575775108106109', '76328567052192004276.3286634056867957577510810610864', 49, 5);
    t('9341865065715748.71038922653450359770952', '9341865065715748.7103892265345035977095284', 39, 3);
    t('1893828187345809.95025475161610431769040580234726', '1893828187345809.95025475161610431769040580234726', 51, 5);
    t('675418012725769678669671724576651465', '675418012725769678669671724576651465.457619554107058395110952042747864102', 36, 5);
    t('722723049408340056285000637093.2', '722723049408340056285000637093.1564846486282233386998', 31, 2);
    t('8039307818523262169309113965.237955673046573', '8039307818523262169309113965.2379556730465735379', 43, 1);
    t('620699795.768772003', '620699795.7687720027642576108299448', 18, 6);
    t('677976409911497700000000000000000', '677976409911497719121490267737187.9281869113', 16, 1);
    t('9913200270263800000000000000000000000', '9913200270263779985846761891158202865.88', 14, 2);
    t('95447550983654000', '95447550983654256.94', 14, 5);
    t('9727301256933984906097986874000', '9727301256933984906097986873657.0', 28, 0);
    t('83100000000000000000000000000000000', '83144797009396341772640576982431626.06487716198510170943250663066', 3, 5);
    t('76723773.04423217932', '76723773.0442321793176553237', 19, 6);
    t('8400000000000000', '8442290748483192.541766601132013892502484734171240', 2, 5);
    t('80224755820119141339232053676610000', '80224755820119141339232053676618497.04561623225397344999', 31, 3);
    t('655778655223372.97', '655778655223372.965', 17, 0);
    t('20184630477822715391390984001199.1866444533896426202887666335307182015', '20184630477822715391390984001199.1866444533896426202887666335307182015', 71, 6);
    t('41930826996134220', '41930826996134221.1974945198118251722677898312235505', 16, 6);
    t('9401169208627113200000', '9401169208627113178099.432092017583166044', 17, 6);
    t('46844285000000', '46844285165612.4277871838', 8, 5);
    t('732653734623947749013990.255', '732653734623947749013990.2554245257179', 27, 3);
    t('817615530000000000000', '817615525922580553380.874869911815006035161427956123678666', 8, 5);
    t('648900000000000000', '648881720909905928.428439970149048878644930752628', 4, 2);
    t('85228991367.461452', '85228991367.46145277847248857183769451804173', 17, 1);
    t('239393776.69622644151182', '239393776.69622644151181623925655', 23, 4);
    t('33546748243104239469877559937.363355786582993', '33546748243104239469877559937.363355786582993', 44, 0);
    t('4000000000000000000000', '3823084385271016278125.11', 1, 5);
    t('77046259820455751626.6026808001342537', '77046259820455751626.602680800134253741613198881', 36, 6);
    t('26687373854107297015683.744', '26687373854107297015683.74432725585956728677271691574', 26, 3);
    t('976672549609984585227222778700000000000', '976672549609984585227222778792212901570.55874088', 28, 1);
    t('2.5616142669200936365611', '2.561614266920093636561053', 23, 2);
    t('94492716203969.2225869573684291964191137943094515', '94492716203969.2225869573684291964191137943094514534843', 48, 0);
    t('990770621178724769771370000', '990770621178724769771371129.99176486560800414', 23, 3);
    t('691412410360950760000000', '691412410360950750128020.48447221650237091009047', 17, 0);
    t('29040659564825201725590.2424704967395', '29040659564825201725590.2424704967395', 38, 2);
    t('89559092584017288521728982.5', '89559092584017288521728982.5833198608864', 27, 1);
    t('1811614362251924.40512130616', '1811614362251924.405121306157849629', 27, 6);
    t('4633501475000', '4633501474602.054640995909183', 10, 0);
    t('678705341261.75297583718425453684090101', '678705341261.7529758371842545368409010097945103677', 39, 6);
    t('564078844253167778.8161248928271829563591373580664298', '564078844253167778.8161248928271829563591373580664298', 52, 0);
    t('88322850583909.7416016241', '88322850583909.74160162410286643', 25, 1);
    t('3846366946174062127758449605.7022758096', '3846366946174062127758449605.702275809555523634154040950898280612811', 38, 6);
    t('89658733400000000000000000000', '89658733356696254546423515335.28349', 9, 2);
    t('4761467.862', '4761467.861150141326149685649922793395', 10, 2);
    t('179.92617', '179.9261669', 8, 4);
    t('75069004857851352687270011001106689339.08735', '75069004857851352687270011001106689339.08735', 46, 5);
    t('3361644609630676688091028947986494293.045392455143082675351', '3361644609630676688091028947986494293.04539245514308267535094798418063548394', 58, 2);
    t('6.7942421735585618046', '6.794242173558561804595', 20, 4);
    t('51657502980255968504333490637.09662892780318174480380404', '51657502980255968504333490637.0966289278031817448038040396600018487', 55, 5);
    t('11855.226308528', '11855.226308528', 16, 6);
    t('61277000000000', '61276347667662.405824737', 5, 0);
    t('4174609640', '4174609636.0', 9, 6);
    t('687936219719625022378.9502', '687936219719625022378.9502057605855', 25, 3);
    t('193757131563910000', '193757131563911904.52682306122527060962641882465', 14, 6);
    t('387.455087', '387.45508661743', 9, 6);
    t('557618201766183005000000000', '557618201766183005041293682.3722668213410766424816487716234273', 19, 6);
    t('21128122508600497.05489', '21128122508600497.05488760', 22, 6);
    t('630252144005109.53366072283861382', '630252144005109.53366072283861381792', 32, 4);
    t('963826907622.364662995276', '963826907622.364662995275692188', 24, 2);
    t('4212653046051913081783119467271.37647511439591502128556', '4212653046051913081783119467271.3764751143959150212855631', 54, 3);
    t('2014467765.30542656127', '2014467765.30542656127', 22, 4);
    t('1479320000000000000000000000000000', '1479326666751842221164864465643932.73817033951340680', 6, 1);
    t('6944862581642049969018412438205282752.728862', '6944862581642049969018412438205282752.72886131507563067413320187385623099', 43, 2);
    t('9838375.64', '9838375.645', 9, 1);
    t('73392897409456861290131541129654.9256075066904', '73392897409456861290131541129654.925607506690362687103367844755771', 45, 4);
    t('89764270000000000000000000000000000000', '89764275311096607184088076035578686331.70483764463153360153147271493', 7, 1);
    t('615391331037547517357702.9541121335087', '615391331037547517357702.9541121335086812', 37, 5);
    t('2617216902482291199973663.484710966125', '2617216902482291199973663.484710966125', 37, 4);
    t('17129271618518517092155339580000', '17129271618518517092155339589985.34795326188', 28, 3);
    t('606535160521504794000000000000000000000', '606535160521504794060646691380816193844.7', 18, 5);
    t('34177380606532061226000', '34177380606532061225426.5490340', 20, 2);
    t('537578.927123', '537578.9271229227', 12, 0);
    t('5008643590729908333990610000', '5008643590729908333990609905.47558686510363696', 25, 5);
    t('4895936188848030000000000000000000000', '4895936188848031794116383643496168477.0801543', 15, 1);
    t('5683079691081606113599129167429526013511.48652875855', '5683079691081606113599129167429526013511.486528758557663781739723321875379906235', 51, 3);
    t('8629922951113.7511448487597458660248521234', '8629922951113.7511448487597458660248521234380600977', 41, 4);
    t('90278102650946226014809000000000000', '90278102650946226014809046003862793.7835979911', 24, 3);
    t('4821625627675322025353636064.57176', '4821625627675322025353636064.57175523338580806993894351876', 33, 4);
    t('201458719700489100373654957.825506491000336059519682', '201458719700489100373654957.8255064910003360595196823588160', 51, 3);
    t('53212790802213594493332', '53212790802213594493332.8650736614584969129110', 23, 3);
    t('8177197143472082210310164500000000', '8177197143472082210310164503390974.89434', 26, 3);
    t('6977664457930000', '6977664457920874.47627073206781', 12, 2);
    t('48200644415566530.498484', '48200644415566530.4984835390397628883909840', 23, 0);
    t('4000000000000000000000000000000000', '4192746732983495210446560254097740.47028995', 1, 1);
    t('755.023209', '755.02320914910644', 9, 6);
    t('66963000000000000000000000000', '66963023876901420169077780295.01885877196', 6, 3);
    t('505291668854707.70906269', '505291668854707.709062685721555135520850266111469308027', 23, 4);
    t('664850.618013', '664850.61801224118433274591608772154387', 12, 0);
    t('785103038952630143044641569204316756.63797054791874616052822183467149', '785103038952630143044641569204316756.637970547918746160528221834671492', 68, 6);
    t('4146164592920910235341149.587', '4146164592920910235341149.58704019746296185479', 29, 3);
    t('579742066579367045736.917004843230287530285579', '579742066579367045736.9170048432302875302855793', 45, 4);
    t('92817.84445', '92817.84445', 13, 4);
    t('772179455775.22422834897', '772179455775.22422834897889', 23, 1);
    t('4670558124138300', '4670558124138345.3777942422216231', 14, 3);
    t('11653701002.1779016933207', '11653701002.177901693320687576204027037389422679', 24, 6);
    t('661971346000462043332006500000000000', '661971346000462043332006546474696174.7167702709539', 25, 4);
    t('7951.2507068', '7951.2507067915133', 12, 0);
    t('83825.91569730694896932060436944', '83825.915697306948969320604369447056', 31, 1);
    t('179989.23863077943963', '179989.23863077943963', 21, 2);
    t('333610464960022048972485697598948476.3274927278', '333610464960022048972485697598948476.32749272770013219331', 46, 2);
    t('567620976076010000000000000000', '567620976076010425435148100001.0364396667132672575976', 14, 5);
    t('25166561739090000000000000000', '25166561739091850523119786031.356417774102', 13, 1);
    t('35769804358466980414449956124427509', '35769804358466980414449956124427509.31366843647801642057', 35, 5);
    t('5624000000000000000000000000000000', '5623994624926506484134510384338582.7213130433076592758171270720102084', 4, 2);
    t('52471579263868610000000000000000', '52471579263868615271801405152771.014318284628313', 16, 3);
    t('800000000000', '754585116996.9768', 1, 0);
    t('6845902810000000000000000000000000000000', '6845902808062306446526711601148487105964.760230279341124548333865130108451155441', 9, 2);
    t('25329930000000000000000000000000', '25329931835642796295008712007359.774', 7, 6);
    t('838205133360935.04', '838205133360935.0450208', 17, 1);
    t('960000', '964483.4958501605537391', 2, 4);
    t('6985940160812637750037427078390489093.5972419', '6985940160812637750037427078390489093.59724194766104006296648646911097283', 44, 4);
    t('85912832521353633900000', '85912832521353633962171.622844738617471351787937', 18, 1);
    t('22537.04', '22537.044131504334747686356', 7, 4);
    t('64820082671929139423989159247959991.27439244549390817', '64820082671929139423989159247959991.2743924454939081683', 52, 2);
    t('81309794072838871318854500310152.093', '81309794072838871318854500310152.09295684998521', 35, 4);
    t('2983.146', '2983.146', 8, 0);
    t('7649851971073922220000000000', '7649851971073922227632383137.12367829277753284010508541882606785617', 18, 1);
    t('4413319948694474085791.6528747', '4413319948694474085791.65287472187114939173359969', 29, 1);
    t('9169434008670526360155.858673', '9169434008670526360155.858673447171266307242005', 28, 4);
    t('87281481541792866.31680021820096', '87281481541792866.31680021820096251746992578412', 31, 3);
    t('1262882505899773546530000000000000000', '1262882505899773546530078940105109255.1340263465677131821878853130', 21, 1);
    t('90957735119480552000.1275792566121', '90957735119480552000.12757925661213812317307', 33, 3);
    t('95094204140327.386800069607', '95094204140327.3868000696069952245821304362528', 26, 6);
    t('782.8315804028037', '782.8315804028036627332091761646906', 16, 4);
    t('4070074897815719013854405000000', '4070074897815719013854404930086.0517196', 25, 2);
    t('7535527006575666259327800', '7535527006575666259327801.3118771661605538067531702602254', 24, 5);
    t('993000000000000', '992026987647463.516668', 3, 0);
    t('4698757980923794411324653119136506649753.9058533965214', '4698757980923794411324653119136506649753.90585339652148316962906302356153227', 53, 1);
    t('338765415.622999075063635041304250288156393', '338765415.62299907506363504130425028815639273842', 42, 6);
    t('444000', '443272.8', 3, 0);
    t('92863547832.26', '92863547832.25594', 13, 0);
    t('227484121340486.542512', '227484121340486.54251108296', 21, 0);
    t('7034', '7034.0', 7, 2);
    t('84677601800000000', '84677601708231089.8035199547', 9, 2);
    t('903992663388421800000000000', '903992663388421818078667246.7494838726767325648', 16, 3);
    t('9440650000000000000000000000000000', '9440656634642185509393801198086015.4', 6, 1);
    t('916417601649800000000000000', '916417601649865583796921805.163932264102471440203390988', 13, 1);
    t('3546786323316815500000', '3546786323316815404869.163629407284618604', 17, 2);
    t('747408692550834866360911935.8990445275165282197802717382548', '747408692550834866360911935.89904452751652821978027173825483', 58, 3);
    t('29804424518.256135774', '29804424518.256135774', 23, 2);
    t('15732791881955773293163440652450', '15732791881955773293163440652451.44291076191144', 31, 1);
    t('82293484047103499038.6975422642147137771867866208', '82293484047103499038.697542264214713777186786620705401', 48, 0);
    t('5288356.17', '5288356.17', 11, 2);
    t('75977708959803656203251104.81675609237952293524970939', '75977708959803656203251104.81675609237952293524970938678', 52, 4);
    t('1958555193503000000000000000000', '1958555193502587409635649733733.584', 13, 2);
    t('61097901998.6487577', '61097901998.6487577', 21, 3);
    t('4966635612645365.09586650722463', '4966635612645365.095866507224630866586307217888782761159', 30, 6);
    t('449349314276161751863885318023458.255606199257691366208268573', '449349314276161751863885318023458.255606199257691366208268573', 61, 6);
    t('3313802055037244406355.1441269459053985', '3313802055037244406355.14412694590539847622267', 38, 5);
    t('77476038634229841225203.057294319605', '77476038634229841225203.05729431960467859218', 35, 6);
    t('532731523000', '532731523075.4101466861580822959', 10, 1);
    t('279000000000000000000000000', '279332110982355758954949172.19150963', 3, 5);
    t('76199428363070721667903.826669566', '76199428363070721667903.826669566165506057243720918746', 32, 6);
    t('297600', '297631.47066887', 4, 3);
    t('8248110729241783273113.562111307100337407', '8248110729241783273113.562111307100337407', 40, 2);
    t('1000000000000000000000', '1236453638891742153022.8348728811', 1, 4);
    t('535330000', '535327351.8008', 5, 2);
    t('4308.58715419601937041522', '4308.58715419601937041522', 24, 5);
    t('13500654611668894713386183570622548.34241', '13500654611668894713386183570622548.34240861579', 40, 5);
    t('995.7155821993173011608323', '995.71558219931730116083231270171862650', 25, 3);
    t('239147599253876718.9504433491173273', '239147599253876718.95044334911732723218139304606724752', 34, 2);
    t('765497722616.8961003646', '765497722616.896100364573390975890478644136740197458', 22, 6);
    t('101979298537.68788178219', '101979298537.6878817821900758850756870223612522530', 23, 3);
    t('54564000000000000000000000000000000000', '54563203006377285590954695127259934956.72304700', 5, 2);
    t('937851991874226678.2', '937851991874226678.191965007088457', 19, 0);
    t('9141378468539383564638997660000', '9141378468539383564638997666387.513564869223', 27, 1);
    t('5140000000000000000000000', '5143738381911696659148152.2424402227698278853166', 3, 3);
    t('20497199023806.45608949', '20497199023806.45608949227987004920339939', 22, 5);
    t('81603182.7', '81603182.616632542195753770061305729184', 9, 2);
    t('1288222268118280000000000000000000000000', '1288222268118271079758562419926059399472.4003871704903395624929468673346838726', 15, 2);
    t('484726059819530000000000000000000000', '484726059819531746596562275263672330.122283607', 14, 3);
    t('680000000000', '675623616750.993125132354', 2, 4);
    t('87687220439051000000000000000000000000', '87687220439051554697153502010194056104.42861', 14, 1);
    t('2844992735024778164396876428400000', '2844992735024778164396876428386392.397101880541673725', 29, 6);
    t('4910468313236974384325553954.559', '4910468313236974384325553954.558968559307060603568610', 32, 4);
    t('7677382468596942606325100814949164462.335409794', '7677382468596942606325100814949164462.335409794', 49, 1);
    t('6958724457786172481654532020985.043', '6958724457786172481654532020985.043', 34, 2);
    t('460800', '460705.8028929240198868371532360965971117977776', 4, 2);
    t('81763325373301098575053681.009551104718485', '81763325373301098575053681.0095511047184850495148390898', 41, 5);
    t('77290000000000000000000000000', '77298476544997406115741457621.48100777755241170668034650809', 4, 1);
    t('2862358127844547877974880690522730000', '2862358127844547877974880690522726982.49409754', 33, 2);
    t('901843956540041489372610610000', '901843956540041489372610614345.487929674', 26, 5);
    t('45031671217607058672836306.48351134', '45031671217607058672836306.4835113355599', 34, 4);
    t('92083948494159391284091144794664446.380059732506918919702', '92083948494159391284091144794664446.380059732506918919702007475113945', 58, 3);
    t('177322371367658020851649000000', '177322371367658020851649147602.975655295218030854293908400956', 24, 3);
    t('4239542600712473433498612753.21060144321763992555366', '4239542600712473433498612753.2106014432176399255536577373285045425339', 51, 4);
    t('989142489.56454046202042259', '989142489.56454046202042259', 29, 6);
    t('39188324760962.64171', '39188324760962.641710', 19, 0);
    t('1784374158800', '1784374158866.4741261545314002975498', 11, 1);
    t('4889335896817404.6714320041572203', '4889335896817404.6714320041572203348794', 32, 3);
    t('3179414685590000000000000000000000', '3179414685592371949442250485606819.26696750990296429734141944', 12, 6);
    t('2903.26350044', '2903.263500440', 13, 3);
    t('1491343.789445631775761446419118151', '1491343.7894456317757614464191181515356911640598', 34, 3);
    t('5500628286601049.843663626181949', '5500628286601049.8436636261819485793447808823573910', 31, 4);
    t('3974711265332457200000000000000', '3974711265332457176491911497563.9361', 17, 4);
    t('93.28105427667321036', '93.281054276673210359', 19, 2);
    t('79910228655846918061600000', '79910228655846918061589185.101353368484158631096929283677512827019', 21, 0);
    t('4389523222041915395738636382677027050000', '4389523222041915395738636382677027049670.297472800121808897652436', 36, 0);
    t('2894245566479613750113978122489420.5686284128195803188', '2894245566479613750113978122489420.56862841281958031877868147223357225', 53, 4);
    t('69268.87', '69268.8680350797908703052101348319784126253336', 7, 5);
    t('9666241652305682011019739397080.00734524875465085184171892', '9666241652305682011019739397080.00734524875465085184171892', 59, 3);
    t('635929.548704563', '635929.5487045626', 15, 5);
    t('76300000000000000000000000000000000000', '76310564821807158327636241720480644515.1231163364457981', 3, 1);
    t('83264642865143610124952.8938841743748792', '83264642865143610124952.89388417437487928505', 39, 1);
    t('92000000000000000000000000000', '92457508117387550193533961053.649117111370385292825169225347531361928', 2, 4);
    t('372212684000', '372212683598.46838580194648', 9, 2);
    t('293317362888487138283898', '293317362888487138283897.94813204526700655025', 24, 5);
    t('540932267948734432302811500000000000', '540932267948734432302811471443249276.08956', 25, 5);
    t('99528255762.2542884876122400046393060705971', '99528255762.2542884876122400046393060705971', 43, 3);
    t('9842064017828774440014665716000', '9842064017828774440014665715259.07392567114931921563803824704', 28, 0);
    t('9257402884539224592545000', '9257402884539224592544998.5235395', 23, 0);
    t('592195536918.89571497', '592195536918.89571496698678', 20, 6);
    t('339361634697985235138010000000000', '339361634697985235138015134706291.4759019746232166421087558397', 23, 3);
    t('413350000000000000', '413357867015514703.0', 5, 1);
    t('979391203987271002320000000000000000000', '979391203987271002317604546377693480297.7160282', 20, 5);
    t('890522255588.73018805', '890522255588.73018805', 23, 1);
    t('25037.2370029415', '25037.23700294156464', 15, 1);
    t('7995420000000000000000000000', '7995421588391041483528210141.23304536200210041105595235895779443253', 6, 6);
    t('88564695659091.46951834452', '88564695659091.4695183445295777003281956', 25, 3);
    t('260000000000000', '264312944832123.1367762400', 2, 6);
    t('4585488016816165814.015643632811011869389', '4585488016816165814.015643632811011869389', 42, 3);
    t('140669577312884632116870000000000000', '140669577312884632116865327734939179.947878360802862', 23, 2);
    t('96209021148142479587160000000000', '96209021148142479587160212319059.9038365', 22, 3);
    t('93762620000000000000000000000000000', '93762613618878400141925884059424443.98519863953502544661275234332363', 7, 0);
    t('8267.5096', '8267.50961103464486', 8, 5);
    t('362424000000000000000000000000000', '362424306795992053311357137129537.8414102030', 6, 6);
    t('95503344561317.6370318237701479895891', '95503344561317.637031823770147989589133607', 36, 3);
    t('4001511668629746217339142490000000', '4001511668629746217339142488051652.94896524824', 27, 5);
    t('68736010855641917958289083273.585584014416', '68736010855641917958289083273.585584014415926607368819445232514201', 41, 2);
    t('6870000000000000000000000000', '6871708368904554915177185203.0', 3, 6);
    t('110898.160011744397077892', '110898.1600117443970778915267926', 24, 2);
    t('6626043270925832716786.485210605', '6626043270925832716786.4852106058793015956475573211428671576', 31, 1);
    t('936650965006742228.52600524260614020916', '936650965006742228.52600524260614020915206023266203478510', 38, 0);
    t('8238145357339080403033.536346', '8238145357339080403033.5363462651613679591293841', 28, 4);
    t('257767134524825922657125.78657581464', '257767134524825922657125.78657581464395208486', 35, 3);
    t('339952052376917906516404363373.98731662403022', '339952052376917906516404363373.987316624030219727477330', 45, 4);
    t('8716129749488751033767283247.40704041', '8716129749488751033767283247.4070404037216', 36, 2);
    t('650493310.2529236530272602982', '650493310.25292365302726029823213801543342203910', 28, 6);
    t('25461791801.521422', '25461791801.521422290366221086', 17, 5);
    t('2267747409765595580', '2267747409765595579.28303', 18, 4);
    t('9694670132400000000000000000000', '9694670132376352453260862522056.2171007311697', 11, 5);
    t('350690', '350687.3193945661852540084656937962117895', 5, 6);
    t('3129043313543749518670352458.5', '3129043313543749518670352458.503656079916403', 30, 4);
    t('9044025.8460543775', '9044025.8460543775', 17, 5);
    t('400', '375.083', 1, 4);
    t('7700000000000', '7696045820777.2', 2, 5);
    t('443.24085276', '443.2408527588027827', 11, 6);
    t('969168337724404191455796109.202', '969168337724404191455796109.2022384545084493540', 30, 4);
    t('66578339440000000000000', '66578339444138406102879.10556956649', 10, 1);
    t('9.781', '9.78091', 4, 4);
    t('565395665366528250192.454055559505779511', '565395665366528250192.45405555950577951134787413536747', 39, 6);
    t('2291197809938635156685146900000000000', '2291197809938635156685146923222136483.2335617135895814877112536629710620685940', 26, 4);
    t('8930000', '8928528.14', 3, 2);
    t('7665000', '7665321.9187089397259198541955255158116376503981', 4, 5);
    t('5000000000000000000000000', '5503306952209997403348929.3398832288866332', 1, 1);
    t('39266215762.45290594550460762', '39266215762.4529059455046076241', 28, 1);
    t('769458324216100000000000000000', '769458324216146062473191568814.733', 13, 6);
    t('2176334000000000000000', '2176333982708480689269.345358976002983', 8, 4);
    t('917418.17190739874566462723224639801471434', '917418.171907398745664627232246398014714336481', 41, 4);
    t('35936559361.52087537643754794806160952', '35936559361.52087537643754794806160951911853825', 38, 0);
    t('320481693849571934020000000000000000', '320481693849571934014908777703626400.356619750849273703334848419631502520136', 20, 2);
    t('277820552825439978335272185827274848000', '277820552825439978335272185827274848375.065939811', 36, 3);
    t('9101344704979041527.091333493387122', '9101344704979041527.091333493387122', 35, 5);
    t('2000', '2007.22029005569772', 1, 4);
    t('3660013391161446845930000000000000000', '3660013391161446845933048060165065592.112371', 21, 4);
    t('75619523206222548265000000000000', '75619523206222548265019369332315.067125469', 21, 1);
    t('36533890639136881014148213120000000000', '36533890639136881014148213123568025477.253998298441', 28, 3);
    t('6230000000000000', '6226078460444062.035673584208', 3, 0);
    t('28375605044097427503224.91', '28375605044097427503224.91', 26, 5);
    t('165182686671902618132878452878.45904146024894606', '165182686671902618132878452878.45904146024894605987846314920686', 47, 2);
    t('52255560860000000000000000000000000', '52255560853735178963967943996648216.4530011651832243192159661049', 10, 0);
    t('8630953896264992974012300', '8630953896264992974012332.227', 23, 4);
    t('4553959846171.192464941436556', '4553959846171.19246494143655599482', 29, 4);
    t('40921024493463159581990', '40921024493463159581989.5809939714721869474575756807216441250576', 23, 4);
    t('7739125004710320.74', '7739125004710320.739535498326400150202463082969893301608', 19, 6);
    t('400000', '313637.40230598', 1, 0);
    t('70000000000000000000000000000000000', '77378294512934932600462786753996959.655752553709837104709431214186984', 1, 3);
    t('7000000', '7020921.195673443260136228110568788', 2, 1);
    t('77965190', '77965188.0366614823', 7, 2);
    t('28432688650884095695520.1181182813950029249167', '28432688650884095695520.118118281395002924916733', 45, 1);
    t('6758623302788423000000000000000', '6758623302788422678482788330732.0', 16, 6);
    t('96709318143167818477864932506.961', '96709318143167818477864932506.9608090055681274767937727870', 32, 4);
    t('60579506511732969487259623201665200000', '60579506511732969487259623201665204840.0', 34, 3);
    t('77409612474928159137380000000000000000', '77409612474928159137372978642610421617.3422994', 22, 2);
    t('3564539259975522389318000000000000', '3564539259975522389318399381402239.78874928', 22, 1);
    t('6076288921559036328.6204437794', '6076288921559036328.620443779320259', 29, 2);
    t('903074959787573.2', '903074959787573.2', 16, 6);
    t('405766946628957757020328743867438673063.47310805108', '405766946628957757020328743867438673063.47310805108', 51, 6);
    t('7785166500000000000000', '7785166460276687023595.868944998673', 8, 6);
    t('571651403459769193.160142534', '571651403459769193.16014253482567897056347917909', 27, 3);
    t('4619517207994467998000000000', '4619517207994467998339344737.51071686946816449039421809158', 19, 3);
    t('1123978038.482067854127511050099931271558', '1123978038.482067854127511050099931271557781', 40, 6);
    t('550043879576120500112.3186473689472021697', '550043879576120500112.3186473689472021697', 42, 5);
    t('2792.991201929', '2792.99120192989904760909918454', 13, 3);
    t('700000000', '778861829.0', 1, 3);
    t('493995796832.3714', '493995796832.371401404530', 16, 1);
    t('1550000000000000000', '1555074147424140721.43592182', 3, 3);
    t('243482916004598157069941104.41101293512921765384834870217', '243482916004598157069941104.41101293512921765384834870217', 56, 3);
    t('530772687633.8777897538', '530772687633.877789753782', 22, 4);
    t('44456046293943470579180414284.7', '44456046293943470579180414284.67867983815805810977613', 30, 4);
    t('1804897278649860484.45711317691108441112', '1804897278649860484.45711317691108441112', 40, 3);
    t('9046500000000000000000000000000000000', '9046540376384433865424104983224383412.794617396480611449096887787', 5, 1);
    t('9547721532.2789273', '9547721532.278927317072672479467365010440158', 17, 3);
    t('85791851047994907019129050000000000000', '85791851047994907019129042204975105769.809', 25, 2);
    t('16000000000000000000000', '16114607671010577899168.0310995664251250652374018', 2, 4);
    t('788724220065202454841.110016583546', '788724220065202454841.110016583546303370', 33, 5);
    t('4804344615391203329294887490654.863003562027769378200924311927518599', '4804344615391203329294887490654.86300356202776937820092431192751859841', 67, 2);
    t('690378989943867106357.19214949', '690378989943867106357.192149490916407', 30, 1);
    t('44183.5551', '44183.5550223531', 9, 0);
    t('401410182200', '401410182219.1', 10, 4);
    t('42380962377009067158018778499329873.6250308', '42380962377009067158018778499329873.6250307837445356995135372083387', 42, 2);
    t('21500000000000', '21522241215872.159', 3, 6);
    t('9828001151085272000000000', '9828001151085271121733219.4571045062764511428734592403214852905', 16, 2);
    t('95299647006776160000000000000000', '95299647006776162177508986330224.19122648459623894024226270345486941', 16, 6);
    t('653564.593871174348', '653564.5938711743478', 18, 6);
    t('35646378667162655924634189810694200', '35646378667162655924634189810694190.2', 33, 2);
    t('93000000', '92513140.005586849991215513794421', 2, 4);
    t('440517293.3834', '440517293.38341314245', 13, 3);
    t('285698', '285697.05793366946523320244', 6, 2);
    t('908873037529215958920518762838500000', '908873037529215958920518762838514526.50654907399067', 31, 5);
    t('26231504574443716000', '26231504574443716010.905605639569485455516679040468519742070', 17, 4);
    t('4.9699753689663', '4.96997536896634774282457632500925505', 14, 1);
    t('986141177664583720474590874086000', '986141177664583720474590874086578.849', 30, 3);
    t('43923471497568090.16785189331580776156204562425822', '43923471497568090.167851893315807761562045624258219496492', 49, 4);
    t('377793591239311706880246505.3706393844', '377793591239311706880246505.37063938435727384253759', 37, 0);
    t('464273510382973094329.29404741', '464273510382973094329.2940474059629811', 29, 4);
    t('8', '8.0', 1, 0);
    t('6252000', '6251999.00175', 6, 2);
    t('7863209438084888424953430716021433308.2021244895687275847838735', '7863209438084888424953430716021433308.2021244895687275847838735', 63, 4);
    t('970000000000000000000', '970280357620529012189.3624345759142969767656736', 3, 5);
    t('65223907242500000000000000000000000000', '65223907242500180359920298917660073566.9939102170589515493', 12, 3);
    t('57548188746663', '57548188746662.67', 14, 4);
    t('862637998581382902977.83', '862637998581382902977.83867631837831323796657120290', 23, 3);
    t('451125410401832463766.5', '451125410401832463766.5', 22, 6);
    t('268412506332521.52179901479161689828297', '268412506332521.521799014791616898282971', 38, 3);
    t('12989534861000000000000', '12989534860995577276739.36288177262050', 12, 6);
    t('5400000000000000000000', '5424106299879971302126.547981004928646', 2, 6);
    t('2600027757896985700000', '2600027757896985722714.38020486164242266219', 17, 4);
    t('4781225159726876.00478', '4781225159726876.00478480', 21, 5);
    t('18044000000000000', '18043885505064325.45835192211', 5, 6);
    t('66209435000000000000000', '66209435089116153606204.542', 8, 3);
    t('507014357229366098929000000000000', '507014357229366098928640781367144.0540759540042058501649474', 21, 4);
    t('30066825658047601.96380421400713862035835324280422614098', '30066825658047601.96380421400713862035835324280422614098', 56, 5);
    t('8310000000000000000000000000000000000000', '8312558538154197986605584172902990944451.584302231700184685305', 3, 6);
    t('80000', '76481.613875965100746486', 1, 5);
    t('2718582329316636409.1604581153526621', '2718582329316636409.1604581153526621993629961', 35, 1);
    t('56921284491000000000000000', '56921284491368077446815531.63733775709791618398002560', 11, 3);
    t('39311405056608918110065579122.848244009', '39311405056608918110065579122.848244009', 39, 2);
    t('23947356675255403719413472572760158911.29075667', '23947356675255403719413472572760158911.290756665203517763031393177814752', 46, 0);
    t('39140218951647271529038657.50533734073424671844811650305686397', '39140218951647271529038657.505337340734246718448116503056863971520', 61, 1);
    t('2063729725148000000000000', '2063729725148726984059451.460283', 13, 3);
    t('605335.772', '605335.77294', 9, 1);
    t('953.6160677374997333', '953.616067737499733318780144256827583160', 19, 4);
    t('9555031.3675', '9555031.36746716459116350609183204763850930', 11, 5);
    t('800000000', '765321903.328383731812011474861394235', 1, 2);
    t('12167179518792632924000000', '12167179518792632924489247.090116917813', 20, 5);
    t('57977195.2483520539550700247712', '57977195.248352053955070024771193978518598', 30, 6);
    t('75159155000000', '75159154987132.925843367257278678006629459732809', 8, 0);
    t('533668485415490000000000000000', '533668485415492109565317254353.0', 14, 3);
    t('7558236.2', '7558236.275145', 8, 3);
    t('9807089192869000000000', '9807089192869059113484.0182942001156962323418364', 14, 3);
    t('695089845288625801679825243520498079.03', '695089845288625801679825243520498079.03', 38, 2);
    t('18398713425.929144167386963084190881334', '18398713425.92914416738696308419088133371697', 38, 4);
    t('6581979578044951653827175010012223.107210537868391667227767', '6581979578044951653827175010012223.1072105378683916672277665937602', 58, 4);
    t('5489566114573992229000', '5489566114573992229211.910872184972714955277492', 19, 6);
    t('237545302902382453099902031798591.19702308', '237545302902382453099902031798591.19702308745012198136743513', 41, 1);
    t('110000000000', '110961214572.203102744', 2, 5);
    t('3542382372827000', '3542382372827150.003661722', 13, 6);
    t('332267769333799900040677452603932', '332267769333799900040677452603931.521170379214', 33, 2);
    t('3836940799185985174.6867557395561591', '3836940799185985174.686755739556159063923707825882483066', 35, 2);
    t('7729190502089342495440.67995563', '7729190502089342495440.67995563', 30, 4);
    t('420040000000000000', '420039711173301803.7012886247193', 6, 2);
    t('68989557051132473567804816513423623.783', '68989557051132473567804816513423623.783', 38, 6);
    t('50718700000000000000000000000000000', '50718652699585353489961994410892219.288523291345811289739999743317883', 6, 2);
    t('31.4', '31.438', 3, 4);
    t('4.726894767', '4.726894766551436297', 10, 4);
    t('122106513055.01658', '122106513055.016576538738665596', 17, 0);
    t('93.54', '93.540', 5, 0);
    t('9637732200000000000000', '9637732222388306782886.36', 8, 3);
    t('56069487484795832478444823478004.506377186', '56069487484795832478444823478004.5063771858886487916032641052512267', 41, 5);
    t('432004443954012288635248026471544882476', '432004443954012288635248026471544882476.236257892998828739', 39, 3);
    t('197958800077609756271401871909299845664.477115', '197958800077609756271401871909299845664.477115', 45, 3);
    t('57258651060391781872602690930062979800', '57258651060391781872602690930062979754.3962270414517041135332016435298029528588', 36, 6);
    t('59450000000000000000000', '59450528150107566870599.0', 4, 1);
    t('469.0744383693', '469.074438369378964', 13, 3);
    t('6025373045810577862305017.207', '6025373045810577862305017.2067561161026303941466721853425103527335', 28, 0);
    t('3623400000000000000000000000000000000', '3623395146076846109974390882778068004.0', 6, 5);
    t('57336888970.16', '57336888970.16226043350165070976048', 13, 6);
    t('2206226431753694580437142.463', '2206226431753694580437142.4626147413336581766517752587431266', 28, 5);
    t('97392610690280844761.651', '97392610690280844761.65057194', 23, 2);
    t('30389610942365757868537.791129', '30389610942365757868537.791128569468', 29, 2);
    t('36427.3386', '36427.33861924664', 9, 5);
    t('1935907299957300000000000000000000000', '1935907299957299383281080621667446614.6988672444446233694675013969', 14, 4);
    t('714371765687134846341422.4325200844855018', '714371765687134846341422.4325200844855018', 40, 1);
    t('56063216321677229326930670633038763566.6698708105', '56063216321677229326930670633038763566.6698708105464739919255396795740995', 48, 5);
    t('6678597472867608250000', '6678597472867608245309.1822159707219', 18, 6);
    t('7837755.89109749', '7837755.891097499938816776', 15, 3);
    t('5956500297724575377583405531032750434', '5956500297724575377583405531032750434.03', 38, 5);
    t('7363700000000000000000000000', '7363743215723349397471152836.917238851086497561123624', 5, 1);
    t('7190855656368997778868000000000000', '7190855656368997778867894126655633.50271322303709138173072341196488864763', 22, 5);
    t('653805372', '653805371.9727377478', 9, 5);
    t('2.16494832', '2.16494832', 11, 4);
    t('83657808343', '83657808342.5803196318709004800453821', 11, 6);
    t('962556120633886757140000000000', '962556120633886757146579327826.71827214128240940155074467687636154775', 20, 1);
    t('1680000', '1674400.95704276660633462', 3, 0);
    t('756498000000000000000000000000000000000', '756497826249893692788015960680818706311.213840638461219077194192911776559', 6, 4);
    t('67535150774849423556560561.33200805967562933726268', '67535150774849423556560561.33200805967562933726267201617040', 49, 0);
    t('9000000000000000000000000', '8945700060829376769005156.7927406002257699220310065971', 1, 4);
    t('63324597.608396736102486858', '63324597.608396736102486858', 26, 0);
    t('369462792120000000000000000000000', '369462792118835681691953350046876.63461090758830176271051327744851', 11, 6);
    t('52034387030', '52034387029.0', 10, 0);
    t('55856255777031.78334', '55856255777031.78334', 21, 2);
    t('83703331657811733873730124954.89136583850876065', '83703331657811733873730124954.8913658385087606461494998236987187', 46, 2);
    t('771539369434188.7793', '771539369434188.7793', 20, 3);
    t('62579578370097.22843542187', '62579578370097.2284354218665597187695', 25, 4);
    t('9990758', '9990757.80571101', 7, 0);
    t('690739.900390080545004167987692100953524830617', '690739.900390080545004167987692100953524830617', 48, 6);
    t('306222481448642969821169836038550.465760094647', '306222481448642969821169836038550.46576009464681881893', 45, 0);
    t('7836824758.84169903994739371685', '7836824758.84169903994739371684921873075343', 31, 2);
    t('114000246.9467', '114000246.946732447167', 13, 6);
    t('91.233432572', '91.2334325717959692047', 11, 6);
    t('510897390250237726969191.409652038', '510897390250237726969191.40965203795442470', 34, 0);
    t('355636663513372553351756000000000000000', '355636663513372553351755543054744485794.9009731268', 24, 5);
    t('841686833094.936617', '841686833094.93661673665211506623', 18, 4);
    t('60163376900000000000000', '60163376949489393647431.1926546149470078', 9, 1);
    t('827650.76302', '827650.763020625498029494246219493', 11, 6);
    t('5261578908406621737168375000000', '5261578908406621737168375374971.9875', 25, 4);
    t('58357054290526050547608119876.69177', '58357054290526050547608119876.69177', 34, 5);
    t('137934713229211052097638372049.53566673853582961080429923', '137934713229211052097638372049.535666738535829610804299230', 57, 4);
    t('51175786670000000000000000', '51175786669293383897149312.6126543', 11, 0);
    t('949994965082745217247823.9673635870799589', '949994965082745217247823.96736358707995889133594021', 41, 0);
    t('9420481616913285304990000000000000000000', '9420481616913285304994917896001657309040.40003480948445724379302213553524106', 21, 1);
    t('3625478000000000000000000000000000000000', '3625477561776942460055313800275838747875.1', 7, 0);
    t('1176032.27794615051841343', '1176032.27794615051841342785764792693036818', 24, 0);
    t('669920557579098.131716', '669920557579098.131716', 23, 5);
    t('36210109945005785834916298574.646', '36210109945005785834916298574.6456003545327715398293548933', 32, 4);
    t('859070538027.4373654065493936', '859070538027.4373654065493936', 28, 5);
    t('5522.2', '5522.2', 8, 0);
    t('624181087990000000000', '624181087985752371409.94175154528983632432500082240161858', 11, 6);
    t('576263564010000000000000000000000000', '576263564009755249865447668033878351.9352082616060', 11, 4);
    t('8629896.71825', '8629896.718250', 14, 0);
    t('590106514193.64486776268633453855236', '590106514193.644867762686334538552360395', 35, 3);
    t('415511438639079437988475110630210', '415511438639079437988475110630203.89166781277393192', 32, 2);
    t('1380135197159.929156854421571', '1380135197159.92915685442157138489476112392336', 28, 1);
    t('2462691640030760200076.93265638', '2462691640030760200076.93265637450566', 30, 0);
    t('860807649648642700000000000', '860807649648642625689234038.379445716250037', 16, 0);
    t('90000000', '88155719.90237020967999', 1, 6);
    t('4228997121469597625636.135', '4228997121469597625636.1343609', 25, 0);
    t('632674120', '632674120.0', 10, 6);
    t('61462425002285000000000000000', '61462425002285081338720243322.5606597679929684092469981075', 15, 3);
    t('7.296799', '7.29679922', 7, 3);
    t('941561842762273739313160.176077322071', '941561842762273739313160.176077322070925948', 36, 5);
    t('620621000000000000000000000', '620620747951074460546745224.72525943332605313870445312816788553', 6, 2);
    t('22431536112672460000000000', '22431536112672453730318033.998558878820772882020742833416384105', 16, 0);
    t('318557846387066660053428324458200000', '318557846387066660053428324458163166.14390850479066962887841394344881', 31, 4);
    t('98056928475951339400000000000000', '98056928475951339431279685210396.3904923442870039112679275634', 18, 4);
    t('61.77022', '61.7702201759441817432795049807736226865141', 8, 5);
    t('4609.428558', '4609.42855754723352953934692827665', 10, 5);
    t('3066.62413429743148619', '3066.6241342974314861935176936', 21, 4);
    t('582205379828459965937000506020265.172971753581412', '582205379828459965937000506020265.172971753581412', 51, 3);
    t('44.4', '44.370231411781621174', 3, 6);
    t('6699347564821666786989200000000000000000', '6699347564821666786989251433013608371049.770', 23, 3);
    t('356966022627275684844577120980082.5', '356966022627275684844577120980082.4989232813415', 35, 6);
    t('7201209046645731866911450000000000', '7201209046645731866911448211523761.656563035177194833412160660', 24, 2);
    t('200000', '209028.8519016', 2, 1);
    t('34865130000000000', '34865121390718261.2', 7, 2);
    t('41582370', '41582366.0', 7, 2);
    t('2894433900000', '2894433971649.0', 8, 1);
    t('14759225267853494.506220539681318051655651155607', '14759225267853494.5062205396813180516556511556068', 47, 0);
    t('23457286.2', '23457286.206', 10, 3);
    t('6046871.5194996153', '6046871.5194996153459619073126847835000974204678', 17, 3);
    t('9037332770648933.633020925444031666211173752491302', '9037332770648933.6330209254440316662111737524913024122371', 49, 4);
    t('25.48302803', '25.48302803', 10, 0);
    t('651220614992700000', '651220614992699372.65621970246999104', 14, 2);
    t('624705304132973.57', '624705304132973.56998', 18, 6);
    t('70434822560656100000000000000000000', '70434822560656117993258499200289630.30744772111707', 15, 3);
    t('7524534817.9286584065609374908344948482394', '7524534817.928658406560937490834494848239421707151', 41, 3);
    t('61219900000000000000', '61219895058203039113.1652542721870', 7, 6);
    t('262731683424620000000000000000000', '262731683424622612704617807468567.684', 14, 1);
    t('624947.41286667698', '624947.41286667697798', 17, 6);
    t('91976569823189214422230320652541413939.335557260568359814049694164', '91976569823189214422230320652541413939.335557260568359814049694164', 65, 6);
    t('302621780000000000', '302621786701102051.012067016', 8, 3);
    t('77634775.9724718655881094555', '77634775.9724718655881094555', 27, 4);
    t('9374841568330829.8694440467375', '9374841568330829.8694440467374740918153155', 29, 0);
    t('213675059347189748035017867759.113082698003417506593974857826', '213675059347189748035017867759.1130826980034175065939748578256', 60, 2);
    t('6688196691086.90631110165735762203278176698959', '6688196691086.90631110165735762203278176698959986', 45, 3);
    t('2370184605043415419367.18314061079671', '2370184605043415419367.18314061079671', 37, 3);
    t('9.667616', '9.66761606008', 7, 5);
    t('938175900016125741748.47982167999785476855', '938175900016125741748.479821679997854768552', 41, 3);
    t('485486971820.001445669297745265', '485486971820.001445669297745265336', 30, 5);
    t('81503716181697586763650382336501.09578087196', '81503716181697586763650382336501.09578087195404', 43, 0);
    t('3579212469767093142871930', '3579212469767093142871930.33680', 24, 6);
    t('663711546230709295732833.923', '663711546230709295732833.92307435123672', 27, 1);
    t('1.576', '1.5763618202116840078011990004382117', 4, 1);
    t('5281389347491463996940620645157090000', '5281389347491463996940620645157086730.643051905445577705303666305', 33, 2);
    t('3918172397800.321139911573294736095733', '3918172397800.321139911573294736095733', 38, 2);
    t('111970390000000000', '111970393405042418.0887493970126379974233091280365243141', 8, 3);
    t('9511632267356265856583220376521085715208.45955266636407336854081278262840513263', '9511632267356265856583220376521085715208.4595526663640733685408127826284051326300', 82, 5);
    t('458435466272059', '458435466272059.26', 15, 5);
    t('953441855745720000000', '953441855745719549856.91023614038', 14, 5);
    t('41923944761.74220302886', '41923944761.74220302885600233057', 22, 5);
    t('19923343330570396592497895.69038581918848541891204721520316301', '19923343330570396592497895.69038581918848541891204721520316300965', 62, 5);
    t('6111.119', '6111.118524960335732960970668', 7, 5);
    t('182500000000000000000000000', '182509086613850656378857305.50510806147702902195468425288', 4, 3);
    t('7336000000000', '7335640816733.4992965170440297210629875688901', 4, 0);
    t('968655.53', '968655.537012191698581793615985142094537936011', 8, 3);
    t('1475879175.8267', '1475879175.8267705635', 14, 3);
    t('4351636838168770948677346.161', '4351636838168770948677346.161868074865084139726278542934981368', 28, 1);
    t('291000000000000000000000000000000000', '291952036290366496064162189976350800.762310277861134', 3, 1);
    t('8785000000000', '8785800140606.020784429019413713094351618941192255', 4, 3);
    t('7072400', '7072449.23490240', 5, 3);
    t('3725613.6', '3725613.573872', 8, 4);
    t('58592879.945', '58592879.945058085056755', 11, 5);
    t('48939406319386930927174.26614359', '48939406319386930927174.26614359', 33, 2);
    t('9834361356627881059631782644639575.15591', '9834361356627881059631782644639575.15591', 42, 2);
    t('986978000000000000000000', '986978694728685901646644.461424444336700531', 6, 3);
    t('74256.9199803399276615', '74256.919980339927661533243393353377', 21, 1);
    t('8.3', '8.3', 3, 6);
    t('2223000000000000000000000000000000', '2223295328891520126318236063245373.9857469', 4, 3);
    t('2882330831868.0265064749593501', '2882330831868.02650647495935007949407263356031770587', 29, 2);
    t('98328396901.172431', '98328396901.17243099851493059944199', 18, 2);
    t('76949906.67702632', '76949906.6770263159332499097199661506699770610048', 16, 6);
    t('65800000000', '65807247118.8', 3, 3);
    t('888592494000000', '888592493563560.71333522887061194', 9, 0);
    t('4127817829107947427300000000000', '4127817829107947427277982505270.506086062140000', 20, 6);
    t('25600000000000000000', '25633767282010437571.0621274747710', 3, 5);
    t('7413352695000000000000', '7413352695374763956451.58', 10, 1);
    t('273656046300426616773.31238982', '273656046300426616773.31238981361804', 29, 0);
    t('9202934399911618042634149551001246516456.503496224', '9202934399911618042634149551001246516456.5034962233891209', 49, 0);
    t('2721429019341157049.671671', '2721429019341157049.671671', 28, 3);
    t('61603735916.63', '61603735916.62084', 13, 0);
    t('71897122155609.654898246', '71897122155609.6548982462366512503044587941118', 23, 1);
    t('402015895890836702510800', '402015895890836702510773.76143075426100572', 22, 6);
    t('712.5165297091', '712.516529709104989915472417906792461269377', 14, 1);
    t('983221318.39947840357640499755', '983221318.39947840357640499755018534075054', 29, 4);
    t('646333865', '646333864.729792182680611462345', 9, 0);
    t('582185374966060399988622097361363761900', '582185374966060399988622097361363761913.7', 37, 6);
    t('5626630000000000000000000000000000000', '5626634509684157752354968158438639456.432066741727', 6, 4);
    t('7524140296482181697771543.455616629727', '7524140296482181697771543.455616629727', 40, 3);
    t('686042632800000000000000000000', '686042632807885533118824764252.0', 11, 1);
    t('6488196127476106185883', '6488196127476106185883.0', 26, 6);
    t('72602.283', '72602.283', 9, 3);
    t('1057000000000000000000000000000000000', '1057495630075528511257326598593080247.2877704', 4, 3);
    t('50409547395.837098', '50409547395.8370988488395', 17, 3);
    t('56910760683566994374600771.65529268023250285223988951', '56910760683566994374600771.65529268023250285223988951386843234', 52, 5);
    t('8000000000000000000000000000000000', '7570653277785218694996754528030605.308770410436496505', 1, 5);
    t('8935287016613076669531433.7494', '8935287016613076669531433.7494', 29, 2);
    t('651521409393371.61383073003641104', '651521409393371.613830730036411040122', 33, 1);
    t('5686842064050076661002407240051949939700', '5686842064050076661002407240051949939703.653612361778370710358134872757', 38, 6);
    t('9844756035317937444.72115821', '9844756035317937444.72115821449618020261795623232', 27, 6);
    t('837353546.9662074781359439587764351720857', '837353546.9662074781359439587764351720857', 41, 6);
    t('61100010000000', '61100010117899.302364782355644', 8, 5);
    t('53506507908340733000000000000000000000', '53506507908340732125183761691434378622.1807338507810083654468022068107', 17, 0);
    t('5885699082300000', '5885699082263176.2676506681058', 11, 6);
    t('5.1531301659143054172614704285', '5.1531301659143054172614704285', 32, 0);
    t('393611.363390329829928', '393611.3633903298299277171285337505933280', 21, 6);
    t('203.401', '203.40135954426808', 6, 1);
    t('774608939742875843500000', '774608939742875843593026.0', 19, 3);
    t('360435000000000000000000000000', '360434645705404210833925724656.451595797512580599488143849', 6, 0);
    t('7404215579719728717495381644855454108.4044595432', '7404215579719728717495381644855454108.4044595432471476515397953', 47, 3);
    t('19852426184807.027576344702101724552129019696', '19852426184807.027576344702101724552129019695062566252', 44, 0);
    t('2905704090188470100000000000000000000000', '2905704090188470039042158369150981201257.0303862400231983807112836505571580', 17, 0);
    t('67145.086174132411917986527781879510033188', '67145.086174132411917986527781879510033188', 42, 3);
    t('957446950000000000000000000000', '957446951161450394570991330748.1602423077378909820', 8, 4);
    t('402.041316', '402.0413160368', 9, 6);
    t('795200000000000000', '795173982597277733.7585105133', 4, 4);
    t('16161300000000000000', '16161308645054711166.78175157', 6, 3);
    t('308937.10877869', '308937.10877869344160179047751121509', 14, 3);
    t('8', '8.0', 3, 1);
    t('936195605200000000000000000', '936195605152411892197739164.8118760424868582321310933173965467', 10, 6);
    t('98260014832417909.47831', '98260014832417909.47830413915267386984294586', 22, 0);
    t('83714.66', '83714.657884', 7, 6);
    t('326693305407136347033622.999515404376423338863420251024', '326693305407136347033622.999515404376423338863420251024', 54, 2);
    t('44822072696770000000000000', '44822072696761625822198860.49933088027626866643327', 13, 2);
    t('534.66479095539237038255', '534.66479095539237038255', 26, 3);
    t('70000', '75576.03046532', 1, 1);
    t('478648084475000000000000000000', '478648084475451187529299471222.0780416231695454861608', 12, 6);
    t('239817.133961446', '239817.133961446133800393', 15, 4);
    t('615389127079050800000', '615389127079050791538.61931153240', 16, 2);
    t('157539712318614480447170347000000000', '157539712318614480447170347298060975.29882315756022455', 27, 4);
    t('9681241653083.944457', '9681241653083.94445772297364178051213', 19, 1);
    t('891200000000000000000000000000', '891113357837202386790297471787.0292', 4, 0);
    t('8711595119260480364739714573000000000', '8711595119260480364739714572764587153.1082413098', 28, 4);
    t('8680536116440', '8680536116437.121048232013873990', 12, 6);
    t('6216974943.1062012582111629902059281272221', '6216974943.1062012582111629902059281272221', 44, 5);
    t('120000000000000', '127238637906283.625242168949830955130305039552035', 2, 1);
    t('575769436711567576.967228009422980551637', '575769436711567576.967228009422980551637248334', 39, 1);
    t('27212332844426010759231644926180567.5997463893128396707', '27212332844426010759231644926180567.5997463893128396707113870340', 54, 1);
    t('790229515300', '790229515330.8334489805813', 10, 5);
    t('83664754725145700000000000000', '83664754725145699752206570012.3828805101759708292322392', 17, 2);
    t('0.36686834', '0.36686834040925216168338920', 9, 3);
    t('15462351985416580713670665046565893.1', '15462351985416580713670665046565893.04130463284129781570938', 36, 0);
    t('6225842932.70590329', '6225842932.70590328718472', 18, 2);
    t('618782819774.67910214574198227731', '618782819774.679102145741982277306209288913', 32, 2);
    t('2697257538758.5181670505389987322', '2697257538758.51816705053899873221196323858582095', 32, 5);
    t('3260858520785471221048015898220625542.764488842585999593', '3260858520785471221048015898220625542.76448884258599959339554896314378', 55, 3);
    t('84996501942077137770114671411789705', '84996501942077137770114671411789705.053942561565669262525207876327', 36, 3);
    t('6156334379398459055.308924052575422', '6156334379398459055.308924052575422', 36, 5);
    t('25148.3786427947', '25148.3786427946502878986076729', 15, 2);
    t('3516630005410529866426.932746298815325649', '3516630005410529866426.932746298815325649', 43, 1);
    t('10846684480000000000000', '10846684479999098803138.428159213603667925349180345515', 10, 4);
    t('36195.6267155648960361', '36195.6267155648960361', 24, 2);
    t('94569000000000000000000', '94569751507641612143446.21', 5, 3);
    t('232008.149165051562704553137785899', '232008.14916505156270455313778589934', 33, 1);
    t('55655.5112051847876', '55655.51120518478760', 21, 4);
    t('465544536653789700342143.765', '465544536653789700342143.76461514957143675462528311799942826', 27, 2);
    t('300', '326.74079572838628493555039909', 1, 5);
    t('72192857000', '72192856781.10471', 8, 5);
    t('73622.41865283254169726', '73622.418652832541697251996227082018993420909', 22, 2);
    t('36369739', '36369739.001199792', 9, 5);
    t('226764103542744342244.9232351253601', '226764103542744342244.92323512536005263120139435253519058628', 34, 5);
    t('3565378588921630000000000000000', '3565378588921626500462886898346.35433105554719294735653907008343452386', 15, 4);
    t('5205580000000000000000000000000', '5205580388656383619432957105234.0194', 6, 1);
    t('3450385.882024', '3450385.88202483', 13, 3);
    t('6310462000000', '6310461280213.3486976008', 7, 0);
    t('4231904810682044742000000000', '4231904810682044741623253921.65691461632849162351151420923385', 19, 2);
    t('259000000000000000', '259065250383248928.841747760201959170452512588629', 3, 5);
    t('53.875032558', '53.875032558', 11, 4);
    t('9998619471000000000000000000000000000', '9998619471126426121333317408776324999.293490469046292596657696', 10, 1);
    t('2270177327140958324800362300000', '2270177327140958324800362300850.602673', 27, 3);
    t('36529719668000000000000000000000', '36529719667682176478109133960644.6814877318265577622423628', 11, 6);
    t('1560300', '1560278.1096573178760776274521860660310393837523', 5, 4);
    t('989351963981283529138904545953100000000', '989351963981283529138904545953131785899.06228447', 31, 5);
    t('108630000', '108630127.215644186276151516210488849525', 5, 4);
    t('2012218800000000000000000', '2012218772344053062981584.0', 8, 4);
    t('3118389569858.82', '3118389569858.821071401137798683267038', 15, 4);
    t('7590170902524000000000000000', '7590170902523334483503139270.806159504392659790437798326022', 13, 0);
    t('15500000000000000000', '15433682422959045052.0', 3, 2);
    t('64725588321287314.806512', '64725588321287314.806511142226153', 23, 0);
    t('8259399044060000000', '8259399044067797361.6021169660303663', 12, 1);
    t('7000', '6069.0', 1, 0);
    t('3.136304565312982421095492656099317', '3.136304565312982421095492656099317', 37, 2);
    t('3155.33026', '3155.33026', 12, 1);
    t('4525923217765595711158300000000', '4525923217765595711158260225527.181937558744', 23, 2);
    t('24106776571840532.29672416243203926699', '24106776571840532.2967241624320392669892412431605872660771', 37, 2);
    t('90528808554681227604423720', '90528808554681227604423724.42559358', 25, 3);
    t('500000000', '599088288.83', 1, 3);
    t('3838747860000000', '3838747851172305.1841', 9, 2);
    t('5126898196444083203000000', '5126898196444083202942240.3365271318145342114', 20, 2);
    t('2900000000000000000000000000', '2855779290615313361871754105.01803578356317', 2, 6);
    t('4.240891958', '4.240891958', 12, 6);
    t('56', '55.810586374272003439023759', 2, 2);
    t('590962280114100000000000000000', '590962280114060102067887623982.69120979918', 13, 4);
    t('234000000000', '234050375065.039', 3, 6);
    t('1351.64085746', '1351.64085746', 13, 3);
    t('568303105114986129226776068000000000', '568303105114986129226776067917842527.985242170663', 28, 0);
    t('3132721000000000000000000000000000', '3132720900441165585626878592575262.932', 7, 5);
    t('25108055963787.4894342', '25108055963787.489434258020164541351292496010049876227', 21, 1);
    t('5464625399334806029120885019298978000', '5464625399334806029120885019298978029.936259122894365', 34, 4);
    t('1427000000000000', '1427193948734531.94648247781437086', 4, 1);
    t('498562523337419240000', '498562523337419232178.03433979863498244635236414651', 17, 0);
    t('728417500000', '728417515464.47274', 7, 1);
    t('18672766746679482606370067644056.13125295016577', '18672766746679482606370067644056.1312529501657696443181244614', 47, 5);
    t('5754535706938.2853584418583', '5754535706938.2853584418583', 27, 0);
    t('400205001730519677447000000', '400205001730519677447312741.5732288465359167995138', 21, 5);
    t('780509189700000000000', '780509189627514667813.97400423216602991036628219188965714', 10, 0);
    t('4886527362106620', '4886527362106622.86', 15, 6);
    t('7159804906.13141273736998', '7159804906.13141273736998', 25, 6);
    t('545258.41', '545258.41102', 8, 5);
    t('900000000000000000000000000000', '876997274534795943437574735101.245', 1, 5);
    t('479151461000000', '479151460135311.6119898217', 9, 2);
    t('430286.868235', '430286.8682354', 12, 6);
    t('769.637089053947', '769.6370890539465007895457', 15, 5);
    t('43853842131848346968421789067399955028.82873654126736', '43853842131848346968421789067399955028.828736541267365247', 52, 3);
    t('7.5226713684941413401253482', '7.52267136849414134012534812268115526', 26, 0);
    t('65914188631741970000000000000000000', '65914188631741970160316532687676098.29895392659230529384408163318105887084', 17, 4);
    t('129488.24122328537861019', '129488.241223285378610189340', 23, 6);
    t('848772.9023588325714008', '848772.90235883257140086579400581210467052', 22, 3);
    t('378.49006335289006114750430269', '378.4900633528900611475043026938603005', 29, 5);
    t('200472549702447791910000000000000000', '200472549702447791911273250596560705.507432630855426260889724350', 20, 5);
    t('2529.66585122645', '2529.6658512264501364', 16, 1);
    t('848082275487822527020869800000000000000', '848082275487822527020869745492026105043.206242972793', 25, 0);
    t('14914147957000000000000000', '14914147957242174503072659.5953477475800', 11, 6);
    t('952313964000000000000000000', '952313964273891334415507049.167673392447682', 9, 5);
    t('35816416930000000000000', '35816416931819946624658.9134497620611441908857527', 10, 3);
    t('240700000', '240763618.90898300020613', 4, 1);
    t('75000000', '74566000.097919779526959481', 2, 0);
    t('3741176347883309065485226556.776245373', '3741176347883309065485226556.77624537219', 37, 0);
    t('15836160053809966.0781', '15836160053809966.0780628124105', 21, 4);
    t('10.619229048325303974', '10.6192290483253039740', 23, 3);
    t('25043555000000000000000000', '25043554812593538101246891.78743408', 8, 4);
    t('528409912489922583037589268.48', '528409912489922583037589268.4899802159', 29, 1);
    t('434276546238724700000000000000000000', '434276546238724652989461715711561206.9002439405488016364954313287352', 16, 2);
    t('6.1858917493914', '6.1858917493913967463450144041294208878972', 14, 6);
    t('41103044585674780000000000', '41103044585674774486552037.69635604244235256273', 16, 0);
    t('409901294100375.5', '409901294100375.5660086', 16, 1);
    t('33076844369073543400000', '33076844369073543386809.8', 18, 2);
    t('5806065.67477242372905716431075', '5806065.67477242372905716431075304636670305417', 30, 6);
    t('811.80220970283262936', '811.802209702832629369', 20, 3);
    t('397813442100000000000000', '397813442088746141260039.8323826563117846', 10, 0);
    t('5440000000000', '5436465279565.4857873374825346145964', 3, 4);
    t('16216425565026700376680878061.97961091195', '16216425565026700376680878061.9796109119588044645', 40, 1);
    t('8197138', '8197137.085688915102419808325180', 7, 0);
    t('1737925781000000000000000000000000000000', '1737925781837602783845632217807620908569.3', 10, 1);
    t('2905717560', '2905717550.5327945454331736810911044093381063647778', 9, 2);
    t('981510075835360000000000000000', '981510075835363834001033597389.14070471795917118', 14, 3);
    t('6438285990300.056341540800517375103', '6438285990300.0563415408005173751031', 34, 6);
    t('762923805603561774028351138357426500000', '762923805603561774028351138357426513381.616776875460808278230809342511309310', 34, 4);
    t('517621.958397153332764351', '517621.9583971533327643510129', 25, 1);
    t('972571263874506352440000000000', '972571263874506352440366976827.42533859522', 20, 4);
    t('708169501038196.077', '708169501038196.07702420720207453140466', 19, 4);
    t('524426372521245400000000000000', '524426372521245426021717598200.101905451851130496609016075562457391', 16, 1);
    t('12625649670065363729862746040023', '12625649670065363729862746040023.666561066780106946387573824145112767', 32, 1);
    t('3796.20820070733695477304', '3796.208200707336954773036198', 24, 2);
    t('660450000000000000000000000000', '660457078569482992688574922179.05014085009786792075291086215845679788', 5, 1);
    t('847727217525365804685101000000000000', '847727217525365804685100970503102531.0076704630583075898004405187', 24, 6);
    t('43813057394717221219268.9020769490083796695623445846006', '43813057394717221219268.90207694900837966956234458460068883364', 54, 1);
    t('4576145106332750054495.41579764377865648125556908295379008', '4576145106332750054495.41579764377865648125556908295379008', 59, 3);
    t('4213882111077115644195.488516866250055032959', '4213882111077115644195.488516866250055032959', 44, 1);
    t('81454681759106893526175502.04446607664781134075170192', '81454681759106893526175502.044466076647811340751701927207', 52, 1);
    t('5770740152452372126684410873258.1', '5770740152452372126684410873258.06', 32, 0);
    t('82323571831930500000', '82323571831930495890.81083176897539065573', 16, 5);
    t('1685823849274092393665970312244825736.3596633134756597227252430227', '1685823849274092393665970312244825736.35966331347565972272524302262138757', 65, 2);
    t('7842.0471901', '7842.0471901194998700033483974270281', 11, 6);
    t('0.37060198047904726202505282797362', '0.37060198047904726202505282797362', 32, 2);
    t('97693109165447512980', '97693109165447512986.7888380289826420533', 19, 3);
    t('123400332434615177504026528919130501066.63688957706038', '123400332434615177504026528919130501066.63688957706038', 55, 2);
    t('9678599.29545', '9678599.2954519211549970', 12, 6);
    t('8485431765056561600000000', '8485431765056561635944415.9', 17, 1);
    t('35000000000000000000000000', '35088914888270294396665983.49510987', 2, 6);
    t('575000', '574313.94409697241208030084011775492', 3, 0);
    t('5082532008193291404085924386.072327396969419409505099609635355764', '5082532008193291404085924386.07232739696941940950509960963535576479', 64, 3);
    t('89268827340000000000000000000000000000', '89268827333095114811740663211119682370.0', 10, 2);
    t('30220493379427539879', '30220493379427539879.0', 24, 2);
    t('70343952490000000000000000', '70343952489781432639487798.33855619486345432987267459786405506876', 11, 5);
    t('4426990000000', '4426993301937.871537212036423', 6, 1);
    t('864249625746247175503725.0461', '864249625746247175503725.046077', 28, 0);
    t('96597310836721610258963.34703476', '96597310836721610258963.347034764688', 31, 1);
    t('8109353456071206832445520828586799.6884596973', '8109353456071206832445520828586799.6884596973281', 44, 6);
    t('607203600', '607203598.4138537131906', 7, 2);
    t('5.66', '5.6589190083813', 3, 2);
    t('72053698828507440473672.3202131', '72053698828507440473672.32021305882247565849580', 30, 0);
    t('48926500000000000000000', '48926446213130500461517.2417595783786304292995', 6, 2);
    t('6300000', '6298589.171488012616545419293492782', 3, 0);
    t('70000000000000000', '72148881608830302.085866235184', 1, 5);
    t('268172.38057437316225574513252260135603', '268172.38057437316225574513252260135603320', 38, 3);
    t('1078940000000000000000000000000000000000', '1078941227888460410700901374672927710879.33411636639808170948074355264', 6, 5);
    t('9986241000', '9986241080.0785038914', 7, 6);
    t('887114.692535605787904518574364', '887114.6925356057879045185743642546', 30, 5);
    t('24274287457020922321768987.1635579744281', '24274287457020922321768987.163557974428004', 39, 0);
    t('589556300000000000000000000000000', '589556345995614117122972564148189.127282490919685159377779380', 7, 5);
    t('498968400000000000000000000', '498968443000691436075559933.177734194325819232422779', 7, 4);
    t('11634715384.9459923879626477', '11634715384.945992387962647674913', 27, 5);
    t('7293549121000000000000000000000000', '7293549120537133154319981569505745.270526732263484660626545254703326536661', 10, 2);
    t('85939411842248498983771298644057462173.1163272878714614372566763597993777784', '85939411842248498983771298644057462173.1163272878714614372566763597993777784', 77, 6);
    t('6559943483739155724920479.286948274210351325', '6559943483739155724920479.28694827421035132534206325', 43, 4);
    t('2200000', '2190601.01463943263', 2, 2);
    t('10076744686920573644121412850385722.3345431931479457567219', '10076744686920573644121412850385722.33454319314794575672181436542909366650', 57, 0);
    t('483577879.389253593526', '483577879.38925359352601917133159610635775', 21, 1);
    t('9103190413239584472156032602275251.619241024318', '9103190413239584472156032602275251.619241024317899', 46, 5);
    t('5807727902288847618914478617950', '5807727902288847618914478617951.236137', 30, 4);
    t('582417171221230929277919661684956', '582417171221230929277919661684956.0', 37, 3);
    t('3689567448623189554.37314055490227460911', '3689567448623189554.3731405549022746091156665492474366845', 39, 3);
    t('9172381963.589989194384', '9172381963.589989194383815737096', 22, 4);
    t('4000000000000000', '3696260271927727.22626119067147', 1, 0);
    t('58906115097160.44', '58906115097160.4391181', 16, 4);
    t('74933042075438834032823822873.9123925614687983269', '74933042075438834032823822873.91239256146879832690999479606153994', 48, 4);
    t('954651359246661767247902045662.68565', '954651359246661767247902045662.6856540190002002473', 35, 3);
    t('82962.66', '82962.655472367232000606454845645568914781205', 7, 2);
    t('70.8944725235405751448066708787', '70.89447252354057514480667087874490546', 30, 3);
    t('32434824796313272070303674.108484663432672792', '32434824796313272070303674.10848466343267279232232323594542835', 44, 1);
    t('312132907834676802899964238058944325689.377535626171346254048708', '312132907834676802899964238058944325689.37753562617134625404870787', 63, 2);
    t('908354647685977371463878.27237', '908354647685977371463878.27237303966556228903800766898511025', 29, 5);
    t('59596987942504279824000000000000000000', '59596987942504279823964467914340191563.185833739009', 21, 0);
    t('525300748767065780.788012121989', '525300748767065780.788012121988773728169251621302093728661', 30, 2);
    t('618136088048293404709806938973322437325.059133633776', '618136088048293404709806938973322437325.05913363377665219085398', 51, 1);
    t('1.1685720994', '1.16857209933852341477', 11, 2);
    t('2.56667', '2.56666500865236739443326655206622287992', 6, 6);
    t('769619248.96051739', '769619248.960517397387219647892053340746', 17, 3);
    t('890580781950762261000000000000000000', '890580781950762261096662480122660923.298884327619', 18, 4);
    t('69140000000000000000000000000000', '69145768397012772924268366909543.5885860779109134323638192', 4, 3);
    t('42114.923057774', '42114.923057774', 17, 0);
    t('1316290.02', '1316290.012651', 9, 2);
    t('1786013302443696020', '1786013302443696016.6757901809', 18, 5);
    t('7681210938614590659483195587827544747896.842', '7681210938614590659483195587827544747896.842055886061888201913972461392913', 43, 1);
    t('4674281298410087824949975548024038679000.1320216349', '4674281298410087824949975548024038679000.1320216349458', 50, 5);
    t('8669465624347275754909868573138897075971.039365198', '8669465624347275754909868573138897075971.03936519780720918864023', 49, 2);
    t('5820239796.75013837611668408', '5820239796.750138376116684079501', 27, 0);
    t('8', '7.9', 1, 6);
    t('64630000000000000', '64625536312023284.637858817791438284540418188843111350', 4, 2);
    t('616060000000000000000000000000000000', '616055615318745236862074991311292945.9290604397847950407597565021968954359', 5, 4);
    t('475122800830120000000000000000000000000', '475122800830120605829354486082189376424.9', 15, 1);
    t('866662773196822503228261515741504947891.338', '866662773196822503228261515741504947891.3377543288775400145538', 42, 0);
    t('377999144522.0380572350446832032453', '377999144522.03805723504468320324520072551', 34, 0);
    t('691542908013991521077481574838949.944715457057', '691542908013991521077481574838949.94471545705761', 45, 3);
    t('2838000000000000000000000000000000000000', '2838653681328833354849317127779033599689.700', 4, 1);
    t('867727079669000', '867727079669493.408812597438029231707257', 12, 1);
    t('9578377974506390860209593158213782078.80142385979464065493592797', '9578377974506390860209593158213782078.80142385979464065493592797032498527845', 63, 6);
    t('667396740529687738822920000', '667396740529687738822916638.75184', 23, 2);
    t('58000000', '57325179.1', 2, 0);
    t('887701835842251844613727971270000000000', '887701835842251844613727971267600638239.01672009943208', 29, 6);
    t('1218269763500000000000000000000000000', '1218269763467093194518067002597306126.91420211537204131835106249503296', 11, 6);
    t('89525298985989691388000', '89525298985989691387929.145260634', 20, 5);
    t('452408788012870006169809627700', '452408788012870006169809627609.215350138983244', 28, 0);
    t('4484883.566669664', '4484883.56666966348711892744156163270', 16, 0);
    t('72.371826856733960335', '72.371826856733960334520746638059990746', 20, 5);
    t('37724373061668655528.522851881285371006409996400788516', '37724373061668655528.522851881285371006409996400788516', 53, 0);
    t('474942224005447175877435461399', '474942224005447175877435461399.0', 32, 1);
    t('872429', '872429.0', 7, 6);
    t('369342159139978456.7827401', '369342159139978456.78274010092287656088709', 26, 5);
    t('83507900000000000000000000000000000', '83507841652414814789715079585462831.8', 6, 2);
    t('744821700000', '744821686907.0', 7, 0);
    t('696924250338500000000000000000000000', '696924250338519899171875048780168296.2522472', 13, 5);
    t('693896881251858774322930586.9', '693896881251858774322930586.8478166830', 28, 0);
    t('98801352047386853139064.50372992043423', '98801352047386853139064.503729920434224422673788529313387', 37, 0);
    t('87977.0670583496674439896', '87977.0670583496674439895874799', 24, 2);
    t('4270000000000000000000000000000000', '4268654792697796076500965336525867.268', 3, 5);
    t('7961429474800000000000000000000000000000', '7961429474742168821189890058057653847049.4654544026148251670855145828048039', 11, 2);
    t('688921500760211155121.136993578475066967967684095429', '688921500760211155121.13699357847506696796768409542932218', 51, 6);
    t('378038845685474358917525251567100', '378038845685474358917525251567054.77279202804690', 31, 0);
    t('124472317581161105332082', '124472317581161105332081.951883864444192470848374185', 24, 5);
    t('52047079582326.203978142209410439510125693140924566', '52047079582326.2039781422094104395101256931409245661023', 50, 5);
    t('5480175342.3044834146637', '5480175342.30448341466363979707154', 23, 0);
    t('242160616794712120000000000000000000000', '242160616794712121052295929234664154044.580445136384750289188804299', 17, 1);
    t('53340543879195600000000000', '53340543879195589981988304.0', 15, 2);
    t('5.4', '5.39802', 3, 4);
    t('913649117905360876750660554055837183.793464479873109806215188591', '913649117905360876750660554055837183.7934644798731098062151885903157165596', 63, 2);
    t('1647000000000000000000000000000000000000', '1647039777336360825296663093644417842936.08227923213404071', 4, 1);
    t('690000', '690146.20050049881810', 2, 6);
    t('55137290133112205980861452575.89406004966957303701', '55137290133112205980861452575.8940600496695730370133357990757260017819', 49, 1);
    t('54089318891874054814154540323869932.1444038195451832595', '54089318891874054814154540323869932.1444038195451832594427', 54, 2);
    t('7371777.159', '7371777.15958616908590091898812', 10, 3);
    t('67023620696530729204373.42296484243', '67023620696530729204373.422964842438044404107404', 34, 3);
    t('72770145929070000000000000000', '72770145929075467964052813951.0', 13, 1);
    t('47115075545610306861930000000000000', '47115075545610306861930206127810465.19323262997238803980257', 23, 1);
    t('162950041193660954583061699278000', '162950041193660954583061699278041.3891705034', 31, 3);
    t('13134079930583942212118400000000000000', '13134079930583942212118421888508654778.2452139529353014167525291651', 24, 5);
    t('62257071362673885780000000000', '62257071362673885776675038884.0403465', 19, 2);
    t('198577580835904889000000000000000000', '198577580835904889014712515426116496.9370822425549198054', 19, 1);
    t('238572.365077', '238572.365077', 14, 3);
    t('6808848537970000000000', '6808848537973757514648.660', 12, 1);
    t('35286574000000000000000000000000000', '35286574008987483287897143202863290.61065', 8, 1);
    t('441880000000000000000000000000000000', '441885439271191809521298031564159629.363120483750', 5, 3);
    t('6748329119.21796659945406633191398188522', '6748329119.2179665994540663319139818852104295507', 39, 2);
    t('49691040576799655882', '49691040576799655882.624938678558054781891162457374592368', 20, 3);
    t('844634.09213439341707034', '844634.0921343934170703315907333070498', 23, 0);
    t('4264363655521037267000', '4264363655521037266750.626998959226026351', 19, 6);
    t('1535331516185209616470642371922806509.84014', '1535331516185209616470642371922806509.8401400808529087972607582281283180755', 44, 3);
    t('2859355391542261216429.0356039863', '2859355391542261216429.0356039863', 34, 1);
    t('8585843755385298439647069375942814147.78336947456434869716467392249', '8585843755385298439647069375942814147.783369474564348697164673922485751', 66, 6);
    t('6968012841029122544125858741688222878.6', '6968012841029122544125858741688222878.554077152525940', 38, 5);
    t('38534.07705034281819300939', '38534.0770503428181930093857553', 25, 6);
    t('30138024047900000', '30138024047956060.282', 12, 3);
    t('46239612142228.02902752077062', '46239612142228.02902752077062980259434016', 28, 1);
    t('56398552949396865998003.81997503353067939', '56398552949396865998003.81997503353067939', 41, 1);
    t('629424172364889697000000000000000', '629424172364889697886263825159256.0', 18, 1);
    t('5402771309840056613806182353543.31934879', '5402771309840056613806182353543.31934879', 41, 2);
    t('62828024378170534281888808292129.503164210810654208', '62828024378170534281888808292129.50316421081065420800888517963276583478', 50, 6);
    t('754.935304590900697', '754.935304590900697', 18, 6);
    t('6133060000000000000000000000', '6133062773995517736094918951.8536342695859286254989071233319263733', 6, 6);
    t('13911001551961706866283.63498117145', '13911001551961706866283.63498117145856817273150480', 34, 3);
    t('293964043884820566.625624464866132', '293964043884820566.625624464866132352791', 33, 4);
    t('5334232018687447000000000000000', '5334232018687446344769341157923.035438870654327744228704253', 16, 2);
    t('32887507.783047569', '32887507.78304756895', 17, 5);
    t('4283638.379182', '4283638.379182504881701800249', 13, 3);
    t('2560340699987845316700000', '2560340699987845316724483.03518357638154153002742', 20, 5);
    t('54735169687449208901740910477101.293291', '54735169687449208901740910477101.2932910006772408783324027555', 38, 5);
    t('4479631988526670000000', '4479631988526662957486.521705724671002675613586901065676171781', 15, 0);
    t('1.063172112925', '1.06317211292514518921430670854', 13, 1);
    t('95270651296238694449762132400576034995.39152964604962834', '95270651296238694449762132400576034995.391529646049628340', 56, 5);
    t('7606300', '7606340.835359183357550106843855928', 5, 6);
    t('462869998143908933517806568025500000', '462869998143908933517806568025506546.6242454', 31, 1);
    t('76921726000000000000000000000', '76921726012538252011694146545.8259', 9, 1);
    t('193753314017770250920901340100000', '193753314017770250920901340080995.77367348019423256588994770626225', 28, 4);
    t('75000', '74442.48700426663320717', 2, 0);
    t('8', '8.0', 4, 1);
    t('20991000000000000000000000', '20990801791887831408182682.19510', 5, 0);
    t('20516365094136092.59095923912194408436381575', '20516365094136092.59095923912194408436381575187630132', 43, 4);
    t('959752895627154820202375593771203', '959752895627154820202375593771203.319588378524484183400316651891710440831', 33, 6);
    t('94901464260000000000000000', '94901464258358135088154837.707933091', 10, 2);
    t('786900000', '786873323.931599247648805226108882908403478593', 4, 4);
    t('899692910225000000000000000000000000000', '899692910225927595312309220960343154867.13432258482281821272', 12, 1);
    t('19847530356670', '19847530356667.177760594653889273904619351058', 13, 5);
    t('7242709061971400000', '7242709061971340721.18868359517733726455179', 14, 0);
    t('3.27810984232723009082902328', '3.278109842327230090829023276166908', 27, 0);
    t('266479615170000000000000', '266479615169762824833230.6002725532860735', 11, 5);
    t('85828028.965492341902', '85828028.965492341902', 21, 6);
    t('925360000', '925361181.04428702860397744040257706812319540', 5, 5);
    t('1956.534575158043', '1956.534575158043', 19, 3);
    t('227594', '227594.0', 9, 6);
    t('33856602652000', '33856602652540.393994532', 11, 3);
    t('597458875043971.77727976333102885184921267359', '597458875043971.7772797633310288518492126735837600', 44, 2);
    t('71805768568330479.5526982826316550195', '71805768568330479.552698282631655019495496096404986', 36, 6);
    t('7160677537275145748479483902505.6425453847', '7160677537275145748479483902505.64254538474874097101895058', 41, 4);
    t('4855000000000', '4855853756013.1567151880531117420', 4, 3);
    t('5230000000000000000000', '5233199585833280950443.04307887', 3, 6);
    t('95686.65541471264944719862515', '95686.6554147126494471986251507', 29, 1);
    t('431994172667230888256451162880.703331204990617927003', '431994172667230888256451162880.7033312049906179270029478572', 51, 4);
    t('96693244310617444931831.77868951307152658352416289465', '96693244310617444931831.77868951307152658352416289465', 52, 1);
    t('22291671900000000000000000000000000', '22291671828746155930592578048104022.4385170', 9, 0);
    t('9543910417522777470508395.7585390106596959072', '9543910417522777470508395.7585390106596959072176924', 44, 6);
    t('278970420.3405', '278970420.3404135106872', 13, 0);
    t('150113494630000', '150113494630660.912', 11, 1);
    t('14667191464362418360141393347446600.47239105700679515', '14667191464362418360141393347446600.4723910570067951500', 53, 2);
    t('5734349508800000000000', '5734349508738410407748.4504441', 11, 2);
    t('9640030029307321205384233266581.22', '9640030029307321205384233266581.22316545491402424', 33, 6);
    t('183922747802512353322000000000000', '183922747802512353321542646355786.85645510716', 21, 2);
    t('338390.02427434046934405601410207133899', '338390.0242743404693440560141020713389897310185', 38, 5);
    t('391775472410571.575406292759927706484955135973', '391775472410571.57540629275992770648495513597343', 45, 1);
    t('85.5', '85.5', 5, 3);
    t('791351616057891.04724', '791351616057891.047231625880869132350535592817237051', 20, 0);
    t('2537844906033592695100000000000000000', '2537844906033592695012357965777637659.2', 20, 2);
    t('48162280000000000000000000000000', '48162284523438168731884478429943.8255883', 7, 6);
    t('29028316014700000', '29028316014706286.401150447526638664094708892816980', 12, 5);
    t('57579530437144010379781699153.117161942077829931804656384447610737', '57579530437144010379781699153.117161942077829931804656384447610737', 68, 6);
    t('6208009246836500000000000000000000000000', '6208009246836524817654532689586607707642.580902126401841764', 14, 5);
    t('87160000000000000000', '87153996745281788154.82392803', 4, 0);
    t('5483253585635193642020450092405667656077.7865334811447', '5483253585635193642020450092405667656077.7865334811446989749995660452364352865', 53, 2);
    t('808320532', '808320532.0', 12, 5);
    t('62987475743752000000000', '62987475743751310696393.212771346860', 14, 0);
    t('9693086525.5782677312864', '9693086525.5782677312864126879', 23, 6);
    t('860706983194333.28240210884472011', '860706983194333.282402108844720107232905443567310784', 32, 4);
    t('341138943249445173460903290142449807.41602354995365', '341138943249445173460903290142449807.4160235499536524033565', 50, 6);
    t('7991888000000', '7991887986360.325968837', 7, 5);
    t('390900.902285227', '390900.902285227346', 15, 6);
    t('5054423130', '5054423126.4761', 9, 2);
    t('27853933346128606850624200', '27853933346128606850624156.70134344438655', 24, 5);
    t('22759177208997076467261073138282946.0613', '22759177208997076467261073138282946.06135439074', 39, 1);
    t('393880856000000000000000000000000000000', '393880855579166647694466270625354608729.10738830447006748471705154', 9, 0);
    t('26690773796.097149285574698997', '26690773796.097149285574698997', 29, 6);
    t('66482002417800651170000000000', '66482002417800651168793960643.61956786498419204996152944925601', 19, 0);
    t('211570846421684764337.813085391136', '211570846421684764337.813085391136', 35, 1);
    t('857566090483580485790776400000000000', '857566090483580485790776351543257692.3196414500757484585417534531823471', 25, 6);
    t('87130000000000000000000000000000000000', '87127110869459234739026073471097514335.128091', 4, 6);
    t('821774754528203371598175063931231.41278926868479587', '821774754528203371598175063931231.4127892686847958699521104101', 50, 5);
    t('636923361880.2', '636923361880.1646871584678008902540715', 13, 5);
    t('43439438147428148707.8296308211519540742008533615', '43439438147428148707.8296308211519540742008533615', 51, 1);
    t('77704224394928160709944206000000000', '77704224394928160709944206070460268.80786921123573000267', 27, 1);
    t('28724.6728810735838260354471659141260583', '28724.672881073583826035447165914126058303', 40, 6);
    t('7286571095242099630815261347014349527461.9185161', '7286571095242099630815261347014349527461.91851603599833113738922', 47, 2);
    t('13372978538967000000000000000', '13372978538967185289109010611.117771', 14, 1);
    t('952000000', '952000455.309', 6, 3);
    t('273154266802184100', '273154266802184104.15', 16, 3);
    t('5900000000000000000000000', '5851134682218860650702715.53880325223763908939', 2, 2);
    t('5726451235842001150000', '5726451235842001156148.9057', 18, 3);
    t('7741.1797093071', '7741.1797093071021050151', 15, 4);
    t('3089114652722941829684578736836.25882', '3089114652722941829684578736836.2588226691873816852567634307168', 36, 1);
    t('757093794762074148.881125', '757093794762074148.8811251366600094138702940649831', 24, 5);
    t('9817674220900000000000000000000000000000', '9817674220943697442771205505149604428559.9522755851', 11, 6);
    t('788295100000000000000000000000', '788295116630101027582223129230.429626054031991882', 7, 3);
    t('205110160662331785000000000000', '205110160662331784611083231293.6637', 18, 4);
    t('984814055163000000000000', '984814055163145068306713.964489259', 12, 4);
    t('27263238599057306', '27263238599057306.813932921303820875', 17, 3);
    t('60122109202435735897.172872', '60122109202435735897.1728718124695137482155450573594321234', 26, 2);
    t('245925812614134950000000000000000000000', '245925812614134950890548170456095489684.755424084123227', 17, 5);
    t('69203540839731000000000000000000000000', '69203540839730148982634330618192075256.8671222202264', 14, 0);
    t('45826604894127696636.8781', '45826604894127696636.8780816357545969796735', 24, 2);
    t('957539712021709434784.71826177196635', '957539712021709434784.718261771966358052', 35, 1);
    t('460372.7310188272160592340554952849857', '460372.731018827216059234055495284985786', 37, 3);
    t('583818192156542869789600', '583818192156542869789525.91006671', 22, 0);
    t('51756895398611313.30333937378894', '51756895398611313.30333937378894114', 31, 4);
    t('7520002699454532058129597019861845.2', '7520002699454532058129597019861845.22575310', 35, 5);
    t('928169325.5888389488748042477434257', '928169325.5888389488748042477434256932', 34, 5);
    t('660498807710585087402490729151.671293692886843921187', '660498807710585087402490729151.67129369288684392118602', 51, 0);
    t('6606529705254822021529800767.230976703873616266', '6606529705254822021529800767.230976703873616266320314', 46, 1);
    t('924891000000', '924891492274.6751273', 6, 1);
    t('270185085178345155490000000000000', '270185085178345155497068535875539.48342937083075206435434320879440159339', 20, 3);
    t('8320185624210000000000000000000', '8320185624214886244520893948035.378193905824711', 12, 6);
    t('2294521214543263587401457276781260000000', '2294521214543263587401457276781261508232.0862539215816209', 33, 1);
    t('4464366800766545.041408492611161705419184975', '4464366800766545.04140849261116170541918497520115', 43, 3);
    t('30.3001', '30.30005434698978503058', 6, 0);
    t('195646389238499.5004347', '195646389238499.5004347822519683', 22, 1);
    t('5.764', '5.764', 7, 0);
    t('8691554261552089017160', '8691554261552089017157.111944453991523903063693067359067921989', 21, 0);
    t('9724851685423325164286210490823631587918.87369504556595324914716', '9724851685423325164286210490823631587918.8736950455659532491471505496671865929', 63, 2);
    t('8000000000000000000', '8213513049787478173.209184448976720026788161102629404544', 1, 5);
    t('60000000000000000000000000000000000', '57237223651221069944574385896707286.7559852196299143710', 1, 4);
    t('260766195.2629905491', '260766195.2629905490103396748643338335674111738897', 19, 0);
    t('15970198116959321317557734022600', '15970198116959321317557734022612.38965113815271', 30, 1);
    t('26087872914645871.2595073962517460184101035344', '26087872914645871.2595073962517460184101035343849', 45, 2);
    t('4116300', '4116309.5757288', 5, 4);
    t('8375414600000000000000', '8375414584615355633037.5904491', 8, 6);
    t('6943431931822129470899358332497266658.28831', '6943431931822129470899358332497266658.28831', 42, 3);
    t('60274340000', '60274341879.1660', 7, 3);
    t('1869479000000000', '1869479680449319.7018', 7, 1);
    t('414543513393299474.152056386467191492131227787', '414543513393299474.152056386467191492131227787', 46, 2);
    t('4529063725717629884022858399347524470', '4529063725717629884022858399347524471.4304773706632110491716', 36, 5);
    t('232599393606558906168669770000000', '232599393606558906168669769527936.2359791162420782706595282324', 26, 2);
    t('732967236397394682313556.5359009900186', '732967236397394682313556.53590099001859630990954635290729267792', 37, 5);
    t('6233.3740991428529634648148650396724612', '6233.37409914285296346481486503967246118', 38, 2);
    t('14206383609638837.70098929917', '14206383609638837.70098929917311777690018437', 28, 3);
    t('2080697560108.137394', '2080697560108.1373933345639', 19, 2);
    t('700000000000000', '693399186451332.45708', 2, 2);
    t('227.35721899812120726034364666642', '227.3572189981212072603436466664221745', 32, 3);
    t('450883041.52', '450883041.515722726113', 11, 2);
    t('9000000000000000000000', '9260630031761440021146.97856839011796057437', 1, 4);
    t('55113237259.858472160386897125284770427592', '55113237259.8584721603868971252847704275921861368111', 41, 4);
    t('995648088', '995648088.09228442333626907385272', 9, 6);
    t('426395270855779004743730243936920099150.02738', '426395270855779004743730243936920099150.02738', 45, 5);
    t('8021817736059.4629999907', '8021817736059.462999990774719713398726680570584147', 23, 3);
    t('8930758810059288498924147125.1', '8930758810059288498924147125.13625922', 29, 5);
    t('413.866191936778389671935', '413.866191936778389671934630738674', 24, 6);
    t('5854717848.69496941', '5854717848.69496941', 18, 1);
    t('890459534829724.7504934205833', '890459534829724.7504934205833', 28, 5);
    t('49547910111794822227716773504.6045596326249999690550592', '49547910111794822227716773504.60455963262499996905505915514309916', 54, 2);
    t('75930400000000000000000000', '75930377428752456303491444.5', 6, 2);
    t('758528.7', '758528.74638225611290654', 7, 3);
    t('25805135999169062499047.64544702873', '25805135999169062499047.645447028724375', 34, 0);
    t('478.04428342', '478.044283420', 13, 2);
    t('837835542450381.16894103417673343838576346', '837835542450381.1689410341767334383857634596880838335510', 41, 4);
    t('100000000000000', '98349300300020.82585446169224160334602060186771901', 1, 5);
    t('296246770000000000000', '296246762068682606616.95', 8, 2);
    t('663844775984843940000000000000', '663844775984843936942337949269.95094911', 17, 5);
    t('523199161892000', '523199161892127.56471634', 12, 3);
    t('36945514870547620000000000000', '36945514870547624996837737024.32', 16, 6);
    t('78030000000', '78024035821.11385', 4, 2);
    t('31676476734586848000686596374385.8184952079175199817', '31676476734586848000686596374385.81849520791751998167290', 51, 5);
    t('83597468724725060991042107.7592339086', '83597468724725060991042107.75923390863688655398108612417', 36, 4);
    t('57860000', '57855620.769655694722841', 4, 0);
    t('2914433118961179567495600000000', '2914433118961179567495595075443.981096', 23, 4);
    t('494326270052.829859713064037823454968', '494326270052.82985971306403782345496848365238', 36, 1);
    t('7536962781743976150120.790093912', '7536962781743976150120.790093912', 32, 3);
    t('51418458713356334375283072.0166459833', '51418458713356334375283072.01664598328674', 36, 2);
    t('1236447853929698060033.94975524163', '1236447853929698060033.94975524162652010946251269407736662', 33, 5);
    t('4.54276888910562', '4.54276888910562', 17, 5);
    t('2746557151963700000000000000000', '2746557151963738900992727724083.50859546', 14, 4);
    t('389.068240286124052078', '389.068240286124052077218', 21, 2);
    t('553804035535829984367568747.8136620033309839931510813672', '553804035535829984367568747.81366200333098399315108136724259979', 55, 3);
    t('302.97767985105', '302.9776798510548448473', 14, 3);
    t('95672006216078759873532428572.1', '95672006216078759873532428572.10', 31, 6);
    t('947656375294082560000000', '947656375294082557739125.854482065735001724882655422', 17, 4);
    t('14681771115505208299358000000000000', '14681771115505208299357847216412362.89', 23, 4);
    t('80262919657407560194393868167979923210', '80262919657407560194393868167979923214.8632676286414', 37, 1);
    t('215797107985768571496.901994105', '215797107985768571496.901994104531882607', 30, 0);
    t('824856867.043250658634604', '824856867.0432506586346040757726275950', 24, 1);
    t('0.159', '0.1590', 6, 3);
    t('1692007344200962018046321704752947', '1692007344200962018046321704752947.188659951595', 34, 3);
    t('37381317.77292170139250796606', '37381317.7729217013925079660619851211111', 28, 6);
    t('973928103000000000000000', '973928102748004057190539.31678603693141930316724055010169', 9, 6);
    t('9587235113.991218739599396129', '9587235113.991218739599396128911', 28, 0);
    t('312640.06', '312640.06', 8, 3);
    t('1413431463.83424', '1413431463.83424349969009699242996820235057774446', 15, 6);
    t('11222735091243931.586066', '11222735091243931.5860656147674', 23, 0);
    t('421294449560', '421294449560.953955214340821643979199673', 12, 1);
    t('761932771542900696079049639176547', '761932771542900696079049639176547.8223218482645054', 33, 1);
    t('221489078232555651247382002203551828272.669', '221489078232555651247382002203551828272.6687861099', 42, 6);
    t('6355792406.77671366710690805600754294545', '6355792406.77671366710690805600754294544520600497', 39, 2);
    t('8707061932000', '8707061931961.7724453324724741007962920309', 11, 5);
    t('650912617730696448489775594488812982.59435128', '650912617730696448489775594488812982.594351280942898556243476319775', 44, 1);
    t('5051417260168300000000000', '5051417260168333680614084.356589087700204585192390546528412425', 14, 6);
    t('21332466059315.806184', '21332466059315.806183545962336965', 20, 5);
    t('9800000000000000000000000000000000', '9727072640311026743540100977567037.282578353398', 2, 2);
    t('18586344746368984973641830.08172', '18586344746368984973641830.081714184', 31, 0);
    t('497300000', '497254121.69475777665334717859957048398', 4, 5);
    t('1269029640000000000000000000000000', '1269029637518155337486505991416927.79', 9, 0);
    t('91619200000000000000000000', '91619223062239517684737476.9', 6, 6);
    t('6942832836313450996280.106269388265912454372893', '6942832836313450996280.106269388265912454372893', 48, 4);
    t('907240000000000000000000000000000', '907234276107550734570355473724567.7771717533818529546', 5, 2);
    t('6026037139815164610209755.76661049', '6026037139815164610209755.766610490', 35, 0);
    t('15229335828000', '15229335828378.97441093862439908204011174136811600787', 11, 1);
    t('766790225235645269076750133990783871226.088457358606453636802', '766790225235645269076750133990783871226.0884573586064536368023699', 60, 4);
    t('7073538446598605255597848413959.0997144280523', '7073538446598605255597848413959.099714428052293225201134539498135047', 44, 5);
    t('229745089966261800000000000000000000', '229745089966261829426834383834927857.827277678470743134420106732692181478980', 16, 4);
    t('38976041950139021201742890000000', '38976041950139021201742890573332.284358062834031', 25, 5);
    t('2458640000000', '2458638513010.8690685589335687689500517801', 6, 0);
    t('19.7594114778674067', '19.75941147786740666353386895227037348', 18, 4);
    t('119579031346977990396.924937685588460464846982', '119579031346977990396.9249376855884604648469817368692427345', 45, 5);
    t('2807970476394397321851546000000', '2807970476394397321851546397150.66401145', 25, 5);
    t('26960000000000000000000000000000', '26962615420336826610024470818431.39548575779439819810582465164530', 4, 1);
    t('7967186234.8', '7967186234.8', 14, 6);
    t('6026481.00423096', '6026481.00423095474222880882505180588190044', 15, 0);
    t('8591221638139587735154215488876500000000', '8591221638139587735154215488876587931454.5239215595157770941438740', 32, 1);
    t('666360108354737750445548.851816452644168548', '666360108354737750445548.85181645264416854803', 42, 6);
    t('9461364652620000000000', '9461364652623408343334.491079115076751870452985', 12, 5);
    t('5338894.185426121105', '5338894.18542612110420628120489389', 19, 0);
    t('942350859784865000000000000000000000', '942350859784864048956272400599573691.912150138', 15, 0);
    t('2094314136812655960725.47', '2094314136812655960725.4702388105846300', 25, 3);
    t('340000000000', '336575786673.427417669348811487986375573', 2, 2);
    t('7410736111858990.1585583739520837770240563172287', '7410736111858990.1585583739520837770240563172286659', 47, 4);
    t('10759006977407826252445826764280.834566343527183760979', '10759006977407826252445826764280.8345663435271837609793', 53, 3);
    t('5486202528598888938673520.122292213788281253', '5486202528598888938673520.12229221378828125305', 43, 3);
    t('8726447375978095040000000000000000000000', '8726447375978095045433494479490364838258.0', 18, 1);
    t('253360724804798949943324472200000000', '253360724804798949943324472154202667.85444515103886330588202591975076718', 28, 5);
    t('599.49413924224451433896228691293358154', '599.494139242244514338962286912933581549038', 38, 3);
    t('928526302049056172393890000000', '928526302049056172393880222530.289053343635715753597328', 23, 0);
    t('761635.0556501299355341510648', '761635.055650129935534151064762446', 28, 5);
    t('522681243131238734567893311.582715740618122250885', '522681243131238734567893311.5827157406181222508845839992', 48, 4);
    t('328897615495547.34324064', '328897615495547.34324063694', 23, 2);
    t('10000000000000000000000000000000000', '9490061846018224477212264521392679.4312', 1, 0);
    t('74.742228752635', '74.742228752635', 17, 2);
    t('1572700000000000000000000000', '1572739601726465512243037569.5419321', 5, 4);
    t('782913138634192669.9536816116749132', '782913138634192669.9536816116749132248198068277525462915', 34, 4);
    t('970.2845', '970.2845', 8, 4);
    t('81332400000000000000000000000', '81332370609107142848428280684.7653630', 6, 6);
    t('3.800350909094177787375499445982', '3.800350909094177787375499445982480', 31, 6);
    t('3572975614961206000000000000000000000', '3572975614961205555297968595936749897.7880441681', 16, 4);
    t('71853356971775189843972060.5808758327563940677478712', '71853356971775189843972060.58087583275639406774787126698140343540', 51, 1);
    t('5856207000000000', '5856206911088814.127809592056646', 7, 5);
    t('533471346771917987021067.58081411', '533471346771917987021067.5808141175913190', 32, 3);
    t('2232184.43810102609238410121202203', '2232184.438101026092384101212022026680', 33, 6);
    t('294874446228946035108.9437275133776804796438182', '294874446228946035108.9437275133776804796438181925912808', 46, 4);
    t('21503463.2', '21503463.13801746818097935238687426101787042697', 9, 2);
    t('6485775134821149037681203046958164517.3999199803190762', '6485775134821149037681203046958164517.399919980319076262055', 53, 1);
    t('860000000000000000000000000000000000', '861062954191618102394422960564696286.7', 2, 4);
    t('386386806478091700703050000000000000', '386386806478091700703046872367215859.899249051605763160380549371', 23, 0);
    t('2684927425720.7', '2684927425720.73201947', 14, 6);
    t('25291822519688600000000000000000000', '25291822519688560978246714672127455.217', 15, 4);
    t('911379467709849597710748486.3197445194706235973249118', '911379467709849597710748486.3197445194706235973249118', 55, 0);
    t('430.2285291652567697407552971675511', '430.228529165256769740755297167551099779499', 34, 0);
    t('75444.43847113', '75444.43847113532662', 13, 1);
    t('3013.5665', '3013.566470946794', 8, 6);
    t('17263865663241441011822485661576323000', '17263865663241441011822485661576322484.93696874985638379394047485183', 35, 2);
    t('2890670', '2890665.20910', 6, 4);
    t('1181642314342598.148443371', '1181642314342598.148443371', 25, 0);
    t('5790389037299187454847757487508.92', '5790389037299187454847757487508.917881629', 33, 2);
    t('800000000', '804192487.740400348197324700607911329822243508', 1, 3);
    t('1.52863320011588', '1.52863320011588585526419762', 15, 3);
    t('573557651538558996127.44699526693329', '573557651538558996127.446995266933290397', 35, 3);
    t('73817814287003937311.1959320781', '73817814287003937311.19593207808342246423552', 30, 5);
    t('3701463295568929824739007226.640686', '3701463295568929824739007226.6406863867497233444188', 34, 1);
    t('52336.4219236047', '52336.421923604699776', 16, 2);
    t('4283416045187205723428626170280.8', '4283416045187205723428626170280.803477795579500512117599', 33, 5);
    t('1726000000000000000', '1726871970075684100.775015214388253784263697789828500033', 4, 1);
    t('3690562719027059489169073316597.52025056', '3690562719027059489169073316597.520250567390285517', 39, 1);
    t('7194344574573551807481094419918246.963210740390201711437031', '7194344574573551807481094419918246.96321074039020171143703063190', 58, 6);
    t('929606568787826337350838653555.542693367425749690545856454884675', '929606568787826337350838653555.542693367425749690545856454884675', 64, 0);
    t('5500695645843.08', '5500695645843.08182526350', 15, 6);
    t('6810000', '6800993.3286207', 3, 0);
    t('160427101000000000000000000000000', '160427100802362993077546304481326.448171', 9, 6);
    t('9694541400000000000000000000000000', '9694541359755774037898496151460066.0', 8, 4);
    t('12347827930964002533032728905130.8852468655922', '12347827930964002533032728905130.8852468655922', 47, 2);
    t('51185606951173492081000000000000', '51185606951173492080481977310741.612817168885601037887500', 20, 2);
    t('1934916700000000000000000000000000', '1934916766430176746997609821009552.192350537503143717336655024789596487', 8, 3);
    t('15064366990436494742032178067397.738845346346155165558674', '15064366990436494742032178067397.73884534634615516555867449307042', 56, 4);
    t('31600000000000000000000000000000000', '31602205141702518372333595924890492.58712', 3, 5);
    t('3235822955208743983544444000', '3235822955208743983544444576.21', 25, 3);
    t('4290000000', '4291687365.6630767833', 3, 6);
    t('1605549265548.84444718', '1605549265548.84444718128116157', 21, 1);
    t('111454102646347347594709834619.045825668454299238056', '111454102646347347594709834619.0458256684542992380560418', 51, 3);
    t('72643105351264550408320717659804503.415812500959', '72643105351264550408320717659804503.4158125009585633653163345684', 47, 2);
    t('5040000000000000', '5045403963254669.435', 3, 3);
    t('1682602746627237431505543.89572587521', '1682602746627237431505543.895725875201307870743131840081', 36, 0);
    t('61419836.366170852', '61419836.3661708526', 17, 1);
    t('581652389000000000000000000000000', '581652389018045200828000947653205.99314', 9, 5);
    t('3277631889894109325639218970.892003251333', '3277631889894109325639218970.8920032513320251750359060171', 40, 0);
    t('36953000', '36953384.22174646714971813609220552710', 5, 5);
    t('43663640341817618200000000000000', '43663640341817618179863638261999.6200513439491911703402339', 18, 0);
    t('7498874.767265905047', '7498874.76726590504707355843190984887463', 19, 1);
    t('592.35', '592.35112117943031799113511458', 5, 5);
    t('20666.88', '20666.8812963937', 7, 3);
    t('81034094983422887.5113569', '81034094983422887.5113569161525277202', 24, 6);
    t('9.3143105', '9.314310472134', 8, 0);
    t('227933532274861478777.60115795676372', '227933532274861478777.601157956763723861485853414927450', 35, 1);
    t('34275590816.7552921636', '34275590816.7552921635716886154110786290931', 21, 0);
    t('135425012025779532784000000000000000000', '135425012025779532783801787711031288403.36142521343475934323144380', 21, 5);
    t('660003910161256642.3247105', '660003910161256642.324710563133139', 25, 3);
    t('1318.471489297740204730907906', '1318.47148929774020473090790585191522552', 28, 5);
    t('6212082256195248221426818413000000000000', '6212082256195248221426818412743303160461.0', 28, 2);
    t('226672000000000000000000', '226672163258187920710589.6420932965179133428881651160541555823', 6, 6);
    t('794312000000', '794312618158.39777604134', 6, 1);
    t('8319394787.749', '8319394787.7495984094260614625163481694495', 13, 1);
    t('331771438081226556131863600000000000000', '331771438081226556131863565696008109159.634617115899853184530756446364', 25, 5);
    t('3507641446.56255512', '3507641446.5625551216900', 18, 4);
    t('217520594487.78675', '217520594487.7867505260352094716832', 17, 3);
    t('2228463161780000000000000000000000000000', '2228463161781177997428056069814769945779.33', 12, 1);
    t('4000000000000000000', '3512324211240173796.25693608', 1, 6);
    t('5860745.457689969', '5860745.457689969', 16, 2);
    t('223854573422908325.1', '223854573422908325.165001974305898194591080405954900', 19, 3);
    t('5310261200000000000000', '5310261203957324044820.984', 9, 6);
    t('914463981686565600000000000', '914463981686565507840759670.8280311547868606', 16, 2);
    t('438336858725526533425000000000000000000', '438336858725526533424917357302465317223.232683992851133582716652778820762821', 21, 6);
    t('455699351676789.536781', '455699351676789.5367810390156', 21, 5);
    t('3.594803875853457', '3.594803875853457129', 16, 1);
    t('35174559459794500000000000000000', '35174559459794505527205051097890.877832899930362389808528464', 16, 1);
    t('89000000', '88786017.9852735623573826512917891', 2, 5);
    t('147152797446865300000', '147152797446865293544.061955276227788070990113', 17, 0);
    t('340.14050532', '340.14050531137843974828631815292', 11, 2);
    t('18347974258134550.908903315309976', '18347974258134550.908903315309975991674725392799325529527', 33, 2);
    t('8670073206.97841040790728093252848258978', '8670073206.9784104079072809325284825897753616981', 39, 4);
    t('519', '519.0', 7, 4);
    t('7895.359121211912664526', '7895.3591212119126645260915826785113721947', 22, 1);
    t('20', '19.752707487', 2, 5);
    t('4153340943737094290992951.4944978033198', '4153340943737094290992951.494497803319804708984', 38, 5);
    t('7262.8101683134', '7262.810168313345915695', 14, 2);
    t('6898666327445650041221000', '6898666327445650041220544.990', 22, 6);
    t('33314320879667980000000000000000000000', '33314320879667979942776417568317110778.395747874160832284587016412053', 16, 2);
    t('571554809100963945213195', '571554809100963945213195.0', 26, 1);
    t('5971000000000000000000000000000', '5971087254799894091246449279737.39104945211307492927411236132', 4, 4);
    t('940000000000000000000000000000000', '932629661371147366576500039015885.39862992178472', 2, 2);
    t('4.26913777580028', '4.269137775800280506731558', 15, 3);
    t('6192581254117677483063840736833.50938694503983009192', '6192581254117677483063840736833.5093869450398300919203570629523', 51, 1);
    t('93498570653712999577440340534.05553895997', '93498570653712999577440340534.055538959967961042320460', 40, 6);
    t('6402165337112.512600305555648109279717', '6402165337112.51260030555564810927971759496110987', 37, 3);
    t('95524.415683032854382881093', '95524.41568303285438288109295335', 27, 6);
    t('846376200000', '846376248781.1823351865531555573', 7, 1);
    t('5510.51', '5510.51', 6, 6);
    t('512689283600000', '512689283609853.8765147934', 11, 1);
    t('13689.620254293559364', '13689.620254293559364', 20, 5);
    t('26392383149300835705287754649156.352074741429555856530724280094059', '26392383149300835705287754649156.35207474142955585653072428009405930', 65, 6);
    t('500.93', '500.928895', 5, 0);
    t('860000000000000000', '861655454643473570.455288635514548495605832839', 2, 1);
    t('78875528527093865414351316761970000000', '78875528527093865414351316761963234868.32982594846', 31, 0);
    t('1537849391433', '1537849391433.248432066792739613982', 13, 4);
    t('218.5629689315', '218.5629689315', 15, 1);
    t('13799843483770.816633666', '13799843483770.81663366624183293014', 23, 4);
    t('941190062089331.88584', '941190062089331.8858376518088', 20, 6);
    t('162803500000000000000000', '162803501200155459695682.689', 7, 1);
    t('923191188167253716861706488747962.926571055', '923191188167253716861706488747962.926571054263863580902', 42, 0);
    t('266.24376890804179', '266.243768908041789181', 17, 2);
    t('979525872200000000000', '979525872150294887622.35088764957', 10, 6);
    t('3615505800000000000000', '3615505801402322758596.28418066068', 9, 4);
    t('3085192750820110030089420939464227000', '3085192750820110030089420939464226700.863618952105598150140078171736255794672', 34, 5);
    t('2607894475016893661316.472613775542387930073809342787134047304', '2607894475016893661316.472613775542387930073809342787134047304', 62, 3);
    t('1715771436210000000000000000000', '1715771436217079320413405840233.0', 12, 3);
    t('3085856600000000000000000000000000000000', '3085856648888901535461400908475054227668.2344254935268425', 8, 3);
    t('80178850677600000000000000000', '80178850677675873052692525556.47742456494324271810', 12, 3);
    t('618958122043772558755661350011116207528.15688578526843828599076706', '618958122043772558755661350011116207528.15688578526843828599076705991218704', 65, 2);
    t('684295951505017907292228272865.15026', '684295951505017907292228272865.15026074262489', 35, 4);
    t('61585970108754824206000000000', '61585970108754824206578719242.74187148272', 20, 1);
    t('1277382244953.908', '1277382244953.90799089613189522027', 16, 4);
    t('574049350581918537879049549.474', '574049350581918537879049549.4743150339', 30, 1);
    t('857.5661648323527555', '857.56616483235275552463567346024', 19, 4);
    t('121824803060326017640297952725691521741.0369307', '121824803060326017640297952725691521741.03693069387951426156018', 46, 6);
    t('17999457.2881316170883805', '17999457.28813161708838053573', 24, 5);
    t('7803770509600.0046693', '7803770509600.004669273509726266000567532432600701', 20, 4);
    t('60030163.7652169387337439', '60030163.7652169387337439443637', 24, 3);
    t('27697500000000', '27697450013257.386675741', 6, 5);
    t('5222051191400000000000000000000', '5222051191378833029445261090785.2110', 11, 4);
    t('6598775918745214043279728955034125120.583481259', '6598775918745214043279728955034125120.58348125858145976261728265543845', 46, 2);
    t('9887325.31991', '9887325.3199073', 12, 6);
    t('628990825.232720181146446549676352659', '628990825.2327201811464465496763526591140261129', 36, 5);
    t('60284', '60283.97408757', 5, 4);
    t('95296784105194059434919939.26829', '95296784105194059434919939.26829', 32, 0);
    t('3741890000000000000000000000', '3741890701501923742115857034.0', 6, 6);
    t('93717912054695210000', '93717912054695207022.55074', 16, 6);
    t('779380000000000000000000000000000000000', '779376738414182119599165901962012420929.412039858413397844', 5, 2);
    t('28326874673984489109821072283261276553.027', '28326874673984489109821072283261276553.02653283925115988665576538879046', 41, 6);
    t('741960000000', '741964607066.789', 5, 4);
    t('14540055781002.17', '14540055781002.170', 19, 6);
    t('719806257040936451.3', '719806257040936451.3301729694336606', 19, 5);
    t('614.61158665521680469185', '614.61158665521680469184098', 23, 0);
    t('4.557123667061', '4.557123667060899439779192213', 13, 5);
    t('80348572591296040608945480034007300.7', '80348572591296040608945480034007300.72328502102214042054', 36, 1);
    t('61199130006.07029927', '61199130006.07029926818160078055', 19, 6);
    t('24854553193868760748104411740223079.1504296', '24854553193868760748104411740223079.1504295975992236037', 42, 6);
    t('2799856690000000000000000000000000000000', '2799856689887788668836696556883491505716.0', 9, 2);
    t('9388854', '9388853.965657841209890812892', 7, 2);
    t('9528282528129225.89802989657', '9528282528129225.89802989656753100662146616', 27, 4);
    t('21858648699876000000000000000000000000', '21858648699876239487024778705556643570.0', 14, 6);
    t('9898298502335147981702496203000', '9898298502335147981702496202671.363169073361400148925418', 28, 0);
    t('2563287465966373852057871363.87205306467323937784', '2563287465966373852057871363.872053064673239377847157291006', 48, 3);
    t('509087957890694.027144', '509087957890694.0271446223662547189454827185928', 21, 3);
    t('2651764303563573459278189862282.787013497081759', '2651764303563573459278189862282.7870134970817589522842360614197', 46, 2);
    t('383000000', '382503381.1423687746592981021179665535230481', 3, 2);
    t('39871483452420824219252000000', '39871483452420824219251679733.0317096239927', 23, 6);
    t('43820172717.34250770993687597', '43820172717.34250770993687597198844188031', 28, 5);
    t('4000000', '4364322.38848978626707964871204835', 1, 4);
    t('4552800773.4', '4552800773.4', 14, 5);
    t('597212000000000', '597211837599005.274247517', 6, 4);
    t('1978833', '1978833.0', 7, 6);
    t('254813330868927948783393.25182', '254813330868927948783393.251820', 32, 2);
    t('7509910870712887931500000', '7509910870712887931481815.8', 20, 4);
    t('229178000000000000000000', '229177664237153447346766.6857570958439165487228414870', 6, 0);
    t('90507475536036204700000000000000', '90507475536036204707701846580043.370997827331050448126103', 18, 1);
    t('68.568861', '68.5688606754270', 8, 2);
    t('7178716.64', '7178716.633436', 9, 2);
    t('650556771280042.8649', '650556771280042.8648367946337145374058675496978', 19, 0);
    t('25405999231620811663908025829000000000', '25405999231620811663908025829114416538.492603015101736712101', 29, 5);
    t('3990248400000000', '3990248366313181.7999638740829132', 8, 6);
    t('4774282241000', '4774282240867.1479647', 10, 2);
    t('66073.0225655753766', '66073.02256557537660190', 19, 1);
    t('4587025500', '4587025565.1731540637164245314446067445', 8, 1);
    t('495668.074', '495668.07441990177179', 9, 4);
    t('23199186728650300000000000', '23199186728650284484034479.7937', 15, 6);
    t('607886497956635834317717797220.08475876456511908', '607886497956635834317717797220.0847587645651190760737963463495', 47, 5);
    t('43565739182979560196012177808.00122181815551800834510137267055', '43565739182979560196012177808.00122181815551800834510137267055', 62, 5);
    t('58358.76', '58358.76756', 7, 1);
    t('52104739726888022407276260635572.01', '52104739726888022407276260635572.008194555678852225409379529', 34, 0);
    t('90160367918209534530000000000000000', '90160367918209534522064600077320451.5', 19, 0);
    t('8700000000000', '8646586789817.54789160058420', 2, 2);
    t('973562481119527707128', '973562481119527707128.0', 22, 6);
    t('5079000000000000000', '5078343495318673114.092500', 4, 2);
    t('582269452656104000000000000', '582269452656103814944130246.25640170270529958', 15, 0);
    t('748149543693647.3722625123933797', '748149543693647.3722625123933797', 31, 2);
    t('40483293017540048400000000000000000000', '40483293017540048447719058673244646541.942943', 18, 4);
    t('747630184307957000000000000000000000', '747630184307956620803669229710484617.6706599551506236499202', 15, 0);
    t('30346635857241586.665917212261501914', '30346635857241586.66591721226150191387028028634', 35, 4);
    t('40164846156236115607.5047', '40164846156236115607.504722328', 24, 5);
    t('6486609.686', '6486609.6860643150699', 10, 1);
    t('543102894131598350130942100000000000', '543102894131598350130942097001017334.8', 26, 4);
    t('553635291450203.503189743406457640797', '553635291450203.503189743406457640796805080748853127', 36, 5);
    t('46031940600000000000000000', '46031940513471214140996872.154252828965233643036417876938835632420', 9, 0);
    t('25774286714.727', '25774286714.7269695356627422', 15, 2);
    t('52931185128009889996858636.632', '52931185128009889996858636.6321063891911248828', 29, 6);
    t('88088709231458000000000000', '88088709231458785989226331.57', 14, 1);
    t('9927560984957292333073334544', '9927560984957292333073334544.2742825399748883624915', 28, 1);
    t('4808315509634.04239797378245542', '4808315509634.04239797378245541613846885670223372532', 30, 2);
    t('16110044458147517754006.87891299641', '16110044458147517754006.87891299641', 34, 3);
    t('100765302819312100', '100765302819312059.099864230924352525606', 16, 6);
    t('718775.58351352357', '718775.5835135235692', 17, 5);
    t('11105351332830.5202728', '11105351332830.52027286', 21, 3);
    t('84674171380000000000000', '84674171387063311442528.261883758840389384550729036712097', 10, 3);
    t('12466649182086705.4421456927540741831838136463203268301', '12466649182086705.4421456927540741831838136463203268301', 54, 0);
    t('307039271437389715972463507606229300596.9926434695642031', '307039271437389715972463507606229300596.99264346956420310467923', 56, 4);
    t('66000000000000000000000000000000000000', '66002014472991402889186780838269855225.36930185259029985644', 3, 6);
    t('512969917035155855959638000000', '512969917035155855959637015850.47008976122680275491099397', 24, 0);
    t('663959423376086075763560528686938236870', '663959423376086075763560528686938236860.0318045883806188389066923349', 38, 0);
    t('659210000000000', '659207394296883.281601744509589481289018945815852', 5, 4);
    t('17249318749.9351142', '17249318749.935114199387', 18, 5);
    t('444838990041882461.9330596334532', '444838990041882461.9330596334532', 34, 3);
    t('1738671356000000000000000000000000', '1738671356217486810883125384722225.80126808622256426363313', 10, 6);
    t('3096000000000000000000000000', '3095976590673285479562287906.9214640827416605510203', 4, 0);
    t('63721614773688986.40694089965346127', '63721614773688986.40694089965346126623218823695062608135', 34, 0);
    t('978948295134530213483635374456833.7820510783295838527916401728', '978948295134530213483635374456833.7820510783295838527916401727519', 61, 0);
    t('45287498.88669551337250444', '45287498.88669551337250444', 27, 2);
    t('7638591440000000000000000000000000000000', '7638591448571211744691844805872401578290.0919304251756051197', 9, 1);
    t('610.2900453767248185445974545769222', '610.290045376724818544597454576922240791307', 34, 6);
    t('2370000000000', '2374836560868.87546876297', 3, 1);
    t('919520000000000000000', '919516533865939479306.70070273212571', 5, 5);
    t('16523735980784388627531200000000000', '16523735980784388627531155036923582.9231386166061833979', 24, 2);
    t('851225333.97847791', '851225333.97847791670219874924264564', 17, 1);
    t('5674496194700000', '5674496194662001.64258318410413485258820591342648', 11, 0);
    t('3006607167565635804970751879873438.5', '3006607167565635804970751879873438.51', 35, 3);
    t('43076640000000000000000', '43076640235689201871428.8329337674402264816355775139472', 7, 5);
    t('815881583599195712042828844112.114461', '815881583599195712042828844112.1144614547925538729216129323521546770631', 36, 3);
    t('406619308963980000000000000000000000000', '406619308963970409822148910493032652815.76936764364873182893999532289260686787', 14, 2);
    t('9800000000000000000000000000000000', '9801263402559707490778892340083302.7', 2, 1);
    t('7042735204727200000000', '7042735204727200263839.842526427943811', 16, 6);
    t('40000000000', '37354217679.6137344024100437249802064150006815', 1, 0);
    t('55344907152.080681398091195452055131382', '55344907152.080681398091195452055131382', 38, 5);
    t('726798235674100000', '726798235674050961.0', 13, 2);
    t('482014.1184119513623039299135428', '482014.1184119513623039299135427873749976114', 31, 4);
    t('1110033261543.283394', '1110033261543.2833940384', 19, 4);
    t('7817917192668452.818386120320887', '7817917192668452.818386120320887032863354009612910222', 31, 4);
    t('347.3041882', '347.30418819083', 10, 4);
    t('361934872153340556.69353411', '361934872153340556.693534109243034865506952', 26, 6);
    t('6479757238179.54024574142736288372765264405648787898', '6479757238179.540245741427362883727652644056487878980', 53, 2);
    t('9073630813403.7127721274852301815737071526', '9073630813403.7127721274852301815737071526', 42, 6);
    t('3212878846324812000000000', '3212878846324811592399253.3832950020', 16, 0);
    t('2484005068901331692325348996407345478.20380522', '2484005068901331692325348996407345478.203805211463532947471731717457691992', 45, 2);
    t('447257138.927', '447257138.9276030566309913925', 12, 3);
    t('8709621689251047000000000000000000000000', '8709621689251047331259370125200005032852.6511043945303191959321784', 16, 1);
    t('92104066890330890000000000', '92104066890330893403180632.12971419629856395181644743095381697', 16, 3);
    t('41068038483546918312608', '41068038483546918312607.998', 25, 0);
    t('27312651147358309848943553660361588410', '27312651147358309848943553660361588418.10447014287', 37, 3);
    t('81506198620000000', '81506198623938042.9795145673363960263760692251128616128396', 10, 5);
    t('32613557484633394694302.755719', '32613557484633394694302.755718922742810071308452716894091467', 29, 6);
    t('681029854424000000000', '681029854423350673956.8696686943', 12, 2);
    t('77599.01946964865147744097345', '77599.01946964865147744097345223071541854', 28, 6);
    t('33997434934343488000', '33997434934343487240.3885', 17, 0);
    t('397385264576.38280428097', '397385264576.3828042809796418353960', 23, 3);
    t('661204877000000', '661204877452594.5649421356769751051853892', 9, 5);
    t('3477075695597549', '3477075695597548.9071679028800', 16, 5);
    t('27099174784708325008.0998', '27099174784708325008.099755088052074382397504612069', 24, 0);
    t('356538829878671.68860696351814518', '356538829878671.68860696351814517524100505788800', 32, 5);
    t('4000000', '4589036.069268', 1, 1);
    t('102380297.553085816', '102380297.553085816381275492559519', 18, 4);
    t('2119809316579162508331309074518408568969.88084720981529150286573337502998414', '2119809316579162508331309074518408568969.8808472098152915028657333750299841397', 75, 0);
    t('88766977872742970937977275819327317223.77956522', '88766977872742970937977275819327317223.779565215019671665', 46, 6);
    t('2960930937437854217641564887365705.0682087986', '2960930937437854217641564887365705.068208798693453131154527302', 44, 3);
    t('9360.687198564', '9360.687198564081040494955276444', 14, 1);
    t('6582600242324501022650013922348778842584', '6582600242324501022650013922348778842584.0', 43, 4);
    t('23094770.162260783155028', '23094770.16226078315502800', 24, 0);
    t('198745601930985127070000', '198745601930985127069365.66385', 20, 6);
    t('5072778000000000000000000', '5072778799669067150706506.55044698961878745614205849469617', 7, 1);
    t('15405000000000000000000', '15404656372242188842656.202993148742519059850', 5, 0);
    t('66736078.13036436', '66736078.130364366939878924656', 16, 1);
    t('831441.5', '831441.408661815496230146', 7, 2);
    t('2700000000000000000000000000000', '2604806111199300818551714901670.55', 2, 2);
    t('65660883000000000000', '65660882759294567016.9238457625', 8, 6);
    t('5079.822962', '5079.822962', 10, 1);
    t('9279173897674978625', '9279173897674978625.2579848416945184996063221820639', 19, 6);
    t('72443147845874299479993.474731', '72443147845874299479993.47473151594062599875770330750', 29, 1);
    t('96853872993995100000000000', '96853872993995025754705094.39965864961523651787249799', 15, 0);
    t('9.847131517331', '9.8471315173312049815497761673', 13, 4);
    t('9575992803116793759038677.97643', '9575992803116793759038677.97642579040183877236150918902094', 30, 0);
    t('474580486363688381790.666', '474580486363688381790.66648580128613221154709511', 24, 1);
    t('25794204749229000000', '25794204749228047488.3054104161830185616764433994914817868', 14, 0);
    t('7509600000', '7509630260.728246919151679230512959854625159203792', 5, 6);
    t('9117133596580672503.58499073522444', '9117133596580672503.5849907352244397', 33, 0);
    t('637732523199824000000000000000000', '637732523199824205547294057820239.164608109', 15, 6);
    t('45.6665578873813169', '45.6665578873813169266', 18, 4);
    t('2174733007639100000000000000000000', '2174733007639108800313270544768877.22081269', 14, 1);
    t('771139500630738600000000000000000000000', '771139500630738674534455262808544547145.456092784', 16, 1);
    t('791867147632451910526036868967.50839548', '791867147632451910526036868967.5083954769146390076023105365963384', 38, 5);
    t('3615802860684743120883746120678010839746.317791332915', '3615802860684743120883746120678010839746.3177913329149', 52, 6);
    t('8532.38781747126492288195990084362946', '8532.38781747126492288195990084362945955', 37, 4);
    t('609398680000', '609398680372.8', 8, 5);
    t('2790028584124602119289699.8746142637176034339', '2790028584124602119289699.87461426371760343390207564942295170445', 44, 6);
    t('987126006921.1074582', '987126006921.10745818243979', 19, 2);
    t('3424.03142973468', '3424.03142973467339', 15, 0);
    t('41.443', '41.4430', 9, 3);
    t('764.527149384918', '764.52714938491783098299358030696021735101', 15, 2);
    t('2200000', '2269368.999317642538266525972094', 2, 1);
    t('93796558522597.99034', '93796558522597.9903403511823', 20, 1);
    t('480993493740477100000000000000000', '480993493740477045443922228519820.90607695', 16, 0);
    t('57412434815497299470517245368832810.163517610769031950239', '57412434815497299470517245368832810.1635176107690319502389615512491843513049', 57, 5);
    t('225056663729753044964235085738212.489101514', '225056663729753044964235085738212.48910151393730', 42, 4);
    t('6046900', '6046866.67143767903', 5, 0);
    t('77020622924237969921469475719773.14286521642994', '77020622924237969921469475719773.142865216429945315550500', 46, 3);
    t('9', '9.0', 3, 2);
    t('15082860131579982439.8939643475211914', '15082860131579982439.893964347521191418082905722577410010', 36, 5);
    t('7619940976771677000000000000000000000000', '7619940976771677004530273461114047397525.154', 16, 3);
    t('85697539715056003147762376327.7951125912', '85697539715056003147762376327.795112591209003', 39, 1);
    t('6787000000000000', '6787144042978272.9944733903625558221763027789', 4, 6);
    t('9476641762393347530600000000000', '9476641762393347530562902254377.48027994770', 20, 6);
    t('46377579209030229377444.891051101283702228679491562761252', '46377579209030229377444.89105110128370222867949156276125214888', 56, 3);
    t('17632350308125247400000000000', '17632350308125247420526418021.4346346606040841495', 18, 3);
    t('5446345729553956384486922522522.2871660073887', '5446345729553956384486922522522.2871660073887', 45, 6);
    t('3386521844598695382092709173601198286.85655233338396800845476773', '3386521844598695382092709173601198286.85655233338396800845476773395', 63, 1);
    t('532121.86042887485', '532121.8604288748588479868190351383935384651', 17, 3);
    t('89691430000000', '89691433473847.59353496531307048187457771322970500', 7, 1);
    t('627552552.82458', '627552552.8245787330', 14, 4);
    t('4261533662900000000000000000000', '4261533662902429091700757172084.7285260574335200191702357914', 11, 1);
    t('8052626.1537741936', '8052626.153774193558021472253', 17, 5);
    t('7076000', '7076114.813', 4, 5);
    t('7876683130897163650911370000000000000000', '7876683130897163650911371603967678777553.142463042155', 24, 5);
    t('6365698547294240612342467614044.1882775340074', '6365698547294240612342467614044.1882775340073951820', 45, 2);
    t('5891108648500618.349480546173211061316903807609008009396', '5891108648500618.349480546173211061316903807609008009396', 57, 1);
    t('298420000000000', '298419766627313.5026956545741877586', 5, 0);
    t('760000000', '762592985.138322315017', 2, 4);
    t('7612333594996286702990217359219513100551.65176579', '7612333594996286702990217359219513100551.65176579', 51, 3);
    t('2138900680353', '2138900680352.517911464572506', 13, 5);
    t('15564707692964912377786539981.7759', '15564707692964912377786539981.77592194772425655199554', 33, 4);
    t('641858961181250914818', '641858961181250914817.3915546039273', 21, 2);
    t('0.3693562375762159', '0.3693562375762158081', 16, 0);
    t('7617049578900895345198470929', '7617049578900895345198470928.8', 28, 4);
    t('1.1297499740078870550057', '1.12974997400788705500571964', 23, 4);
    t('946200591173446024430250328001547890000', '946200591173446024430250328001547889184.2524614', 35, 2);
    t('32709.966', '32709.96604648796292357', 8, 3);
    t('3620000000000000000000000000000000000000', '3619839973223046578758187592325204629761.8859942', 4, 2);
    t('90000000', '83649327.7476477075940893091850490739370000', 1, 2);
    t('495000', '495243.21896843855432604', 3, 1);
    t('5000000000000000000', '4758188271119880748.57816371495723097345938290521631', 1, 5);
    t('93419893998110879692824000', '93419893998110879692823535.511212491699', 23, 2);
    t('8033207691842959874180936703.572556855838504556', '8033207691842959874180936703.5725568558385045550361914', 46, 2);
    t('73013497623518342142241149674.06667023022023060859', '73013497623518342142241149674.06667023022023060858717', 49, 0);
    t('642.5819166', '642.58191657854', 10, 4);
    t('49808.01366517291572563585163', '49808.013665172915725635851633179678367707019', 28, 1);
    t('48982614000000', '48982614008194.1074867', 8, 1);
    t('9533.1561196786', '9533.1561196786000050316406851681', 15, 4);
    t('97367607351654.72853', '97367607351654.7285277223917499060815034026905177720', 19, 2);
    t('6554371300000000000000000000000000', '6554371208660735587035802666339911.6889044331488880241899294233706', 8, 0);
    t('63162770593514143021742570.9118', '63162770593514143021742570.911765955362282309424427888818125002', 30, 6);
    t('5453374862586694.205511869714', '5453374862586694.205511869713870', 28, 6);
    t('6369039661082741228490432734063397680521.560703489858', '6369039661082741228490432734063397680521.560703489858319632274', 52, 3);
    t('64929912368644744420.158719', '64929912368644744420.15871867542915559317201', 26, 0);
    t('6041385481000000000000000000000000', '6041385481428052998591796563431343.9492346857531084711466658082666757084759', 10, 1);
    t('771980480561766107646857961', '771980480561766107646857960.71700068326846', 27, 5);
    t('20507298553644.4066861909502957485', '20507298553644.406686190950295748501105200944087239', 34, 1);
    t('85.197032946', '85.197032946', 13, 2);
    t('7927481301330.07920615414132814', '7927481301330.07920615414132813517667312', 30, 2);
    t('118760448449332300000000000', '118760448449332245508790897.805146155691143606636', 16, 2);
    t('55507627962820000', '55507627962820810.24487', 13, 5);
    t('81180960161598232657894.421866608811249578521581172849', '81180960161598232657894.421866608811249578521581172849', 56, 0);
    t('86133754811.312129494966034921660895851', '86133754811.3121294949660349216608958510808', 38, 5);
    t('5535847511900000000000000000000000000', '5535847511821022472936166202493417734.9452109134187106184', 11, 2);
    t('9933.6362281842774', '9933.63622818427740', 19, 3);
    t('97843575586006000000000000000000000', '97843575586006008508539613974277067.523095132554085', 15, 3);
    t('860000000', '859365108.0', 2, 0);
    t('584116302546390089467408860864437930303.67509', '584116302546390089467408860864437930303.675086582315502501', 44, 0);
    t('699105686817664673975187340000000', '699105686817664673975187341876530.58699159', 26, 3);
    t('79341615699338788770955954735000', '79341615699338788770955954734272.01125847122547', 29, 0);
    t('89312152.7458914658972', '89312152.7458914658971850047575398993607523416', 21, 5);
    t('12000000000000000', '12177088524504911.22312414494112278904', 2, 1);
    t('125800', '125819.435501491664346251839256987142832676999', 4, 3);
    t('620438257338252958424887982643.37248165', '620438257338252958424887982643.3724816543943815714498754', 38, 1);
    t('1988417018001771801181486324514274.6', '1988417018001771801181486324514274.6839243737616287207523006551968682088574', 35, 1);
    t('300000000000000000000000000000', '345716568446441498684879346904.26790396338', 1, 3);
    t('4272123153029827300000000', '4272123153029827290795068.3756223994827587', 18, 0);
    t('30893542514345547870000000000000', '30893542514345547873546522314490.365302499373180281129948946', 19, 5);
    t('1358079936441623022.318282572497127779622045357', '1358079936441623022.3182825724971277796220453571643447', 46, 1);
    t('920000000000000000000000000000', '919603351657253788308545594260.42481574513111428766', 3, 2);
    t('91284.2775401572', '91284.277540157157085789045766471196', 15, 5);
    t('2509096257.47', '2509096257.461712782211000780', 12, 0);
    t('459787.62227382017319', '459787.6222738201731865076576174215883304026', 20, 6);
    t('4370900000000000000000', '4370856230530620812296.0', 5, 0);
    t('8834682615296315980870441.68062808796', '8834682615296315980870441.6806280879687261507', 36, 1);
    t('7098532178124896584.80437956995187408401719686379', '7098532178124896584.80437956995187408401719686379', 49, 2);
    t('3804206872400000000000000000000000', '3804206872374466201898540025556941.441419087160658167368', 11, 4);
    t('3946358425390609213589130.6949209842394457256364', '3946358425390609213589130.6949209842394457256364', 48, 2);
    t('1237454797874092286920385148486995', '1237454797874092286920385148486994.8356201766259537231979358361131600693033', 34, 5);
    t('67864988721846030372970713214395800', '67864988721846030372970713214395786.62783342432598', 33, 0);
    t('198499774880.452', '198499774880.45126665272961', 15, 0);
    t('4108400.72100086', '4108400.72100086', 15, 4);
    t('113513072324000385.568', '113513072324000385.568', 23, 4);
    t('25', '24.29556067217011742396399', 2, 2);
    t('7328318030000000000', '7328318033731853006.29', 9, 5);
    t('940754212007660825121043451291', '940754212007660825121043451291.978315193778910640701322127301', 30, 3);
    t('48772340028367330300', '48772340028367330266.008', 18, 5);
    t('226908389000000', '226908388988425.97461813531', 10, 0);
    t('903402018290.1037055560329961200752', '903402018290.1037055560329961200752444525558555802', 34, 6);
    t('152900000000000000000000000000000000000', '152877011725054785484411291262746406108.692780', 4, 0);
    t('226885800000', '226885866001.259266820948', 7, 1);
    t('647123.866988466437830037', '647123.8669884664378300365479846124251', 24, 4);
    t('193834927.842266', '193834927.842266', 17, 0);
    t('8200000000000', '8155108797512.962969885007463015', 2, 2);
    t('99210310000000000', '99210310012075687.58623074397', 9, 5);
    t('883665507006.7538287088222668176', '883665507006.7538287088222668176', 32, 5);
    t('1000000000000000000000000000000', '1649386889628931919996015616852.9142673361245073331786736', 1, 3);
    t('80384230031.98710508284742380699675269212', '80384230031.987105082847423806996752692129067', 40, 3);
    t('607121428361814267329767126000000000000', '607121428361814267329767125965362574532.52', 28, 6);
    t('98251995395799750219698833.434746218821654581132506851', '98251995395799750219698833.4347462188216545811325068515984', 53, 3);
    t('27.8513825276933', '27.85138252769337857', 15, 1);
    t('1729062289616208683.838791', '1729062289616208683.838791', 25, 0);
    t('4835078252195107261', '4835078252195107260.149693005491645602480203', 19, 0);
    t('3200600000000000000', '3200592713214712854.8655117741363956517171803391446', 5, 4);
    t('71571141389156958763316358974719.327059', '71571141389156958763316358974719.32705875', 38, 0);
    t('5422.3', '5422.2733913212597596548492799819139', 5, 4);
    t('31991009603.729381895813819672', '31991009603.729381895813819672', 32, 4);
    t('60839043121679005627909706643.4709980292888001986172', '60839043121679005627909706643.47099802928880019861719719379', 51, 6);
    t('875386072059703907875473595750', '875386072059703907875473595746.7277629810983597699218331587315', 29, 2);
    t('24780728389317697.819685', '24780728389317697.81968440867963427819086123532591756270', 23, 2);
    t('4456608709191048529300000000000', '4456608709191048529205664234671.0148117704860', 20, 0);
    t('19456058000000000000000000000000', '19456057447931996672780750437726.869435429776449934529102823774013247367', 8, 0);
    t('270090.7189', '270090.71894807', 10, 6);
    t('7224214368486646438000000000', '7224214368486646438023855936.007501328403265945', 20, 4);
    t('877502.9314151704263333', '877502.9314151704263333', 25, 5);
    t('320000000', '320444948.9834850256852', 3, 1);
    t('17.604735481031986558840050282', '17.6047354810319865588400502829986187466089', 29, 1);
    t('991740000', '991737996.372554760225554', 5, 6);
    t('68000000000000000000', '67811357407967219927.0', 2, 5);
    t('9268657705.314275393683984', '9268657705.31427539368398433302383097', 25, 3);
    t('201670750684764563381.81949920163433831902', '201670750684764563381.8194992016343383190207', 41, 1);
    t('7668028.3464151652871281730587042924', '7668028.3464151652871281730587042924025', 35, 5);
    t('1316104110518449.7755088697706498771086984973', '1316104110518449.7755088697706498771086984973367703636221', 44, 5);
    t('71394617040598735664.5029020530437834', '71394617040598735664.5029020530437834', 38, 0);
    t('3719313802954450928544999510', '3719313802954450928544999503.79300942347945781323148318793410045', 27, 2);
    t('386303130000', '386303128892.367130216', 8, 2);
    t('50127055347335265.134752943', '50127055347335265.134752943', 28, 0);
    t('34860772387164007552606848000000000000', '34860772387164007552606848234678929905.07246559179831633780', 26, 6);
    t('1336273748208419338751238827122911.8713', '1336273748208419338751238827122911.871312495956538076', 38, 3);
    t('351328157.56379278820226', '351328157.5637927882022605116198065', 23, 6);
    t('78780000000000000000000000000000000', '78789152689083966794982265269399169.3472254', 4, 1);
    t('993.10260499921214203431235932276924', '993.10260499921214203431235932276923782', 35, 0);
    t('6749291057953800000000000000000000000000', '6749291057953821468371672863861043863882.122799633121428971104867311418', 14, 3);
    t('408188067461018874188244211938037433.946551790013148655199210311816', '408188067461018874188244211938037433.946551790013148655199210311815905015556', 66, 2);
    t('466067483450292786621.4901258813733588578081697523334551', '466067483450292786621.490125881373358857808169752333455093', 55, 2);
    t('2496729.0533609', '2496729.053360990', 14, 3);
    t('4782.446670358144837436425', '4782.4466703581448374364247997481622143', 25, 0);
    t('18160192.3', '18160192.30272846565611868889211594', 10, 6);
    t('48000000000000000000000000000', '48106002212371632123410355250.946708748480492520472859063734578500', 2, 4);
    t('396141329930264597249458406787820221680', '396141329930264597249458406787820221684.0837523240956441', 38, 4);
    t('8324539547635002.3423', '8324539547635002.3422825168732639857588870', 20, 5);
    t('53000000000', '53598559272.722', 2, 3);
    t('4594.21510604212657828265136', '4594.21510604212657828265136', 29, 4);
    t('4907376409917233239720000000000000', '4907376409917233239722158697670645.735959264796', 21, 1);
    t('366000000000000000000000', '365730375933732918353009.850709', 3, 6);
    t('474390049387752893', '474390049387752893.2320015547', 18, 1);
    t('145305260288166305767342884484.944628682663', '145305260288166305767342884484.94462868266347', 42, 6);
    t('53897339583598.120797663128702', '53897339583598.1207976631287019437308102', 29, 4);
    t('533987176227275.209916198457108362232672', '533987176227275.2099161984571083622326725008400', 39, 1);
    t('301564723835480000000000000000', '301564723835483301737880841179.03380401008', 14, 4);
    t('584908578677.04121741478583250767598', '584908578677.04121741478583250767598295', 35, 1);
    t('8738854249220954971614710700000000000000', '8738854249220954971614710709607500887784.901460985737', 27, 1);
    t('2125474248191772000000000000000000', '2125474248191771922277452278095853.7640293969590620827735', 16, 0);
    t('6769182960700', '6769182960651.488566324555197', 11, 5);
    t('13381225797084098260684930000000000', '13381225797084098260684927114658462.972654257151625001099797375129', 25, 4);
    t('5520572141719860.1510385342016466909587917228674289', '5520572141719860.1510385342016466909587917228674289', 50, 1);
    t('49660425684732367962.133934812364', '49660425684732367962.1339348123635478154134737182972756171138', 32, 0);
    t('78463326217556840.452536407502891401444828385866598577', '78463326217556840.4525364075028914014448283858665985770', 54, 3);
    t('94788682276223953520710995355094.4', '94788682276223953520710995355094.40894408628163344', 33, 1);
    t('9173134667772928.960404694173684997', '9173134667772928.9604046941736849962742148', 34, 0);
    t('635955.746089029138732158585005235', '635955.746089029138732158585005235198945632927', 33, 6);
    t('800000000', '845707251.96835494', 1, 3);
    t('751802547110849595605943964953000000000', '751802547110849595605943964952397252489.12324641492249666964473117419941440', 30, 2);
    t('69884.067455815394', '69884.0674558153949082760742', 17, 3);
    t('73492149496065710541180000000', '73492149496065710541182908013.4937575417282', 22, 4);
    t('1614325767742200.357839', '1614325767742200.357838993157819027840026275684', 22, 5);
    t('14569000000000000000000000000000000', '14569016905135303575570801240854613.55665408366554265922845465870802', 5, 5);
    t('77871553758.16580361', '77871553758.16580361', 19, 2);
    t('3483.97374085551619285', '3483.973740855516192840313', 21, 2);
    t('212848034137636843824816181600', '212848034137636843824816181510.0870395295', 28, 0);
    t('931', '930.9695016409126494961316861153484', 3, 5);
    t('3257987769760000000', '3257987769765713787.60831264697098275633', 12, 1);
    t('100000000000', '96442499544.28', 1, 6);
    t('27.2955', '27.295464696716225578', 6, 6);
    t('3400920616721224201572000000000000000000', '3400920616721224201572497248551756826999.709764325425', 22, 1);
    t('4957626412201223472119453195706000', '4957626412201223472119453195705960.978080765231758663232682037005380991', 32, 0);
    t('150354649189774058074060287000000000000', '150354649189774058074060286548341723323.937656489', 27, 5);
    t('2501.108', '2501.1078217784', 7, 2);
    t('607227000000000000000000000', '607226895007988464883248187.94661057810', 6, 4);
    t('46210233866233871', '46210233866233870.773663514487296', 17, 0);
    t('357117153589613212284.15175586113', '357117153589613212284.151755861123802760642445969016', 32, 0);
    t('7959547177465973548251580900', '7959547177465973548251580909.9040064432844545780456145279510604', 26, 4);
    t('728458890975100061153585.916217010098828963307069', '728458890975100061153585.916217010098828963307069095', 48, 4);
    t('3303847936373895451690781000', '3303847936373895451690780026.62582008545929372135780172794294156', 25, 2);
    t('4689776135000000000000000000000000', '4689776135323880070583913902305330.69680174103123353957919', 10, 4);
    t('811308881246000000000000000000', '811308881246792623148150211620.070456231878', 12, 3);
    t('714.2398961995594', '714.2398961995593751469913955534317562909', 16, 0);
    t('8067056460831967588556879464723000000000', '8067056460831967588556879464723395371232.46848', 31, 5);
    t('58719044100000000', '58719044178341940.41017943018918603400534019554680', 9, 1);
    t('2017190745070625525544317389446245500000', '2017190745070625525544317389446245534299.9437924852230', 35, 6);
    t('213589860250000000000000000000000000000', '213589860248909347823201087489643146635.685', 11, 4);
    t('32995384247364113488166323381000000000', '32995384247364113488166323380965149905.769959183615902738683534', 29, 0);
    t('6039897.965', '6039897.96512175188663274433', 10, 1);
    t('8927220000000000000000000000', '8927220397749943792620398447.447229479026276657799165668016477', 7, 5);
    t('21140756359269989756241450229171.463148884', '21140756359269989756241450229171.4631488844628224982025', 41, 4);
    t('38130576830000', '38130576839953.65077', 10, 3);
    t('9434438340823384.448171272963', '9434438340823384.448171272963', 31, 0);
    t('43000000000000', '43451010360962.799531136266', 2, 5);
    t('179860000000000', '179851290954245.41999591220264996407', 5, 0);
    t('56.29793108552503126', '56.2979310855250312626160452925466128293339', 19, 6);
    t('14571789400000000', '14571789327029811.629490133670523', 9, 2);
    t('67924155200542.75796', '67924155200542.7579668544306', 19, 1);
    t('92634.009106149156390793429054', '92634.009106149156390793429054', 29, 4);
    t('4962376056479277655685534200000000000', '4962376056479277655685534197388578569.2688806778075244217200038843', 26, 2);
    t('548299165350306642891481689859219626207.634363065364692357685463', '548299165350306642891481689859219626207.634363065364692357685463', 64, 0);
    t('9000000000000000000', '9073182415306029608.565629', 1, 3);
    t('263675045490000', '263675045499541.767433596084981809232037781499695353', 11, 1);
    t('80', '80.8377328506743152169', 2, 1);
    t('3072796962107369221099131.264159078125292397797969754181325530724', '3072796962107369221099131.2641590781252923977979697541813255307240', 68, 3);
    t('651338205490376018607927148773.3225300473134530685', '651338205490376018607927148773.3225300473134530684929298823', 49, 0);
    t('13149783797005420500153929010294.037', '13149783797005420500153929010294.03689', 35, 2);
    t('457029325437547368069556419.1835335974712027553337404', '457029325437547368069556419.1835335974712027553337404290', 52, 4);
    t('74387240824591593299651733520103.42', '74387240824591593299651733520103.414482350203683766', 34, 2);
    t('643192000000', '643192284518.7448669605882259978300', 6, 3);
    t('371097129669812400000000000000', '371097129669812424645700612539.69508', 16, 1);
    t('11421961443220000000000000000', '11421961443220710837557609272.6518293750631808899475931351724224', 14, 3);
    t('36470.728', '36470.728', 10, 6);
    t('66308.14976876', '66308.1497687580105', 13, 6);
    t('62971570000000000000000000000000000000', '62971572945303451313886065403578690307.0580984926728914635816344120975036', 7, 6);
    t('13885', '13884.9731128619995306', 6, 2);
    t('86909665827214877037000000', '86909665827214877037254395.40051', 20, 4);
    t('7786.96', '7786.96', 9, 4);
    t('47841518458900000000000000', '47841518458864503251471405.81360585', 12, 5);
    t('3856700.277459495409523020259', '3856700.277459495409523020259379045844154488508', 28, 4);
    t('59198402210934800000000000000000000000', '59198402210934827711078653405050952575.8739', 15, 5);
    t('7383678442297206', '7383678442297205.8358549489', 16, 5);
    t('621777400000000000000000000000000000000', '621777416657954137690612974300823465049.217349638406024506994164340745084', 7, 4);
    t('7521588147950000000000000000000000', '7521588147950460692804680673601945.8320487471325458499003307907554', 12, 4);
    t('6221487001281383604344000000000000', '6221487001281383604344092082081187.233544450524253192350543009063', 23, 1);
    t('3522949409618541398.1636865879574', '3522949409618541398.1636865879574', 35, 3);
    t('4526050889002522.4', '4526050889002522.4', 20, 6);
    t('3167974946842669549870000000', '3167974946842669549864557190.7023607168445921', 21, 0);
    t('7407268.7771555144531', '7407268.7771555144530749990922', 20, 6);
    t('7694799860000000000', '7694799850931133185.471995205544512', 9, 0);
    t('51448524770304776071562734888143.2860119172375901908', '51448524770304776071562734888143.286011917237590190780990125027897', 51, 0);
    t('899112985660370000000000000', '899112985660368700985819283.1767', 14, 5);
    t('4645740100549854859075810.306329668606215949140818081', '4645740100549854859075810.3063296686062159491408180807593953', 52, 5);
    t('4000', '4263.04102234187341361739865287799326694', 1, 1);
    t('362068327835359037061330380000000000', '362068327835359037061330382661833277.03237396477217769259156053603', 26, 4);
    t('645000000', '645954147.0', 3, 1);
    t('3000000000000000000000000000000000000000', '2279431651529143452456614770432858937900.05362872888984208598853717562990', 1, 0);
    t('357841670802087188.9355124', '357841670802087188.93551243', 25, 6);
    t('79078566125022339288.128382591128', '79078566125022339288.12838259112877946088', 32, 1);
    t('688170000000', '688171056500.836968', 5, 1);
    t('668378677409000000000000000', '668378677409178263584469413.4119382946333552872293629069351353619031', 12, 5);
    t('166689.036233498384422162419', '166689.0362334983844221624189232706631', 28, 2);
    t('4440477952100000000000', '4440477952093354793071.0642852880', 11, 4);
    t('21725500000', '21725466408.96002464512027956176555', 6, 2);
    t('6486912766399000000', '6486912766398896055.2694660387537033966950978215711974427', 13, 4);
    t('2531049265993635941204.8115497719868868552', '2531049265993635941204.8115497719868868552376965243031061267315', 41, 4);
    t('8013246779880341074682.38148225', '8013246779880341074682.381482245216124209128397739167751379175', 30, 5);
    t('221047326862135721634547894400062412.539', '221047326862135721634547894400062412.539730289', 39, 1);
    t('822011625878400000', '822011625878311406.657260352355996731439057', 13, 2);
    t('812487237169705923908139353934762600', '812487237169705923908139353934762599.7170', 34, 6);
    t('36949082750290463320834741148588870.079506248563170996770993132903166375', '36949082750290463320834741148588870.079506248563170996770993132903166375', 73, 5);
    t('14654525658339993235849274000', '14654525658339993235849274255.9504278862', 26, 1);
    t('55916225576000000', '55916225575100248.4465036489601128811266131281106', 11, 0);
    t('645518883110973093861805823.4491902534164703759147542113679', '645518883110973093861805823.4491902534164703759147542113679', 61, 0);
    t('358598255669734.629', '358598255669734.6298059383258333892689316', 18, 1);
    t('740689207935700', '740689207935711.657417237564530498407298628092', 13, 5);
    t('2830529153357390140000000000000000', '2830529153357390143239464329358746.950876196', 18, 5);
    t('1650000000000000', '1640840867686293.5647134264723812966337372206061', 3, 0);
    t('3606602119459549783260469.4762536660577676748718446', '3606602119459549783260469.476253666057767674871844566774649', 50, 0);
    t('34291131952000000000000000000000000', '34291131951906424167228785662961140.192589712252256', 11, 5);
    t('72381861796434900000000000000000', '72381861796434868764304448966897.47175467294', 15, 2);
    t('2054866028540036166650062.44980397881578534902286742162263', '2054866028540036166650062.449803978815785349022867421622635', 57, 3);
    t('92117059982082500413900000000000', '92117059982082500413804787764240.342373111611', 21, 0);
    t('227757102524210000', '227757102524206134.7488778893034640337639597701932841', 14, 0);
    t('531000000000000000000', '530836544672422923128.37103655531278312510694460700374', 3, 2);
    t('3470953271456.204767973646214538028854143098', '3470953271456.204767973646214538028854143098029006', 44, 5);
    t('634780192130000000000', '634780192127622499179.6666629866288727547496211731649', 11, 0);
    t('82775105467', '82775105467.3', 11, 5);
    t('664400000000000000000', '664355276421949863428.13080633286561', 4, 5);
    t('896246695670597268763713413622054.8', '896246695670597268763713413622054.7260540457119630968841746582053660546600', 34, 2);
    t('108773555790900501222000', '108773555790900501221912.95245773146723197947280', 21, 2);
    t('709026210000000000', '709026204611640483.527807', 8, 0);
    t('280089643422.15486639434841', '280089643422.15486639434840995236', 27, 6);
    t('435677856419172037.7713', '435677856419172037.771381492971805965861162213211851502', 22, 1);
    t('67887322661701450.0863168511307393301059381091747', '67887322661701450.086316851130739330105938109174744', 48, 6);
    t('4725922910000', '4725922912709.1', 9, 4);
    t('55387481050000000000000000000000000000', '55387481055183351283811506946400256911.58800507610759226', 10, 1);
    t('49517621748860734873747747226167058877.859738405379132620740501604921931143', '49517621748860734873747747226167058877.859738405379132620740501604921931143', 76, 2);
    t('338071892085911928210836137.2622', '338071892085911928210836137.2621114265620234093003898333476553', 31, 2);
    t('28071746.041', '28071746.04050231906399068824049855052', 11, 2);
    t('6401124349403577194000', '6401124349403577193520.674349425297036232949594635372', 19, 6);
    t('3210000000000000000000', '3213130913892933503859.0', 3, 4);
    t('11788794020063416666254793.588779527838312578557', '11788794020063416666254793.58877952783831257855664607539', 47, 5);
    t('25044914938067579447598459.5', '25044914938067579447598459.5988567', 27, 3);
    t('5264165273402488453497000000', '5264165273402488453497220557.37', 22, 1);
    t('4021947427.420883226087493324', '4021947427.4208832260874933239', 28, 4);
    t('27.14', '27.14440', 4, 4);
    t('4132353.199857435065006244633', '4132353.199857435065006244633', 30, 0);
    t('4709992940309486701548267.7893', '4709992940309486701548267.7893237031352848850837069789212257042', 29, 3);
    t('32195.73995683232747785538851', '32195.73995683232747785538850278', 28, 0);
    t('2634400000000000', '2634376271790061.1', 5, 4);
    t('7700690756441442245975276494001102370', '7700690756441442245975276494001102372.239754560423', 36, 5);
    t('67258800000', '67258849329.528208759232004458400756971189096', 6, 5);
    t('7.6459139234166537848', '7.64591392341665378480', 23, 0);
    t('6074310000000', '6074307530292.72276', 6, 2);
    t('6788', '6788.0864232', 4, 3);
    t('702651090978378470091133000000000000', '702651090978378470091133484098464530.7520391767071488447086763459428796560151', 24, 6);
    t('7375135702784906050277086486.46957273364826', '7375135702784906050277086486.4695727336482652946664166457', 42, 1);
    t('978379015765928486174503299000000', '978379015765928486174503298654060.9328556450384828007932529269648770780', 27, 4);
    t('4344139496.65337637540642121475999885', '4344139496.65337637540642121475999885366', 36, 4);
    t('7023223639062851239767.39218000883400212876503190216747', '7023223639062851239767.39218000883400212876503190216747', 57, 4);
    t('39028806962172428434100000000000000', '39028806962172428434126113166830097.903539621167', 21, 3);
    t('32740000', '32736500.27132593865128', 4, 2);
    t('494.6200868226863006514117239', '494.6200868226863006514117238274013', 28, 2);
    t('500000000000000000000000000000', '460404219362958335598880058394.948', 1, 2);
    t('842870000000000000000000000000', '842872069628293640848762648185.7971559009', 5, 3);
    t('288742463789903483800000', '288742463789903483772013.259141217915317359', 19, 2);
    t('32402681510000000000000000000000000', '32402681509069715290882768053895390.3595398660386468748431163102016325', 10, 5);
    t('92.1673', '92.167287238577553764149877617837', 6, 0);
    t('2886807097545505316737185093182143248.77918637171922337622305282536619643', '2886807097545505316737185093182143248.7791863717192233762230528253661964254023', 72, 4);
    t('7818659083147481.72', '7818659083147481.71621629998011423382107522384394', 18, 0);
    t('79032899425011532217017487.00309018693479317855364739715', '79032899425011532217017487.0030901869347931785536473971569', 55, 3);
    t('32476233948000000000000000000000', '32476233948178293667714073630381.10591126396335783075016579', 11, 3);
    t('994160835319296110000000000000000', '994160835319296115990219272262227.66322135', 17, 1);
    t('33384911722755217737364663588703455879.551680118', '33384911722755217737364663588703455879.551680117600480548527662590008563041', 47, 0);
    t('3011969224061.1470999732495908055', '3011969224061.1470999732495908055', 32, 2);
    t('52170603373183855596259572898429869.72049467357503651998020982', '52170603373183855596259572898429869.7204946735750365199802098263591748401233', 61, 1);
    t('61651635245952756088515000000', '61651635245952756088515050377.1', 23, 4);
    t('3000000000000000000', '3471646976941515796.042582', 1, 6);
    t('737681128944426963000000000000', '737681128944426962543922971079.5999', 18, 6);
    t('22990615593799039882356.515400932432881645088046995', '22990615593799039882356.515400932432881645088046995', 52, 1);
    t('767418348801.999710467210156153276755817655638', '767418348801.9997104672101561532767558176556376', 45, 0);
    t('271263779237', '271263779236.5409173405833038640', 12, 5);
    t('54005000000000', '54004942625818.6423102307', 5, 6);
    t('4743420198217669660158042.332105070778379226676', '4743420198217669660158042.3321050707783792266763', 46, 6);
    t('1824000000', '1823638067.10', 4, 2);
    t('19076971753809147138989434.0384', '19076971753809147138989434.038401782247915', 30, 1);
    t('493188912177087830605938067286566208.474449301262798437941227', '493188912177087830605938067286566208.474449301262798437941227', 60, 4);
    t('175755512234552760407269000000000000', '175755512234552760407269008772693139.709452300', 24, 6);
    t('5137.4815571807262033469391576418', '5137.48155718072620334693915764180107', 32, 6);
    t('5796085533.812833278541553', '5796085533.8128332785415528346224448', 25, 5);
    t('19891600', '19891565.98547122420919720021471158685691', 6, 6);
    t('15535955963086464620600000', '15535955963086464620600274.0077903615259341745656801', 23, 5);
    t('7783527067465801.079783', '7783527067465801.079783', 24, 1);
    t('763701468733387894090485803744152591.4234225515039', '763701468733387894090485803744152591.42342255150389198813954119637151790', 49, 6);
    t('75789965430953565000000000000000000000', '75789965430953564709112811165286524239.67851522080514', 17, 4);
    t('5825046305310370000000', '5825046305310374601656.4666416360617', 15, 3);
    t('759259', '759258.1407870491180497', 6, 0);
    t('405659100000000000', '405659142763807066.9257887', 7, 1);
    t('8729968693647731984668867.5878333758746501', '8729968693647731984668867.5878333758746501', 43, 3);
    t('21528896409196004366', '21528896409196004366.1538430057157', 20, 1);
    t('1596973523474903.975798723416636', '1596973523474903.975798723416636140', 31, 6);
    t('7898509157451433814804238341.1077', '7898509157451433814804238341.107678042083024499089309', 32, 0);
    t('68179224091046745649410876014000', '68179224091046745649410876013426.6767', 29, 2);
    t('456747973150569.1094783998420071171', '456747973150569.1094783998420071170538030761710803295327', 34, 2);
    t('6852643.374076972180904', '6852643.37407697218090420490699033094', 22, 3);
    t('863563956510000', '863563956509100.72', 12, 2);
    t('341469943734916050000000000', '341469943734916050853727977.815488837706531204108082141', 18, 3);
    t('614507811297624.24668288929171', '614507811297624.24668288929171', 31, 2);
    t('91466630000000', '91466633937966.59114749152693885536602393032465395', 7, 6);
    t('460275943642', '460275943642.01', 12, 4);
    t('92283800000000', '92283833614982.49008290', 6, 6);
    t('189393326439226081.45814', '189393326439226081.4581428302', 23, 5);
    t('47863345130000', '47863345136537.2355040193131578', 10, 1);
    t('12343923099.014145618', '12343923099.0141456179645165401412793012216645155', 20, 6);
    t('158609989298386894728896676274592.5115003', '158609989298386894728896676274592.51150038426654871852', 40, 3);
    t('590346188000817716791939408770.1', '590346188000817716791939408770.04331069471279465667683791189024275', 31, 2);
    t('4545145597940000000000000000000000', '4545145597940213575674802007050467.8722805275831088812143322735725158', 13, 1);
    t('71161899725630738277084772318195323.758535367687854', '71161899725630738277084772318195323.7585353676878541127', 50, 3);
    t('829198777285586367377505514374.35042233965813', '829198777285586367377505514374.35042233965812919753', 45, 0);
    t('84269752601810028316403.817634102507607703', '84269752601810028316403.8176341025076077031333771666343585677566', 41, 1);
    t('4201225814021720848650000', '4201225814021720848652179.524152583918753125284358', 21, 5);
    t('68770831512385.491332229', '68770831512385.491332228514771665581586824982607', 23, 5);
    t('677946544849778324700000', '677946544849778324670053.485', 19, 0);
    t('30081138918523833162014.890626682144618', '30081138918523833162014.89062668214461833', 38, 6);
    t('6812626600000000000000000000000000000', '6812626646769107032327052128751748200.928615331608101628556359790061118241942', 8, 5);
    t('48581541131413383539911000000000000000', '48581541131413383539910889640323417491.5034591060199283960958655753110821489', 23, 5);
    t('28060257876.17697527', '28060257876.176975269241812403041857333', 19, 6);
    t('1.89393420948351132', '1.89393420948351132', 20, 5);
    t('72184719.2', '72184719.249486', 9, 5);
    t('4371363071372493960170467708060.2208067634649561285172662968872119548', '4371363071372493960170467708060.2208067634649561285172662968872119548', 70, 6);
    t('200.900330772054', '200.900330772054111626', 15, 1);
    t('706967355366834711950946655262685.544145056337', '706967355366834711950946655262685.54414505633667', 45, 5);
    t('6008113321510000', '6008113321509556.7193409219896487052919182579442', 13, 0);
    t('7288017017117000000000000000000', '7288017017117574193046750482188.24', 13, 1);
    t('5416889.25123830698407746679397202407', '5416889.25123830698407746679397202407', 37, 4);
    t('4746036947432988161127572977531233631.3289826795778545698', '4746036947432988161127572977531233631.328982679577854569814028984', 56, 3);
    t('637976582097611335227300', '637976582097611335227308.48', 22, 6);
    t('43183697.08742', '43183697.08742843567558924160723464435455555', 13, 3);
    t('99265705644225754922000769994740000', '99265705644225754922000769994749024.27489243084452355944914498905525', 31, 3);
    t('70615587289000924342020', '70615587289000924342019.764873', 23, 4);
    t('7286393000000000000000000', '7286393081444379737006140.3152399259092', 7, 1);
    t('66500.550247265044201979', '66500.550247265044201979', 25, 1);
    t('638128794000000000000000000000000000000', '638128793852566015974550237140413074884.3704245315138515368081974756492331204', 9, 0);
    t('639776995422431707274507021858200000000', '639776995422431707274507021858167095611.761849127115582', 31, 0);
    t('537588229831910677193710000000000000', '537588229831910677193717328511374328.17', 23, 3);
    t('1506210893684596381932000000000', '1506210893684596381931638269358.579078256019128351', 22, 4);
    t('59776338066547033313684981700000000', '59776338066547033313684981778964007.490093827893072065033397456883331', 27, 3);
    t('139942074848502442550593497179484944000', '139942074848502442550593497179484944262.17648232439178536', 36, 6);
    t('732646867527591037827715599902368960725.7806183', '732646867527591037827715599902368960725.780618347684', 46, 1);
    t('8091216574951011485757278800', '8091216574951011485757278877.9897212259420894', 26, 1);
    t('249586363414570753873974550733949620', '249586363414570753873974550733949616.836760458957752', 35, 2);
    t('379441003903904559374686536.598228306858', '379441003903904559374686536.59822830685840825719661757', 39, 3);
    t('9915418494187602346.36244933165720336614126', '9915418494187602346.362449331657203366141257319432096', 42, 6);
    t('7343279396546.8764419088206077', '7343279396546.876441908820607653686805518353', 29, 5);
    t('1900345899500000000000000000000000', '1900345899510547961856684263909525.787453311501255342930242525263284350366', 11, 3);
    t('44331159166272.9471454579635857172', '44331159166272.9471454579635857172', 36, 5);
    t('9501280312623990434908652485.4', '9501280312623990434908652485.40396', 30, 3);
    t('492920218032766973184509224720320593584', '492920218032766973184509224720320593584.36629645569199202522418', 39, 1);
    t('467521.0226496833105611752', '467521.02264968331056117515563122993101', 25, 6);
    t('9595083379034524606164.71059984', '9595083379034524606164.71059984', 31, 0);
    t('9970000000000000000000000', '9965482043965987598637733.5583210749501409728', 3, 2);
    t('1592338.971883076', '1592338.971883076', 17, 4);
    t('7333767695306400000000000000000000', '7333767695306382568470511272838909.38796720966964950191844707080349251', 14, 2);
    t('32800000000000000000', '32876911196847202475.291', 3, 1);
    t('74242348398312192291428.673', '74242348398312192291428.673423842088606879', 26, 6);
    t('8386486108186860095841440.11730856', '8386486108186860095841440.117308558873442559935861736285736374447', 33, 2);
    t('80947810000000000000000000000', '80947807162749742411836443948.706994282953', 7, 4);
    t('9195641658476168463768340936.76188994861', '9195641658476168463768340936.761889948610197959977737786402852906886', 40, 1);
    t('4578927381348845.5985555207901936', '4578927381348845.598555520790193572751522533', 32, 5);
    t('90465052283290169919908879000000', '90465052283290169919908878868468.905908671955794022459213015110874', 26, 6);
    t('54547701007984170000', '54547701007984169975.91223209002505925', 17, 2);
    t('979787521006681794473550', '979787521006681794473554.3006822695304752748535940310897632701', 23, 1);
    t('7622842626897436617.07', '7622842626897436617.06688', 21, 4);
    t('653093024482578.796', '653093024482578.7951857714517102028500161600627', 18, 0);
    t('9833966314261094003.018695', '9833966314261094003.018694758385422', 25, 5);
    t('15830000', '15829083.057524504524234', 4, 2);
    t('961750000000000000000000000000', '961747772066287628937112422597.3817215646', 5, 2);
    t('355424801500000000000000000', '355424801540457063574782862.12697540776577178037467419053183354', 10, 1);
    t('157000000000000000000', '156481812617478598304.73178', 3, 2);
    t('5861785294.796', '5861785294.79661', 13, 1);
    t('182225501209978000000000000000000', '182225501209977688205618014373798.0', 15, 2);
    t('3738345000000000000', '3738345405763729028.7752042632909875432389295604576675688599', 7, 3);
    t('41084850583049628980622145057509398548.8190604', '41084850583049628980622145057509398548.8190604843293625567811059253', 45, 3);
    t('53021470000000000000', '53021463475466030072.41324254204200289', 7, 0);
    t('4422813108757117230000', '4422813108757117235733.368525730128833856317079780867', 18, 1);
    t('189232967180.7056416660639', '189232967180.7056416660639547537942134250776280649', 25, 1);
    t('3224251146374659306359.77314112838955387983728605588484463873', '3224251146374659306359.773141128389553879837286055884844638733', 60, 4);
    t('8600000000', '8597184202.7', 2, 0);
    t('3.115106', '3.115106696315723132076001869', 7, 1);
    t('7087892306431595280582615538940026200', '7087892306431595280582615538940026236.40864513918323835', 35, 1);
    t('8779033903291644915487146794281306357.13755271835304049285254286', '8779033903291644915487146794281306357.13755271835304049285254286', 65, 5);
    t('8839716025004465601060469108336560376613.076951', '8839716025004465601060469108336560376613.07695120337636053497434023685410433694', 46, 3);
    t('4275536079721224.6939983771317331340817', '4275536079721224.6939983771317331340817', 41, 5);
    t('4119369292321042699138254863900', '4119369292321042699138254863925.916', 29, 6);
    t('2000000000', '2490819916.3', 1, 4);
    t('347020000000000', '347022391133103.69', 5, 5);
    t('32.8177254975495761052855062741', '32.817725497549576105285506274021', 30, 0);
    t('6410000000000', '6414991659617.62580', 3, 1);
    t('7813884249729159606824.73899541339089931882', '7813884249729159606824.73899541339089931882289865313148715', 42, 4);
    t('1307698104295029293564444728385517996459', '1307698104295029293564444728385517996458.679583748', 40, 4);
    t('79199269306160169700057625859500.75', '79199269306160169700057625859500.748498500907905913442887488721506', 34, 5);
    t('1774284784107000000000000000000', '1774284784106843222936411292325.8095840661997998', 13, 4);
    t('96180958515851420047364059533.45', '96180958515851420047364059533.4500', 34, 3);
    t('5096577600000', '5096577646405.002009274998306919759095', 8, 4);
    t('35495991462515089667988035168084.069684665058835048922', '35495991462515089667988035168084.0696846650588350489211203398', 53, 2);
    t('1301540485567727308005102.653073598273814', '1301540485567727308005102.653073598273814352683', 40, 1);
    t('3963594790294442000000000000', '3963594790294441943327812598.375812877531', 16, 0);
    t('64594305402307716771208391189249.76', '64594305402307716771208391189249.76', 37, 3);
    t('7700000000000', '7665889554206.06357537958563530747169', 2, 2);
    t('59329033', '59329033.5495', 8, 3);
    t('332438298511038453550000000000', '332438298511038453547200947956.2192981469502292273494', 20, 2);
    t('8397858102848942599.34017105558993239', '8397858102848942599.340171055589932385598244796639', 36, 2);
    t('576909126.83784127994', '576909126.83784127993456201111155339', 20, 0);
    t('393961250.7006808533185915831665', '393961250.70068085331859158316649952', 32, 0);
    t('72241336442883432977410072016789326.85909917403', '72241336442883432977410072016789326.8590991740269249902167680', 46, 5);
    t('3278426934943396285754165700000000', '3278426934943396285754165732885483.06596256396074260171', 26, 1);
    t('736543296510453987773.605509008', '736543296510453987773.605509007181332059349490', 30, 2);
    t('110000000000000', '111495776833142.0', 2, 4);
    t('45340022682492689648807110000000000000', '45340022682492689648807109769143470012.1084149526', 26, 6);
    t('87000000000000000', '87217962600936114.8854899406121', 2, 4);
    t('43550000000000000000', '43546349704633032746.317726010358', 4, 5);
    t('38674592.062075641', '38674592.0620756406895194200906791200545', 17, 4);
    t('8799153537641000000000000000000000000000', '8799153537641003070643136611836476962663.45959762796159306483279783213626', 14, 6);
    t('9034306.944996082', '9034306.944996082', 18, 5);
    t('66909.7030016', '66909.703001606016261', 12, 6);
    t('456161.7', '456161.7', 9, 1);
    t('73489941078390234.4964486419890071038', '73489941078390234.496448641989007103838414836851921306436', 36, 3);
    t('1923645556', '1923645556.0', 12, 6);
    t('4453480502942468513972218347117175676.8556856', '4453480502942468513972218347117175676.855685593499', 44, 5);
    t('4526083791053541695.3115884260861412696242231283', '4526083791053541695.31158842608614126962422312832814802', 47, 6);
    t('1735294034300000', '1735294034352688.8004588017016288793464854923210346394', 11, 1);
    t('4081195502300000000', '4081195502302699823.4560841368861046', 12, 5);
    t('49781669423198412000', '49781669423198411198.0', 17, 2);
    t('109995899280702335182325284264923.81916175', '109995899280702335182325284264923.8191617472824694092', 41, 2);
    t('47408473000', '47408473435.0', 8, 6);
    t('932978086371834995409298037000', '932978086371834995409298037784.5640', 27, 1);
    t('1306443435440000000000000', '1306443435433698711710394.340932765590083799', 12, 0);
    t('1827686396871299439394788932673263747.66', '1827686396871299439394788932673263747.669853927728407512178192674886688', 39, 1);
    t('4900000000000000000000000000000000', '4841121706354427548419072490466753.383548126030428119', 2, 0);
    t('24283467000000', '24283466617739.8', 8, 4);
    t('767105665646623001939968053926.728417', '767105665646623001939968053926.72841624625344', 36, 0);
    t('25517505.034089', '25517505.034088176', 14, 0);
    t('737440405369700000000000000000000000000', '737440405369614993849903124666404118624.819154016104968389481992305318545394', 13, 0);
    t('6844183807190466.846621817', '6844183807190466.8466218172882018', 25, 1);
    t('1.371455056656', '1.3714550566563', 13, 6);
    t('99239.74535303', '99239.7453530349', 13, 1);
    t('7324556628.008826655544013705536', '7324556628.008826655544013705536', 33, 6);
    t('61528629766932.383', '61528629766932.383063', 17, 1);
    t('26177', '26176.898365', 5, 0);
    t('73299598.949237274603980082155776829803522987', '73299598.949237274603980082155776829803522986507', 44, 0);
    t('62226313923410956793306.70191135384145114489242', '62226313923410956793306.701911353841451144892428767886226239', 46, 3);
    t('221374620616798288514773053908832701.91219881948882', '221374620616798288514773053908832701.912198819488822318', 50, 3);
    t('52915031594003589732.226', '52915031594003589732.225583247394680000', 23, 0);
    t('8051342770408498662211166314.12978623482', '8051342770408498662211166314.129786234823161725325', 39, 4);
    t('979075431491845287238326307986108000568.09', '979075431491845287238326307986108000568.09', 44, 3);
    t('2597398855201', '2597398855201.0', 15, 3);
    t('708543766436276721865041805.46', '708543766436276721865041805.459', 29, 5);
    t('237900000000000000', '237877583040690220.9179286635509916910112201173265', 4, 2);
    t('8481743052179513077951505680106275241657.785', '8481743052179513077951505680106275241657.784404362894', 43, 2);
    t('590315856670000000000', '590315856660996059247.80123435997708', 11, 0);
    t('67623269958079021270', '67623269958079021264.10087752556931149642358223252871', 19, 0);
    t('24260141371575324.036549326388228', '24260141371575324.03654932638822808911077', 32, 5);
    t('311128.437029668', '311128.437029667773213203', 15, 2);
    t('4683981205998904101000000', '4683981205998904100724696.9260167298916', 19, 6);
    t('87300000000000', '87290083695691.788815500880174631185242225091', 4, 2);
    t('582918471744892863460969538385683350294.5', '582918471744892863460969538385683350294.5164646080758340043867868', 40, 6);
    t('1300000000000000000000000000000000', '1305597128575923563913433734264637.47', 2, 1);
    t('52953698620966184858676280.7507', '52953698620966184858676280.75070283177973902859681856673421', 30, 6);
    t('1186231136878568100000', '1186231136878568118839.79', 17, 6);
    t('99115508905461000000000', '99115508905460018708631.933878813825297873202612011592', 14, 2);
    t('5875456974377.312755094', '5875456974377.31275509394', 22, 0);
    t('81216871.9', '81216871.87101287038603250', 9, 5);
    t('42034440262237500', '42034440262237545.695288671225005', 15, 3);
    t('25943689905954849470983646104638', '25943689905954849470983646104637.583166054397250511', 32, 6);
    t('224106987515664588637177222760942846293.434541647165052159796141', '224106987515664588637177222760942846293.434541647165052159796141052647407100', 63, 4);
    t('9131673888785493.22342384', '9131673888785493.223423840233', 24, 3);
    t('0.832476711281182', '0.832476711281182', 15, 1);
    t('1513760378759061.7900697', '1513760378759061.79006969374', 24, 0);
    t('523353000000000000000', '523353027287352104649.339549955694097681762331045556', 7, 3);
    t('91853629.8', '91853629.8994454858437050074834429708', 9, 1);
    t('2671543889233943884120198595000', '2671543889233943884120198595468.40741754782', 28, 3);
    t('41586946834.233970576', '41586946834.2339705761110305', 20, 3);
    t('359874415570455742645391.490204462', '359874415570455742645391.4902044614741', 33, 2);
    t('8440291.2067646', '8440291.20676451822534914026', 14, 2);
    t('7887193371077893796899405294168', '7887193371077893796899405294167.12304333192877747156', 31, 2);
    t('7771425937460000000000000', '7771425937457182166344979.4281554195120574592916886', 12, 0);
    t('2398840124250526657118090000000', '2398840124250526657118093418514.586884', 24, 5);
    t('8034.999240128930545865837400195083831036', '8034.999240128930545865837400195083831036175', 40, 5);
    t('53350770282401151022359684479.4150017265871', '53350770282401151022359684479.4150017265871', 43, 0);
    t('1836.2901716965', '1836.290171696448013', 14, 0);
    t('53900000000000000000000000000000000', '53811973569266486547521534194682815.775061310240830', 3, 2);
    t('97195339145807133836581649.49', '97195339145807133836581649.498461817380505159636792273592383619540', 28, 1);
    t('8781354885.40521396225001', '8781354885.40521396225001', 26, 4);
    t('4195533919413670199233591760983063.044588578956739921516049721165233925', '4195533919413670199233591760983063.0445885789567399215160497211652339249', 70, 0);
    t('4.2', '4.2274679655', 2, 4);
    t('2000000000', '2356157692.863500888313706747', 1, 4);
    t('700000', '703438.4086', 2, 6);
    t('8579776801536797847516153209877562533676.8750685231437489210524230987', '8579776801536797847516153209877562533676.8750685231437489210524230986564420', 68, 2);
    t('51867617908268321110783346204580461212.281', '51867617908268321110783346204580461212.2813618556831', 41, 5);
    t('649393249438816557050596861.546597478389602', '649393249438816557050596861.54659747838960146185981668055330793', 42, 0);
    t('30000000', '33607645.0', 1, 5);
    t('74088713747862470000000000', '74088713747862475663611878.024940913', 16, 3);
    t('43570349798700000000000000000000000', '43570349798755872271871551924166737.593446362653813853206404179862697301', 12, 1);
    t('2218618077613108196308831974574040', '2218618077613108196308831974574038.694370405197', 33, 5);
    t('926348200000000000000', '926348175950548532418.938252947508488691379699853956756365', 7, 4);
    t('4984339030732470000000000000', '4984339030732464404596140545.0', 15, 2);
    t('741000000', '740663984.67441413120010', 3, 4);
    t('71970898094536046411.87467194369', '71970898094536046411.87467194369', 34, 6);
    t('240060872648074297074288841047227088.297024818734423', '240060872648074297074288841047227088.29702481873442394', 51, 3);
    t('59347917140216675110698714.41681', '59347917140216675110698714.41681904399541', 31, 1);
    t('153600000000000000000', '153652728557017061675.717105590', 4, 1);
    t('1.4', '1.41304736793703701518', 2, 1);
    t('5185400686562240770312869.33382785309533', '5185400686562240770312869.33382785309533191', 39, 5);
    t('79103755569332', '79103755569332.0', 17, 4);
    t('63.788087373094917842', '63.78808737309491784170631103', 20, 0);
    t('610000', '610455.675852217079461275640962760626810328911', 3, 3);
    t('1079.6839', '1079.6838846', 8, 0);
    t('557340000000000000', '557335816675082302.5626727537432964165888656942302889', 5, 0);
    t('522352103517.082550293858237196990189282459', '522352103517.08255029385823719699018928245955', 42, 3);
    t('69744392349703568.192187065478739', '69744392349703568.1921870654787393', 32, 6);
    t('491259800000000000000000000000000000', '491259748670478216050872371655823803.80177296584382248634041159227198266536', 7, 2);
    t('109339.854731965898436', '109339.8547319658984355', 21, 2);
    t('50900.421375874', '50900.4213758735466011242124890702667371910', 14, 4);
    t('1084167026217154607803778000000', '1084167026217154607803777548500.57719394868335403780789892598384', 25, 2);
    t('82512334536860276958376415886509279926.9', '82512334536860276958376415886509279926.947', 39, 4);
    t('64635.225676303', '64635.2256763031707', 14, 4);
    t('4013599375010487721983844', '4013599375010487721983844.0', 28, 3);
    t('418763602200000000000000', '418763602168088402361749.0992331628682761962322558413806655975', 10, 0);
    t('35121245874000000', '35121245874348112.804', 11, 6);
    t('2091072600487459492037900', '2091072600487459492037943.032274313824', 23, 1);
    t('786944012509200564187289859114332164.886', '786944012509200564187289859114332164.88597776776032254703006551', 39, 5);
    t('5510012359536586295525090000000000000', '5510012359536586295525082450888793459.2282826131843152014995', 24, 0);
    t('5074011888628821330000000000', '5074011888628821328542802558.3873605619910448429', 18, 5);
    t('101141660925871596132818016582', '101141660925871596132818016582.47377189240025859565460187997456018784', 30, 5);
    t('24403282227517.471966', '24403282227517.471966738302486934', 20, 1);
    t('848995591995663049509.43385779093167891113654066569203457277', '848995591995663049509.4338577909316789111365406656920345727643', 59, 0);
    t('2000000', '1812639.3', 1, 5);
    t('52551956.412107848295552852356376301662', '52551956.41210784829555285235637630166177117909', 38, 2);
    t('82716.10215118992200146540494', '82716.10215118992200146540494', 28, 3);
    t('70000', '70166.0', 2, 4);
    t('56047460146555000', '56047460146554739.7257883678943424269159042630401', 14, 6);
    t('712998066121562772546229433804494.835878090588297757461267671326', '712998066121562772546229433804494.8358780905882977574612676713264196446', 63, 5);
    t('99387111276248408952831136918082110', '99387111276248408952831136918082105.45987990177743738332', 34, 2);
    t('114.434129047', '114.43412904789684', 12, 3);
    t('7977961400548494050.84728662093908732719343685', '7977961400548494050.8472866209390873271934368531824', 45, 5);
    t('100000000000000000000000000000000000000', '97272558382697322666784133923630728650.015113557822546220026', 1, 2);
    t('6226986078745224747403247677799730124.709963671990586', '6226986078745224747403247677799730124.709963671990586316', 52, 5);
    t('379938730159133994554623349000', '379938730159133994554623349277.142801570416444', 27, 5);
    t('2712371636875.8876255166807730622135941', '2712371636875.88762551668077306221359415217750', 38, 3);
    t('7646133661486233.40005164399246921700070707', '7646133661486233.400051643992469217000707068134895947', 42, 6);
    t('6224260477700253782615479279421593031.219613357040709677', '6224260477700253782615479279421593031.21961335704070967692894128496', 55, 0);
    t('31.28263044', '31.2826304374279193', 10, 5);
    t('5338587100000000', '5338587094595692.189029309718883067557794142', 8, 0);
    t('7.059669', '7.059669', 7, 0);
    t('14393752968082197146780000000000', '14393752968082197146783367135371.257115956149362191764183', 22, 5);
    t('45946102668923.8512634475473800661143274', '45946102668923.85126344754738006611432737562882344', 39, 5);
    t('7389100000000000000000000000', '7389165426229872560402668498.67992315851321516587001595983167724072', 5, 3);
    t('467854849853345030397732540000000', '467854849853345030397732536724972.99245862', 26, 5);
    t('413958038223115250663874661218522.37896756434331643800654262', '413958038223115250663874661218522.3789675643433164380065426275', 59, 3);
    t('299266671297334058.474402301545', '299266671297334058.474402301545', 31, 0);
    t('1496630', '1496627.4799482274357047323140', 6, 4);
    t('116363400658592.84246392541', '116363400658592.8424639254051688509124968', 26, 0);
    t('81112461790.297', '81112461790.297', 17, 3);
    t('9342092648950356525137.0297', '9342092648950356525137.02972717415', 26, 5);
    t('6274493534.71318536138744291506374', '6274493534.7131853613874429150637355899', 33, 2);
    t('3490113339968166888558774127650067816063.53120129448345142410619243580148489', '3490113339968166888558774127650067816063.53120129448345142410619243580148489', 76, 0);
    t('523257182156289968600000', '523257182156289968695732.283726034208120502645993022154644', 19, 1);
    t('3893860000000000', '3893852788036514.88525944944', 6, 2);
    t('9745.887493', '9745.88749293', 11, 2);
    t('406174178400000000000000', '406174178494283010337060.5188965714995121791713772444', 10, 1);
    t('0.5056', '0.50564', 4, 6);
    t('405963232789318632.613', '405963232789318632.61300407300', 23, 6);
    t('1790829.9084993799', '1790829.9084993798690400', 17, 6);
    t('1640000000000000000', '1638700705317251432.49363', 3, 2);
    t('75000000000000000000000000000000000000', '75287555831753195069005206090058900444.09133190034400996', 2, 3);
    t('51432891', '51432890.7485', 8, 5);
    t('76497169892995002059311.07631021679438420096653', '76497169892995002059311.07631021679438420096653', 49, 0);
    t('59805658828550516480633954971623094.6200898815205326660379', '59805658828550516480633954971623094.620089881520532666037925', 57, 4);
    t('161000', '161772.432027496636124397858615', 3, 3);
    t('75269965075697558772957063876.83109049531256', '75269965075697558772957063876.8310904953125578225640304045483851605189', 43, 2);
    t('621051696362856602801738177194530565.112', '621051696362856602801738177194530565.1116884315975326394837239706328', 39, 5);
    t('90150000000000000000000000000000000000', '90154892309886766495783362582775544757.369674986765162774046626255864', 4, 4);
    t('258703411000000000', '258703410922153223.60475', 9, 5);
    t('3709704331642000000000000', '3709704331642244200712999.6170288032921256065731206685375', 13, 5);
    t('865554088000000000', '865554088558920675.7349605304', 9, 3);
    t('1701316772160.103', '1701316772160.1029', 16, 2);
    t('46745807992862580.7', '46745807992862580.720881779764764346805812', 18, 1);
    t('49585278347036000', '49585278347035695.782147031010', 14, 4);
    t('96713953824118.1821274868476422346040613626', '96713953824118.18212748684764223460406136258829', 42, 5);
    t('783956363906400000000000000', '783956363906410059509130968.96305467333', 13, 4);
    t('43105.844', '43105.843549344', 8, 5);
    t('829211298828600000', '829211298828668699.221294293842622484212438698', 13, 3);
    t('37196000000000000000000000000', '37196294573136385613340634306.290692298188', 5, 5);
    t('248295476114604010.896705382754283508', '248295476114604010.896705382754283508339', 36, 4);
    t('84423424502558134979280670603873000', '84423424502558134979280670603872746.2265033665276422', 32, 5);
    t('2809713489011123812961337990980603434581', '2809713489011123812961337990980603434581.12518685410537223733814208', 40, 4);
    t('172574.3584172', '172574.35841725', 13, 3);
    t('7924946249000000000000000000000000', '7924946249977573911672885156907431.067063211162', 10, 1);
    t('8683597943692398064100', '8683597943692398064161.851847698936750210418282976', 20, 1);
    t('69.0914431363288438564076', '69.0914431363288438564076', 24, 4);
    t('79793568589313600000000000000000000', '79793568589313596333612873530424427.4180837875', 16, 6);
    t('7796000000000000000000000000000000000000', '7796296719768579262285552574438319036946.4360634477727793909333566069238361773343', 4, 1);
    t('27635061002361998926997557652.6687317', '27635061002361998926997557652.6687316873552376374115178880871922', 36, 0);
    t('304.3574514667822228535474989881513617724837', '304.3574514667822228535474989881513617724837', 45, 1);
    t('6000000000000000000000000', '6483360962952099756240718.4896978424381725128583170', 1, 1);
    t('8363320899953874012.2017093850081725', '8363320899953874012.20170938500817246975904001', 35, 0);
    t('758220000000000000000000000000', '758222759936151260819259093895.951641916086729436609851298', 5, 5);
    t('91498547535920.817460894069121630583182731', '91498547535920.8174608940691216305831827313907783776485', 41, 1);
    t('439443228097661.31443614704', '439443228097661.3144361470460307561511611359578787723', 26, 1);
    t('8109363087368', '8109363087367.90', 13, 4);
    t('39200000000000', '39257648625692.931100828492169990746385930', 3, 1);
    t('5723631930549247182000000', '5723631930549247182333695.3423428113212', 19, 5);
    t('8433881399280554978486653792494.1300591089612', '8433881399280554978486653792494.1300591089611708995164658', 44, 4);
    t('1708568565656089316993482.277402', '1708568565656089316993482.2774019', 31, 4);
    t('5229509593.347218623510970353538086', '5229509593.3472186235109703535380857120887', 34, 5);
    t('1918864577748586967860098328985816319360.015606', '1918864577748586967860098328985816319360.015605932701466300341253760', 46, 6);
    t('0.43136323867307', '0.43136323867307', 16, 3);
    t('8334000000000', '8334082234405.0', 4, 5);
    t('84750531776787132592447241.568', '84750531776787132592447241.567748055400890104172602257091', 29, 2);
    t('584314287643000000000000', '584314287643393548021297.706880716265', 12, 6);
    t('405007058132550075901817.24593070292778979511020730744097283', '405007058132550075901817.245930702927789795110207307440972839', 59, 1);
    t('43519639353329164740780000', '43519639353329164740788998.3147762967743957252845965', 22, 1);
    t('739522336927466646469422217608536000000', '739522336927466646469422217608536045198.6705809', 33, 5);
    t('61464000000000000', '61463792830919931.7731178453635819621767147', 5, 6);
    t('4876027814244430271211000', '4876027814244430271210808.324126531366118118592674249657188519372', 22, 5);
    t('66961404930924220883562230746171276.0171355945', '66961404930924220883562230746171276.01713559450932479562015922933235003', 46, 1);
    t('936355668354895726619165200633505311.0647656769928606764806201308272', '936355668354895726619165200633505311.06476567699286067648062013082719660954', 67, 6);
    t('403.6152816916801960814', '403.61528169168019608140', 26, 4);
    t('84423144200', '84423144200.0', 10, 6);
    t('0.6', '0.6', 1, 6);
    t('76798153.6006117640951621404723', '76798153.6006117640951621404723', 31, 5);
    t('6000000000000', '5781186404903.094105632051316931', 1, 6);
    t('62.05369819740251195', '62.0536981974025119505927695', 19, 6);
    t('699654100000000000', '699654099414093693.7146750', 8, 0);
    t('7880083449938296.8746842', '7880083449938296.87468416408772550194129579542322', 23, 6);
    t('9171900000000000000000000000000000000000', '9171831128687558129281515394234036228735.4191677843157921813233', 5, 0);
    t('16015842434756146984896307254900000', '16015842434756146984896307254975181.5249921958691070261012', 30, 3);
    t('22649042169776063601825433445939200620', '22649042169776063601825433445939200617.4640059994619247282591561983069', 37, 4);
    t('240000000', '245023458.0', 2, 1);
    t('4118836670659770000000000000000000000', '4118836670659769679868567856549228309.86582346', 15, 4);
    t('9209289412000000000000000000000000000000', '9209289412001754306020014655415621208715.05157575857311168896403679274', 10, 6);
    t('7012510000000000000000000000', '7012514117099125191877135519.0088826968859106', 6, 5);
    t('8938889129571058777000000', '8938889129571058777078041.9', 20, 3);
    t('51279160696558160438071990.5', '51279160696558160438071990.5', 29, 6);
    t('96452920840860000000000000000000', '96452920840863725032547861450757.0176295547374322523339174625344238215646', 13, 6);
    t('755633364326626435077416331324559362.41380891894682475192996827910395', '755633364326626435077416331324559362.41380891894682475192996827910394132529', 68, 0);
    t('63551391234347500000', '63551391234347538885.682988985', 15, 3);
    t('6971219912319657298934153292162576681.227063', '6971219912319657298934153292162576681.227062997479376', 45, 0);
    t('387395548890747594110.73258141734354272251795', '387395548890747594110.7325814173435427225179401318886611', 44, 2);
    t('473.135', '473.135', 8, 0);
    t('927016096185582521171986421909107498.9207522053989857', '927016096185582521171986421909107498.92075220539898572321175518134876', 52, 6);
    t('63768.01000584590787574', '63768.01000584590787573744', 22, 6);
    t('38.3', '38.3', 6, 5);
    t('7491.6665620227', '7491.66656202274015', 14, 6);
    t('97092599105366000', '97092599105365993.26547422', 15, 2);
    t('3120148093.7523904', '3120148093.752390449694975030520582607', 17, 1);
    t('17327157417227558706354.17807392329604613728425', '17327157417227558706354.178073923296046137284251507', 46, 3);
    t('281000', '280167.7092864994917055775653688782707', 3, 2);
    t('32785854917768933220000000000', '32785854917768933224303039038.2058193020746240873968346994928712352422', 19, 6);
    t('35667912400000000000000', '35667912380813089868488.250584861', 9, 5);
    t('899596403053525089647480000000000', '899596403053525089647484759178959.916473021152397407891338', 23, 6);
    t('14745037187570022846107646644930000', '14745037187570022846107646644931325.38053731468723756458145869', 31, 3);
    t('33574797454691460275790.76', '33574797454691460275790.755045375', 25, 4);
    t('65031200000000000000000000000000000000', '65031262077165350238305531433379099671.7718568162727500206128207677082684945643', 6, 3);
    t('574632868512343128887331094723783700000', '574632868512343128887331094723783610067.144766298255005336980609781650438587645', 34, 2);
    t('9994870316709399533104.760022976329513397803886900936', '9994870316709399533104.760022976329513397803886900936967455407', 52, 1);
    t('716218617798116968092.18618', '716218617798116968092.1861730403641', 26, 0);
    t('795772272699962336654650000000000000000', '795772272699962336654654163110504958221.8071786483', 23, 3);
    t('5757.243049', '5757.243049547380169445193591241', 10, 3);
    t('59404487201000000000000000000000000', '59404487200792689654248137974864846.7', 11, 6);
    t('55074017152', '55074017152.0', 13, 0);
    t('0.6', '0.6', 3, 6);
    t('296219478315000000000000000000000000', '296219478314162392373713474016497135.6118597604677540263', 12, 0);
    t('276910694131263061322333280489468758116.9217861994071100288', '276910694131263061322333280489468758116.9217861994071100287521592758667135167110', 58, 6);
    t('3093430404.7', '3093430404.710017935458355451', 11, 6);
    t('900000000000000000000000000000000000', '965458260348828828944266573842209844.3732258429238965164197273600', 1, 3);
    t('66262286.9409504746706274365892587037797', '66262286.9409504746706274365892587037797427795120', 39, 3);
    t('22354062184079842644000000000000', '22354062184079842643735834447606.6800330412324887841685318200', 20, 5);
    t('26183218000000000', '26183217460464198.97', 8, 2);
    t('237614400000000000000000000000', '237614372017734708285810241221.42814134245888', 7, 4);
    t('825450494959282667497344701413995059.2723579093615742', '825450494959282667497344701413995059.272357909361574170381810543', 52, 2);
    t('93748060.528069293698504863657297', '93748060.5280692936985048636572966744', 32, 4);
    t('69626272309081096760583959976183200000', '69626272309081096760583959976183182825.3667872', 33, 4);
    t('1979.44509351627108093', '1979.445093516271080928575243', 21, 6);
    t('4415110628241252823308652816777297394.3942029367', '4415110628241252823308652816777297394.39420293660365458839', 47, 2);
    t('779231418794.449031966175634', '779231418794.4490319661756340025266', 27, 3);
    t('50', '45.69933296', 1, 2);
    t('6079.048887541962107868623935', '6079.048887541962107868623934532226316337798', 28, 5);
    t('93529472100000000000000000000000000', '93529472186320534415761296507138470.905280067059921116429144824', 9, 3);
    t('29898706650888894766171.6228944331', '29898706650888894766171.6228944331', 33, 1);
    t('922469795490253433535679091.062088', '922469795490253433535679091.062088290895805005314002547508', 33, 5);
    t('886600', '886571.60853335895055712447461273388', 4, 2);
    t('5609573962001812216992069259.930151', '5609573962001812216992069259.930151154912844425540631', 34, 6);
    t('45007044', '45007044.2086008047991042816131308872682', 8, 4);
    t('515688608128573684421179114100089113548.5257221606', '515688608128573684421179114100089113548.5257221606412316163084178774706252637', 49, 6);
    t('33081983026436645300000000000000', '33081983026436645256794953512623.698898733309834993545634', 18, 2);
    t('4447970000000000000000000', '4447976713289303679880070.9025261589467531117050172199', 6, 1);
    t('3903539864331348354864.765499107422', '3903539864331348354864.7654991074223411506949490883', 34, 3);
    t('984.025831747975455', '984.025831747975454838916188', 18, 5);
    t('581869610680217033571716502000000000000', '581869610680217033571716501999105429150.18648056812383517110349', 29, 4);
    t('595536346220703901411504802959406295.42383', '595536346220703901411504802959406295.4238391852814166902374770866016', 41, 3);
    t('94148126897556706760500284224.99723009363881145', '94148126897556706760500284224.9972300936388114498128', 47, 4);
    t('9715291106439557879466836795142517.781731033356111646', '9715291106439557879466836795142517.781731033356111646', 55, 4);
    t('82188353813971708733422689598227279.282476114041612077181694', '82188353813971708733422689598227279.282476114041612077181694', 62, 1);
    t('6294691074225860001965900000000', '6294691074225860001965899922418.89704561203607759919', 25, 0);
    t('6269698242304637000000000000', '6269698242304636020015369132.78424538515870397917428', 16, 0);
    t('4112676513272614989660000000', '4112676513272614989662611630.0', 21, 3);
    t('963033817.6', '963033817.616305319473', 10, 6);
    t('80246657958668073322316.6486987566008799223446172515708423915', '80246657958668073322316.6486987566008799223446172515708423915', 60, 2);
    t('693077517345181781403.440159259', '693077517345181781403.440159259000711920300899046684517345722', 30, 3);
    t('5499.7542', '5499.7541982143424097727706267067904033811', 9, 5);
    t('137679905313.5', '137679905313.50920005385380998167873840', 13, 5);
    t('39041690890014680359537720.467', '39041690890014680359537720.467851777573801', 29, 1);
    t('549542696010421538881075566508.71802045606183083628', '549542696010421538881075566508.7180204560618308362717637092511485434996', 50, 0);
    t('134997.142', '134997.142', 12, 2);
    t('97062841', '97062841.0', 9, 5);
    t('7716229142748262412.2182892204301', '7716229142748262412.2182892204301', 33, 6);
    t('298055382083161367153757.670234', '298055382083161367153757.670234165956730', 30, 4);
    t('4487.65135387719123626', '4487.65135387719123626', 21, 3);
    t('450871919575419788029382449.36246724734281790975668141079057', '450871919575419788029382449.3624672473428179097566814107905633864', 59, 0);
    t('66509604712294030927000000000000', '66509604712294030927705014933920.411161820552972509760891713474171187', 20, 3);
    t('4437840000000000000000', '4437848923500848221297.082710339692399', 6, 1);
    t('4000000', '4394457.793', 1, 1);
    t('24944720316894000000000000000000000000', '24944720316893358451636954228782192382.9047', 14, 2);
    t('929764221522108318487689.31', '929764221522108318487689.31', 29, 5);
    t('1916615685096.0437914', '1916615685096.04379143521812122277153112384491318036', 20, 6);
    t('90000000000000', '91902079995852.925006180666828', 1, 4);
    t('8000000000000000000000', '7208784027522887444672.11413061677955913612838834837', 1, 0);
    t('1167057364346164000000000000000000000000', '1167057364346164053938715478441117766069.7831864260947057063729556843646017', 17, 3);
    t('32604860504700', '32604860504745.860568239', 12, 1);
    t('758236232476453121251900', '758236232476453121251805.180883075117527428751196305141453', 22, 0);
    t('15778845903952108694027199.865720581665', '15778845903952108694027199.8657205816647482665', 38, 5);
    t('71836448621951813916.608168680984106761619715465365353', '71836448621951813916.6081686809841067616197154653653533', 53, 4);
    t('4571.2608729831649246553', '4571.2608729831649246553236386243949999812', 23, 3);
    t('671646112127.9629339711', '671646112127.962933971014753739203614331', 22, 2);
    t('51212124.114306691493', '51212124.11430669149276460', 20, 4);
    t('26583009145983740841060.671451973166754228836197273523562862', '26583009145983740841060.671451973166754228836197273523562862', 60, 1);
    t('516221626290783121176276857256250861123.72716427904973377592', '516221626290783121176276857256250861123.7271642790497337759201224997360827540176', 60, 3);
    t('2009.12559915063415946719009354', '2009.12559915063415946719009353211976', 30, 0);
    t('29508653765153371073.52531165945247759', '29508653765153371073.52531165945247759', 40, 4);
    t('95098509871366.9899013', '95098509871366.9899012726246', 21, 6);
    t('80943756831056578.51', '80943756831056578.5114', 19, 6);
    t('92977009470226089300', '92977009470226089307.718836672', 18, 5);
    t('616753228580.552348', '616753228580.55234721477089240988', 18, 0);
    t('7381621811408903838.0728550343820371401742853482037', '7381621811408903838.07285503438203714017428534820378', 50, 3);
    t('78440027207572109240642678954.30380293199343866219668352768437896928', '78440027207572109240642678954.30380293199343866219668352768437896928', 67, 2);
    t('710160000000000000000000000000000000000', '710156435694131392755601183057984543189.26599', 5, 4);
    t('193.14829', '193.148294149622867178', 8, 4);
    t('835940043159212928727189.560696', '835940043159212928727189.56069635592524928041278128102489', 30, 3);
    t('786263.56', '786263.550955552709347744961090', 8, 0);
    t('814184930000', '814184920729.7578354', 8, 0);
    t('8954740616479411900000000000000', '8954740616479411900406110289318.800166943976086150936335882', 18, 3);
    t('641476000000000', '641475405551989.4002413950530282993286703067', 6, 2);
    t('72500029015790529783610606151531647000', '72500029015790529783610606151531647508.230819605288', 35, 3);
    t('8716769016400274.041333681306721', '8716769016400274.0413336813067205917278', 31, 4);
    t('474.4098884609752164038752669081', '474.409888460975216403875266908092', 32, 2);
    t('12826501204331428850868.1445955492638933660158739', '12826501204331428850868.14459554926389336601587394269', 48, 1);
    t('3288404860000000', '3288404860433517.0677507053', 9, 3);
    t('490503205814234045255800000000000000000', '490503205814234045255799801888267152366.3844545778608751158532095051985515586969', 24, 2);
    t('66971700', '66971739.976685305247400552574348929986838165976', 6, 1);
    t('66055300000000000000000000000000', '66055343524111814825189719198717.97719038693469244479094766', 6, 6);
    t('715434425.16', '715434425.1597147607134344986691081', 11, 4);
    t('677788267259789.660084', '677788267259789.66008429356847131308349399', 21, 5);
    t('492924661203442663187593270426680258045.39', '492924661203442663187593270426680258045.385351039559134', 41, 2);
    t('46715891.4', '46715891.308313500235869', 9, 0);
    t('8971559207272230000', '8971559207272231719.2384', 15, 6);
    t('121206.0138', '121206.013823961', 10, 5);
    t('3164247.4801004093807831', '3164247.480100409380783140', 23, 1);
    t('33742324699984958558106741.96786474378', '33742324699984958558106741.96786474378000556468130348826335053', 39, 1);
    t('56301192172', '56301192172.3094253029158', 11, 5);
    t('3166182.73842615803', '3166182.73842615803514216', 18, 1);
    t('1068370733643416796.276', '1068370733643416796.276256', 22, 4);
    t('86964883053706313638.417548922214447', '86964883053706313638.417548922214447', 36, 1);
    t('914500000000000000000000', '914408117838045540734957.5596', 4, 2);
    t('738842000000000', '738842625594662.54457465523415003175', 6, 1);
    t('37075700000000000', '37075798411630398.04975', 6, 1);
    t('1937368589434800000000000000', '1937368589434807233194457506.490', 14, 5);
    t('0.862860097319343064284', '0.8628600973193430642845', 21, 1);
    t('7630000000000000000000000000000000', '7629188822179808172225386839313058.50186363423704803771863033122', 4, 2);
    t('41793419.387338858', '41793419.3873388575259633046557509883805', 17, 2);
    t('12269892108620240.5124', '12269892108620240.51236126103819', 21, 2);
    t('46485234293000000000000000000000', '46485234292774790419605606949019.874292', 11, 0);
    t('45156365600', '45156365596.186714121869560114559010', 10, 6);
    t('2437979.4003', '2437979.40039482797782711139', 11, 3);
    t('50000000000', '53802186794.9', 1, 1);
    t('974291699274862810628200', '974291699274862810628271.050167756809066172076', 22, 3);
    t('3595746.479178336069954137118272659', '3595746.4791783360699541371182726596211410', 34, 3);
    t('451000000000000000000000000', '450697813387512390822981500.936699353451653630416221983459', 3, 2);
    t('30830879930.84239583841406899266474534', '30830879930.84239583841406899266474534002', 39, 5);
    t('6765040762256568000000000000000000000', '6765040762256567872769091158308331735.673711032213567225', 16, 5);
    t('1849502270850000', '1849502270845397.374194614552537', 12, 0);
    t('3985776400068053390.941054461557867820653145085', '3985776400068053390.941054461557867820653145085', 48, 3);
    t('261986515000000000000000000000000000', '261986515037008668637616957098156006.68973240716278797168837297548430346', 10, 1);
    t('324155310382714685541700000000', '324155310382714685541658615030.24977362928983808413209896218', 22, 6);
    t('110', '110.59823201834384622651', 3, 3);
    t('4.52858912476101', '4.5285891247610101575698', 15, 5);
    t('346060760146288791845572.74310164188', '346060760146288791845572.74310164188484448294', 35, 1);
    t('464741553818272018082832612568690015.7808188687585023866869913374789325111', '464741553818272018082832612568690015.7808188687585023866869913374789325111', 76, 0);
    t('53018915800000000000000000000', '53018915773045253355923481800.0', 9, 0);
    t('465033.37821497741991846349774563111601117398', '465033.37821497741991846349774563111601117398', 44, 1);
    t('83307.6666136', '83307.66661354204082581754234790', 12, 0);
    t('820518876760280164', '820518876760280163.927719603020978538957', 18, 6);
    t('67456775700000000', '67456775652135444.7983', 9, 4);
    t('9408485026319982198200000000000', '9408485026319982198248761280366.385154674', 20, 4);
    t('4700', '4695.213034873065550', 2, 5);
    t('48431.815160141', '48431.81516014092121715420008', 15, 2);
    t('1314434.04728673237649', '1314434.04728673237649656', 21, 3);
    t('914988000000', '914988263918.3311634276220', 6, 3);
    t('841268754718402.6884777283578683032', '841268754718402.68847772835786830326', 34, 3);
    t('2179532100000000000', '2179532086218158618.652211', 8, 0);
    t('6178609299078741300.5450786283702', '6178609299078741300.545078628370279363123470', 32, 1);
    t('59122034983300000', '59122034983268316.4733035814068', 12, 0);
    t('36.523989588901912968669070807135017', '36.523989588901912968669070807135016167', 35, 2);
    t('870161465512', '870161465512.0982629887', 13, 1);
    t('858843058.12545', '858843058.125440590016960', 14, 2);
    t('9996762205000000000000000000000000000', '9996762204907474022086663508750476538.05882', 10, 5);
    t('96942573172624683753.6', '96942573172624683753.63950620611234046948298614637835', 21, 4);
    t('5039986331700000000', '5039986331760447336.43377931183000568993836939933', 11, 3);
    t('4395344185395015938.15817497062190937', '4395344185395015938.1581749706219093686508892832783', 36, 2);
    t('438380', '438378.54875', 5, 0);
    t('65655105.078885296730319422775228', '65655105.0788852967303194227752283390', 32, 4);
    t('918753247256864908347200000000000000000', '918753247256864908347137421610191711949.49', 22, 0);
    t('4928374830226850115900000000000', '4928374830226850115903324915175.41199473016321150578005773671098', 21, 6);
    t('28194029298327832230842265605547.119962721164', '28194029298327832230842265605547.1199627211639', 44, 5);
    t('88882719672914626625.1', '88882719672914626625.1', 21, 2);
    t('307700000000000000000000000000000', '307657926873348921505369460894606.91049840431722545040', 4, 6);
    t('147359899962930105854475829600000000000', '147359899962930105854475829557263522450.0888110004766', 28, 2);
    t('81175601931439.826', '81175601931439.8262554644776', 17, 5);
    t('8821042105313760896891610572552.7649589440893', '8821042105313760896891610572552.7649589440892969730233123871012', 45, 6);
    t('949510087923875246711117848.5438', '949510087923875246711117848.54388423535027861541558841766966596377', 31, 1);
    t('43968155450821640788634893308.353841604028418392', '43968155450821640788634893308.35384160402841839207', 47, 4);
    t('89417610609780000000000000000', '89417610609777753894413519271.04042745131620287241607', 13, 6);
    t('45152819502.479474785204667052364854507288', '45152819502.479474785204667052364854507288', 41, 1);
    t('5395872193697960371980987688.8205029491466018736', '5395872193697960371980987688.820502949146601873662899006', 47, 1);
    t('76.8365866602826646', '76.8365866602826645702', 18, 2);
    t('86335650484592234604437.00561795853', '86335650484592234604437.005617958530737368026', 34, 3);
    t('6107714394301454693233614.3061021723', '6107714394301454693233614.306102172202150', 35, 2);
    t('22444893106787039297970000', '22444893106787039297969342.1233788335744063239940285370285062', 23, 0);
    t('468000000000000000000000000', '467850495746072938117755154.525948863332123406434125345', 3, 0);
    t('1301.12', '1301.11517858315', 6, 2);
    t('1000000000000000', '1052897176565950.936187871176229130035252', 2, 3);
    t('5285000000000000000', '5284525973764752630.2', 4, 0);
    t('29404958922500820944766659044818718.60102', '29404958922500820944766659044818718.60102', 40, 4);
    t('10000000000000', '9968153770946.146882277', 2, 2);
    t('6920561620000000000000000000000000000', '6920561611600970241880252726909316379.9378976', 9, 2);
    t('3485.2', '3485.2', 7, 4);
    t('1939997828100512355800000000000000000000', '1939997828100512355815513317388328374394.60138659', 20, 6);
    t('977442857784484457.20580118086925556686773083136289185529', '977442857784484457.205801180869255566867730831362891855290', 59, 3);
    t('654851972433000000000000000', '654851972432550439841258976.8158964233404290719860891', 12, 2);
    t('7312', '7311.9202347435536', 5, 2);
    t('18192327683213.68', '18192327683213.6836413189386', 16, 5);
    t('69797147533480731.162099516482', '69797147533480731.1620995164823', 29, 1);
    t('74184600969483503964290', '74184600969483503964290.3705294223610529329091126887590', 22, 4);
    t('67089990', '67089981.573828444111', 7, 0);
    t('7070000000', '7069099215.705407506839921205993507940804', 3, 5);
    t('7688291992801746557000000000000', '7688291992801746556789627904603.26470705170', 19, 2);
    t('7132330331.2846375829415958785968209', '7132330331.2846375829415958785968209', 38, 0);
    t('6519934.68', '6519934.68403836579587381105880690948988', 9, 6);
    t('306946000000000000000000000000', '306946215447574733917447642881.893294471042430', 6, 6);
    t('47220166226930042336925683555000000000', '47220166226930042336925683555313445250.49681308482917184000272', 29, 4);
    t('48430235362608.849247044918685854', '48430235362608.8492470449186858540', 33, 3);
    t('158271371.32036939', '158271371.3203693903878869363427525795502684151', 17, 3);
    t('582917829800000000000000', '582917829770539629316662.069244710050168', 10, 0);
    t('4109790949100555585437819055.4380840382172937860171558878', '4109790949100555585437819055.4380840382172937860171558877992805', 56, 6);
    t('9747014992220.80045295509', '9747014992220.80045295508789080632127500', 24, 5);
    t('240883460000000000000000000000', '240883458186572745837657238159.73076', 8, 4);
    t('276971.54090865051616072', '276971.540908650516160717018546952372871066141', 23, 5);
    t('52722420000000000000000000000000000', '52722417362931760245500669638129086.32119572765441472489402', 7, 5);
    t('645324454808645431451743843069847462492.36287140556010196887633283761293', '645324454808645431451743843069847462492.36287140556010196887633283761293', 73, 0);
    t('97465014648781810000000000000', '97465014648781810450346149017.90700', 17, 3);
    t('4184.1982738', '4184.1982738', 12, 5);
    t('7', '6.606702', 1, 6);
    t('8669497738721172.1640592008325984245914889430955', '8669497738721172.1640592008325984245914889430955047239903', 48, 1);
    t('593', '593.0', 5, 1);
    t('70000000000000000000000000000000000', '65526536556779129709543979009837631.575845569859', 1, 5);
    t('62880701999103337789430284866161052.275947744303417358736212967434852', '62880701999103337789430284866161052.275947744303417358736212967434851557', 68, 2);
    t('727101903058194500000000000', '727101903058194570982013126.6350', 16, 1);
    t('3845178060000000000000', '3845178061141544419480.25073', 9, 4);
    t('8984576688628408581748581.85790200140839506', '8984576688628408581748581.857902001408395060241211386763601248', 42, 3);
    t('37', '36.60372975668624764', 2, 4);
    t('515282763146119429591203728543515569.7', '515282763146119429591203728543515569.7030618164', 38, 5);
    t('680805.640220885052969083', '680805.640220885052969083954', 24, 1);
    t('170', '174.46152064537229609', 2, 4);
    t('25476099867495372800184766.827928326084', '25476099867495372800184766.8279283260840773757925458188316676', 38, 6);
    t('71671607721775.0333946661276300246612843', '71671607721775.0333946661276300246612843', 40, 4);
    t('4691460044381337580.158419986', '4691460044381337580.15841998568638504', 28, 4);
    t('1108940.6149247987489303262091', '1108940.6149247987489303262090163', 29, 0);
    t('750.7742936568414', '750.77429365684130497', 16, 2);
    t('74863692882000000000000000000000', '74863692881543806931109471337029.850311183265881878998514344609333', 11, 0);
    t('626519094166613586467863792081138.35216352641567', '626519094166613586467863792081138.35216352641567532546434014', 47, 3);
    t('24935436229250000000000000000000', '24935436229252253280756919891987.03234785513250830869728679775178554619', 13, 5);
    t('980944598681120080194390740557964.475005828704710196973392338', '980944598681120080194390740557964.47500582870471019697339233797952027528', 60, 6);
    t('3783900', '3783919.9', 5, 5);
    t('459030541287566504186.581444', '459030541287566504186.58144354332342596179204957688156279233', 27, 0);
    t('561086920000000000000000000000000000000', '561086928584654788365035814896063496317.5', 8, 1);
    t('94206000000000000', '94206288306323100.7302089168', 5, 1);
    t('91780', '91783.572174321', 4, 3);
    t('57139636057587524025228223240316813.9', '57139636057587524025228223240316813.9031276996613', 37, 1);
    t('3525485978.162926897', '3525485978.16292689759858306199037527535369093363', 19, 3);
    t('399876522996.79680668176894391', '399876522996.79680668176894391', 31, 1);
    t('379113287149510960599134.648625873', '379113287149510960599134.64862587318800419', 33, 3);
    t('11489.22922295140367', '11489.229222951403669962308738048998948435', 19, 5);
    t('8698525693358000000000000', '8698525693357839212254773.0803521320864319027822170142988798671', 13, 2);
    t('37631080242000000000000000000000', '37631080241732058316193699115191.7979195627264906140376450177716024416', 11, 6);
    t('40000000000000000000000000000000', '32180891491914323854230373273741.62057080', 1, 0);
    t('344306114700000000000000000000000000', '344306114785668105651018964037392870.1038849181797660177378026766185821', 10, 3);
    t('68870.581611338574', '68870.581611338574054586248586110433887054', 17, 3);
    t('33678295164324182256988137783068609226.40640612056371', '33678295164324182256988137783068609226.406406120563716832490515647253', 52, 1);
    t('598.9274', '598.927356948141356898989666013426359411', 7, 2);
    t('143390373.5658', '143390373.5658', 14, 6);
    t('6638552559700360404632140762391950772473.6', '6638552559700360404632140762391950772473.6', 43, 3);
    t('7407478563477.7980116', '7407478563477.79801156337691099', 20, 6);
    t('2482632263691558110501.3346081395996', '2482632263691558110501.334608139599666', 35, 3);
    t('1660346000000000000', '1660346847120272300.7962', 7, 3);
    t('65340622682256025750', '65340622682256025759.818381147762554318385732922', 19, 1);
    t('85499', '85499.289632', 5, 6);
    t('1979041202129485533901580053150.934100058072932197476231', '1979041202129485533901580053150.934100058072932197476230762', 55, 5);
    t('6973894789190808227.65413', '6973894789190808227.654130', 24, 0);
    t('158167218322287699993031700', '158167218322287699993031767.0614', 25, 3);
    t('20695492663.5583410187981', '20695492663.558341018798159266', 24, 3);
    t('6182448000', '6182448999.381496', 7, 3);
    t('5293417662278951600211000', '5293417662278951600211204.041772917785153116600151411136892096623', 22, 4);
    t('545897210889765', '545897210889765.0', 16, 4);
    t('12215313197.886', '12215313197.886', 15, 0);
    t('2260951.816840778', '2260951.81684077821403853864265472444357322', 16, 6);
    t('9056876438', '9056876438.0', 11, 5);
    t('89976693077401068843347000000', '89976693077401068843346092214.0', 23, 2);
    t('80108722842.9', '80108722842.89370445893027845966', 13, 2);
    t('45991646190529700000000000', '45991646190529674797985739.9687085896269485', 15, 0);
    t('38979185598986014467500', '38979185598986014467510.551771158545489148136084760965342973153', 21, 1);
    t('12643425.87063862', '12643425.870638625452612834830', 16, 3);
    t('766899654017779463132462120.04846', '766899654017779463132462120.04846345567420667154187870412052197', 32, 4);
    t('9537648743664911903034.40143714332136032985318385906', '9537648743664911903034.401437143321360329853183859064875653', 51, 1);
    t('3832518926284411160854280440.72129778716839492378928', '3832518926284411160854280440.7212977871683949237892816976', 51, 6);
    t('5131222550094694666260973774273.7766379079571747', '5131222550094694666260973774273.7766379079571747305486917708945997313597', 47, 4);
    t('423.629371', '423.6293706468406883431471641841', 9, 2);
    t('657237.66', '657237.662241685857', 8, 1);
    t('28458928000000000000000000', '28458927999547148787479802.3113110996981652846865491952600819754', 11, 2);
    t('2003858270479321.212', '2003858270479321.21187594312', 19, 2);
    t('1559171512601877400000000000000', '1559171512601877390115110469335.5491487121', 17, 5);
    t('2677575763969633524940856532936.359159256528', '2677575763969633524940856532936.359159256528084', 43, 3);
    t('2838.73330611625048', '2838.73330611625048727147988476', 18, 1);
    t('17385852096357.1', '17385852096357.1', 15, 0);
    t('69735867325481101645980.72', '69735867325481101645980.71905117841548', 26, 2);
    t('8433171392581590735904134036607100000', '8433171392581590735904134036607109447.3543947944561637914231955310', 33, 3);
    t('3990009389793956075764683385202.792', '3990009389793956075764683385202.79248208423695000727175653', 34, 1);
    t('64856072757279525558.95', '64856072757279525558.946453547742250302167378974', 22, 2);
    t('160.7056535957', '160.705653595704444251252734525415', 13, 1);
    t('1274046489777272086500000000000000000', '1274046489777272086511437035450185347.760032357540450288896670070085', 20, 3);
    t('93395644524857227596039968896046.843122292', '93395644524857227596039968896046.843122291328039743002364', 41, 0);
    t('1000000000', '1038731425.99970500596782894434154140912581953314', 2, 3);
    t('192228943430.4', '192228943430.4', 16, 3);
    t('757508341777251667750366.61397466686333784260505857722747', '757508341777251667750366.61397466686333784260505857722747', 59, 0);
    t('1578166804822223.833909711', '1578166804822223.83390971088520454122448477878955521', 25, 2);
    t('2724344033.2153', '2724344033.215298409141492211546805195004140', 15, 6);
    t('54075497377129202620000000000000', '54075497377129202613717882187752.60', 19, 2);
    t('80792346990988232586552030000000000000', '80792346990988232586552021872978157357.860567064', 25, 2);
    t('778759317397.82', '778759317397.8217813', 14, 4);
    t('694543583492110395.48575698333232967314813413457', '694543583492110395.4857569833323296731481341345622024', 47, 2);
    t('5178035740000000000000000000000000', '5178035738939689499500599161754671.378', 9, 6);
    t('561387245.46444048154304598313574', '561387245.4644404815430459831357403725272036', 33, 3);
    t('6826156098923375864730525.152', '6826156098923375864730525.15199138455461465532534777219826791169', 28, 6);
    t('4643208426713695900826464149.70542331', '4643208426713695900826464149.70542331156', 36, 3);
    t('3141421.536291322048290091381', '3141421.536291322048290091381', 28, 5);
    t('956257500.84288951739222068607201123', '956257500.84288951739222068607201122695', 35, 6);
    t('7299960401146.5', '7299960401146.528819', 14, 6);
    t('76', '76.0', 6, 1);
    t('442534784764510000000', '442534784764516227260.8879952516', 14, 3);
    t('9230349601469827223062574722124.1875377', '9230349601469827223062574722124.18753764888702615661223', 38, 2);
    t('9156122095431962378.1141', '9156122095431962378.11414720044212611', 23, 4);
    t('55080625689204913.72', '55080625689204913.72169667986305658216', 19, 4);
    t('70486242000000000000000000000', '70486242046424899180028666967.63', 9, 1);
    t('9797.6', '9797.6', 5, 2);
    t('992405808301251032279275800', '992405808301251032279275809.554724595376878555023569257169436751', 25, 3);
    t('166436855532907138647554000000000', '166436855532907138647554212041203.97', 24, 3);
    t('695577651779415210271667.95043567774518561076132534244965217', '695577651779415210271667.95043567774518561076132534244965217', 62, 1);
    t('7644316028630130010.94509137318', '7644316028630130010.945091373172', 30, 0);
    t('624503789911821.7199', '624503789911821.7199704703366034849922704720925318117016', 19, 3);
    t('60904575787222500000000', '60904575787222426254911.203765912483894', 15, 0);
    t('393300000000000000000000000', '393384655785641779238058726.322148', 4, 3);
    t('60656546520703093568.2180902453411175834899276', '60656546520703093568.218090245341117583489927649144', 45, 5);
    t('917968808922451.2', '917968808922451.2', 19, 3);

    t('494724446222447272134436118293973049042208245586819785140130787553487687286873291327585520984443687211379965781311260864429223913901690400000000000', '494724446222447272134436118293973049042208245586819785140130787553487687286873291327585520984443687211379965781311260864429223913901690493449434874.049', 136, 1);
    t('3720280055397389210963282787773692958098570405431190000', '3720280055397389210963282787773692958098570405431182098.1847115996658646089352534779938945383741184042776917885782743812931817237846474528789801', 51, 0);
    t('759188909745876678229347606524330959678969850255912440680257260789800000000000000000000000000000000000000000000', '759188909745876678229347606524330959678969850255912440680257260789793508531411911105662225225721357883400623102.881476662045026352517268398904498921045685377855106763679514746737986172605517588709155551610890784878010336772838372711497238975980355571170655725075915881399899070771273619112223837705596667049149981193', 67, 0);
    t('4534304542844843415257043214798844424046834211704366877715023555011150000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '4534304542844843415257043214798844424046834211704366877715023555011146990876705027962738522155031711274213446497330677820779740785545273317321438460934917531081249158109061429290323393161606391411793.277725444395', 69, 5);
    t('527244430907269587850000000000', '527244430907269587847819161754.09994360681344184942353237241149292559482211118493881252182204930127002210488011149670127752911923929048735411637853031498542495512941893603181502188788354208091361643795263910287', 20, 5);
    t('500274199038784701912854356303000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '500274199038784701912854356303168604036329678593776947367198603066622399541094534586945152898619439257327038632128821666524202122658068057432922418958129519085677022999535035083.95723140131476417422632523022711778228314707489949388264508533995136005795122359991', 30, 5);
    t('5333760730338237.358128061070265324699616103754480796605292719724271045', '5333760730338237.35812806107026532469961610375448079660529271972427104505398437046671245161686878493124070786536064182194621929641261600482019485630475569955198761522647338125822382150219203146448476750498789080703982026782691017063636782481300', 70, 5);
    t('9387762026.7326734940421184446066226146106694714282947260976784456146415402', '9387762026.73267349404211844460662261461066947142829472609767844561464154020', 75, 4);
    t('67323858.24741072854262989306161704601101', '67323858.2474107285426298930616170460110072782018778697292704532242132189290371593478382802207909998627911041939764517795222755210476244995305187831767572511625381158475111177037691940475964756887806055996078204829400918467078858084732712280713071351617031612115767601030103857883944259837811618215851568286334400246721991754374286168046011298802236522', 40, 2);
    t('618280971398998509245590744944289036083345801467550000000000000000000000000000000000000000000000000000000000000000000000000000000', '618280971398998509245590744944289036083345801467546352624671931393813580337957902732550378613100649571499058004021009366633559571.2699009128705660239887951528338903748244427453200035937983818076678373616991455573', 50, 5);
    t('3625212716653626154269212443601969756896707.1563164088370829935116368726319072553580520490616568', '3625212716653626154269212443601969756896707.15631640883708299351163687263190725535805204906165678662625271', 95, 2);
    t('279794024756694785000215326702581040764281424692.94416985247012017560069970214089414772133468917746186164351199574782564560451796356697926186392269112385101341672332415427389576261072927099738586782426824776935594024948', '279794024756694785000215326702581040764281424692.94416985247012017560069970214089414772133468917746186164351199574782564560451796356697926186392269112385101341672332415427389576261072927099738586782426824776935594024947823170180686', 218, 6);
    t('35828450460504270469704936980280835584645741698641855543514563095433541570683138929790029105719699910360000000000000000000000000000000000000000000000', '35828450460504270469704936980280835584645741698641855543514563095433541570683138929790029105719699910361577052023426681751310480362480069300311248293.15348414532497112250157766681332507724698060759080257429160605105296732683871024', 103, 6);
    t('43761815435821313568192740041070231887308670966477475362204961804611459661458966510873830168.716748692490194916680548288518654884099098683415405589451450981728769866802762326739', '43761815435821313568192740041070231887308670966477475362204961804611459661458966510873830168.716748692490194916680548288518654884099098683415405589451450981728769866802762326738670533948709724357936162347635031179866244', 176, 2);
    t('9940289055903627512125768245502992585069900170400000000000000000', '9940289055903627512125768245502992585069900170399503819515972524.13568735283721983801757702683853241247769318405229726113276035561474837566466532875876277432576610830142188269641845050335097656359864498101396326004092685742327518181101473104745549408201476216574093693733574369506676184899821529267868141813088352688031339966846243166921375083445250920', 47, 4);
    t('47308500490651000672647597245353550222048575.807214284239202', '47308500490651000672647597245353550222048575.8072142842392016063606507543350997595316613513575837073421103918284314912676250088610064455343463233565051257974791197641171878251624763412467738389496245200149213715671349161', 59, 4);
    t('7242493899945961184657078755114354.8275694408198985015345976108000512015550224515265362601958484547530262', '7242493899945961184657078755114354.827569440819898501534597610800051201555022451526536260195848454753026282935514547388140', 104, 1);
    t('4984463523458535754905264911455285850008993268231053980.767410232956686367696506915282296297', '4984463523458535754905264911455285850008993268231053980.767410232956686367696506915282296296709367816894555456665962274719887159413483764425930136761749360042065357118267648394706165913383308575864809558314678492797545419', 91, 0);
    t('468731093093191466545712621915352193328466285392879958862956252122614534959310655639000000000000000000000000000', '468731093093191466545712621915352193328466285392879958862956252122614534959310655638837578478177944653136138911.41840552882713558913968005741142943', 84, 5);
    t('8722796170843759397737441130128801601794238071659409548210220188790585826046201831648630668253198051544084586185006178347748234700', '8722796170843759397737441130128801601794238071659409548210220188790585826046201831648630668253198051544084586185006178347748234759.9434511722086056368943', 128, 3);
    t('67926499505419960439966855715456849140496804252323842323150939357112.592612688760262', '67926499505419960439966855715456849140496804252323842323150939357112.5926126887602628389494013461282357363300181486827630507100652279617012866942377850016203733098433364051', 83, 3);
    t('1402457858069260843701831542961702005330000000000000000000000000000000000000000000', '1402457858069260843701831542961702005338943536372264316099532808299709640182133143.009979906034715196177295226439746913418590464152810027317068521003207924233662128607269663291833', 39, 1);
    t('1106350604097326705139313335803761878947015866282894852274609129769275668488439922176607333011405146481732245401758539986540.94882628447605933696571359709430425378075223612860671428379', '1106350604097326705139313335803761878947015866282894852274609129769275668488439922176607333011405146481732245401758539986540.94882628447605933696571359709430425378075223612860671428378614749989787085552416655512628791179233', 183, 2);
    t('9735483775685609464322320000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '9735483775685609464322320520902637418698700323093604913106065125437383414521799911105265912341567164193794949895999422938985317071738708834029942382911270457249378913787.1103832980594904520955978916205703025866141635300875243', 24, 1);
    t('489529.68306403435338777825958322163255657451479551721256707088073420494749504360454448131356633', '489529.683064034353387778259583221632556574514795517212567070880734204947495043604544481313566331313388535912869238239769635054162766247017293340447968973686886334804349955536713950306242814646600928789897172058572510482913298938731094634890755913526004290737129748224090241001770940', 95, 1);
    t('5916066038388477.282777699215057097344308799257479415453402583912978373402137006855767835024785902898403750237821320238828483034980698804153055853499', '5916066038388477.282777699215057097344308799257479415453402583912978373402137006855767835024785902898403750237821320238828483034980698804153055853498816858707539873771777970182477242160360615721826161461903035810812584828', 148, 5);
    t('1492565201039912150941642914851.0188694977647271424918935209921129749064571021507470715331012037266466375367808113130910695243443128008252945554605720303655', '1492565201039912150941642914851.01886949776472714249189352099211297490645710215074707153310120372664663753678081131309106952434431280082529455546057203036546314199747521356134765285561227686693949347029078827406886675694947', 155, 6);
    t('9640.823515736412358055650895513069103270437767616274081289588691585664302621579615021666966091360826575410661596403164479300452152684973160363097309608067', '9640.823515736412358055650895513069103270437767616274081289588691585664302621579615021666966091360826575410661596403164479300452152684973160363097309608066484458989787971253015176864265186897585966764740584557961889884222991340777882611717168551975010970695517417044829720982931484654856360759782530622', 154, 0);
    t('2192917244999869343205916374337677.13670389941440829125766924482174747890075138055235242239241690214', '2192917244999869343205916374337677.13670389941440829125766924482174747890075138055235242239241690213762402416630046739017177822', 99, 6);
    t('63537288045407814614632100373927391371818274269554962.476', '63537288045407814614632100373927391371818274269554962.47539574376947926528280013008702010901972495291817972524796755262276716657735757278', 56, 2);
    t('2811000000000000000000', '2810271650889132939296.56710151819671988902086681773799853021233325407901168571657754153770180269049570864269302522698770485556146', 4, 0);
    t('315495111097618519547112164879699292983604105741729996616794447864850000000', '315495111097618519547112164879699292983604105741729996616794447864849595569.4732429578938004170666887599581675624557787373713415172397346993864722937610311872942069612761272487587892003117667431497194318776589121114621644677670506783812384346194108237410689252765415071112547238246125156991556184611980318715999133743922164645414546327450499594633872455804056396617969187240487543621620', 69, 6);
    t('204.054713812637763806802452', '204.054713812637763806802452', 28, 1);
    t('283401218864063400000000000000000000000000000000000000000000000', '283401218864063396558389343623404646465639303535411612743365055.0226824948376888', 17, 6);
    t('92229.61758321528485812953091790666786058628679345291222636317052305267319561885415781143567404759366953', '92229.6175832152848581295309179066678605862867934529122263631705230526731956188541578114356740475936695278089617909281727991815396972738914904573285717078598330120780900701772340175710598595012522421761264348230172003217', 103, 5);
    t('272730132423593904649945011230989.07441427998296751796431596590750797141803801848817960594928685477662435073578350760298700878428378237445274', '272730132423593904649945011230989.0744142799829675179643159659075079714180380184881796059492868547766243507357835076029870087842837823744527376015781771828747087898039912031366854731319283802355925346531872038489621488183273173863', 140, 2);
    t('834882716459088686085166110241864368.2585566209634844197272577905771502073368091374111995246662548601100626717411222217154101996014', '834882716459088686085166110241864368.258556620963484419727257790577150207336809137411199524666254860110062671741122221715410199601367804869545194', 130, 2);
    t('17069318170447559670824010313506648026528166586129880725091542343600550422997254786851016215940786230013064180480826097444101000000000000000000000000000000', '17069318170447559670824010313506648026528166586129880725091542343600550422997254786851016215940786230013064180480826097444101017117212949291335428804169206.4789414896747948034966909141270082719485508246318636726728954481', 125, 4);
    t('459159626088903852606631649784227594935229835299060973799218279323000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '459159626088903852606631649784227594935229835299060973799218279323713336570937633103595944583645177507124617629953771475688119656568329467957028808451626080.2157052326772496024245144704394972923671967963191881122863487049025806804492486980444650765391761189341478929086845849828893436948962388818123422', 66, 1);
    t('8964893258158025627760261546686868902639753180000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '8964893258158025627760261546686868902639753182992598513279568321959004014846384717934802529654264364024872333833301734962238301019.830652683', 45, 5);
    t('364360883189274660000000000000000000000', '364360883189274654915125395493219398572.2474137512117378243053958296641927737578354492483712457777086640826940365951543677980800757991055403096197453940892', 17, 0);
    t('62798805854239559247586786671700000000000000000000000000000000', '62798805854239559247586786671691665719360680992293791198823228.818590188166', 30, 4);
    t('2919.39098219640843017608744567594536808520282595700914805687657135531319', '2919.39098219640843017608744567594536808520282595700914805687657135531318032160509536225908603462589415339166427523162209654606757107524863883661608768843802643273536183881182868488521872273354102426789809964810179453662086615915871323302711182265370022', 72, 2);
    t('498518410121335882889426203225015440023332938864257782616315008715408773039963902751839000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '498518410121335882889426203225015440023332938864257782616315008715408773039963902751838767267915743484873806263287060097982670860870385502952319232270631241658522169392870221178660715803158866477319937662.846785402930088836652911499666200645819', 87, 5);
    t('520941351764055886289735827417855514032432912931774105456330522144490686401420180612123874549554669346321262900000000000000000000000000000000000', '520941351764055886289735827417855514032432912931774105456330522144490686401420180612123874549554669346321262896008462609741586754054364569546461.2429089475342569777092100730166828570271471', 109, 2);
    t('967458518129341217253800000000000000000000000000000000000000', '967458518129341217253709716236449897709859578471385569703767.5910317303224815591685773736902737751471450963092018032255057533766036103870318753356442960596256042598', 22, 0);
    t('9662196143301839800000000000000000000000000000000000000000000000', '9662196143301839705031220496896691190294225138121274279925596712.0203460202028757446953506798750015378274648720904316033651383902176708372706425613310383514346763263467758182468735328971481487656805110460945915528406864172765257723043582531000130586899875561444151372514512291889832677047615547733065535747728797524316358866980760695549025240545242091006010837915568433047688055375245194021392164240208719535184898', 17, 0);
    t('6255329573046153649127956419129000000000000000000000', '6255329573046153649127956419128933860694304373701726.250880165836121004198907852796247214381039590207008830549703854123026933129805499102004754384730169342348270791032292637169670624056036232474455500077254576414258097695787156006722094117374281805864', 31, 4);
    t('2938023661311694369523053387474862627005054567575647185741.9696914269', '2938023661311694369523053387474862627005054567575647185741.9696914269535847720347586814036993775875202454386522900613491733950638770759772801319163707493309651581703579484894219139777152989661627184715672798874751283359330975953440284643879510593891970483429196974653165942791286085891251330175955865947061105293742671856983020020931413079689929286714773', 68, 3);
    t('89644275248076280135996431694494585216142764960596271863233685229500000000000000000000000000000000000000', '89644275248076280135996431694494585216142764960596271863233685229478151361977754297719381884893417997149.7276942035156644361594153201750497989145999432982563104134226323602103', 66, 2);
    t('3621398595588474509066402790068487932332614685252608067437382034371038394916218312812622138438642454452599751032587560096600491173601747127497561833256000000000000000000000000000000000000000000000000', '3621398595588474509066402790068487932332614685252608067437382034371038394916218312812622138438642454452599751032587560096600491173601747127497561833256444389256825145320238855233411588813336509320308.3367165088481830153692092228829737996756183745658278', 151, 6);
    t('92415801217038909730022901378119107982333014308178287386995051914112588361983933297785722729827126724555614209018870066.3893588', '92415801217038909730022901378119107982333014308178287386995051914112588361983933297785722729827126724555614209018870066.3893587475430940038769455129740126488007171767911876764144901490907900148699745671034211458124840605474358419339759832274318140870966448940204797288703631002883899112896464', 126, 2);
    t('7560105202638076591304298529587525012457504942849899112746563698130000000000000000000000000000000', '7560105202638076591304298529587525012457504942849899112746563698122439956530954488734755312827788.6417022065606389585802431393841424894366329257001522258205727538216448527464235329825940759871469256931', 66, 2);
    t('66995001649030211664119047687225086030956570000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '66995001649030211664119047687225086030956566204447845404789316540213090066206078940682480932810099656061555024770099139160542850090.122334522545303463441221026121', 43, 6);
    t('14490397893155132667456885597338604732985379268397981.2081419441252461', '14490397893155132667456885597338604732985379268397981.20814194412524602224011423146752015300159176873', 69, 2);
    t('86861471129532148054116681766300312963711215993008832380584929043511317.1', '86861471129532148054116681766300312963711215993008832380584929043511317.1466258586963221165670547764967205521866908637878320717641', 72, 3);
    t('73834711577547798638031291211636300000000000000000000', '73834711577547798638031291211636311934357660747950323.755152169908968929834255238397734571093192384408210864185466198517657109578864075505486272989945628854333336905036312457194027658496282384810706', 33, 5);
    t('47320095527171643384344591500000000000000000000000', '47320095527171643384344591508977039334416352634041.57113715399196502557782993225397539307154301103953180996525029028153077218955706731379381552220219569878724086327034686612180719089657800025671348735659119045165918181042170882334023354573107788134631556801431885371030182847692891060921430137357898706467690365873192264792320805073351188743336875050498012', 27, 5);
    t('656602068401382732588960833366235917779616250.5636526994362', '656602068401382732588960833366235917779616250.5636526994361945014918158318282957627382214', 59, 2);
    t('902104013140757696027282612431066124000000000000000000000000000000000000000000000000000000000000000000000000000', '902104013140757696027282612431066124061537464554760082052905196121530071374572743312060732667407624715040558091.72015125590106111274390681676950441733844337981103022486026418039654695350245345670858956495278932974603533245310112377237787700927456294773086049548782381514975671428630359433099391923092754028617649781483578', 37, 1);
    t('3660790171446361864950742787131889453341601519.3348997770735366816055966374137', '3660790171446361864950742787131889453341601519.3348997770735366816055966374137103812177465935394035', 77, 4);
    t('5820672707747092665792977577228917423752482118008191157229.109742653843952044630520671469786', '5820672707747092665792977577228917423752482118008191157229.109742653843952044630520671469785960317902540021629891575925086455370761455004286189998825202292767430309991534835297', 91, 5);
    t('5162518717191234940170385173.99825485', '5162518717191234940170385173.99825484726769997216086057519586172970589510020418841313952728974455323049863192560018033611211676626462688063570', 36, 2);
    t('138403120000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '138403117685272283605042450842097277593922286183089934942255569055902622309463413735896057181326252325919090690822832980334081747853267738186929314923712222957431268.54884985379007553067573782572396170956289355694442034980354498312', 8, 6);
    t('23951866586162768395524948731420622224659006274166689482627061806901981039.88337850147313921522241850472544924850306', '23951866586162768395524948731420622224659006274166689482627061806901981039.88337850147313921522241850472544924850306290598547924095268939043158062840494981917', 115, 4);
    t('1420239497131716291787071181638326264822353708160487646389367416516218100408.5664369507868262504659885357644881', '1420239497131716291787071181638326264822353708160487646389367416516218100408.56643695078682625046598853576448800693169421', 110, 0);
    t('17096722453619005072569489945034770852327870859331600000000000000', '17096722453619005072569489945034770852327870859331571214082505560.303937879060753522783440060485926696840894635578707498970953196878460669087418547488782940781613935774777163186171854071703500046287098586043979298335520822349', 51, 0);
    t('6801100000000000000000000000000000000000000000000000000000000000000000000000000000000000', '6801046529969677325899971480644257426586960584187894931035067336000549516020105511735752.53607149476534465142721581188804312', 5, 2);
    t('64096699541740951000522871129459359442357061769306984000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '64096699541740951000522871129459359442357061769306983999290169120671414467098782248086499322736039497598877718544717492313229611403580297813.0260047536504561834614946878649134477649911055772794241232018600', 55, 5);
    t('147176262421381179082043436873718550029597313316260934311152863185168583160388560565038363850210264604836729403000000000000000000000000000000000000000000000000000000000000000000000', '147176262421381179082043436873718550029597313316260934311152863185168583160388560565038363850210264604836729402242062799835486741157139186218356021958569476952057319191090194148153.341102356049', 111, 2);
    t('4636092730850291720622270414.59397183980812235115238645538943708148898771418708096423686471314066860296120514163884909322677', '4636092730850291720622270414.593971839808122351152386455389437081488987714187080964236864713140668602961205141638849093226770254711717290607664511641430106144204889906783456707665938722497375099086323641832517852798493827587', 124, 1);
    t('936223108811876521308502894295449422976839978502750311488979146235136795.309434552', '936223108811876521308502894295449422976839978502750311488979146235136795.3094345520649312210517383367357603030567467993175990372668359868514021152301126858949007904432825217527118264802392721920880409898779947284764', 81, 5);
    t('1709040123735155362317.3966113165689635181713199405552594557540713048778859987081574867514077285585734311646723191', '1709040123735155362317.39661131656896351817131994055525945575407130487788599870815748675140772855857343116467231905512338769750966329289691486336248559993060043110686977416693960208639318400592402322658860308656623113942876105769117677529686343428242185439463940076202503075073114787806291460679057725', 113, 2);
    t('417053298.97922233553105161045066283626769598610761', '417053298.97922233553105161045066283626769598610760854886392155621988826868361783843895502317214612762012245420778927395847747769087524411745263396834957883714115058141594747269690380110618973158171125745225071282117355226415048486144604474205947049611515907359440557991798661688734629397463194567219435890785560436838462649392129030910714545419562587720429725', 50, 6);
    t('10214418722953581805066453826.4730588926066948204678685336679914570537084', '10214418722953581805066453826.473058892606694820467868533667991457053708453899522228', 72, 1);
    t('2169486468260.260791685173852905', '2169486468260.2607916851738529048416660153672501217056761661671752463899381157394488512478190527186859986529152830892428804718743591', 31, 2);
    t('395996257284019593716684307325245872995921690627947.577442548483219510569631535210317295476201430765753803386706888331333985422686042545231031733768364979739744832574259440218031448583312997189858379779482234', '395996257284019593716684307325245872995921690627947.577442548483219510569631535210317295476201430765753803386706888331333985422686042545231031733768364979739744832574259440218031448583312997189858379779482233622315966857928540909677819100057', 207, 5);
    t('74792704839840686038542104994919712157688022694199717543064569532034170795064580544597189679926365206078967961677804752510145106693751505489702000000000000000000000000000000000000000000000000000', '74792704839840686038542104994919712157688022694199717543064569532034170795064580544597189679926365206078967961677804752510145106693751505489701140874881429743453419305310264478016125574911502508.0220376046634516742897529695044172754876036742401337', 143, 2);
    t('6174253550', '6174253549.6142014221795834215750848', 10, 0);
    t('53.3719542139573041258731110524537215007150343134410059935561453054778542963482727403567899005816068320993658722421277533268536439893608871684968883292952127641991426535818671924623848211032205984', '53.3719542139573041258731110524537215007150343134410059935561453054778542963482727403567899005816068320993658722421277533268536439893608871684968883292952127641991426535818671924623848211032205984876267217', 195, 1);
    t('3144239440845522409.62932130241733391963', '3144239440845522409.62932130241733391963', 42, 3);
    t('9882946517217822853644622945258678141060794836198424328071100164278064067618292044.607499066625900922372148168842447513948236369996', '9882946517217822853644622945258678141060794836198424328071100164278064067618292044.6074990666259009223721481688424475139482363699965412736021541765983086283679387735381608089604384001', 130, 1);
    t('4000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '3956992117614684282782509057317718386012674558507095649281672324993051027702459657632730227484411561614108800272700490553871139892701369558412221547850412476394970117853231550359519298.97928109630248', 1, 5);
    t('630601289481393540021014309928703773863121125914909225721944761961216542979929169126806056805280.8105910218711643', '630601289481393540021014309928703773863121125914909225721944761961216542979929169126806056805280.810591021871164331495367323345858816966636852001096978099609', 112, 5);
    t('781670477331191037740111670143690204106.67985025987482555011990816247555542397302913412615246350677073443254383467310457235123789022737387721446222210607645399527001851744102450708827541572192', '781670477331191037740111670143690204106.679850259874825550119908162475555423973029134126152463506770734432543834673104572351237890227373877214462222106076453995270018517441024507088275415721920696429567024314787939830020073806986413341481855200660237744450371363825525032075', 191, 6);
    t('518195726210613100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '518195726210613085655771803405045106899499980128446964634865112632739146629471390828782257047099996969574853502441127697940.8883915196168746899340142788360701822084102561992594179100602817718102417046525392638596925898261021496325208927845801106770892', 16, 2);
    t('9923463308381428141503725653889346659610160352466621380112614259518664917950478587876829774458444357135018803601572138114559501310085632900000000000000000000000000000000000000000000000000000000000000000000000000', '9923463308381428141503725653889346659610160352466621380112614259518664917950478587876829774458444357135018803601572138114559501310085632939836982644916060504787497294323236857068754823080610113140878913437612819.373210730331399202793999549387111428066077678124249614', 137, 5);
    t('32830982465208949238667045462837067704746232605055891965749923186929654007298417081000654390318072945333065060000000000000000000000000000000000000', '32830982465208949238667045462837067704746232605055891965749923186929654007298417081000654390318072945333065058664731658351085039720544375164492372.180739863304351569605666446870', 109, 5);
    t('49427947641710385952697255888182297266711657080270735964468073800362710862862133948814136718.36', '49427947641710385952697255888182297266711657080270735964468073800362710862862133948814136718.366543924977547753390547010206045662680890000160801066427794570010737903344332188152998545783781560414128262314353241711032827528312693970280005614344889895622293373698764733886010983078054729917284967941031047770733', 94, 3);
    t('4425215392947570200634324621658791546696341840.62542917', '4425215392947570200634324621658791546696341840.62542916564570895238332619787026151614670571634731422924977315070336341685668028309426', 54, 2);
    t('9325944092080097816035045593458262.5383408286977', '9325944092080097816035045593458262.5383408286976526042783273441481356407952494179513255493871158', 47, 4);
    t('3515517763912128893412491259716214806000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '3515517763912128893412491259716214805840698122819828408011014085336143510148609156273869929956642563865843107881282013776655552.90234735', 37, 5);
    t('10808820000000000000000000000000000000000000', '10808817716867638466637380144756911806364153.94555634831511598587806420788068660096998341811967687735011280418596660322970825079163248127776212795945264924990985576518531368826342', 7, 6);
    t('2315458027610800652169398656662943777315143479720799223750667670069566048679846910960', '2315458027610800652169398656662943777315143479720799223750667670069566048679846910955.701017739708429187696716017282376323890133551961723040879322960221409809479544367531215047642130291346044540008742737582727158778086037727917120476868638170330588658545444079985780975951806105414867557989603380890901680251161', 84, 6);
    t('8985827786375262202427712610274996993798622651810803220598716379534389083624755422338947042405357190214131128066079876971049238699249375691303179869969456430000000000000000000', '8985827786375262202427712610274996993798622651810803220598716379534389083624755422338947042405357190214131128066079876971049238699249375691303179869969456430158592060304027433.35447999085718059733925682200331772312351706144971622762685368887640337', 156, 5);
    t('524887707409451589603101704022412866152633.26202996280249182330354609135653224419254061976625205093704973425289', '524887707409451589603101704022412866152633.26202996280249182330354609135653224419254061976625205093704973425288968275497453361557041937676667063369202617038435651534405807988866824118668550758232748553506318926672022617761757868787259', 110, 2);
    t('75989805055092603186952215742967142823', '75989805055092603186952215742967142822.5127631895185452222789969053494', 38, 0);
    t('41221000768791290020763128870254781814655735068988243303321891318732566805597997.19214146354229070764998435360882072452541242829814139769342', '41221000768791290020763128870254781814655735068988243303321891318732566805597997.192141463542290707649984353608820724525412428298141397693421234646746261528711059619521215164905659788455709535891455722910540351146615355408657254', 139, 3);
    t('50802468595762714268718353706484312473718182148363533968710688829234462425161290235914945921539238272739490066731860586859400882438376967131941153590604452503170.59324532993461143975974073', '50802468595762714268718353706484312473718182148363533968710688829234462425161290235914945921539238272739490066731860586859400882438376967131941153590604452503170.5932453299346114397597407321594161434635960270904322', 187, 5);
    t('46016123534000000000000000000000000000000000000000000000000000000000000000000', '46016123534180655993990149482451381767821427859617245850101468015903490979175.858275591244039600775598672568121374192621196527576626125507673143739883', 11, 4);
    t('4888699710779351265487213208939551024178969806297447600000000000000000000000000000000000000000000000000000000000000000000000000', '4888699710779351265487213208939551024178969806297447686249991247518170908388553929407290668501930612070068050216212815734519968.72027500484297906270711609970760', 53, 1);
    t('66354823527821482875432450000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '66354823527821482875432449739872296495350461549878806451275365308439362926632276029264397947546369466760889448268413776963880829296039901939215543133882724224910020416503845583577043641336786257.81900279378866058672892928599755546238512591819740399068002633606645', 25, 2);
    t('1131724633884365075592543.007', '1131724633884365075592543.007574418233292044432797401042670534953576621122968025219662925164382324776900778329000069614518083238402331937678180905889553144617166163230724981954947827642425412318440054733211931119806554425653695408237524', 28, 1);
    t('7016839035242305815314604212561836712500585251554744663365087954255785824.01887558904923134362384813448468171743348906467584773877392179193355924', '7016839035242305815314604212561836712500585251554744663365087954255785824.01887558904923134362384813448468171743348906467584773877392179193355923527097448977020919566957865204454517054362382', 144, 5);
    t('63994039341754628062618027602733689075049904053497840926648736705576852178066349240992838702350737765310197169845380274938752299659381775302138.6052118271918140142543773010165013819542245867920719', '63994039341754628062618027602733689075049904053497840926648736705576852178066349240992838702350737765310197169845380274938752299659381775302138.605211827191814014254377301016501381954224586792071919', 195, 1);
    t('63.817192126788117271388061543033824849814777852487697495691479', '63.817192126788117271388061543033824849814777852487697495691478906619489229409824', 62, 4);
    t('997.120791', '997.12079034890602480500857737223608987636673617183', 9, 2);
    t('69701803000000000000000000000000000', '69701802620236551497733457888204125.4768168', 8, 2);
    t('608548851570018943315726164421613635755901110494372569824250602538712615163928440206561562880467076284420187121263717916328762466407947715014727733408986386904186410114773671315.6327902', '608548851570018943315726164421613635755901110494372569824250602538712615163928440206561562880467076284420187121263717916328762466407947715014727733408986386904186410114773671315.632790175989314813338590202370103403116220259935742582', 184, 5);
    t('81844036840670441363709426366942403290000000000000000000000', '81844036840670441363709426366942403290713499970351007905549.073583510123856893999941', 37, 6);
    t('9199247062878474.982190008998493921', '9199247062878474.982190008998493921439986012615347636616881679626495', 34, 6);
    t('933696555368354825732071429840797169595925317758958564990244000000000000000000000000000000000', '933696555368354825732071429840797169595925317758958564990244137328002637490548395911275301529.0489880924559735671528017623579646790586676293009007289064413643873695379523475211150808', 60, 3);
    t('48647105188843195010683443206138754072663690604488000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '48647105188843195010683443206138754072663690604487890224067117558784480089851636890696599496482160689903224679278886903069665652154817.5408854422124059603939367331181466957668631211393757975723665633156', 50, 2);
    t('1594446037714663638121546879381779706084161158444586362821048334192049143638752637435594091361648865008465288.213715060607330324105490052796026760147245077403237466237518926062', '1594446037714663638121546879381779706084161158444586362821048334192049143638752637435594091361648865008465288.2137150606073303241054900527960267601472450774032374662375189260613947132419620843190203078174041698033019588178091327633356091344132823192630990616357298537481', 175, 2);
    t('555343049226788442801323893428901517149209098781000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '555343049226788442801323893428901517149209098780934479891747383031966797635157053943672526986852540323505809372185956359271161751531887496395097001337151703880665210426512014370596390232595707062044544827.2146866390675513183', 48, 5);
    t('8804699900372494595947348117553000000000000', '8804699900372494595947348117552866071438456.75652178014356224754117456504868658806619212190864429840212829486669803444902196041423126932595361230050507694439317379754013480361764652470785533891553705305063813471938962675072589053804978547075364729803302890621185791282128033169666697702850483773966172233202127', 31, 0);
    t('267936813615385062390089188427938622354401006629238118135026467827.534896536876212282379400338742134219612255', '267936813615385062390089188427938622354401006629238118135026467827.53489653687621228237940033874213421961225482194191964', 108, 0);
    t('44014310404925523123737717017297863285966319452970591167811633784913365000000000000000000000000', '44014310404925523123737717017297863285966319452970591167811633784913364664825283684661050866508.9651955653444284298139227997363475241262012856347618706045966911719999487291481480002381770542711072447650303504732458291506991796021110355521437170615173817868', 71, 0);
    t('8456597724371161363638916678058396286510832995909252886874166708729659497362936768779740032898569595101525.015770795747479196514522319649', '8456597724371161363638916678058396286510832995909252886874166708729659497362936768779740032898569595101525.01577079574747919651452231964921864920907000243726501077387641460989734073417955699421444302620062046204624685667862560028363867689604395339372100401707937929339003250654979909882847025102657674443982994890061332', 136, 4);
    t('118331848386657013492875087234803544727000000000000000000000000000000000000000', '118331848386657013492875087234803544727270099606301500951522167705469031026807.58066001', 39, 3);
    t('13991863293343664500348407053013386579639484942701095402128804429957527532284521825700788449881407472100000000000000', '13991863293343664500348407053013386579639484942701095402128804429957527532284521825700788449881407472115415577806950.2655681199022583982734953692545054015071513038861554105446531677792557255306084648819863108617805258133523892018914328217101598780', 102, 1);
    t('6175626977448514045901256794382148477757792628342358911084387700224451950964697440642082199343310569172091477224662361768423846969717260983888323608051836095985229000000000000000000000000000000000000', '6175626977448514045901256794382148477757792628342358911084387700224451950964697440642082199343310569172091477224662361768423846969717260983888323608051836095985229510531655491075626901616748005385653.824239528308774310378049401519100568304981526', 163, 1);
    t('76045360623298367134969301079000000000000000000000000000000000000000000', '76045360623298367134969301078095488608499489614172121047609550481170004.164', 29, 0);
    t('230683737958685598762953724712960347938798338430041771395887825887914495727667444822271115197524242062650353151103571387279555800', '230683737958685598762953724712960347938798338430041771395887825887914495727667444822271115197524242062650353151103571387279555756.846', 127, 5);
    t('5380657860604526967587319340419994016422048037202670000000000', '5380657860604526967587319340419994016422048037202661774765693.4023405953294535980308489051198657218304704104833953049470996001581136838916', 51, 0);
    t('7333992978878509400000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '7333992978878509477832276132142280293760641979224749802153360748249736781059426826081765181653710316666128397722400800289839675804105601848144036487797.6906026163362086029836271190467869199593331531669511989474682843147', 17, 3);
    t('494140147025930800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '494140147025930719179154365271874657528499654182025728593281977225006366591642370596030862485778092974308871888289988809212.88924963867277568502219542486267104242548140244698872605772785464974218220915960', 16, 0);
    t('630116312552955238193051.91521795403242374649696538944167478011840612798487067270827638', '630116312552955238193051.91521795403242374649696538944167478011840612798487067270827638554659866486650974320865577368237968759520861107869234023170214620212294467550688372295926283507755633312985927861964849469283358034419061318714514741036597761922670053007229114779416966342359788598393975503524647667578254360689864231707314094069138540606473948373', 86, 1);
    t('17653417176492364495249917528821790953119544124659959916862517167112491749097724233524044142505734362190535263689551274671048379193364836.9719096828777570821961708623154', '17653417176492364495249917528821790953119544124659959916862517167112491749097724233524044142505734362190535263689551274671048379193364836.97190968287775708219617086231535280454', 168, 5);
    t('5714907844665418027015168518652995101129749298414316535037611517945222373036153432833.382445478053665882697549', '5714907844665418027015168518652995101129749298414316535037611517945222373036153432833.382445478053665882697549', 110, 4);
    t('599119064541883397448477558019634742373986268152047938047374068134341667050081814053832145168030937499325479117698968596417632003768934299163441342251990097176332170712011171436067779258448714317682366197400000', '599119064541883397448477558019634742373986268152047938047374068134341667050081814053832145168030937499325479117698968596417632003768934299163441342251990097176332170712011171436067779258448714317682366197359397.86929624334083177688461751102077266778397', 205, 4);
    t('5738307720920803000000', '5738307720920802528034.864677947879666090176750031667015776064030525280715417753381336190363579757479164471367136492272726030762472772787581724405289236365395651313178020428568165850852716588106857850467282531212334145204521159837284269775398552048978004696189663413971566094012513934095696086446709848621184505948397891677598599', 16, 0);
    t('191.416306802132580483644268506305358', '191.41630680213258048364426850630535786486690213469856816415863371781229506142698157983570978387507680967455824525', 36, 6);
    t('8569892418606286230328926773907855314168071997125761618851906016466164461041426.79045', '8569892418606286230328926773907855314168071997125761618851906016466164461041426.79044905836179669119434461812843372014505549940430583386870689432517957598234831408775565894602131018968509291878221684252923168606714641003321853235473693143462884359545057828907917400558198', 84, 2);
    t('64751184332858677925044144556195161717436471305898195839546221011107445713000000000000000000000000000', '64751184332858677925044144556195161717436471305898195839546221011107445713879192831905711001670721466.99096049347399367233044246450288897546801252643374021675085680929125200046', 74, 3);
    t('801008945206013576397402000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '801008945206013576397402622060952361967797173967040447258467341752859233321289556931283484639185153554882944053119747806524385918128650018723619589.7726979693899611428039844849250', 24, 1);
    t('916383291706215000923778580371.1279186176059806298227187815510581571430293927238722745331625367151', '916383291706215000923778580371.12791861760598062982271878155105815714302939272387227453316253671508074336723438802965503425374153258095039038833624621', 97, 5);
    t('2476317808404560000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '2476317808404560529170069533881866505505488366533034307240685953353067956458199276939007930108303131923730314934680149431383191867462362097602734931932058179399938091950865820233639090282410684.0198821070237128901128331383835129961849517968664494333860955060019190799273085811590328', 16, 3);
    t('230199782408968047340483343283109934515549649256531953792208282159062856120061.04644842785423', '230199782408968047340483343283109934515549649256531953792208282159062856120061.046448427854232739892127214089772635290142316742466763984308881578444250129452112172914551100467', 92, 3);
    t('2212325.3537833697297004506016168918118589897264250372883897', '2212325.353783369729700450601616891811858989726425037288389717651611615874688423229191581709', 59, 1);
    t('994693278690983851768415168386297851153664524202769027124646895763167785456119108917090742.07060443724052389681717069678436469262602351073137827985293512251485030004149645162318', '994693278690983851768415168386297851153664524202769027124646895763167785456119108917090742.0706044372405238968171706967843646926260235107313782798529351225148503000414964516231823626338771043316613321141059436153923660805530062104215243', 176, 3);
    t('29764523098471944.1050805765305690140702288', '29764523098471944.105080576530569014070228848257873415548677575470633330115018598585805776430158482772866133815124184727835914927743890381309925484282620325601617378812765756158151415687410134912330849242266837388288150425906328430889538369530446559719788627530525261911940520287', 42, 4);
    t('75227469636243830727758700586984772800821.80844205913235650944103521017042232467261924675445159212577782929539717926632550010711915392085247155040856361', '75227469636243830727758700586984772800821.8084420591323565094410352101704223246726192467544515921257778292953971792663255001071191539208524715504085636125305732036285948769060818665762869600904021305183014151885738034613573197534406828157401910683410801452708217425228017471606106249696236211167866234340377147516262526962', 151, 4);
    t('19515430547496242050490994603860000000000000000000000000000000000000000000000000', '19515430547496242050490994603869790131386246726267615872904762099125098419773064.73192828615742618336601654501222545553554464612346518363718115770718681652664479952128183301759951006229669025251441055144202123368993187244557845794309099161323436664612394688796436212', 31, 1);
    t('3660252032699537548168470403075725667664134592786537330731273210109704333588765532984113073054860551688196634.459408886658', '3660252032699537548168470403075725667664134592786537330731273210109704333588765532984113073054860551688196634.4594088866570393233972391420954946521628715226623911701678649454881296242499459294468259199627588612428160469463466101941153956174026241869629965224585975026567547985490016209497606664211416960801644243571973709445744491802854914', 121, 2);
    t('15495173429303665885148981284455000000000000000000000000000000000000000000000000000000000000000000000', '15495173429303665885148981284454827352971371598599967506595767056683891132738035441818425186402099444.82475487083689070932673601644900242451092', 32, 0);
    t('36791719306499678605294382231892899423559543237544376949913517154851709731405540109857292207189000000000000', '36791719306499678605294382231892899423559543237544376949913517154851709731405540109857292207188079345552069.735687054506032390868584717995599451750', 95, 2);
    t('93087910097842759995875625944676481161715966548708853711932670912.8480621347724413038', '93087910097842759995875625944676481161715966548708853711932670912.848062134772441303765784028462', 84, 4);
    t('55790752740243495601177160026774673996483530856521574469204473438506580215498954100480013418429330291725209507285875928403034268221804600636083793000000000000000000000', '55790752740243495601177160026774673996483530856521574469204473438506580215498954100480013418429330291725209507285875928403034268221804600636083793275239236573918247509.087861767210315482747570163370341146397626738832421553193376832269188477303064772662558976482167410554219691152578142092623472', 146, 1);
    t('283784837400039260851982776.663095095357539292941129414865867833502931814651707865730203457147594197484019237454872628398645363220405256999913363253426396684593361035369519936012', '283784837400039260851982776.66309509535753929294112941486586783350293181465170786573020345714759419748401923745487262839864536322040525699991336325342639668459336103536951993601152080779485717344719521645035017583667539564192654805782042312448586978830667237975008317010398647516', 177, 0);
    t('3378540', '3378544.185', 6, 3);
    t('958.9858779478386831439705715976308049584366368355375369029009052101303659760324406763248505267', '958.985877947838683143970571597630804958436636835537536902900905210130365976032440676324850526725', 94, 3);
    t('58940183809164660000000000000000000000000000000000000000', '58940183809164669132422706009643971800571331759869460594.4586050654709084856804392868286671855025007448468783967240494648073760130', 16, 1);
    t('473654513158650897172947911724931453983000000000000000000000000000000000000000000000000000000000000', '473654513158650897172947911724931453983303817056337241016884436361351577596555632514517516457873791.57570737304276095483461543587718668897', 39, 5);
    t('777389624244355544793539476347012247326409209214056415226467335731486390711508869357560613526640608651121690773365389480413946400424892.7648309571243281113583054762339891318394', '777389624244355544793539476347012247326409209214056415226467335731486390711508869357560613526640608651121690773365389480413946400424892.76483095712432811135830547623398913183940237814488941416282953508629095107453890345148543351576090854', 175, 5);
    t('1727168077868785119202371816164980216597564790153290813810147917000000000', '1727168077868785119202371816164980216597564790153290813810147916082555083.86850190603928014838258458941554549593238096658429332830640437182977', 64, 0);
    t('5651427510143522797305683200224035081444975848273609047948930790001147604806358370427718689016944119929280740129114581589423337164464809284429000000000000000000000000000000000000000', '5651427510143522797305683200224035081444975848273609047948930790001147604806358370427718689016944119929280740129114581589423337164464809284428619277276052696521437063366466195020644.9738938507150254859894053706373724353744849617620644532123627121', 142, 2);
    t('6417715550000000000000000000', '6417715554296680747390535582.7589375783117450206677541825912400572039874153526608914336344590663911214692990909306708654082243888772447064247', 9, 1);
    t('5685563949399415374554463906606318904719673677112872586079785799583288884590316199908525053743013350405210162373573008.59902926086167074372679551170162521769166751727135132699328804427841928428082221635304521637517', '5685563949399415374554463906606318904719673677112872586079785799583288884590316199908525053743013350405210162373573008.59902926086167074372679551170162521769166751727135132699328804427841928428082221635304521637516011854533635707442', 213, 0);
    t('20686353961050976378383974066021592639036675764493387351950019058617374222332369272836961279538956985967360468604332414939474444480577223700000000000000000000000000000000000000000000000', '20686353961050976378383974066021592639036675764493387351950019058617374222332369272836961279538956985967360468604332414939474444480577223680517381274787159397870500807264095329691591322.437126597518713881196338638297612334895809', 138, 5);
    t('622773230399769423161316347213124067256421948177332805035318236418544781730329341310352296263933659283484472797649315355317075128025414806618.9', '622773230399769423161316347213124067256421948177332805035318236418544781730329341310352296263933659283484472797649315355317075128025414806618.89090198940642193173077542443922974124153020570580321191292481140592392961811914312455402143558682819760342046862519886757911469600468335615317', 142, 6);
    t('2286950140660670398292409262876876341857149.2646153308244328481401118369832699238426461846737531374749504825618330210225434997411976878078380437551678', '2286950140660670398292409262876876341857149.2646153308244328481401118369832699238426461846737531374749504825618330210225434997411976878078380437551677348677094099451288670480931752387670234953462297711296787952', 149, 2);
    t('48736299001362325593016422297184242523366489296097738053648032521197235791100081849221674104490992089130020043399840508933834.145124685174356867746', '48736299001362325593016422297184242523366489296097738053648032521197235791100081849221674104490992089130020043399840508933834.1451246851743568677460840060666918652861459689306956348324703260697480962128562703090744595474806329309604118315614983532172268686171347953418100', 147, 3);
    t('641922133750521706277865905716690248733844369916717447605553717037492593459847584707481234606810153871933870.71324124288343013252112281169335794733811', '641922133750521706277865905716690248733844369916717447605553717037492593459847584707481234606810153871933870.71324124288343013252112281169335794733811418578183384125598979368084341516958257267686234255741747971754', 149, 4);
    t('25497851219416763875242.88099809101664231905262780638986421956055282687', '25497851219416763875242.88099809101664231905262780638986421956055282686865947580882271228030519529620367098454541941639212511861367588849311', 70, 6);
    t('3686958861463023424.0337', '3686958861463023424.03374572660975953416562965736110210158073066875257489365223625064983044191780300603601475807039707015896103547311649518622357766586157247956625216317215689826762166405860382028169683472609139386021082347317961058230731027484245404009660529230102185317769153511695590860081473512602719531928724333', 23, 1);
    t('16375785438103292421197470807475421549192794576807324947827951725277000053030464201150678699200000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '16375785438103292421197470807475421549192794576807324947827951725277000053030464201150678699204970968461252590734918353817808800071931304384742160999528655034367360994114828141884659.773330299251946637454148169290683109416239884577003897', 94, 6);
    t('90573368394931739963298668628303395384900000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '90573368394931739963298668628303395384902717479717223536819335041943816698585798943965325659920175577589725044699257849128126216326287.501969631490149388019616969232108354884059613498862636969751056485920061298760103711026166266762286605065006377349798335966176858103543226767387', 40, 3);
    t('53773999632024209935070504550933200065714537651235620075727999273072918231706337521724851982292199815547128421382069228483706129.6668372752', '53773999632024209935070504550933200065714537651235620075727999273072918231706337521724851982292199815547128421382069228483706129.66683727520653631351845152540037923532668803830', 138, 1);
    t('5450743247353477288055672569190993307691890469426322029379653241127147127373950368081247696154411342497731785409362736805016152437031911816481007850504452059387384873069997700000000000000000000000', '5450743247353477288055672569190993307691890469426322029379653241127147127373950368081247696154411342497731785409362736805016152437031911816481007850504452059387384873069997635209341540276662002333.9473082529952825325417225981740081357889783252475882818', 173, 0);
    t('250354565870506920000000000000', '250354565870506922176280052723.0450715294615446332281209757723973876001815278456528748452742087812071804534164373312852781406894821981242488789798646866947676117816163565806500983273408508197186949505744394591188644776465665943675338', 17, 3);
    t('46519307224392751255027984564176672805833059734261371608488131785035718047547342472674856405825100042540080611664694433732899836984057081381692614135487015680200353626926500000000000000000000', '46519307224392751255027984564176672805833059734261371608488131785035718047547342472674856405825100042540080611664694433732899836984057081381692614135487015680200353626926492434230558804349414.314890003876481478202088719913717964', 171, 2);
    t('756803234574170746380.84', '756803234574170746380.840003675756826506909395728130816221363679157064021866413768973761034616806578332378257278356614341861616129643497252437731603902395586', 25, 1);
    t('453231636777181806203813713051735144438884807.97891911181735187472154567455917918328871110618182865592120447919428374702192700601299570100624951545483460025291640223128318122512', '453231636777181806203813713051735144438884807.978919111817351874721545674559179183288711106181828655921204479194283747021927006012995701006249515454834600252916402231283181225124631', 176, 5);
    t('594749263000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '594749263079631890539899347849671513173017672037471494320013921759164892105559454602687775658363542153248189782204081203109950196563.23419003602671210', 9, 6);
    t('2726297893480867623616918272036989190940491407810000000000000000000000000000000000000000000000000000000000000000000000000000', '2726297893480867623616918272036989190940491407809153790920942609729007195723067784763932865070166444754685233931401472291657.020004624461714173116406127985783639335656612617871964118287617691535821481301935505957215396339458328030375789559681687309367243997526628347973914575051', 49, 0);
    t('4476108350350779908003453898345776397006837544352661055391141490200000000', '4476108350350779908003453898345776397006837544352661055391141490100125586.581349056228024551134457815735202483937707250533736500785549757274556368289761837350988775196494724892147814446580922760304657238416017337', 65, 0);
    t('395738.0616958786052245533191073099513143069654718275070985744879635312783955258503282136513741713742449682028054933694390447901028875550651848', '395738.061695878605224553319107309951314306965471827507098574487963531278395525850328213651374171374244968202805493369439044790102887555065184824618790604250', 142, 5);
    t('55146564959896072259977065229475550000000000000000000000000000000000000000', '55146564959896072259977065229475554060554255749480086977277062181607591272.203667437265264330881541964547846188815806247335792664929725429893239453192998898478116527888', 34, 5);
    t('809659468352886026939463819892631770010267047185343.8062954131707035698091575744677858398047074505243259149068371655894160388432059136338266441740432246933803879936822392208777432867154', '809659468352886026939463819892631770010267047185343.80629541317070356980915757446778583980470745052432591490683716558941603884320591363382664417404322469338038799368223922087774328671540', 185, 5);
    t('47495342176992746620865496130163210050221600', '47495342176992746620865496130163210050221622.64637656350032043857559152804634124218281614950767232385742676177127141900592783899571260974687688474619126327814273896207310610371885488424086955198698556995578370909040118275250967491137850939352710837932190736937017650708227252207375437593220091349620365417487886502594977754255717767105538052492', 42, 4);
    t('1541796028661937656686542905775765917830765070764057716470.81638611803357944702192772303262380766256582924888133102067852227608276232834280461637196279351206864992', '1541796028661937656686542905775765917830765070764057716470.816386118033579447021927723032623807662565829248881331020678522276082762328342804616371962793512068649924509777311498443065150070872203193888237552855186228539863498002', 162, 1);
    t('264764842367287017133548701926354627322557319442000000000000000000000000000000000000000000000000000', '264764842367287017133548701926354627322557319441601756028063984056347242704048354999096371299500328.4166396662901044178802253385161155044772640121269315766181', 48, 4);
    t('49657688356331991445586053343513822475620831344519283427937082728312454902295265644059590341274929074380796210825966143723170000', '49657688356331991445586053343513822475620831344519283427937082728312454902295265644059590341274929074380796210825966143723165835.26406846272150830208721801586480251573792080187993867749012751596596332074090599031856249822964337042772177359334417520439130212606025983583233732260366990850875458586', 124, 4);
    t('45773311434894949467340360320447303169417190901478851768129455000000000000', '45773311434894949467340360320447303169417190901478851768129454887147257547.7365199502051338542735409162512116', 62, 0);
    t('29960609449539826306659280436376522557518976.418373649752465334562939061733766599079771958183914567069557801779013731041514', '29960609449539826306659280436376522557518976.41837364975246533456293906173376659907977195818391456706955780177901373104151392071225957193051086507728545865587104526053596354771412166109031807489136250439129916444586151784150818130285541289639978483822533896482963041261579497867698035960175128861907', 123, 2);
    t('15828239163206072.454931496895822592624904929487341817779072725340094924228379761699833190208297583729665503390815668547393969718574663', '15828239163206072.454931496895822592624904929487341817779072725340094924228379761699833190208297583729665503390815668547393969718574662591372889591193871685717611505147816963541952401664666528842870776284195061150762869663004968701286885405562147112977694371242144301218367050457750696922571', 134, 0);
    t('8076235257757126146041617.6106111376132296770388070742193459103588064967154685336410739705996', '8076235257757126146041617.610611137613229677038807074219345910358806496715468533641073970599661766544856387402463465440256601222790506431798597343314081966297016168690644588341821554729948109876077', 92, 3);
    t('37863737676080317350488736904841756692117663694317363763973730252503458.37768693602456972', '37863737676080317350488736904841756692117663694317363763973730252503458.377686936024569711289384101840829053587872072961173984496223629424092193820760960548286129852971513816', 88, 0);
    t('44971500208017953698955438980000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '44971500208017953698955438987384940477175909684034771115186431664578965338126651787093772232834187914031273978208099962115498990542745575020294754767182545453732248876481.19', 28, 3);
    t('427541717674221793858920465159733070866600390000000000000000000000000000000000000000000000000000000000000000000000000000', '427541717674221793858920465159733070866600393178179731797036236235351850001027167688467470339045610339984352108957660345.73342640738369948642657054670106564643659381292844482304432753063469762398621143041993839011471079999335980703581393462151627327748326365512030832735806527464498423244618099039667255171084126214214311289204801555794940', 44, 6);
    t('72257775932903792577610000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '72257775932903792577619036434043554408866614440351043327581290658522720449190331101801292987430901060298286916419107884719978204489274963021766140736518936593314898837052.581974037978012413370566597', 22, 3);
    t('89612927274087877988080290492841499932381499362217494666777144486866375575855055839493985214647059578354.505881426075193', '89612927274087877988080290492841499932381499362217494666777144486866375575855055839493985214647059578354.5058814260751929822039292907102689312857202873528567312006741484151368557603101451', 119, 5);
    t('81286019249566216671337049398506287614086502282110410561201839754036700970232265498116226571097404271121166057386251720800000000000000000000', '81286019249566216671337049398506287614086502282110410561201839754036700970232265498116226571097404271121166057386251720777188230142854846029.29994909444027652622386282348118329781632655223849392752966241461', 120, 6);
    t('98518625441254686599563044061844700948790282466410586318009278056644986541931719270761927224520353447416000000', '98518625441254686599563044061844700948790282466410586318009278056644986541931719270761927224520353447415962365.2851172183045041281535637659695953037666639138422832832923', 105, 5);
    t('127671636938', '127671636937.987123529006117946924624057923993128622997069414900392770945898583863161870803063360', 13, 4);
    t('2993243306460972305729512070870583015118573775010166888424891196969072408420696287236932837236814745178842191728186859842918341162506367838887185072.99577961000068530864603540937326665083011688970459', '2993243306460972305729512070870583015118573775010166888424891196969072408420696287236932837236814745178842191728186859842918341162506367838887185072.99577961000068530864603540937326665083011688970459', 198, 5);
    t('296470695635653023169889007326000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '296470695635653023169889007326855450893138784079639836362923246341138400087012052143853316515307484629116565183809499215974209563888172478127941764001020669956246577547995985241785336225747018992110.820929', 30, 1);
    t('34423811953642173764571656372530178223058238367860335272450239217817853703656772565811353011425019743676693906956762197536507057566108670742576486924220474401304020000000000000000000000000000000000000', '34423811953642173764571656372530178223058238367860335272450239217817853703656772565811353011425019743676693906956762197536507057566108670742576486924220474401304013988603857287378023721478840711237724.738479595806847347934911706', 163, 2);
    t('718123899335986533567.7086097170906355911269646201699044704313306507725531746184046714648573585255701', '718123899335986533567.7086097170906355911269646201699044704313306507725531746184046714648573585255701057276992126136765596575255451770251417173468151520505302145879872755993028450886549245798607858649549320104633703673622202528140594899138907796136377912807206321', 100, 5);
    t('95436099298702504504343768.3002025574175148292371429', '95436099298702504504343768.3002025574175148292371429703126068967187', 51, 3);
    t('30.1192061973800701277575242613889722582', '30.119206197380070127757524261388972258292153627398406289906249002271077279894371527948041927118692922635219840512142580028608948432369426819007505120027619357236067456764643499962589220626853391661231126682106923122557278574909978782265554850219031750826040502241324431685746', 39, 3);
    t('2562794118207498366831439617659049196175835639157512988446987390099280000000000000000000000000000000000000000', '2562794118207498366831439617659049196175835639157512988446987390099285545260154805048751414167639846390784154.4078443692560291164', 69, 3);
    t('2466416651730690015500000000000000000000000000000000000000000000000000000', '2466416651730690015507396953795579515481624696456342979225750698780379220.89761352905', 21, 3);
    t('9071114185074947968414843056892599498799140347535381800637937365272567045644380441808263835710765864866958252.518745262780567545484694124492821468268159589169', '9071114185074947968414843056892599498799140347535381800637937365272567045644380441808263835710765864866958252.518745262780567545484694124492821468268159589168239743221767642960163282703335142333', 157, 2);
    t('914963700451998049743880036699329379994566514986975061720000000000000000000000000000000000000000000000000000000000000000000000', '914963700451998049743880036699329379994566514986975061720546776672888884702818595613566423278635546067529708543403062452633591.712549927214249133251329541204995265102081450168695487471363688243567413140044633244771882287891520224020919425936844301964572305198464099424308485460943913607046473068916591918956', 57, 1);
    t('89748288219753433422104575331947913060326246689737806205622356114099534266855430844238989824595178358169607933931596905365018816607575435129062641000000000000000000000000000000000', '89748288219753433422104575331947913060326246689737806205622356114099534266855430844238989824595178358169607933931596905365018816607575435129062640959448599746183542772923916480439.7033201698339434867387818862342807581', 146, 6);
    t('9263330976594319988276274145833226643930108115060125932826301325998710748729381640588271763391365901249263604474103951224678256611520273820981.25643933081797198520326964967522898348056026246', '9263330976594319988276274145833226643930108115060125932826301325998710748729381640588271763391365901249263604474103951224678256611520273820981.256439330817971985203269649675228983480560262465673763660847663038115609146605753039204284', 189, 1);
    t('1764764545487223101800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '1764764545487223101860369975671780993129042755280403759934678340159548362998896616210560830198293764973309648485160066386277451438802986454668123344205652928.4271738395435684172318416703652382945077334483185274013756215316754763978093395621985758291157392450031657964806805394206304464750555765554738805', 20, 3);
    t('79601028700688760638.59365922095089169346530905689769375518315810269787989958887068792331', '79601028700688760638.59365922095089169346530905689769375518315810269787989958887068792330109150676733936590486170311802684966688361413070551160381447178011452807844124721', 88, 0);
    t('954.2528411898266859106045859337252158409336835679392517025785217184843363871051107682681943367093811671', '954.252841189826685910604585933725215840933683567939251702578521718484336387105110768268194336709381167102524192478392162389917482382710569749996018086939097974449950', 103, 1);
    t('7885045669635394030805991242531008176157510000000000', '7885045669635394030805991242531008176157514260954672.48746273385329591822707870846688365670520142988174482143336823989892216431970967091729585924620784111976203090788937749536816091138706032808530462588014311154308915110288610498412546117631500855129912162782722306285813208292158554794000340235431523838615935265274442883184638217796150710099667927926719928534111509697206', 42, 6);
    t('8706692581872478093562875521973830000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '8706692581872478093562875521973829946750081880691488238301811776875230926753944442432680524230422255874042047777386214370445450984476700.13484446390151917286172236821631221323034764443417956156944859061015480781823289448214751013493552796973762143482134380915703511971358425472404288743251912092289078039708063856063639796051695316887022185', 34, 5);
    t('8194066786445121042939000000000000000000000000000000000000000000000', '8194066786445121042938714002505212446683070904447612153980961528916.757504824610260299527211337258848789705228719657868665877985186234634267374', 22, 4);
    t('12939271390460000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '12939271390460423480494249727669895881798514634354314922382507003600450484739398514075045815838107272143256.76222301548554355367010648136218202825038158971196160495084047806464269039447473683932805096353536211577243521513224923319332379392820891854429908546781473498616091707957546488846965694355', 13, 4);
    t('53906274305544685.33', '53906274305544685.329248529693833600767256678227094876161829526905244077875143992759462768013726457633137934301463193450231215305805526669198078577656410166782080341881971076078475031528214604194951750175650407985950072292820482226881311677991', 19, 0);
    t('3322845047032785483726788561682143094848232271700000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '3322845047032785483726788561682143094848232271660963380609338112181734137794875102890846745364033317397476337901129632827804360077279847417445330044969430724789886.48346028654312327491371656325872509938658795398528473978983631179182490871379613743227711631969806190368745255454700410032856390672949751', 47, 4);
    t('13282415918327614969937818228329145895292034603886632584976217095507434910300000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '13282415918327614969937818228329145895292034603886632584976217095507434910294973098064168134036839842559188294244995391568288688609361635795481231836339557948179524959559837148.481510848004712115304701387756818844256521689404680512445', 75, 0);
    t('5584243065533932640764466242547159474977901731774508702723556207015429940914794776.12709548', '5584243065533932640764466242547159474977901731774508702723556207015429940914794776.127095480025989541975396618', 91, 1);
    t('2992276286400498.55268867321389052095527060255708615070518585988946112917094547913679625198542652193806866333487254153009010702', '2992276286400498.5526886732138905209552706025570861507051858598894611291709454791367962519854265219380686633348725415300901070296515789263212388143920676263417397716555818208313337002460093329656951682564751491010503508608662354174201171423', 126, 1);
    t('6099352708417164815147696947086174291904565503699751349089756441122.932765203722303005662214460064414669107004607885621902351741777268353939315', '6099352708417164815147696947086174291904565503699751349089756441122.9327652037223030056622144600644146691070046078856219023517417772683539393147599351798474749395581104459918874164807743554444524230327101249497132903299941234', 142, 0);
    t('7.39874262605616778783451485747500641533365', '7.39874262605616778783451485747500641533364757784513672783144320551581318', 42, 0);
    t('27961438269900000000000000000000000000000', '27961438269899700188740218072218219673542.90672120565531074994103283227343988802377251784839710290841085114189375935184308415266943627145358790819906879000010691992991695431432866340541500023837148576917517011882121209493367005967137314749833593035971251884577778316665635834522424868529205029103935549024371794059661539531213253877734740945230634320736004724456815462', 12, 5);
    t('7260994897513444230601438780639787072278017272.872898282805529043892181720364235461566664574337463685524576239011856244966227556145246496', '7260994897513444230601438780639787072278017272.872898282805529043892181720364235461566664574337463685524576239011856244966227556145246496', 137, 5);
    t('85000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '84558604301488171843118785812280192438662052034119641682088579634477835829438901854533675885270529772955940296826469838035016085315535152966300557.48807603601192973553624702475516189102822935022432373462549287153259710172686608690473242562800840303551906865', 2, 6);
    t('75089703266408485374779750827773520390194.184948486082331613006522624344840294', '75089703266408485374779750827773520390194.184948486082331613006522624344840293578935924702284652810360110948248897503446057856329986200654456587059992409446031504253', 77, 2);
    t('2168389323103766314258230770516291853615380049710571057575852334031533988358010000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '2168389323103766314258230770516291853615380049710571057575852334031533988358009138625333203538729206028734440876046235653499226289718640315342448658313443775673872995838991244171810621849776132208137270339816015881733925.126134', 78, 0);
    t('57470133247211.368567232356', '57470133247211.36856723235621302591124672395301157521708687893901064500600618755643980922312291857419599622780293', 26, 3);
    t('8750268844272052704328703425796043796522066846838248889392512918649566954000000000000000', '8750268844272052704328703425796043796522066846838248889392512918649566954091321865762501.43617681528791507015881826052374873688831914370887105987849069910661336269576450201952215748724814029398909634984800943047585943405922596367394029', 73, 5);
    t('967212270030947576828571785043724536311214244451047182542044218587658500000000000000000000000000000000000000000000000000000', '967212270030947576828571785043724536311214244451047182542044218587658496464522041534147495798872888137955009520399195631037.73885115032716604231332625293', 70, 5);
    t('396082914710054101000', '396082914710054100918.53839109097008364646804428199857227630868613529133472564035597803083342505779943184496967787676239118600631863032559599740444128918704756090329189336689190019325957108907722489060616250884616305237744494718849262740318403504369729946132287772528817441288832983982027802713620637077103742043103803665673366913068883548306741487103437671384523746014433775061371023265086', 19, 0);
    t('4505941013653922371706146991675673429278648973655575063030568894477371289717639703601950506978379707599681994407934288875785280311292697913783162203490834412065196539708894000000000000000000000', '4505941013653922371706146991675673429278648973655575063030568894477371289717639703601950506978379707599681994407934288875785280311292697913783162203490834412065196539708894439081800759171797088.36323773637459330136048226419', 172, 5);
    t('93724307911776893320233311060171286122883164768065071360166196816997456985985000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '93724307911776893320233311060171286122883164768065071360166196816997456985984456026292561302091354991433404957396875434161932235377166508855998557606708598193472513041352663299721409.8255919823810832337809490577328478641659043944186721970', 77, 0);
    t('69403832863.683461451181529721244483104604163509528974441073335823819581032496475967313642574558720758072844763479894', '69403832863.6834614511815297212444831046041635095289744410733358238195810324964759673136425745587207580728447634798935206053106751349835210381282653140742232016800125585924513665565450627394079122717192106577554858451783286769', 116, 6);
    t('9810404208052800000000000000000', '9810404208052796662842586463232.4650650649741110580683921859518746962200309115007549903302487525559636277649771137426795207288937847787703860295189519305473463354283497289070262247786140629268644126795696422256910634562127568779179750395820395505214093781856150199761295969711097293939263608450653113628652999733400455423562616010659989530849822837111153417848491482671580829066359192', 15, 6);
    t('589069234720578881431160000000000000000000000000000000000', '589069234720578881431159768674665985726153346569562571629.9400', 23, 6);
    t('923771985359528985044171829310092336947520700065075106506139508474536302158627743886977859639947229378799102314558970579906642359637784538360000000000000000000000000000000000000', '923771985359528985044171829310092336947520700065075106506139508474536302158627743886977859639947229378799102314558970579906642359637784538359549699527466711348104725899742049505.52807606932840404378132183544799157986071492000617136877207873423810949605288570661760257976067848961212793', 140, 5);
    t('2971748551847848321317259251178342889538459260000000000', '2971748551847848321317259251178342889538459263852622606.958343099203049844811972041233154681908062412110594542563221428126318548812885703302277160836869820435842604926023546281249583332950641128137820728584125697042142085866970391503496248036115287807083602', 45, 3);
    t('180317943337009477808.213215536213805463790773009138494548922979061998625754290629388663363368174228719946', '180317943337009477808.2132155362138054637907730091384945489229790619986257542906293886633633681742287199458596373537750483666320212695998536750431665138726350699674451825969963867582232086409158031748497022774586013701029292943253425078631370202444385694928335599137475222255486728945228643352170371765899009227002224863273535522038750159603185473129078054', 105, 0);
    t('9510465404.0210463642754227636395076593472870384984608182436038094087982', '9510465404.02104636427542276363950765934728703849846081824360380940879821786574477528237891451457805903414698107028445439201745709343056054506648299764673589583061507955662991344228630406121386542105152785497101972888447782874054212052754442', 71, 6);
    t('73286448189844806538901096356704695563.12419155543016946063649263811057807816', '73286448189844806538901096356704695563.1241915554301694606364926381105780781636905875860632066594587852891118', 76, 5);
    t('12461030900000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '12461030895445519659573976837760142105167126736680959714635399305790322847610872406625536860246485334291529798445041625583798912838044612903478501167086926739836075441.24111395965340000293380863656642598069920448592175787556765849503606238104039588', 9, 6);
    t('2991779548216226406940408.115957256909996311264299770206193511362298510536042044463580516023435612011508147519', '2991779548216226406940408.115957256909996311264299770206193511362298510536042044463580516023435612011508147519468037100922431147580438638504072383393579741715348019040030857891551146254520672781179744156226511430306135679606883748516330737084166178420676970690933370250941', 109, 1);
    t('40340000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '40335710243061493390471438775185604035380071757776022488014763660980531332014419343529332954629503687259460422638844893051276757140995684621655556315.38584579200943592128125146823995036909145515486594230252141669257925858132807176810906824070876574501524020547779296092458905951800767310793542580560', 4, 6);
    t('919499294478025420305647342300000000000000000000000000000000000', '919499294478025420305647342271656258696507505925825407348418743.0457368763260727065630833173893599879682740406739373400974926161585557037167075149340326038750759617', 28, 4);
    t('8214174211571756229962323955059595131232520682600000000000000000000000000000', '8214174211571756229962323955059595131232520682562346697478994397172265647634.393110069502', 47, 5);
    t('221025291704689607933288998247012154561204609288371116278817.16470503281337711483291036539150916928133215400009713383624472485961006373308488159', '221025291704689607933288998247012154561204609288371116278817.16470503281337711483291036539150916928133215400009713383624472485961006373308488159', 143, 2);
    t('31513721094871217343766910266729.9558553611170671014223927496486570774163170526354251108586983541771708216185225782542558662715', '31513721094871217343766910266729.95585536111706710142239274964865707741631705263542511085869835417717082161852257825425586627147147792016582279668573874859888459658451120712690401584671957300319987883687035376855341184698898706', 126, 2);
    t('91844612000000000000000', '91844612342635396898651.680355851', 8, 1);
    t('93454285276336593562285414105112345.21454532362366427129256658573008323586106993455224759', '93454285276336593562285414105112345.2145453236236642712925665857300832358610699345522475831243992719606573089046710362361628338008298870294648565150753362568835159065997163242856039701506640582672187459827569639485479774357874550651829315924055198191839981859291525932279479366090158558618485936825145570860703697708894979420207815576476083575566164612', 88, 2);
    t('9692076269737223471413084023859863881.315961257749329804726889368792949715974919659734461304850834244946426710975411158752338746269832945', '9692076269737223471413084023859863881.315961257749329804726889368792949715974919659734461304850834244946426710975411158752338746269832944878776044317013677383307799311', 136, 0);
    t('586037127409209100163148025473699568695776965312021747611792655587801987650000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '586037127409209100163148025473699568695776965312021747611792655587801987649942667002592863178758248443720288481163152644337474498563193026680686994354423652521108504321455360739327.4581', 75, 6);
    t('8474392202198054072192131418760215140450000000000000000000000000000', '8474392202198054072192131418760215140448023953115118608823796743555.203234957032632398598955207083689844797406635933537', 39, 5);
    t('66737902410200000000000000000000000000000', '66737902410252166383673490891119880929396.769564106157927102375779116222453544081648510439500070090038221939148718992492367185082043001751626192258290408787025764402261302351592036449033836958149529238756999191608369145218072092132029972672209254925717533342239885853463535647579601704204288516606892313878995083952057531315780', 12, 3);
    t('168921616307017079461113797083964427522645067138489629938481021876908451008482248463339836167008510040104.630813', '168921616307017079461113797083964427522645067138489629938481021876908451008482248463339836167008510040104.63081210198470014330740008580356673196658552691314476774482129306812215567986489836123166101136046881124206918501876317628854151823440451177061392314230872655694038003581099670361901974693021493943501843726526178142819511131898240696651969130305827899863', 111, 2);
    t('509417327088535317747256396962840357313223493677013050211.3529245232024048702970505859943425620455', '509417327088535317747256396962840357313223493677013050211.3529245232024048702970505859943425620455168565142284272823905298675441913195871762285147450462654059328391929147107569541043546442948020704375810659545311139322900149822987898000806081004207779719136799373514598254475757', 97, 1);
    t('539460433491034970463207003642224597772731071149676301808592785994614174913000000000000000000000000000000000000000000000000000000000000000000000000000', '539460433491034970463207003642224597772731071149676301808592785994614174913078581350238701073774513062464987123974450512681934260357505013240225558362.161522254391558992194554203763023081574695735749276387555223993143820262475734296620699559331626203441904421479461088937388801', 75, 1);
    t('4506600328786175081233423595.60331525137752746800149303767928379661233363866', '4506600328786175081233423595.60331525137752746800149303767928379661233363866267328514556832027397413482954374403', 75, 4);
    t('8871986356122750914801819799152503543178338272528.402656575307467155480428309913366296620820435428404184176084426986363939586781953216838089311292456597445981805124728160303121333052', '8871986356122750914801819799152503543178338272528.402656575307467155480428309913366296620820435428404184176084426986363939586781953216838089311292456597445981805124728160303121333051953861682704262867458', 181, 2);
    t('33832043194724941529377819803931.609094869619645947325640564853421150887147593947610416145268426901906346952758249172705', '33832043194724941529377819803931.6090948696196459473256405648534211508871475939476104161452684269019063469527582491727052539272833968786274217001171265008679459966658622797274724885135078029909534853211194301060043462140531788535132185257874011683', 119, 1);
    t('6187746015760748439430974303253184215843759775611008.596905511', '6187746015760748439430974303253184215843759775611008.596905511986721979268657396394', 61, 1);
    t('8379.7156130822396059884246702', '8379.715613082239605988424670199587985678184011719957569814374041598815077104512436897315166719664168466949141530547790461424138', 29, 0);
    t('4700501493020131248178724925583910545170225.45856824778731877523995490391751', '4700501493020131248178724925583910545170225.4585682477873187752399549039175074747939341309000051317173150301304503387915759849544972717390212749879288857836430551929204564', 75, 2);
    t('8554588103068736296106.8360212997595971838249714316031026189436328770214875676', '8554588103068736296106.83602129975959718382497143160310261894363287702148756760754645713', 77, 4);
    t('2991937938748391790326415116063209990907714080883453751084698290692468357852395.13115759', '2991937938748391790326415116063209990907714080883453751084698290692468357852395.1311575899545824339695291672744896363174593979502520061659149005343959730502257202595381091292514533070', 88, 4);
    t('2723403404810373858719735404893269779658697455108675364752984926621850231815143601942033810490131414008144161817147041562964751596491045570754503104821518515529000000000000000000000000000', '2723403404810373858719735404893269779658697455108675364752984926621850231815143601942033810490131414008144161817147041562964751596491045570754503104821518515528607570355855704783652380796.5202521622234473685714401851696198684912708494230', 160, 6);
    t('53721929226735644065316916040273045324001538004676288146012253208941196884211092398252533978432782801444.889210003629266100568167582289447990118067549', '53721929226735644065316916040273045324001538004676288146012253208941196884211092398252533978432782801444.88921000362926610056816758228944799011806754885638855891188747522288532538463501899723104433416358075546649903955633916496454065039812083614779110359347921712438739368992', 149, 0);
    t('65360616404582055016378455911466078989506274172593088513770300.376133304466031015852756081582226395621172997987772383388792183190483289565847032434460583033852459399219290356', '65360616404582055016378455911466078989506274172593088513770300.3761333044660310158527560815822263956211729979877723833887921831904832895658470324344605830338524593992192903559923650117373394882655306787344854286474263289129408685767910855183', 173, 4);
    t('796400000000000000000000000000000000', '796431324480232746622263717518113449.175731283273671278018192162520908755529565588925452', 4, 6);
    t('90791363906366120715892303629050830210276678356597958517263310408053426092738633463801817.19983432903123547299543206336478660053026519292392', '90791363906366120715892303629050830210276678356597958517263310408053426092738633463801817.199834329031235472995432063364786600530265192923910440888398782885767323090952021052848803402898347767183718989576', 139, 0);
    t('89160422295000000', '89160422294600639.903992226409300728198638761424103', 11, 2);
    t('6556883230981683946684024529077687297772787350610078662799056301162082947622437898332112995102357340945912177.1', '6556883230981683946684024529077687297772787350610078662799056301162082947622437898332112995102357340945912177.095795714764668777721590857382593475293113502833562653414815040681239172889947902686617453945695393572914891326860831403445689203257421803292645807998877595054726435487896652793960178052549871139182870068511168312559426', 110, 6);
    t('94500123494235211859301860062268000000000000000000000000000000000000000000', '94500123494235211859301860062268161598762043584123066613190554248856329603.76714779894136098479603831159383372816510883913201788792568795501303540618613389942477572370', 32, 1);
    t('792987637344257621525447246867115542.42233091821299348097275334097995172523244095542264417641128262667559837135098282522311474918174463592730222891181879929045', '792987637344257621525447246867115542.42233091821299348097275334097995172523244095542264417641128262667559837135098282522311474918174463592730222891181879929045098664644465374541295577498901734737478087057862603143', 158, 3);
    t('804137857183929696066278148323874034563890534875700000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '804137857183929696066278148323874034563890534875735143710561408189030831287485408668761237255735537554072609675282082393570158835977004990.541023212122196288656966402134608812804433984615727784237402355477824289069223', 49, 6);
    t('73921999753363029698360136453856772041359392010435446186510.160541227613051393759', '73921999753363029698360136453856772041359392010435446186510.16054122761305139375859082910897041670630471665100519862320132745052133776497974145840567550826647678983426506756489407069418804557290227543150902700642042776969359741105118504188261769611686270293804899766687497549855789733883608368856680329088564161782216033', 80, 2);
    t('63833988069228897409749953.11647293833768376878313845', '63833988069228897409749953.116472938337683768783138441992180721686715372821471158658562509', 52, 0);
    t('15247338562325.206284300011156080848086930333807736434754918105813934457496152301999111857681932357693844752585431677558816518287529042657206172794796523422331366431385710570761', '15247338562325.206284300011156080848086930333807736434754918105813934457496152301999111857681932357693844752585431677558816518287529042657206172794796523422331366431385710570760864687012397207675891457161867775382014798721261555670900608900759856148219321156549467186650259568978060043007', 176, 5);
    t('28097800900000000000000000000000000000000000000000000000000000000000000000000000000000000000', '28097800911860788745059904257083543710087079273420546318471672003028234286747978185311161679.58829759696037053500646446165586784380424138286228813060543629909285668646445114794414416236124160420531438841418225219823567567931621615724512219579124021797819875588143147727363834025754266109072260654916131803037987289936759365097472426540759968593589142027791796114917636622249441', 9, 3);
    t('824393985936336580000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '824393985936336581082736446340416512353809236887242825039451718342496073397975449825169226619761891147350196648446424463049369549210441640268681164524.255248978787984847370814606420026509926980136135932724246987278391189160979980981366977000343440837439279340028636851', 17, 1);
    t('8110607662539172630000000000000000000000000000000000000000000000000000000000000000000000', '8110607662539172630079438172587847566292780629523897186533843558455512446658098644887530.11258197860629151807095912107029270444251454562643', 18, 1);
    t('484112292509638417146433298666230879655836212810574378773942329962461512303992597.2655661720450928907093055510975353', '484112292509638417146433298666230879655836212810574378773942329962461512303992597.2655661720450928907093055510975352620603768290335965933085053844012462178748225319523688267588206265585052080551227202491091276849502547', 115, 0);
    t('98126026809035203211183515263888684169368962531843821829018828737401943275830000000000000000000000000000000000000000000000000000000000000000000000000000', '98126026809035203211183515263888684169368962531843821829018828737401943275829058250312359708123622077722144051248630993776512860508123310565751715880695.0', 76, 4);
    t('6616998694296271814756579986724468488749143839667311625693712485220073972775070317200000000000000000', '6616998694296271814756579986724468488749143839667311625693712485220073972775070317219501183449657274.89038307033607375831288204523217021969639933678627204221', 83, 1);
    t('794115480752405946789450000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '794115480752405946789449591782589670412625553917662641550148302163604997697700334925291760786361584527039565159238002048730953455050475937034560910221004816614632204415181099369786035170317568.96855450117939205161525769301360821151664812518', 24, 5);
    t('870352899738.96351363787739524575713457815463291089562884064911074255089191106447910487240600696967427630264441210026471548414885128630301087098659097574564925619388071891669086', '870352899738.96351363787739524575713457815463291089562884064911074255089191106447910487240600696967427630264441210026471548414885128630301087098659097574564925619388071891669086626328829999861780378996889603201398654247037328318059674579247785231238258241584405309', 176, 1);
    t('496136031092228258679710000000000000000000000000000000000000000000000000', '496136031092228258679703568512226696528352736810451556875314453879398847.866802730785348555379909599627036423773260718507469706219502410214747036', 23, 2);
    t('9074629546367424082933717158131118276452298951406700760934635522732482609894798366710915.4174833856999975365445345394', '9074629546367424082933717158131118276452298951406700760934635522732482609894798366710915.41748338569999753654453453939186136449734861661019141802812508852098785806344974498011623097157613691', 116, 2);
    t('675300000000000000000000000000000000000000000000000000', '675331606463642280113856003500956838929135348139644075.9063467412806587951327152877118831632818350668537499037181615664734727899602660394511670009238421973972470712618373105226787885035415846270180729579224038739210049666436823630772915404798330032711159675491039547594907948744563636144379042720093559545175877287135825939783192440574753917194324454756159953012958450624997467176671889084808332711012766159634905057606329586', 4, 5);
    t('3002936419765889503834051829208566974432126888338948733598296519295170442035185270714166966713013159148794142008848001667807021000000000000000000000000', '3002936419765889503834051829208566974432126888338948733598296519295170442035185270714166966713013159148794142008848001667807020351608134945358230615056.2853795500974750352344857749043785286700225987714187744812912469251796047097528290452828665813413615866844481490841277575971351369', 127, 0);
    t('408.41277256500984897347099238585184328189914010379582042253152286858306155525501218926046838321139934842275568856362490725563556641303799801798994686352740595195', '408.41277256500984897347099238585184328189914010379582042253152286858306155525501218926046838321139934842275568856362490725563556641303799801798994686352740595194742801850988522629893617283919219193103998799303969248532869852753', 161, 5);
    t('92534360020268062210783409389422217098294203756372617487540638896942143813629889354497007606327765013044404126.700330428072527512132', '92534360020268062210783409389422217098294203756372617487540638896942143813629889354497007606327765013044404126.70033042807252751213206554002609276361567677594642761123201502979512356342215220088516310086279735918135132493747930335610947239159696274111457873360278154315375532373039836876260110067118', 131, 3);
    t('698017126110178878952860161951171376759477035877563644020510170671493092727239862504408992003000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '698017126110178878952860161951171376759477035877563644020510170671493092727239862504408992002124594635765242084291468908269288666829143920128823081413068002188106720370762719672762098265080128818162560502817198677938.1', 93, 2);
    t('831627455937552.6251483261782373682762', '831627455937552.6251483261782373682761396965493424354574032571609339984196782774575820201463249401469671614573705939801568287', 37, 2);
    t('27645439133335268428214220000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '27645439133335268428214218711553433645468409036971305403749835908265645460816706235923705920636889738335329173977285877103284731.40704297228090983485168788327117788054221959238564073558188232370413460631388656106854366880998191409469084711304138737923954539593807344', 25, 2);
    t('4886155693436750833496073005237.686612117984057221152433244608551640404109915662468701664862142953294368959', '4886155693436750833496073005237.6866121179840572211524332446085516404041099156624687016648621429532943689589746321181720013282234582070448825318852760946', 106, 0);
    t('6094594640000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '6094594645642016289925063981903088433416110725788927130012919095959751886524984317482727254046101649459339916509538446738022968196.7993926712611225007607958283544137126803150970280599927507188803935133821984774125502789909585924521376903085805935823831780700656', 9, 3);
    t('6157242279003212184310215176701350375140273149246444964829825986033105522633060523187343761988031267972662087976061169638896138117707257593900', '6157242279003212184310215176701350375140273149246444964829825986033105522633060523187343761988031267972662087976061169638896138117707257593957.42620618920971391367783756684832079238791939148514434169693', 140, 3);
    t('29073255811249437472775238595226366554630413468121310520133136272137218221571984692406819830230550088898549274164723.4308539865345385852770965981413310725494', '29073255811249437472775238595226366554630413468121310520133136272137218221571984692406819830230550088898549274164723.4308539865345385852770965981413310725494338', 156, 3);
    t('377833742650383951563850764422333962457243.837283764972645286660111148002275704148889321395758169', '377833742650383951563850764422333962457243.8372837649726452866601111480022757041488893213957581694500726984882205', 96, 4);
    t('2602606636759466860950321361599324472618237166044973366252347815741420308668903174709960257982504121430326752.43374492002', '2602606636759466860950321361599324472618237166044973366252347815741420308668903174709960257982504121430326752.433744920014347981623126391972783912567894331384850024492860771805335829791348450731689666500975456845603910375506295441216010480825411860916278580293897025779646702773914166516554557037977798063732102545504932578725648976744', 120, 0);
    t('7325757186422962313754695423000000000000000000000000', '7325757186422962313754695423348277264992641475000052.8482', 28, 5);
    t('2.8659941359591150603266275387378301', '2.865994135959115060326627538737830039796782073981629299819378867330242044247677424490551808598460621983091650546682837658329774492821084417287161447451898910761257187810078354808482589', 35, 2);
    t('5212671000000000000000000000000000000000000000000000000000000000000000000', '5212671897559623468404083840111163375693944002590364154936825600826828140.993934987477974417491994765155019572934590860897295487269359432204535', 7, 3);
    t('50027831168536088768405764146623542814712392716845386481874094015137020349385236178594481378230.8411669550054528616731066154521486715467410045295435374079073230050335724189279', '50027831168536088768405764146623542814712392716845386481874094015137020349385236178594481378230.841166955005452861673106615452148671546741004529543537407907323005033572418927887163005874320104388842293486588266715633526527141955245112996021576', 174, 2);
    t('6179662623804128262607907364502284042324450346084774144176075595890293721515534532748837766205552481010100000', '6179662623804128262607907364502284042324450346084774144176075595890293721515534532748837766205552481010010853.806498001052839175324159448797361030689762947520040997035167651295847769616764050737429278445684604421539387842779868687343570243541839183645260900074939291', 104, 0);
    t('347056689198811306326442102227253.5596535933586070562908969877532620862258856412715065219713031503957395959998', '347056689198811306326442102227253.5596535933586070562908969877532620862258856412715065219713031503957395959997553932138792224975864801740803982081252307318197779883208780015289807697121534996122739468606542765138307293414638257760975457802031912270398488234668745042515515282981107229362122160322389434641', 109, 4);
    t('85047444000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '85047444663431935448553195364154706135016408829534899233460104612086302748644347652775373330426147611098.449884', 8, 3);
    t('97541.824', '97541.82404720000769552604010955885502038293107010529854372077088358174300207943475020704399391592181566021813682907260153793863407074910295984761005741793439907881730', 8, 3);
    t('6466044201587638423181231587138576399195511200000000000000000000000000', '6466044201587638423181231587138576399195511200957354993903231618799793.7801510502671412815177633886403460180009631468225270346214359376938104192406436', 45, 1);
    t('2587367983000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '2587367982201096039729271223404991212964037143111948507639694692560844993351470973153758412493610821283958030967887941546985224047535926004104357665659519026291431957181083862108256422130815933346325574087501.52946520028955582070980493131228997880', 10, 0);
    t('808409658.23988586255576035212401032232451065625', '808409658.23988586255576035212401032232451065624938264268232', 47, 5);
    t('181109940134973656996790761450186767319271884620097989158994578187000000000000000000000000', '181109940134973656996790761450186767319271884620097989158994578187550442474430094687679516.4861872496996622381277108631987023234914905358143422541510810078372474784956421184656491809405368091095878533010469573571906272178075842788402143813742970224003372786451113578829431320285866863188926811428713452952774551677887933654203657947842629530', 66, 1);
    t('82144898527000000000000000000000000000000000000000000000000000000000000000000000000000000', '82144898527025819759313670232793448628228468227360462631787763577724184827305548077623238.8984732856947777690822788584607890797906723103633916150775305251689320091724768622127288346455315212787472445998095961961135215596874615062133781074099962558005579601245121237223287853478331469292128039166869', 11, 1);
    t('143586549164844974498100427408560303779334254405465160031579883057890168325330757063999284147503881046705464918582.64302876604193007071642093677140865763857728', '143586549164844974498100427408560303779334254405465160031579883057890168325330757063999284147503881046705464918582.643028766041930070716420936771408657638577286028985137316467054818455673743188482746171945130616348157161058304569140389933040849956698780971487449621100979205086628870220127277688749085862686769527480', 158, 1);
    t('986180747733167047867781085479055656893578173921310606787037682241621700042713051981951778623222929523537500000000000000000000000000000000000000000000000000000000000000', '986180747733167047867781085479055656893578173921310606787037682241621700042713051981951778623222929523537493670677406317154843113292293106719783371435545816672033262768.909902447289049710782', 107, 2);
    t('5739859.034189486063460259210963588637493', '5739859.03418948606346025921096358863749334112945995678', 40, 3);
    t('571.9537420152338088236358845912630720576019', '571.95374201523380882363588459126307205760182632239934743705896572951738677351685264628466544478637664001183545136947360940778666913028306518461915525591024919631491499388215285765', 43, 2);
    t('2357841656344030125946996053574287659500000000000000000000000000000000000000000000000000000000000000000000', '2357841656344030125946996053574287659553792276651044519367453906035603314666830411708542824857402873000912.795542042812813018217175', 38, 3);
    t('73731962808249512023615431297635312146852442139307425621025035933536559582217512160420826852159115522216347604161988344014911013967512060799274417253169804695684.36962011461894883193001478', '73731962808249512023615431297635312146852442139307425621025035933536559582217512160420826852159115522216347604161988344014911013967512060799274417253169804695684.36962011461894883193001477782046563642860695370781603383456284464441590672607457128622977742327620086997706163', 187, 5);
    t('731132', '731131.864', 6, 6);
    t('3726085374102165620844643157610768116760873494.14254980864763311711253585387361827', '3726085374102165620844643157610768116760873494.14254980864763311711253585387361826552756092812799215456900480181636602348568465039745676473226790623691936554810780317872629821573185144382129953927903607042935437318643440638076015922605745629498424868728289195633380775307572805776881410411322012871785472800', 81, 5);
    t('10875410682244698800900426001129566841301306639545886.784111349049896877414390015318217729451598170031463887105295181792', '10875410682244698800900426001129566841301306639545886.7841113490498968774143900153182177294515981700314638871052951817918478487473217644249500616682833', 119, 0);
    t('7555547839417818272855.97523709387756771820900039493760346492237327921481262', '7555547839417818272855.9752370938775677182090003949376034649223732792148126242581272735865100987340199352510457615287153053617211492342297118342926051697863294218955080454843362540211319420884728292298392187854551658629329434761454942805024078790670246859838855', 75, 3);
    t('8839455747266643369065809.4244983040034254', '8839455747266643369065809.4244983040034253641256166918816027676', 41, 2);
    t('29255941738730978919442957299163150806328775247932125488384617419950000000000000000000000000000000000000000000', '29255941738730978919442957299163150806328775247932125488384617419955962141984129910782228398420813538008337759.36923', 67, 3);
    t('36928018828254700000000000000000000', '36928018828254665187945388146087916.7922360112684245710995245693', 15, 5);
    t('570438939437452028142767207981829442108737940000000000000000000000000000000000000000000', '570438939437452028142767207981829442108737936453884318760180978236998381460506948489084.38124801949689025682752479647757340958010938284218208430905121130576632', 44, 4);
    t('315198680876297382343817615667420858564586132872404116971868431408147662299295851717881974881782.5960111387478354037921', '315198680876297382343817615667420858564586132872404116971868431408147662299295851717881974881782.59601113874783540379218754721431068606635258210270306582570996983535518169826015809990973142159716525022988155486534784600614928589107395963651553906300491983015059054846530562901732972276271327036710582302500144487047360641617', 118, 1);
    t('58445127697.896049', '58445127697.89604898251', 17, 5);
    t('10365882608613217047522344.1948944645923871101696234042809885954833291233040596221777691944735719497294431069025341', '10365882608613217047522344.19489446459238711016962340428098859548332912330405962217776919447357194972944310690253405793275038212469405088867279689299254049132699693460626002609919878706897378925251840078280125012871329500934677300706975488770359590014', 114, 0);
    t('64934423609917780000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '64934423609917780230218533322096710404647386737570966296951803258809122994730951904697973146442583850939939107923443390504802435603824.3', 17, 5);
    t('5501120520455294072971443532.079819358902302701641402582902440189894660246636129544718768843920373465326052654701721850571782681069386955186133815277880942904448557', '5501120520455294072971443532.07981935890230270164140258290244018989466024663612954471876884392037346532605265470172185057178268106938695518613381527788094290444855686303487853284685893694470174145573928457926419426122687050120675071230532042993978490930521465566709017061739844865324478244099922582937293047', 163, 4);
    t('9024339000', '9024338122.49259', 7, 0);
    t('66846930587154618580138773081155538.16806091618910192337335105980620390853369423537112026106172701151160847374349422478299082995200939019637622108072795816683870543407177824697664187', '66846930587154618580138773081155538.16806091618910192337335105980620390853369423537112026106172701151160847374349422478299082995200939019637622108072795816683870543407177824697664187171103015357008709240715872347283873781593671913969591520879693679359469014802830695007356565577044296501513339', 181, 5);
    t('2244524372577962643489481507749712379478817698039676230107733649208938995669496.48909784964475691155105658449021870756721652712373516265530222839861773546752570632995967', '2244524372577962643489481507749712379478817698039676230107733649208938995669496.489097849644756911551056584490218707567216527123735162655302228398617735467525706329959672460863918882862306973013971293076069468062609312046193372874917351665', 168, 3);
    t('30256762883569158134135401302884448544694899297641790669243209300708443511358084207941715733251380363613565119737440932765629098329693449860000000000000000000000000000000000000000000000000000', '30256762883569158134135401302884448544694899297641790669243209300708443511358084207941715733251380363613565119737440932765629098329693449852570726442346676358310947094170486273611786034499103.466467337436400219522839', 139, 0);
    t('2822286557526419662971591874766975726917999147198754814405206579765354820181128053903858421211763883497140000000000000000000000000000', '2822286557526419662971591874766975726917999147198754814405206579765354820181128053903858421211763883497139300889612944559531721552328.680892860710719', 105, 4);
    t('9714057524000000000000000000000000000000000000000000000000000000000', '9714057523684742449540299598903005976487324998203929600933626030084.1031296814603517165040707751335698659072728549818731522243976463773572627285665615580772035888189476701748596221982091139105258969709532306287253156', 10, 6);
    t('371094390352126227808200182053712151549295004856658942158288021687722320937736900000000000000000000000000000000000000000000000000000000000000000', '371094390352126227808200182053712151549295004856658942158288021687722320937736891887044145031972071000765531151909065989648999057849238658210194.3386328533901716941998400082352297605818902435536962838328174231033766597786252719', 79, 2);
    t('185964762048546616273672633451538714182355843928341935331803950026434584795194995592349809841453835003319019834682239351.855829668651416850433115276331795628770263544540678688979', '185964762048546616273672633451538714182355843928341935331803950026434584795194995592349809841453835003319019834682239351.85582966865141685043311527633179562877026354454067868897946418917618088617850317641560629338110770570172', 177, 4);
    t('827688633786471985224867395206147093696831173710019147262839608240676070968198942377635442768416004305587434610495969201278510403677966604639055540945112333063151633231393595323271252391582.72001877897082095', '827688633786471985224867395206147093696831173710019147262839608240676070968198942377635442768416004305587434610495969201278510403677966604639055540945112333063151633231393595323271252391582.72001877897082094684625370607448191291435554190541040336479', 206, 4);
    t('349320263797353124402678223136972048214560884509057140169949322193155005957668222295435145403211760435014542218305879000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '349320263797353124402678223136972048214560884509057140169949322193155005957668222295435145403211760435014542218305878007488122639904621394840134965194021829418009257021136334775486625231174491155238489449363978.2249', 117, 0);
    t('21639951746042091167333048989072068714840.4887976703471283932918273660752925764144307906347260425225754264771', '21639951746042091167333048989072068714840.488797670347128393291827366075292576414430790634726042522575426477103399567131935999559887271328535523318217502841670232918118080678016839238182642405278091221482288232734212932696730007131909537495956586802316360340608009915101254818893256820472156', 109, 3);
    t('24208222707455481198604000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '24208222707455481198604394606818591027240183946877688876979103814981312619125318639531445507125300713220598096183949848575283901.1623486', 23, 6);
    t('15379546809603556899047636531002951550520360339885497143405705391243326642628615246089984343144538231352129011207668004415595648930000000000000', '15379546809603556899047636531002951550520360339885497143405705391243326642628615246089984343144538231352129011207668004415595648927112226370909.38800791040414199193140830996262535481646916616291458664646620256989260884162294978765711449507552659749348264007', 130, 2);
    t('45587692750188239198433802350', '45587692750188239198433802354.984543638320863', 28, 4);
    t('173773322854416146162061160266885560375883172135412189220109728075175926138131896305182828', '173773322854416146162061160266885560375883172135412189220109728075175926138131896305182828.23247699139340538724776757678984424683359026230034640567698051707932484357710754298538720212699157052564', 90, 1);
    t('8189381785742309015153751.322249852408942522657870365', '8189381785742309015153751.322249852408942522657870365072783333', 52, 6);
    t('9131307811716743634590448622777940000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '9131307811716743634590448622777941157535908850837541749033627479100389837511749333901137819457040496322332390274530611266918457330899744899365363199722.47258791271240587625887978562559179595315880226056730556023510049048331726993668649594053077429330693003537127153426652227278998901282', 33, 4);
    t('49991.763343564', '49991.76334356343472', 14, 2);
    t('551250000000000000', '551250038763987495.8215460251860340664257077254380431732071959854636684284356899690893191592714205059666944456420169097882818210815150577908274313614441315531198345707158292815618561440743294681534659148200165115266789057622352051323210355738533414', 6, 5);
    t('911744312201314962232169735454744892761577772604742444514798129284531559018297433283727035227479800989934281885786887662973024911868896714065639121357624165398001653892164834932829024720000000000000', '911744312201314962232169735454744892761577772604742444514798129284531559018297433283727035227479800989934281885786887662973024911868896714065639121357624165398001653892164834932829024724452184345351.2652706731218251825184', 185, 3);
    t('191363740168000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '191363740168019577895516708689180899007171911065789642644406095471877961652710153833247901094742055601776613526338.65536629807213233160561898489391677210630690794166189673007343165338832423417269616455649060397728174705121985047648997938247447074627160333769102201780709512239445137429342762183785328313108405387201024500321840324123230484253784', 12, 1);
    t('7623260011072350986143959146000000000000000000000000', '7623260011072350986143959146145619591393586945463498.750995044565682330647582825411291660956359501103153448897275753374944324458428491474849291216103818095860288616728468652963697140355287046414572247003882847538375784329038057579472120357477586128322624254108427', 28, 4);
    t('246206173688835562207291043658364239480645.697704748664615887001027515950177207597891869205375956434182094547639454021202067904187710507977406628129296698947653064', '246206173688835562207291043658364239480645.6977047486646158870010275159501772075978918692053759564341820945476394540212020679041877105079774066281292966989476530643188992112202220102826927444758611757987743976831325942617109861062768839827165243110866205468360662352886207865819', 162, 1);
    t('786187448946716531382680890185555090082068789357396053500000000000000000000000000000000000000000000000000000000000', '786187448946716531382680890185555090082068789357396053426478713646851550977580763656715234340362784484330955178854.4551258031057069020138795386992826992562217065407715322', 55, 0);
    t('782869276540086352886956488886125350672667720403669609684010938198430858.9384650628456914058594', '782869276540086352886956488886125350672667720403669609684010938198430858.938465062845691405859334232608381279246361748355530265535288150178981668183293498127172613996553539383587203066741606939774452980041278299296401631017723100190815410244137785214186', 94, 0);
    t('731078542427574962269148205649103576953518186473282038543713479348610691698868825198365749313082165823164.1171589166967938223184851779901425875060782446990888842626885914706207392522050976393', '731078542427574962269148205649103576953518186473282038543713479348610691698868825198365749313082165823164.1171589166967938223184851779901425875060782446990888842626885914706207392522050976392801307948298', 190, 4);
    t('1134378193173000000000000000000000000000000000000000000000000000', '1134378193173417130287471217649534732903423438545387667168976831.20943473326243153946', 13, 5);
    t('89007529534975279798107800518116771576321728119560334463403132031391338256660000000000000000000000000000000000000000000000000', '89007529534975279798107800518116771576321728119560334463403132031391338256658204919431478786621663539966542119754202004996454.61276899302397452538293421627089387986758186322508552338035880185844860697224838825004989019602976370789943784901693452196345053988883135191096174040927338804402042420387', 76, 4);
    t('619657221856572203208934072364980861496688886753350053917550742703235706942093578747005560749622048010904814.71504566786425455818', '619657221856572203208934072364980861496688886753350053917550742703235706942093578747005560749622048010904814.7150456678642545581808153870877950583020262021252013327253196210', 128, 3);
    t('18775025155.1487918654805203663539513846768796421397665', '18775025155.148791865480520366353951384676879642139766556201044225755584688824102187896107503834445262093154405150653212142238884749848270164605068955336438527199330654380416004507754793706217912177626560787645231642701176007708490472150387578131680182559648427849459947371', 54, 3);
    t('2663324600000000000000000000000000000000000000000000000000000000000000000000000000', '2663324614652779106760612209204417539633178126761288048996411638831494123503128176.7759613053886434686371701221368652281037418908838499038584384382267078796525806578312811126191639304773706971227324266030326554753557006719551628806', 8, 3);
    t('98539367639866246611265545129977915890541326813405091493744409438984000000000000000000000000000000000000', '98539367639866246611265545129977915890541326813405091493744409438984323382406135803404118968937866738232.0258137723362759472193927378124963024964134198563400068803135670802899576300124771305457867532812584787725051622928705694080072217706', 68, 4);
    t('60180702132708906673300665401761859543808994940300499916216048601906594675048816112893587833789837398006593417887430190352541409427328315776167118324661933.3866298', '60180702132708906673300665401761859543808994940300499916216048601906594675048816112893587833789837398006593417887430190352541409427328315776167118324661933.38662976742317340259343313002583534565531127817572543770389365721917177317235259421512321003161877173195261240586882540149737207976062445494928592959075385', 162, 2);
    t('8804485398218883095646902640147744762574340761295513694700517414365614928949438984600000000000000000000000000000000000000000', '8804485398218883095646902640147744762574340761295513694700517414365614928949438984554689230067424414525902779745208085262868.1483925670925281630889001643', 83, 0);
    t('2903018011504287826977837083078876618243536108691.79821821201391532744997835383755403754', '2903018011504287826977837083078876618243536108691.7982182120139153274499783538375540375383235292397185642497491495876798777089425611506982491828538', 87, 0);
    t('7450706948480263931815676534089666201543249501343620288333013.2363390095574565725753636930888727', '7450706948480263931815676534089666201543249501343620288333013.236339009557456572575363693088872695681', 96, 0);
    t('92931624430026795194601052050422978081935074004431330570898880966283901954040616867104013289897871059480000000000000000000000000000000000000', '92931624430026795194601052050422978081935074004431330570898880966283901954040616867104013289897871059476313239940573481725777439028674257740.44482224686022736666802379353596450177436360646540862097332', 103, 0);
    t('82990306900000000000000000000000000000000000000000000000000000000000000000000000000000', '82990306851493230157734491803606943186554885228825262510587906573078084401741285156675.5687', 9, 6);
    t('6222401701370060720995124192269775581874727492587331000000000000000000000000000000000000000000000000000000000000000000000000000000', '6222401701370060720995124192269775581874727492587331402531191235169526534331223833807492636818544564841473182582130883746271463758.33864577565454126287399133742667982377223970892324398921449416', 52, 5);
    t('9.3533883287368171333854475', '9.35338832873681713338544748490384014140164115677793405', 26, 6);
    t('7689143167690139182381885513183101140785982633117983145927000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '7689143167690139182381885513183101140785982633117983145926666098194195795805371649009730935383407639852812224940860185026631011307718736195101486004426009148067036220993623270703.554893914546498639904645379896385701743309641213340430258976872', 58, 5);
    t('9160920046780490000', '9160920046780490824.603676290680949462863643755188841993991104004181683287109036132495264520514762812457960795100957398771850406883934380751667004163169223166226873696375534112267346075324620328074036799168580443682759762839531019921173831038090552197014766965875279843', 15, 4);
    t('597.54597393055996102138423167817005238851089090188267749914587703132648695583', '597.545973930559961021384231678170052388510890901882677499145877031326486955832198895130', 77, 1);
    t('41812731403252171576653993755255680000000000000000000000000000', '41812731403252171576653993755255688008029063758118419055146430.39674494247206578441783399803409329218303600009463378042761175660695625070660197606229410656431891273767', 34, 3);
    t('97.2763724676118480861454894446946', '97.276372467611848086145489444694586945687106217826', 33, 4);
    t('367624717243103071259842503977651514706.098864840555096413425890871357950469954720167083181124507336219747991345280970488998793726095103108721307166546154966430083760168823050771318539997', '367624717243103071259842503977651514706.0988648405550964134258908713579504699547201670831811245073362197479913452809704889987937260951031087213071665461549664300837601688230507713185399977782349467266994837449518872498201470273074540057150629721606067232522911897', 186, 3);
    t('655221.278814451835193877739226178121062121512336985605869594921390241476564439228936744361547402', '655221.27881445183519387773922617812106212151233698560586959492139024147656443922893674436154740102650690359007153180659078904516474351036029168046998432031036742537443253330246497424651452061352458978833080071646504346145567359599660346861661588415351894510615380533743952979499683922605154712', 96, 0);
    t('80223349820298084420839172.6595724759319017610252098719828782248060879281242826581667480106302235864992857676961283598341594008056703994033303850947543837564724030148961414721803542682665907947571284', '80223349820298084420839172.6595724759319017610252098719828782248060879281242826581667480106302235864992857676961283598341594008056703994033303850947543837564724030148961414721803542682665907947571284203951312070968', 198, 1);
    t('22777163657923410463252464.2953263874427785414', '22777163657923410463252464.29532638744277854143553276387225063429568835138249099', 45, 4);
    t('982424207470113907891356433701320880315181929224300546323427805203.496', '982424207470113907891356433701320880315181929224300546323427805203.4956708774200951973132842430724383202347793382162147677423685746991552906308235793981578440001184180251517561756382764090616259312112897288186124265082856453157849140912843823944711943595261383336072725621771517674349975782033545198237516220499036994297872443880770322159434135579', 69, 2);
    t('13277304526.2086649772860051630076', '13277304526.20866497728600516300758218', 33, 0);
    t('461269620490000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '461269620489281726081462143015905386866073992603506314012381782575777277617656817932385413368073662002136777951692384037469047399731051588440561273000715329084941177352524.1021564224262326516239584704386783468721550716595349168453555331627885364706784250767425704426723598885201585891', 11, 6);
    t('3701093096023409833482844331433499766978470459253394261725498413035168016364.91435491245072064098814314', '3701093096023409833482844331433499766978470459253394261725498413035168016364.91435491245072064098814314872535186256862400420629647291869518514558039702565831823207033350123045147040624314450418301', 102, 3);
    t('987073387632686044916385352538242320515213500416017433835072596300000000', '987073387632686044916385352538242320515213500416017433835072596255022850.13660568039761362553959902486991196616645965774711562009359661833491752857543308372', 64, 2);
    t('89860270230175431535025474863491817093989573864016740870189383675152288091671747566699520666.086590324779681586633624079', '89860270230175431535025474863491817093989573864016740870189383675152288091671747566699520666.086590324779681586633624078320067400835619492361981800727792', 119, 0);
    t('70507406208878219774821804588645846000000000000000000000000000000000000000000000000000000', '70507406208878219774821804588645846076733566353431253339032381176520671632974157236536574.1719000834356639345335104145250938351627013080467301320604524492643700544072692509226092674458089128765203065315943547659266458245329400268444843937289007115976259790401842152580037510938328987297750558932470356888661640509873118850420588768001578624715549', 35, 6);
    t('9425430652894442864692556162480358016167216566914175095236742.5729', '9425430652894442864692556162480358016167216566914175095236742.5729360476266998716190185928129728504293571102350783912547230303810292860142130140101978964184869139142354083641827', 65, 5);
    t('72907879538769706399123045864941737908869944520787179021513623081960602068071281227959012134992664362017285259651440000000000000000000000000000000000000000000000000000000000000000000000', '72907879538769706399123045864941737908869944520787179021513623081960602068071281227959012134992664362017285259651433826561342683014158794695705659963815865135108471911374496715847867731.66332916929115333154697444027281003065629321152254375', 115, 2);
    t('443303216272243950390701682.7368958897669539092700571666881766398238366920699507621800279978928820480734450768029064348492781172171046857', '443303216272243950390701682.736895889766953909270057166688176639823836692069950762180027997892882048073445076802906434849278117217104685683900192991847096592793234667877379277932', 136, 0);
    t('5977435298566072878539311050303564332968823800624929931518447113199505583374826370103101050264607839698969700000000000000000', '5977435298566072878539311050303564332968823800624929931518447113199505583374826370103101050264607839698969689803341964723421.8120315890314465991351657216246621114855560434536201108143553711327196480736829356', 107, 5);
    t('252765635253930331298.8881097483277201048819816713907484324771617300896050988282158386658656804485695524233105058516414915943451667063603750933450594011139198837889562098514543525683221085892542051046882677593', '252765635253930331298.88810974832772010488198167139074843247716173008960509882821583866586568044856955242331050585164149159434516670636037509334505940111391988378895620985145435256832210858925420510468826775926030284601438335445651955377707817463570818', 208, 4);
    t('8654326509901663192970322367754139000000000000000000000000000000', '8654326509901663192970322367754138565435957254375438402623721480.750152399152322036857581509611550544116247481665092586569267817736130470408375009646037477227649615318678660335011161004286705779223651051331477061522521840862690349369360755145037371000472615087773869626172268572262641959572836701871157291022695204283106314220938736096308122810394056749911020', 34, 2);
    t('4443884762195668869392149095245783364704315580283493705947668839922216887753207422924384999049222549218570892763679485103944503384255600000000000000000000000000000000', '4443884762195668869392149095245783364704315580283493705947668839922216887753207422924384999049222549218570892763679485103944503384255577491689890833978657784084340580.306065986548152836364109671426225746415039423579318466034967440804005', 134, 4);
    t('75393531889678393323366105196047164695081515272259506387320000000000000000000000000000000', '75393531889678393323366105196047164695081515272259506387323064677929886636981811390273466.87881332', 58, 5);
    t('562660367308608908145744353799348708282580858251038407903065530487563252202860469991813804872744579517563974718881476765669904951755205159000000000000000000000000000000000000000000000000000000000000000', '562660367308608908145744353799348708282580858251038407903065530487563252202860469991813804872744579517563974718881476765669904951755205159070092805594675802768375231107125006564485770401142241839648650.6153003699862326520592576843658150932072953520415873951594', 139, 1);
    t('944631405093113755545229923894.097290195077', '944631405093113755545229923894.09729019507654719556365954119771007410271816381342746450045093518053461883159793714289585521323343806705707416355008153802590609891013479249584662181734892553062133662759220254708499875824203472827418171923334950416346823472835224143353731230681', 42, 2);
    t('4188159674682619016563587858827829618367610000000000000000000000000000000000000000', '4188159674682619016563587858827829618367609878468511248309283963172420033459858762.90553310161352862148713155881674844789616078431643595195939093177800977565471953', 42, 2);
    t('1321182374341404862786688063304055.6789762321548591160848185916897139623190999864435372198684932062158116217419435799545348302092607950461702769934397798464542440924282195215304229604330434300273150586', '1321182374341404862786688063304055.678976232154859116084818591689713962319099986443537219868493206215811621741943579954534830209260795046170276993439779846454244092428219521530422960433043430027315058604843629855881275033768398688744786566934104160335195634901199', 201, 3);
    t('5037410418356897269800000000000', '5037410418356897269744609591477.751852', 20, 2);
    t('934276225498851916280957627162677651421126.929999061459115447308920027573311724266364093042298093475510992750131620042419743461213771183282489114701407114768741589099811329742156262818330896645604', '934276225498851916280957627162677651421126.92999906145911544730892002757331172426636409304229809347551099275013162004241974346121377118328248911470140711476874158909981132974215626281833089664560482888796409035220563337961497206487368678908544362174327861824155312782789321474755', 195, 3);
    t('7.344096931882775976764247510358340095264981528339953201679799735319607877143776374264295792197725013671209495726364129599009854820290415889299751', '7.344096931882775976764247510358340095264981528339953201679799735319607877143776374264295792197725013671209495726364129599009854820290415889299750969467388702393867', 146, 0);
    t('865934000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '865934398600494122272164806372432952529512764127636377849480569444421156158565194098492003908639.816977699456925843723457333860759951810037955600889047372219302396922', 6, 1);
    t('94284208074359432305720998658038348354997030815687786329649632490436844000000000000000000000000000000000000000000000000000000000000', '94284208074359432305720998658038348354997030815687786329649632490436843815876202617641452311804260483942138484526416229593399025885.85913876544548253809880471162494539086525339649790681586523882445026952612144141699525214681368289274841889288462575111188335467313218232224986335681861803146168552286771286019138902351650623671195660091833', 71, 0);
    t('74123913822831939.35812656554023778799092603009831542355812497773706', '74123913822831939.358126565540237787990926030098315423558124977737061323821917682998269046882234447679338549431799645075449964650443289858695300645611493001', 67, 5);
    t('539570480926423156.1922267838233', '539570480926423156.1922267838233109886999535796758556738822793732840924625145634110146072987401142359300669124676862245306612773331345515707697130012903737734556827660347239076225346610082182058909237489699006288975419028374676403899775963906534304197757534462999007342371810009832721508668148515323233764877843736084528912093871106711287234512672046804197665876961188792517847966011129150', 31, 5);
    t('10603608.932421424481911625939324750822758251316', '10603608.932421424481911625939324750822758251316407118765255112166889509036586368624821177299230565696804046799349128730', 47, 3);
    t('6680.73340810116681324126601499367864449565016062292548488846066', '6680.73340810116681324126601499367864449565016062292548488846066018199357538480540242715371072088302672195855729284234918788808498369163054346879317682850233777437284911442789947314105148967356026423324830', 63, 1);
    t('81063973521700322399271134884033443461044960699353164148754665336759584576605836671939000618051601639639623685223977156844000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '81063973521700322399271134884033443461044960699353164148754665336759584576605836671939000618051601639639623685223977156844119037282788428834812307317615619304979247528103797747348447118795279080849199405936.018099782214179089687887770618964493293213467939434304', 122, 4);
    t('20', '18.3714', 1, 5);
    t('728614302798517976829254653294745545041101947208720640764.54885990277994908533264475747131718818686873010272697806673301431613601247614584981434528769241888726', '728614302798517976829254653294745545041101947208720640764.5488599027799490853326447574713171881868687301027269780667330143161360124761458498143452876924188872643099553352252317197601995669433826217127566274433188504121932328100318335360208407329035', 158, 6);
    t('2064868706915894760388649149792641278158594100000000000000000000000000000000000000000000000000000', '2064868706915894760388649149792641278158594065000821625153140896963061734756653289755492099454123.657994688829998181092128904', 44, 0);
    t('48404731639323378981722.3416345624901061524559832344783', '48404731639323378981722.341634562490106152455983234478218684279393130193926979348829861222113830777385785382863826268699305966716010', 54, 2);
    t('2008349342030138495637636998940545035424857254311989943617272362747586229702.88439645842422138549589154094', '2008349342030138495637636998940545035424857254311989943617272362747586229702.884396458424221385495891540948133856869890211761956357175716690709410288710794160304250581166397740358933855215417356478460341990866403937500998845968332944836477', 105, 1);
    t('190310727013247330244274133215840438611919244874463193143000000000', '190310727013247330244274133215840438611919244874463193143112987713.97816580369878869313576785413957290169378910942856869981035529934232130225814211933164906628540497867251651901788859513009706086437578500568037896374478957281548998581074878533241374846651323596439066822616150376070830021927323212316581515728017361796530893377354696665537527591793599998545041382507093633471', 57, 1);
    t('55093275333792142382517960000000000000000000000000000000000', '55093275333792142382517959908392361298485511522210429721692.0', 25, 5);
    t('2321089210865045688041430796133164598866420300477751566023565843085264.17872053347340245909677853207406951077548317737106925207083816364341941869630468333837474912345', '2321089210865045688041430796133164598866420300477751566023565843085264.17872053347340245909677853207406951077548317737106925207083816364341941869630468333837474912345', 165, 5);
    t('780768482258164.05077197021889278787353274182785882764239267657292579308944', '780768482258164.0507719702188927878735327418278588276423926765729257930894485574893472975513410576165386', 74, 1);
    t('520688419812080805406341741702163437353299710698799573836689884668569165182420299900000000000000000000000000000000000000000000000000000', '520688419812080805406341741702163437353299710698799573836689884668569165182420299871423919855074790784932015770577134074596641209947548.6242848202686519828541297', 82, 4);
    t('61760866160180466838831656195289976503664637722070796115081014705910798477891194963274479567884046415820934276057298945858490309110933654030642408706960000000', '61760866160180466838831656195289976503664637722070796115081014705910798477891194963274479567884046415820934276057298945858490309110933654030642408706955441840.13444303026156131513', 151, 0);
    t('57826516046100', '57826516046175.43630286889974953490563086648313916703376', 12, 1);
    t('731553909320535982135404281435649588044637486541219613008751070743854815787.130092900529215255171982906320222483683826360513', '731553909320535982135404281435649588044637486541219613008751070743854815787.130092900529215255171982906320222483683826360513907731485705823843138892728546763149214186983670119427597250276831222614983241441017705541969754010320643467334188682827448369813394483360065875510893111437721122404569085773892548889414496970893501556709650176530216559063749657', 123, 1);
    t('3818286288372819756926163510257097833781880672235547000000000000000000000000000', '3818286288372819756926163510257097833781880672235547353879899992071217312194434.43532971082259859918610718415715527806914182781142656886790905451880790592911329590078696429350158586437298962590694518570537472235291812919552047147715046506953231733312148629304596697963895799460972', 52, 3);
    t('8550870000', '8550865296.386261875418042208156314410530017953212569453462771245594164185085207470100041646790628775856434982161895086346450314120420285806421097156956399171751474671773605516242839965516372500449893695319647457120618209567029227630428193514802517567551315097709879567812557794212385475', 6, 0);
    t('863454850353013710225373986736536686898287892653000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '863454850353013710225373986736536686898287892653485655996229099646673238196685925158386708252059530922348899334521675569616189571730075596905717750102801325199672556085798918174947854371935392139859588640140651.72660402432286118498', 48, 3);
    t('69789743219872180022007460905639830352229760492473465461391848584316715153770515037478833437539400810750713873464246340000000000', '69789743219872180022007460905639830352229760492473465461391848584316715153770515037478833437539400810750713873464246331574812441.9661690044909378973391658233082645044639209768129130308685433581894885998496038510428903', 118, 2);
    t('903761418207035.4091410078607481228599348685727064440860875540898215909873544584141432992925962912', '903761418207035.409141007860748122859934868572706444086087554089821590987354458414143299292596291243363854966505213759680674750851866664130636913354847501301360486980110776616351007395699084963', 97, 1);
    t('81482321542080168111422670609602279121361882874237944880595111983339000870242028466224288244113851737315000000000000000000', '81482321542080168111422670609602279121361882874237944880595111983339000870242028466224288244113851737315630775199699611084.4687084704516285007163976224671743277751618276240357130271967760399545558331747969038193524622312831208333444753', 104, 1);
    t('2686622739445955843491621374260.635066036018764105078470648113701', '2686622739445955843491621374260.635066036018764105078470648113700948139368422370705491501220487665930549946504164459937393359660476355630915536024810379581805901164416079158', 64, 5);
    t('894546502848282428032017723660606302531221894628059280757653164938969149147600000000000', '894546502848282428032017723660606302531221894628059280757653164938969149147556617398550.706512160778950648285338503331891639354227617329246038970329535927', 76, 6);
    t('340410899511374697707136109377511314377257998071866529831856470.545183219865025746244160361682861465396624884006984399438668876039953757235718198274894027588752438', '340410899511374697707136109377511314377257998071866529831856470.54518321986502574624416036168286146539662488400698439943866887603995375723571819827489402758875243800203477986428233863328387987115453081050481659350999419762258180092934335711299185960684807208204124574124139774571647732364731', 163, 1);
    t('8102216694904203451821050619050989558332669062916534516239910874989559942318749161071969985316456677330431782493599663133454872724725560276686414839985284483525326379397355577038195027696875044027445815277523406320580.7', '8102216694904203451821050619050989558332669062916534516239910874989559942318749161071969985316456677330431782493599663133454872724725560276686414839985284483525326379397355577038195027696875044027445815277523406320580.732668', 218, 6);
    t('28199349932800000', '28199349932765738.33458894221111288923706962580810323677302213932869437992686342595258291912150583530979042742086447049821164370770878378354402162058877481845225930542764013608319917707256854', 12, 5);
    t('8087271383752242009990778669139.494947343406519', '8087271383752242009990778669139.49494734340651937962463428342142109', 46, 6);
    t('8156362974725938795740751954521660442055999679927460043263895306575991648741531451019458732867046103772.20359374090609824431551041794980141736987577205010076287830930355137764855390010160225440820612541191', '8156362974725938795740751954521660442055999679927460043263895306575991648741531451019458732867046103772.203593740906098244315510417949801417369875772050100762878309303551377648553900101602254408206125411911279828468988014959452247895', 204, 4);
    t('3647741845713936455371861308909701154724299878074836479600000000000000000000000000000000000', '3647741845713936455371861308909701154724299878074836479646924550030580636433895637430601153.82134744461893197068417832804626766917', 56, 4);
    t('79681306047729880512698632124683422444071384729659205737903319488092774328492174524395085746100000000000000000000000000000000000000', '79681306047729880512698632124683422444071384729659205737903319488092774328492174524395085746095383128513364210261961721316226920216.54597861377820847383964953335052200523680881356', 94, 0);
    t('535800527902646836304945861350822626913408563699982754110495620553575140324874336925192674768405585904861936819.72571098849117', '535800527902646836304945861350822626913408563699982754110495620553575140324874336925192674768405585904861936819.725710988491176954112762069026964883325911430555798014711712', 125, 3);
    t('45414057972242947399910.3578669506389350568448178811525863272482934007951610650997663549775985008969563191480602842', '45414057972242947399910.357866950638935056844817881152586327248293400795161065099766354977598500896956319148060284284418452003453318257877121047143308071421228899597714385051069818414153195553191513819167265206528521840039172022440342401844220715814942228343832220562972713996155477', 114, 1);
    t('3191.583407589', '3191.583407589', 14, 4);
    t('993257994980226727044850807819822354888998425648868.0924267799209482691318708934304616', '993257994980226727044850807819822354888998425648868.092426779920948269131870893430461546967602282691786', 85, 2);
    t('4387767939429749065000000000000000000000000000000000000', '4387767939429749065281066430033365656012271296201536552.48327509808711635732857279917511011123438955314817829582980967595222886583891739965107577953323280035742056711594855573699859525196091224073166593598158373154046163863143705573016086823633580009093', 19, 6);
    t('19907140725349706397216444677518989730088662640507325721862345953096291838791228042193302343600000000000000000000000000000000000000000000', '19907140725349706397216444677518989730088662640507325721862345953096291838791228042193302343628267135293467219674863359867961429039920842.2655386403756461319224247583264311316165524036620279140378024652615854132346785906048894906500338731031327117432508591942192515187786610119130785807284853866893464138273898635133448150560509', 93, 4);
    t('1810677599541525783109666000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '1810677599541525783109665982496411732943890207461902743143840078528440493637472297227629698329018310968539022725969.9724851928532302585322786562897088462405630967073059226693981123543821226198637677886297557069148980607967282676445852491491497399414774475930176219542832261747861229519337038313268902516695729436978', 26, 2);
    t('240580614780766169813850285147593814013484872139114514550000000000000', '240580614780766169813850285147593814013484872139114514552426941616970.067324774067055956559129472526869859207709775190352079807124414051188507427247445240698393880354398631941159776398261217788708069999752516250602797205', 56, 3);
    t('5110997908585841142957666850850000000000000000000000000000000000000000000000000000000000', '5110997908585841142957666850852981190058935013354443043693675711806322529550575685791403.14080782581179124007055258723454754644758278919668360801393707092978757131495853808808921244063670031792014275601510779870794484717744915430220590328194457554133671621565870971316471729163391513321573135569110807455328', 30, 3);
    t('47851047965565281601141420691656361630414356054830899550118140400820335534085309448647644947010343550952179657130777677435922247107711441900000000000000000000000000000000000000000000000000000000000000000000000000000000000', '47851047965565281601141420691656361630414356054830899550118140400820335534085309448647644947010343550952179657130777677435922247107711441980833325443210716066182980743143457414819199798704823798215287052894945387685013394.3655', 138, 1);
    t('3404081940255704326331802175718000000000000000000000000000000000000000000000000000000', '3404081940255704326331802175718084324265452130390396791651997585489982137940775672837.53842037267716003691725093508319075770110261873892866206330397757424036966680946253361150843074554279210558', 32, 3);
    t('1696251060727373274784356703261663716143055704494916040256011250285122500862243592423874018850154450000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '1696251060727373274784356703261663716143055704494916040256011250285122500862243592423874018850154451612844167026194771480989958329245955886645583715633756258465038522500606219800395961468217155236179.0494847649015717089', 99, 1);
    t('339263634087958341164368150899809456751525212064208339510591089984543413437887129820112900002005031479248743218953223968797754500000000000000', '339263634087958341164368150899809456751525212064208339510591089984543413437887129820112900002005031479248743218953223968797754452960929744915.868201485401314483705692686439756064399078752638497826925869730148574362869141498327177622338139490557', 127, 0);
    t('341750.155632476302245552868219769736692569398', '341750.155632476302245552868219769736692569398130966600417543901383433973046356882941633091496', 45, 3);
    t('466608723846733609200738141994683628400000000000000000000000000000000000000000000000000000000', '466608723846733609200738141994683628334671051759015364102918094860931065416175375881041528001.09859039371421004555018261530802124987830823425139612193777504000612355972917030042253395097876346143907724607279295654814556176662062258731669995881575808788389379074286224377232983551957194352882198683141569975', 37, 0);
    t('4015798670906419486499009806361122875528472113112162175478727834604320758200299252639892055132537561700000', '4015798670906419486499009806361122875528472113112162175478727834604320758200299252639892055132537561719884.72323068756744841669698462286586541962237340427500708737433341898617890789317', 101, 3);
    t('583500761446910949223687816000000000000000000', '583500761446910949223687816135061678346038035.68904381630170412966529391128339076188233569165355605125762890099757253445036120034111515224858902981247', 27, 4);
    t('36488328168080782883764726564399355219806928789775198657001257600671086115185222812355637774906214757539.7749850528020680535950771406859884883594615681886766580912072920758966112214783', '36488328168080782883764726564399355219806928789775198657001257600671086115185222812355637774906214757539.77498505280206805359507714068598848835946156818867665809120729207589661122147825923157035129876444245907327505816692802747764549177044496300505819677107', 183, 6);
    t('277.18332249035246473276697875605718682492745973595', '277.1833224903524647327669787560571868249274597359598174063', 50, 1);
    t('82475301134010000000000000000000000000000000000000000000000000000', '82475301134008328497232030923434159614453041963125593865735306339.5335589632533122928688971819214219704971874750615614255850897472173864211437243693630318081376788578258760172455463627581042219087032985334105415938941555312107374887278240276469093814136337143361922316568940968482490561642934257', 13, 6);
    t('989876775155857340651766468093417176586421228588171826476885336560111308330373293566557907075476114833486734885584181500179549054349157983901689663943328478683930227102105885.4264035990680328038692502040380349144703949514914355962', '989876775155857340651766468093417176586421228588171826476885336560111308330373293566557907075476114833486734885584181500179549054349157983901689663943328478683930227102105885.4264035990680328038692502040380349144703949514914355962107516', 229, 3);
    t('52326318423775538538925283977680344118101228573836264897507870000000000000000000000000000000000000000000000000000000000000000000000000', '52326318423775538538925283977680344118101228573836264897507875807795971315360429087103988229983136780775250087763512953469946556742495.800010643', 61, 1);
    t('22211335521268463979947775796088957275923204590695733514193937959600450190993403014939000000000000000000000000000000000000000000', '22211335521268463979947775796088957275923204590695733514193937959600450190993403014938291591938312910602123635498290005902172440.5345728038729516924881819609226', 86, 0);
    t('2947825252857101289827642383612408972369651630153636697618384429287025359346504224593041976087168058803631353520348773716417953875885197711728942199991131342632756746516115386581767995254233277000000000000000', '2947825252857101289827642383612408972369651630153636697618384429287025359346504224593041976087168058803631353520348773716417953875885197711728942199991131342632756746516115386581767995254233276969947104647714.108107669423626591904956134542729049784', 193, 2);
    t('8565471712461470322759774070743371555146253550531851307111371651228525976833122105902935404435074320000000000000000000000000000000', '8565471712461470322759774070743371555146253550531851307111371651228525976833122105902935404435074320795376828417668211883713028554.3737637888931223205557311776544095942292699219530183604222531747827664977599922906693079659153355681766009749509155672103379461214263839014685974706961', 99, 1);
    t('24689840079729618645222478266499621545420354927701301397139678280862839843528450198035000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '24689840079729618645222478266499621545420354927701301397139678280862839843528450198034029351016815968891684056923583441263836546440359874232710436304452621393960157700692137633295.2168809704559606020110813931511549673058061015143728012551225012157160', 86, 2);
    t('89522381462052433237551310520198358411288634700710717395276339941542867935764011642386630850083788668419916932963724186979856280964.40394329441544403174312632240416581475382550014622145815681654609212579216488558769', '89522381462052433237551310520198358411288634700710717395276339941542867935764011642386630850083788668419916932963724186979856280964.4039432944154440317431263224041658147538255001462214581568165460921257921648855876874434844383922933172242834341763', 214, 6);
    t('5370265030253648157366450102148822469957873138143421790170557721419093116287242821625698025613891538314621033527982174848000202115128874057627290000000000000000000000000000', '5370265030253648157366450102148822469957873138143421790170557721419093116287242821625698025613891538314621033527982174848000202115128874057627283997467039539130782091301005.860', 144, 0);
    t('1287639580695534261204542626542772436311426369957206684881712075358546019718896135720740763245959757404304803583472255267244870365683471960715725596383.192', '1287639580695534261204542626542772436311426369957206684881712075358546019718896135720740763245959757404304803583472255267244870365683471960715725596383.191158697227896069634120361522905333645323494140303166334908642988652963307563549150250776240764341392623104270693', 154, 0);
    t('451366094312141734649919881670056328164496913630236697787575994146332920525736739164', '451366094312141734649919881670056328164496913630236697787575994146332920525736739164.605430695752810835104088313618979697249579825566552636316592', 84, 1);
    t('690000000000000000', '683289980805740647.90272473675', 2, 0);
    t('7319782656864782352848558160793866096674820.163360225457849', '7319782656864782352848558160793866096674820.163360225457849890838274098354360291607184078039480866830207435081741647515700344232553125413161647334542106902574629655241774332021469862745791809892420432404274702708472060792141958219091883139887693570678845048514060659389493249838866196498410383941518548303335893619397729722457182533483892620792247', 58, 3);
    t('29803441429957.81292354748015659124902298288212691708309391217339', '29803441429957.812923547480156591249022982882126917083093912173392375839339461056643013136748317682794198410495607185982376859739106818264240462', 64, 6);
    t('880777917364478017087392685120824461738208000000000000000000000000000000', '880777917364478017087392685120824461738208329671168701902518618714355499.70183636880213926729247522542286630136840', 42, 3);
    t('6.94377666090829347', '6.943776660908293474214807034646989717410795843860974654219976286129560798083092860517679822347426154922384038835441555000838874082423838729476532095947331225367832435333547130535681115446401108073357207699081279439379301032591301520905960840029839469123378', 18, 1);
    t('7030784159119428150599772.7202929241', '7030784159119428150599772.7202929240398793828219873720211881295455621406299248148943351637512672513028083843799832883022485473800562443103348021256611498331616039597778904204024064069755451114803732488718445105079861291874508049129252107371825080478420673510214584424318040566871065242592380372617595628739621215086179641858696905247153531627217131317340569509662024547580638939971001227569484706361171', 35, 2);
    t('567670072400000000000000', '567670072405010271962060.8296543695848706922271240907912701277104', 10, 4);
    t('24796341637000724390646971161697311238037760000000000000000000000000000000000000000000000000000000000000000000000', '24796341637000724390646971161697311238037764445047250596623511290140358648183927238505433519964933653274564079285.41658387', 43, 3);
    t('902841599383560717561393592545842564286387807322056.202466772083136786884077268744050506469134539714195989991165433429128029763859661922603802387380468290633', '902841599383560717561393592545842564286387807322056.20246677208313678688407726874405050646913453971419598999116543342912802976385966192260380238738046829063345948582776939894003742463992134801665854468266461525336989093236435430661866243704446049518506458360473721302449475744658992543668043527246', 156, 4);
    t('6466235795971184469528543711595635703945521301563874944267700760855036893911677177850000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '6466235795971184469528543711595635703945521301563874944267700760855036893911677177845283037551448568308948068591009407191818199730798913330011348990241013824684459935635470898733.195764272937512902015882391870705970201947538952138272943060', 84, 0);
    t('15552634205013291298658445582047367845302568514169520000000000000000000000000000000000000', '15552634205013291298658445582047367845302568514169517516772176177856215137109749748001688.404674341455912754308510799890190558172849996081976831579331158743863', 52, 6);
    t('346318332549646608968296825926831978668686376.76931728918615339925270359896980353927835584626', '346318332549646608968296825926831978668686376.76931728918615339925270359896980353927835584626218798095858626981254820671581612746565907266444455708225282176858363627309137663567838289656011152453273406812669968116849617550929721767853270965965217398819014216615020517550131944921177302209460392354634082486562646945698495915205061014252413806409436934481326171559574619244915178407996', 92, 6);
    t('214163098489051757063489849211512919069257108400303239981386400000000000000000000000000000000000000000000000', '214163098489051757063489849211512919069257108400303239981386316339894921406254598610850835094056835487011920.1815174303902778554305989510967743307447078514174853018950769559252049780666694635436911961541257109833601117747168712093133317681006645730185511170403340023763736424169466989172294846', 61, 0);
    t('2951743119342985461847159119040690768984725000208772422261251844753671145603221689350352759475379822043995239495419740812610395095397629162536606273802149480673284614660987880481751000', '2951743119342985461847159119040690768984725000208772422261251844753671145603221689350352759475379822043995239495419740812610395095397629162536606273802149480673284614660987880481750505.472056638027978346052388356678586610132', 181, 6);
    t('11000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '11410029693623425603611972777567488519697803430077743315341468171354466070416367199002729142572868262073789060466842674559946603.3083648182516414199170351736741651118886364214223880592926010268493072651013916800393795233', 2, 1);
    t('278991936617705294.244443412845189739861497631126305408407194034416549445675187606210630833753062727890689834370412', '278991936617705294.24444341284518973986149763112630540840719403441654944567518760621063083375306272789068983437041129388053266870804268572641953903133943301866829216', 114, 2);
    t('46599604816838305199186135162866748158274743429139775232200397860024184213575779414287948307201149803456768837800000000000000000000000000000000000000000000000000000000000', '46599604816838305199186135162866748158274743429139775232200397860024184213575779414287948307201149803456768837792706050255797980747071406499001945937450121393689952447154.34292785419366316488329363896641226897250181049246563655312355310', 112, 2);
    t('823717422836.865667354184161849648504934255533835659253824376137994471641509454', '823717422836.86566735418416184964850493425553383565925382437613799447164150945376239299102444264507987800779137851561636464114040497870159863436344044708212231103416914960578459353837040577272317097977072941360570563668643758535399048083024771052251', 78, 4);
    t('790994109031394278069380360873501535240700201254212706235757287838388772265484137760188746418079360000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '790994109031394278069380360873501535240700201254212706235757287838388772265484137760188746418079356396330793087255349523489053414112717735030702487433110975176396312248030082120544577118424148270390.6955508221080257014', 98, 4);
    t('67592663787321089969614696614472467837799405523310825730563900707152253693839839240456306863272799200513137529360979801797410671324405026829418134375282886419537858410000000000000000000000000000000000000000000000000000000000', '67592663787321089969614696614472467837799405523310825730563900707152253693839839240456306863272799200513137529360979801797410671324405026829418134375282886419537858411153696650957071193652816296413238212460536749821336911460.5102518369565233943', 166, 4);
    t('2684294525133065755010138120382917524672483574131068402332463366729779080551299700000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '2684294525133065755010138120382917524672483574131068402332463366729779080551299756544630036445237324037251824589867465265736039691305717512093150719022742927923062163902405265184686589154996415651110523244740431657428912203341.1130610118660752515385', 80, 3);
    t('472227186520667429881538758172493122003395896383795618927732525326256165100900000000000000000000000000000000000000000000000000000000000000', '472227186520667429881538758172493122003395896383795618927732525326256165100964289581200978723678084886492546256919118633193846770039022517.307', 76, 1);
    t('83657244708707444817258472780650682882566176917464633281660063906531511830497036486180753561332304510739565221713116254628999281133383924261481994443601558798000000000000000000000000000000000000000000000000000000000000000', '83657244708707444817258472780650682882566176917464633281660063906531511830497036486180753561332304510739565221713116254628999281133383924261481994443601558798301960076231312954556904226706962999924978597893438519098090866.1011659470946967460', 158, 5);
    t('77500871021751185524391170909628061204202226025966413859027706159600000000000000000000000000000000000000000000000000000000000000000000000', '77500871021751185524391170909628061204202226025966413859027706159616578901759249263670771366559882307904878099611763677744059658184971340.07561806683130719563062813388165053430137171646051359158612382443154301907156892704791741784391432513557791112874607265308421404595997374', 66, 6);
    t('2834070003205513826140738491796837583798432512953221588640559225275604799348709943910163936892621454382715318855616113409999183013159713000000000000000000000000000000000000000000000000000000000000000000000', '2834070003205513826140738491796837583798432512953221588640559225275604799348709943910163936892621454382715318855616113409999183013159712384878098750337053741691334351640281157947961912780186694635547411003.986585566180883858592989726403444232792575321182', 136, 2);
    t('6299687000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '6299687871575141968888015547260987313814397425993139178293515784824694846373090105645003425288764797.7735097127955426173348652791758700502940529197401711281132753525911693474884858870088430759787932720838772336727341799111684871756935598419112219864351847028726300992286937057531612974887493327371870858550500289', 7, 1);
    t('776021134911756441565786062224242000000000000000000', '776021134911756441565786062224241906441252675967427.097668885343153848267872324654617228749352591891091463536966901580781907226484729601888446605973839514545673571805490283335893528506106400245004168503503609049287817297979352950651879452815805433035686', 33, 6);
    t('1201576594347397521971270871444629935657263801231251123547796566688614286485434751930968445.33169878280431415652605638256792691012183677125891446695069163324324512444960457457291087792', '1201576594347397521971270871444629935657263801231251123547796566688614286485434751930968445.331698782804314156526056382567926910121836771258914466950691633243245124449604574572910877923732589798837', 183, 3);
    t('46977516561122966264900000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '46977516561122966264921570202786790048947251289237799357310836701483387901043264411808197584078361128447570664873112220024.7458471987826828007555195427773002298427872952312707080497935196671603352872596978324993591686964956130317215903054815398197138958608', 21, 1);
    t('655339968565335000000000000000000', '655339968565334933497035180660255.0084230587185499106515300700903610611146931731564065154864759313758909553583789463366741826135931689359487308354947925304094297210886463522237483508398237362211472', 16, 0);
    t('26992417487882795201113087680705502881.159923263687680843532440717015672165734800131397013312883445321384701483837042368914384306049583526676475039707946111444017471906758168144499', '26992417487882795201113087680705502881.1599232636876808435324407170156721657348001313970133128834453213847014838370423689143843060495835266764750397079461114440174719067581681444981721673011106818278546808081095607990222288684680134540564152427969384153156843', 179, 2);
    t('21491419366782168442193261217744048875821360070019566148657000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '21491419366782168442193261217744048875821360070019566148657346513074288147191371748109962744363054666909410741606134878736216774091327505806554799373.200661094557', 59, 3);
    t('793635702908282611380706092.9556587889812912284555223385608697504144981387528869946904520388039472930262214872491526161016973902169485791', '793635702908282611380706092.9556587889812912284555223385608697504144981387528869946904520388039472930262214872491526161016973902169485790946923297462856215349885215457143434657317830546028921905125440588281768588667257436161099339612994913943597147895391991662227598411816047203260388295637890428151372824707952738718120457168719122680', 136, 0);
    t('809096314761166156505910284062366456968788695385314012253399600773794467718752606318188403983258288368050782754617.805844516077806589134083581062422526422949599872491878874028012', '809096314761166156505910284062366456968788695385314012253399600773794467718752606318188403983258288368050782754617.8058445160778065891340835810624225264229495998724918788740280117888200819852880539729294593747565599354612886589', 177, 2);
    t('837503728221283163362026359413260769206435400000000000000000000000000000', '837503728221283163362026359413260769206435373331726574793703630869991875.6311900297338073401800935145687478111439963736459699555398', 43, 2);
    t('401195215161444513070115712936219399029815754051039233346499460876887170838182773215643166723.4739088306294979870878486337', '401195215161444513070115712936219399029815754051039233346499460876887170838182773215643166723.4739088306294979870878486336335584983663924163305310690316591334583441774861614117838830492794027944999459411825532', 121, 2);
    t('560768897953444083268445195345239590290181531407940034016420767637786531555823676865064259982435701056777104855419534131.604', '560768897953444083268445195345239590290181531407940034016420767637786531555823676865064259982435701056777104855419534131.6040864586180244', 123, 3);
    t('566771158194510482646463775889441102866766129638589564286642576108858451345635851.176594842412206101707821832898871', '566771158194510482646463775889441102866766129638589564286642576108858451345635851.1765948424122061017078218328988712055610060923672586522282366563957322291443430878355099574262256339930517750018782132243', 114, 1);
    t('6222442482536296989202321839973023169860415199895874452554959810510562317515951288.23660397545544349631', '6222442482536296989202321839973023169860415199895874452554959810510562317515951288.236603975455443496308215880147337968022828945', 102, 2);
    t('15787616720408025665490879013780790708259985554600368760460641367783007027793680052812916556695480630350900209349929954910705747940268435577960325566042252.940283588413673728', '15787616720408025665490879013780790708259985554600368760460641367783007027793680052812916556695480630350900209349929954910705747940268435577960325566042252.94028358841367372765660267865349180541875424070076614734370076917413946547167791362766520617', 173, 5);
    t('96516501111942447330466319475466.474482953045462319588', '96516501111942447330466319475466.474482953045462319587481324727766654360167455509273736367427120583431152179534457305105311205043159505685246813856160393491487061492386496782072484155182741332704259434787454914595619820250414914985809974239961169131442044325213712541017264997505948715752529233212626933423190150873031256642046559725834825095089796618365650789979856437578873399', 53, 2);
    t('409500135676058824738963406859332600281666354595324191927676860401888553076263726047255675119125696235120814.2713242035956046631681162380797766349062346360318862', '409500135676058824738963406859332600281666354595324191927676860401888553076263726047255675119125696235120814.271324203595604663168116238079776634906234636031886230041611449140969277906818420648179287047125084791458781564097122730186843376338', 160, 3);
    t('4655336656975930244053334102238945136556425968029828255622000751351354817894188002277854214644377066570391112764500000000000000000000000000000000000000', '4655336656975930244053334102238945136556425968029828255622000751351354817894188002277854214644377066570391112764554497438162779750121253559876509373410.30876173871116990247885468638961210666572', 113, 3);
    t('469732505545328930444786580330818575843535830148532817415267886883025911308843912.87126356941814558807826888191484607208161326697241687600160799671968739153620482749734283116271', '469732505545328930444786580330818575843535830148532817415267886883025911308843912.8712635694181455880782688819148460720816132669724168760016079967196873915362048274973428311627138900172876843038595205037033648958148960854578777034032741920164045568599746657490020', 176, 1);
    t('103.804063843326550906', '103.804063843326550906302218880894581530792964020713952789630849381679115288828578298390500906435743880022019214578718545072358915096407015631050599000206950682985975407770219498984990261593947953658117591839849558162614019683984932058561484140741887751195369289541291916722867341844978943080210527073856034628852335785494035434861719819117', 21, 1);
    t('3015294588413200284803720820982225309810668195567286772234470810.3838028800763269471', '3015294588413200284803720820982225309810668195567286772234470810.383802880076326947140332350014186974098904478151531414352517588791918728784123975664073216931475950854301321123916079270644325008261023773', 83, 6);
    t('7505582695285686095206751476116802408803922137681178944106550.407757514354597514808110436184072636971935997990919795575958139997803621881440159991012151921200041190807937839373744919188824', '7505582695285686095206751476116802408803922137681178944106550.407757514354597514808110436184072636971935997990919795575958139997803621881440159991012151921200041190807937839373744919188823723367510836751305264839118326123729579972117323643602509084', 187, 2);
    t('3792495069499973047733644104200000000000000000000000000000000000000', '3792495069499973047733644104237463387881028440047815921276920004601.0', 29, 5);
    t('920590411185024929623637017373727232120650188799732344447567190279612301735075705639906036033491390337933048768767670294.7228987254740427941907278992883516767524608846879776535973698', '920590411185024929623637017373727232120650188799732344447567190279612301735075705639906036033491390337933048768767670294.722898725474042794190727899288351676752460884687977653597369717056263750265039871517513916130', 181, 2);
    t('284741478486898663752062889788911022344230000000000000000000000000000000000000000000000000000000000000000000000000000000', '284741478486898663752062889788911022344232262217213015742672566377247880072042073196928874881455185565947732038239805822.5000940645804214118219767259811613701890686463', 41, 3);
    t('7135297521416123674965.385083109129550843406403047890000033970757527220403202850303892836', '7135297521416123674965.385083109129550843406403047890000033970757527220403202850303892835058131755112596832076467868604395144480199099108239500168345909674698129465821793740144087004147711016398253270267589898698018652804179715023454104465122117793459245293525455854234092990229541294921906518822166290033119261794854141721418', 88, 2);
    t('958921.51415285128136715646065112278233898370444916878332068245860888754', '958921.5141528512813671564606511227823389837044491687833206824586088875437253310917657874546914229407154402659312463039482980789387249260005357057', 71, 5);
    t('4193527144896046841836076407470000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '4193527144896046841836076407461992788706442835944485090763176358101515827436296572313643584020349651841115924459207558908569954824787659.4066468792734405853368465792753108703586766329331314358730313061221019047359808174619001439425', 30, 2);
    t('1209646657.832757332153255354800132', '1209646657.832757332153255354800131377623310936598198971857668229121393466933716455329933486148760096392377575380047557136328093481265644138880979896912210080682144032548584081636371740458157137198381672435183761271295704064659362374540259091884680411808957990797627788357374051712884821022366', 34, 2);
    t('40855933334394160810786811327986605432299725427808964740519692242567898639948106482319180824268372000', '40855933334394160810786811327986605432299725427808964740519692242567898639948106482319180824268371878.028414742036970032305574122304259862354322771227', 98, 5);
    t('715363658063531.7716384942004619983587524228499324326292476422846807545125033346656488495227228264422712556288886123898521653', '715363658063531.771638494200461998358752422849932432629247642284680754512503334665648849522722826442271255628888612389852165293659883123478962166994671279584323061397010318465526876203863376048825950279846349672455835588509707729607552895620173340177007089297', 124, 6);
    t('95780278998764396364850860295689904.3698998829117', '95780278998764396364850860295689904.36989988291178524447748250067831423769678324243913539315791555764133971392428761558505395416327062926523023716529539071269748410824955658742219758240835344349633118394658818461663765512515173340860402296559091627943970799600306024664691652982115777926239617335096', 48, 3);
    t('517856639056307.73263988632463444881690437912024579996279293234051403341257909911439274531554433045219766037', '517856639056307.732639886324634448816904379120245799962792932340514033412579099114392745315544330452197660370902275486100175285641914157152262851960571288623867672858232631727770190204515444439803189532555714893387786900027846430030453811061674762083300657045202557210406030238584694180944244139', 107, 1);
    t('590180316821002420851933926116241595082052965230994359223461577.1041557766211955915719854907757139', '590180316821002420851933926116241595082052965230994359223461577.10415577662119559157198549077571393161012435680830897', 97, 1);
    t('16802899368112203034887936761226644115748292102044561291992563839000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '16802899368112203034887936761226644115748292102044561291992563839157546521781784446002390839782007841041827757303627281610231177243534693892325713660083189283447017602944774.65378628982753127718826095223714725021668147869915804577452031243166265785', 65, 3);
    t('382302696302783571547011095349400000', '382302696302783571547011095349389952.7729782719', 31, 5);
    t('538979037966934305549258023290451591596193309467414218974763481.522103542977124993052', '538979037966934305549258023290451591596193309467414218974763481.522103542977124993052036648084', 84, 1);
    t('7072118055853679198401751472207402253790049831902970477830985031052655475661975662990000000000000000000000000000000000000000000000000000000000000000000', '7072118055853679198401751472207402253790049831902970477830985031052655475661975662985133961399385083867719523768505421932905731752358485596731942126350.33980342498148533581', 84, 2);
    t('6488438105114388815329799774812456554970114918799169535689677309189028308251350346747012185492855242129189864199521149728995000', '6488438105114388815329799774812456554970114918799169535689677309189028308251350346747012185492855242129189864199521149728994956.4402894063385887384211363602537035886849554739359805081227212083654029231811902928207383582229725531665054665922374978150585407772', 124, 0);
    t('517816497578713884279376955081202866162429340842621330092862246079376643267690972111580048569721658819240877656121498725144.00991395266639109190225566019174584918775886749776152242422115235567', '517816497578713884279376955081202866162429340842621330092862246079376643267690972111580048569721658819240877656121498725144.0099139526663910919022556601917458491877588674977615224242211523556770544063994210809644381121235579', 191, 1);
    t('401572523580983096831494584.9572442957426011', '401572523580983096831494584.957244295742601100004192220299799708600200749040047134017144439147364346269', 45, 1);
    t('57795317859627157467241188534255390768017488243095545114040887799356.13619621174078055198547169983289028858329015660706561192721292246217', '57795317859627157467241188534255390768017488243095545114040887799356.1361962117407805519854716998328902885832901566070656119272129224621628896159670632065233501370778362786443734522169818627008574847475540269615335178904382917978265717744088798061605801190058', 136, 0);
    t('12751477952368032379173849759525549328103982631857215035504584435625169858728515448104275849977278202675848044137466177593804035038126519873397371960973381012985026288773398623001698993000000000000000000000000000000000', '12751477952368032379173849759525549328103982631857215035504584435625169858728515448104275849977278202675848044137466177593804035038126519873397371960973381012985026288773398623001698993611331292073402472780734749105324.324069', 185, 3);
    t('650371258.634439', '650371258.63443856941518666294221994935216515708472744567238559240042189514709081748899583240694834964927183302432648770356650496278268083533071019278823218133202882691691953075898160962519955876619798092923494950851091467644569179304990379075864760216997171840660510107566643759538737651112492352985699235415110667640859606683046903769329297', 15, 6);
    t('69376723101193196443773.9470498664650812270073126847217576164801764272438', '69376723101193196443773.9470498664650812270073126847217576164801764272437005767904404646350843975472483091295246008466974087319830955011668785698114640799741903219171832540279654933022265915473276903665974460037792097240214125632668599222780312749492608112111243', 72, 2);
    t('9917616043337235439094236899751102650967820291005860846740273783914911826376024537636586150700', '9917616043337235439094236899751102650967820291005860846740273783914911826376024537636586150651.7784092487603657717150', 92, 4);
    t('3636257910723001829765912060987281054.2181348791961338050036670701949873442326114481600169874621004754', '3636257910723001829765912060987281054.2181348791961338050036670701949873442326114481600169874621004754060196308671919742425978057808356085642376089112234422655096161835359627024244353528788640359860447920591952231416085495257701303614844352018914336749', 101, 6);
    t('51697892800000000000000000000', '51697892790870799108149230797.89816888922061293128052924928002391533305626693708171819925362625209412469589996418819517866639721037645333776088296276488673711074987870047936085782063358562031874', 9, 4);
    t('710556967625053266030885400497279960209254266051982375141256651761690000000000000000000000000000000000000000000000000', '710556967625053266030885400497279960209254266051982375141256651761680561088093136742799509193281981526001906273295258.82281498362536573711244073693819807887595144532930147989791078038338285533', 68, 0);
    t('95533180567713525531555726228072881139316543441714740570304409265489233349147536231900543327837800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '95533180567713525531555726228072881139316543441714740570304409265489233349147536231900543327837803722378581183269739463486564534266434619176788631986630194755264116397960720616012728099282762.563267315452300731802587883874550984269', 96, 1);
    t('512970009205574966589023927422462592361899174464248429759888301555293187594703296175648160629051645591209163171625.538988987624441795148', '512970009205574966589023927422462592361899174464248429759888301555293187594703296175648160629051645591209163171625.53898898762444179514826357393066620116260128083177327078137361028542628461639188351879120543504493928622498658887814495218656113338545768235891145178604006486003957885961873350281443292170864533949488573227740484002973581889', 135, 3);
    t('71441315006045475', '71441315006045475.91764098564204749043831918619287560541044153107322604334728606858704847751167193866081673355447795951279367437994458191410609116772204316384336637940647517058093145805569473778461418167424426368320590689035967878764747992887901018235212088083777993540164707248237668491362253983913915399962901698593033755201624974496766928642719958', 17, 3);
    t('900000000000000000000000000000000000000000000000000000000000000000000000000000000000', '932294269120764011106254432565458007696943968692678718160325976276263874035501557895.1332201757970470102814301619028575078643392414519015718072242172071973', 1, 1);
    t('5102484136115709782552535904502456781141116627355863180572.18152961196805822848358873484444469785245365053255816741124460517302314377558776084189672340217971775757', '5102484136115709782552535904502456781141116627355863180572.1815296119680582284835887348444446978524536505325581674112446051730231437755877608418967234021797177575678961728404433725722624200950223918363893044794051469395003586727876534458723739765774498909967071', 162, 0);
    t('788192499297300501525810616198860882705487300000000000000000', '788192499297300501525810616198860882705487340651766174137794.8941821939212487452226936685415601735105580719233254659936548365977057268252676', 43, 4);
    t('91084879372063457938.80552813043096353637955113896712429334731837931841352929572158312627577750484422941914931221446431754782666750664', '91084879372063457938.80552813043096353637955113896712429334731837931841352929572158312627577750484422941914931221446431754782666750664002043290211869065226242588538669267162858810742118103956425637906725623358594855607911651948509442844784897378703431689818230505676810971853011829513549811004844942942456819174654', 133, 4);
    t('507514025639750448289445311978677847772464914310537967142413416209309479470183912146218109330000000000000000000000000000000000000000000000000', '507514025639750448289445311978677847772464914310537967142413416209309479470183912146218109322377215537895104287628673830002182044762559968141.420948901991108179539960067251072850585324835153785044698552396276670672087461072816218226612962759465318117136814779429984246789421362', 92, 2);
    t('2080510097179581624544680752275329153356221614340545287411365220000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '2080510097179581624544680752275329153356221614340545287411365225833204869247005818199410900352140004151645443663476903355959968017107961446735587445238245974766455999053020060402766388410302829.5896615930948615458949966193168921329863374485741354379126851903772753480742514', 63, 3);
    t('54376830000000000000000000000000000000', '54376839743160154466047519259227565487.430560541172058931707758', 7, 3);
    t('3779221099434139096002899015767520912.145184757', '3779221099434139096002899015767520912.145184757', 49, 5);
    t('5992434798981748.559128088184947104910638057126097368', '5992434798981748.55912808818494710491063805712609736763856898647505419585002576863250044877395999405697210990579787588506085089873390522209261894434238387240949012137310281108179711063044378934725726370385017368021121221745521798575504758372592600833305045937866423000024', 52, 4);
    t('9049823098925668369092007887096255009929017270621145079890787964647485423435031786915014759618789472783334014980769641783639669130.870114589506858143314636027815445227138351', '9049823098925668369092007887096255009929017270621145079890787964647485423435031786915014759618789472783334014980769641783639669130.870114589506858143314636027815445227138351373170706482463431511501483705554127174563460396716269073834137565', 172, 1);
    t('882820520153895270241572283947242243175043158062439088702918959303352031175995407838843423240000000000000000000000000000000000000000000000', '882820520153895270241572283947242243175043158062439088702918959303352031175995407838843423241309319065822506933255469419112232974990470592.27699392389963267024996771131871632653851149321597671866106967465571834707173623706591542898032043914484645699684827913874792898870568209089322857365', 92, 4);
    t('66115230467890505.99322994829450524565845476771065514258573744221319934263524377248873', '66115230467890505.993229948294505245658454767710655142585737442213199342635243772488728320363470760789230230067101547307479730715', 85, 0);
    t('47798248650973225975612240848853550489266603698099495211806305571519358.609180321337288528547028704842643822165079037706277746505217', '47798248650973225975612240848853550489266603698099495211806305571519358.6091803213372885285470287048426438221650790377062777465052168601216179633615986232472716404274972036755961421', 131, 0);
    t('51323561216992178175075481702314853144576064791157572485474294267321.870649487220210829296905801', '51323561216992178175075481702314853144576064791157572485474294267321.87064948722021082929690580003662430140737625235188913021002145215567863871498791015127351184777057121365489885739397226913181309166522018596062398882198033403910585114055092650839737513190969974756399878301', 95, 0);
    t('821784138927860844821.477452392', '821784138927860844821.47745239197809071247302104007410010468491006152659932370111378950498607660588813752760478140464262886734911808319368074616036469591024383220602867357587110398147390928115846185867331006398732835469484908847514832181288462737250832919782', 30, 5);
    t('895000087.996', '895000087.9960913155986998937', 13, 3);
    t('174301840649601064164657732283404380857858867307823627900000000000000000000000000000000', '174301840649601064164657732283404380857858867307823627940438292223546783097856847553150.926695582407333080547307323269622471262371', 55, 4);
    t('17527170978324873776666986844696439949000', '17527170978324873776666986844696439948686.14669438205150496490390398191731979634354', 38, 4);
    t('5628045210000000000000000000000000000', '5628045212118467374174826580665660081.31570943013305326920940905302331333564735777', 9, 6);
    t('33671642014599003720701628185663889608742655212640681786287779601638542521593037993824741854874013127000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '33671642014599003720701628185663889608742655212640681786287779601638542521593037993824741854874013126414506635355131123635285189297295398468295915182837601310245916050103040562080468846425963029.859325876631842784859194780100236948', 101, 2);
    t('221567395495185577125328394265056550305189403234038977758444153974.91854025074735751673601573740940697887240996480909719541162333027343551669261150523540358438715082290961534735669356372660156588009122045', '221567395495185577125328394265056550305189403234038977758444153974.918540250747357516736015737409406978872409964809097195411623330273435516692611505235403584387150822909615347356693563726601565880091220445472801492993191868600780670', 203, 0);
    t('70444253802207298921006326266676328707557199351173604322325241577727814518778998659244369773340864535255285637681188860065130442934381384854123834445315808621047427679185025259035187915230755737.9434249015115190861381617200758023', '70444253802207298921006326266676328707557199351173604322325241577727814518778998659244369773340864535255285637681188860065130442934381384854123834445315808621047427679185025259035187915230755737.943424901511519086138161720075802299711170578', 230, 5);
    t('20917565779741657799744899594270.140214907126629049182940625836664689968393574188160666878', '20917565779741657799744899594270.14021490712662904918294062583666468996839357418816066687820603382614430', 89, 3);
    t('3382309898399151991687718577419346.963888628', '3382309898399151991687718577419346.9638886276318308476583561988872230800901247317321228275038800184539229853081541894334128231684999730132422763709058165978042723721479594441434781684128092144619164191981012419609401527403315424182914238461501920206467325322455093583535641412966794659584020932943152312651927411128878823737447212877891229015369078595253327470655867976540564180928908', 43, 5);
    t('819284963448875798644810156701633878750906622788526629.9910725681256496497183797319894576280058706572364325069214427297089', '819284963448875798644810156701633878750906622788526629.991072568125649649718379731989457628005870657236432506921442729708828044924214332336211821726295700281846946217644763550503198042715131633063001357578271', 121, 2);
    t('96363152617131787.739199659360404919440317001534417867169485659071093233681633149835323593', '96363152617131787.7391996593604049194403170015344178671694856590710932336816331498353235933440895062483283134094036709681380872692661553149218750587979078711410510304135944859640810104788991886166121986610804186051415552884105745122226057739615571969077382337148421167711', 89, 3);
    t('8556651168226810038980776025326235004954019298688651.208793', '8556651168226810038980776025326235004954019298688651.208792457685750715274608964680851039979344844656588584739358701930591915989797770647956719072337560704157793729390849438', 58, 2);
    t('827335214000000000000000000000000000000000', '827335214324946440822874717481577422126657.8173378456884', 9, 5);
    t('735692413963571662852967624176801372981084959161727318935444248457001466373053470422549136000000000000000000000000000000000000000000000000000000000000000000000000', '735692413963571662852967624176801372981084959161727318935444248457001466373053470422549135875316686662800809082072956888499491399793473960200524968561500196974737.00624592119802517351753183182782895062528437552237660610976578482850578962329971713413897706431520886388394013953079178176154101663133499056181887', 90, 5);
    t('8981862871000000000000000000000000000000000000000000000', '8981862871146136860885168404981963243075100840535813816.585755411136228766506708187975372178205272606031036421484249689776004885073709905930291626196473244332925016930393976438045366174176366242881717609733807413042165934918982886674802334467132792387850443540692957817574066184833268447823150147839417288', 10, 3);
    t('9501134025528298826351523737244468231123051249141762010000000000000000000', '9501134025528298826351523737244468231123051249141762010669239464599582264.0225561711349979322522851837142036244045002809964795440008531964290403281264086815069664575138329997503031143598245434219189710390258557307', 54, 3);
    t('515401571121407000000', '515401571121407475681.9719519569954644025068450082540825819991904638452283941417157727714492310093535871356979865178391283', 15, 5);
    t('43482197121668528142328870837299842055437277753027133626248088801950215241362325782865600000', '43482197121668528142328870837299842055437277753027133626248088801950215241362325782865642513.19607398555098085004313312099237093581498423178984244472671147163968117567641522169980116736196338744602752990742600552343579081609102260599623508472288758364228429652920911961310597524891682915362808717957', 87, 6);
    t('354413859388520370379110685625549681092.68290073833004040543483395', '354413859388520370379110685625549681092.6829007383300404054348339519255327629153310381904021060735045071596843020303275646550775729147905840987399476011722356348189980004660323339695027395532759159535966367154335739964558919871823931627956140808147058487450517946579604525347191740941407298394571183689341076890942743279902311265452660086518123044154277476512577634369121509692802885039108736658653', 65, 6);
    t('734772961029362500076065124419061049470659952618686751.1372338554263125629131', '734772961029362500076065124419061049470659952618686751.1372338554263125629131026112132445247173583691902867693679709206574231805866556439519635865811690117726303288654441815844086717241092849050646200813854071298624116621604090179403514141659031194119626384915041', 77, 6);
    t('7740070238561989943158473363922955759843778471067688903944547215656320600000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '7740070238561989943158473363922955759843778471067688903944547215656320592772318420363726082716511833737039128007705790456230185958633266164459707451640744075313600618714065992722327.268930651842822684964654138953439396938462889903118093722055232866240342271452702617489809741924488129493', 71, 6);
    t('2541806397974982475881130927136713192263060630090380963799994436194425959012355460971331567649532310383381.9693561358837212150913900174917032448061964652', '2541806397974982475881130927136713192263060630090380963799994436194425959012355460971331567649532310383381.96935613588372121509139001749170324480619646518527496371220359294380281656', 152, 2);
    t('5101603194569949284613438071481597369335611055613894055454179246534706464674410.440690754898969385470797253107002646865964081002085463299', '5101603194569949284613438071481597369335611055613894055454179246534706464674410.4406907548989693854707972531070026468659640810020854632990595716741742743952256361427124681851433266480433133403266727577724470070541620505148377', 137, 3);
    t('7668259072603120421844943398393386899656363693468321206142000000000000000000000000000000', '7668259072603120421844943398393386899656363693468321206142301786925051222807286655059225.1043653066272968046893164319212353065465833120614537595', 58, 6);
    t('40647264743530506541384704433817742288976126240756944213019309444039295898.75402527670056341338631950832550865565755774945898473268247840725196773937953454703858788822595734219197650370136192042125161838', '40647264743530506541384704433817742288976126240756944213019309444039295898.7540252767005634133863195083255086556575577494589847326824784072519677393795345470385878882259573421919765037013619204212516183809532930296781018', 202, 6);
    t('73346887189489769281418.0939843565223434271494587856352280972059553557220262021410213495611958623881', '73346887189489769281418.09398435652234342714945878563522809720595535572202620214102134956119586238808866678919019845526600800967075973221952841083329222452884265045844881412452005765957767826182818563797085479394097154294403054837135891636122162183324', 99, 4);
    t('1635665532177450596318361279.57645156114973828939366794891510806709124490929241295562204336109509021825315362044482515544966712645425392390268607', '1635665532177450596318361279.57645156114973828939366794891510806709124490929241295562204336109509021825315362044482515544966712645425392390268607257959186787', 144, 1);
    t('408715910997658011882602579932950467458601796921364338583687982313012489216095323992176957.54833479547310531067145988', '408715910997658011882602579932950467458601796921364338583687982313012489216095323992176957.548334795473105310671459876818903870342099', 116, 6);
    t('976453474688238330242184985576334131772374526595559178614156831426794332249183793379243037222207097071091850.142284', '976453474688238330242184985576334131772374526595559178614156831426794332249183793379243037222207097071091850.14228377392035617523222377894854634102105120586602795599809263788722239893829858595901109985035234899862131587349', 114, 2);
    t('4588324365185450598778.61040783742', '4588324365185450598778.61040783742458472737319625022328346216219612714571479814405039801326557344746338644944824904888177574419581406656149327662774214477291370446865090872', 33, 1);
    t('21558292168419201790857802962477704769514545373329483226189934539658892633.1250242698604418508173825193011698', '21558292168419201790857802962477704769514545373329483226189934539658892633.1250242698604418508173825193011698421651302496495666591750512849759681440', 108, 6);
    t('79885381419254951162609772750023.3842188233882102852630734596506642447188155406737358', '79885381419254951162609772750023.38421882338821028526307345965066424471881554067373588', 84, 1);
    t('922730964585582418660577420360621539857247754977813138075444759775013190684285415976146598615251731.649398815987397756912538125971', '922730964585582418660577420360621539857247754977813138075444759775013190684285415976146598615251731.6493988159873977569125381259717462178698841694282520361760520490443434745615367645689587728812963879078521138', 129, 1);
    t('4440964151844075364956777955950907548988478750986515462536994452049565158660896505588.9571427', '4440964151844075364956777955950907548988478750986515462536994452049565158660896505588.9571426958838689612609613121000568392328226463313200431457214738399190525018694123195518835000795407764484639596451257294825750665867264523803', 92, 6);
    t('4560051385034446704872484929956696008141959732352721332.285', '4560051385034446704872484929956696008141959732352721332.284279285850371896296619524032254841011875865784977981535873988162228181529348627754883838909081986047315933584777223813164883164829676566689275673671306784961873975525583322578564591080258105395814079249590172676', 58, 0);
    t('3432057473042540689847850396655168838025201.660499637866857092451140343197366054935291447233833', '3432057473042540689847850396655168838025201.6604996378668570924511403431973660549352914472338325869626780359986433060960332115589801281666730657320795947033961857645156107635907040567110371405982507169785602759065856398420336496389337436184942514518313888155645658337588476165276054114843', 94, 2);
    t('9154571348704392396782990424407517685.928671706630776479355000280800113857713348395', '9154571348704392396782990424407517685.92867170663077647935500028080011385771334839582809741861283098352807988976927609995473171655003305120', 82, 3);
    t('1910736119602242229780927065533180260955541237319000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '1910736119602242229780927065533180260955541237319928818936164189998298827510705361008182750948401901407796306604797093612612768802835072839051884339.4501596202461036448307173496633159765271887277', 49, 1);
    t('13091910086792561477312315.37661116764866331269351213801332052966929865092006388080264347228062704630893081695587765712966782058912596435515902471115646813', '13091910086792561477312315.37661116764866331269351213801332052966929865092006388080264347228062704630893081695587765712966782058912596435515902471115646813', 157, 0);
    t('40720068992.3907920490170073426139977225009024149825649203813399611695395097166126509779275', '40720068992.3907920490170073426139977225009024149825649203813399611695395097166126509779275038130772932091198264650531414575865245604344608309357282318143606068330882288849275257908153326522419589978588248740283793605429964482724355057043553', 90, 1);
    t('497194931825325546249245919513670728155862513028438812794848294005536294722002521295038403501707124940807183406462871692369901377644721974953171347408154416372095765712093566774952727753.31167489961838486661532402885049025697334', '497194931825325546249245919513670728155862513028438812794848294005536294722002521295038403501707124940807183406462871692369901377644721974953171347408154416372095765712093566774952727753.311674899618384866615324028850490256973345834863340359828', 227, 3);
    t('4300593231473217687083633433748451168809332663350701714272085519029319113910650627541841085280690805873085507.920685374975472568139922039623265954767702075905388622317726011540606380121147998', '4300593231473217687083633433748451168809332663350701714272085519029319113910650627541841085280690805873085507.9206853749754725681399220396232659547677020759053886223177260115406063801211479972668654964119592319478920623699929889074003549368825394504453', 190, 2);
    t('700807855129658268894599764776502940057263505973440409184579333013923961189932098776653252804695349794404737371201469616614331013481.28643194110883734443983791858938822662910335489917217', '700807855129658268894599764776502940057263505973440409184579333013923961189932098776653252804695349794404737371201469616614331013481.28643194110883734443983791858938822662910335489917216999292343575629740316325548198317', 187, 5);
    t('7456.37627084154528485133803934088422373926939894977494357726557677', '7456.37627084154528485133803934088422373926939894977494357726557677', 69, 1);
    t('1716680901399001362841717739161008152000000000000000000000000000000000000000000000000000000000000000000000000000000000', '1716680901399001362841717739161008152281445208435133555936686100421473958370093801129563572905135474470448445665848130.590349691909285925055729917860739517141847552085646927651159939677977462', 37, 6);
    t('94858780289773612607045402764339996874879756995518790296387812174549841279307329695585692010017023195725496860490487529379394422407.22579902907800588', '94858780289773612607045402764339996874879756995518790296387812174549841279307329695585692010017023195725496860490487529379394422407.225799029078005875278215', 148, 4);
    t('6917283774233394650873600', '6917283774233394650873609.57026519760526202428260660760566084831408668936992265974086300852422118732259674439017273647318987690489638229453305857239137618922789228047266223977196364428543030548387105822667671659555632318116793772139223716598277207422703597617541073804340157668682760274013028208324972596197070437299118120389668051532710063457585407044766323', 24, 3);
    t('69261784116369931496565597463404979074037227200562303047737607995493614768591088947695409057600835150000000000000000000000000', '69261784116369931496565597463404979074037227200562303047737607995493614768591088947695409057600835150312035989321279615118039.538531366818423323701601532799451675309', 101, 4);
    t('57510040486515037867501033970000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '57510040486515037867501033970269003843263597826917066492965686054921721253855424248803808148387611733624772636775776764701093639027598054785174140760.049596456687731907634612330865984453870023222347829274668467263275', 28, 1);
    t('25073153340963326636064848277131893946387401486343593547187532458672029823637809845355915290399821719875754346705591815154754048616334755761235034984288328034354293510646523531921937.2143757310665', '25073153340963326636064848277131893946387401486343593547187532458672029823637809845355915290399821719875754346705591815154754048616334755761235034984288328034354293510646523531921937.21437573106653978344980550387355403633', 195, 4);
    t('69671970466708159457883878571873106748916866206977492123672058362717630356265522400000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '69671970466708159457883878571873106748916866206977492123672058362717630356265522492208743736069589383465133546698177099290206239359281120193419131470044736457519183457493.3787233196282390678787373063540609988874588793904573645207852422505183601658037863367140264', 81, 3);
    t('3519193267652209009372099523070284444811653220405920265366564428300000000', '3519193267652209009372099523070284444811653220405920265366564428225296641.80329830893002867688959521544817938', 65, 0);
    t('485575894300355336282509173349700000000000000000000000000000000000000000000000000000000', '485575894300355336282509173349639998913806814123403142782758994141439130264854073899510.54244745621004266', 31, 2);
    t('2091007274244937176305542869549777121237427408221031651118790699040552855073722959890360.01166493511869764243331337379494449040124784386089176888557988218668988378354416556', '2091007274244937176305542869549777121237427408221031651118790699040552855073722959890360.011664935118697642433313373794944490401247843860891768885579882186689883783544165565075527498577973372232448462520319335928836535387840594174901790056910016720223312663908120', 171, 3);
    t('9882257439617665718090487396515820852983740242695668436618533738882434853015107473410.738307612694471969738790094877889677865736329332027487339', '9882257439617665718090487396515820852983740242695668436618533738882434853015107473410.738307612694471969738790094877889677865736329332027487338921074368460243435075', 142, 2);
    t('9000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '9547586432171602276083532263049228401265226603586631226612676039268761383699584349007515943134456099443621482899332790036150654361580130358395095545605778724684043826050670028775138447319315081984753373416517592.93533750750020649', 1, 1);
    t('602581758917175810314349225551545.352242740961804096909498446967794353445454068240678242857293607226369949244581085310355974996455246533773750358', '602581758917175810314349225551545.35224274096180409690949844696779435344545406824067824285729360722636994924458108531035597499645524653377375035720326844314739204870929082984966514818822061490471312075816', 144, 2);
    t('8373060126353211813285975879603399691708700352586688363401125400408467057010362399004835550891085825932123814782690000000000000000000000000000000000000000000', '8373060126353211813285975879603399691708700352586688363401125400408467057010362399004835550891085825932123814782691995019187737001788717394337247312875625112.09402671292215784720', 114, 4);
    t('491610969577293000786666.223273739061779139327803433224578432394070622891554807627158556349317756100064712962989909454484031832173638620157127761723562171428713580546133924', '491610969577293000786666.22327373906177913932780343322457843239407062289155480762715855634931775610006471296298990945448403183217363862015712776172356217142871358054613392389371887723071777357383703973663191746286414491579264679753488933', 171, 4);
    t('7277378106668536.727432705796697693360767712651329198688677513625161943976156432872451955321970733120497897005615209082618495606937842389532805025956551165054819700524206222187772210097219', '7277378106668536.72743270579669769336076771265132919868867751362516194397615643287245195532197073312049789700561520908261849560693784238953280502595655116505481970052420622218777221009721853631785116442439475214227391366924948886679070824719640030729995616336870570957862675046363288', 187, 0);
    t('5833694304917364548512089654151072740230000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '5833694304917364548512089654151072740236508419319502278833412535215139449346121521225875270234242176220863638271878829946357261580700524630777961936693261673.52551258994650942896490811399122165308672322063269', 39, 3);
    t('2951579400619328.3569877906365481546250033', '2951579400619328.35698779063654815462500327816894080582331513636578023101042373617590', 41, 6);
    t('87158112008921330691611117480499424875050767211244447412167275887150653951380994397200000000000000000000000000000000000000000000000000000', '87158112008921330691611117480499424875050767211244447412167275887150653951380994397238694752732610339274100940108221224791365526259619071.96590778245414546530982710622660502366750432688667941077699590961309875827777520556799403325671455111469715347633377422360721925304103435146650700232253011435805984612020830407955906915243349507929693', 84, 6);
    t('87116.7996010302285126411858429091923640410212', '87116.799601030228512641185842909192364041021292482972983512967946284035215397603355427512470097791870204021654955581074267936302633526868891382056530802', 45, 1);
    t('64190585667179274490367473792583992156528773061674634019958967.3', '64190585667179274490367473792583992156528773061674634019958967.2804632680461565475774651746330101022324833541008195607362545760753397442539505469912902301918102644974601119918790702832474874922169977432', 63, 5);
    t('59349317775144560276933221953908264029.536', '59349317775144560276933221953908264029.53501225764', 41, 2);
    t('738165820795020064335288426300214594553930646353111021447821167692677353900000000000000000000000000000000000000000000000000000000000000000000000000000000', '738165820795020064335288426300214594553930646353111021447821167692677353922178257810119288713689039039718379946053292211218081391415135070724056953469134.88671746560908810089954420271444708789609993942972251225139659383180729445169559875896752197734503', 73, 6);
    t('78478252699397581028069170954969215908260849542851637966068872221006282907222024814476290621080144861248584493992537464052697523184957105107131329018552487025839817704895269299000000', '78478252699397581028069170954969215908260849542851637966068872221006282907222024814476290621080144861248584493992537464052697523184957105107131329018552487025839817704895269298857311.0', 176, 5);
    t('36612465705292062347666291467802724126474866530840138627826340842036932607349390641806990000000000', '36612465705292062347666291467802724126474866530840138627826340842036932607349390641806986287436000.865385528037726406110580652060744369269641988836', 88, 0);
    t('30759717903676017261266114826531494115235872369586155454494227075013126.4597923', '30759717903676017261266114826531494115235872369586155454494227075013126.459792299449374509585385550157558822319011587548679520508972773782184172', 78, 2);
    t('95858173096692050000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '95858173096692044109486931371349375812247360339145729169082448243023018212853693605481443233589410464119069109071789549429.0921197892746103288250822', 16, 0);
    t('598530697076900000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '598530697076902515566742442940096570303977343194301352607840631988967259168685361488169779744098844940385301845398705.8157228985561137222616023618414252792634358748879817426579141092700373660541073897672415353830696176078187364092690187847636784380294823899785027038643632820603871229653539836732200477393510591828015154', 13, 4);
    t('383948489876359890000000', '383948489876359881063435.96622516671713964', 17, 2);
    t('6067385709701967162074808394972239923679298310537084770196452840361419271895069386485846790503585207893605640049910000679103059424314271895856839153942491076181461075414971254098176660145431231926052620000', '6067385709701967162074808394972239923679298310537084770196452840361419271895069386485846790503585207893605640049910000679103059424314271895856839153942491076181461075414971254098176660145431231926052616022.824698137707836633235017604026370450', 201, 4);
    t('737676859205990690911899971375766058386642873000450465800053756647491761976206714932222539043558.3456928229069856306741926337631074595364064266815642', '737676859205990690911899971375766058386642873000450465800053756647491761976206714932222539043558.3456928229069856306741926337631074595364064266815642604101681199738483068991826097424668743', 148, 3);
    t('81668415671452445018595295843440348566949155280057853726276752916258351288718320270411159300792.505171572854407987527474613', '81668415671452445018595295843440348566949155280057853726276752916258351288718320270411159300792.50517157285440798752747461285887540233609269434711221798420094915714103119506', 122, 5);
    t('52763645.5252431228842582759548763254438135892645240487763463256657298923316465537343807800309', '52763645.525243122884258275954876325443813589264524048776346325665729892331646553734380780030916260389840895675540801799656378720953200075039264298902086491682795708419754843995727339229231774402817484', 93, 1);
    t('5226964486288473368842087240551733215840260920325336686255535523134094299631297231503270770011435781396702005648969647185395761035413636200000000000000000000000000000000000000', '5226964486288473368842087240551733215840260920325336686255535523134094299631297231503270770011435781396702005648969647185395761035413636209120926811602730057966977228947299191.558299722830565945788002517234742838836902895775515505063275351707452307698350311354227639823598469839465467', 138, 3);
    t('1926.56442251313104', '1926.5644225131310438775162685911104440885469548549478642851965455806495', 18, 4);
    t('61650609594555.68048457', '61650609594555.6804845700', 25, 6);
    t('2712249075095210523507104253748037612356056804839933586072281768038177482497497777489166421520646312939421963710816520598439199581756.76110672642471066087513903264151861613487', '2712249075095210523507104253748037612356056804839933586072281768038177482497497777489166421520646312939421963710816520598439199581756.7611067264247106608751390326415186161348784095011324953719831260442464603960603149402353500862594826067274160', 174, 3);
    t('201387621797238489271536636449591863824143261927586610918916338245560895352106410312723048354927720.2441954725388399037906913149158644535001767100299965583458145078873067869974811504267668365216939300290059174311564504099989975', '201387621797238489271536636449591863824143261927586610918916338245560895352106410312723048354927720.24419547253883990379069131491586445350017671002999655834581450788730678699748115042676683652169393002900591743115645040999899747404406321824', 226, 6);
    t('6242170103233678225934149229327784056.38621659249', '6242170103233678225934149229327784056.38621659249', 48, 0);
    t('520081163800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '520081163785404729455458313151783050574607334679576362465343966825411083197444381156732642923121755728581776949217395154736800184920172936928410353279574034411028.706090931858778413421067332699054837527024564195637425221345228459783527633357066087474', 10, 2);
    t('97352650903785332472620947189286099800069061080117369303148629700000000', '97352650903785332472620947189286099800069061080117369303148629652078170.5941141658108528613233262772745630074517426319964937101253089032569170184361246342120609092488422065377285653875302981841436569063668503315072932240214139920848141100024808763629051393736', 63, 5);
    t('6161266151229583081794525207501940441844199680748857045600389002674332437321000000000000000000000000000000000000000000000000000000000', '6161266151229583081794525207501940441844199680748857045600389002674332437321125739326261992179521201169715900635047143097725295793614.785767954619468097605052154927106300384236352017429566910261014380373329170596663066243786656048883944675421572566044625416324', 76, 4);
    t('91117080636060602077275988324619930395037405488167569602020556294282130119662142397429339392193823979538949178773420520986255888081037446732070347917730974525582.8314511885500666229281518657887204421424582683', '91117080636060602077275988324619930395037405488167569602020556294282130119662142397429339392193823979538949178773420520986255888081037446732070347917730974525582.8314511885500666229281518657887204421424582683', 208, 6);
    t('55633768760182115213040960398999904119409123063216139412056955099708858338241168337310070056125685165000', '55633768760182115213040960398999904119409123063216139412056955099708858338241168337310070056125685165335.966393067488670499975831571907520844796061748633131074055483388147525288413810864020968481218796449893871414603807845227086206408742726108', 101, 1);
    t('150609600000000000000', '150609621875974422688.9920188009580700176046359815863259993902550352282953130875635137444037442541493584217645800655640090615678128354341947508993814569017', 7, 5);
    t('1629.53', '1629.529311948403357689140326389249909603625046118258800756550888649933', 6, 6);
    t('83652898112890560658144837966392000000000000', '83652898112890560658144837966391182752002454.7226406013073844082290741338830129778387384609920096224566653714290444704180469363622808402502568018346170143868068732987971542387876693174673165748', 32, 0);
    t('79874146455586433281019855135607828069818673.8998663256400498755344773531297191562267541922446489661656447573855', '79874146455586433281019855135607828069818673.899866325640049875534477353129719156226754192244648966165644757385411967912636942985573376148042270319105215914142470151971821991946442105490408357018814705763832428275904644328845923681375292470130563167882319040065927216849628218946493996014821786691636684526182690423170735', 111, 2);
    t('858173909391000', '858173909390852.61015655697500980003556144784650479534077463135230', 12, 2);
    t('344595105234635543877577901405379895160', '344595105234635543877577901405379895166.0021', 38, 3);
    t('311012753257958919148206088169933415693238912077631813981024307993208887680444164444349.87883224451259592895970141498081067258273569845007461355238871126660717636279723846', '311012753257958919148206088169933415693238912077631813981024307993208887680444164444349.87883224451259592895970141498081067258273569845007461355238871126660717636279723846085', 170, 3);
    t('98371822604113956817680694769305253104150000000000000000000000', '98371822604113956817680694769305253104146065168989378491088817.81150700128447849678806474280005297374997605939234695889942196759442803763541085422553185534605660662429259667880371075485552', 40, 6);
    t('56425588910945245663310925333091239640738600081211687106617611602271517881417035084273848973692058155563013211851355171343536175763917664290000000000000000000000000', '56425588910945245663310925333091239640738600081211687106617611602271517881417035084273848973692058155563013211851355171343536175763917664294198180357856836523068189.7501935743986189939641843014324500142778600447860797313', 139, 3);
    t('518503775268077141.37550225997773390224312361936678542896410723352322173588865107339847119294719580294530617511720926162351743689809883421691330529365186276566357094962378992485888', '518503775268077141.375502259977733902243123619366785428964107233523221735888651073398471192947195802945306175117209261623517436898098834216913305293651862765663570949623789924858874019917730071746689073155308757977', 179, 2);
    t('69617460000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '69617463207537184461468517494478360647946894598555509583117701261816209139599830436781516902035945418374060279642480516259728012803603065264111478959558756.9', 7, 6);
    t('980635915026549175777515879343601105317606228547447186407006895422105069002478483975858819597672722341119490137087.6946424155428291365858650968', '980635915026549175777515879343601105317606228547447186407006895422105069002478483975858819597672722341119490137087.6946424155428291365858650967543033526659427192403601016416257915457092748832687850851077976301670211899464557842019210905055767642967233735565248921898115116261471053314', 142, 4);
    t('43466443547797851673040957547071628000000000000000000000000000000000000000', '43466443547797851673040957547071627748203828537768138250428771217863476200.1032510776540879819357583915825779773965884116418564605711832172141481734786029116989600531285288815099531654', 35, 0);
    t('823840286.0280168196946744', '823840286.0280168196946743247651650196920235974152729364690763591471221320825743231248568135517154023695918891224379465678772', 25, 0);
    t('97510477161417058989642000000000000000000000000000000000000000000000000000000000000000', '97510477161417058989642225549610842440264616926387729534119588319021269884299559864161.35832652047826620519814107571707014969224791155938759214149394433045857852189840150878064895965686990233750985502384517364444765277001903789126541292074011806781517830693370131796460817522849886333095122208837743202368764087962383614', 23, 5);
    t('52458709478258048697820222187144814134178862110399000000000000000000000000000000000000000000000000000000000000000000', '52458709478258048697820222187144814134178862110398215373360405534589929104078247747160099275050386303975820240067966.0', 50, 0);
    t('7655894187193872710701876330275845599688230.2624713383456200093913990735907698482063336960696331892691747', '7655894187193872710701876330275845599688230.26247133834562000939139907359076984820633369606963318926917461988346335946368804197408991275397519901124100204033330329357610586678764302050810171462418243770960998512026503152057674838889202766342114060062161623689837710058453109722', 104, 2);
    t('5061708806007066892483541738209288838006103.1709528419', '5061708806007066892483541738209288838006103.1709528418404840626822981092506959941830330111413474536166097564530029870190675088723935496297441790764496087494195169597469429577131105769519439242426459030776639485611117598407954905731110052304287704692630690779070230417918475172834', 53, 2);
    t('80172578532343921180243307574602010305793020135683404.414425511171329137851216351745670912682903310650288836484769085553', '80172578532343921180243307574602010305793020135683404.414425511171329137851216351745670912682903310650288836484769085553795970', 119, 3);
    t('9526202922885821760663505732238949140.73840402463886813994924024429240463338951127898018896816407676396458797518217047932284136021152834264211696602877678156962675587910173604', '9526202922885821760663505732238949140.73840402463886813994924024429240463338951127898018896816407676396458797518217047932284136021152834264211696602877678156962675587910173604', 174, 5);
    t('6447506274470623062968929579945668576828671476.34', '6447506274470623062968929579945668576828671476.34835624837497875798997279560938022367182291608973365736904399703068827900493977411377410685403931431055889699836905763965276297136903774073992955308623', 48, 3);
    t('41549216502203256811212461410400393044663414699567844999079610528316856021749272250808204227638154901408305000416241303247990731352947751129486281767830521850617.6789621493584149712', '41549216502203256811212461410400393044663414699567844999079610528316856021749272250808204227638154901408305000416241303247990731352947751129486281767830521850617.678962149358414971247767959725017', 180, 4);
    t('6902729762214451617003171572824667344846609156497695957879838314030599180133336185336.5640060378922051640840361765992937679654751243536392995457', '6902729762214451617003171572824667344846609156497695957879838314030599180133336185336.5640060378922051640840361765992937679654751243536392995456648557867820383089703062561643910559555652218208527864231787445166850444357194136562004173623907934689055316241615300', 143, 2);
    t('879155825292821060505930599785612572900559913461687476440380422512054039987204523766424162188636981029600660906746821428182625471690809234401709551.2978913105', '879155825292821060505930599785612572900559913461687476440380422512054039987204523766424162188636981029600660906746821428182625471690809234401709551.29789131054167132799279291672028317447360192943617360612776895304799452983525043952738208891005851014682486793716804576879119269', 157, 4);
    t('3497450695194025946712668114015725921346419696946931088956960859243833209295889504032381817408166425800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '3497450695194025946712668114015725921346419696946931088956960859243833209295889504032381817408166425823986801914161214930853266677857963254814548299270738686739552738220208409347397194944696.305704395158', 101, 5);
    t('796057504829377485004301220980832704659749583448872888778524235278947201555614422807475014609593200810431850330441443937250015686719093226378276630625918828108201448000', '796057504829377485004301220980832704659749583448872888778524235278947201555614422807475014609593200810431850330441443937250015686719093226378276630625918828108201447713.53507554705616606370162756806586709007937791043011135056297522139461', 165, 5);
    t('4121916884664976627406989458.0199634983735073470229171800481849', '4121916884664976627406989458.019963498373507347022917180048184965942072459387', 62, 3);
    t('196.657845533606032092052281044925792947802490464120272129465341620451885521780149', '196.65784553360603209205228104492579294780249046412027212946534162045188552178014919664764762702036863070533341642566365921016203680354061576970413278688855516713636200464515354632841', 81, 4);
    t('41367649899.15525125446571277017165613628044374695699984495136745445711', '41367649899.155251254465712770171656136280443746956999844951367454457105844', 70, 4);
    t('8873.187149341751810548484187178356375073852175479369008168932223582932952767709415688746970482355958447822926582848227758936880291578899996468709410782434803162034436902501171282982544152', '8873.1871493417518105484841871783563750738521754793690081689322235829329527677094156887469704823559584478229265828482277589368802915788999964687094107824348031620344369025011712829825441527074658076737294957144738594233959505', 187, 1);
    t('4943537068937.311582577928871271489174738811104060572744189882533606411328967424336657380676904086383027103', '4943537068937.3115825779288712714891747388111040605727441898825336064113289674243366573806769040863830271033400836415480749828144147537475695402309459771102019684914021224077741737422216401727657293432161084842892852006712', 106, 3);
    t('9972476183341059725893805649.179069520587653230777436267690516911', '9972476183341059725893805649.179069520587653230777436267690516910760593410386930883331739878866395409465917650355717686986132520806390735388910923962969466', 64, 5);
    t('5483584394722419317048902623428547254905701012937964452741907288631256875987194920950828396139063507499275346520337933416400000000000000000000000000000000000000000000000000000000000000', '5483584394722419317048902623428547254905701012937964452741907288631256875987194920950828396139063507499275346520337933416312319056442054531157527104679761776711552087897018460859017625.427709', 122, 2);
    t('814675062016.9793608684051517011982021706861871213088378685132442487138849988236138494911', '814675062016.97936086840515170119820217068618712130883786851324424871388499882361384949116533262473827333127936595628436903745933608374145382214927951796413869489635489847533386864650011442672843426415139775193158635280977849336312568525948884318457048399283739259156983890111547750741538325826765905372379737524212420669258866565234636467737115206680988545165239720', 88, 3);
    t('4189393884566955798939864147160491327313000000000000000000000000000000000000000000000000000000000', '4189393884566955798939864147160491327313875429364356613884470500465855742715871230933045771826099.05976269257139787788485589405912362961993489595812861841679189880596438620636471596375855198413708680754109397369593022328362879932945767034272927693520846661114197572837045886708273947675341195445510747093', 40, 3);
    t('55050484198814400461034896015.043578037', '55050484198814400461034896015.04357803651195', 38, 4);
    t('5039686353460628680000000000000', '5039686353460628677076650575075.297707244779342830505662334660032727289', 18, 2);
    t('77.81469653679202991667853327570329115723480246705200680766180372240147090375406065177105769', '77.8146965367920299166785332757032911572348024670520068076618037224014709037540606517710576871892176664270776894883397542906705004567352286666456320339', 91, 6);
    t('158069657768914969551344761970758524502443310226627806531322836423917253720215713444294479763649777655104365748204491446328979835311304267487951.229562622060576800574155366742953970773177202830428846859824889', '158069657768914969551344761970758524502443310226627806531322836423917253720215713444294479763649777655104365748204491446328979835311304267487951.22956262206057680057415536674295397077317720283042884685982488923610546428820302444498277557069395', 207, 5);
    t('51552124484589191751254534040567204143368791502819034743952119397439252412265192986791945131860749097886555431919539050070736000000000000000000000000000000000000000000', '51552124484589191751254534040567204143368791502819034743952119397439252412265192986791945131860749097886555431919539050070735000484778878716736200805874328976120765530.320622827895803206216883884302366242656342502366528638358995645611295', 125, 0);
    t('716060000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '716058712577379126816583831630505701862980065578091689731850979183625654757531915075809073541692205430176203825322803667318714909136.4587218197732516524510054536839911991940737289231960220784869154', 5, 6);
    t('47830887510836652771335592581224562308640144294155.8073449197', '47830887510836652771335592581224562308640144294155.8073449196392006752206208330604929225974272480293801289440309897979712156716076536448199408672271328049914315200721087273933491428950233743278763187575128636068189433800306947294468309497556414642951155815522587121205592315493939220060146864111088166819486128241436413855607092174821600424869283990063087107934566755', 60, 2);
    t('812159519781778883468277315173202547335578492956911598341029093888402388291603514.1919545176183994434681888079726016944122731087180235183382587688107404180417216736408320839847520863866692320499357', '812159519781778883468277315173202547335578492956911598341029093888402388291603514.1919545176183994434681888079726016944122731087180235183382587688107404180417216736408320839847520863866692320499356741565809093781545311084486813207043253227326301298721619', 196, 4);
    t('1517965676263777089424223727694311820331041415634411320000000000000000000000000000000000000000', '1517965676263777089424223727694311820331041415634411324404495143801105774891857671656817263071.562687503695896664031', 54, 1);
    t('3467839655493012100000', '3467839655493012139082.7820', 17, 1);
    t('660260794246404088983209070801106715967772741529780721868723648020100000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '660260794246404088983209070801106715967772741529780721868723648020016322794285096047693324758977549028319295811600573332667521627974664566534906513222396.1', 67, 0);
    t('24571316714137644479570176937517049023601300000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '24571316714137644479570176937517049023601304105446607324131972458381104838679090874328712814327588810324149363854721290910887330093719870734360546246589589018419088310206474242206025646.7626274947871818800775181008740203318500499157293448119', 43, 3);
    t('23770142585.5948541194826701585433499438546419045514791878927654711059371230723320054753506346759346488683601492708254752762959462577567363147093047478856895498623770460966123851325613', '23770142585.59485411948267015854334994385464190455147918789276547110593712307233200547535063467593464886836014927082547527629594625775673631470930474788568954986237704609661238513256134523266', 183, 6);
    t('1608983914499358103582000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '1608983914499358103582206735696166675482307060059601568745225584710996665591503143678371984480927533321408267447327689193752.1715531075115874556504', 22, 3);
    t('367358348720124893680796.3874183734848016340928396360478321200750217369703691192176827515356151416340861297195625500162769669607545476068851015393847554647646141196291925', '367358348720124893680796.387418373484801634092839636047832120075021736970369119217682751535615141634086129719562550016276966960754547606885101539384755464764614119629192519310312', 169, 6);
    t('59250953118830402064779394282860000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '59250953118830402064779394282850996515938931844774235129889902848869423112947406410242976923946106300370272490705278726079983609394.71066411451402501834940734769073764534330285470967521609584321746848282717841430082326235815556709253734289317577455965451876072538050390047117133989091', 31, 2);
    t('484925100183240950269793652051457.3161796091624759159530038054362904510704940398786427899816485', '484925100183240950269793652051457.3161796091624759159530038054362904510704940398786427899816484865402742948868500313686692430525124831128457566458859425', 94, 5);
    t('52471855091560953692215682701005497442217061404752217686211229283.69163867180242339326541890823067152663773816', '52471855091560953692215682701005497442217061404752217686211229283.6916386718024233932654189082306715266377381546808637113512942044319721107829267019889313221142668122427701562433597005304893614114193481299897849671908796', 109, 2);
    t('908419753438744070141076370000000', '908419753438744070141076378169410.74333983261427949269621988795693110973116206526223036921309', 26, 3);
    t('81935411189176625061439946915996312928897103014803980406503499542345931755841537016634694650820157145479682242251417452663910293362820000000000000000000000000000000000000000000000000000000000', '81935411189176625061439946915996312928897103014803980406503499542345931755841537016634694650820157145479682242251417452663910293362810504331768203685250177813088729862071085505179418731145604.6572457807790648084690676177719870120171058374742887751201259510850048983273767318479088224824', 133, 2);
    t('902360986633409288324390515617260600317559542404560734080653636072635297440996260755954338285757918685170123942546217767669241026300000000000000000000000000000000000000000000000000000', '902360986633409288324390515617260600317559542404560734080653636072635297440996260755954338285757918685170123942546217767669241026289479947747990910007402371806149102108152125848428225.491122710545420556394084833981120643729', 130, 0);
    t('9829860316000000000000000000000000000000000000000000000000000000000000000', '9829860315704528895061964818551982065567550027539631611458430220868699020.19776654684753006431594395706556722301519836500139740740432125776426010879501035349266725108199243882667218591449688785435865359067536370841134884556237192208489222907986605872587967110372138327942009241764333348082401409075782660425650236565363387007714816757164914899659620651927877085609750158423456296160', 10, 6);
    t('8376178219.9619576494', '8376178219.961957649316993599711948161082321933466311763013037969624431564189833911023674708801916301684637183421778308667563065083567014574955149855803350224640387138901475048996027952118670371567576588391782774540541540277813835169342731383781016398548595694399871104142025550947873661971381075369039269792558916722869371771106972459950', 20, 0);
    t('2892264500819648740534428255237033781341433507478808757356669729580207884.788639542327604952192003744620113365243717169003245809227528483364256123944918854865482', '2892264500819648740534428255237033781341433507478808757356669729580207884.7886395423276049521920037446201133652437171690032458092275284833642561239449188548654829244', 160, 1);
    t('78102384321305436835613600080162832418680850876267924215786200000', '78102384321305436835613600080162832418680850876267924215786202680.6501855812201292176242913725178193602675060488544303664859478203088358462095131454871097004011721215642251223929073137306305433625138056524487827518652283052748647353025286671373', 61, 3);
    t('35221425263628325017136954529782729565025008135623909512704000', '35221425263628325017136954529782729565025008135623909512704364.5638856473145', 59, 5);
    t('3626068871242134029428118842606665000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '3626068871242134029428118842606665191556407273100588379950583462172331483246762158615103384106869948534475457362922460.95925919211389873178891675988413358965', 34, 5);
    t('12998196.011320853291310949593652730446846639239555854242241269830864326314846', '12998196.01132085329131094959365273044684663923955585424224126983086432631484625756354', 77, 6);
    t('1298676591260045491253069297603047415969848553', '1298676591260045491253069297603047415969848552.7612810650092670501160701146091623831136151813093', 46, 4);
    t('3233825.388962530951338997432718042973214347716231076409077222563803489575358319895901095336702436682631908833332863852939003595404859128011747913588954962329259423275829404697836247150056874', '3233825.3889625309513389974327180429732143477162310764090772225638034895753583198959010953367024366826319088333328638529390035954048591280117479135889549623292594232758294046978362471500568740', 193, 4);
    t('17779740790419142377362050729585603836302569323817052308522702895992672634508613113715229591205919717124508809.5724', '17779740790419142377362050729585603836302569323817052308522702895992672634508613113715229591205919717124508809.5723423238783779216796484362176547550609047142258558646973144095758416645493599917465359460374', 114, 0);
    t('2940000000000', '2939865676752.772546593989', 3, 6);
    t('58708293820853304793054085540673185419509742117307168596487517068666737004500343765849080915916520316341586255737000000000000000000000000000000000000000000', '58708293820853304793054085540673185419509742117307168596487517068666737004500343765849080915916520316341586255736546214520423614710156334470698924505969888.46887935348628101649068458929', 113, 5);
    t('50407293429151664192393297347463936855819836763938269280305276990364107310789929211550.903', '50407293429151664192393297347463936855819836763938269280305276990364107310789929211550.90302688247410708341623987834978791182817811592060713654602', 90, 5);
    t('4491028782447168059560762390631646926622826415804639302079500', '4491028782447168059560762390631646926622826415804639302079509.51939885934', 59, 4);
    t('62247978944611582092744743490438394647653027820566005951610237278063658681891498010535575734917139059937962.72857274799273767', '62247978944611582092744743490438394647653027820566005951610237278063658681891498010535575734917139059937962.7285727479927376719198642307530078159796624439251396061840263795679688178957721092647910608914694144678729240681742931317905078356192750830564145773493914957084033116202498244', 124, 4);
    t('766149691690933055860356709629756178689877846622145225745991121825553483760111016259274006001426102.7420097779964405110486205225948618203934866937920502509044019854644382', '766149691690933055860356709629756178689877846622145225745991121825553483760111016259274006001426102.742009777996440511048620522594861820393486693792050250904401985464438162896596578452877076707277904615679829929610341759918888207', 169, 4);
    t('7562803639800597676894955320.28155896858619228080806961007045573540946236278847926913655754874707763619884196949114534505566171777795695077121476684069280798135916624863468102332295451807', '7562803639800597676894955320.28155896858619228080806961007045573540946236278847926913655754874707763619884196949114534505566171777795695077121476684069280798135916624863468102332295451806980717131553319', 186, 0);
    t('8257232538923691763628143623505648274.7389983100870556148001976607688464279021958757857889306432157097235456965', '8257232538923691763628143623505648274.73899831008705561480019766076884642790219587578578893064321570972354569647027188991406552895546939830093598128001303747756190281428586830200628145302015220314375588646622111490907654392271320126830007293890134926417914424068984396659129091833999190051559867264626348388205041187044869160252829641189914887', 110, 5);
    t('73632239200000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '73632239155251475847057532541406463097842364134225037019967885647508855069702841642049092042217228038176742159712211.103147881119826706046090681123406626218885727322915828742238479484184009646237974570714999573197141429191550377593980640492978961339', 9, 5);
    t('72298241143571247.531911652315922888494564', '72298241143571247.531911652315922888494563753260303370006649', 41, 4);
    t('844567996955099295218259326510765642283201485376855634807275585206171456543576573647532732721640015339600869563285963489531408189835938499.427492999174', '844567996955099295218259326510765642283201485376855634807275585206171456543576573647532732721640015339600869563285963489531408189835938499.42749299917367828279878972960377633372010674092289620208301531189258079714821974860', 150, 2);
    t('566810545271336887657824059216824144189979171017454954605381554300438519381.819614138157893220721093454708608774701204135553447584803', '566810545271336887657824059216824144189979171017454954605381554300438519381.8196141381578932207210934547086087747012041355534475848026342676889657053018288182652194407123761904325294391763528030392980734802307584525648580397143711991580656952671504645623066553118127886912647527095252917441739335181055353618603816932591766852358132978080189', 132, 2);
    t('6902359143674147619503709790249585476485605736874890395714016065106164260366648069034933597763127992168367715566605242012739656942387998063875250404964601862583211459.9933', '6902359143674147619503709790249585476485605736874890395714016065106164260366648069034933597763127992168367715566605242012739656942387998063875250404964601862583211459.993247156022554211486573776849896131123973127538451066297463', 170, 2);
    t('87708633493478986628986617692642755809772027224772550.6414105616605567057562429171497691910816548177397576283828434189385906', '87708633493478986628986617692642755809772027224772550.6414105616605567057562429171497691910816548177397576283828434189385906246911457649998052960510661877525766747662744468196801552139253066772848513125564591852104068183409738152939474681971181364', 123, 4);
    t('21000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '21322413035720604931308478773224822895177917818071019684423359162992566025746626028093140806335525865561449617881179679300398352325828899240116853545991968325931181310552522541144308057578635071181071.895342839776617732232282119194959925171161286647721698037410171260', 2, 3);
    t('92853714973495712425779712154866761730112427604658108204171078709238536385641597345856662405016056305607009786562216423567661040540284362801765915330510855640513000000000000000000000000000000000000000000000000000', '92853714973495712425779712154866761730112427604658108204171078709238536385641597345856662405016056305607009786562216423567661040540284362801765915330510855640512868355705321982392692190440283169004588876431998833.608', 161, 4);
    t('3624609466386894479180163750561619478538100000000000000', '3624609466386894479180163750561619478538073646953013643.51070999442312317480432360358077045189185821164302916446759170978389611872582368005388042021365815553272687488069145043019014082112137763701676237376775857914699215603131', 41, 2);
    t('9826459047927854789866555401653891045540967675624028176182523147298625069274766567799512039695650040289935920409.794064290534', '9826459047927854789866555401653891045540967675624028176182523147298625069274766567799512039695650040289935920409.7940642905344312776152538171399369684871601056713469907062464361912232134764024123466449068582639014264963898502902568297093333915400424633143579113853131202520232078877374161164991680429554465005107279133335929743073121049528', 124, 4);
    t('592071841877297866031476511700307234020412990210924977837259026577167811534243844319759112231159393612532912778799274030556099330000000000000000000000000000000000000000000000000000000', '592071841877297866031476511700307234020412990210924977837259026577167811534243844319759112231159393612532912778799274030556099330305826221266904602312389789014777662391949498648151627.19066189118052325951496789533946625529', 128, 6);
    t('12485333768062056752189203606699228088380477094276982010743500548786168279140365372878278531621564758873851950509639449731661702371000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '12485333768062056752189203606699228088380477094276982010743500548786168279140365372878278531621564758873851950509639449731661702371826614107962895887848054113201840894632364460422555324257215448656288150830043584888.47239126', 131, 1);
    t('5910673938123813309455838661625440800000000', '5910673938123813309455838661625440798377731.57342787313950078824949', 35, 4);
    t('7629472733638704242162008770410371394505232087343341373610645088827.996', '7629472733638704242162008770410371394505232087343341373610645088827.99596848841551364349762945237611807386103516621299796831864558915971925364226004376366329967954907075332796917856807026152746169734604937022095239673087726509549602489550105221292429832971331500468298836381581067281674886810463068854723255459805028732536069461271857508503682959243186054781622218058479724966373463814983', 70, 2);
    t('723468147327381796115407508781875546500000000000000000000000000000000', '723468147327381796115407508781875546599538752436279376332785321186027.907700507576880646647030456937939336069006278148101917574', 37, 1);
    t('541214043.1829', '541214043.1829937924272070591', 13, 3);
    t('4092970800689123936240666417350365683434150551348883758526532328704.8', '4092970800689123936240666417350365683434150551348883758526532328704.7892581991451744007868368101896713139194719471071736590', 68, 2);
    t('13.675046012041803922080622764784', '13.67504601204180392208062276478442061688688558482377473962430861631991658217403543743200376393422629516', 32, 6);
    t('2743485125558445242334477808359019466032688447586634340000000000000000000000000000000000', '2743485125558445242334477808359019466032688447586634336454310287447012525296995668798531.5131', 54, 2);
    t('1772401090316424736150807969416410631166445627770320253777919374613500000000000000000000000000000000000000000000000000000000000000000', '1772401090316424736150807969416410631166445627770320253777919374613510616110700239207438776772317339798094793403493173012640923250018.981365116660710', 68, 6);
    t('419295417565065521416028847540000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '419295417565065521416028847545609097340552348744912445947703865887444968036064680981216543532468109428686015577160133136502409044026145313534209633456029177971692392840312265887207159897987470458294190345231799.7478547172908907289896221787236496679172809928483252', 29, 3);
    t('7445669041295733193706269203990988284804646887754165876733609579595087393917918702000000', '7445669041295733193706269203990988284804646887754165876733609579595087393917918701693957.528296514557659248141475086247999114949943608257911431887171530611225060327219795726441711805915098892706089577947797477325896143576333483218416285314501494362985081304613296333188686073080', 82, 6);
    t('535529719819272621815704062199117389844.5', '535529719819272621815704062199117389844.470288376775718661053503572339205133', 40, 6);
    t('6044410000000000000000000000000000000000000000000', '6044410840073060304479832509284792065420488640517.33679692311430654188436925369656618854256482929475108562518683981692537363865749613684903044406413307679121901959331772063595185156402968805758141586380872573691621989944406031581769253215293650004099353602644105', 6, 3);
    t('29712318090999787701366409119644795394446271979572359682789435866.114573681756088226773778423879318139839', '29712318090999787701366409119644795394446271979572359682789435866.1145736817560882267737784238793181398399211971300393795448068596208', 104, 3);
    t('6244396027056550000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '6244396027056555952140839417353206621814791434147330249374605097615704748955685197338801860198168819207067826779433188219513235298134728941676420957041558333646794.95865011886403445037257504001284551193260453098562410745042483685855182309459790428610854366424278381079226170949249330522461939634', 15, 3);
    t('7879994549467110203327021331587090937717239787538820000000000000000000000000000000000000', '7879994549467110203327021331587090937717239787538820466477414916990642872110354554310839.9905048787687124908598789929305148', 51, 1);
    t('82398433144803551078382412231002139952693491337330051897067553175795381549796974358861101255358594648726296673073628513282522076688631499502190928313562208772293761139561984466764450241.865323008276277645358', '82398433144803551078382412231002139952693491337330051897067553175795381549796974358861101255358594648726296673073628513282522076688631499502190928313562208772293761139561984466764450241.865323008276277645357581709551653285206', 206, 2);
    t('4116779742265948483180153626231768079629222000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '4116779742265948483180153626231768079629222481544222163715387510573918723724276334881193767326399334315313323809217160650758845225.675439698874646861910449048261961454923818261694118470065292733669884636751539600619408762641486015132545515614994917899397207234883755341526', 43, 5);
    t('34.6766246099903260938603524676592514152918941770083271707462142234234469079011715069205244965449736315505569132523349023090523', '34.676624609990326093860352467659251415291894177008327170746214223423446907901171506920524496544973631550556913252334902309052259646182188503072454841937529760921977647351338176504575477737318948060358642963674697572493868062304004984537958901465473026457813416681656515349993045286672656503415280550754426751807909232219174626921463184775950', 126, 6);
    t('389708710056484965746662375753.42472810307', '389708710056484965746662375753.42472810306598229797638946828857644946534137438978921642583787213928177526710002459796326280913697132751005296709955631736869588619514203346329701156945585100142071096150970633245462732981401488277756165247072255208334124733441255054775807014017981899325476125788643996592452304979575851256245262136451364077158708012413562566062489954899251031252231889152547138892027870495', 41, 4);
    t('89430482377668062698481837000361221993576553360000000000000000000000000000000000000000000000000000000000000000000000', '89430482377668062698481837000361221993576553356341227347973363161275437672473153819548221850987777717424806792829297.70670242419994178555086426581568831387499765626757358126014423995257512863530995362902599431089722159232489158556456295254767921362200634310099802356039959642915449708014193012112375282', 46, 4);
    t('58552654789301344727755745691415949597503361662811282843811313485440147036264639395166984871671009100000000000000000000000000000000000000000000', '58552654789301344727755745691415949597503361662811282843811313485440147036264639395166984871671009050173909110086033539808240622794699707912310.459061037612178981820533219254669619907363669759128066259039055943573217495180350673545357400951177606412293610917517901632707348476504600164611525510264148339262102586435963150522', 99, 0);
    t('74383172933959844389520336820627290779757.786544614371129777491', '74383172933959844389520336820627290779757.78654461437112977749124618308326705811882343', 62, 5);
    t('48413155308104859290895778413594097980592213370758046995522702098147586932882075058200035798917530891732063910455197448099288637952.624170380763323811796597306615430875926449453446220266424468', '48413155308104859290895778413594097980592213370758046995522702098147586932882075058200035798917530891732063910455197448099288637952.624170380763323811796597306615430875926449453446220266424468309820559784949108299168699385367981076469', 191, 4);
    t('9920425902000763630241864901951162774389144189594814880710840586509781211400816890117623129137410036697278104778753319161938592080670100831995866396.277100866430642957242058322278709056047705139556486', '9920425902000763630241864901951162774389144189594814880710840586509781211400816890117623129137410036697278104778753319161938592080670100831995866396.2771008664306429572420583222787090560477051395564856606447929715018306995203286168134330', 199, 5);
    t('527891544211600439736024024376712514613461296559506199858435579272876423096595414862258945475953454455747500770000000000000000000', '527891544211600439736024024376712514613461296559506199858435579272876423096595414862258945475953454455747500769761769675828316947.895788921246993727267829124212915951986598941206330075010886307833389167135335793534529842789636679092264704939966713031211391253686280328484052233697369190995629859651538350520363234852859534755', 111, 0);
    t('42654219500907081176158192.504232261123755983754445906', '42654219500907081176158192.504232261123755983754445905657175056563', 53, 0);
    t('714637106140000000000000000', '714637106137728142116427470.9158782928128099614517107391518482756701911860945449771693433846141102339942262606764539513538012751110744213550719577682729575188965539987621392827543597380244651264686206547464311511239124786886687449704915318527950224328039009354018367650050725790816588414551128027719828', 11, 2);
    t('5080199165060761968076270234579.125673237578799', '5080199165060761968076270234579.12567323757879926653790324631558986239479378267201678739847772378802569541564575941994285578470034955109261090085804936044302432120952398656140083808717118495692699305653998689753872810703845405394091438162534167', 46, 3);
    t('5513946403291896765323178319643914853834233978680450049988280903340676707728497659711780000000000000000000000000000000', '5513946403291896765323178319643914853834233978680450049988280903340676707728497659711778436013484318766696303404168688.70642922712348282', 87, 4);
    t('408917556239.802292939232343554674368239911368740928706606982175436035835839990191038267188468100048258248097809366906931842790392181', '408917556239.80229293923234355467436823991136874092870660698217543603583583999019103826718846810004825824809780936690693184279039218135959600414036638761852122059890632629797464371899945649188532238734670880058399703919787900395937547244832925968027', 132, 6);
    t('324134482535480253000000000000000000', '324134482535480252315375239949006878.52530682652800858906538184671808723664155947933198134959615611348171559', 18, 2);
    t('980000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '980313676458577918374124470941044209412885614582955190262520275004758173762850541992070001091419049942549418604452377458322204102844.103160631632953451039059458469183702314228116048640201298833403928835014084876917807892625477357275324823985403720218508073', 3, 4);
    t('34848381365173805819539856247192526039353736794923331980470083305578504880790000000000000000000000000000000000', '34848381365173805819539856247192526039353736794923331980470083305578504880785820930887441547767271875545162150.014374186236814072861015212840930835201691760227780246442432342545401857120319446125977876823614945621941790752648198402857647051378951616123319630265', 76, 4);
    t('77592239204334208989681432082483274117258946024601231845994927885116101788000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '77592239204334208989681432082483274117258946024601231845994927885116101787586755795878897590131610437673556915190939946850670065107304332331735213067326434531468952590246122795063765.138819520154860007617443128317101949122095684971034947832118330638716573418117961543263615313868', 74, 5);
    t('37105513675699706173228985021930305824751949917417197145069188365439385962671608508009735800503609758026649419840097511846723124318677887886028426853200972226951598292100797105705195000000000000000', '37105513675699706173228985021930305824751949917417197145069188365439385962671608508009735800503609758026649419840097511846723124318677887886028426853200972226951598292100797105705194418812080915656.754634158870019269561218', 182, 2);
    t('9718620049237051794232428002436135535649259922761297315099411923873088921009565056861486169415987753267701965994475704589108143127127762993.7279535450144632875', '9718620049237051794232428002436135535649259922761297315099411923873088921009565056861486169415987753267701965994475704589108143127127762993.72795354501446328753999124299765903093360821832373506061593960367928804103022320250597668663420714787261001609952836466267008050239764345440240703703942270784039121539625379', 158, 6);
    t('341294874835534912587346605054410091464591785262899677457930251743631038431000000000000000000000000000000000000000000000000', '341294874835534912587346605054410091464591785262899677457930251743631038430477703475909035049467817457514227193941511962306.691470351507510020119857538518323857016940877028626554030006383967502429976146442531728612187186062371983171565759517719825674470499833912', 75, 2);
    t('11272349040220201279900000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '11272349040220201279877924214076411668924359539268451335401203230227958371064919348765741892964444815627311678312131998186185007726145018523659333642350.824315385463385382437070128322595291640160232892526989127633466422147790051529304316172', 21, 0);
    t('7094774007217256160279281999661948193530378009508160607000000000000000000000000000000000000000000000000000000000000000000000000', '7094774007217256160279281999661948193530378009508160607653428176582895585008590928014674114762856814181531304175457937809699644.72974423472711586919653383166529447530467349604425247465655369739157', 55, 1);
    t('230426106681940004777795540205566926292731583187905354.3765087810257117086262036574340388300204377237435', '230426106681940004777795540205566926292731583187905354.376508781025711708626203657434038830020437723743417407576619208734362429107591770318453405761013644531671491408745954974309440884253054895939608', 103, 2);
    t('2112456523814021.7644527032310965738695555696052453047436545661534957871583346831543694695878790205410364968516955304254099102037', '2112456523814021.76445270323109657386955556960524530474365456615349578715833468315436946958787902054103649685169553042540991020368261288510849580749261238409709668359718629487638512656096969272944634328431406233936999396332084814680360956466144420710', 128, 5);
    t('2447121274625718819422265563414512314368109348978836877685570217.4897592804498116319535083457600725291', '2447121274625718819422265563414512314368109348978836877685570217.489759280449811631953508345760072529149', 101, 6);
    t('702263071397928605222423998090390706447181128515619010369852560468489063775789849520916188108024418490174125543814186566942236297036593980541107590', '702263071397928605222423998090390706447181128515619010369852560468489063775789849520916188108024418490174125543814186566942236297036593980541107592.047950709645951865301370630883161925636939650927022975507027281863213575968484033334402162619227934514', 146, 5);
    t('749557470184.02868644465207776320742872855547610288827681799942418942776006394944021681289243846187277831545529728975449216853664085490365472836', '749557470184.02868644465207776320742872855547610288827681799942418942776006394944021681289243846187277831545529728975449216853664085490365472836780474057791330566030833601219279', 143, 1);
    t('56653420388092519269138469028923132420056382820.95877475243510904406382797866977196040713189308318949974043217673253158', '56653420388092519269138469028923132420056382820.9587747524351090440638279786697719604071318930831894997404321767325315839003262766665497616833155248632892047349519641649196300622412851265761462286326330598277915164', 118, 3);
    t('35893455552835803998253243009228043784199760817032102117243208003409056145499396989026273295356207388000000000000000000000000000000000000000000000000000', '35893455552835803998253243009228043784199760817032102117243208003409056145499396989026273295356207388764385909831641968274993220202186645117065876821667.8935832', 101, 3);
    t('6870656389537678559371824433860282742365894.690155930549958745716690490056148181215246333099786369970740925874326602967160541115', '6870656389537678559371824433860282742365894.690155930549958745716690490056148181215246333099786369970740925874326602967160541114942756039150413949937855969958415398985', 127, 0);
    t('82941608763885079890503002447800511972694028374385477775.20992069798214127069574932369991961733150995953761902263784383826723295799026293307839', '82941608763885079890503002447800511972694028374385477775.2099206979821412706957493236999196173315099595376190226378438382672329579902629330783904202258027485904299635352085418518828841929076007858737552190900752703881829668919007105224311158971917822', 142, 3);
    t('8593919000', '8593919446.464881959061679800074321239899', 7, 4);
    t('2060256322617493951000000000000', '2060256322617493951616235610102.8', 19, 3);
    t('6570984602793936137822388830656965127101615871545238271674566525078057155.95737040291292513045510750370853651982881', '6570984602793936137822388830656965127101615871545238271674566525078057155.95737040291292513045510750370853651982881050519296373483641534146701980742610740905813651131793670672231333866638004732769727936431506579176789101040', 115, 1);
    t('827859885893365365070551350585209298221625571100000000000000000000000000000000', '827859885893365365070551350585209298221625571061739954142010427116528074823367.450096935232631672520793154497713845107376773975352103570361850122184043602059649032097030883555626290250770869532765691174537781206180130', 46, 2);
    t('412380551983767111639515815976749868543440693477845969999954329277521093363093215.0182922565007376617876294178643699634142832158171735520559', '412380551983767111639515815976749868543440693477845969999954329277521093363093215.01829225650073766178762941786436996341428321581717355205585527908375031898639656402753829150288365237044351912093236', 139, 6);
    t('49437096299510419399016066045335684650389592027404627380546360299020444745277.962636820758293011988942570972047524', '49437096299510419399016066045335684650389592027404627380546360299020444745277.9626368207582930119889425709720475239542065113099355311492656', 113, 6);
    t('28635105047043563659789184764931925824306654430569099039088973769297085004837038268718759593607837610254341225139111014865905658186152030933251989190283109.2673886357936', '28635105047043563659789184764931925824306654430569099039088973769297085004837038268718759593607837610254341225139111014865905658186152030933251989190283109.26738863579358953991028024759123646109641260184583218649836065912705817483730571793552', 168, 0);
    t('908863180000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '908863184419985081280539513381136630862421449148794613283814066843782395232576251052047582728214991268542255557010876700936258288607558685.30466743328397191478492785201133575066258332273500454022688661394184901137169102981933515491210793698000625381261389062099783789023892245323112358042024012034582333667206286512497111621059', 8, 5);
    t('822536392548341670313629566844199235933706444921103153735248659797928502230505962584033863286643054087922000000000000000000000000000000000000000000000000000000000', '822536392548341670313629566844199235933706444921103153735248659797928502230505962584033863286643054087922400771725938832747718540665322178434598061766154995332310.74501273035941686892502288076007511073341479564591564014012986925940992245247519504733402654038079546097251111933218772333753231442980940', 105, 6);
    t('933604911216957901253114333994322485712386099034611718629000000000000000000000000000000000000000000000000000000000000', '933604911216957901253114333994322485712386099034611718629327566854184376934579999131408462674462686266902367680325017.7564767476810385541021267688537974', 57, 6);
    t('280741904989762959479622701090407057.5', '280741904989762959479622701090407057.50029578459765230080115383835976863725732676', 38, 5);
    t('9.7779973315626', '9.77799733156251119001152', 14, 2);
    t('3669439229465326596871655950916401388705094300482864070694384087286229900827221.85738029447283445029', '3669439229465326596871655950916401388705094300482864070694384087286229900827221.85738029447283445029', 99, 4);
    t('601525431892284607086351674722292311904042910.1636960221803801891708496563315368844046101267437065115231299614492777668929674837242035788195611187661148423367357549895', '601525431892284607086351674722292311904042910.163696022180380189170849656331536884404610126743706511523129961449277766892967483724203578819561118766114842336735754989525339362824796550961', 166, 1);
    t('3240000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '3247781135317921071803873350251302939383498145780940974379845006843292673037658072933231941570065341321687111949873466774308965105814113956803004.851969534684498627866413687881709163573827272863950847694253457704915993222674654561710435945021145325691', 3, 3);
    t('315766922591064858752529780687651.757311395094990499637280101395617284359619500273713472739724653075396562736810005002263524128826895770274449263', '315766922591064858752529780687651.757311395094990499637280101395617284359619500273713472739724653075396562736810005002263524128826895770274449262985070943636652', 144, 2);
    t('4497808392106840014614624054000000000000000000000000000000', '4497808392106840014614624054248690615879138822444728152571.48030731940462297750602012686213916569490098047769637157415674429803397008072599504832186823536463426687427747894506971748824234904694873888034594061131533148732', 28, 4);
    t('3717633275398721024838340785023465457000094475291974380216838937544567967737.53028257583418033504247', '3717633275398721024838340785023465457000094475291974380216838937544567967737.5302825758341803350424680205271374946283725502631807206796788191606458863044117240630620028129406145734913102001665902229142521299085457659101805497247480429118020479888100654925846951008284771318924335111687673421402663004236857683250659417408719065941163312', 99, 2);
    t('1417832535554691294029875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '1417832535554691294029874957660038199887089650715895249069240334473183871026130934937305594879053633263405676304709900343329709022586010685214202170930294958.34077620143', 25, 6);
    t('54184963268203813400', '54184963268203813496.21299521380896156620240536532402046119094', 18, 3);
    t('733957486439512267473267294787355232077543954453965652179794505949963080953683515735397375298416865494927838613129753253580097975659271546764560.96686564', '733957486439512267473267294787355232077543954453965652179794505949963080953683515735397375298416865494927838613129753253580097975659271546764560.966865631202146504265695686422120444795090937378698064310310691848459965275724410177663796467276485733476473680882576057', 152, 0);
    t('79230286274303255118926231971259992808934633806130388448129710589555742923490514918373874958910000000000000000000000000000', '79230286274303255118926231971259992808934633806130388448129710589555742923490514918373874958915232113104055099785080013960.78642329886146516446419747637687069325042609315271623706175467963943662148042987494427786767877784775760362288331439277235035774515997647345280186237547948780920103742111967146206487434379466577', 94, 1);
    t('9773927005491058835879036252355998872279215202373655580320971.025683266693042781654753636935146593373101021716254403053480346423006055931061323801576726093445073619495283588793595014211864337957078653366238478933984104112', '9773927005491058835879036252355998872279215202373655580320971.02568326669304278165475363693514659337310102171625440305348034642300605593106132380157672609344507361949528358879359501421186433795707865336623847893398410411168024997852588751137971907', 220, 5);
    t('525689821414056991193081392725859768338662423428646524973162999790317000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '525689821414056991193081392725859768338662423428646524973162999790316795155526322064900667523402264907684204747087066510095015348192108429798970554084922933896112631371418757523570258113230.49465866263353575258656433', 69, 6);
    t('737813563633621504582318871274496563257156894114497941657173149813544334704083684245247097.3122662699049', '737813563633621504582318871274496563257156894114497941657173149813544334704083684245247097.31226626990488038902547553707706365789583199700781364428232391825869865', 103, 2);
    t('185821082383432758941600000000', '185821082383432758941643327604.77267690', 22, 3);
    t('63857826397024632282169082414889532439615954291104773243315542683530469109788962128590935473183134049110000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '63857826397024632282169082414889532439615954291104773243315542683530469109788962128590935473183134049114280281436060887899573892396085657163557274567721611261118394960261679870504863969406909.8406499', 103, 6);
    t('10044701744956714324433113471390819235101461274629430645000000000000000000000000000000000000000000000000000000000000000000000000000000000', '10044701744956714324433113471390819235101461274629430645988609498624108140933338059660169631314870811444800044313445693142184778994825571.53425451', 56, 3);
    t('3218917089390000000000000000000000000000000000000000000000000000000', '3218917089385949852172834776455742717964588641790942343490483708095.82481406031947961155711248292429430143394856627080672455290617699290563525846666486957169112878101294159543210616166118315105416859017730778197853022887', 12, 2);
    t('3528272891263019784691068177.69960377908006532953755589832438044446522978641817579209734651873884089047187812149564729145530757114', '3528272891263019784691068177.6996037790800653295375558983243804444652297864181757920973465187388408904718781214956472914553075711445124273459', 129, 1);
    t('192553522229555403.0874351976330645273968701055154391105001278169130930948316426855116815613990893372525769499696988338515868045457483629395832723', '192553522229555403.08743519763306452739687010551543911050012781691309309483164268551168156139908933725257694996969883385158680454574836293958327230980217556735098014319437095537630474606308312770264397030362459842316729019153530167409405159201590133083936845260416341738074472987698579446160', 145, 4);
    t('93042441521808984594458524132066078562885209947488722218598747571493545609167640639107914000000000000000000000', '93042441521808984594458524132066078562885209947488722218598747571493545609167640639107914248724266806606441572.53246107102503466059858370164451523869388706602360571887723074261272699489060548797646991468482171031159128852864026763904445627316', 89, 4);
    t('416677260700000000000000000000000000000000000000000000000', '416677260682506200478982108686838924953761402730492179608.7122294638185244701486831158458297745510145534022913680271795621396827512537', 10, 5);
    t('910850168612770086113526107147348585365869052279099794291239173740170.593800526209179208356191013783', '910850168612770086113526107147348585365869052279099794291239173740170.59380052620917920835619101378332704127095683121904503485415608382379643389767995780792105124941826113035878096656938699226031914150073076632979341347378505216293494', 99, 3);
    t('52317794020185400000000000000000000', '52317794020185395986266997016004539.6784431173048469198919139469974949851648576985035080218431430907311335468733887497180', 15, 4);
    t('3784575792040101335281447532961128424318903165981.66247949650895671697361', '3784575792040101335281447532961128424318903165981.66247949650895671697360808135229461334465559175', 72, 4);
    t('142179119057434575.918341039508323187587548567733581158873213120115135718551635383486074364131344829097972095313153', '142179119057434575.9183410395083231875875485677335811588732131201151357185516353834860743641313448290979720953131536784866693475147932967157585241873842503774355405420261234916730170626018487337636186566899368937227182353255400538143356420956751656996738281244007251925506721282212413458474336461316225462330085331160634228641421468492371650684650047', 114, 1);
    t('30115413108934169328447737175571094763917365444312236553023405000741802698264757953793581187304003351923870064723834.2208956082192531341284075505112830273316155889317141737647091771033001252598817925', '30115413108934169328447737175571094763917365444312236553023405000741802698264757953793581187304003351923870064723834.2208956082192531341284075505112830273316155889317141737647091771033001252598817924978739', 198, 5);
    t('71749654000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '71749654020015474257844958817969046061680671408571066330443676442389574633680534153443486159.0578327711965262079389536069685889330601556398395134553371985595185230191959766083830438531316498386559436830327844903022299463549217430379364402971015384135589697', 9, 4);
    t('14704487698559696503376320161696345133.3491817828387322314568391050691401660418293704419907207433964821506', '14704487698559696503376320161696345133.3491817828387322314568391050691401660418293704419907207433964821506532618076211713989731345503236717573450491499828813180211360442726948758165429244342902299414704097438529065200620203182199298575377183612699206031393411003109677603769558206958225827365449577561609417673813944453057737398320020412068695720456869', 105, 3);
    t('10130420039026661037545862332854852354.2123509817792191028508814359967050145299560592323166992602161', '10130420039026661037545862332854852354.2123509817792191028508814359967050145299560592323166992602161348219564177155887016566168623123174584197830031938936912836321597504299862024531017365060648987436163892741514341426374050502305050698609380175350803813796109343206560126795801407428448', 99, 1);
    t('853226013794494334952448668048746023105847945065884631255107440596301203834203480345807205067883117728300873887025488401556198407470000000000000000000', '853226013794494334952448668048746023105847945065884631255107440596301203834203480345807205067883117728300873887025488401556198407474746315584506901812.15520331', 131, 6);
    t('4026065434113110892231045507078360071960330758357593755423057564000699973811589526349660021714718214657304290585.65468', '4026065434113110892231045507078360071960330758357593755423057564000699973811589526349660021714718214657304290585.65468236218901562169469231540407464639670383400652145411102186701914675913660002286849973297083947294062050', 117, 4);
    t('710229597635160603148505641496608057330418601423589720118232895518873204772614989105417995904455300354878213554352751082769451211851567619514104552885371441149787573000', '710229597635160603148505641496608057330418601423589720118232895518873204772614989105417995904455300354878213554352751082769451211851567619514104552885371441149787573047.92399819', 166, 5);
    t('329677513523687470745261910122202200110207594189842969917584406715143389633060000807291977523597043849883385336970424244903477839519836118584946761492.76040232918991216937', '329677513523687470745261910122202200110207594189842969917584406715143389633060000807291977523597043849883385336970424244903477839519836118584946761492.76040232918991216936957169643670150583981451', 170, 0);
    t('4790098000000000', '4790098547017577.44368580923363209250815899108169987137576625853589505395209568342902474253739309468280063380231199504312392136871274778272329138596848457886002771592692220737691336690783997', 7, 3);
    t('2023517184829906148861235759981190894222724805531830614523239579304022386210378996850718222968.46148157396734770714766267847087679618192806416998250079620318466998', '2023517184829906148861235759981190894222724805531830614523239579304022386210378996850718222968.46148157396734770714766267847087679618192806416998250079620318466998097499426030365219778662054974895868862351084866636655323787259994224886202264745279976476023561038316652389794757912303295723801', 162, 6);
    t('89543675698737163031407295740145970930000000000000000000000000000000000000000000000000000000000000000000000000', '89543675698737163031407295740145970936982047231953473923591895614240395019507585433979720012121447838173907049.5153606510918315788613848085105872', 37, 1);
    t('361614560812460800', '361614560812460792.60128657642360910273360836907199719710314087685220934035501909323304581592173682434163505780939183122080648915532848297218311860419676703269653202865473825358809612302716110079650228413734708284721256784826118750596854003739644151040476712730791310374590324935836212508807567170111742166319921742', 16, 2);
    t('89977662771718400374456973081311912895157829615589008470.90463013597', '89977662771718400374456973081311912895157829615589008470.90463013597048991069637274668748288977055977190788433018710324653007947867253829823687216038258895738544197192039216432866005159944284747092537455420945123731966582235488683515177224817494727936224332120915', 67, 6);
    t('6930793562821742708143254456176113494456876323828082668603473.6690126222611181098532290213322288360749823779676306534021401972900324881', '6930793562821742708143254456176113494456876323828082668603473.6690126222611181098532290213322288360749823779676306534021401972900324881027787526088348457250498188876544060994596643633', 134, 3);
    t('8236635537276607028315601311774058276466325093774784679730318103906455951052646720204838854940406249745389433567046101210791805455500491034772859549627403319744.882219371', '8236635537276607028315601311774058276466325093774784679730318103906455951052646720204838854940406249745389433567046101210791805455500491034772859549627403319744.88221937190933197025250169242095863605215250211868663008741758500250986153559626236400811430468990580113253226865716566005227721090', 169, 1);
    t('4550565208654600877540027366421399048442902287308913327238842312494672605700581498825282615077777617022315638297367040.67245850524959093496202000309438291487258252003945451', '4550565208654600877540027366421399048442902287308913327238842312494672605700581498825282615077777617022315638297367040.6724585052495909349620200030943829148725825200394545099245142558165258722979259070480488992081932999', 172, 2);
    t('346497801241823776714947456165804092247826277125608267983475760576775473624850221770263154189697202661005379546416628152926268600000000000000000000000000000000', '346497801241823776714947456165804092247826277125608267983475760576775473624850221770263154189697202661005379546416628152926268643761450431177868559484487540941.075593698506427', 127, 3);
    t('4202669785183946628583760000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '4202669785183946628583755540633812539738699862227740542995216405756992768958105926606935096165487262417991138354492282.176411699328865481781166378989440123989097318383342415670059703569221999835035096986149885163202322654375029558284266669524452645198431', 24, 4);
    t('972129055258245470000', '972129055258245471979.793390404944033361381818835336955723303958284266698252690048729701482780114624929745255532307479304260162477091447280082810397998411046849008706843158311402157972435515346038937360729453304', 17, 3);
    t('3970751274783953566802615596253527564965151714652014964080000000000000000000', '3970751274783953566802615596253527564965151714652014964080999601965332570172.8201131924012234517307711409892853632090451437581490975003', 57, 4);
    t('4570603649226373115989445605266140247839.646462565021653630214374627240656515735550367218256267433952650211887302655', '4570603649226373115989445605266140247839.646462565021653630214374627240656515735550367218256267433952650211887302654569636800242340664774678186553227972910574447008208767153731687971787765956053460932361476974710768', 115, 4);
    t('32851154696821735840833035.048447648935237751479572948', '32851154696821735840833035.04844764893523775147957294787759567931150117812677663642847763987793010615788924616105882528427546480755356834573764369307666715467336067091900543048166649853072265744393785464741410280503839024458650910282825403593829038092951045609368023272664111326', 53, 0);
    t('988114642683022858882647343295243304927989625636494411431779233887952121449500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '988114642683022858882647343295243304927989625636494411431779233887952121449422404603206885594161079196730840681234275899458132330967621612980292686111080734721315117212597648789944553040874120166656175567156272674706033019.17118', 76, 2);
    t('41938694322401354833943416472656183212000000000000000000000000000000000000000000000000000000000000000000000000', '41938694322401354833943416472656183211546638962470248128823661145031849536980087464840670292658694344790691406.56330137307434195691889571454592912903766813438171091195823694277296857106267233679037569012128091845501939976560799983869146502319555381501060152622920463056197649188793433332813416638616288609675239736385182087605927930681424818838332075544249454', 38, 2);
    t('4553787777618701287712937081423348278.350031834821844849168711381454099810235297970256488579479077843040529965236544869242678636256810202954315', '4553787777618701287712937081423348278.35003183482184484916871138145409981023529797025648857947907784304052996523654486924267863625681020295431552855794979470018159509500560342664864045958454740113864145945318258', 142, 3);
    t('289336002078027826928906750681464953646975066446.4090245509929622762461189475536039799', '289336002078027826928906750681464953646975066446.4090245509929622762461189475536039799823157326900345629672968862109551761215776367131578166601240744981792044297109855468470558888927696627587356491309212883', 85, 1);
    t('6645721398557031592620308924000000000000000000000000000000000000000000000000000000000', '6645721398557031592620308924345376201470307217751849731201910403405688001595810656655.9318969522580288533255769665093639075492713142320493574654719708236611746856894821984655', 28, 1);
    t('2484404807862586344441997.70064749062175456745403917153805394624', '2484404807862586344441997.7006474906217545674540391715380539462399825559751669479538189086906811481020687244292892838466858371913954289002237503138776207967347729800318411815997300301223765488691927696925471943911888394672867100197713336231938672274534833838122943045459012195805841552893147444850888309591174442839623585046631250764883165404502467896080647303530942843664257137970011157997657413211689748859733533', 63, 4);
    t('2676940665349275034282071747695356780153373034919920447544880067195609648438783360000000000000000000000', '2676940665349275034282071747695356780153373034919920447544880067195609648438783363506699000715204083756.92955449183828977483391006560443628001806417836627832937220600742062514732689', 81, 3);
    t('797867133185896662931300589470352055784237866342004.521489309673', '797867133185896662931300589470352055784237866342004.52148930967272275519751424125777', 63, 5);
    t('991566454868676634135604285269233151457989411350.5237933304131005908394411102782274040549627089901293211969993741509459968374191470053100953055424494953768800955008707', '991566454868676634135604285269233151457989411350.523793330413100590839441110278227404054962708990129321196999374150945996837419147005310095305542449495376880095500870712914733256121788162835029696027249417974257103616943462032712448471457052458458106371271438445262371', 166, 5);
    t('508800035718928040000', '508800035718928030675.49248045797678096368643615459488842716465262526505105677758422635482314470607491', 17, 0);
    t('368747404454539821783395399701617992401.019280083917024876088027446338621924776', '368747404454539821783395399701617992401.019280083917024876088027446338621924776456611620553536503677581305035558599018429778780779371836061769312683297535952094456151164175497694629673208735912902119178428', 78, 1);
    t('625242168831139068858904610940710267666115484335653899797989200922890513064601354513727209191409592969532000000000000000000000000000000000000000000', '625242168831139068858904610940710267666115484335653899797989200922890513064601354513727209191409592969532061463671430601591798961521234385670581994.767540611388794654089985823944442302039806883533355', 105, 4);
    t('99326416226240', '99326416226240.776888425957305886937532942082208016595091650673113882926815541951551319901649794', 13, 4);
    t('182616997577209684927124529232225257584994922201430330497421592315950074824012956378580982933932139050630.43613984132743195170648628424062', '182616997577209684927124529232225257584994922201430330497421592315950074824012956378580982933932139050630.436139841327431951706486284240617758492205728112540068261946567475407697925026066380483681042092256427175395026212229256503059017688940979831971992037586409684605310884', 137, 5);
    t('65996021916824609605512699435173552420478819614703638000000000000000000000000000000000000', '65996021916824609605512699435173552420478819614703637970114481640370659848680734277509673.1438383712668572004760893602547948900487480571266260365187284329224997984116633657632595262815563170968860846880474600948674240650564582029230129102698514076057125492885514721047918676289', 54, 6);
    t('6484255066785658403683346626127391927539728184934563325088207073309031203339335704410296538902039008842534841778423846232956264982473262400000000000000000000000000000000000000000000000000000000000000', '6484255066785658403683346626127391927539728184934563325088207073309031203339335704410296538902039008842534841778423846232956264982473262468433234652921037012097550682213449526295536636673598290088708.31711664933404573985514671842776450661293', 137, 1);
    t('2565042366817410604752564151509171859148391649606633110132681263753786615290000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '2565042366817410604752564151509171859148391649606633110132681263753786615284400845556302868967465245426277745825207106356347215672729211305092417868737893794967743.82960177586463828851752309228680031539364405502734486072015', 75, 0);
    t('37763855138691163.35076714929993249960227597929957571956365442902832011891169749584612114641603585959072299197887527521616967318855043730341136265', '37763855138691163.350767149299932499602275979299575719563654429028320118911697495846121146416035859590722991978875275216169673188550437303411362643778947585735681311583259077396250507123202933989059575267485412420596982375298455817790061885137070757800045525547071563260642907764443', 145, 2);
    t('786820789489686762319942207413604000000000000000000000000000000000', '786820789489686762319942207413603499494025096732478697261643422618.133747506506301848842905035059491677233768845888814737864267', 33, 2);
    t('99386571358061602750885413729671861764478179169607943633817300000000000000000000000000000000000000000000000', '99386571358061602750885413729671861764478179169607943633817292349873314495667586396794722637225417486845112.6426321527822193278462852950929184063456720274481213530559324667775684841561431850474981113508958', 60, 0);
    t('7709.8414536762330906088087832769125042738420312511350415', '7709.84145367623309060880878327691250427384203125113504150549819247222186233248818322648538082398455688526968302762909471860748592024340940300901170188617454163667282791568879', 56, 4);
    t('155797784875634744181938614800221992887254700000', '155797784875634744181938614800221992887254716384.45919348240051958968867936918923894465403795845082839580497773364502118396518972833984089640288028976866530384080159168577588540502705371690636741754767955521961691684286718828727787607892745793049597234011842090075796057097152037478561722537258605891260073623086030742573440931151041686628', 43, 1);
    t('997816478757005678988147272824101563448468775996.677511396489490889930594300367737152986140034444973', '997816478757005678988147272824101563448468775996.677511396489490889930594300367737152986140034444972644616492796', 99, 6);
    t('645113239428488651258102738933762753847396258123896988596386686053185725744425714912.4469290469587362398436152216399420433119024876386316645280753523837351809748257860295048467', '645113239428488651258102738933762753847396258123896988596386686053185725744425714912.4469290469587362398436152216399420433119024876386316645280753523837351809748257860295048467010807997339361845914496334915252528934796750230875359740657569235314436526777603852281293049853530537900', 176, 3);
    t('93274152804739294580544308872236155796276071194401345550748173408439163308346166326143127914278538677609966657972512158546460051051040177766425610760869434273850013679485879932057185357475793417109043690294345100', '93274152804739294580544308872236155796276071194401345550748173408439163308346166326143127914278538677609966657972512158546460051051040177766425610760869434273850013679485879932057185357475793417109043690294345141.75068240666547978598323788', 210, 5);
    t('45500636287317800848839531369312876286194933907418624434755228859.598407254137351705219770055814416274980552632151473755043', '45500636287317800848839531369312876286194933907418624434755228859.5984072541373517052197700558144162749805526321514737550436143792830474955815225964782464979186887', 122, 1);
    t('68080721808051794496614471241939392874594874000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '68080721808051794496614471241939392874594874204233369571539024811455711867804854600791561281985761342907369046901635552368503304354805371029257460352717132463337704984983893.08573530725150324739285689173784114519778905738811581', 44, 4);
    t('873376870641522640000000000000000000', '873376870641522639701607619065750060.96933', 18, 0);
    t('9816047129282632708063870734821536990191104745440038732449259493507609927095428798381272816407076250812000000000', '9816047129282632708063870734821536990191104745440038732449259493507609927095428798381272816407076250811704269197.600424199', 103, 6);
    t('80108472302646771097382157153122270147932620463221271744387466409755801.660951338033687308220708526953341324284907613830646606545064980704919657562541630197769011448014342930167971', '80108472302646771097382157153122270147932620463221271744387466409755801.6609513380336873082207085269533413242849076138306466065450649807049196575625416301977690114480143429301679709798129999188337370071257388712', 179, 0);
    t('4000000000000000000000000000000000000000000000000000', '4460019063976506868135604153756690447781153709631332.12113203121117613705667647917427278606447809637340696260243705466641412257621260681906382244438593887188283923279092401313729485067698500118732993041391030529840827860867038487700709692445293731147643350668578196204633377554383252385950843694345', 1, 1);
    t('55210059132209877552070539023184418093000221398291676553805624868843640170535.0306', '55210059132209877552070539023184418093000221398291676553805624868843640170535.030624987', 81, 1);
    t('9686028110726214117447998043926670794934483953397716240920160338864017756629997839840000000000000000000000000000000000000000000000000000000000000000000', '9686028110726214117447998043926670794934483953397716240920160338864017756629997839843055120467866541100317887177107746045835829826096570331622805431461.3116247166770931480905593023772017771508665525120053875140775097661359695713128487278695168230640790699744168982962464767925261794046686644504717', 84, 3);
    t('123679249680932827147011384378481363907602015648298246974212980193414156041102249328848086381581654701895420468646613044166014258756487.3030267129283437', '123679249680932827147011384378481363907602015648298246974212980193414156041102249328848086381581654701895420468646613044166014258756487.303026712928343717169221886644335163947921832389279491558544896792470167311413937865663577377262675458203497521062332201738753303115915695', 151, 3);
    t('538389302360271498265702873230000000000000000000000000000000000000000', '538389302360271498265702873227443816220078985789607879584644120228990.32081878857124639316723274524930340931345660879969762098256262958444701217282057637620120194595871301315008427786405888660783919821943249939097468712984753388225403785635613329551', 29, 2);
    t('406088603447286182872.2720885757179280591204732035300673930692610969580693', '406088603447286182872.2720885757179280591204732035300673930692610969580693631153806008307090744186743939851207862840150845602581', 73, 3);
    t('55177093624097065961617548529493522744291617919288125332644024106318047576333010855142909513912120040000000', '55177093624097065961617548529493522744291617919288125332644024106318047576333010855142909513912120034241187.810313701407433990890146005956522578003581680112928257485276411374324268493552890232747491', 100, 0);
    t('96301290433438267763250011057803090433027991628517048930441265737056378570043021323115913242772753200000000000000000000000000000000000000000000000000000000', '96301290433438267763250011057803090433027991628517048930441265737056378570043021323115913242772753193213804221656149307877741604195303789202094699105868843.2587266831027081', 99, 0);
    t('10285437262130948747813295616.93009414243396604656954983189000487371561875317839222048807900894776499469691819371004168874070289908750538558923855379168434121700657863056204931176954892057021013103362', '10285437262130948747813295616.9300941424339660465695498318900048737156187531783922204880790089477649946969181937100416887407028990875053855892385537916843412170065786305620493117695489205702101310336193399570641661893312691700291664042640810600506238609270171822690974714120628', 199, 6);
    t('228240249658974409861769399038561754370555500000000000000', '228240249658974409861769399038561754370555541442973611232.67334535414836177416375123951438589054902920966199593377353818685652309446126406319890401762272133989', 43, 6);
    t('888343594360240946564342279640151636326935654951.98', '888343594360240946564342279640151636326935654951.980196994493129947403493471765201427356558466642719715584585304149464225593350310163658483372063714794419648426549550829299246708928094237635302459635220906385659652438908303253433954465020608018132457177920224142043144300310808757552155274451709419916339298251835929242215834349748903089', 51, 3);
    t('31920411276029888790422315558270619344000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '31920411276029888790422315558270619344305595106728100941947288186723860021378265462831338077788943669140263636580235974581645.9616337587094563424792983357532417245510115518989548188948450150397273994878343444230531816186254271104078999671653159512909686096657858435263612526412905167898449694000227994395449119145753404804726338297497', 38, 4);
    t('2995891771324798968218639549149559107770750146017381650918320354241036394663639617838370941128129162274464318876670', '2995891771324798968218639549149559107770750146017381650918320354241036394663639617838370941128129162274464318876668.25821487223493974295831164897', 114, 6);
    t('8.846765324272458628508964872949313181196', '8.8467653242724586285089648729493131811961814495737609127543137913650620', 40, 5);
    t('52712825126305471768071738368.4239', '52712825126305471768071738368.423823438738577030847471930930859204087137227462821889388895800915594324906215015595401199628242078774042655750123231334802213302969916', 33, 2);
    t('45458001817617673573675587468862052036718314127202503241208678465825982828329904894536253723822674628743408821547800000000000000000000000000000000000', '45458001817617673573675587468862052036718314127202503241208678465825982828329904894536253723822674628743408821547739129189084680453992035909263024365.7970780057639116019149836107063686843748482503435547552654907627963417548270159169039105074119292', 114, 2);
    t('31876658623916984984110748098395089918253689827285549106667048558267969069572190844569639961816907982746000525893341695264132794099690915000000000000000000000000000000000000000000000000000000000000000', '31876658623916984984110748098395089918253689827285549106667048558267969069572190844569639961816907982746000525893341695264132794099690914941591998943575192751129985959637174582607773349143980808420811.567085071949832382934287248092491405', 137, 2);
    t('27339141149365219086561392.3736', '27339141149365219086561392.37369320492325446005793386959013089763123249188221245734965290331296390429470131224971234491745381834198224256634216491180758205583791843512296409570335022888472880507892138826253491824906041966261869473702874059297696237509324212397358015917571228162557607618840438622323047339122725429337731429396708946049723245072291942275475839', 30, 1);
    t('724624084685891604510000000000000000000000000000000000000000000000000000000000000000000', '724624084685891604516535951694648993203981817233435889280096255243743422047886778634294.32732140584853071555822472730820550563818797956388424079874604801332587387180547937666752901663521614749309742218394543573828574010965932224597699349121145928328666039719302134774228284529796508817035627887286674413615', 20, 1);
    t('987433953289436513470955635279236297804534000000000000000000000000000000000000', '987433953289436513470955635279236297804533647505219104142255991378813582282112.92656552840677822830959267845253788212523811071614158100203480', 42, 0);
    t('78598744465666863567578992841900000000000000000000000000000000000000000000000000', '78598744465666863567578992841904580929751290310952415438605583863966193332514186.5872407464930829603199549420574740453220827389948387', 31, 4);
    t('560685002913854188938043503238066629209010180134481370680681888523280414963494821377220000000', '560685002913854188938043503238066629209010180134481370680681888523280414963494821377217364893.5766804076823986809118686596439656747185420134256', 86, 2);
    t('746775036412577214930209215845173153156', '746775036412577214930209215845173153156.31529197345625314920179109243042414853017101494869707967412395441749289403123220884915361966127982490135717972114806465', 39, 4);
    t('29952798591768616581232336428000000000000000000000000000000000000000000000000000000000000000', '29952798591768616581232336428179315219500852763465195835209692521251152207330612075272165625.0250775400117938754461187171275519927285643839971969796786154915386957461376153120689204457675221756103546120553132266629279361060207600416445985557602949015750314617194621930731613120982927969750224868837447863548697', 29, 3);
    t('79718774329817982032004830881792097761374501970639604951849201.77009819912236482807308', '79718774329817982032004830881792097761374501970639604951849201.77009819912236482807308408509650539249848731177862251660693037239133268312833401387003184397800926841519199523718014831631602693392103520515168972611980300573369736521716098900242660243411932115625603560496361501902301', 85, 1);
    t('779601233424108366518673.308140514241741955679168529073274228', '779601233424108366518673.30814051424174195567916852907327422832626486908677700118220948737320933598558781657281750737053354779941771353008173981927863196473140144269311300866394687616949565724933022874994645017400169234503261103212367', 60, 4);
    t('852934977550517804793098038663330351448084347101505365.53650998646545108098428621413110971375514413', '852934977550517804793098038663330351448084347101505365.536509986465451080984286214131109713755144129476008646982830965318736114806272567887062030071612402013609429910', 99, 0);
    t('433101380703035702777673294508022207333794577958150530462807415844485267734901192346511380235736061756316669395439043169422675400000000000000000000000000000000000000000', '433101380703035702777673294508022207333794577958150530462807415844485267734901192346511380235736061756316669395439043169422675359644700257379531508935408495376237744699.5733768682639494648398734779055075146246986881009731657602181444617621102923043771363', 127, 2);
    t('49733996718891861556042003995224816763865689594590374593773227708.34183459813399843703928034699253822718406937', '49733996718891861556042003995224816763865689594590374593773227708.341834598133998437039280346992538227184069366131649868095257754928301784434', 109, 0);
    t('13427677029307594120106055332597319690000000000000000000000000000000000000000000000000000', '13427677029307594120106055332597319687534019273841402756731395097694141294221851916117163.06586360340731169032176153803381422696830987440881776693297964815367775030717310403170547200617486838553267', 37, 5);
    t('6077558843061190208394947089063.98389937176033722244113266276140158875956443964821549000244381651339501592358104423833281623275025714048164459948091280847767472802284111714', '6077558843061190208394947089063.983899371760337222441132662761401588759564439648215490002443816513395015923581044238332816232750257140481644599480912808477674728022841117140018752180729345694003616536680579149', 171, 5);
    t('8991353849977649164002541121858277421967233606235066745254546180100481632686374495592146130633932500762503.570218208513058394181928526355027500768962687403559', '8991353849977649164002541121858277421967233606235066745254546180100481632686374495592146130633932500762503.570218208513058394181928526355027500768962687403559212858784644310', 157, 1);
    t('3231908393141166100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '3231908393141166051207719549508782761904188264000367117855986684028291722709141871560013419192212354549574081749089.6179787489661991088819039370600279837110884785637001637893143949031456732185677249039182995303437534113481979693853698765056085525492374973913243997868083544626507036329720541419452206841232147', 17, 4);
    t('34505106292710353613500241.413225657076924534707860903994719102', '34505106292710353613500241.41322565707692453470786090399471910106582003249533133398204295470822269123736472852595732911242781599163293100633895', 62, 2);
    t('10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '15148100830271726374398777480269674734213353638810935885251049514266345918721388996187610611364.5466209604577775507986544331960849023514654983847692643600652316761713222465326797262684469593135416979020696078701249118293114417948892044643706207923', 1, 1);
    t('85038449041926952008088414345823227277409238749.293257327570523127594', '85038449041926952008088414345823227277409238749.29325732757052312759472862137483240158559413312362352792154704649585891037189632232', 68, 3);
    t('4455917060947902959655.17071595889172572701374417004075419487113', '4455917060947902959655.170715958891725727013744170040754194871131008079845948895838440889915489680824480727564660981473795301550935966603556789524843279931884636634137991542787390046666697903636822582622509441445110278482812640362137938148266004911742418242849201467738577615558426308812760147873282547719488473831465119022857664456097696191108', 63, 5);
    t('124102483508757086287433374930186516266371604572417062134.31633261276249319906542207716479770271327970330349393894993785145372966583503', '124102483508757086287433374930186516266371604572417062134.316332612762493199065422077164797702713279703303493938949937851453729665835032958647419206130561880362978690', 134, 1);
    t('6816781351279212342242043644762648522767138000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '6816781351279212342242043644762648522767138166549899011681294030429958235134369167724608929105796715107148979322688572721122295003854497518058924626476519434806.6869204149476484447646780560674713', 43, 5);
    t('92295184074.2645784061473360792660572886233004659675583082994141198020226629718992726593574', '92295184074.26457840614733607926605728862330046596755830829941411980202266297189927265935746609518986610717669969956085867211574786296176909498609640263703210512180431712333041132735307682781649966455077562248218160350808660933998148930040781837306985075148836931', 90, 3);
    t('6226060887330732156139303364532321211352238.4721362472567657652892965545273727550355503339790077620366561716412346749773911264999862001087518147285668040176814243', '6226060887330732156139303364532321211352238.47213624725676576528929655452737275503555033397900776203665617164123467497739112649998620010875181472856680401768142434339597829240186845901608326866974713146336398257708555', 161, 1);
    t('959562120516903125448065494180567180075792297719455998153892018262549066756768071719607768175868.11998928656285432515658154118349015209', '959562120516903125448065494180567180075792297719455998153892018262549066756768071719607768175868.119989286562854325156581541183490152085510117880629896606110383016260656283462854435508472426256338170599532278745666054750597000540748417113003634584042005791614247990201095052197927793115070622129955161163517579973772148949438', 134, 6);
    t('5957961655958801263120000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '5957961655958801263118997397237895149431597522464360568994757686439724360789743076979523424848581834772460154319495220737998242758662479493819124571565276353329899900.374008475414450756890723615778029649630833111155384467180623515268971039206454358488894872027335561598929843064250272697450878425461', 21, 2);
    t('45446735144743266618340341584514257362814452403146168110585443825889754129768906217704685076356296030787746910565698050906.8654041274696305457845792092181317884077948', '45446735144743266618340341584514257362814452403146168110585443825889754129768906217704685076356296030787746910565698050906.865404127469630545784579209218131788407794765433447225018181670155601034922629790264835266277424125201997724890229110564484722381343264394385710651889616387808758283', 165, 4);
    t('887430782250150076150252553859697852608507614367800000000000', '887430782250150076150252553859697852608507614367782875563300.7311131485499819537050020797466659075091954561913985694734470959710431340615700845589608273045397309393483782764138920819120047913023556312344118957723134360789257297991912305856428884008088902395146728425218877131788358', 49, 6);
    t('92485438800000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '92485438798816207080262628218364906707241176857949005089288924166977504958760466898081373973619772067069873572316040538027375768352477184978338389785127648340334.83397594606', 9, 2);
    t('604890374315071521474515258544236308856232575355626712047293339236648760656987618979636040968684662839000', '604890374315071521474515258544236308856232575355626712047293339236648760656987618979636040968684662838191.290', 102, 2);
    t('404931223561877662422607121533422955404801.36284229269793446885696141531017762564856786974999252155772993225963263399025261350947969447630175019928743525827338949', '404931223561877662422607121533422955404801.3628422926979344688569614153101776256485678697499925215577299322596326339902526135094796944763017501992874352582733894916639457867417316303675813245021250230139028135732', 161, 5);
    t('40936940573216170626226.6567565235855823376821', '40936940573216170626226.65675652358558233768204660230089177475139525650179519237994645907389329430320876819355', 45, 0);
    t('49316852571533827730792.13424888385334', '49316852571533827730792.13424888385334673201836578443171604230712266855796011518409451123331053657393902801577192837730136569057039883440492905602685467843183251771173949011594203535407910668674088702386643022797344666425', 37, 3);
    t('864539682203680309603710665652986392003303174290776256668330256713545423912800195919664.51332275224135239472849624825212447865138254559816224343567705894094435', '864539682203680309603710665652986392003303174290776256668330256713545423912800195919664.5133227522413523947284962482521244786513825455981622434356770589409443558694946309265297325534940385994375278606338204030149334606925079806', 158, 1);
    t('9312846.36275981891881809058781654515839707797373056023272155253111969937103557933136512775327856945038024760460165794522422', '9312846.362759818918818090587816545158397077973730560232721552531119699371035579331365127753278569450380247604601657945224219428899111851694224645427029126', 123, 6);
    t('3037658591.64020161309519610016735318890799644013021652241019563662', '3037658591.6402016130951961001673531889079964401302165224101956366228130766984934572326856484611179169972104174659744940427717580282486738595060918438332904193327145937326796', 66, 5);
    t('35305733359897030003318284801754779683243830297113756875962566936813516429.9773615547557840930299748614945939950375510486283943123', '35305733359897030003318284801754779683243830297113756875962566936813516429.97736155475578409302997486149459399503755104862839431229813649209126971526294024922573827397962448457809539012899149000069385757963487942265826244584740612727238171980884593308188253566274861132449444642055427995849458313087351679867781252133', 130, 6);
    t('5269378211720082443401590566034774281761991883100112316611359056234659.17445439193333765064995191627', '5269378211720082443401590566034774281761991883100112316611359056234659.174454391933337650649951916268005575933420381823810972064473925330955508049989535278947575', 99, 5);
    t('7395449561.257914786579545476810053884472373676', '7395449561.25791478657954547681005388447237367528067669506458195797712627612581064617068251182406500253008551018756930311867421185378750843961747248915153842782401416064066877617889756679442', 46, 0);
    t('37027785303.34325', '37027785303.343250417755', 16, 1);
    t('7158880354248766042792019624272.7498988228956288', '7158880354248766042792019624272.74989882289562880120987920799441469250824064453079', 48, 3);
    t('4384994879574956819638167162980890000000000000000000000000000000000000000000000000000000', '4384994879574956819638167162980885889355141416642053498435774579535302212451144536649496.8971973580369170201467960003163605214823366086796154345791883229238463700409832853056088141907770749156197793728887363866023853519527789362054813544827767050889684264122641652280308512533260092901', 33, 5);
    t('9868034639.8011566054448068', '9868034639.80115660544480676078754036095851061473951470123337012138031380347632317236895150640614823530386029690252184333188985077296118258', 26, 4);
    t('2985358.5862066295861774862875700230734861782952819068785003', '2985358.586206629586177486287570023073486178295281906878500228783210374826684537031306217528292353984440659099798870365776910298336747107093593886089737072352020304612126859', 59, 0);
    t('536155339924934683763653700000000000000000000', '536155339924934683763653687320417299386174058.366946202636667407506291820070797125665565987641937469862683119410345130942731789611614695099791370456875896482208128304978407734674283097241443547', 25, 6);
    t('57430764080096075620847255069952118347344076932332000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '57430764080096075620847255069952118347344076932332710279270103000253222435657658689909108504262700491815800880496499596324312211191917042683540415093.1547937987410444111681247138301133968636337959267997165776381518473503892792995483118076061754', 50, 3);
    t('743606305306.5049744078029844869844862833715465729420629939176393984705072465116963980834445146678367839124685575827056495192526037085111516565954407534984122219777171321961454321740492', '743606305306.50497440780298448698448628337154657294206299391763939847050724651169639808344451466783678391246855758270564951925260370851115165659544075349841222197771713219614543217404917751566248088405216897381542256', 184, 5);
    t('712724485169701048059267500069821623039541000000000000000', '712724485169701048059267500069821623039540546590043858673.8075258167735848575299000226842611406095493572389059714180652474276298782717', 42, 5);
    t('93426862053406382413019601459297017253270675824367022456770138394320172446023186620395643470528938530271256165297508203554343809094173099625526304315468001840000000000000000000000000', '93426862053406382413019601459297017253270675824367022456770138394320172446023186620395643470528938530271256165297508203554343809094173099625526304315468001843143137529414111599139489.13822943142982', 157, 3);
    t('2841733697576864723444015425219.806499832413461534498256132654624581242037878322', '2841733697576864723444015425219.80649983241346153449825613265462458124203787832199953134213270071918132397722566431136310648', 82, 5);
    t('6971071901734240699614546791152772509032300099743132815144878052266634978140141973104124000568130778839157659369400188500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '6971071901734240699614546791152772509032300099743132815144878052266634978140141973104124000568130778839157659369400188477197739228067999372257748968556408456755732533336046911429621283965797548161002917634513.05', 119, 4);
    t('629295720155904827688146394513459068935222775160582121602437418308049303933416024133623927200128599049168612300779297865654881846832459.2425103563693', '629295720155904827688146394513459068935222775160582121602437418308049303933416024133623927200128599049168612300779297865654881846832459.242510356369318077178654767668425872294084', 148, 5);
    t('24258474725106000000000000000000000000000000', '24258474725105160663295244552967127572634927.3194022975608001000646784048155', 14, 0);
    t('36205835865209519334904132034664465440551164780221570828635467521227098480875787590000000000000000000000000000000000', '36205835865209519334904132034664465440551164780221570828635467521227098480875787582129596706490473150355909322707625.621233738362268412598701628551635247939858830560461845717985432715689540972091476158969374188110936543', 82, 0);
    t('83801183609627382559335265286564834495388988396223310947.79086105178704504128932251152659791398484788294012439453493281993776130514630578245352267241', '83801183609627382559335265286564834495388988396223310947.7908610517870450412893225115265979139848478829401243945349328199377613051463057824535226724113996204560051007902653714989501123412170611677852855844343866709827388506105830744915962031', 148, 3);
    t('5047375304226642024646081978467292138054542394030000000', '5047375304226642024646081978467292138054542394033359489.2200581907011210007434192605020110574226934878002532750817928395316652561355392353589078494815701597777386086246390810297129897288497477986379696468351292152918492874194510199603883676066184252789664734463082112914103881212064019636965959542065404814570393894020241129611017614897102609190183437342663197', 48, 6);
    t('5575.4', '5575.38406125179538314463266994958741696603874687323274923958079421951960508319331441164738130323276829884241082772900905717381446941865329481161662614750668963872045892792787861255898052400487315998244345165969977233154842292823655236009411079409639572350898417523741017970691935175900543373083730964450023399421972656227425590315928133747285', 5, 0);
    t('210769591180064197435247560688569690906727787267687350431592.629577226258640575645839051668527817548220812951969334603201581314093934899611500163355721552391436294911716051267430516837', '210769591180064197435247560688569690906727787267687350431592.6295772262586405756458390516685278175482208129519693346032015813140939348996115001633557215523914362949117160512674305168379463551600229394861618313839773395958210560796785603285733429', 183, 3);
    t('725642433199783894824980267150417722457302792667634392884163282145934272818688801480947848676370332800648169803641483260649050207917698085097589.26145', '725642433199783894824980267150417722457302792667634392884163282145934272818688801480947848676370332800648169803641483260649050207917698085097589.26145', 151, 6);
    t('983171081689636633030766387525042298358060815687096403294144356764829417939842304451871605326190530750.4676005753029055459068020657323898597131847287560584744104946729343877342472', '983171081689636633030766387525042298358060815687096403294144356764829417939842304451871605326190530750.4676005753029055459068020657323898597131847287560584744104946729343877342472', 178, 3);
    t('274.27007832860400762', '274.270078328604007611424301653566296730097735814085924000852536786424689114379965969377327370203177219729951692039042770898713510420528103092111372735876101294811927459992503831382303166790689976250115768153226162097277812698566506927651262325629438366544255506010291730538017211631236817412', 20, 2);
    t('479643052335617889093893416228533238955501602589031839477588670000000000000000', '479643052335617889093893416228533238955501602589031839477588670612701596468931.9938092772069539429761915409887853800772083326369485073460902695210344361664652162331218169140730472189828798223995007444578766725210970222950521608072312709132113009080736', 62, 1);
    t('6922892189283966418515098461261417581572724527473416368792700000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '6922892189283966418515098461261417581572724527473416368792790149245282176474224718644231851810564658657960644633720262611156343085210809414109025457906770.80546135877329183165860167565885168291612752754952225873578425966804976757883973391444119', 59, 3);
    t('91052700000000000000000000000000000000000', '91052797806790557665666600333047336915600.6006568113778165146439644578551844983725487458438092763366718819375056090715215740130039256745807733473617982457689295058876141533934806733135655046689207105718723861107153', 6, 3);
    t('507176882818.9916148545338346358132433847816342', '507176882818.9916148545338346358132433847816341639342152993332330917768222797585022116291105194617513623750339457362372789669214070061448750234745008426491200999842993296806028277359180312279431519913772616201425262509763815727946362928413818749630888', 46, 4);
    t('1186335777860704388640850397000000000000000000000000000000000000000000000000000000000000000000000000000', '1186335777860704388640850397322319085362236773190921940922202936726876616916615098917995011639430836005.341510661266975083102170', 28, 4);
    t('8049703225199990412276931027470505250752888334413451394193768613888845822746002554751620598430759540148553354256000000000000000000000000000000000000000', '8049703225199990412276931027470505250752888334413451394193768613888845822746002554751620598430759540148553354256695738425142702247201709745977107311020.0122807368199571329542502201417495', 112, 3);
    t('87878092846307299698852233672897.52150051132222019417637199862075224519304926462552980537419788', '87878092846307299698852233672897.5215005113222201941763719986207522451930492646255298053741978778378826684460754743730410356607736862598554401019822475895763473870314317905821221', 94, 0);
    t('67143922170380271075183395498376122.1597564824533170264779104603', '67143922170380271075183395498376122.1597564824533170264779104602519854711943145500483784087525348905433474576257048142780561694', 63, 5);
    t('23577369521740717487961308566439438626.7785220729569378477468182883418498941494290474595', '23577369521740717487961308566439438626.77852207295693784774681828834184989414942904745957150966050531651950077152342163259955184439890870690659962512976247204137156504', 87, 3);
    t('64957604189098460745938269474842987795007890000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '64957604189098460745938269474842987795007895710023214517273257084041186001512739679629716925929463562345943901058569889048543158693508625952017405784281985840730725482605666006220546652032976202609868897123915879.9375033097456163346708482610139936945901015389363379', 43, 3);
    t('3228780885684373962232058038565.06919386374042191016', '3228780885684373962232058038565.069193863740421910159463722056218716121699117169472811805604854639596573887', 51, 6);
    t('4740974162109662716659271671981589803716385827459392730506701840558791743766553658828256786667000', '4740974162109662716659271671981589803716385827459392730506701840558791743766553658828256786667057.5737134199035203400210148552039639551588530936384374370987741864306328465635017069536799647779178250820024569031667653291833061645812660244564841179047332376481339102652277511875012099599842569411138561184557527999226368248557425447065446210309583821227337008507080948662861601', 94, 5);
    t('60538401510614373864798077132070934651213513830288186938663427752267306.2085598910425348676196330230007476729978406559015206721967763286223589184464548', '60538401510614373864798077132070934651213513830288186938663427752267306.2085598910425348676196330230007476729978406559015206721967763286223589184464547410358148618651613793249616910728994514424223451310993204850679353685176335584152233', 150, 2);
    t('1929574558305162907746.96691553883056229452943929748', '1929574558305162907746.966915538830562294529439297482454986465280774042664012040583282771196371137050320173066272308162078569002989820622720207119648599899180579713715383091971311033770081251472719634711443488873848547410834052934237251981916073164812663846739767261465204256442051542087005061275782', 51, 6);
    t('57719461670979034536713307964236210395561629582972894257899333637281909.4736376829427717287230660207722474432666582533018', '57719461670979034536713307964236210395561629582972894257899333637281909.473637682942771728723066020772247443266658253301751343908648019450267676319607998706', 120, 2);
    t('321816817986416912450449901639517862603839945256095998171371805220410094212016413735011968913134737060611434698993271721080683650996922367315000000000000000000000000000000000', '321816817986416912450449901639517862603839945256095998171371805220410094212016413735011968913134737060611434698993271721080683650996922367315038774932640961384388989670608231.7247700362204857611', 142, 4);
    t('1211749267.218699246854', '1211749267.2186992468535282059590443899956724705712362944368929430698529213852478259581930746413931109736375807', 22, 2);
    t('28000000', '28145122.692030633988040261319661010018229240789021936919032451998657229652344087831463486810690705830203667688092837362022562577450983247804741696860456684382947046622736244569059649058669668945867215280938953638845120021126478670059745413519822311154551199038530131050405538018259415482577750760502792197995333407607470784353173486308982433217378471328499044', 2, 4);
    t('70421234905147786797988571595540000000000000000000000000000000', '70421234905147786797988571595540838902836736045344671215627153.8015346343521474996051739875431685958101971395215505594748077907266416570390581831729496559515511443834374350612204230394066876336137344958211042509266596860917564', 32, 3);
    t('5034018215165458848008139495523489874.03376336096645373740584643854077', '5034018215165458848008139495523489874.03376336096645373740584643854076962907123850798576272013744814476973662198925043205770660009312988310052082199046173230090038611836383812580300432749265926780857416932980603220096610051561', 70, 5);
    t('83465116483633329469892035899051214304833627081502534592536093571062537022461983038812804997332624606326980616761922998108321600000000000000000000000000000000000000000000000000000000000000000000000000', '83465116483633329469892035899051214304833627081502534592536093571062537022461983038812804997332624606326980616761922998108321680358559141400636855443060787356675791738187060632977884989818609747285992.4573', 126, 1);
    t('412026414399525071650594164923833936844850000000000000000000000000000000000000000000000000000000000000000000000', '412026414399525071650594164923833936844847273338573087154708906358381953581099486810466578558024416943275309299.1547285762679429823393431675351533167176328000593466154043859014661271664839568100113702022246192745211768782431699764391316694208124611848708840012780858930258789485729938310307149660630436665693341162045323153311070630675672019254696', 41, 0);
    t('24184281511162954354065479822988787658082488726029230964332108040623402309561801834624558861814492790548942373623472968824705013783170041934277445860710235574.9469863724343351110285675726533981998753264470394929595706054654539', '24184281511162954354065479822988787658082488726029230964332108040623402309561801834624558861814492790548942373623472968824705013783170041934277445860710235574.9469863724343351110285675726533981998753264470394929595706054654538860048', 225, 6);
    t('6528437875892691976548211518.24685079120011734820004207086140979168460245178917826434371997305141022847756041431375899105325', '6528437875892691976548211518.24685079120011734820004207086140979168460245178917826434371997305141022847756041431375899105324919785163855396689289511220736318428730997310359267318200728995651261553931749829029709309507777494337592244025093149733053834434331483018149604299765767492651748934740409204987337649', 124, 0);
    t('54840723345518316839235479.66298142041328062300439459932019790214795159930329911069342913247303369789405040373951962555445126692221316169309551520966435184986056649962041208952112820340474643', '54840723345518316839235479.662981420413280623004394599320197902147951599303299110693429132473033697894050403739519625554451266922213161693095515209664351849860566499620412089521128203404746436311696539001017796280457567020755477077608467790258', 190, 1);
    t('807991788710800290255468609318490845294840702742890475823369213502138441000000000000000000000000000000000000000', '807991788710800290255468609318490845294840702742890475823369213502138441473090507453171912732042988153868992514.06531203192016922009332409425555004520154760559043270547023041152076671', 72, 5);
    t('31188584966303127381503115718940547032615250764733945646803033.16801497739', '31188584966303127381503115718940547032615250764733945646803033.1680149773855979794823927265756991', 73, 0);
    t('8775400000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '8775466085414543073779209543057574932452788824441566732084759573870458439888006373669780377114235514168393104587.019325002484995191289455714', 5, 1);
    t('288.800623027304134541338806431884837474405810890016831959644312513765093874624940672214516819509051642914533844179037022375084617611170562223478541026325106635595681179195549628753278792341609852897791218125111473098120032', '288.8006230273041345413388064318848374744058108900168319596443125137650938746249406722145168195090516429145338441790370223750846176111705622234785410263251066355956811791955496287532787923416098528977912181251114730981200319638331244143925', 222, 5);
    t('78664117485196.26080145473732434147776948936075389060695497936755618934575601433579013586199090597827627898197174878630805725226155165823', '78664117485196.26080145473732434147776948936075389060695497936755618934575601433579013586199090597827627898197174878630805725226155165822379561335273668245708676272217985538163123', 136, 0);
    t('226652668776510356745673128804092024522000000000000000000000000000000000000000000000000', '226652668776510356745673128804092024522212860684406140705936036730300924742664473214634.52519567227832436168798010', 39, 6);
    t('76187726770000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '76187726770146714488473861067022637329338457054509204141149168751402208540667602830040870679811569378796817886578761925303158089285033456479051240199551952705477882146489966944458966504563022491268963016079717897383268.591', 11, 4);
    t('107418863393352656205587963420617762313433063250939036826347847138912512868978.7959698779979939004739234639552557309749', '107418863393352656205587963420617762313433063250939036826347847138912512868978.7959698779979939004739234639552557309749212088149355059075402582354234084555841458054540285104916739910424854709109514071370587697', 118, 5);
    t('46339929840600000000000000000000000000000000000000', '46339929840667602784728776298829187576269703792020.352902344501962851261072596846904654878638617857268484406071133983152462924725888787992422455089066687994341012982776010283632963474546763031928', 12, 3);
    t('73803831740084947712932480718.370453064080575630834065304188259601541977153801881194314571779722633346527595704401012668393515351684762424578563688815472177946816315358243895619101702075674431002147968', '73803831740084947712932480718.370453064080575630834065304188259601541977153801881194314571779722633346527595704401012668393515351684762424578563688815472177946816315358243895619101702075674431002147968832970406914274961841250650880349934068042222932460698863781', 200, 1);
    t('664762307274158434513912466237481165672508817932493794897940709800245416267908117128830202171811643985992038662932184499802408840000000000000000000000000000000000000000000000000000', '664762307274158434513912466237481165672508817932493794897940709800245416267908117128830202171811643985992038662932184499802408841446628928071016814941949373673441380688986398497568.43696334563468250818484292617449396278175160453902853150', 128, 4);
    t('684833382705735000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '684833382705735366022021221610718448812717565083645986755871173722416348980364620368092725272043436079122766673599371083360089295398781829495316124306915326031112050555020847105404848794.98064765927538509032711705003415532921697928012044422160394647448908486462274235247022', 15, 6);
    t('9523229252273031213852937756883030194662367574461969565997190423703233657823225547332912431.514691127079022744903160530559286297010362179', '9523229252273031213852937756883030194662367574461969565997190423703233657823225547332912431.514691127079022744903160530559286297010362178661650490970407522724257841414861815868919326764201126359001580385345573229592122255584580886424680678153939255741430656682245647086228115243811612563779', 136, 4);
    t('957831069598107843372326791377060065755846828520806962625151646870671403533795317859319291342905278879700156276715199551003072985800479935151161192140981640759273421130000000000000000000000000000000000000000000', '957831069598107843372326791377060065755846828520806962625151646870671403533795317859319291342905278879700156276715199551003072985800479935151161192140981640759273421129195622199875565188307072280623467856272709.362938513', 167, 6);
    t('324924987300039452802549776686212440241186995269477898200000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '324924987300039452802549776686212440241186995269477898200722764400467113722125079520434677584795409815119498076231384334164933319432675909257132574107959724074586678067352857024901866950876577019513.919205653046862675275248325864245', 56, 4);
    t('122284171543071259543194240460241623184291087513673279693486586539998.617849876080087217050408744664722357774977792735523', '122284171543071259543194240460241623184291087513673279693486586539998.61784987608008721705040874466472235777497779273552253234651728804467359989592745627535553091765468181193044358450735727092243874350274266033514954939483859885893259401477497220054315695357898954835009122504585299151237752022783062337881546179643668179570', 120, 5);
    t('31722554813771566030041180344518485971525788776654774133354709714081109144314851.7725714445172811762226132704524627132148051705216356556019479045049094117100399980050356484474139670618660114', '31722554813771566030041180344518485971525788776654774133354709714081109144314851.7725714445172811762226132704524627132148051705216356556019479045049094117100399980050356484474139670618660113755899876759', 189, 6);
    t('18648373828695525304964840000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000', '18648373828695525304964844613132761026424800339722780396244376541040426090070552337115611235624961988104587372507645741474248738.556020613780050801', 25, 3);
    t('5760443204443416175879517790929784554812.36470988298773171937307412285', '5760443204443416175879517790929784554812.3647098829877317193730741228497915221108498170497521469016750756728060010580160520067424596234553800691202614552525060835882040102493247095790403438233017963635842839571283813', 69, 5);
    t('7930371537197228431503600000000000000000000000000000000000000000000000000000', '7930371537197228431503622323003269145590901915043964968929950673803457111513.6', 23, 5);
    t('194927704759716999737952637950749467657827600', '194927704759716999737952637950749467657827559.919403770840822175972786017803199068405043178810250571694416212858918272496519', 43, 6);
    t('366763985061296627375995000000000000000000000000000000000000000', '366763985061296627375994988520498401354427141647683539962362297.823510352274776652694253563878392985469220455118158824859937764475089228072088702703863070436530913536177584336650185707427759022894822571', 24, 5);
    t('7483978023392620879223025348065115057653025858013483463606682807652114675009756211425640344986476725604422263784508630866327980720727327697183007274924845457864758981447718084335691255091414929451655538943143277846.270484986861', '7483978023392620879223025348065115057653025858013483463606682807652114675009756211425640344986476725604422263784508630866327980720727327697183007274924845457864758981447718084335691255091414929451655538943143277846.270484986861458493444574936108272', 226, 5);

    Test.isException(function () {new BigNumber('12.345').precision(NaN)}, ".precision(NaN)");
    Test.isException(function () {new BigNumber('12.345').precision('NaN')}, ".precision('NaN')");
    Test.isException(function () {new BigNumber('12.345').precision([])}, ".precision([])");
    Test.isException(function () {new BigNumber('12.345').precision({})}, ".precision({})");
    Test.isException(function () {new BigNumber('12.345').precision('')}, ".precision('')");
    Test.isException(function () {new BigNumber('12.345').precision(' ')}, ".precision(' ')");
    Test.isException(function () {new BigNumber('12.345').precision('hello')}, ".precision('hello')");
    Test.isException(function () {new BigNumber('12.345').precision('\t')}, ".precision('\t')");
    Test.isException(function () {new BigNumber('12.345').precision(new Date)}, ".precision(new Date)");
    Test.isException(function () {new BigNumber('12.345').precision(new RegExp)}, ".precision(new RegExp)");
    Test.isException(function () {new BigNumber('12.345').precision(7.5)}, ".precision(7.5)");
    Test.isException(function () {new BigNumber('12.345').precision('-1.1e1')}, ".precision('-1.1e1')");
    Test.isException(function () {new BigNumber('12.345').precision(0, 1)}, ".precision(0, 1)");
    Test.isException(function () {new BigNumber('12.345').precision('0')}, ".precision('0')");
    Test.isException(function () {new BigNumber('12.345').precision('-1')}, ".precision('-1')");
    Test.isException(function () {new BigNumber('12.345').precision(-23)}, ".precision(-23)");
    Test.isException(function () {new BigNumber('12.345').precision(MAX + 1)}, ".precision(MAX + 1)");
    Test.isException(function () {new BigNumber('12.345').precision(MAX + 0.1)}, ".precision(MAX + 0.1)");
    Test.isException(function () {new BigNumber('12.345').precision('-0.01')}, ".precision('-0.01')");
    Test.isException(function () {new BigNumber('12.345').precision('-1e-1')}, ".precision('-1e-1')");
    Test.isException(function () {new BigNumber('12.345').precision(Infinity)}, ".precision(Infinity)");
    Test.isException(function () {new BigNumber('12.345').precision('-Infinity')}, ".precision('-Infinity')");

    t('12.3', '12.345', 3);
    t('12.34567891', '12.3456789123456789', 10);

    Test.isException(function () {new BigNumber('12.345').precision(1, NaN)}, ".precision(1, NaN)");
    Test.isException(function () {new BigNumber('12.345').precision(1, 'NaN')}, ".precision(1, 'NaN')");
    Test.isException(function () {new BigNumber('12.345').precision(1, [])}, ".precision(1, [])");
    Test.isException(function () {new BigNumber('12.345').precision(1, {})}, ".precision(1, {})");
    Test.isException(function () {new BigNumber('12.345').precision(1, '')}, ".precision(1, '')");
    Test.isException(function () {new BigNumber('12.345').precision(1, ' ')}, ".precision(1, ' ')");
    Test.isException(function () {new BigNumber('12.345').precision(1, 'hello')}, ".precision(1, 'hello')");
    Test.isException(function () {new BigNumber('12.345').precision(1, '\t')}, ".precision(1, '\t')");
    Test.isException(function () {new BigNumber('12.345').precision(1, new Date)}, ".precision(1, new Date)");
    Test.isException(function () {new BigNumber('12.345').precision(1, new RegExp)}, ".precision(1, new RegExp)");
    Test.isException(function () {new BigNumber('12.345').precision(1, 7.5)}, ".precision(1, 7.5)");
    Test.isException(function () {new BigNumber('12.345').precision(1, 9)}, ".precision(1, 9)");
    Test.isException(function () {new BigNumber('12.345').precision(1, '-1.1e1')}, ".precision(1, '-1.1e1')");
    Test.isException(function () {new BigNumber('12.345').precision(1, '-1')}, ".precision(1, '-1')");
    Test.isException(function () {new BigNumber('12.345').precision(1, -23)}, ".precision(1, -23)");
    Test.isException(function () {new BigNumber('12.345').precision(1, MAX + 1)}, ".precision(1, MAX + 1)");
    Test.isException(function () {new BigNumber('12.345').precision(1, MAX + 0.1)}, ".precision(1, MAX + 0.1)");
    Test.isException(function () {new BigNumber('12.345').precision(1, '-0.01')}, ".precision(1, '-0.01')");
    Test.isException(function () {new BigNumber('12.345').precision(1, '-1e-1')}, ".precision(1, '-1e-1')");
    Test.isException(function () {new BigNumber('12.345').precision(1, Infinity)}, ".precision(1, Infinity)");
    Test.isException(function () {new BigNumber('12.345').precision(1, '-Infinity')}, ".precision(1, '-Infinity')");

    BigNumber.config({ EXPONENTIAL_AT: 0});

    t('-7.66e+6070140', '-7.66E+6070140', 4, 0);
    t('-7.9674989422401853010973497105652818e-8963939', '-7.9674989422401853010973497105652818966315450635741064116E-8963939', 35, 1);
    t('-4.06442845741835e+4', '-40644.2845741834341373016286331077593043129764966', 15, 0);
    t('4.09025241411015488e+717', '4.0902524141101548818266218311183397635475335E+717', 18, 4);
    t('7.36428e+190072', '7.36428E+190072', 7, 3);
    t('-3e+3081875', '-3.267981E+3081875', 1, 6);
    t('-1.23124161515264434734535288146151136598822e+9395349', '-1.23124161515264434734535288146151136598821988E+9395349', 42, 0);
    t('-4.4983972008593697e+44878', '-4.4983972008593696209789148459E+44878', 17, 0);
    t('-4.1375e+3', '-4137.49217', 6, 3);
    t('-8.95980419651e+22886', '-8.95980419650920346716242088426056438478151380E+22886', 12, 4);
    t('9.2026300320824367e+44', '9.2026300320824367E+44', 17, 0);
    t('-1.64105581e+91', '-1.64105581E+91', 9, 0);
    t('1.98291451081e+74', '1.9829145108144E+74', 12, 5);
    t('7.6348532140506e-80910', '7.6348532140506E-80910', 17, 6);
    t('9.059e+5218', '9.05937196874816625610E+5218', 4, 5);
    t('9.23e+5908668', '9.228916335834469051E+5908668', 3, 5);
    t('5.44543328367417165611416e-3866', '5.4454332836741716561141541766451000323105927150236E-3866', 24, 0);
    t('3.32201381681300432698705473e+6515', '3.3220138168130043269870547397294958572352372E+6515', 27, 1);
    t('5.3543068140922848122399446448033e+53771', '5.3543068140922848122399446448033E+53771', 34, 6);
    t('-7.608284024e-8356', '-7.6082840240E-8356', 10, 5);
    t('-1.100872801064263741678e+464181', '-1.10087280106426374167810621907E+464181', 22, 6);
    t('-3.2473379e+7279', '-3.24733794E+7279', 8, 5);
    t('-2.545908227379844801445331551443e+433', '-2.545908227379844801445331551442699203986416069185920937E+433', 31, 0);
    t('-3.43002030696946e+10799', '-3.43002030696945751808E+10799', 15, 3);
    t('4.745483222968536138779919876e+9993', '4.745483222968536138779919876E+9993', 28, 1);
    t('-3.43052070592372e-56099', '-3.430520705923719555682478758967295244174056812478480256E-56099', 16, 6);
    t('2.3e+8', '2.2053928E+8', 2, 0);
    t('1.919e+0', '1.918722775040248055932652325', 4, 2);
    t('-4.38245617493930057658036e+5410', '-4.3824561749393005765803605139961715E+5410', 25, 2);
    t('4.4388936844084512832090453337825382985888008689e-9017', '4.4388936844084512832090453337825382985888008689027793412589E-9017', 48, 5);
    t('-6.931278521e-51', '-6.9312785210246E-51', 11, 6);
    t('2.98337650636741163327150135558127286071425633112e+272652', '2.983376506367411633271501355581272860714256331120005363E+272652', 50, 3);
    t('-8.566874804756e+5506', '-8.566874804756E+5506', 14, 4);
    t('9.9691915e+9973', '9.9691914630694767701986871E+9973', 8, 0);
    t('1.8856995929e-575', '1.885699592832896413671E-575', 11, 2);
    t('-4.407661624e+2726968', '-4.40766162467365502E+2726968', 10, 1);
    t('1.202e+5410', '1.20199212066707E+5410', 4, 0);
    t('1.3465968471523e+190', '1.346596847152251544417550241163997602190526752298467158944E+190', 14, 0);
    t('-4.6844572027253e+614', '-4.68445720272534591309596295269774329198700E+614', 14, 1);
    t('8.643477444e+1', '86.4347744392225110929', 10, 6);
    t('1.21e+532425', '1.210E+532425', 7, 4);
    t('4.6e+1795', '4.5136583697101494149760944E+1795', 2, 2);
    t('3.608417010945906135e-1', '0.3608417010945906135088', 19, 5);
    t('4.408469287631117363108958479986697077756e+396', '4.40846928763111736310895847998669707775584E+396', 40, 0);
    t('5.6498469401e-307', '5.6498469401806184596045354572096752707425481768E-307', 11, 3);
    t('-8.855333399e+1142', '-8.85533339893608E+1142', 10, 4);
    t('-6e+101', '-6E+101', 4, 0);
    t('-1.113466906796262619251424082293519197801557945525e-8035', '-1.11346690679626261925142408229351919780155794552503675E-8035', 50, 1);
    t('6.0351e+1', '60.350932000765743765948', 5, 0);
    t('-4.2243657735450142974712525032448e+115042', '-4.2243657735450142974712525032447594337811580E+115042', 32, 0);
    t('4.0737423e-289799', '4.07374228747738429354308050E-289799', 8, 0);
    t('6.7142168606150075836935902129e+7', '67142168.6061500758369359021289968337706', 30, 2);
    t('-2.06e-48', '-2.06E-48', 6, 1);
    t('-4.1406537826747838501847227e+1', '-41.40653782674783850184722733775173400593481597880791974103', 26, 1);
    t('-5.2344941324311479774252677816127451347e+5', '-523449.41324311479774252677816127451346204366670', 38, 3);
    t('-3.723419790392802480440325694999142764e+976263', '-3.7234197903928024804403256949991427646881230189E+976263', 37, 1);
    t('-7e+7414', '-7E+7414', 4, 5);
    t('1.626202703780595e+15', '1626202703780594.49462802780387359712846275172328755998954792', 16, 2);
    t('8.32946093792427993586778885872e+68249', '8.329460937924279935867788858724032019E+68249', 30, 6);
    t('2.86651842e+9921', '2.86651841708239170240815444449258748921E+9921', 9, 6);
    t('2.611365177e+4757', '2.6113651770206670771421106822690209141654E+4757', 11, 1);
    t('-3e+2207', '-2.31680484E+2207', 1, 3);
    t('-6.0652750450609601468080954656174019077172561075841612509e+378', '-6.0652750450609601468080954656174019077172561075841612509E+378', 57, 4);
    t('-4.51e-42886', '-4.506454671E-42886', 3, 5);
    t('-2.94324e+64', '-2.94324E+64', 7, 1);
    t('-8.096437128422e+43', '-8.096437128422E+43', 16, 3);
    t('3.6e+903756', '3.595941214E+903756', 2, 5);
    t('-8.493214e+646', '-8.493213972482556638199288267891E+646', 8, 4);
    t('-9.321586e+43570', '-9.32158562195556489258044E+43570', 7, 5);
    t('-2.214635013209782e+37', '-2.21463501320978194425416356E+37', 17, 3);
    t('8.5e+346', '8.4521609612906797429879512441743844780079E+346', 2, 4);
    t('3.394294e-83527', '3.394294E-83527', 8, 6);
    t('-5.790809e+858032', '-5.7908084785816319987683035704990E+858032', 7, 3);
    t('-4.3809800223050634e+16356', '-4.3809800223050633180390529847313003224214494464288E+16356', 17, 3);
    t('-7.71478248344912760438270191805727030659073e-1', '-0.771478248344912760438270191805727030659073291', 42, 4);
    t('8.9473411956136952863603e+0', '8.9473411956136952863602873002842938153248244520287677768', 23, 6);
    t('3.98512165e-406', '3.98512165E-406', 9, 2);
    t('-3.3679e+4748', '-3.3679000E+4748', 9, 5);
    t('-5.7950879061744896240195549309351e+6745', '-5.79508790617448962401955493093511584E+6745', 32, 4);
    t('6.3012025838e+6565', '6.301202583767076815074413E+6565', 11, 5);
    t('1.3401461557642378095953194598208e+621371', '1.3401461557642378095953194598207610667180E+621371', 32, 4);
    t('-5.38685005685513084177264173254e-5', '-0.0000538685005685513084177264173253967376083861788045831495495483', 30, 4);
    t('-6.64e-55', '-6.64E-55', 5, 1);
    t('-4.53119445193802e+1497', '-4.5311944519380192081604E+1497', 15, 6);
    t('2e+19637', '2.177620111E+19637', 1, 1);
    t('-1.82250309497649679310085e+6', '-1822503.09497649679310085232806332467204', 24, 2);
    t('7.634005607e+50', '7.63400560612E+50', 10, 0);
    t('-1.048e+8', '-1.047785E+8', 4, 4);
    t('-7e-5176057', '-6.54E-5176057', 1, 6);
    t('6.770016207269424951e+1250', '6.770016207269424951700025480165903296857062476013255E+1250', 19, 3);
    t('2.7e+175', '2.68838599432180398822019088089980012453852436837612475555E+175', 2, 0);
    t('-1.34e+892', '-1.33935704824554293883E+892', 4, 0);
    t('1.4688699e-5085', '1.4688699E-5085', 11, 3);
    t('5.978956262004798481009688482e+0', '5.97895626200479848100968848162368542', 28, 0);
    t('9.286975e+4703178', '9.286975E+4703178', 8, 1);
    t('-2.704929748188632263993259177116e+699506', '-2.70492974818863226399325917711547567029283135771051086455E+699506', 31, 0);
    t('6.2180638313738e-5832', '6.2180638313738283397154672139E-5832', 14, 6);
    t('5.6882354157817323408190154291872e+0', '5.688235415781732340819015429187247405275796993462668778', 32, 5);
    t('-8.8438040088488221e+46', '-88438040088488220224564882589653601905847261849', 17, 3);
    t('6.041429451633647195846e+625', '6.041429451633647195846E+625', 25, 1);
    t('6.915234e+534', '6.9152339796380626E+534', 7, 2);
    t('8.88127e-52248', '8.8812797094076919346602414910E-52248', 6, 1);
    t('8.43464364312791528837682017827958746027908893605e-744', '8.4346436431279152883768201782795874602790889360584608841238E-744', 48, 1);
    t('-2.15e+755952', '-2.14313139E+755952', 3, 3);
    t('7.693493611e+7386', '7.6934936114114902537748379043E+7386', 10, 5);
    t('4.733710060196692e+95368', '4.733710060196692075936228E+95368', 16, 6);
    t('4.666777246136556e+6', '4666777.24613655632469634924609210636778826726380354', 16, 6);
    t('2.712452957e+78', '2.712452956152588934925903E+78', 10, 2);
    t('-9.13335846961498158373253379e+982184', '-9.13335846961498158373253378807256368527276426E+982184', 27, 5);
    t('5.2855634628134e+92', '5.28556346281336612826959543809286E+92', 14, 4);
    t('-4.464508151077081e+56', '-4.464508151077080660609205938E+56', 16, 5);
    t('-1.247443292597675571926985586e+99', '-1.247443292597675571926985586E+99', 29, 2);
    t('9.6e+3', '9655.37921092371791714258918059279097354', 2, 1);
    t('-4.270758863850020793238607781950702086699177710655536971164e+95159', '-4.270758863850020793238607781950702086699177710655536971164E+95159', 59, 1);
    t('7.480547833670792e+7062', '7.48054783367079219581590748175836330E+7062', 16, 3);
    t('-3.8867365998135819452509619e-384353', '-3.886736599813581945250961910244083596880388954E-384353', 26, 4);
    t('5.189910195483756653e+286046', '5.189910195483756652610448704129539266E+286046', 19, 0);
    t('-1.10944e+3104234', '-1.1094464786909778116768534008366316470680778108338936E+3104234', 6, 2);
    t('-2.73725098713116601853714691783947615e+4402378', '-2.73725098713116601853714691783947615123192E+4402378', 36, 5);
    t('3.0733361501920979424e-34076', '3.0733361501920979424E-34076', 23, 6);
    t('-9.08241e+593935', '-9.08240642417562290132127650738600E+593935', 6, 3);
    t('6.9102192372884348128663475e+888925', '6.910219237288434812866347586E+888925', 26, 3);
    t('7.2630276255596214848420346385268351395048896649e+93654', '7.26302762555962148484203463852683513950488966490022943594866E+93654', 49, 3);
    t('2.285667632707481655489750735437552177542e-176543', '2.2856676327074816554897507354375521775418E-176543', 40, 0);
    t('4.7707364648929253418e-9220014', '4.7707364648929253418409217972963362424373E-9220014', 20, 5);
    t('6.23233617494703423e+90667', '6.232336174947034232282802373419465608630127175426076E+90667', 18, 3);
    t('6.44069e-19', '6.44068374265737505445920653236475590371753936931116E-19', 6, 2);
    t('-6.5e+6', '-6572480.030064718776761', 2, 1);
    t('-6.3836111e-44', '-6.383611100193E-44', 10, 4);
    t('-1.87458e-22', '-1.874576864398001461019153755602408406903448992468673822216E-22', 6, 4);
    t('-5.88843833e+798916', '-5.88843833440456E+798916', 9, 2);
    t('4.112293400431443026367344957439e-65410', '4.112293400431443026367344957438568896666205943E-65410', 31, 6);
    t('9.102374577555726095242038544771699729168e-244971', '9.102374577555726095242038544771699729167210E-244971', 40, 2);
    t('5.94028510371395456e+622110', '5.940285103713954551626294515849E+622110', 18, 0);
    t('4.18666602418926623029340813935e+895', '4.18666602418926623029340813935227050340057377941936317362559E+895', 30, 3);
    t('2.32242583195962062384e+2', '232.242583195962062383193', 21, 0);
    t('-2.47458985736996328194365099545069e+2', '-247.4589857369963281943650995450694527946366843', 33, 1);
    t('6.126631571e+9589', '6.1266315710232234E+9589', 10, 3);
    t('8.243601188412475518224553720463561663805655509347003e+679', '8.2436011884124755182245537204635616638056555093470029E+679', 52, 5);
    t('2.37694029243227408727012419862589389e+2080695', '2.376940292432274087270124198625893881468999565978253E+2080695', 36, 0);
    t('8e+77967', '7.285864474E+77967', 1, 0);
    t('-7e-75', '-7E-75', 1, 1);
    t('8.2987748868553e+3866504', '8.2987748868553E+3866504', 17, 5);
    t('1.3742663357466387e+398879', '1.3742663357466387E+398879', 17, 0);
    t('-6.87856066456573788e+4', '-68785.6066456573787511048192214619169366348896006', 18, 3);
    t('-8.092986361732e+746', '-8.0929863617323016314E+746', 13, 1);
    t('2.04313029494e-136', '2.043130294941755E-136', 12, 3);
    t('-3.678395571013e+4943080', '-3.67839557101385225147047122741279233373656005947123539345E+4943080', 13, 2);
    t('6.8270900273e+314', '6.82709002733121035E+314', 11, 4);
    t('9.754e+77', '9.754645017028710911408494910830742114195979114753282913263E+77', 4, 1);
    t('4.573256499048099560424985704491138414670385322887955e+8964', '4.573256499048099560424985704491138414670385322887955E+8964', 52, 5);
    t('-5.20641765e+5370015', '-5.206417654581211249181962648679978613023900E+5370015', 9, 4);
    t('-1.495810569652186870608983e+9109', '-1.49581056965218687060898341393E+9109', 25, 1);
    t('-3.9277982906082234684384255881242632776333672900668629285e+6042', '-3.92779829060822346843842558812426327763336729006686292846E+6042', 56, 6);
    t('-8e+5189', '-8E+5189', 1, 3);
    t('-4.010117191647555937588208084600101678e+9332', '-4.010117191647555937588208084600101678738352E+9332', 37, 1);
    t('6.3e+22206', '6.30585027395050718024910615679322453436994539E+22206', 2, 1);
    t('-7.8442e+2243460', '-7.8442E+2243460', 8, 0);
    t('-8.5687833331755e+62', '-8.568783333175528911E+62', 14, 1);
    t('-4.71e+25325', '-4.701E+25325', 3, 0);
    t('-9.8e-695', '-9.8E-695', 4, 6);
    t('-9.95089286211712788e+4', '-99508.92862117127875269189726705754650024096803723101765', 18, 6);
    t('1.5e+8431166', '1.4588286163E+8431166', 2, 6);
    t('3.25698256850983242002097559269e+655', '3.2569825685098324200209755926865165412882632395635543E+655', 30, 5);
    t('4e+1387', '4.02017775978978244E+1387', 1, 3);
    t('1.625314e+6079', '1.6253141185267727641E+6079', 7, 4);
    t('-4.29746293e+76502', '-4.297462939748E+76502', 9, 2);
    t('-3.7635270830697521572358550073548e+48350', '-3.7635270830697521572358550073548711E+48350', 32, 1);
    t('1.886474913643e-2', '0.0188647491364370859', 13, 3);
    t('-5.42795648e+967', '-5.42795647996830766229953531992969498121079633174394E+967', 10, 4);
    t('-9.6e+29170', '-9.6E+29170', 5, 0);
    t('-4.7248348129493958e+665', '-4.72483481294939574286E+665', 17, 3);
    t('3.89540565472e+747', '3.89540565471814305720029596702393663941E+747', 12, 5);
    t('-7.974e+44', '-7.9740614142591147928185750261917057279246E+44', 4, 6);
    t('3.513618576137939e+2611534', '3.513618576137939E+2611534', 17, 3);
    t('-9.911e+4867168', '-9.91016916746601216674376858319753150305E+4867168', 4, 3);
    t('-6.348510493418474517923631621064599e-5', '-0.0000634851049341847451792363162106459910953016825272', 34, 1);
    t('1.2427966894898043295108e+9', '1242796689.4898043295108', 23, 1);
    t('8.1143598457862025513376231533681504672565903897e+7838', '8.11435984578620255133762315336815046725659038969418243205618E+7838', 48, 0);
    t('-7.583e+58189', '-7.58288768457471610251E+58189', 4, 3);
    t('1e+24332', '9.4880080489873655448903883496039395722932239390E+24331', 1, 2);
    t('-3.5165477093e+8808', '-3.51654770924421026710533044562645488595988456289E+8808', 11, 0);
    t('8.9193408e-8', '8.9193408E-8', 10, 0);
    t('6.7627931443586183163993663907626445829e+4325', '6.76279314435861831639936639076264458293338543157439E+4325', 38, 4);
    t('-5e-9070409', '-4.08457270E-9070409', 1, 3);
    t('-5e+9244071', '-5.6266301487E+9244071', 1, 1);
    t('-7.2e-6206718', '-7.2E-6206718', 5, 6);
    t('1.60594726740667199920052025419431412065792e+6', '1605947.26740667199920052025419431412065791689232852870404543', 42, 5);
    t('-3.2013e+619', '-3.2012975378579830605832754949307752402269325903512212471637E+619', 6, 5);
    t('7.5961116814783999035997653e+202', '7.59611168147839990359976538492E+202', 26, 1);
    t('-2.81585040663400229908788936485913315780802321e+3683', '-2.81585040663400229908788936485913315780802321E+3683', 48, 3);
    t('3.7e+671', '3.681142E+671', 2, 0);
    t('-6.109e+1953', '-6.109336176410647968173711511541006722000536743261980088381E+1953', 4, 6);
    t('-3.2e+585322', '-3.2811E+585322', 2, 1);
    t('-8.3586461873575368924060096e+434', '-8.3586461873575368924060095121085462201212314038E+434', 26, 0);
    t('1.996314e+66', '1.99631392959152E+66', 7, 4);
    t('-9.28296465e+2125', '-9.28296464790006743061E+2125', 9, 0);
    t('1.19e+2', '119.3', 3, 6);
    t('6.509993372544e+282536', '6.50999337254445577733E+282536', 13, 6);
    t('5.484128477071997414159230299105483265835e-12505', '5.4841284770719974141592302991054832658358740358544461250047E-12505', 40, 1);
    t('-4.4714776e+143', '-4.4714775705954021348E+143', 8, 6);
    t('-9.57e+8622646', '-9.56835534E+8622646', 3, 0);
    t('-6.33117454850645697739696509448025e+151', '-6.33117454850645697739696509448024089E+151', 33, 3);
    t('-7.137390398428604e+2979', '-7.1373903984286037254E+2979', 16, 3);
    t('-7e-7205', '-7.85416945008792766125264836461E-7205', 1, 1);
    t('-5.6015179879e+421087', '-5.601517987898113130622933E+421087', 11, 4);
    t('4.1966e+858', '4.1966403617302590759E+858', 5, 3);
    t('-9.2792013968515105549791151e+32', '-927920139685151055497911512658558.34599782687535149908', 26, 1);
    t('5.09e-5', '0.000050899243', 3, 0);
    t('-7.506554295028719229649058531685e-72979', '-7.5065542950287192296490585316846890797440889669116005081E-72979', 31, 4);
    t('-9.1792836314359860387882450320865042650485329957164e+4284', '-9.1792836314359860387882450320865042650485329957164E+4284', 51, 6);
    t('-4.2933216429417509294224189998249488475743327e-1397', '-4.2933216429417509294224189998249488475743327E-1397', 47, 6);
    t('9e-216637', '9.087924216118133862E-216637', 2, 1);
    t('3.1702930329914108e+55429', '3.17029303299141087080069647726603E+55429', 17, 1);
    t('2.2936e-8938210', '2.29355733289074040471E-8938210', 5, 4);
    t('-6.113e-1298', '-6.1130E-1298', 8, 2);
    t('-6.34387679e+31476', '-6.34387679E+31476', 12, 2);
    t('-8.9359311686840016e-629113', '-8.93593116868400158072549818891784774605798568745492849386044E-629113', 17, 3);
    t('6.1335811157648260984882e+88591', '6.1335811157648260984882078508802E+88591', 23, 3);
    t('-1.5853e+2964', '-1.585277888545693341591017969569681715196073125544E+2964', 5, 4);
    t('4.38433059e+6150', '4.384330590864683698711811528E+6150', 9, 1);
    t('-2.7532e+0', '-2.753145085267', 5, 0);
    t('-2.9055e+864321', '-2.9055E+864321', 6, 2);
    t('9.928e+806', '9.927271E+806', 4, 0);
    t('-9.709860045777352807038e-9704424', '-9.709860045777352807037742E-9704424', 22, 0);
    t('9e-17', '8.9241340198E-17', 1, 5);
    t('-6.074043984814937e-7673816', '-6.07404398481493781E-7673816', 16, 1);
    t('-1.7451928023043557432512219212511306e+2', '-174.5192802304355743251221921251130525020603766268688', 35, 0);
    t('-3.5e+7916169', '-3.4079876387367E+7916169', 2, 0);
    t('-8.3e+7857411', '-8.30515102364E+7857411', 3, 2);
    t('-3.8767426109515175e+2', '-387.67426109515175', 19, 5);
    t('1.913e+1764', '1.913E+1764', 5, 4);
    t('-7e-956710', '-6.974387278866782555350536251455442500E-956710', 1, 4);
    t('1.8527e+972', '1.8527E+972', 5, 2);
    t('9.782073215494517877298968126249468882305366059003e-95486', '9.7820732154945178772989681262494688823053660590035E-95486', 49, 5);
    t('-5.4e+10', '-5.40964E+10', 2, 2);
    t('3.46246731369282e+5', '346246.7313692820', 19, 6);
    t('-4.807312501229969057464130082752557969824883948501e+26', '-480731250122996905746413008.2752557969824883948500294533', 49, 3);
    t('-3.61789987207897895e+0', '-3.6178998720789789585732082298256489650526511', 18, 2);
    t('5.456692e+9', '5.45669177E+9', 7, 2);
    t('8.75161144e+7', '87516114.40', 10, 3);
    t('8.3e+5583', '8.34829619433286213892616555798342184E+5583', 2, 3);
    t('4.978e+80745', '4.97813995093110784185700347476762967200972086691E+80745', 4, 6);
    t('-2.93894831062e+3', '-2938.94831062173', 12, 2);
    t('-8.66944e+8', '-866944038.98785814735172840487738630', 7, 6);
    t('-2.805654737004596806509496836889e+8458218', '-2.805654737004596806509496836889415462308364E+8458218', 31, 6);
    t('-7e+2332272', '-6.670267431319511014723460690562220039145649407982839E+2332272', 1, 6);
    t('-4.63e-18', '-4.632E-18', 3, 1);
    t('1.64e+1719', '1.6430039259283586E+1719', 3, 5);
    t('-1.054e+1595', '-1.05410006E+1595', 4, 6);
    t('-5.87237421659e+734609', '-5.87237421659537110291652280627288E+734609', 12, 2);
    t('9.006629642200129103836e-926', '9.006629642200129103836969765E-926', 22, 1);
    t('-2.454995026312844896337353435e+3', '-2454.9950263128448963373534355620898', 28, 1);
    t('3.610608e+3', '3610.608', 8, 6);
    t('-4.12727638e+301', '-4.12727638E+301', 12, 5);
    t('6.5e+544600', '6.5375397250148428442161727901E+544600', 2, 1);
    t('-2.665714352151038e-7513214', '-2.6657143521510388345129539902365040727904382618E-7513214', 16, 1);
    t('8.090648256244503574155400306029208465e-985453', '8.09064825624450357415540030602920846500467124904074850977222E-985453', 38, 1);
    t('9e-8689', '8.058E-8689', 1, 0);
    t('8.98192783175e+3', '8981.927831749', 12, 5);
    t('5.0283886601979836995462e+29046', '5.02838866019798369954617194896446689515E+29046', 23, 5);
    t('-9e+94', '-9.0793909314887E+94', 1, 2);
    t('6.34564893576e+9103075', '6.34564893576239555081318028768760122920E+9103075', 12, 5);
    t('5.9009478154776530110233e-98646', '5.900947815477653011023322251230439694800132E-98646', 23, 6);
    t('7.14274913993087665088554356877484e+283', '7.142749139930876650885543568774834E+283', 33, 0);
    t('6.69990179558e+715103', '6.699901795582163727E+715103', 12, 3);
    t('6.6714329715808808456063359819835587550928855e+79', '6.6714329715808808456063359819835587550928855E+79', 44, 1);
    t('-6.89587846377e+6', '-6895878.4637698847091', 13, 3);
    t('-6.720876397e-708', '-6.720876397954E-708', 10, 2);
    t('-2.98388450018745e+21033', '-2.983884500187449321232059E+21033', 15, 4);
    t('1.1722e+3736756', '1.172155564706E+3736756', 5, 0);
    t('7.3889552926469701945e+156749', '7.3889552926469701945E+156749', 20, 0);
    t('9.46243e+4333', '9.462438622127824E+4333', 6, 1);
    t('-3.6435e-383', '-3.643545388264045634009521901053203913747529055318E-383', 5, 1);
    t('6.2059028e+78538', '6.205902806487317324004137E+78538', 8, 4);
    t('8.53311e+522783', '8.53310693361292235649168133824816288322762977067506E+522783', 6, 0);
    t('-1.325e-271', '-1.3245043304808175640641614278354668611319976136702E-271', 4, 6);
    t('-3.7e+0', '-3.6500070832998551178945053002168', 2, 6);
    t('4.077e-3828', '4.0760303362617886144E-3828', 4, 2);
    t('-6.315e+3121', '-6.315569405934295E+3121', 4, 1);
    t('7.3003999048e+91480', '7.300399904745465175E+91480', 11, 0);
    t('-1.97550150359271408734265576e-8861', '-1.97550150359271408734265576E-8861', 27, 0);
    t('2.8881937515941659838372386870241678169190980954e+6', '2888193.75159416598383723868702416781691909809534891232', 47, 2);
    t('-2.504321861e+29', '-250432186102214810967305298086.418143246710499945284', 10, 6);
    t('-2.21911759837e+986', '-2.2191175983789229228004149122700E+986', 12, 1);
    t('-7e-60811', '-6.94598988120900497546719104132320760E-60811', 2, 3);
    t('8.72003351417e+2', '872.0033514170438001559913292452', 13, 3);
    t('-9.1149088123029e+431', '-9.1149088123028866605312109427765187127E+431', 14, 4);
    t('-2.6874e+7824205', '-2.6874E+7824205', 6, 5);
    t('9.105915e+14587', '9.105915179211403558E+14587', 7, 6);
    t('6.06983178718902662348950241823835301e+74512', '6.069831787189026623489502418238353012617904733832075E+74512', 36, 4);
    t('3.04700582e+8487714', '3.04700581715E+8487714', 9, 4);
    t('9.547296574550214629414645e+5294002', '9.54729657455021462941464498013501E+5294002', 26, 0);
    t('-3.8e+685034', '-3.8E+685034', 4, 2);
    t('2.2723223499344907627908e-9', '2.272322349934490762790877264508171629787937075E-9', 23, 3);
    t('4.559474879040682125518534893409431e+90', '4.559474879040682125518534893409430377698470E+90', 34, 0);
    t('-6.6082107891010044567552e+242479', '-6.6082107891010044567551500094056753067766206E+242479', 23, 6);
    t('-1.1982320835430829444212e-482020', '-1.19823208354308294442125105E-482020', 23, 1);
    t('9.96428864e+593', '9.964288644617348720259413133792E+593', 9, 1);
    t('-8.699e+0', '-8.69900304478038685809', 5, 4);
    t('-7.8325952072e+2286781', '-7.83259520711638926773253738848033322520757E+2286781', 11, 0);
    t('1.600715e-8762', '1.60071452E-8762', 7, 4);
    t('7.395011552033747581e+829882', '7.395011552033747580996314430051532018784631853731E+829882', 20, 5);
    t('-7.827376198663921586419479612519720808189e+3315831', '-7.82737619866392158641947961251972080818938423375E+3315831', 40, 2);
    t('1.5986616496817933978735e-5860', '1.5986616496817933978735E-5860', 26, 5);
    t('3.693780992432786510036232257e+290', '3.693780992432786510036232256781261999697773222979976334E+290', 28, 4);
    t('2.40046030272496787023668e+8914195', '2.4004603027249678702366897724617792E+8914195', 24, 3);
    t('-8.80967e+44', '-8.80966692924510586872015E+44', 6, 3);
    t('-9.060765170589856619538771265978e+5212031', '-9.06076517058985661953877126597763635913720606867609752526789E+5212031', 31, 6);
    t('-6.564e-16', '-6.564469061511952E-16', 4, 4);
    t('-1.94493375e+943716', '-1.94493375E+943716', 10, 3);
    t('-5.6094449426385906826652e+39', '-5.609444942638590682665118000237E+39', 23, 3);
    t('7.7954591118090523540122694554991395416092506658391032458e-8280', '7.79545911180905235401226945549913954160925066583910324579941E-8280', 56, 5);
    t('-4.03361529613234853138107968764041499007857758e+770', '-4.03361529613234853138107968764041499007857758E+770', 46, 4);
    t('9.35607862458e+5906', '9.3560786245755552696936564528928E+5906', 12, 0);
    t('4.56289370661472791146242008967730750837178199373832166e+32', '456289370661472791146242008967730.750837178199373832166', 57, 6);
    t('5.277e+78', '5.277058E+78', 5, 3);
    t('1.23860172285758646220297971394135e+1454', '1.23860172285758646220297971394134779958837369280E+1454', 33, 5);
    t('4.33660647884007637699270829910381723204312340399e+9998665', '4.33660647884007637699270829910381723204312340399E+9998665', 48, 0);
    t('1.7862961569785459834314e+1', '17.862961569785459834314', 26, 3);
    t('-1.48e-497838', '-1.48E-497838', 6, 5);
    t('-8.90413249105559261667616588522006187619e+4587', '-8.9041324910555926166761658852200618761952459783254308065E+4587', 39, 1);
    t('-5.02260283155e+8208', '-5.022602831557355931916007234717956095294607758558287155E+8208', 12, 1);
    t('-4.8611618232245352e+67006', '-4.8611618232245351474E+67006', 17, 0);
    t('3.072637476403189601331745441279482141716e-564', '3.072637476403189601331745441279482141715568162339E-564', 40, 0);
    t('5.02664605799684270099951604169317654972639165e-8848993', '5.026646057996842700999516041693176549726391650536877801652E-8848993', 45, 6);
    t('-8.07467098653129e-785', '-8.074670986531290E-785', 17, 2);
    t('7.19767276569168e+551875', '7.19767276569167928746515864362187284819325005219467E+551875', 15, 2);
    t('5e+4386655', '5E+4386655', 4, 4);
    t('2e+239', '1.9467211742070218E+239', 2, 0);
    t('-1.2058388225887357e+785', '-1.205838822588735741387785252769320912659712392987110367E+785', 17, 6);
    t('-6.4157699202496358214115e+502465', '-6.415769920249635821411514E+502465', 23, 5);
    t('1.2e+67', '1.11058776205121E+67', 2, 2);
    t('-8.1493429488729326217541646631e+185', '-8.14934294887293262175416466316742776702775E+185', 29, 1);
    t('-1.66874955040543874671736066613116684e+80545', '-1.66874955040543874671736066613116683988311377410792E+80545', 36, 6);
    t('-2.804847454602773057560513e+0', '-2.8048474546027730575605127', 25, 6);
    t('-3.1521e-2952', '-3.1521080821596942294464E-2952', 5, 1);
    t('3.689972530194980964771e+527895', '3.68997253019498096477115476909557370042685E+527895', 22, 4);
    t('8.910505326474732534078363449e+41', '8.910505326474732534078363449E+41', 31, 4);
    t('-9.91056e+175', '-9.910561334818E+175', 6, 4);
    t('2.2425e+7080', '2.24245525574527469867560065337399E+7080', 5, 6);
    t('-9.996088797356723446515682606e-8848', '-9.996088797356723446515682606657161359753640E-8848', 28, 1);
    t('5.66801218832321860843824655777255401588288486898444039e-72', '5.66801218832321860843824655777255401588288486898444039E-72', 57, 5);
    t('-3.4497949e+81294', '-3.4497949E+81294', 9, 6);
    t('-3.544302804e+215167', '-3.544302804234201383460668303870171E+215167', 10, 1);
    t('7.73448778e+3503361', '7.73448778760938462686545E+3503361', 9, 1);
    t('4.579876125e+40017', '4.57987612415299703146624253230E+40017', 10, 2);
    t('5e-4', '0.0005060', 1, 6);
    t('7.68e+6465110', '7.682E+6465110', 3, 3);
    t('-2.3e+59823', '-2.293161003728825144304719511028282408569E+59823', 2, 6);
    t('-6e-1918', '-5.9890E-1918', 2, 3);
    t('6.18424685387723766058154e-643', '6.18424685387723766058153878619367133970622727804507163945E-643', 24, 2);
    t('3.4883e+491', '3.48828156083928704549187622002859083630172428E+491', 5, 0);
    t('-5.11e+27198', '-5.11E+27198', 5, 3);
    t('-5.3342621389273488804773698318886661e+65968', '-5.334262138927348880477369831888666175128E+65968', 35, 1);
    t('-5.0538e-2', '-0.0505378613267352613415922806678', 5, 6);
    t('-9.356e+8641', '-9.356E+8641', 6, 1);
    t('3.24326e+7642', '3.24326E+7642', 6, 2);
    t('-6.68e+3200661', '-6.6759921328381958278730724001893404367E+3200661', 3, 5);
    t('7.104500894457089486e+586467', '7.1045008944570894863363847E+586467', 19, 1);
    t('-2.0795440346326703348643404347e-55871', '-2.079544034632670334864340434698174498306496919561E-55871', 30, 3);
    t('2.5334e-4991641', '2.53344911012184190694962E-4991641', 5, 6);
    t('4.95563737729233481703338301e-7791887', '4.9556373772923348170333830061570E-7791887', 27, 0);
    t('-5.146139971341569e-55569', '-5.1461399713415689429903976049E-55569', 16, 6);
    t('-6.0033605285200510840552013504866220952798e-3', '-0.00600336052852005108405520135048662209527973159298918212600', 41, 3);
    t('6.88005e+8799150', '6.88005091408130E+8799150', 6, 3);
    t('-8.18194e+4', '-81819.32892107713311267792590379858214280030439011411244', 6, 3);
    t('-3.590711347228753e+9', '-3590711347.22875251481351194849551818145219724', 16, 3);
    t('-5.385064703950286052004e+60882', '-5.38506470395028605200453518650931090484061922E+60882', 22, 2);
    t('-2.3588290704067116464264389015e+1555115', '-2.35882907040671164642643890150056938E+1555115', 30, 4);
    t('-2.107232017313772399825603063301153657381914e+47', '-210723201731377239982560306330115365738191488395.372', 43, 2);
    t('-8.37e+397603', '-8.376E+397603', 3, 2);
    t('-5.892562e-3', '-0.00589256156', 7, 0);
    t('8.0964e+458984', '8.09633685289916020182E+458984', 5, 2);
    t('7.433330767966e+76057', '7.43333076796575098042863729019264347874039699474E+76057', 13, 6);
    t('5.1798437206367294133907437423855020240854549e+20', '517984372063672941339.074374238550202408545488591848', 44, 5);
    t('9.10152743e-74', '9.1015274214338923215292236337711858950705671416387E-74', 9, 2);
    t('-3.0837839602596928e+3', '-3083.7839602596928205378643609134808026254498190075461220407', 17, 2);
    t('-6.141e-7301273', '-6.141E-7301273', 5, 1);
    t('-2e+717', '-2.0E+717', 3, 2);
    t('-1.3665608148990461527844985751715e+73', '-1.36656081489904615278449857517150E+73', 36, 3);
    t('7.9636e+754', '7.963589505242977095546449826544152648535667799E+754', 5, 4);
    t('-2.91e+685', '-2.91E+685', 3, 4);
    t('6.068922808060398e+27251', '6.06892280806039791704489258734411261636125872171407E+27251', 16, 4);
    t('-5.48609664912067964792811635959e+35', '-548609664912067964792811635959013072.3', 30, 2);
    t('-2.093521914841644358469645433266290764688512e-23', '-2.093521914841644358469645433266290764688511683397E-23', 43, 0);
    t('3.384685165854137303255736388705640840568899e+894654', '3.384685165854137303255736388705640840568898687355630230656E+894654', 43, 5);
    t('-7.3808092821574359178073725962077e-1181564', '-7.380809282157435917807372596207699535977034093204E-1181564', 32, 5);
    t('2.46e+7', '24550122.55331348690769123914718981220141506859057', 3, 6);
    t('-4e+714', '-4.1916617E+714', 1, 4);
    t('9.5922592e+976508', '9.5922592348670245392376859E+976508', 8, 1);
    t('-9.96716934647254585827723e+84', '-9.96716934647254585827723224182980624584340755943365597584E+84', 24, 1);
    t('6.99747e+97679', '6.99747060415763791E+97679', 6, 5);
    t('-7.11423205926e-6566', '-7.11423205926E-6566', 13, 3);
    t('1.884791059046866821073e+645132', '1.88479105904686682107284577E+645132', 22, 6);
    t('-6.032030423316542e+353249', '-6.032030423316542E+353249', 18, 3);
    t('7.79965003769688179012649e+702412', '7.799650037696881790126494535404297482909271121468863E+702412', 24, 3);
    t('9.693660463e+6822710', '9.69366046326643E+6822710', 10, 4);
    t('2.805273012e+29960', '2.8052730129984867118599149262452394E+29960', 10, 1);
    t('-8.732870984134597059233338881458e+51734', '-8.732870984134597059233338881458E+51734', 33, 3);
    t('4.67813135e+2', '467.8131343090017449829', 9, 0);
    t('8.3762055703188973e+63600', '8.376205570318897303276827745E+63600', 18, 3);
    t('8.374622193631334586e+681', '8.374622193631334586046010377160431348646E+681', 19, 6);
    t('-9.88048659558632753384e+377622', '-9.88048659558632753384E+377622', 24, 1);
    t('5.1034068584976260922828765795708220990400757994803e+7169675', '5.103406858497626092282876579570822099040075799480277746529E+7169675', 50, 0);
    t('8.5359233e+9894091', '8.53592335E+9894091', 8, 3);
    t('5e-978380', '5E-978380', 1, 0);
    t('-7.371470663e+1468', '-7.371470663E+1468', 13, 2);
    t('6.7522939985874061556702186064650212913e+5440', '6.752293998587406155670218606465021291289033638196890016E+5440', 38, 4);
    t('-9.923744208538918071e+56', '-9.923744208538918071313058E+56', 19, 1);
    t('-6.427832654969132e+7511841', '-6.4278326549691325468349282080130969E+7511841', 16, 2);
    t('5.6679263146179264226e-26196', '5.66792631461792642260301182213006E-26196', 20, 6);
    t('4.299929721123892854900795e-583', '4.29992972112389285490079548767342932803930253419964E-583', 25, 4);
    t('5.3412154629944419e+7', '53412154.629944419444586294857', 17, 1);
    t('-8.131177903650305692456787025341898095e+831043', '-8.131177903650305692456787025341898094383E+831043', 37, 0);
    t('3.9466641231978e+4330', '3.9466641231977941504010972E+4330', 15, 2);
    t('-2.06125356183e-6131', '-2.06125356182975871308924306461894884815064372375098730E-6131', 12, 6);
    t('9.2326899603547e+1228549', '9.232689960354682049257541629195181E+1228549', 14, 5);
    t('-7.11116e+4315', '-7.1111515801294991969112909555390059635995822032258E+4315', 6, 0);
    t('-4.4373208373587224450466939385841693720522015022067e+718875', '-4.437320837358722445046693938584169372052201502206680065993E+718875', 50, 4);
    t('-4e+524', '-3.96E+524', 2, 3);
    t('-5.339008861318178284335955048737074258481812431421791571e-3370054', '-5.3390088613181782843359550487370742584818124314217915710E-3370054', 59, 2);
    t('8.539504e+8396', '8.539504E+8396', 8, 4);
    t('4.5947e+74992', '4.5947945E+74992', 5, 1);
    t('6.30510478e+570606', '6.30510477538E+570606', 9, 6);
    t('2.5e+976', '2.47E+976', 2, 5);
    t('-7.8031120126955021456052078571172387e+682', '-7.8031120126955021456052078571172387E+682', 38, 6);
    t('5.121367e+9716771', '5.1213673451535160318692948778552270E+9716771', 7, 4);
    t('-6.56034057157504771604350526e+74', '-6.5603405715750477160435052600E+74', 30, 1);
    t('3.279995117e+100358', '3.279995116924116E+100358', 10, 6);
    t('2.6005752859164089822615788675995755332146855e+31868', '2.6005752859164089822615788675995755332146855E+31868', 44, 2);
    t('-4.8355815273790134100536393459096792862e-80', '-4.83558152737901341005363934590967928623824459831E-80', 38, 1);
    t('-1.951846674768e-79', '-1.951846674768E-79', 14, 4);
    t('-7.3328697e+49', '-7.3328696488586938E+49', 8, 3);
    t('-2.882014541656e+2779347', '-2.882014541656197649267E+2779347', 13, 4);
    t('8.314e-1905', '8.31448E-1905', 4, 3);
    t('-1.1752225e-637070', '-1.1752224988694631E-637070', 8, 6);
    t('-1.76514278892e+993792', '-1.7651427889246804059416797823440212022E+993792', 12, 5);
    t('2.55824877812e+8771395', '2.5582487781207856113196827968031842917553675E+8771395', 12, 5);
    t('-8.558401192286028757049e+343706', '-8.558401192286028757049E+343706', 23, 6);
    t('-3.5e+840', '-3.4401074070621905590555159112E+840', 2, 3);
    t('-2.99672648482721009608289562424547e+9', '-2996726484.82721009608289562424546600936599626301451', 33, 4);
    t('6.8674363184202233e+800798', '6.86743631842022329172936897814E+800798', 18, 0);
    t('5.36050497306587365452710885e-944011', '5.360504973065873654527108849714E-944011', 28, 0);
    t('3.92468691e+70937', '3.924686908865201356E+70937', 9, 4);
    t('7.80152038356565725255086e+723', '7.80152038356565725255086E+723', 27, 3);
    t('-3.42777716943308485007304e+47026', '-3.42777716943308485007304E+47026', 27, 0);
    t('9.665576035e+286', '9.66557603542207494389442966116662940E+286', 10, 5);
    t('-4.8908306e+87541', '-4.89083060087099469781987306E+87541', 9, 6);
    t('1.79753900244538662792669214227529592054e+7893690', '1.79753900244538662792669214227529592053767E+7893690', 39, 2);
    t('-8.4793940387746676870951188505669765132733407e-20', '-8.47939403877466768709511885056697651327334070079203E-20', 45, 1);
    t('-2.7e+1729', '-2.78E+1729', 2, 1);
    t('2.827689960150689114253944677197e+499151', '2.82768996015068911425394467719674833277893427E+499151', 31, 2);
    t('-3.64348171834114400597e+2849647', '-3.64348171834114400596315840603908197914E+2849647', 21, 0);
    t('-4.2078086e-3', '-0.004207808627401708339092142055881', 8, 6);
    t('-4.3126127291530433315e+529', '-4.31261272915304333150249582400275534806639973118376772880E+529', 20, 2);
    t('-3.8e+4999689', '-3.793446857151237703384178895759447278786456922659725040E+4999689', 2, 5);
    t('1e+4587256', '9.890478135719743490580514815339872078243225E+4587255', 1, 0);
    t('8.70474445277395586925894857810961e+1174', '8.704744452773955869258948578109601600817906E+1174', 33, 0);
    t('-2.861579e+438271', '-2.86157885025930E+438271', 7, 6);
    t('6.041358333376001374613386284859e+4395', '6.041358333376001374613386284859E+4395', 31, 1);
    t('-3.836154e+6114', '-3.836154703467862884010086306545730E+6114', 7, 2);
    t('-9.2243280235374058719628e+9148465', '-9.224328023537405871962766815353483816189375897881847609E+9148465', 23, 5);
    t('-4.65168248409524474207e+662', '-4.6516824840952447420677515097E+662', 21, 3);
    t('8.63330141062483922800025128244e+519', '8.633301410624839228000251282437796476E+519', 30, 5);
    t('-2.9574e-57', '-2.9574E-57', 6, 1);
    t('-3.5235366065838619853469424349615376423079638814618139e+93', '-3.5235366065838619853469424349615376423079638814618139E+93', 54, 4);
    t('-6.5013636e-480', '-6.5013636E-480', 8, 5);
    t('-2.10027484797043543909416008220040756104562779474190846757e+9', '-2100274847.970435439094160082200407561045627794741908467571', 57, 2);
    t('-2.9638e+77985', '-2.963702240E+77985', 5, 3);
    t('8.7107204748e+9619', '8.71072047475081951635189521299626198499315431828E+9619', 11, 5);
    t('-6.4232834e+21', '-6.4232834E+21', 11, 1);
    t('-1.012e+74', '-1.0117478467818320359819987765E+74', 4, 0);
    t('-9.300376331641989822e+932', '-9.300376331641989822347391840594753928947E+932', 19, 4);
    t('9.32320761660799985340765205828049019e-85126', '9.3232076166079998534076520582804901885234445061E-85126', 36, 2);
    t('-8.5e+35', '-8.512679E+35', 2, 2);
    t('-6.7018036086e+75', '-6.701803608598427597768253950E+75', 11, 3);
    t('-8.58041e+794977', '-8.580409357157961088730078365426004823218367758425364E+794977', 6, 5);
    t('9.58589849117033587e+22451', '9.58589849117033586144724289060686E+22451', 18, 0);
    t('5.92985636453440274751024e+737', '5.92985636453440274751024469055940493E+737', 24, 1);
    t('4e+6388', '4.0E+6388', 3, 5);
    t('-6e+42261', '-6.1081141931288364053E+42261', 1, 5);
    t('-2.56523082870809016375835191e+7', '-25652308.287080901637583519065115611982608920565789769134', 27, 0);
    t('-7.321698712e+7', '-73216987.1219445635514155', 10, 1);
    t('-4.23966608396248001510917340233655786e-245302', '-4.23966608396248001510917340233655786E-245302', 38, 6);
    t('8.7025664e+81133', '8.70256633E+81133', 8, 2);
    t('2.0062732e+42736', '2.006273226703575512E+42736', 8, 5);
    t('9.21e+6898', '9.207314E+6898', 3, 5);
    t('-4.72528252425070876255067987007e-362', '-4.72528252425070876255067987006811688237724E-362', 30, 5);
    t('-6.0702982495097970749213568803453064606417813268843e+25243', '-6.0702982495097970749213568803453064606417813268843E+25243', 53, 5);
    t('8.24e+0', '8.24', 6, 3);
    t('-4.39862654146454573713715963e+7562', '-4.398626541464545737137159630899038E+7562', 27, 1);
    t('1.19528e+7310', '1.195280E+7310', 9, 0);
    t('-2.6130570886580000482e-89', '-2.613057088658000048140843094531E-89', 20, 3);
    t('5.254294114568445415042285409896e-379765', '5.2542941145684454150422854098962590E-379765', 31, 3);
    t('-9.8213229327546375629e+234415', '-9.821322932754637562925570607704166878090E+234415', 20, 4);
    t('-3.573203473558881048e+95', '-3.573203473558881048E+95', 20, 0);
    t('2.63e+12497', '2.63E+12497', 5, 6);
    t('-1.07663825075e+1', '-10.7663825075', 15, 2);
    t('3.588829193397e+4', '35888.2919339690165573909', 13, 6);
    t('-9.1403231742090404230488934093861e-53', '-9.1403231742090404230488934093861060990049524E-53', 32, 6);
    t('1.639e+635', '1.638604635522197375641912425724880531575114E+635', 4, 0);
    t('-6.46596850944e+2550', '-6.46596850944E+2550', 12, 4);
    t('-7.661876093570487253311496368213e-9687604', '-7.661876093570487253311496368212218884541809080240960861893E-9687604', 31, 0);
    t('-1.125e+943', '-1.124997261108011571645563873402952E+943', 6, 4);
    t('7.6993801629633e+1911685', '7.69938016296334529471041E+1911685', 14, 1);
    t('6.862478734806e-92768', '6.862478734805963358227622491001428489E-92768', 14, 4);
    t('2.2109847343455339e+2', '221.0984734345533994638268608301246', 17, 3);
    t('-1.2366e+544', '-1.236545845289913942704729737986266389826087006E+544', 5, 0);
    t('-8.79893576e-29990', '-8.7989357573E-29990', 9, 6);
    t('6.1484866820895025882262053026e+1', '61.4848668208950258822620530260220738857220455609790446489526', 30, 1);
    t('-7.43424344069123943802993119537533e+8', '-743424344.0691239438029931195375334662351', 33, 2);
    t('-3.7649976148957416363239273766163965424677728e+412002', '-3.7649976148957416363239273766163965424677728341687E+412002', 44, 2);
    t('-1.20850953157416e-569', '-1.20850953157416990E-569', 15, 2);
    t('-6.053647681648879e+358577', '-6.053647681648879E+358577', 19, 0);
    t('1.646969805703345e+620', '1.6469698057033450737E+620', 16, 5);
    t('1.654735820553784188829468994e+122802', '1.654735820553784188829468994362684140283E+122802', 28, 3);
    t('7.7226288428221069734089839e+1', '77.226288428221069734089839633683279293', 26, 1);
    t('2.82992201e+146', '2.8299220136999892314E+146', 9, 5);
    t('-6.182602e+782', '-6.1826014365277571361500627843587364E+782', 7, 0);
    t('-9.313e+9560', '-9.313145E+9560', 4, 6);
    t('-2.2242994e+0', '-2.224299483613225382', 8, 1);
    t('-8.89777373770592235578e-1', '-0.889777373770592235578392779', 21, 1);
    t('-6.4457850490707257487722e+5', '-644578.5049070725748772160764928840472774088872', 23, 0);
    t('-9e+118077', '-8.9996668732770963938593388335E+118077', 1, 0);
    t('3.51714912015551311801e+7', '35171491.2015551311801', 21, 2);
    t('-3.32353810966223e-9', '-3.32353810966223042404136997337466230623043748759023517E-9', 16, 2);
    t('3.7737482191944382319225869047387888e+5', '377374.82191944382319225869047387887983525851', 36, 5);
    t('4.3783e+9567', '4.37826838589510261123909E+9567', 5, 6);
    t('-3.985e-2', '-0.03985070188803290', 4, 2);
    t('1.65197396866300583907e-3', '0.00165197396866300583907', 21, 2);
    t('7.18503097437965513662e+782', '7.1850309743796551366240612E+782', 21, 1);
    t('-7.73242565208328098297381025950920089121514584420287e+5508', '-7.73242565208328098297381025950920089121514584420286539486143E+5508', 51, 0);
    t('4.62e+2790305', '4.617E+2790305', 3, 0);
    t('5.8e+783953', '5.8E+783953', 3, 5);
    t('7e-8', '7.5E-8', 1, 3);
    t('-3.703e+9480736', '-3.7032972342352394E+9480736', 4, 1);
    t('-7.8193e+2', '-781.9322838959582540801075467997413947441104', 5, 2);
    t('1.7997138330074987748478365385e+1', '17.997138330074987748478365384948636276743736', 29, 6);
    t('1.2571804822997082e-351476', '1.2571804822997082E-351476', 17, 5);
    t('-7.87203993859788329e+5', '-787203.99385978832900', 21, 1);
    t('-1.9711e+512', '-1.9711E+512', 7, 6);
    t('-6.458147e+0', '-6.458147', 9, 2);
    t('-9.965341105616582435e-3', '-0.0099653411056165824347595722007756515232971182837876107500', 19, 5);
    t('-9.71685658005581348171982623888e-5', '-0.0000971685658005581348171982623888', 32, 4);
    t('2.974354561e+24', '2.9743545616887E+24', 10, 3);
    t('-3.7060339384902261e+286176', '-3.706033938490226161179003097636260948204E+286176', 17, 2);
    t('2e-4', '0.000194283720', 1, 5);
    t('-2.5586117223089085937150934e+6314', '-2.558611722308908593715093336003543953851480914E+6314', 26, 3);
    t('3.373337836399702092209516453181178958490430664635829e+2', '337.3337836399702092209516453181178958490430664635829', 52, 0);
    t('-8.7712889764269490080349160394265975992822029043078e+452034', '-8.7712889764269490080349160394265975992822029043077604875046E+452034', 50, 0);
    t('-5.1257027650068071859293493660340238701e+818916', '-5.125702765006807185929349366034023870128382973296E+818916', 38, 1);
    t('-4.8229721230334924619225587990386e+2', '-482.29721230334924619225587990385994651481', 33, 5);
    t('2.6868e+4761575', '2.686754433401575579698072090448979893606534281374639826E+4761575', 5, 0);
    t('-3.02517098818e+0', '-3.0251709881899', 12, 2);
    t('-3e+832277', '-3E+832277', 3, 6);
    t('-7.05504287197929899496862298902170284961275938785759583e+2580', '-7.05504287197929899496862298902170284961275938785759582542774E+2580', 54, 6);
    t('-9e-4607115', '-9.382959279273049281089E-4607115', 1, 2);
    t('-4.229180768636231729e-69', '-4.2291807686362317299724960792108E-69', 19, 1);
    t('-5.97943311271501043695766450737e+91306', '-5.9794331127150104369576645073739177394535642942750317237E+91306', 30, 4);
    t('4.31e+1', '43.03226234319746568', 3, 0);
    t('-5.028753472056497383982195753029849e-7284', '-5.0287534720564973839821957530298494763856502002179E-7284', 34, 4);
    t('-6e+10', '-5.5E+10', 1, 6);
    t('2.74e+99653', '2.74E+99653', 3, 1);
    t('-9.7160755250939056056e+6174975', '-9.7160755250939056055128865E+6174975', 20, 3);
    t('1.64703260594999873265508198931627723774207175868771193e+233856', '1.64703260594999873265508198931627723774207175868771193E+233856', 57, 3);
    t('-6.4762149849938963e+7920', '-6.4762149849938963E+7920', 19, 0);
    t('-3.840388265558069445342020515117e+756', '-3.84038826555806944534202051511765456276320E+756', 31, 2);
    t('-3.839088274337955335e+798708', '-3.8390882743379553353357323139204867334E+798708', 19, 6);
    t('3e+6451143', '3.2553353651E+6451143', 1, 3);
    t('-6.5603394064144024105319192732e+8771347', '-6.56033940641440241053191927327138338481082667976394207E+8771347', 29, 2);
    t('5.0797093675671126439717207030262656429e+774094', '5.0797093675671126439717207030262656428167641427406445E+774094', 38, 2);
    t('-4.3398358136589e+5571', '-4.3398358136589757267208930041885E+5571', 14, 2);
    t('6.090157657337714e+858449', '6.090157657337714042852762514567E+858449', 16, 6);
    t('-4.735197645902557646e+7', '-47351976.45902557646097', 19, 5);
    t('-4.9943833093335881e+4236', '-4.99438330933358815109747407E+4236', 17, 2);
    t('-6.81911213140603910130903153447406e+7', '-68191121.314060391013090315344740588262097', 33, 0);
    t('-6.84732483876501375073554948e+1562512', '-6.84732483876501375073554948E+1562512', 29, 1);
    t('-1.98e+3735', '-1.97867877339126E+3735', 3, 5);
    t('4.4e+79', '4.4E+79', 2, 2);
    t('-4.71162086413259436570589405271943452248941723e+81573', '-4.711620864132594365705894052719434522489417229E+81573', 45, 4);
    t('-1.6420114878653176248e+1507645', '-1.6420114878653176247989461814021919E+1507645', 21, 5);
    t('7.090214e+413228', '7.09021416E+413228', 7, 1);
    t('7.53e+4', '75390.00348060062031', 3, 3);
    t('-7.184793397411864082259e-47', '-7.1847933974118640822590E-47', 26, 2);
    t('1.8714579341875637e-54', '1.87145793418756374298307262498563796357693943E-54', 17, 1);
    t('9.7742419409154131601268845464492522813e+151', '9.7742419409154131601268845464492522813E+151', 38, 5);
    t('4.8642840149222236275685485884149597931e+6573', '4.86428401492222362756854858841495979316612617406293289E+6573', 38, 3);
    t('-8.345442898479435663415193380384e-461124', '-8.3454428984794356634151933803840E-461124', 34, 5);
    t('4.978503728e-496414', '4.9785037272814786147E-496414', 10, 2);
    t('-9.0262030141891668783479732117672912086489060999e+2817', '-9.026203014189166878347973211767291208648906099973579717746E+2817', 47, 2);
    t('7.92811253509863e+916', '7.9281125350986249267E+916', 15, 0);
    t('-6.0022e-83169', '-6.002248896680703510613227314711143565007736E-83169', 5, 6);
    t('8.73163565e+2537182', '8.73163565699815154009868837014772362073E+2537182', 9, 3);
    t('-8.844956545211218e+42951', '-8.84495654521121817797538688766632444805672348523080555540706E+42951', 16, 4);
    t('-8.506780484518922265820936622361681721025792338679e+0', '-8.50678048451892226582093662236168172102579233867949750', 49, 6);
    t('7.0771150422262867732044720846843475352095e+1270', '7.077115042226286773204472084684347535209533422982541142656E+1270', 41, 1);
    t('4.316147920924e+4087', '4.316147920923227493E+4087', 13, 2);
    t('4.956349113275277950742396e+3156542', '4.9563491132752779507423957585E+3156542', 25, 2);
    t('-8.71654e+4381841', '-8.716540485683585474573875244088495741473771892296269E+4381841', 6, 4);
    t('9.505e-6', '0.00000950500465788', 4, 6);
    t('-5.299609854720926506243874451588818195124203607e+4498', '-5.299609854720926506243874451588818195124203606720435867E+4498', 46, 6);
    t('-9.5318904654055584e+0', '-9.5318904654055584', 20, 1);
    t('-7.757743850327846079e+219344', '-7.7577438503278460794813366437279570E+219344', 19, 4);
    t('-9.09940269714098543594535e-2457', '-9.09940269714098543594535E-2457', 25, 0);
    t('-2.845837654e+297', '-2.84583765399532860761406604278239971429962479387446427955471E+297', 12, 0);
    t('9.7397135741805224948e+783551', '9.739713574180522494783671398111862387937869484E+783551', 20, 4);
    t('-2.50202086058358348393e+49', '-25020208605835834839282044121721829590365039893844.91036101', 21, 3);
    t('5.728630981323067853028929894910289375332856367e+53', '5.728630981323067853028929894910289375332856367E+53', 49, 1);
    t('6.881471e+50', '6.881470823078821003684E+50', 7, 0);
    t('-4.9439665803162249675924329074068e-56', '-4.943966580316224967592432907406881081795265771404E-56', 32, 2);
    t('4.433364652090877121187774137933281785720218226780910285e+529578', '4.43336465209087712118777413793328178572021822678091028554091E+529578', 55, 3);
    t('-9.87786065607668166877052298377541608325491754182025e+14100', '-9.877860656076681668770522983775416083254917541820245376E+14100', 51, 3);
    t('7.58735956165834555893520606152375998596e-603', '7.58735956165834555893520606152375998595975224031192116336897E-603', 40, 4);
    t('-5.3e+52520', '-5.3357852095481E+52520', 2, 2);
    t('-2.155920057e+9250', '-2.1559200573558277583834192221274402861497506E+9250', 10, 2);
    t('-3.27527575795249e+691', '-3.275275757952480495922777951729054338E+691', 15, 0);
    t('-3.5978e+44338', '-3.5978E+44338', 6, 2);
    t('7.337714431779147803325e-340', '7.33771443177914780332499563E-340', 24, 2);
    t('4.760343e+35092', '4.760342515171675359569114965540808144656514570E+35092', 7, 0);
    t('6.1e+8518000', '6.103859E+8518000', 2, 6);
    t('9.69318e+32948', '9.6931841138E+32948', 6, 5);
    t('4.466074195543396064e-385', '4.4660741955433960641035447E-385', 19, 4);
    t('6.715021e+47746', '6.7150212755061885245804935804808E+47746', 7, 3);
    t('-6.96179216341587283860816340174266277e+5', '-696179.216341587283860816340174266276955393440577845829973159', 36, 5);
    t('-2.18611761e+834270', '-2.1861176055439583222E+834270', 9, 0);
    t('-1.587120844251016734486593e+80767', '-1.587120844251016734486593E+80767', 27, 5);
    t('4.68318007830336e+125423', '4.683180078303362132783981517786257741584E+125423', 15, 5);
    t('4.7412845967846656075530184917702878e+80', '4.7412845967846656075530184917702878047544771537E+80', 35, 1);
    t('-5.58740154e+38', '-558740153048935033987425045398734336237.14638580351522', 9, 3);
    t('4.92e+35935', '4.9213468856913814176233255210193E+35935', 3, 1);
    t('-2.13e+4592', '-2.13E+4592', 4, 3);
    t('8.188162594553237765367155521324363754e+165', '8.188162594553237765367155521324363754E+165', 39, 1);
    t('-8.8e+364851', '-8.8E+364851', 4, 1);
    t('-2.63134e+5560', '-2.63134187853713610200299E+5560', 6, 5);
    t('7e+2', '673.48452557', 1, 6);
    t('-3.6542241109977e+7', '-36542241.109977077461198934056', 14, 6);
    t('2.55818717460121114408886103292237e+385444', '2.55818717460121114408886103292237242216442164971E+385444', 33, 6);
    t('3.063e-9480773', '3.06236687879E-9480773', 4, 2);
    t('-1.0868e-2026922', '-1.0868017608E-2026922', 5, 5);
    t('2.889181881373201e-349', '2.889181881373200995781137411E-349', 18, 5);
    t('-5.3641835676e+85873', '-5.364183567552347476930458997157083E+85873', 11, 5);
    t('-8.952562e+642', '-8.952562037353765E+642', 8, 4);
    t('-9.88e+92823', '-9.88E+92823', 3, 3);
    t('2.4430151572986223e+2994170', '2.44301515729862229125616087982683721102338042900E+2994170', 17, 0);
    t('4.51e+7791', '4.51E+7791', 4, 4);
    t('-5.187396e-2950', '-5.187395628772943100E-2950', 7, 3);
    t('2.195967415939620487329702061632918511776413e+360', '2.1959674159396204873297020616329185117764134E+360', 43, 4);
    t('2.66715e+536107', '2.667149040697930655576123782024258086843997840816E+536107', 6, 4);
    t('4.472306784425e-5', '0.000044723067844251958792463535780524', 13, 3);
    t('-5.1614936645387e+687', '-5.16149366453870E+687', 15, 0);
    t('5.200536e+3', '5200.53595722388562690469550228025433632396059662661901173655', 7, 5);
    t('-3.827e+7', '-38267710.5064379201601626921330', 4, 5);
    t('-3.173e+219', '-3.17284561261E+219', 4, 0);
    t('-8.7028719619254419354e+86', '-8.7028719619254419354E+86', 22, 0);
    t('2e-41802', '2.0620334676178509147815612710000761156996908532223388E-41802', 1, 4);
    t('-3.3429057e+38323', '-3.3429057E+38323', 11, 6);
    t('-8.196522161628225766450552474559564923e+82', '-8.196522161628225766450552474559564923E+82', 39, 0);
    t('-4.4906195215540381336516051013698e+954', '-4.490619521554038133651605101369807972971E+954', 32, 2);
    t('4e+5910075', '4.039168E+5910075', 2, 3);
    t('4.2e+127', '4.23755613632598514243896E+127', 2, 4);
    t('1.6118204e-468', '1.61182037095294436031528254256E-468', 8, 0);
    t('-3.10389584661634027684e+66', '-3.1038958466163402768302351223590549785010169E+66', 21, 3);
    t('-9.80337024645126792010294817e-6', '-0.00000980337024645126792010294817', 28, 6);
    t('-9.146e+798', '-9.1457808405747569775691191383908645358E+798', 4, 4);
    t('4.568e+0', '4.5672174401332565777988623', 4, 0);
    t('-8.24915305858710156541523554079e+2', '-824.91530585871015654152355407949155119615', 30, 4);
    t('-1e+893154', '-1.0451609E+893154', 2, 4);
    t('-3.875107046615123973646e+5688', '-3.8751070466151239736460820E+5688', 23, 1);
    t('9.7005336678985e+405', '9.700533667898551911745951958859846338985234481833887E+405', 14, 1);
    t('-7.137120879332517e+779256', '-7.13712087933251669352E+779256', 16, 4);
    t('-1.19282072168685647315024114892595192498163338296910515e+9881370', '-1.1928207216868564731502411489259519249816333829691051497607E+9881370', 54, 5);
    t('5.47506463974e+91889', '5.4750646397422308149496797817355956612798719169105E+91889', 12, 1);
    t('9.9244434e+6070156', '9.924443429740401790110543583244854097620706487E+6070156', 8, 6);
    t('4.292820427625019143760151e+122', '4.29282042762501914376015128623681942921348E+122', 25, 6);
    t('-9.42e+614613', '-9.42E+614613', 4, 2);
    t('-9.6e-3658666', '-9.553449357332696970559243360040025750E-3658666', 2, 4);
    t('-6.192e+5791', '-6.1916511938529602919E+5791', 4, 0);
    t('4.4545137439187013208e+243429', '4.454513743918701320809208403318588E+243429', 20, 3);
    t('3.6595217576936318386477898709431709e+9543', '3.659521757693631838647789870943170986459332714381908E+9543', 35, 1);
    t('6.0377651918071037865814e-871', '6.03776519180710378658133583752181895516465643180408407538E-871', 23, 2);
    t('1.86e+906', '1.86E+906', 5, 3);
    t('5.22e+7', '52249843.394967265134968193994598284307435485636091998677510', 3, 1);
    t('3.576931147358621e+3', '3576.931147358620469813465959773208092209560310073974', 16, 0);
    t('8e+2751592', '7.1E+2751592', 1, 0);
    t('-1.855288299891982146e+173', '-1.855288299891982146E+173', 21, 2);
    t('1.40107e+1224626', '1.4010719610933900660081466533708212087336047335E+1224626', 6, 3);
    t('8.357417593e+2', '835.74175929767016609362', 10, 0);
    t('7.66989e+417932', '7.6698896100228873E+417932', 7, 0);
    t('6.30105e+9', '6301055982.69016095514401922027618249097611893781344', 6, 1);
    t('8.9100073667171e-68063', '8.91000736671710337E-68063', 14, 6);
    t('7.323869152119302031456895e+83', '7.323869152119302031456894900945679361E+83', 26, 0);
    t('-7.362291759320817675300263853735594e+950', '-7.362291759320817675300263853735593188717165E+950', 34, 3);
    t('6.6492503055280986e-95365', '6.64925030552809869905383549279251121861986419697041E-95365', 17, 3);
    t('5.34718150085879402441849695656074e+2', '534.71815008587940244184969565607398176', 34, 6);
    t('4.902958493595450560107371838147e-838885', '4.90295849359545056010737183814790905412430522911872E-838885', 31, 1);
    t('-8.189e+55512', '-8.1894166174147063820810097870615510E+55512', 4, 1);
    t('-2.8314877622105e+6', '-2831487.7622104624575535212701370', 14, 4);
    t('3.89255e-7941', '3.89255713159591974603584686214294742E-7941', 6, 3);
    t('-3.5247e+67233', '-3.524656675073615311712686618730205934621108528028846E+67233', 5, 6);
    t('-9.72817e+34864', '-9.7281745115551383E+34864', 6, 6);
    t('-8.097814782422950317232400927298682e+11', '-809781478242.29503172324009272986813384', 34, 3);
    t('-2e-81689', '-2E-81689', 2, 3);
    t('-9.19565743927e-49458', '-9.1956574392648E-49458', 12, 3);
    t('4.536693e+6', '4536693.1497687075463642928', 7, 5);
    t('4.4853963108791945903769729589132e+9', '4485396310.8791945903769729589131358794034707403380516005575', 32, 2);
    t('-8.141848452925248089405126361475851419073e+19783', '-8.141848452925248089405126361475851419073E+19783', 43, 1);
    t('-5.2435730849379153168212800831446296199077048602608651349e+9163', '-5.2435730849379153168212800831446296199077048602608651349E+9163', 59, 3);
    t('-1.150049161e+8625781', '-1.1500491608757162459352E+8625781', 10, 0);
    t('-1e+17958', '-9.7787E+17957', 1, 5);
    t('-9.52370985948e+73114', '-9.523709859487945395998592337323597662137307E+73114', 12, 1);
    t('6e+3517', '6E+3517', 4, 1);
    t('-7.71813423335e+311170', '-7.71813423335884120814803325033E+311170', 12, 2);
    t('-2.686852817394765322049191e+845', '-2.6868528173947653220491905033929709428160289191E+845', 25, 4);
    t('-8.31004e+77', '-8.3100460867713103E+77', 6, 2);
    t('-6.287790422040372030104e+53', '-6.28779042204037203010403E+53', 22, 2);
    t('9.884273077483605929216e+235', '9.8842730774836059292158418466575646924547058208438933514176E+235', 22, 2);
    t('-6.5636990192880074264081853982725967337173723489178e+7', '-65636990.192880074264081853982725967337173723489178275', 50, 6);
    t('4.6197596005276949418167226159792e-2750080', '4.61975960052769494181672261597918275884439847328142025190E-2750080', 32, 5);
    t('-8.099e+730', '-8.099E+730', 5, 2);
    t('-6.625e+2612', '-6.62469E+2612', 4, 5);
    t('2.5e+31', '2.50E+31', 3, 6);
    t('9e+996605', '8.4E+996605', 1, 0);
    t('3e+40', '2.987511518319277875887E+40', 2, 6);
    t('-3.550969772138267134156883117635e+376', '-3.5509697721382671341568831176352996E+376', 31, 6);
    t('4.470934604741057e+83822', '4.4709346047410567675336322293545439E+83822', 16, 0);
    t('2.999026064966589628e+9486', '2.999026064966589628E+9486', 19, 6);
    t('-8.1115e-801856', '-8.1114766833960309621588974241938756653641679570971806411E-801856', 5, 5);
    t('-4.42863531e+6439592', '-4.4286353127E+6439592', 9, 6);
    t('5.4311e-270552', '5.431065681532234796772221883481707691378719315136713815E-270552', 5, 2);
    t('8.518292289896948690337413e-7466', '8.5182922898969486903374136871659700E-7466', 25, 3);
    t('2.960298047343473309275821769682517571051e+72926', '2.96029804734347330927582176968251757105109749002261343E+72926', 40, 5);
    t('-1.0011e+3298', '-1.0010979611192E+3298', 6, 0);
    t('-4.43361592050307015e+2422', '-4.43361592050307015231E+2422', 18, 1);
    t('-4.631245463104814239004929663544249807e-5531', '-4.63124546310481423900492966354424980670095776578024942866E-5531', 37, 4);
    t('4.270794609051e+9962159', '4.2707946090514654326481371585E+9962159', 13, 5);
    t('-8.665421e+1086801', '-8.6654214644388E+1086801', 7, 4);
    t('-4.56662004e-86', '-4.566620032141954438E-86', 9, 0);
    t('7.9e+50', '7.9E+50', 2, 1);
    t('-9.425351e-24', '-9.425350307E-24', 7, 0);
    t('1.658968919693332e+657', '1.6589689196933319175E+657', 16, 2);
    t('-5.899013201060474764524863934e-33057', '-5.8990132010604747645248639339647154701E-33057', 28, 5);
    t('5.5705560728309939608452372e+8399910', '5.57055607283099396084523726124227E+8399910', 26, 3);
    t('-7.8e-4095158', '-7.82913083E-4095158', 2, 4);
    t('-2.71229e+8757', '-2.71229249030158607501711532471290742E+8757', 6, 2);
    t('-8.1108035399489e+34', '-81108035399488995137367506353175934.646649', 14, 3);
    t('2.0814421141940890607829644e-88995', '2.0814421141940890607829644203982225863869983807934608586E-88995', 26, 4);
    t('8.38084275115529265587196e+6123343', '8.380842751155292655871960E+6123343', 25, 3);
    t('-4.055e-311465', '-4.0557664497E-311465', 4, 1);
    t('-4.21499e+6163', '-4.214983001335E+6163', 6, 3);
    t('-2.118428e+9', '-2118428110.2', 7, 6);
    t('9.3441e-3742821', '9.3440466200308592235E-3742821', 5, 0);
    t('1.57618570134515860759e+5', '157618.5701345158607592238761069154', 21, 1);
    t('9.01468e+92', '9.0146795902817463820607498822984767816143677120489593248306E+92', 6, 5);
    t('-1.71950674239546e+3', '-1719.506742395465199571153078497254191955756771', 15, 2);
    t('6.1e-788398', '6.16645267348568798416809877587645E-788398', 2, 3);
    t('8.90983997e-77110', '8.909839965454374329446126996547715689777E-77110', 9, 5);
    t('8.1401958e+478', '8.140195766175631E+478', 8, 0);
    t('7.1027357762993e+3150', '7.1027357762992752089942053192794264944227735962722893431052E+3150', 14, 6);
    t('-4.707e+4498997', '-4.7067E+4498997', 4, 5);
    t('-6.181e-919610', '-6.181E-919610', 6, 6);
    t('1.176405160499e+43', '11764051604990212746004564375866229949497399.31', 14, 4);
    t('4.822041439e+82866', '4.82204143940908346397825449356591439519370274012826E+82866', 10, 4);
    t('-7.8225116034705331155e+9943825', '-7.8225116034705331154990472301475E+9943825', 20, 4);
    t('5.128911606411e+432', '5.128911606411E+432', 13, 6);
    t('3.8e+0', '3.8214767', 2, 3);
    t('3.1676499687904862353168009234834495131318366245638e+6323867', '3.1676499687904862353168009234834495131318366245638129E+6323867', 50, 6);
    t('9.520918537226072707725362e+9', '9520918537.2260727077253625388272954', 25, 3);
    t('8.810053e-10', '8.81005346E-10', 7, 4);
    t('-3.172844815750470629850904548e+228', '-3.17284481575047062985090454786257688623393E+228', 28, 3);
    t('3.037e+43', '30368403038653212826328177465214921135151050.9364458100', 4, 2);
    t('-8.63e+496507', '-8.63931054901152203626786964517419314886149012E+496507', 3, 2);
    t('4e+51', '3.0588030088043E+51', 1, 2);
    t('7.758071695649e-2', '0.0775807169564957920711071187096215656178023184757689', 13, 3);
    t('-5.220526710012919424034078555e+93026', '-5.2205267100129194240340785553887E+93026', 28, 4);
    t('-3.6885e+487229', '-3.68846871691376965426E+487229', 5, 4);
    t('-5.03440887777333e+7488086', '-5.03440887777333E+7488086', 18, 3);
    t('8.257641e+6', '8257641.378731581', 7, 4);
    t('6.85870002938634037608e+42672', '6.8587000293863403760782566E+42672', 21, 6);
    t('5.34157e-81443', '5.34156775632235670783246479563862E-81443', 6, 6);
    t('-9.4073e+46044', '-9.407347937552E+46044', 5, 1);
    t('-8.28172145838480615760451783e+8', '-828172145.8384806157604517829582998', 28, 4);
    t('9.536153883955755360758034836e+3602179', '9.536153883955755360758034835087830647450035E+3602179', 28, 2);
    t('-2.0800032915703368e+7', '-20800032.915703367688846081204385979440915281582', 17, 5);
    t('-9.8562e+729', '-9.8562E+729', 8, 5);
    t('-7.39950899686e+1775569', '-7.399508996859071729517E+1775569', 12, 4);
    t('5.369993e+7113604', '5.36999264E+7113604', 7, 5);
    t('-6.4964531420684e-87743', '-6.4964531420684003712553239252913008570048465170802024810E-87743', 15, 6);
    t('-5.508e+13228', '-5.508E+13228', 5, 5);
    t('-2e-7683602', '-1.8E-7683602', 1, 5);
    t('-4.5727259e+22', '-4.572725899003608E+22', 10, 3);
    t('3.463572e-4', '0.0003463572', 7, 2);
    t('5.8854e+5910672', '5.8854001222518179836849023839470597770275737670013170979584E+5910672', 6, 5);
    t('-9.63093899458164501505169e-72326', '-9.6309389945816450150516819E-72326', 24, 3);
    t('-5.40121954164248e+1121484', '-5.4012195416424760893922061606285903117665884E+1121484', 15, 4);
    t('-5.94e+299', '-5.939975E+299', 3, 4);
    t('-9.6054105889640443e+8386', '-9.60541058896404426692E+8386', 17, 5);
    t('9.1671666e+8', '9.1671666E+8', 10, 4);
    t('-1.7296765139e+21116', '-1.7296765139E+21116', 11, 5);
    t('-6.75615539954e-1733542', '-6.75615539954E-1733542', 15, 0);
    t('-4.838096e+980976', '-4.83809601077181998439896269E+980976', 7, 4);
    t('-4.08e+384257', '-4.0753332921086823559845194882783381440560771E+384257', 3, 0);
    t('-1.44527954926094945e+9689', '-1.44527954926094945900137173110316072E+9689', 18, 2);
    t('-6.5363e+4', '-65363.3004650', 5, 1);
    t('1.94663e-26', '1.94663E-26', 7, 5);
    t('4.09787056723637227342263e-99023', '4.097870567236372273422632E-99023', 24, 1);
    t('5.17982341566503e-165102', '5.179823415665039980519774970E-165102', 15, 3);
    t('-3e+5050699', '-3E+5050699', 4, 2);
    t('2e-8606348', '1.56255396747802874E-8606348', 1, 5);
    t('-6.926252213505507086985445e-59', '-6.9262522135055070869854458314E-59', 25, 2);
    t('4.350606e+0', '4.350606', 8, 4);
    t('1.6178014156167093476541265884256212791683979281642e+821', '1.61780141561670934765412658842562127916839792816419E+821', 50, 5);
    t('2.1218657914707836373087e+2439', '2.121865791470783637308639E+2439', 23, 2);
    t('3.26691968270084138249697e+489990', '3.26691968270084138249697E+489990', 27, 6);
    t('4.791892189370184341921142929863765319797e+2', '479.18921893701843419211429298637653197969087839848', 40, 4);
    t('-8e+89525', '-8.1213651078761757487828080292361477473497E+89525', 1, 6);
    t('-8.81556952167635721641024815212335932908018691859035657392748e+99', '-8.81556952167635721641024815212335932908018691859035657392748E+99', 62, 6);
    t('6.8e+67904', '6.8275062777393401286729234613E+67904', 2, 4);
    t('-6.98381126005242938453338983576750412e+25220', '-6.98381126005242938453338983576750412E+25220', 39, 3);
    t('-8.58017811412463071689221e-1569035', '-8.580178114124630716892203267230468759274324587548317800E-1569035', 24, 0);
    t('7.527591735e+566', '7.52759173553529912E+566', 10, 3);
    t('5.654701968203e-8637', '5.6547019682025853841976288628431149635289778864319949E-8637', 13, 4);
    t('-2.3231074585379073e+82040', '-2.32310745853790730138744827071683025519346212E+82040', 18, 1);
    t('3.9517638245032988595264589e+52589', '3.95176382450329885952645890653999981862919479379227214973613E+52589', 26, 3);
    t('-1.2226876729551759956378951947407499e+758', '-1.2226876729551759956378951947407499488E+758', 35, 5);
    t('-4.872088e-6', '-0.000004872088', 7, 3);
    t('-9.32e+421898', '-9.3220888E+421898', 3, 1);
    t('-4.919000512823e-557', '-4.919000512822308E-557', 13, 3);
    t('-4.36794456373219367264474995e+6196339', '-4.367944563732193672644749945E+6196339', 27, 0);
    t('3.1e-7279900', '3.11850889481678457629317460E-7279900', 2, 1);
    t('-9.601382052844e+2411', '-9.601382052844E+2411', 16, 6);
    t('-4.635646585266705129105701244587406626788519681481062511259e+1434', '-4.6356465852667051291057012445874066267885196814810625112599E+1434', 58, 2);
    t('7.283974735293644e-555', '7.283974735293644092287454742972135795736771389081885E-555', 16, 5);
    t('-2.5982e+7672084', '-2.5982E+7672084', 5, 1);
    t('-1.65884786158855793838610799883469186e-65259', '-1.658847861588557938386107998834691858019357786147E-65259', 36, 6);
    t('-3.00066e-4', '-0.00030006600170716491249330994753157', 8, 4);
    t('2.084244357481e-2', '0.02084244357481237716095184146187287384969469053246556793', 13, 1);
    t('-5.20115555627285843144032209757185921014841893e-334694', '-5.20115555627285843144032209757185921014841893E-334694', 48, 3);
    t('-8.1e+6', '-8.087E+6', 2, 0);
    t('-2.7969164953157050824017172988955e+3063586', '-2.7969164953157050824017172988954707924E+3063586', 32, 4);
    t('5.669172109606e+74', '5.6691721096060E+74', 17, 3);
    t('-9.2399017527832098008909930378489983456723744148e+1492', '-9.2399017527832098008909930378489983456723744148E+1492', 49, 5);
    t('2.74074541476211346997434597244591557e+30', '2740745414762113469974345972445.9155731193170721350799879667', 36, 1);
    t('-3.98705988696658132e+652', '-3.987059886966581318675039513024E+652', 18, 0);
    t('-6.9e+8043000', '-6.802616E+8043000', 2, 3);
    t('6.0074229997692e+92', '6.007422999769204228409586722031674318E+92', 14, 6);
    t('-2.919159901439191458341802201454759e+88608', '-2.919159901439191458341802201454758951E+88608', 35, 0);
    t('-7.5162371e+27', '-7.516237032705921632388519E+27', 8, 0);
    t('9.19582044e-9297853', '9.1958204362250E-9297853', 9, 2);
    t('8.955581623209e+602492', '8.9555816232094157E+602492', 13, 4);
    t('7.1529205005467794e+9140', '7.15292050054677939554569674596661104E+9140', 17, 2);
    t('1.350569467555028281609809643256354490143995248e+8006', '1.350569467555028281609809643256354490143995248045608245570E+8006', 46, 5);
    t('5.417488235773859903783996533e-862648', '5.417488235773859903783996533121223567618780565712364997855E-862648', 28, 1);
    t('6.453141804476312956379075686639e+9517413', '6.4531418044763129563790756866394209273389324576554236E+9517413', 31, 6);
    t('4.1057138085229988398888417771048636e+7852087', '4.10571380852299883988884177710486352981497247E+7852087', 35, 0);
    t('4.3919450069e+627885', '4.391945006910833324856245885E+627885', 11, 5);
    t('7.4452109419807577865115679041365678159906e-854', '7.445210941980757786511567904136567815990624917E-854', 41, 6);
    t('4.198257248e+262', '4.1982572477E+262', 10, 0);
    t('5.115065133773903499548541e+438653', '5.1150651337739034995485408860968951843945076009636763663324E+438653', 25, 0);
    t('2.8040407746e+1', '28.0404077460', 11, 0);
    t('7.60753613229e-67', '7.6075361322805171021336042807754318651942755861E-67', 12, 2);
    t('5.9436292e+206479', '5.94362920E+206479', 10, 5);
    t('2.143149e+3325', '2.14314830917586223819341531485E+3325', 7, 2);
    t('5.517271e+6', '5517271', 9, 2);
    t('5.313896909576011e+826', '5.31389690957601085302439620188537962685472405595964714531E+826', 16, 5);
    t('-7.01870588207542785528e-482323', '-7.0187058820754278552805638058475598519195658946312E-482323', 21, 5);
    t('-7.19379078688705099072430430609124e+8', '-719379078.6887050990724304306091241561130787299495521', 33, 2);
    t('-3.3233507706603114239824785989229924707228129968e-2357411', '-3.3233507706603114239824785989229924707228129968957019038E-2357411', 47, 1);
    t('2.447626895811422011897246737414538895e+12', '2447626895811.42201189724673741453889402', 37, 2);
    t('-2.848497673e+26585', '-2.848497673E+26585', 10, 1);
    t('-7.3622380265e+4012', '-7.36223802657534696220E+4012', 11, 1);
    t('9.15742743482358933990101e+0', '9.15742743482358933990100783264135911', 24, 4);
    t('-5.880153180079319930184991e+89157', '-5.880153180079319930184990593403650160495825799E+89157', 25, 3);
    t('5.641125298493280005532846e+62261', '5.64112529849328000553284530053884E+62261', 25, 2);
    t('-3.529e-82295', '-3.52897E-82295', 4, 3);
    t('-1.1552750946426377014757877374e+24', '-1155275094642637701475787.7374', 32, 3);
    t('7.187e+2', '718.6747172648354290', 4, 6);
    t('7.11005868980322525870051425015911268693000179022467e+7', '71100586.89803225258700514250159112686930001790224665377', 51, 4);
    t('4.382049e-93844', '4.3820487927010354867784878437270698230551762E-93844', 7, 5);
    t('-9.6138104024501e+10281', '-9.61381040245010E+10281', 14, 4);
    t('9.994714e+5363814', '9.9947142942619889445692135381068360057833312566990754460E+5363814', 7, 1);
    t('6.19e-48', '6.1855E-48', 3, 5);
    t('1.2486424617810472002038e+3', '1248.6424617810472002038096492673638719827034097971', 23, 6);
    t('-8.84781466010525295728517228346766703827552e+5897', '-8.8478146601052529572851722834676670382755197202857E+5897', 43, 4);
    t('-4.3142571509809464898312346047974884632e-48', '-4.314257150980946489831234604797488463242235171443990551E-48', 38, 1);
    t('-1.3161332e-20', '-1.316133233688096102908212620120731092552327E-20', 8, 4);
    t('-7.74566504e-235', '-7.745665038778179875324E-235', 9, 0);
    t('-1.3905690775255247e+16906', '-1.3905690775255247E+16906', 17, 1);
    t('5.076289248e+510966', '5.0762892481983395E+510966', 10, 6);
    t('1.5073771050022329553556e+6', '1507377.1050022329553556896577660913603070315328', 23, 3);
    t('8.085874534357827360055954456507e+96', '8.08587453435782736005595445650712709579E+96', 31, 3);
    t('7.74e-775189', '7.74E-775189', 6, 0);
    t('4.605554824866717091985751992576167521e-884868', '4.6055548248667170919857519925761675210E-884868', 39, 1);
    t('5.117e+166021', '5.11704384677983E+166021', 5, 4);
    t('-4.22e+698', '-4.21852734E+698', 3, 6);
    t('-6.082104e+5734', '-6.0821039592435726E+5734', 8, 3);
    t('-4.09463550220845706024242798647728654330126148725e+1471310', '-4.094635502208457060242427986477286543301261487245367428858E+1471310', 48, 0);
    t('-4.45687141812372036021135220205127076e+66933', '-4.45687141812372036021135220205127075533921E+66933', 36, 0);
    t('4.780995514579026024492087075664405736e+10', '47809955145.7902602449208707566440573536626494', 37, 0);
    t('7.51157e+2802528', '7.511568957564449681704831396E+2802528', 6, 0);
    t('-1e-71', '-9.52927778E-72', 1, 5);
    t('-1.44078650804439866689025727992858790033145841755082587e+318', '-1.440786508044398666890257279928587900331458417550825870E+318', 58, 3);
    t('-9.4e-9864', '-9.4E-9864', 3, 3);
    t('-8.97271976700257834881887401378419e+92016', '-8.972719767002578348818874013784193447567E+92016', 33, 5);
    t('-6.58530780360854070637e-894', '-6.585307803608540706372897184163132E-894', 21, 4);
    t('8.63063070362260215223210716e+145081', '8.63063070362260215223210715810326497041472183915185455951406E+145081', 27, 2);
    t('-4.43817714016944e+162', '-4.43817714016944E+162', 16, 2);
    t('2.23294153345393920760083273306229451792e+643', '2.23294153345393920760083273306229451791130628377E+643', 39, 0);
    t('6.576315e+146', '6.57631596814405769E+146', 7, 3);
    t('1e-5210837', '9.813E-5210838', 1, 5);
    t('4.0599515e-7470', '4.05995149E-7470', 8, 0);
    t('5.547429656549567e+6', '5547429.6565495674090270850341655001746501', 16, 4);
    t('6.408e+97', '6.40784E+97', 4, 4);
    t('7.4249620488e+35307', '7.424962048841527447E+35307', 11, 6);
    t('4.83116332e-1435', '4.83116331642870986979512867015168E-1435', 9, 4);
    t('7.8581e+20', '785809133917993806152.766899036364576133748018', 5, 6);
    t('5.8675707566116779772315220765765490270232737179e+87109', '5.8675707566116779772315220765765490270232737179E+87109', 48, 2);
    t('8.590354486277233031075067e+4596', '8.5903544862772330310750666233E+4596', 25, 6);
    t('-6.4248009991478939026509e+24', '-6424800999147893902650879.80472855366', 23, 4);
    t('-6.825557023431606803503993e+84475', '-6.8255570234316068035039934992139772004436248114121E+84475', 25, 1);
    t('7.878290453692163901227132723658396291156275637356e+227', '7.878290453692163901227132723658396291156275637356E+227', 49, 1);
    t('8.036e-2', '0.0803509719028880560326561090873054775832971715307161481', 4, 2);
    t('3.11818e-83', '3.1181716436E-83', 6, 0);
    t('-6.496618466065e+0', '-6.496618466065219896930852265561981874833459073419402415', 13, 1);
    t('8e-424201', '7.96489731579205E-424201', 2, 5);
    t('-5.749904942276462e+73237', '-5.7499049422764612644E+73237', 16, 3);
    t('2.127655182548166e-972', '2.1276551825481656E-972', 16, 5);
    t('-9.14290951265979179589545804258e+3839895', '-9.14290951265979179589545804257745836327404953837693078575127E+3839895', 30, 4);
    t('5.516102e-8621', '5.516102294067266060370332960012972305962409219783629216446E-8621', 7, 6);
    t('-7.3598274577296714811025797387778519704e+418204', '-7.3598274577296714811025797387778519704E+418204', 40, 5);
    t('7.873343874513487761628890528252357340449966365007719401147e-82', '7.8733438745134877616288905282523573404499663650077194011462E-82', 58, 0);
    t('-9.295591101707e+4431', '-9.29559110170721703378556774593448300E+4431', 13, 4);
    t('3.956207943500324622049974268796032389e+690647', '3.956207943500324622049974268796032389E+690647', 38, 1);
    t('-6.726e-2', '-0.0672602340026', 4, 4);
    t('3.794617966936738e+258', '3.7946179669367377902038476944141649473585147688E+258', 16, 2);
    t('-9.240467782906726739501666100105084293e-26', '-9.24046778290672673950166610010508429338242220599E-26', 37, 1);
    t('-5.846062e+224770', '-5.846061751E+224770', 7, 6);
    t('2.92e-74', '2.9201E-74', 3, 6);
    t('8.2367e-90617', '8.2367E-90617', 5, 2);
    t('-5.86403631389014941e+6741979', '-5.86403631389014941E+6741979', 18, 2);
    t('-8.66858287972627583851e+8', '-866858287.97262758385100842191295475602', 22, 2);
    t('9.4729874640677367e+698017', '9.4729874640677367E+698017', 19, 3);
    t('6.10826087e+681389', '6.108260860818664977E+681389', 9, 0);
    t('3.329509588199078827e+4694', '3.329509588199078827009271571782065369E+4694', 19, 6);
    t('9.9e+93853', '9.985E+93853', 2, 3);
    t('2.5667433176e-40986', '2.5667433176603523E-40986', 11, 1);
    t('9.2615288929e+8983', '9.2615288929000232E+8983', 11, 6);
    t('-4e+6', '-4.63522E+6', 1, 2);
    t('4.434459735e+773640', '4.434459734172417332873634027119671838027810233472650221991E+773640', 10, 0);
    t('-9e-4', '-0.0009', 1, 0);
    t('-5.8748855e+548', '-5.8748855172688246709E+548', 8, 4);
    t('4e-980662', '4E-980662', 4, 6);
    t('5.39e+76523', '5.3902842093454160036251979943920075037361649404766714749E+76523', 3, 1);
    t('-7.0247270654388e-832', '-7.02472706543884872660500304923813232E-832', 14, 2);
    t('-9.0281521769748073586747024497133530800759630754870161e+4954689', '-9.028152176974807358674702449713353080075963075487016101E+4954689', 54, 5);
    t('1.506538808544156762758136e+582769', '1.506538808544156762758135815E+582769', 25, 0);
    t('2.6834e+2672', '2.683408339002603852066268794014289503E+2672', 5, 6);
    t('-3.65548035337839e-77', '-3.655480353378390475387328881640088312490302619338572897690E-77', 15, 5);
    t('-7.118e+11696', '-7.118E+11696', 5, 5);
    t('-4.2203527780623951958469e+999', '-4.220352778062395195846988E+999', 23, 1);
    t('9.89068849803e+1', '98.906884980225142', 12, 0);
    t('1e+95582', '1E+95582', 2, 4);
    t('-6.28344427124400442956585091015e+9855970', '-6.2834442712440044295658509101434613001490985570E+9855970', 30, 3);
    t('2.31153e+108', '2.3115316438236536737198421053873571514832452E+108', 6, 5);
    t('2.76208958699636632522037071701e+1200', '2.7620895869963663252203707170087505029853142566410779539E+1200', 30, 2);
    t('-5.4403558573031e+749306', '-5.44035585730305171127501625843920568655751931833135618536E+749306', 14, 5);
    t('-5e+8670265', '-5.0E+8670265', 5, 2);
    t('4.642e+5524', '4.64176E+5524', 4, 2);
    t('3e+2082', '3.02635190828878195315E+2082', 2, 5);
    t('7.9355738e+2', '793.55737142268702766', 8, 0);
    t('-4.41980021444e+73', '-4.41980021444537638784207450916942776614E+73', 12, 2);
    t('-3.998508001798430993666098478e-424', '-3.998508001798430993666098478050061182309180016E-424', 28, 4);
    t('-4.73750772595e+48807', '-4.73750772594037143E+48807', 12, 0);
    t('3.6759845e+83', '3.6759845E+83', 9, 2);
    t('-2.01180417374131e+65143', '-2.01180417374130857146531862253591E+65143', 15, 5);
    t('-5.4169238231381e+779185', '-5.416923823138046507571980670858737494990472400E+779185', 14, 0);
    t('-2.18157656705665043069e+238072', '-2.18157656705665043069E+238072', 23, 3);
    t('9.4737666e+6', '9473766.56712', 8, 4);
    t('5.71359057418893608571391378831209627661584636e+404', '5.713590574188936085713913788312096276615846355E+404', 45, 6);
    t('-6.2839845e+33041', '-6.28398447E+33041', 8, 4);
    t('9.5878592972e+2788328', '9.5878592971578016312097181298264286991923594E+2788328', 11, 2);
    t('-8.416673361685513849e+53533', '-8.41667336168551384802628961363712620465008E+53533', 19, 0);
    t('3.72111831057266918e+30', '3721118310572669188108566749501.7802186856399624954504662', 18, 1);
    t('9.15633277e+145', '9.15633277E+145', 9, 3);
    t('-7.9645919837e-10061', '-7.96459198367E-10061', 11, 3);
    t('-1.122144e+95', '-1.12214441337771251447464435257005658E+95', 7, 5);
    t('-9.6555953966e+1388205', '-9.6555953965405172450996407E+1388205', 11, 0);
    t('-5.78e-75', '-5.77558751184765500458675866E-75', 3, 6);
    t('7.3253657892323423380416805372711e+41911', '7.32536578923234233804168053727115848970644022E+41911', 32, 1);
    t('-2.35854175320395928697648982647959512565e+942', '-2.3585417532039592869764898264795951256592167123E+942', 39, 2);
    t('1.457003951233067734e+18', '1457003951233067734.8382464963', 19, 1);
    t('-3.897e+85444', '-3.8970E+85444', 6, 1);
    t('-9.8166283498424144296690737733752077809123248e+387', '-9.8166283498424144296690737733752077809123247746E+387', 44, 3);
    t('-5.1873448962904627025e+870', '-5.187344896290462702461350396588507E+870', 20, 5);
    t('3.0411078226897997226193e+2605', '3.0411078226897997226193E+2605', 26, 6);
    t('-3.8595710023381860673844604147641253598717405772283e-6269', '-3.8595710023381860673844604147641253598717405772282579003E-6269', 50, 4);
    t('-3.787025e+7672', '-3.7870252446795925614194795E+7672', 7, 2);
    t('8.6e+9', '8.58E+9', 2, 2);
    t('2.3639986e+0', '2.3639986021263097233', 9, 3);
    t('-3.31892536e-3479327', '-3.31892535925987756330661E-3479327', 9, 0);
    t('-9.8e+63', '-9.8499141125118915903867097660451754339665413793443945892E+63', 2, 1);
    t('-7.65291e-19', '-7.65290923432775701351088E-19', 7, 3);
    t('-9.575456285186607e-1058', '-9.57545628518660696019953400950015563017706103035E-1058', 17, 6);
    t('-5.4679290039144475e+6724976', '-5.4679290039144474882671380229896676846052775601632605319240E+6724976', 17, 0);
    t('8.35e-3711', '8.350143428493398625525367318282751215E-3711', 3, 5);
    t('-5.8844099637873017e+8', '-588440996.37873016103971575225987127', 17, 0);
    t('-7.31579862e+6553', '-7.31579862E+6553', 11, 5);
    t('-8e+4886', '-8.22E+4886', 1, 6);
    t('2.523413224857883289235120316146164e+467124', '2.52341322485788328923512031614616353574142E+467124', 34, 2);
    t('-6.74609e+803', '-6.74608944627069008E+803', 6, 6);
    t('6.2003249650746863467203938e+467', '6.2003249650746863467203937645809600576215166E+467', 26, 6);
    t('7.3717565407620811e+171', '7.37175654076208100853380885E+171', 17, 0);
    t('-6.86468476713e+5732', '-6.86468476713076647E+5732', 12, 2);
    t('8.4418301271e+9', '8441830127.19541', 11, 1);
    t('-6.25495707975e-409984', '-6.254957079749570821617050585476E-409984', 12, 0);
    t('-8.42365291604375397e+269', '-8.4236529160437539798707780423E+269', 18, 2);
    t('1.9037429155084166363403e+125', '1.903742915508416636340308E+125', 23, 6);
    t('5.013e+747331', '5.012638583054913389873345075643E+747331', 4, 6);
    t('-2.375645e-7987', '-2.3756454E-7987', 7, 1);
    t('-5.80942e+71831', '-5.80942E+71831', 9, 0);
    t('-6.044393037731063430015926196230302647042693925500211702e+305', '-6.044393037731063430015926196230302647042693925500211702E+305', 57, 0);
    t('-5.26350625897e-32', '-5.26350625896175741087140991895E-32', 12, 3);
    t('6.235771864e-70979', '6.23577186398920064256454E-70979', 10, 0);
    t('-1.5434147960333337e-5196814', '-1.54341479603333361902846332270185519675612896E-5196814', 17, 3);
    t('8.38882481539305134459975e+2', '838.88248153930513445997450521560709779262233633731813', 24, 4);
    t('-8.936774472712055006319e-2084199', '-8.936774472712055006319E-2084199', 23, 6);
    t('-6.823655550183522197869278026467682425427e+71407', '-6.82365555018352219786927802646768242542701681594372E+71407', 40, 1);
    t('-3e+502584', '-2.25732E+502584', 1, 0);
    t('-6.510314409159854361843517552190247454001758347622583524e+24', '-6510314409159854361843517.552190247454001758347622583523569', 55, 6);
    t('3.519769482025802975e+10690', '3.5197694820258029747577949514610498619516159E+10690', 19, 6);
    t('2.241315729471781637e+65', '2.241315729471781636566E+65', 19, 4);
    t('-8.26206188053767626050172161618e+3614', '-8.262061880537676260501721616184403382874709E+3614', 30, 5);
    t('-6.465709474959193e+80', '-6.4657094749591930264321012E+80', 17, 1);
    t('-2.7657e+3', '-2765.72763431586', 5, 5);
    t('6.652227870355e+24', '6652227870354933356473280.97387447800271', 13, 5);
    t('2.05263015138926449216743e+232', '2.052630151389264492167427072425910803E+232', 24, 2);
    t('-1.712399e+1766', '-1.712399E+1766', 7, 2);
    t('2.70649240031485243264801146373645658e+582', '2.70649240031485243264801146373645657014721816641E+582', 36, 2);
    t('-7.800583465595144148559e+5945', '-7.800583465595144148559574772E+5945', 22, 2);
    t('-2.3097e+14761', '-2.30969974E+14761', 6, 3);
    t('-6.180296929288269778433512704815503782e+9753', '-6.18029692928826977843351270481550378162585247500044951244830E+9753', 37, 6);
    t('6.6992e-4455912', '6.69922087E-4455912', 5, 3);
    t('-1.3644503e+7840', '-1.3644503E+7840', 9, 1);
    t('5.99087840045e+10551', '5.990878400449345784106227994995E+10551', 13, 2);
    t('-3.095131910463318401907525871782426e+7100', '-3.095131910463318401907525871782426666706699333241576344885E+7100', 34, 2);
    t('-4.04e+96', '-4.048114292085776040886902427462002275999088903414255E+96', 3, 1);
    t('-2.963193e+328', '-2.963192839639737E+328', 7, 5);
    t('-6.422939023e+1005', '-6.422939023E+1005', 12, 0);
    t('-1.49e+17', '-148917393095787555.2705871780235', 4, 0);
    t('3.7848495937328806738626215682874668775078e-5168843', '3.78484959373288067386262156828746687750777553624238448E-5168843', 41, 5);
    t('-1.781364423752529018e+8996727', '-1.781364423752529018124035342642141936036E+8996727', 19, 2);
    t('-1.7191244e+312', '-1.719124439527E+312', 8, 5);
    t('-2.3731618901480022986e+0', '-2.3731618901480022986477454508432084282', 20, 5);
    t('4.654760486e+40', '4.6547604855944959444151902500316274486E+40', 10, 6);
    t('7.120882477484270996042664069e+450', '7.1208824774842709960426640688127372629454188399310946682E+450', 28, 4);
    t('-2.962627047773e+5306369', '-2.962627047773E+5306369', 15, 2);
    t('-9.835212540348e+4', '-98352.12540347873262382710087145355', 13, 3);
    t('-6.638616906205051696e+170', '-6.63861690620505169599065784481214024373028E+170', 20, 6);
    t('2.36e+901064', '2.36E+901064', 4, 5);
    t('-4.69e+594', '-4.6912718632358824E+594', 3, 2);
    t('3.7e-470038', '3.67908774446338229517448994204938E-470038', 2, 2);
    t('-3.7e+91', '-3.7E+91', 2, 3);
    t('4.226765441608898006650203970757e+335', '4.2267654416088980066502039707570181001472212149696251E+335', 32, 3);
    t('5e-170400', '5.0710780789008065794569611440668882448965562E-170400', 1, 5);
    t('-1.7474543083716099e-63699', '-1.7474543083716098998995678255734E-63699', 17, 6);
    t('2.02064610751816125262953552208291284630237e+55523', '2.020646107518161252629535522082912846302369281E+55523', 42, 6);
    t('2.5373684474e+5', '253736.844743737922084692824113975', 11, 3);
    t('2.7985269489e+3317', '2.798526948999882149007619185924757010916798136E+3317', 11, 1);
    t('2.4e+966', '2.4E+966', 5, 4);
    t('-8e+2', '-750.36', 1, 4);
    t('-2.8497119933004103896211116626542633977e+2323194', '-2.849711993300410389621111662654263397695182617E+2323194', 38, 5);
    t('7.1477376604873315794448297415406515e+2', '714.77376604873315794448297415406515', 36, 0);
    t('-8.596775463927891936470624429009457842e+161', '-8.596775463927891936470624429009457842E+161', 39, 0);
    t('-9.6405634305210709e-83716', '-9.6405634305210708988E-83716', 17, 4);
    t('3.84e+3942385', '3.84E+3942385', 5, 2);
    t('2.4e+88', '2.4239116792770354106190781611172E+88', 2, 6);
    t('-5.450520013722e-3361069', '-5.4505200137216142771326146783140178951E-3361069', 13, 5);
    t('-5.71e-93', '-5.70994E-93', 5, 3);
    t('-8.52779857394291542107615755254891600211036036e+585', '-8.5277985739429154210761575525489160021103603598166390759774E+585', 45, 0);
    t('9.1971e+6', '9.19710E+6', 7, 5);
    t('1.29e+6312', '1.29E+6312', 6, 2);
    t('-5.8695924879677785162328689e+5154', '-5.8695924879677785162328688925326002351354867558904E+5154', 26, 0);
    t('-7.70624982975226143e-406', '-7.70624982975226143486613542852432483171921863E-406', 18, 5);
    t('-1.8773513e+968', '-1.87735126920528275661818766991137309736259644E+968', 8, 6);
    t('3.17e+5', '317136.797686906565813585739437429', 3, 5);
    t('-8.01918321067420976410193166e+9863', '-8.01918321067420976410193166437325974122278588369325143352E+9863', 27, 2);
    t('4.038562045624977016749e+4698014', '4.038562045624977016748955673441741727786523099732E+4698014', 22, 4);
    t('-2.699579479e+6', '-2699579.47875487814574990825', 10, 5);
    t('6.21585226039504e+648981', '6.2158522603950320618665229342330189E+648981', 15, 2);
    t('1.306247742204936303800948757e+2089', '1.306247742204936303800948756472585E+2089', 28, 2);
    t('-2.633172892759e+155', '-2.63317289275895527207863723929856439E+155', 14, 3);
    t('2.5739444994156512113239e+685', '2.573944499415651211323842803674357595739063080003150E+685', 23, 0);
    t('8.6080144842252051295812947229722915584e+2', '860.80144842252051295812947229722915583560006730725561028', 38, 5);
    t('-1.313030073031911896870975353613046659e-36649', '-1.313030073031911896870975353613046659E-36649', 38, 2);
    t('1.812755e-92', '1.81275416622397871962712E-92', 7, 0);
    t('-7.23098e+629819', '-7.23098E+629819', 6, 0);
    t('-2.4840493901e+4492529', '-2.48404939005052653840023621557847029329062681851675E+4492529', 11, 5);
    t('2.6e-1813059', '2.58142064496533988687913587399475687276497735E-1813059', 2, 2);
    t('-7.0170908483697e-12539', '-7.01709084836965862477501E-12539', 14, 3);
    t('4.6739486703652448e+710', '4.6739486703652448928485E+710', 17, 3);
    t('1.57958956033751439045441e+4', '15795.8956033751439045440186685341114868015970', 24, 0);
    t('6.1e+8551', '6.0915867964940180E+8551', 2, 0);
    t('-3.8e-83787', '-3.82E-83787', 2, 2);
    t('-8.93298e+1', '-89.329884', 6, 2);
    t('-9.206793026326099055663101869903e-8200538', '-9.206793026326099055663101869903E-8200538', 32, 5);
    t('2.2843e-232409', '2.28432119057680323514719448292795086317E-232409', 5, 6);
    t('1.88927091e+838791', '1.88927090276556198321287637811856873454042E+838791', 9, 2);
    t('1.166214080789564544702313981934268e-9', '1.166214080789564544702313981934267947846E-9', 34, 6);
    t('7.13935e-595', '7.13935E-595', 8, 3);
    t('-3.64788925572847734194185611895e-580', '-3.64788925572847734194185611895794E-580', 30, 1);
    t('-4.6673e+6021439', '-4.66733E+6021439', 5, 4);
    t('-2.94e-40405', '-2.94E-40405', 5, 4);
    t('-9.166720790664e+37966', '-9.166720790664654469463839371760E+37966', 13, 1);
    t('-3.990856292981534208234837869e+686327', '-3.9908562929815342082348378693E+686327', 28, 1);
    t('2.17137144e-87', '2.171371443962273485122E-87', 9, 4);
    t('-9.6594088922940320370848431313796880540044990028e+935', '-9.65940889229403203708484313137968805400449900270412305247E+935', 47, 3);
    t('9.27483501870838833786206228883e+48378', '9.27483501870838833786206228883164587749272E+48378', 30, 1);
    t('-4.1578235675789e+492031', '-4.1578235675788577857129742940E+492031', 14, 6);
    t('-5.74294028e+461', '-5.74294028205E+461', 9, 2);
    t('2.1451712412298e+1025', '2.1451712412298103E+1025', 14, 5);
    t('-7.0151181239817133077167649200803088e-5080319', '-7.0151181239817133077167649200803087217798089863E-5080319', 35, 0);
    t('-6.89932716670703814855e+2', '-689.932716670703814855', 21, 4);
    t('1.833e+489', '1.832505948806343263988648214428811571767718528025793084E+489', 4, 2);
    t('7.3609383192443334147561141986e+65330', '7.3609383192443334147561141985812337438063E+65330', 29, 0);
    t('9.5038035030880654053e+539890', '9.50380350308806540537718013961384213839879339517558488928450E+539890', 20, 3);
    t('-4.058205104e+1461', '-4.05820510408096591226097E+1461', 10, 5);
    t('-5.97343481415978508721681e+1212', '-5.97343481415978508721680931436543750132476965776021E+1212', 24, 0);
    t('8.62002372e+96', '8.62002372700515669E+96', 9, 1);
    t('-4.365604132927e+59723', '-4.36560413292663798362915210E+59723', 13, 3);
    t('-8.35336e+60', '-8.353359511744124663404E+60', 7, 4);
    t('-1.30258e+21384', '-1.3025801410118627389E+21384', 7, 2);
    t('-6.51081e+75833', '-6.51080772929785024834669618399388220400815097E+75833', 6, 6);
    t('8.715848092239727e+31', '87158480922397264205385014585152.7900447605', 16, 2);
    t('-1.196952647839886147738175961613e+1204364', '-1.196952647839886147738175961613225195525474477216014170E+1204364', 31, 2);
    t('8.5455215356547669660422e+6', '8545521.535654766966042184788964505650622133194', 23, 0);
    t('2.473e+55858', '2.473E+55858', 7, 5);
    t('5.1428e+7358', '5.14288841747917278E+7358', 5, 3);
    t('6.6154697386479655e+845', '6.615469738647965536950083496831E+845', 17, 6);
    t('-3.92e+81', '-3.9155798829705008905889195932928744881741190511013208869E+81', 3, 6);
    t('8.15656372610408983438241717323e+53', '8.156563726104089834382417173234876454E+53', 30, 6);
    t('6.61567423866100833742665834806051043948666e+4', '66156.7423866100833742665834806051043948665532964609', 42, 6);
    t('2.4952900173843940113e-43', '2.4952900173843940112927849799E-43', 21, 2);
    t('4.0325963895226128551361e+66307', '4.03259638952261285513607147E+66307', 23, 2);
    t('-1.552777456034285277355681e+6', '-1552777.45603428527735568089500', 25, 5);
    t('2.2e+943', '2.2257E+943', 2, 1);
    t('4.46841651e+6974731', '4.46841650506544750942218E+6974731', 9, 2);
    t('-7.37494096536042421373869500170183399e-9', '-7.374940965360424213738695001701833985251301E-9', 36, 4);
    t('-7.2e-28396', '-7.2E-28396', 5, 2);
    t('5.15093093464e+7096048', '5.15093093464E+7096048', 13, 0);
    t('-9.0514772952588311759819e+235', '-9.0514772952588311759818964082263E+235', 23, 4);
    t('-1.09e+8', '-1.09E+8', 6, 1);
    t('-2.2067011e+1', '-22.067010950767899054145914421011755452', 9, 3);
    t('3.50307027873227195053828732915844652288634e+4', '35030.7027873227195053828732915844652288634', 43, 5);
    t('7.602672902252192234143043e+660', '7.60267290225219223414304237417197202580581934E+660', 25, 2);
    t('-1.9224972180856201e+379940', '-1.922497218085620090391E+379940', 17, 4);
    t('7.538533729e+981840', '7.5385337285123302291E+981840', 10, 2);
    t('4.92876635646474938574471027359009597143886099999509e+96172', '4.928766356464749385744710273590095971438860999995085793E+96172', 51, 6);
    t('1.78188535098414155e-34', '1.78188535098414154533E-34', 18, 0);
    t('4.696133381912e-18', '4.69613338191196167066E-18', 14, 5);
    t('5.1459831124969363149181337180165525240729400788e+96', '5.1459831124969363149181337180165525240729400788E+96', 50, 4);
    t('5.4186e+0', '5.4186008235', 7, 3);
    t('-3.4375300510539e+0', '-3.4375300510539123895165', 14, 1);
    t('-6.46821098990364e+2067', '-6.46821098990363723444674881466645030969287678361E+2067', 15, 4);
    t('5.6654e+94', '5.665436101E+94', 5, 6);
    t('4.2e+948', '4.15661204964768643245084795166733E+948', 2, 2);
    t('1.3e+91', '1.36859898361106026199751888E+91', 2, 3);
    t('-7.390081539803136189503043765352316429396578e+42', '-7390081539803136189503043765352316429396578.828', 43, 2);
    t('-2.81411e+18', '-2.814109E+18', 6, 4);
    t('3.676634446286698611148238e+8344488', '3.676634446286698611148237845613270036183172E+8344488', 25, 0);
    t('-2.401712093e+746976', '-2.401712093458885488545828031381001600405156890946618043884E+746976', 10, 2);
    t('-3e+2', '-299.6865281', 3, 0);
    t('-8.992214e+797727', '-8.992214805085806948434393979874128540506027E+797727', 7, 1);
    t('1.580364867597260327354853e+6', '1580364.8675972603273548532', 25, 1);
    t('5.439153206682068544677425835649285370066e+9', '5439153206.6820685446774258356492853700660845432305218068727', 40, 6);
    t('7.645976648867677694727952045511976483074085552e-74', '7.645976648867677694727952045511976483074085552E-74', 48, 6);
    t('8.15e+58433', '8.1460198456041947E+58433', 3, 2);
    t('6.11e+73108', '6.112172181222372955961784E+73108', 3, 5);
    t('-3.083e-10101', '-3.083294170690508976033E-10101', 4, 1);
    t('-3.93e+47404', '-3.93488626896712340391634875743E+47404', 3, 5);
    t('-5.14e+89', '-5.139537438396588969394874582417822749362223847602781273426E+89', 3, 6);
    t('-6.294507394808243582094922e+4344', '-6.294507394808243582094921127257005642438227903E+4344', 25, 3);
    t('-5.70402055e+900848', '-5.70402055627616071271374610383374089680610643256942393E+900848', 9, 2);
    t('-6.639682635432503935443e-451407', '-6.6396826354325039354430045014045439067007471E-451407', 22, 4);
    t('8.483161969187252604769e+197', '8.48316196918725260476849813597607427184911E+197', 22, 0);
    t('3.25684369108469307276586206046252945e+47', '325684369108469307276586206046252945462745959174.35850', 36, 6);
    t('8.13265270647335535341164777678741734e-226361', '8.13265270647335535341164777678741733980588268690645922711E-226361', 37, 4);
    t('-5e-753', '-4.95014205114397060450783828E-753', 1, 6);
    t('-1.0201e+983', '-1.020053E+983', 5, 3);
    t('-5.7532326896335e+12', '-5753232689633.501547851280219', 14, 5);
    t('7e+5', '784885.326885902066', 1, 1);
    t('1.66704829e+47097', '1.667048298599870756743136877165580853561E+47097', 9, 1);
    t('8.990129568888134308769282072778741847097631791386e-7680', '8.990129568888134308769282072778741847097631791386E-7680', 49, 0);
    t('-8.9e-5562302', '-8.90024057E-5562302', 3, 6);
    t('-5.69888132031145710073953452975168322651202277653e-287', '-5.698881320311457100739534529751683226512022776530541614E-287', 48, 2);
    t('-8.286038e-16', '-8.286038E-16', 7, 4);
    t('4.9124905580407e+177138', '4.91249055804070421301470E+177138', 14, 4);
    t('-2.66121060761023028810909e+1', '-26.612106076102302881090906368418625582346', 24, 5);
    t('2.58670152885787988741876e-4076', '2.5867015288578798874187564322601329578916E-4076', 24, 0);
    t('4.21441011859934305176422088022e-28237', '4.21441011859934305176422088022E-28237', 33, 6);
    t('7.67048212872e+42601', '7.67048212872012401223815558E+42601', 13, 4);
    t('9.8250586500236057e+7243', '9.8250586500236057322224804085948070975486E+7243', 17, 3);
    t('-5e+14', '-596384951106423.792350994548640720653099056982989', 1, 2);
    t('-6e+1259', '-6E+1259', 1, 3);
    t('-4.3924795e-33', '-4.3924794964308305420939662515339388E-33', 9, 5);
    t('-6e+462840', '-5.07745985104741384463E+462840', 1, 3);
    t('-6.2789118650622644089635e+34', '-6.2789118650622644089635E+34', 25, 3);
    t('3.158049587698e+5380', '3.15804958769840598618714839E+5380', 13, 1);
    t('6.43e-3716', '6.43E-3716', 5, 0);
    t('-2e-92451', '-1.892140441722006657321261393975708684E-92451', 1, 0);
    t('7.17906574072797415e+90', '7.1790657407279741482260134E+90', 18, 5);
    t('-2e+94225', '-2E+94225', 4, 3);
    t('-2.4369e-3269128', '-2.4368547722014191624204604402259811123571E-3269128', 5, 4);
    t('4.5857101150245963051546392491560954e+122', '4.5857101150245963051546392491560954231118302045040595527451E+122', 35, 4);
    t('-3e+71', '-2.246042759103370E+71', 1, 3);
    t('2.195100265185621e+7722223', '2.1951002651856214225696984637889946686219288724073E+7722223', 16, 3);
    t('1.572102853648346e+251', '1.57210285364834604229672396339256078674408333595E+251', 16, 4);
    t('-1.93151e+71797', '-1.931517748303051190238322635E+71797', 6, 1);
    t('-1.339034688163837e+605111', '-1.3390346881638367E+605111', 16, 0);
    t('-9e-705671', '-9.4004563979159046E-705671', 1, 4);
    t('6.7098952181713051886080959011438515164993e+37585', '6.70989521817130518860809590114385151649928780360841684E+37585', 41, 6);
    t('-3.2354e-3', '-0.0032354', 7, 2);
    t('-7.899462898841796e+903817', '-7.89946289884179577918426159240502391417765510E+903817', 16, 6);
    t('-6.4011243962680042874053e+11814', '-6.4011243962680042874053141864169E+11814', 23, 4);
    t('-9.24911071966889081465e+5', '-924911.0719668890814643445900277875396491030252214514515', 21, 3);
    t('-4.14473261338283143e-8420808', '-4.14473261338283143172031109901722820647058311261233319275E-8420808', 18, 4);
    t('-6.7801425575567718979027099380244e+5714', '-6.780142557556771897902709938024437574E+5714', 32, 6);
    t('1.6570952968763203898e-79', '1.657095296876320389851E-79', 20, 1);
    t('4.26663226338287247584737510812355e+4673', '4.2666322633828724758473751081235477926E+4673', 33, 2);
    t('-5.98103695e+4331803', '-5.981036943494796111971553462895989911E+4331803', 9, 3);
    t('-6e+61560', '-6.037E+61560', 1, 6);
    t('7.69528964969150956959873e+328', '7.695289649691509569598738223409797697022016069E+328', 24, 1);
    t('-2.53625167914e+66', '-2.53625167914E+66', 15, 3);
    t('-1.63885720319963952278997684186091326e-36', '-1.6388572031996395227899768418609132507E-36', 36, 0);
    t('-3.258053035378732272e+583', '-3.2580530353787322710283998882E+583', 19, 0);
    t('-7.6441829e+1071', '-7.6441829E+1071', 11, 2);
    t('-4.5145586589692e+657364', '-4.514558658969169326839406258682255E+657364', 14, 5);
    t('-9.5497714880414e+3382781', '-9.549771488041403443E+3382781', 15, 4);
    t('-5.061820474200211394425193053995187e+0', '-5.061820474200211394425193053995186618267119320057', 34, 0);
    t('1.35060401154595512793e+7', '13506040.11545955127930003436389300769443170028096383', 22, 3);
    t('1.496503873486841900811467e-60', '1.49650387348684190081146749087103979E-60', 25, 3);
    t('-3.5049e+115682', '-3.50484605212E+115682', 5, 0);
    t('8.7089515262501891531126208380741028353004435678606062e-378', '8.7089515262501891531126208380741028353004435678606062E-378', 56, 1);
    t('9.4186024077623837e+92', '9.418602407762383689654E+92', 17, 0);
    t('6.8941364674693303672967786107e+4', '68941.3646746933036729677861067024283325268123569771', 29, 6);
    t('4.10513361827432312e-4052', '4.105133618274323115192649511118024473357802E-4052', 18, 4);
    t('4.7857755977234135215e+431339', '4.78577559772341352150E+431339', 20, 3);
    t('2.4e+715', '2.4147E+715', 2, 1);
    t('5.5307e+78', '5.5307E+78', 8, 4);
    t('-4.5125449107234557943408e+691', '-4.5125449107234557943408E+691', 24, 1);
    t('7.391610285e+2', '739.16102852701', 10, 3);
    t('-5.18827983709629488706479039963941464809771022983437902e-92', '-5.18827983709629488706479039963941464809771022983437902E-92', 54, 6);
    t('-3.6e+433', '-3.6102354738334252E+433', 2, 2);
    t('-6e+78', '-5.656645043472733249461060E+78', 1, 5);
    t('9.65033e+932', '9.65032182621852251426458145993391926562129609439699553593E+932', 6, 2);
    t('9.82851496057798747058e+15116', '9.8285149605779874705793E+15116', 21, 2);
    t('-6.507705908040783413623054744817431324017008203697451494178e+1510', '-6.5077059080407834136230547448174313240170082036974514941780E+1510', 62, 4);
    t('7.31701757847968782e-3229', '7.31701757847968781736913825509767286310E-3229', 18, 0);
    t('2.65091439513147953022e+684312', '2.65091439513147953021798611890792063348113104511624309239E+684312', 21, 5);
    t('-2.8692332435615892e-430', '-2.86923324356158914E-430', 17, 3);
    t('-1.006616006e-956087', '-1.00661600602E-956087', 10, 4);
    t('1.92212998755673874e+1937', '1.922129987556738737627E+1937', 18, 6);
    t('2.6514334921383079495e+6604', '2.6514334921383079494369918620203365264678555406E+6604', 20, 0);
    t('3.8840929e+61389', '3.88409296171974425191643287912667794203003605892200422E+61389', 8, 1);
    t('-9.6e+5', '-955550.030950952412107393018630', 2, 6);
    t('-4.360258509123578738190855066155343581277868213e-204', '-4.360258509123578738190855066155343581277868213186E-204', 46, 2);
    t('7.93587844612633e+65', '7.93587844612632539E+65', 15, 0);
    t('-6.566e+9', '-6565955901.927666035665215452916733618436776', 5, 6);
    t('4.7489e+96615', '4.7489E+96615', 8, 3);
    t('-7.22e+75', '-7.2206097906381336760706750689175620234E+75', 3, 4);
    t('2.5988722886726744954187e+8906825', '2.5988722886726744954187797646143708446390506959288313169E+8906825', 23, 3);
    t('5.1327825053441265e-990931', '5.1327825053441265E-990931', 19, 3);
    t('8.8452438056e-3', '0.008845243805565967754734179236657745778124546299517', 11, 4);
    t('-7e-35', '-7E-35', 3, 5);
    t('-5.7e+3097', '-5.65718243169E+3097', 2, 6);
    t('6.6733103017e+614', '6.673310301670678818128606E+614', 11, 6);
    t('-3.940103161994451647164790244567630107094e+3', '-3940.1031619944516471647902445676301070939713', 41, 5);
    t('4.54062812216763803545271124622652915451935562e+654724', '4.540628122167638035452711246226529154519355621E+654724', 45, 6);
    t('-6.007e-73194', '-6.00702232371226969584235817986037051734773176917351E-73194', 4, 2);
    t('-3.03638503699426979082181610276510596852067e-46', '-3.03638503699426979082181610276510596852067E-46', 42, 6);
    t('7.64349686049786e+558390', '7.6434968604978624E+558390', 15, 4);
    t('-8e+3', '-8E+3', 3, 6);
    t('-3.075125798647937824005405180307721116e+5676881', '-3.07512579864793782400540518030772111513187740936866E+5676881', 37, 0);
    t('5.504806134915e+9562', '5.50480613491507764E+9562', 13, 1);
    t('9.4e+98707', '9.44307094846E+98707', 2, 5);
    t('4.569e-35230', '4.56929146033544E-35230', 4, 1);
    t('-2.1831297388e-7888', '-2.1831297387682691574094348E-7888', 11, 5);
    t('-7.0800418e-5', '-0.0000708004178300999409', 8, 3);
    t('-7.3921436155883e+826513', '-7.3921436155882122161E+826513', 14, 3);
    t('-5.222e+508963', '-5.2219582E+508963', 5, 0);
    t('3.51974665e-2893556', '3.5197466470389917746147753109557223867959915E-2893556', 9, 4);
    t('-2.7446325925660416729023e+5', '-274463.25925660416729022807303552330', 23, 0);
    t('-2.73411982275259828e+1226189', '-2.734119822752598279649003E+1226189', 19, 3);
    t('7.4741e+82', '7.4741E+82', 7, 4);
    t('-1.74519e+984', '-1.74518594E+984', 6, 6);
    t('-2.2928505636e+97', '-2.2928505635707176806310118376735248777703802E+97', 11, 3);
    t('7.3622504397649290426018e+5905', '7.3622504397649290426018E+5905', 24, 2);
    t('-7.3122e+2027', '-7.312107973539552079E+2027', 5, 0);
    t('7.511082e+43', '7.511082E+43', 7, 1);
});

