//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Support/Type/Unit.cpp
//! @brief     Implements class Unit
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Support/Type/Unit.h"
#include "Base/Const/Units.h"
#include "Base/Util/Assert.h"
#include <QString>

double convert(double d, Unit from, Unit to)
{
    if (from == to)
        return d;

    if (from == Unit::angstrom && to == Unit::nanometer)
        return d / 10.0;

    if (from == Unit::nanometer && to == Unit::angstrom)
        return d * 10.0;

    if (from == Unit::angstrom2 && to == Unit::nanometer2)
        return d / 100.0;

    if (from == Unit::nanometer2 && to == Unit::angstrom2)
        return d * 100.0;

    if (from == Unit::angstromMinus2 && to == Unit::nanometerMinus2)
        return d * 100.0;

    if (from == Unit::nanometerMinus2 && to == Unit::angstromMinus2)
        return d / 100.0;

    if (from == Unit::radiant && to == Unit::degree)
        return Units::rad2deg(d);

    if (from == Unit::degree && to == Unit::radiant)
        return Units::deg2rad(d);

    if (from == Unit::other || to == Unit::other) {
        ASSERT(false); // no conversion possible
        return d;
    }

    ASSERT(false); // no conversion implemented
    return d;
}

QString unitAsString(const Unit& unit)
{
    switch (unit) {
    case Unit::unitless:
        return "";
    case Unit::nanometer:
        return "nm";
    case Unit::nanometer2:
        return u8"nm\u00B2";
    case Unit::nanometerMinus2:
        return u8"1/nm\u00B2";
    case Unit::angstrom:
        return u8"\u00c5";
    case Unit::angstrom2:
        return u8"\u00c5\u00B2";
    case Unit::angstromMinus2:
        return u8"1/\u00c5\u00B2";
    case Unit::degree:
        return "°";
    case Unit::radiant:
        return "rad";
    case Unit::other:
        ASSERT(false); // this function should not be called for Unit::other
        return "";
    default:
        ASSERT(false); // no string implemented
        return "";
    }
}
