/* $Id: fors_remove_bias.c,v 1.7 2013-04-24 14:14:14 cgarcia Exp $
 *
 * This file is part of the FORS Data Reduction Pipeline
 * Copyright (C) 2002-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2013-04-24 14:14:14 $
 * $Revision: 1.7 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <math.h>
#include <cpl.h>
#include <moses.h>
#include <fors_dfs.h>

static int fors_remove_bias_create(cpl_plugin *);
static int fors_remove_bias_exec(cpl_plugin *);
static int fors_remove_bias_destroy(cpl_plugin *);
static int fors_remove_bias(cpl_parameterlist *, cpl_frameset *);

static char fors_remove_bias_description[] =
"This recipe is used to subtract the master bias, produced by the recipe\n"
"fors_bias, from one raw data frame. The overscan regions, if present, are\n"
"used to compensate for variations of the bias level between master bias\n"
"and input raw frame. The overscan regions are then trimmed from the result.\n"
"In the table below the MXU acronym can be alternatively read as MOS and\n"
"LSS.\n\n"
"Input files:\n\n"
"  DO category:               Type:       Explanation:         Required:\n"
"  LAMP_MXU\n"
"  or SCIENCE_MXU\n"
"  or STANDARD_MXU            Raw         Raw data frame          Y\n"
"  MASTER_BIAS                Calib       Master bias frame       Y\n\n"
"Output files:\n\n"
"  DO category:               Data type:  Explanation:\n"
"  LAMP_UNBIAS_MXU\n"
"  or SCIENCE_UNBIAS_MXU\n"
"  or STANDARD_UNBIAS_MXU     FITS image  Bias subtracted frame\n\n";

#define fors_remove_bias_exit(message)        \
{                                             \
if (message) cpl_msg_error(recipe, message);  \
cpl_image_delete(raw_image);                  \
cpl_image_delete(master_bias);                \
cpl_propertylist_delete(header);              \
cpl_table_delete(overscans);                  \
cpl_msg_indent_less();                        \
return -1;                                    \
}

#define fors_remove_bias_exit_memcheck(message) \
{                                               \
if (message) cpl_msg_info(recipe, message);     \
printf("free raw_image (%p)\n", raw_image);     \
cpl_image_delete(raw_image);                    \
printf("free master_bias (%p)\n", master_bias); \
cpl_image_delete(master_bias);                  \
printf("free header (%p)\n", header);           \
cpl_propertylist_delete(header);                \
printf("free overscans (%p)\n", overscans);     \
cpl_table_delete(overscans);                    \
cpl_msg_indent_less();                          \
return 0;                                       \
}


/**
 * @brief    Build the list of available plugins, for this module. 
 *
 * @param    list    The plugin list
 *
 * @return   0 if everything is ok, -1 otherwise
 *
 * Create the recipe instance and make it available to the application 
 * using the interface. This function is exported.
 */

int cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe );
    cpl_plugin *plugin = &recipe->interface;

    cpl_plugin_init(plugin,
                    CPL_PLUGIN_API,
                    FORS_BINARY_VERSION,
                    CPL_PLUGIN_TYPE_RECIPE,
                    "fors_remove_bias",
                    "Subtract bias from input frame",
                    fors_remove_bias_description,
                    "Carlo Izzo",
                    PACKAGE_BUGREPORT,
    "This file is currently part of the FORS Instrument Pipeline\n"
    "Copyright (C) 2002-2010 European Southern Observatory\n\n"
    "This program is free software; you can redistribute it and/or modify\n"
    "it under the terms of the GNU General Public License as published by\n"
    "the Free Software Foundation; either version 2 of the License, or\n"
    "(at your option) any later version.\n\n"
    "This program is distributed in the hope that it will be useful,\n"
    "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
    "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the\n"
    "GNU General Public License for more details.\n\n"
    "You should have received a copy of the GNU General Public License\n"
    "along with this program; if not, write to the Free Software Foundation,\n"
    "Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA\n",
                    fors_remove_bias_create,
                    fors_remove_bias_exec,
                    fors_remove_bias_destroy);

    cpl_pluginlist_append(list, plugin);
    
    return 0;
}


/**
 * @brief    Setup the recipe options    
 *
 * @param    plugin  The plugin
 *
 * @return   0 if everything is ok
 *
 * Defining the command-line/configuration parameters for the recipe.
 */

static int fors_remove_bias_create(cpl_plugin *plugin)
{
    cpl_recipe    *recipe;
/* Uncomment in case parameters are defined
    cpl_parameter *p;
*/

    /* 
     * Check that the plugin is part of a valid recipe 
     */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) 
        recipe = (cpl_recipe *)plugin;
    else 
        return -1;

    /* 
     * Create the (empty) parameters list in the cpl_recipe object 
     */

    recipe->parameters = cpl_parameterlist_new(); 

    return 0;
}


/**
 * @brief    Execute the plugin instance given by the interface
 *
 * @param    plugin  the plugin
 *
 * @return   0 if everything is ok
 */

static int fors_remove_bias_exec(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    
    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) 
        recipe = (cpl_recipe *)plugin;
    else 
        return -1;

    return fors_remove_bias(recipe->parameters, recipe->frames);
}


/**
 * @brief    Destroy what has been created by the 'create' function
 *
 * @param    plugin  The plugin
 *
 * @return   0 if everything is ok
 */

static int fors_remove_bias_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    
    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) 
        recipe = (cpl_recipe *)plugin;
    else 
        return -1;

    cpl_parameterlist_delete(recipe->parameters); 

    return 0;
}


/**
 * @brief    Interpret the command line options and execute the data processing
 *
 * @param    parlist     The parameters list
 * @param    frameset    The set-of-frames
 *
 * @return   0 if everything is ok
 */

static int fors_remove_bias(cpl_parameterlist *parlist, cpl_frameset *frameset)
{

    const char *recipe = "fors_remove_bias";


    /*
     * CPL objects
     */

    cpl_image        *raw_image   = NULL;
    cpl_image        *master_bias = NULL;
    cpl_image        *dummy       = NULL;
    cpl_table        *overscans   = NULL;
    cpl_propertylist *header      = NULL;

    /*
     * Auxiliary variables
     */

    char        version[80];
    const char *master_bias_tag = "MASTER_BIAS";
    const char *raw_image_tag;
    const char *pro_image_tag;
    char       *instrume = NULL;
    int         lamp_mxu;
    int         lamp_mos;
    int         lamp_lss;
    int         science_mxu;
    int         science_mos;
    int         science_lss;
    int         standard_mxu;
    int         standard_mos;
    int         standard_lss;
    int         nbias, nframe;


    cpl_msg_set_indentation(2);


    cpl_msg_info(recipe, "Check input set-of-frames:");
    cpl_msg_indent_more();

    nbias = cpl_frameset_count_tags(frameset, master_bias_tag);
    if (nbias == 0) {
        cpl_msg_error(recipe, "Missing required input: %s", master_bias_tag);
        fors_remove_bias_exit(NULL);
    }
    if (nbias > 1) {
        cpl_msg_error(recipe, "Too many in input (%d > 1): %s", 
                      nbias, master_bias_tag);
        fors_remove_bias_exit(NULL);
    }

    nframe  = lamp_mxu     = cpl_frameset_count_tags(frameset, "LAMP_MXU");
    nframe += lamp_mos     = cpl_frameset_count_tags(frameset, "LAMP_MOS");
    nframe += lamp_lss     = cpl_frameset_count_tags(frameset, "LAMP_LSS");
    nframe += science_mxu  = cpl_frameset_count_tags(frameset, "SCIENCE_MXU");
    nframe += science_mos  = cpl_frameset_count_tags(frameset, "SCIENCE_MOS");
    nframe += science_lss  = cpl_frameset_count_tags(frameset, "SCIENCE_LSS");
    nframe += standard_mxu = cpl_frameset_count_tags(frameset, "STANDARD_MXU");
    nframe += standard_mos = cpl_frameset_count_tags(frameset, "STANDARD_MOS");
    nframe += standard_lss = cpl_frameset_count_tags(frameset, "STANDARD_LSS");

    if (nframe == 0) {
        fors_remove_bias_exit("Missing required input raw frame");
    }
    if (nframe > 1) {
        cpl_msg_error(recipe, "Too many input raw frames (%d > 1)", nframe);
        fors_remove_bias_exit(NULL);
    }

    if (lamp_mxu) {
        raw_image_tag = "LAMP_MXU";
        pro_image_tag = "LAMP_UNBIAS_MXU";
    }
    else if (lamp_mos) {
        raw_image_tag = "LAMP_MOS";
        pro_image_tag = "LAMP_UNBIAS_MOS";
    }
    else if (lamp_lss) {
        raw_image_tag = "LAMP_LSS";
        pro_image_tag = "LAMP_UNBIAS_LSS";
    }
    else if (science_mxu) {
        raw_image_tag = "SCIENCE_MXU";
        pro_image_tag = "SCIENCE_UNBIAS_MXU";
    }
    else if (science_mos) {
        raw_image_tag = "SCIENCE_MOS";
        pro_image_tag = "SCIENCE_UNBIAS_MOS";
    }
    else if (science_lss) {
        raw_image_tag = "SCIENCE_LSS";
        pro_image_tag = "SCIENCE_UNBIAS_LSS";
    }
    else if (standard_mxu) {
        raw_image_tag = "STANDARD_MXU";
        pro_image_tag = "STANDARD_UNBIAS_MXU";
    }
    else if (standard_mos) {
        raw_image_tag = "STANDARD_MOS";
        pro_image_tag = "STANDARD_UNBIAS_MOS";
    }
    else if (standard_lss) {
        raw_image_tag = "STANDARD_LSS";
        pro_image_tag = "STANDARD_UNBIAS_LSS";
    }

    if (!dfs_equal_keyword(frameset, "ESO DET CHIP1 ID")) 
        fors_remove_bias_exit("Input frames are not from the same chip");

    header = dfs_load_header(frameset, raw_image_tag, 0);

    if (header == NULL) {
        cpl_msg_error(recipe, "Cannot load header of %s frame", raw_image_tag);
        fors_remove_bias_exit(NULL);
    }

    instrume = (char *)cpl_propertylist_get_string(header, "INSTRUME");
    if (instrume == NULL) {
        cpl_msg_error(recipe, "Missing keyword INSTRUME in %s header", 
                      raw_image_tag);
        fors_remove_bias_exit(NULL);
    }

    if (instrume[4] == '1')
        snprintf(version, 80, "%s/%s", "fors1", VERSION);
    if (instrume[4] == '2')
        snprintf(version, 80, "%s/%s", "fors2", VERSION);

    cpl_msg_indent_less();
    cpl_msg_info(recipe, "Load input frames:");
    cpl_msg_indent_more();

    master_bias = dfs_load_image(frameset, 
                                 master_bias_tag, CPL_TYPE_FLOAT, 0, 1);
    if (master_bias == NULL)
        fors_remove_bias_exit("Cannot load master bias");

    raw_image = dfs_load_image(frameset, 
                               raw_image_tag, CPL_TYPE_FLOAT, 0, 0);
    if (raw_image == NULL) {
        cpl_msg_error(recipe, "Cannot load %s frame", raw_image_tag);
        fors_remove_bias_exit(NULL);
    }

    overscans = mos_load_overscans_vimos(header, 1);

    cpl_msg_indent_less();
    cpl_msg_info(recipe, "Subtract the master bias from input %s...", 
                 raw_image_tag);
    cpl_msg_indent_more();

    dummy = mos_remove_bias(raw_image, master_bias, overscans);
    cpl_table_delete(overscans); overscans = NULL;
    cpl_image_delete(master_bias); master_bias = NULL;
    cpl_image_delete(raw_image); raw_image = dummy;

    if (raw_image == NULL) {
        cpl_msg_error(recipe, "Cannot remove bias from %s frame", 
                      raw_image_tag);
        fors_remove_bias_exit(NULL);
    }

    cpl_msg_indent_less();

    if (dfs_save_image(frameset, raw_image, pro_image_tag,
                       header, parlist, recipe, version))
        fors_remove_bias_exit(NULL);

    cpl_propertylist_delete(header); header = NULL;
    cpl_image_delete(raw_image); raw_image = NULL;

    return 0;
}
