/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*
* ## Notice
*
* The original C code, long comment, copyright, license, and constants are from [Cephes]{@link http://www.netlib.org/cephes}. The implementation follows the original, but has been modified for JavaScript.
*
* ```text
* Copyright 1984, 1991, 2000 by Stephen L. Moshier
*
* Some software in this archive may be from the book _Methods and Programs for Mathematical Functions_ (Prentice-Hall or Simon & Schuster International, 1989) or from the Cephes Mathematical Library, a commercial product. In either event, it is copyrighted by the author. What you see here may be used freely but it comes with no support or guarantee.
*
* Stephen L. Moshier
* moshier@na-net.ornl.gov
* ```
*/

'use strict';

// MODULES //

var floor = require( './../../../../base/special/floor' );
var ldexp = require( './../../../../base/special/ldexp' );
var isnan = require( './../../../../base/assert/is-nan' );
var MAXL10 = require( '@stdlib/constants/float64/max-base10-exponent' );
var MINL10 = require( '@stdlib/constants/float64/min-base10-exponent' );
var PINF = require( '@stdlib/constants/float64/pinf' );
var polyvalP = require( './polyval_p.js' );
var polyvalQ = require( './polyval_q.js' );


// VARIABLES //

var LOG210 = 3.32192809488736234787e0;
var LG102A = 3.01025390625000000000e-1;
var LG102B = 4.60503898119521373889e-6;


// MAIN //

/**
* Returns `10` raised to the `x` power.
*
* ## Method
*
* -   Range reduction is accomplished by expressing the argument as \\( 10^x = 2^n 10^f \\), with \\( |f| < 0.5 log_{10}(2) \\). The Pade' form
*
*     ```tex
*     1 + 2x \frac{P(x^2)}{Q(x^2) - P(x^2)}
*     ```
*
*     is used to approximate \\( 10^f \\).
*
*
* ## Notes
*
* -   Relative error:
*
*     | arithmetic | domain      | # trials | peak    | rms     |
*     |:----------:|:-----------:|:--------:|:-------:|:-------:|
*     | IEEE       | -307,+307   |  30000   | 2.2e-16 | 5.5e-17 |
*
*
* @param {number} x - input value
* @returns {number} function value
*
* @example
* var v = exp10( 3.0 );
* // returns 1000.0
*
* @example
* var v = exp10( -9.0 );
* // returns 1.0e-9
*
* @example
* var v = exp10( 0.0 );
* // returns 1.0
*
* @example
* var v = exp10( NaN );
* // returns NaN
*/
function exp10( x ) {
	var px;
	var xx;
	var n;

	if ( isnan( x ) ) {
		return x;
	}
	if ( x > MAXL10 ) {
		return PINF;
	}
	if ( x < MINL10 ) {
		return 0.0;
	}

	// Express 10^x = 10^g 2^n = 10^g 10^( n log10(2) ) = 10^( g + n log10(2) )
	px = floor( (LOG210*x) + 0.5 );
	n = px;
	x -= px * LG102A;
	x -= px * LG102B;

	// Rational approximation for exponential of the fractional part: 10^x = 1 + 2x P(x^2)/( Q(x^2) - P(x^2) )
	xx = x * x;
	px = x * polyvalP( xx );
	x = px / ( polyvalQ( xx ) - px );
	x = 1.0 + ldexp( x, 1 );

	// Multiply by power of 2:
	return ldexp( x, n );
}


// EXPORTS //

module.exports = exp10;
