///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file DisplacementDataChannel.h
 * \brief Contains the definition of the AtomViz::DisplacementDataChannel class.
 */

#ifndef __DISPLACEMENT_DATA_CHANNEL_H
#define __DISPLACEMENT_DATA_CHANNEL_H

#include <atomviz/AtomViz.h>
#include "DataChannel.h"

#include <core/scene/animation/controller/Controller.h>

namespace AtomViz {

/**
 * \brief The data channel stores the atomic displacement vectors as well as several important parameters that affect
 *        the display of displacement vectors.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT DisplacementDataChannel : public DataChannel
{
	Q_OBJECT

public:

	/// \brief Serialization constructor that creates an empty data channel.
	///
	/// \note This constructor is only used when a data channel is loaded from a scene file.
	///       It must not be used to create a new data channel.
	DisplacementDataChannel(bool isLoading = false);

	/// \brief Constructor that creates a standard data channel.
	/// \param which Specifies which standard data channel should be created.
	///              This should be DataChannelIdentifier::DisplacementChannel.
	///
	/// Data type, component count and name are automatically chosen by this constructor.
	DisplacementDataChannel(DataChannelIdentifier which);

	/// \brief Renders the channel' contents in a viewport.
	/// \param time The current animation time.
	/// \param vp The viewport into which the channel should be rendered.
	/// \param atoms The AtomsObject to which this DataChannel belongs to.
	/// \param contextNode The scene object the AtomsObject belongs to.
	virtual void render(TimeTicks time, Viewport* vp, AtomsObject* atoms, ObjectNode* contextNode);

	/// \brief Renders the channel's contents in high-quality mode to an offscreen buffer.
	/// \param time The current animation time.
	/// \param atoms The AtomsObject to which this DataChannel belongs to.
	/// \param view Describes the projection parameters.
	/// \param contextNode The scene object the AtomsObject belongs to.
	/// \param imageWidth The width of the image buffer in pixels.
	/// \param imageHeight The height of the image buffer in pixels.
	/// \param glcontext The OpenGL rendering context.
	virtual void renderHQ(TimeTicks time, AtomsObject* atoms, const CameraViewDescription& view, ObjectNode* contextNode, int imageWidth, int imageHeight, Window3D* glcontext);

	/// \brief Returns the bounding box of the channel's geometry when rendered in the viewports.
	/// \param time The current animation time.
	/// \param atoms The AtomsObject to which this DataChannel belongs to.
	/// \param contextNode The scene object the AtomsObject belongs to.
	/// \param validityInterval This is used to return the validity interval during which the bounding box doesn't change.
	/// \return The bounding box of the rendered geometry or an empty box if the channel is not rendered in the viewports.
	virtual Box3 boundingBox(TimeTicks time, AtomsObject* atoms, ObjectNode* contextNode, TimeInterval& validityInterval);

	/// \brief Lets the channel clear all its internal caches.
	///
	/// This method is automatically invoked for each DataChannel by the AtomsObject::invalidate()
	/// method. It informs the data channel that the AtomsObject it belongs to has
	/// changed in some way.
	virtual void clearCaches();

private:

	/// Initializes the object. This is called from the constructors.
	void init(bool isLoading);

	/// Renders the displacement vectors.
	void renderDisplacements(TimeTicks time, AtomsObject* atoms, bool isPerspective, Vector3 viewDir, const Point3& viewPoint, Viewport* vp);

	/// Controls the color of the displacement vectors.
	ReferenceField<VectorController> _arrowColor;

	/// Controls the width of the arrows in world units.
	ReferenceField<FloatController> _arrowWidth;

	/// Controls the scaling factor applied to the displacement vectors.
	ReferenceField<FloatController> _scalingFactor;

	/// Controls whether solid or flat arrows are being rendered.
	PropertyField<bool> _solidArrows;

	/// Controls the flipping of the arrow direction.
	PropertyField<bool> _reverseArrowDirection;

	/// Controls the flipping of the vectors.
	PropertyField<bool> _flipDisplacementVectors;

	/// This is the cache array that contains the displacement line segments for viewport rendering.
	QVector<Point3> displacementLines;

	/// The bounding box of the displacement lines vertices.
	Box3 displacementLinesBB;

	/// Indicates if the line segments have to be updated.
	TimeInterval displacementLinesValidity;

	DECLARE_SERIALIZABLE_PLUGIN_CLASS(DisplacementDataChannel)
	DECLARE_REFERENCE_FIELD(_arrowColor)
	DECLARE_REFERENCE_FIELD(_arrowWidth)
	DECLARE_REFERENCE_FIELD(_scalingFactor)
	DECLARE_PROPERTY_FIELD(_solidArrows)
	DECLARE_PROPERTY_FIELD(_reverseArrowDirection)
	DECLARE_PROPERTY_FIELD(_flipDisplacementVectors)
};

/**
 * \brief A properties editor for the DisplacementDataChannel class.
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT DisplacementDataChannelEditor : public PropertiesEditor
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

private:

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(DisplacementDataChannelEditor)
};

};	// End of namespace AtomViz

#endif // __DISPLACEMENT_DATA_CHANNEL_H
