///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __OVITO_PROPERTY_FIELD_H
#define __OVITO_PROPERTY_FIELD_H

#include <core/Core.h>
#include <core/plugins/PluginClass.h>
#include <core/undo/UndoManager.h>
#include <core/reference/RefMaker.h>

namespace Core {

class RefTarget;					// defined in RefTarget.h
class ParameterUnit;				// defined in ParameterUnit.h
class PropertyFieldDescriptor;		// defined below

/******************************************************************************
* RefMaker derived classes must use this helper class to store their
* properties.
******************************************************************************/
class PropertyFieldBase
{
public:
	/// One has to call init() once to fully setup this property field.
	PropertyFieldBase() : _owner(NULL), _descriptor(NULL) {}

	/// Connects the property field to its owning RefMaker derived class.
	/// This function must be called in the constructor of the RefMaker derived class for each of its property fields.
	inline void init(RefMaker* owner, PropertyFieldDescriptor* descriptor) {
		OVITO_ASSERT_MSG(owner != NULL, "PropertyFieldBase::init()", "The PropertyField must be initialized with a non-NULL owner.");
		OVITO_ASSERT_MSG(descriptor != NULL, "PropertyFieldBase::init()", "The PropertyField must be initialized with a non-NULL descriptor.");
		OVITO_ASSERT_MSG(this->_owner == NULL, "PropertyFieldBase::init()", "The PropertyField has already been initialized.");
		this->_owner = owner;
		this->_descriptor = descriptor;
	}

	/// Returns the owner of this property field.
	inline RefMaker* owner() const {
		OVITO_ASSERT_MSG(_owner != NULL, "PropertyFieldBase", "The PropertyField object has not been initialized yet.");
		return _owner;
	}

	/// Returns descriptor of this property field.
	inline PropertyFieldDescriptor* descriptor() const {
		OVITO_ASSERT_MSG(_descriptor != NULL, "PropertyFieldBase", "The PropertyField object has not been initialized yet.");
		return _descriptor;
	}

protected:

	/// Sends the REFTARGET_CHANGED message or another to the dependents of the field owner.
	CORE_DLLEXPORT void sendChangeNotification(int messageType = REFTARGET_CHANGED);

	/// Calls the RefMaker::onPropertyFieldValueChanged() method on the field owner.
	void notifyPropertyFieldValueChanged() { owner()->onPropertyFieldValueChanged(*descriptor()); }

private:
	/// The reference maker this object is a member of.
	/// This will be initialized after a call to init().
	RefMaker* _owner;
	/// The descriptor of this property field.
	PropertyFieldDescriptor* _descriptor;
};

/******************************************************************************
* An instance of this helper class stores a non-animatable property value
* with a primitive type.
******************************************************************************/
template<typename property_data_type, typename qvariant_data_type = property_data_type, int additionalChangeMessage = 0>
class PropertyField : public PropertyFieldBase
{
public:
	typedef property_data_type property_type;
	typedef qvariant_data_type qvariant_type;

	/// Default constructor that does not initialize the property value.
	PropertyField() : PropertyFieldBase() {}

	/// Constructor that assigns an initial value to the property.
	PropertyField(const property_type& value) : PropertyFieldBase(), _value(value) {}

	/// Constructor that calls the internal value constructor with one parameter.
	template<typename K1>
	PropertyField(K1 param1) : PropertyFieldBase(), _value(param1) {}

	/// Constructor that calls the internal value constructor with two parameters.
	template<typename K1, typename K2>
	PropertyField(K1 param1, K2 param2) : PropertyFieldBase(), _value(param1, param2) {}

	/// Constructor that calls the internal value constructor with three parameters.
	template<typename K1, typename K2, typename K3>
	PropertyField(K1 param1, K2 param2, K3 param3) : PropertyFieldBase(), _value(param1, param2, param3) {}

	/// Constructor that calls the internal value constructor with four parameters.
	template<typename K1, typename K2, typename K3, typename K4>
	PropertyField(K1 param1, K2 param2, K3 param3, K4 param4) : PropertyFieldBase(), _value(param1, param2, param3, param4) {}

	/// Cast the property field to the property value.
	inline operator const property_type&() const { return _value; }

	/// Changes the value of the property. Handles undo and sends a notification message.
	inline PropertyField& operator=(const property_type& newValue) {
		if(_value == newValue) return *this;
		if(UNDO_MANAGER.isRecording() && descriptor()->automaticUndo())
			UNDO_MANAGER.addOperation(new PropertyChangeOperation(*this));
		_value = newValue;
		notifyPropertyFieldValueChanged();
		sendChangeNotification();
		if(additionalChangeMessage != 0)
			sendChangeNotification(additionalChangeMessage);
		return *this;
	}

	/// Changes the value of the property. Handles undo and sends a notification message.
	inline PropertyField& operator=(const QVariant& newValue) {
		OVITO_ASSERT_MSG(newValue.canConvert<qvariant_type>(), "PropertyField assignment", "The assigned QVariant value cannot be converted to the data type of the property field.");
		return ((*this) = (property_type)qVariantValue<qvariant_type>(newValue));
	}

	/// Returns the internal value stored in this property field.
	inline const property_type& value() const { return _value; }

	/// Returns the internal value stored in this property field as a QVariant.
	inline operator QVariant() const { return qVariantFromValue<qvariant_type>((qvariant_type)_value); }

	/// Saves the property's value to a stream.
	inline void saveToStream(SaveStream& stream) const { stream << _value; }

	/// Loads the property's value from a stream.
	inline void loadFromStream(LoadStream& stream) { stream >> _value; }

private:

	/// This undo class records a change to the property value.
	class PropertyChangeOperation : public UndoableOperation {
	public:
		PropertyChangeOperation(PropertyField& _field) : field(_field), object(_field.owner()) {
			// Make a copy of the current property value.
			oldValue = _field;
		}
		/// Restores the old property value.
		virtual void undo() {
			// Swap old value and current property value.
			property_type temp = field;
			field = oldValue;
			oldValue = temp;
		}
		/// Re-apply the change, assuming that it has been undone.
		virtual void redo() { undo(); }

	private:
		/// The object whose property has been changed.
		PluginClass::SmartPtr object;
		/// The property field that has been changed.
		PropertyField& field;
		/// The old value of the property.
		property_type oldValue;
	};

	/// The internal property value.
	property_type _value;
};

/******************************************************************************
* An instance of this helper class stores a pointer to a RefTarget derived class.
******************************************************************************/
class SingleReferenceFieldBase : public PropertyFieldBase
{
public:
	/// Default constructor that initializes the internal pointer to NULL.
	/// One has to call init() once to fully setup this reference field.
	SingleReferenceFieldBase() : PropertyFieldBase() {}

	/// Destructor that resets the reference before the object dies.
	~SingleReferenceFieldBase() {
		// The internal intrusive pointer must be cleared first before the reference count
		// to the target go to zero to prevent infinite recursion.
		intrusive_ptr<RefTarget> null_ptr;
		pointer.swap(null_ptr);
	}

	/// Returns the RefTarget pointer.
	inline operator RefTarget*() const {
		OVITO_ASSERT_MSG(owner() != NULL, "ReferenceField pointer operator", "The ReferenceField object has not been initialized yet.");
		return pointer.get();
	}

protected:
	/// The actual pointer to the reference target.
	intrusive_ptr<RefTarget> pointer;
	/// Replaces the reference target.
	CORE_DLLEXPORT void setValue(RefTarget* newTarget);

	friend class RefMaker;
	friend class RefTarget;
};

/******************************************************************************
* This is the templated version of the SingleReferenceFieldBase helper class.
* It stores a pointer to a RefTarget derived class given as the template
* parameter.
******************************************************************************/
template<typename RefTargetType>
class ReferenceField : public SingleReferenceFieldBase
{
public:
	/// Constructor that initializes the internal pointer to NULL.
	ReferenceField() : SingleReferenceFieldBase() {}

	/// Read access to the RefTarget derived pointer.
	operator RefTargetType*() const { return (RefTargetType*)pointer.get(); }

	/// Write access to the RefTarget pointer. Changes the value of the reference field.
	/// The old reference target will be released and the new reference target
	/// will be bound to this reference field.
	/// This operator automatically handles undo so the value change can be undone.
	RefTargetType* operator=(RefTargetType* newPointer) {
		setValue(newPointer);
		OVITO_ASSERT(pointer.get() == newPointer);
		return (RefTargetType*)pointer.get();
	}

	/// Write access to the RefTarget pointer. Changes the value of the reference field.
	/// The old reference target will be released and the new reference target
	/// will be bound to this reference field.
	/// This operator automatically handles undo so the value change can be undone.
	RefTargetType* operator=(const intrusive_ptr<RefTargetType>& newPointer) {
		setValue(newPointer.get());
		OVITO_ASSERT(pointer == newPointer);
		return (RefTargetType*)pointer.get();
	}

	/// Overloaded arrow operator; implements pointer semantics.
	/// Just use this operator as you would with a normal C++ pointer.
	RefTargetType* operator->() const {
		OVITO_ASSERT_MSG(pointer, "ReferenceField operator->", QString("Tried to make a call to a NULL pointer. Reference field '%1' of class %2").arg(QString(descriptor()->identifier()), descriptor()->definingClass()->name()).toLocal8Bit().constData());
		return (RefTargetType*)pointer.get();
	}

	/// Dereference operator; implements pointer semantics.
	/// Just use this operator as you would with a normal C++ pointer.
	RefTargetType& operator*() const {
		OVITO_ASSERT_MSG(pointer, "ReferenceField operator*", QString("Tried to dereference a NULL pointer. Reference field '%1' of class %2").arg(QString(descriptor()->identifier()), descriptor()->definingClass()->name()).toLocal8Bit().constData());
		return *(RefTargetType*)pointer.get();
	}

	/// Returns true if the internal is non-NULL.
	operator bool() const { return pointer != NULL; }
};


/// \brief Dynamic casting function for reference fields.
///
/// Returns the given object cast to type \c T if the object is of type \c T
/// (or of a subclass); otherwise returns \c NULL.
template<class T, class U>
inline T* dynamic_object_cast(const ReferenceField<U>& field) {
	return dynamic_object_cast<T,U>((U*)field);
}

/******************************************************************************
* An instance of this helper class stores a list of pointers to a RefTarget derived classes.
* RefMaker derived classes must use this helper class to store their
* vector reference lists.
******************************************************************************/
class VectorReferenceFieldBase : public PropertyFieldBase, boost::noncopyable
{
public:
	/// Default constructor that initializes the internal pointer list to an empty list.
	/// One has to call init() once to fully setup this reference field.
	VectorReferenceFieldBase() : PropertyFieldBase() {}

	/// Destructor that releases all referenced objects.
	CORE_DLLEXPORT ~VectorReferenceFieldBase();

	/// Returns the stored references as a QVector.
	operator const QVector<RefTarget*>&() const { return pointers; }

	/// Returns the reference target at index position i.
	RefTarget* operator[](int i) const { return pointers[i]; }

	/// Returns the number of objects in the vector reference field.
	int size() const { return pointers.size(); }

	/// Returns true if the vector has size 0; otherwise returns false.
	bool empty() const { return pointers.empty(); }

	/// Returns true if the vector contains an occurrence of value; otherwise returns false.
	bool contains(RefTarget* value) const { return pointers.contains(value); }

	/// Returns the index position of the first occurrence of value in the vector,
	/// searching forward from index position from. Returns -1 if no item matched.
	int indexOf(RefTarget* value, int from = 0) const { return pointers.indexOf(value, from); }

	/// Clears all references at sets the vector size to zero.
	CORE_DLLEXPORT void clear();

	/// Removes the element at index position i.
	CORE_DLLEXPORT void remove(int i);

	/// Returns the stored references as a QVector.
	const QVector<RefTarget*>& targets() const { return (const QVector<RefTarget*>&)*this; }

protected:
	/// The actual pointer list to the reference targets.
	QVector<RefTarget*> pointers;
	/// Adds a reference target to the internal list.
	CORE_DLLEXPORT int insertInternal(RefTarget* newTarget, int index = -1);

	friend class RefMaker;
	friend class RefTarget;
};

/******************************************************************************
* This is the templated version of the VectorReferenceFieldBase helper class.
* It stores a list of pointers to a RefTarget derived objects given as the template
* parameter.
******************************************************************************/
template<typename RefTargetType>
class VectorReferenceField : public VectorReferenceFieldBase
{
public:
	typedef QVector<RefTargetType*> RefTargetVector;
	typedef typename RefTargetVector::ConstIterator ConstIterator;
	typedef typename RefTargetVector::const_iterator const_iterator;
	typedef typename RefTargetVector::const_pointer const_pointer;
	typedef typename RefTargetVector::const_reference const_reference;
	typedef typename RefTargetVector::difference_type difference_type;
	typedef typename RefTargetVector::size_type size_type;
	typedef typename RefTargetVector::value_type value_type;

	/// Constructor that initializes the internal pointer to NULL.
	VectorReferenceField() : VectorReferenceFieldBase() {}

	/// Returns the stored references as a QVector.
	operator const RefTargetVector&() const { return *(const RefTargetVector*)&pointers; }

	/// Returns the reference target at index position i.
	RefTargetType* operator[](int i) const { return (RefTargetType*)pointers[i]; }

	/// Inserts a reference at the end of the vector.
	void push_back(RefTargetType* object) { insertInternal(object); }

	/// Inserts a reference at the end of the vector.
	void push_back(const intrusive_ptr<RefTargetType>& object) { insertInternal(object.get()); }

	/// Inserts a reference at index position i in the vector.
	/// If i is 0, the value is prepended to the vector.
	/// If i is size() or negative, the value is appended to the vector.
	void insert(int i, RefTargetType* object) { insertInternal(object, i); }

	/// Inserts a reference at index position i in the vector.
	/// If i is 0, the value is prepended to the vector.
	/// If i is size() or negative, the value is appended to the vector.
	void insert(int i, const intrusive_ptr<RefTargetType>& object) { insertInternal(object.get(), i); }

	/// Replaces a reference in the vector.
	/// This method removes the reference at index i and inserts the new reference at the same index.
	void set(int i, RefTargetType* object) { remove(i); insert(i, object); }

	/// Replaces a reference in the vector.
	/// This method removes the reference at index i and inserts the new reference at the same index.
	void set(int i, const intrusive_ptr<RefTargetType>& object) { remove(i); insert(i, object); }

	/// Returns an STL-style iterator pointing to the first item in the vector.
	const_iterator begin() const { return ((const RefTargetVector&)*this).begin(); }

	/// Returns an STL-style iterator pointing to the imaginary item after the last item in the vector.
	const_iterator end() const { return ((const RefTargetVector&)*this).end(); }

	/// Returns an STL-style iterator pointing to the first item in the vector.
	const_iterator constBegin() const { return begin(); }

	/// Returns a const STL-style iterator pointing to the imaginary item after the last item in the vector.
	const_iterator constEnd() const { return end(); }

	/// Returns the first reference stored in this vector reference field.
	RefTargetType* front() const { return (RefTargetType*)pointers.front(); }

	/// Returns the last reference stored in this vector reference field.
	RefTargetType* back() const { return (RefTargetType*)pointers.back(); }

	/// Copies the references of another vector reference field.
	VectorReferenceField& operator=(const VectorReferenceField& other) {
		clear();
		for(const_iterator o = other.begin(); o != other.end(); ++o)
			push_back(*o);
		return *this;
	}

	/// Returns the stored references as a QVector.
	const RefTargetVector& targets() const { return (const RefTargetVector&)*this; }
};

/// Special version of the Qt Q_FOREACH() macro that can be used with vector reference fields.
/// This is needed because the VectorReferenceField class is not a real container class
/// like the other Qt container classes (QVector, QList etc). So we have to turn
/// a VectorReferenceField into a QVector first using the targets() method before it
/// can be used in the Q_FOREACH() macro.
#define foreach_ref(var, vectorField)	Q_FOREACH(var, (vectorField).targets())

/// Bit-flags that control the behavior of a property field.
enum PropertyFieldFlag
{
	/// Default behavior.
	PROPERTY_FIELD_NO_FLAGS				= 0,
	/// Indicates that a refernce field is actually a vector of references.
	PROPERTY_FIELD_VECTOR				= (1<<1),
	/// With this flag set no undo records are created when the property value is changed.
	PROPERTY_FIELD_NO_UNDO				= (1<<2),
	/// Controls whether a REFTARGET_CHANGED message should
	/// be sent, when the property value changes.
	PROPERTY_FIELD_NO_CHANGE_MESSAGE	= (1<<3),

	/// The target of the reference field is never cloned when the owning RefMaker is cloned.
	PROPERTY_FIELD_NEVER_CLONE_TARGET	= (1<<4),
	/// The target of the reference field is always shallow/deep copied depending on the mode when the owning RefMaker is cloned.
	PROPERTY_FIELD_ALWAYS_CLONE         = (1<<5),
	/// The target of the reference field is alwayes cloned completely when the owning RefMaker is cloned.
	PROPERTY_FIELD_ALWAYS_DEEP_COPY		= (1<<6),
};
Q_DECLARE_FLAGS(PropertyFieldFlags, PropertyFieldFlag)
Q_DECLARE_OPERATORS_FOR_FLAGS(PropertyFieldFlags)

/******************************************************************************
* This structure describes one member field of a RefMaker that stores
* a property of the object.
******************************************************************************/
class CORE_DLLEXPORT PropertyFieldDescriptor
{
public:

	/// Constructor	for a property field that stores a non-animatable.
	PropertyFieldDescriptor(const char* identifier, PropertyFieldFlags flags,
			QVariant (*_propertyStorageReadFunc)(RefMaker*), void (*_propertyStorageWriteFunc)(RefMaker*, const QVariant&),
			void (*_propertyStorageSaveFunc)(RefMaker*, SaveStream&), void (*_propertyStorageLoadFunc)(RefMaker*, LoadStream&))
		: _definingClassDescriptor(NULL), _targetClassDescriptor(NULL), _identifier(identifier), _flags(flags), singleStorageAccessFunc(NULL), vectorStorageAccessFunc(NULL),
			propertyStorageReadFunc(_propertyStorageReadFunc), propertyStorageWriteFunc(_propertyStorageWriteFunc),
			propertyStorageSaveFunc(_propertyStorageSaveFunc), propertyStorageLoadFunc(_propertyStorageLoadFunc) {
		OVITO_ASSERT(_identifier != NULL);
		OVITO_ASSERT(!_flags.testFlag(PROPERTY_FIELD_VECTOR));
	}

	/// Constructor	for a property field that stores a single reference to a RefTarget.
	PropertyFieldDescriptor(const char* identifier, PropertyFieldFlags flags, SingleReferenceFieldBase& (*_storageAccessFunc)(RefMaker*))
		: _definingClassDescriptor(NULL), _targetClassDescriptor(NULL), _identifier(identifier), _flags(flags), singleStorageAccessFunc(_storageAccessFunc), vectorStorageAccessFunc(NULL),
			propertyStorageReadFunc(NULL), propertyStorageWriteFunc(NULL), propertyStorageSaveFunc(NULL), propertyStorageLoadFunc(NULL) {
		OVITO_ASSERT(_identifier != NULL);
		OVITO_ASSERT(singleStorageAccessFunc != NULL);
		OVITO_ASSERT(!_flags.testFlag(PROPERTY_FIELD_VECTOR));
	}

	/// Constructor	for a property field that stores a vector of references to RefTarget objects.
	PropertyFieldDescriptor(const char* identifier, PropertyFieldFlags flags, VectorReferenceFieldBase& (*_storageAccessFunc)(RefMaker*))
		: _definingClassDescriptor(NULL), _targetClassDescriptor(NULL), _identifier(identifier), _flags(flags), singleStorageAccessFunc(NULL), vectorStorageAccessFunc(_storageAccessFunc),
			propertyStorageReadFunc(NULL), propertyStorageWriteFunc(NULL), propertyStorageSaveFunc(NULL), propertyStorageLoadFunc(NULL) {
		OVITO_ASSERT(_identifier != NULL);
		OVITO_ASSERT(vectorStorageAccessFunc != NULL);
		OVITO_ASSERT(_flags.testFlag(PROPERTY_FIELD_VECTOR));
	}

	/// Returns the unique identifier of the reference field.
	const char* identifier() const { return _identifier; }

	/// Returns the RefMaker derived class that owns the reference.
	PluginClassDescriptor* definingClass() const { return _definingClassDescriptor; }

	/// Returns the base type of the objects stored in this property field if it is a reference field; otherwise returns NULL.
	PluginClassDescriptor* targetClass() const { return _targetClassDescriptor; }

	/// Returns whether this is a reference field that stores a pointer to a RefTarget derived class.
	bool isReferenceField() const { return _targetClassDescriptor != NULL; }

	/// Returns true if this reference field stores a vector of objects.
	bool isVector() const { return _flags.testFlag(PROPERTY_FIELD_VECTOR); }

	/// Indicates that automatic undo-handling for this property field is enabled.
	/// This is the default.
	bool automaticUndo() const { return !_flags.testFlag(PROPERTY_FIELD_NO_UNDO); }

	/// Returns true if a REFTARGET_CHANGED message is sent, each time the property value changes.
	bool sendChangeMessageEnabled() const { return !_flags.testFlag(PROPERTY_FIELD_NO_CHANGE_MESSAGE); }

    /// Returns the human readable and localized name of the property field.
	/// It will be used as label text in the user interface.
	QString displayName() const;

	/// Returns the next property field in the linked list (of the RefMaker derived class defining this property field).
	PropertyFieldDescriptor* next() const { return _next; }

	/// If this reference field contains a reference to a controller than
	/// this method returns the unit that is associated with the controller.
	/// This method is used by the NumericalParameterUI class.
	ParameterUnit* parameterUnit() const;

	/// Returns the flags that control the behavior of the property field.
	PropertyFieldFlags flags() const { return _flags; }

	/// Compares two property fields.
	bool operator==(const PropertyFieldDescriptor& other) const { return (this == &other); }

protected:

	/// The unique identifier of the reference field. This must be unique within
	/// a RefMaker derived class.
	const char* _identifier;

	/// The base type of the objects stored in this field if this is a reference field.
	PluginClassDescriptor* _targetClassDescriptor;

	/// The RefMaker derived class that owns the property.
	PluginClassDescriptor* _definingClassDescriptor;

	/// The next property field in the linked list (of the RefMaker derived class defining this property field).
	PropertyFieldDescriptor* _next;

	/// The flags that control the behavior of the property field.
	PropertyFieldFlags _flags;

	/// Stores a pointer to the function that can be used to read the property field's
	/// value for a certain RefMaker instance.
	QVariant (*propertyStorageReadFunc)(RefMaker*);

	/// Stores a pointer to the function that can be used to write the property field's
	/// value for a certain RefMaker instance.
	void (*propertyStorageWriteFunc)(RefMaker*, const QVariant&);

	/// Stores a pointer to the function that can be used to save the property field's
	/// value to a stream.
	void (*propertyStorageSaveFunc)(RefMaker*, SaveStream&);

	/// Stores a pointer to the function that can be used to load the property field's
	/// value from a stream.
	void (*propertyStorageLoadFunc)(RefMaker*, LoadStream&);

	/// Stores a pointer to the function that can be used to access the single reference field's
	/// value for a certain RefMaker instance.
	SingleReferenceFieldBase& (*singleStorageAccessFunc)(RefMaker*);

	/// Stores a pointer to the function that can be used to access the vector reference field's
	/// values for a certain RefMaker instance.
	VectorReferenceFieldBase& (*vectorStorageAccessFunc)(RefMaker*);

	/// The human-readable name of this property field. It will be used
	/// as label text in the user interface.
	QString _displayName;

	/// The ParameterUnit derived class that describes the units of the parameter
	/// if this property field stores numerical controller object.
	PluginClassDescriptor* _parameterUnitClassDescriptor;

	friend class RefMaker;
	friend class RefTarget;
};

};	// End of namespace

#include "NativePropertyFieldDescriptor.h"

#endif // __OVITO_PROPERTY_FIELD_H
