"""
Copyright  2016-2019 Maël Azimi <m.a@moul.re>

Silkaj is free software: you can redistribute it and/or modify
it under the terms of the GNU Affero General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Silkaj is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Affero General Public License for more details.

You should have received a copy of the GNU Affero General Public License
along with Silkaj. If not, see <https://www.gnu.org/licenses/>.
"""

from re import compile, search
import math
from time import sleep
import urllib

from tabulate import tabulate
from silkaj.network_tools import get_request, post_request, HeadBlock
from silkaj.crypto_tools import (
    get_publickey_from_seed,
    sign_document_from_seed,
    check_public_key,
)
from silkaj.tools import message_exit, CurrencySymbol
from silkaj.auth import auth_method
from silkaj.wot import get_uid_from_pubkey
from silkaj.money import get_sources, get_amount_from_pubkey, UDValue
from silkaj.constants import NO_MATCHING_ID


def send_transaction(
    amount, amountUD, allSources, output, comment, outputBackChange, yes
):
    """
    Main function
    """
    tx_amount, comment = cmd_transaction(
        amount, amountUD, allSources, output, comment, outputBackChange
    )
    seed = auth_method()
    issuer_pubkey = get_publickey_from_seed(seed)

    pubkey_amount = get_amount_from_pubkey(issuer_pubkey)[0]
    outputAddresses = output.split(":")
    check_transaction_values(
        comment,
        outputAddresses,
        outputBackChange,
        pubkey_amount < tx_amount * len(outputAddresses),
        issuer_pubkey,
    )

    if (
        yes
        or input(
            tabulate(
                transaction_confirmation(
                    issuer_pubkey, pubkey_amount, tx_amount, outputAddresses, comment
                ),
                tablefmt="fancy_grid",
            )
            + "\nDo you confirm sending this transaction? [yes/no]: "
        )
        == "yes"
    ):
        generate_and_send_transaction(
            seed,
            issuer_pubkey,
            tx_amount,
            outputAddresses,
            comment,
            allSources,
            outputBackChange,
        )


def cmd_transaction(amount, amountUD, allSources, output, comment, outputBackChange):
    """
    Check command line interface arguments
    """
    if not (amount or amountUD or allSources):
        message_exit("--amount nor --amountUD nor --allSources is set")
    if not output:
        message_exit("--output is not set")
    if amount:
        tx_amount = float(amount) * 100
    if amountUD:
        tx_amount = float(amountUD) * UDValue().ud_value
    if allSources:
        tx_amount = 0
    comment = comment if comment else ""
    return tx_amount, comment


def check_transaction_values(
    comment, outputAddresses, outputBackChange, enough_source, issuer_pubkey
):
    checkComment(comment)
    for outputAddress in outputAddresses:
        if check_public_key(outputAddress, True) is False:
            message_exit(outputAddress)
    if outputBackChange:
        outputBackChange = check_public_key(outputBackChange, True)
        if check_public_key(outputBackChange, True) is False:
            message_exit(outputBackChange)
    if enough_source:
        message_exit(
            issuer_pubkey + " pubkey doesn’t have enough money for this transaction."
        )


def transaction_confirmation(
    issuer_pubkey, pubkey_amount, tx_amount, outputAddresses, comment
):
    """
    Generate transaction confirmation
    """

    currency_symbol = CurrencySymbol().symbol
    tx = list()
    tx.append(
        ["pubkey’s amount before tx", str(pubkey_amount / 100) + " " + currency_symbol]
    )
    tx.append(
        [
            "tx amount (unit)",
            str(tx_amount / 100 * len(outputAddresses)) + " " + currency_symbol,
        ]
    )
    tx.append(
        [
            "tx amount (relative)",
            str(round(tx_amount / UDValue().ud_value, 4)) + " UD " + currency_symbol,
        ]
    )
    tx.append(
        [
            "pubkey’s amount after tx",
            str(((pubkey_amount - tx_amount * len(outputAddresses)) / 100))
            + " "
            + currency_symbol,
        ]
    )
    tx.append(["from (pubkey)", issuer_pubkey])
    id_from = get_uid_from_pubkey(issuer_pubkey)
    if id_from is not NO_MATCHING_ID:
        tx.append(["from (id)", id_from])
    for outputAddress in outputAddresses:
        tx.append(["to (pubkey)", outputAddress])
        id_to = get_uid_from_pubkey(outputAddress)
        if id_to is not NO_MATCHING_ID:
            tx.append(["to (id)", id_to])
    tx.append(["comment", comment])
    return tx


def generate_and_send_transaction(
    seed,
    issuers,
    AmountTransfered,
    outputAddresses,
    Comment="",
    all_input=False,
    OutputbackChange=None,
):

    while True:
        listinput_and_amount = get_list_input_for_transaction(
            issuers, AmountTransfered * len(outputAddresses), all_input
        )
        intermediatetransaction = listinput_and_amount[2]

        if intermediatetransaction:
            totalAmountInput = listinput_and_amount[1]
            print("Generate Change Transaction")
            print("   - From:    " + issuers)
            print("   - To:      " + issuers)
            print("   - Amount:  " + str(totalAmountInput / 100))
            transaction = generate_transaction_document(
                issuers,
                totalAmountInput,
                listinput_and_amount,
                issuers,
                "Change operation",
            )
            transaction += sign_document_from_seed(transaction, seed) + "\n"
            post_request(
                "tx/process", "transaction=" + urllib.parse.quote_plus(transaction)
            )
            print("Change Transaction successfully sent.")
            sleep(1)  # wait 1 second before sending a new transaction

        else:
            print("Generate Transaction:")
            print("   - From:    " + issuers)
            for outputAddress in outputAddresses:
                print("   - To:      " + outputAddress)
            if all_input:
                print("   - Amount:  " + str(listinput_and_amount[1] / 100))
            else:
                print(
                    "   - Amount:  "
                    + str(AmountTransfered / 100 * len(outputAddresses))
                )
            transaction = generate_transaction_document(
                issuers,
                AmountTransfered,
                listinput_and_amount,
                outputAddresses,
                Comment,
                OutputbackChange,
            )
            transaction += sign_document_from_seed(transaction, seed) + "\n"

            post_request(
                "tx/process", "transaction=" + urllib.parse.quote_plus(transaction)
            )
            print("Transaction successfully sent.")
            break


def generate_transaction_document(
    issuers,
    AmountTransfered,
    listinput_and_amount,
    outputAddresses,
    Comment="",
    OutputbackChange=None,
):

    totalAmountTransfered = AmountTransfered * len(outputAddresses)

    listinput = listinput_and_amount[0]
    totalAmountInput = listinput_and_amount[1]

    head_block = HeadBlock().head_block
    currency_name = head_block["currency"]
    blockstamp_current = str(head_block["number"]) + "-" + str(head_block["hash"])
    curentUnitBase = head_block["unitbase"]

    if not OutputbackChange:
        OutputbackChange = issuers

    # if it's not a foreign exchange transaction, we remove units after 2 digits after the decimal point.
    if issuers not in outputAddresses:
        totalAmountTransfered = (
            totalAmountTransfered // 10 ** curentUnitBase
        ) * 10 ** curentUnitBase

    # Generate output
    ################
    listoutput = []
    # Outputs to receiver (if not himself)
    if isinstance(outputAddresses, str):
        generate_output(listoutput, curentUnitBase, AmountTransfered, outputAddresses)
    else:
        for outputAddress in outputAddresses:
            generate_output(listoutput, curentUnitBase, AmountTransfered, outputAddress)

    # Outputs to himself
    rest = totalAmountInput - totalAmountTransfered
    generate_output(listoutput, curentUnitBase, rest, OutputbackChange)

    # Generate transaction document
    ##############################

    transaction_document = "Version: 10\n"
    transaction_document += "Type: Transaction\n"
    transaction_document += "Currency: " + currency_name + "\n"
    transaction_document += "Blockstamp: " + blockstamp_current + "\n"
    transaction_document += "Locktime: 0\n"
    transaction_document += "Issuers:\n"
    transaction_document += issuers + "\n"
    transaction_document += "Inputs:\n"
    for input in listinput:
        transaction_document += input + "\n"
    transaction_document += "Unlocks:\n"
    for i in range(0, len(listinput)):
        transaction_document += str(i) + ":SIG(0)\n"
    transaction_document += "Outputs:\n"
    for output in listoutput:
        transaction_document += output + "\n"
    transaction_document += "Comment: " + Comment + "\n"

    return transaction_document


def generate_output(listoutput, unitbase, rest, recipient_address):
    while rest > 0:
        outputAmount = truncBase(rest, unitbase)
        rest -= outputAmount
        if outputAmount > 0:
            outputAmount = int(outputAmount / math.pow(10, unitbase))
            listoutput.append(
                str(outputAmount)
                + ":"
                + str(unitbase)
                + ":SIG("
                + recipient_address
                + ")"
            )
        unitbase = unitbase - 1


def get_list_input_for_transaction(pubkey, TXamount, allinput=False):
    listinput, amount = get_sources(pubkey)

    # generate final list source
    listinputfinal = []
    totalAmountInput = 0
    intermediatetransaction = False
    for input in listinput:
        listinputfinal.append(input)
        inputsplit = input.split(":")
        totalAmountInput += int(inputsplit[0]) * 10 ** int(inputsplit[1])
        TXamount -= int(inputsplit[0]) * 10 ** int(inputsplit[1])
        # if more 40 sources, it's an intermediate transaction
        if len(listinputfinal) >= 40:
            intermediatetransaction = True
            break
        if TXamount <= 0 and not allinput:
            break
    if TXamount > 0 and not intermediatetransaction:
        message_exit("Error: you don't have enough money")
    return listinputfinal, totalAmountInput, intermediatetransaction


def checkComment(Comment):
    if len(Comment) > 255:
        message_exit("Error: Comment is too long")
    regex = compile(
        "^[0-9a-zA-Z\ \-\_\:\/\;\*\[\]\(\)\?\!\^\+\=\@\&\~\#\{\}\|\\\<\>\%\.]*$"
    )
    if not search(regex, Comment):
        message_exit("Error: the format of the comment is invalid")


def truncBase(amount, base):
    pow = math.pow(10, base)
    if amount < pow:
        return 0
    return math.trunc(amount / pow) * pow
